/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Jonas Regier <jonas.regier@open-xchange.com>
 */

define('io.ox/office/presentation/view/popup/slidepanecontextmenu', [

    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',

    'io.ox/office/tk/control/button',
    'io.ox/office/baseframework/view/control/compoundbutton',
    'io.ox/office/baseframework/view/popup/contextmenu',
    'io.ox/office/presentation/view/presentationcontrols',

    'gettext!io.ox/office/presentation/main'

], function (Utils, Forms, Button, CompoundButton, ContextMenu, Controls, gt) {

    'use strict';

    // class SlidePaneContextMenu =================================================

    /**
     * A context menu for sheet tab buttons. Provides menu actions to
     * manipulate the active sheet, or the sheet collection of the document.
     *
     * @constructor
     *
     * @extends ContextMenu
     *
     * @param {PresentationView} docView
     *  The presentation view containing this instance.
     */
    function SlidePaneContextMenu(docView, rootNode) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------
        // TODO enable key like in speadsheet??
        ContextMenu.call(this, docView, rootNode, {
            delay: 200,
            lazyInitHandler: initMenu
        });

        // private methods ----------------------------------------------------

        /**
         * ... before the context menu will be shown.

        function contextMenuPrepareHandler(event) {
        }
        */

        function initMenu() {

            // remove all existing controls from context menu
            self.destroyAllGroups();

            self.addGroup('slide/insertslide', new Button({ label: gt('Insert slide') }), { focusTarget: rootNode }) // TODO focus target documentieren
                .addGroup('slide/insertlayoutslide', new Button({ label: gt('Insert layout slide') }), { focusTarget: rootNode })
                .addGroup('slide/deleteslide', new Button({ label: gt('Delete slide') }), { focusTarget: rootNode })

                .addGroup('slide/hideslide', new Button({ label: gt('Hide slide') }), { focusTarget: rootNode })
                .addGroup('slide/unhideslide', new Button({ label: gt('Unhide slide') }), { focusTarget: rootNode })

                .addSeparator()
                .addGroup('layoutslidepicker/changelayout', new Controls.LayoutSlidePicker(docView, {

                    // overwrites
                    tooltip: null,
                    label: gt('Change Layout'),
                    icon: 'fa-chevron-right',

                    // specialized additional parameters
                    iconPos: 'trailing',
                    caret: false,
                    anchorBorder: 'right bottom top left'
                }));
        }

        // public methods -----------------------------------------------------
        this.getXY = function (event) {
            // TODO doc stick to slide
            if (!event.stickToSlide) {
                return {
                    pageX: event.sourceEvent.pageX,
                    pageY: event.sourceEvent.pageY
                };
            } else {
                var t = Utils.getNodePositionInPage(rootNode.children('.selected').find('.slidePaneThumbnail'));
                // calculate the position right to the slide with 5px offset
                return {
                    pageX: t.left + t.width + 5,
                    pageY: t.top
                };
            }
        };

        // initialization -----------------------------------------------------

        // preprocessing before the context menu will be shown
        //this.on('contextmenu:prepare', contextMenuPrepareHandler);

        // to keep the focus after pressing esc to close the popup
        this.on('popup:hide', function () {
            rootNode.focus();
        });

        // hide contextmenu if the user start scrolling
        this.listenTo(docView, 'change:scrollpos', function () { self.hide(); });

        // destroy all class members
        this.registerDestructor(function () {
            self = docView = rootNode = null;
        });

    } // class SlidePaneContextMenu

    // exports ================================================================

    // derive this class from class ContextMenu
    return ContextMenu.extend({ constructor: SlidePaneContextMenu });

});
