/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Peter Seliger <peter.seliger@open-xchange.com>
 */

define.async('io.ox/office/presentation/view/control/layoutslidepicker', [

    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',

    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/control/radiolist',

    'io.ox/office/presentation/view/layouttypetitlemap'

], function (Utils, Forms, LocaleData, RadioList, LayoutTypeTitleMap) {

    'use strict';

    // class LayoutSlidePicker ==================================================

    var
        THUMBNAIL_HEIGHT         = 50,
        VERTICAL_SPACING         = 45,

        css_value__thumb_height,
        css_value__thumb_width,

        css_value__button_height = ((THUMBNAIL_HEIGHT + VERTICAL_SPACING) + 'px'),

        $thumbContainerBlueprint = $('<div class="thumb-container" />'),
      //$thumbContainerBlueprint = $('<div class="thumb-container io-ox-busy" />'),

        i18nLayoutTypeTitleDB    = {},

        // request the proper language data targeting layout type titles
        i18nDataRequest = LocaleData.loadResource('io.ox/office/presentation/resource/layouttypetitles').done(function (data) {
            i18nLayoutTypeTitleDB = data;
        });

    /**
     * @constructor
     *
     * @extends RadioList
     */
    function LayoutSlidePicker(docView, initOptions) {

        var
            self      = this,                   // self reference

            menu,                               // the menu reference

            docModel  = docView.getDocModel();  // the document model

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({

            // hints
            tooltip: '+++ missing wording for this tooltip +++',
            label: '+++ missing wording for this label +++',

            // defaults
            icon: 'fa-clone',
            updateCaptionMode: 'none',
            toggle: true,
            itemDesign: 'grid',
            gridColumns: 5,
            smallerVersion: { hideLabel: true }

        }, initOptions));

        menu = this.getMenu();            // referable after having been extended.

        // private methods ----------------------------------------------------

        function getMasterTitle(/*key*/) {
            var
              //defaultTitle = LayoutTypeTitleMap['master'];  // - linter warning.
              //defaultTitle = LayoutTypeTitleMap[key];       // - does work, but too complicated.
                defaultTitle = LayoutTypeTitleMap.master;     // - go with what the linter did suggest.

            return (i18nLayoutTypeTitleDB[defaultTitle] || defaultTitle);
        }
        function getLayoutTypeTitle(slideId) {
          //return LayoutTypeTitleMap[docModel.getSlideFamilyAttributeForSlide(slideId, 'slide', 'type')];
            var
                defaultTitle = LayoutTypeTitleMap[docModel.getSlideFamilyAttributeForSlide(slideId, 'slide', 'type')];

            return (i18nLayoutTypeTitleDB[defaultTitle] || defaultTitle);
        }

        /**
         * hook on create items, only called once per type,
         * sets the button text and fill the chain-data for deferred rendering
         */
        function renderLayoutThumb($thumbContainer, slideId) {
          //Utils.info('+++ renderLayoutThumb :: $thumbContainer ', $thumbContainer);

            var
                $thumbPromise = docView.requestThumb(slideId),
                $layoutThumb;

            $thumbPromise.done(function (data) {
                $layoutThumb = $(data.thumb);

                // center align via math for it can't be done by css rules like `vertical-align: center`
                $layoutThumb.css('left', (Math.round(($thumbContainer.width() - $layoutThumb.data().width) / 2) + 'px'));

              //$thumbContainer.idle();
                $layoutThumb.appendTo($thumbContainer.empty());
            });
        }

        function createThumbContainer(slideId) {
          //Utils.info('+++ createThumbContainer :: slideId ', slideId);
            var
                $thumbContainer = $thumbContainerBlueprint.clone();

            $thumbContainer.attr('data-container-id', slideId);

            $thumbContainer.css('width', css_value__thumb_width);
            $thumbContainer.css('height', css_value__thumb_height);

            return $thumbContainer;
        }

        function prependThumbContainer(event, $buttonNode) {
          //Utils.info('+++ prependThumbContainer :: event, $buttonNode ', event, $buttonNode);

            $buttonNode.css('height', css_value__button_height);

            var
                slideId         = $buttonNode.attr('data-value'),
                $thumbContainer = createThumbContainer();

          //$thumbContainer.busy();
            $thumbContainer.prependTo($buttonNode);

            renderLayoutThumb($thumbContainer, slideId);
        }

        function getSectionId(slideId) {
            return [

                'master_id_',
                docModel.isMasterSlideId(slideId) ? slideId : docModel.getMasterSlideId(slideId)

            ].join('');
        }

        function createMenuSection(slideId) {   // TODO - do i18n.
            var
              //copy = getMasterTitle('master') || ('master / ' + slideId);
                copy = getMasterTitle() || ('master / ' + slideId);

            self.createMenuSection(getSectionId(slideId), { label: copy });
        }

        function createOptionButton(slideId) {  // TODO - do i18n.
            // first parameter is the `data-value` payload value that will be passed along with all form control events.
            var
                copy = getLayoutTypeTitle(slideId) || ('layout / ' + slideId);

            self.createOptionButton(slideId, { label: copy, tooltip: copy, section: getSectionId(slideId) });
          //self.createOptionButton(slideId, { label: ('layout / ' + slideId), section: getSectionId(slideId) });
        }

        function createMenuItem(slideId/*, idx, list*/) {
          //Utils.info('+++ createMenuItem :: slideId ', slideId);
            if (docModel.isMasterSlideId(slideId)) {

                createMenuSection(slideId);

            } else if (docModel.isLayoutSlideId(slideId)) {

                createOptionButton(slideId);
            }
        }

        function clearMenuContent() {
            menu.clearContents();
        }

        function createMenuContent() {
            clearMenuContent();

            docModel.getMasterSlideOrder().forEach(createMenuItem);

          //Utils.info('+++ createMenuContent +++');
        }

        // initialization -----------------------------------------------------

        function initialize() {

            self.registerUpdateHandler(function (newState, lastState) {
                if (lastState !== newState) {
                    Forms.selectNodes(self.getNode().find(Forms.BUTTON_SELECTOR), newState !== false);
                }
            });

            // destroy class members on destruction
            self.registerDestructor(function () {
                self = initOptions = docModel = docView = null;
            });

            if (!css_value__thumb_height) {
                css_value__thumb_height = (THUMBNAIL_HEIGHT + 'px');
            }
            if (!css_value__thumb_width) {
                css_value__thumb_width = (Math.round(THUMBNAIL_HEIGHT * docModel.getSlideRatio()) + 'px');
            }

            self.getMenuNode().addClass('layout-slide-picker');

            self.listenTo(menu, 'popup:beforeshow', createMenuContent);
            self.listenTo(menu, 'create:item', prependThumbContainer);
        }

        initialize();

    } // class LayoutSlidePicker

    // exports ================================================================

    return i18nDataRequest.then(_.constant(RadioList.extend({ constructor: LayoutSlidePicker })));
});
