/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/presentation/model/slideformatmanager', [
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/tk/utils'
], function (DOM, TriggerObject, TimerMixin, Utils) {

    'use strict';

    // class SlideFormatManager ===============================================

    /**
     * An instance of this class represents the slide formatting manager. This manager
     * keeps track of unformatted slides. They are formatted on demand for performance
     * reasons.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends TimerMixin
     *
     * @param {PresentationApplication} app
     *  The application instance.
     */
    function SlideFormatManager(app) {

        var // self reference
            self = this,
            // a reference to the model
            model = null,
            // a collector for all unformatted slides
            // -> this is an object, whose keys are the IDs of the unformatted slides
            allUnformattedSlides = null,
            // a collector for all slides that are currently formatted
            // -> avoiding that slides, whose formatting takes a long time, are formatted more than once
            runningTasks = {},
            // the order of the document slides
            docOrder = null,
            // the order of the master and layout slides
            layoutOrder = null,
            // a function that checks for idle time to update slides in backgroud
            idleTimeChecker = $.noop,
            // the idle time (in ms), after that a slide formatting is triggered
            idleTime = 2500,
            // the left offset that is used during updating a slide
            leftOffset = '10000px';

        // base constructors --------------------------------------------------

        TriggerObject.call(this);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Removing a task from the collector for all unformatted slides.
         *
         * @param {String} id
         *  The ID of the slide that is no longer unformatted.
         */
        function removeTask(id) {

            // removing slide from the collector of unformatted slides
            if (allUnformattedSlides && allUnformattedSlides[id]) {
                delete allUnformattedSlides[id];
            } else {
                if (allUnformattedSlides) {
                    Utils.warn('SlideFormatManager: Trying to remove non existing task: ' + id);
                } else {
                    Utils.warn('SlideFormatManager: Trying to remove task: ' + id + ' from non existent container of unformatted slides.');
                }
            }

            if (model.isStandardSlideId(id)) {
                docOrder = _.without(docOrder, id);
            } else {
                layoutOrder = _.without(layoutOrder, id);
            }

            if (_.isEmpty(allUnformattedSlides)) {
                allUnformattedSlides = null;
                docOrder = null;
                layoutOrder = null;
            }
        }

        /**
         * Element formatting for one slide and its content.
         *
         * @param {jQuery} slide
         *  The jQuerified slide node.
         *
         * @param {String} id
         *  The ID of the slide that will be formatted.
         *
         * @returns {jQuery.Promise}
         *  The promise of a deferred object that will be resolved after the specified slide is formatted
         *  completely.
         */
        function formatOneSlide(slide, id) {

            var // making slide visible for formatting
                visibilityChanged = model.handleContainerVisibility(slide, { makeVisible: true }),
                // the paragraphs in the slide
                paragraphContainer = $(),
                // the slide styles object
                drawingStyles = model.getDrawingStyles(),
                // collector for all drawings on the slide
                allDrawings = slide.find('.drawing'),
                // the content root node
                contentRootNode = app.getView().getContentRootNode();

            // updating one drawing on the slide
            function updateOneDrawing(drawing) {

                var // the paragraphs in one drawing
                    paragraphs = $(drawing).find(DOM.PARAGRAPH_NODE_SELECTOR);

                if (!model || model.destroyed) { return; } // the model might have been destroyed in the meantime

                drawingStyles.updateElementFormatting(drawing);

                // also updating all paragraphs inside text frames (not searching twice for the drawings)
                // -> using option 'suppressFontSize', so that the dynamic font size calculation is NOT
                //    started during formatting of a complete slide.
                model.implParagraphChangedSync(paragraphs, { suppressFontSize: true });

                paragraphContainer = paragraphContainer.add(paragraphs);
            }

            // always shifting the slide horizontally so that it is not visible during formatting
            contentRootNode.css('overflow-x', 'hidden'); // -> avoiding horizontal scroll bar
            slide.css('left', leftOffset);

            // updating the slide ...
            model.getSlideStyles().updateElementFormatting(slide);

            // updating all drawings via iteration
            return self.iterateArraySliced(allDrawings, updateOneDrawing, { infoString: 'SlideFormatManager: formatOneSlide' })
                .then(function () {
                    // updating the lists also synchronously
                    model.updateLists(null, paragraphContainer);
                })
                .always(function () {

                    // shifting back the slide after formatting
                    contentRootNode.css('overflow-x', '');
                    slide.css('left', '');

                    if (visibilityChanged) {
                        // making the slide invisible again
                        if (!model || model.destroyed) { return; } // the model might have been destroyed in the meantime
                        model.handleContainerVisibility(slide, { makeVisible: false });
                    }

                    // this task is no longer running
                    if (runningTasks[id]) { delete runningTasks[id]; }
                })
                .done(function () {
                    removeTask(id); // updating the collector of unformatted slides
                });
        }

        /**
         * Formatting a full set of slides. The caller of this function must make sure,
         * that after all specified slides (in container slideIDs) are formatted, the listener
         * can be informed about a fully formatted slide (all required layout and master
         * slides are also formatted).
         *
         * Info: All slide IDs that are specified in the container 'slideIDs' must exist
         *       in the global collector 'allUnformattedSlides'.
         *
         * @param {String[]) slideIDs
         *  The IDs of the slide(s) that need to be updated. It is necessary, that this
         *  is always
         *
         * @returns {jQuery.Promise}
         *  The promise of a deferred object that will be resolved after all specified slides
         *  are formatted completely.
         */
        function formatSlides(slideIDs) {

            // -> check for running tasks, so that a slide is not formatted again
            var newSlideIDs = _.reject(slideIDs, function (id) {
                return runningTasks[id];
            });

            // blocking these IDs for further formatting
            _.each(newSlideIDs, function (id) {
                runningTasks[id] = 1;
            });

            // -> iterating asynchronously over the collected slide IDs (formatting slide(s) asynchronously)
            return self.iterateArraySliced(newSlideIDs, function (id) { return formatOneSlide(model.getSlideById(id), id); }, { delay: 'immediate', infoString: 'SlideFormatManager: formatSlides' });
        }

        /**
         * Registering the handler to find idle time for formatting slides.
         *
         * @returns {undefined}
         */
        function registerIdleTimeChecker() {
            idleTimeChecker = self.createDebouncedMethod($.noop, idleSlideFormatHandler, { delay: idleTime, infoString: 'Presentation: idleSlideFormatHandler' });
            model.getNode().on('keydown mousedown touchdown', idleTimeChecker);
        }

        /**
         * Deregistering the handler to find idle time for formatting slides. This
         * can be done, if there are no more open tasks.
         */
        function deregisterIdleTimeChecker() {
            model.getNode().off('keydown mousedown touchdown', idleTimeChecker);
            idleTimeChecker = $.noop;
        }

        /**
         * Determining the ID of the slide that needs to be formatted next. This is done
         * with an idle time, so that it is possible to format any of the remaining
         * unformatted slides.
         *
         * @returns {String}
         *  The ID of the slide that shall be formatted.
         */
        function getNextFormatSlideId() {

            var // whether the layout/master view is visible
                isMasterView = model.isMasterView(),
                // the id of the slide, that shall be formatted
                slideID = isMasterView ? _.first(layoutOrder) : _.first(docOrder);

            // Fallback: simply take one arbitrary slide ID
            if (!slideID || !allUnformattedSlides[slideID]) { slideID = _.first(_.keys(allUnformattedSlides)); }

            return slideID;
        }

        /**
         * Handler to make slide formatting, if there was no user input for a specified time.
         */
        function idleSlideFormatHandler() {

            var // the ID of the slide that will be formatted
                slideID = null,
                // the set of affected slides
                set = null,
                // a collector for the unformatted slides
                formatCollector = null;

            if (!model || model.destroyed) { return; } // handling closing of application

            if (self.unformattedSlideExists()) {

                // Trigger formatting one slide
                slideID = getNextFormatSlideId();
                // -> this must be a full set of slides
                set = model.getSlideIdSet(slideID);

                if (set.id && allUnformattedSlides[set.id]) {
                    formatCollector = [set.id];
                }

                if (set.layoutId && allUnformattedSlides[set.layoutId]) {
                    if (formatCollector) {
                        formatCollector.push(set.layoutId);
                    } else {
                        formatCollector = [set.layoutId];
                    }
                }

                if (set.masterId && allUnformattedSlides[set.masterId]) {
                    if (formatCollector) {
                        formatCollector.push(set.masterId);
                    } else {
                        formatCollector = [set.masterId];
                    }
                }

                if (formatCollector) {
                    // formatting the collected slides
                    formatSlides(formatCollector).done(function () {
                        // informing the listeners
                        informListeners(formatCollector);
                        // triggering next slide update
                        if (self.unformattedSlideExists()) {
                            idleTimeChecker();
                        } else {
                            deregisterIdleTimeChecker();
                        }
                    });
                } else {
                    // triggering next slide update
                    if (self.unformattedSlideExists()) {
                        idleTimeChecker();
                    } else {
                        deregisterIdleTimeChecker();
                    }
                }

            } else {
                deregisterIdleTimeChecker();
            }
        }

        /**
         * Listener for the event 'change:slide', that is triggered from the model,
         * if the active slide was modified.
         *
         * @param {jQuery.Event} event
         *  A jQuery event object.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.newSlideId='']
         *      The id of the standard slide that will be activated.
         *  @param {String} [options.newLayoutId='']
         *      The id of the layout slide that will be activated.
         *  @param {String} [options.newMasterId='']
         *      The id of the master slide that will be activated.
         *  @param {String} [options.oldSlideId='']
         *      The id of the standard slide that will be deactivated.
         *  @param {String} [options.oldLayoutId='']
         *      The id of the layout slide that will be deactivated.
         *  @param {String} [options.oldMasterId='']
         *      The id of the master slide that will be deactivated.
         *  @param {Boolean} [options.isMasterChange=false]
         *      Whether only the master slide for a layout slide was changed.
         *
         * @returns {jQuery.Promise}
         *  The promise of a deferred object that will be resolved after all specified
         *  slides are formatted completely.
         *  Info: This promise should not be used, except in the case of the function
         *        'forceSlideFormatting'. This is required for the first visible slide
         *        that needs to block the document load process. Typically all affected
         *        processes must be informed via the event 'slidestate:formatted'.
         */
        function changeSlideHandler(event, options) {

            var // a collector for the unformatted slides
                formatCollector = null;

            if (!self.unformattedSlideExists()) { return; }

            if (options.newSlideId && allUnformattedSlides[options.newSlideId]) {
                formatCollector = [options.newSlideId];
            }

            if (options.newLayoutId && allUnformattedSlides[options.newLayoutId]) {
                if (formatCollector) {
                    formatCollector.push(options.newLayoutId);
                } else {
                    formatCollector = [options.newLayoutId];
                }
            }

            if (options.newMasterId && allUnformattedSlides[options.newMasterId]) {
                if (formatCollector) {
                    formatCollector.push(options.newMasterId);
                } else {
                    formatCollector = [options.newMasterId];
                }
            }

            if (formatCollector) {

                // formatting the collected slides
                return formatSlides(formatCollector).done(function () {
                    // informing the listeners
                    informListeners(formatCollector);
                });
            } else {
                return $.when();
            }
        }

        /**
         * Listener for the event 'request:formatting' that can be triggered by an
         * external source to request a formatting update of the specified slides.
         * external trigger of slide formatting
         *
         * The formatting of the slides is done asynchronously.
         *
         * This function also handles that only 'complete' formatting is done. If
         * a document slide is in the specified list 'allSlides', the corresponding
         * layout and master slide are also formatted, if this is necessary.
         *
         * Listeners are informed with the event 'slidestate:formatted' from the
         * model, that the specified slides are formatted.
         *
         * @param {jQuery.Event} event
         *  A jQuery event object.
         *
         * @param {String|String[]|Object} allSlides
         *  The IDs of the slide(s) that need to be updated. This can be a single string,
         *  an array of strings or an object, whose values are interpreted as string IDs.
         *  Such an object is created by the function 'getSlideIdSet' from the model.
         */
        function externalFormattingHandler(event, allSlides) {

            var // the container for the slide IDs
                container = null;

            if (_.isString(allSlides)) {
                container = [allSlides];
            } else if (_.isArray(allSlides)) {
                container = allSlides;
            } else if (_.isObject(allSlides)) {
                container = _.values(allSlides);
            }

            // shortcut, if there are no unformatted slides
            if (!self.unformattedSlideExists()) {
                // informing the listeners that all slides are formatted
                informListeners(container);
                return;
            }

            _.each(container, function (id) {

                var // the set of affected slides
                    set = model.getSlideIdSet(id),
                    // a collector for the unformatted slides
                    formatCollector = null;

                if (set.id && allUnformattedSlides[set.id]) {
                    formatCollector = [set.id];
                }

                if (set.layoutId && allUnformattedSlides[set.layoutId]) {
                    if (formatCollector) {
                        formatCollector.push(set.layoutId);
                    } else {
                        formatCollector = [set.layoutId];
                    }
                }

                if (set.masterId && allUnformattedSlides[set.masterId]) {
                    if (formatCollector) {
                        formatCollector.push(set.masterId);
                    } else {
                        formatCollector = [set.masterId];
                    }
                }

                // Is it necessary to inform the listener, even if all slides are already formatted?

                if (formatCollector) {
                    // formatting the collected slides
                    formatSlides(formatCollector).done(function () {
                        // informing the listeners
                        informListeners([id]);
                    });
                } else {
                    // nothing to do, required slide is already formatted
                    informListeners([id]);
                }
            });

        }

        /**
         * Informing the listeners, that the slide with the specified IDs are formatted.
         * This includes a check, if the model was destroyed in the meantime, because the trigger
         * is executed with a delay.
         *
         * @param {String[]} allSlideIDs
         *  An array with all formatted slide IDs.
         */
        function informListeners(allSlideIDs) {
            self.executeDelayed(function () {
                if (model && !model.destroyed) { model.trigger('slidestate:formatted', allSlideIDs); }
            }, 50, 'Presentation: Trigger slidestate:formatted');
        }

        // methods ------------------------------------------------------------

        /**
         * Whether there is at least one unformatted slide.
         *
         * @returns {Boolean}
         *  Whether there is at least one unformatted slide.
         */
        this.unformattedSlideExists = function () {
            return allUnformattedSlides && !_.isEmpty(allUnformattedSlides);
        };

        /**
         * Whether the slide specified by its ID is an unformatted slide.
         *
         * @param {String} id
         *  The id of a slide.
         *
         * @returns {Boolean}
         *  Whether the slide specified by its ID is an unformatted slide.
         */
        this.isUnformattedSlide = function (id) {
            return allUnformattedSlides && !_.isUndefined(allUnformattedSlides[id]);
        };

        /**
         * Adding additional IDs to the container of unformatted slide IDs.
         *
         * @param {String|String[]} IDs
         *  A single string or an array of strings with the IDs of slides, that need
         *  to be formatted later.
         */
        this.addTasks = function (IDs) {

            if (_.isString(IDs)) {
                if (model.isValidSlideId(IDs)) {
                    if (!allUnformattedSlides) { allUnformattedSlides = {}; }
                    allUnformattedSlides[IDs] = 1;
                }
            } else if (_.isArray(IDs)) {
                if (!allUnformattedSlides) { allUnformattedSlides = {}; }
                _.each(IDs, function (id) {
                    allUnformattedSlides[id] = 1;
                });
            }
        };

        /**
         * Forcing an update of one specified slide and its layout and master slide (if required).
         * This process is especially required for formatting the first visible slide.
         *
         * @param {String} id
         *  The ID of the slide, that needs to be formatted.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the specified slide and all its parent slides
         *  are formatted.
         */
        this.forceSlideFormatting = function (id) {

            var // the set of affected slides
                set = model.getSlideIdSet(id);

            return changeSlideHandler(null, { newSlideId: id, newLayoutId: set.layoutId, newMasterId: set.masterId });
        };

        /**
         * Setting the initial tasks for the slide formatting manager. This is typically done with
         * the object 'idTypeConnection' from the model.
         *
         * @param {Object} idObject
         *  An object, that contains the IDs of the slides, that need to be formatted as keys.
         *
         * @param {Array} docSlideOrder
         *  Copy of the container with the document slide IDs in the correct order.
         *
         * @param {Array} layoutSlideOrder
         *  Copy of the container with the slide IDs of master and layout slides in the correct order.
         */
        this.initializeManager = function (idObject, docSlideOrder, layoutSlideOrder) {

            allUnformattedSlides = idObject;
            docOrder = docSlideOrder;
            layoutOrder = layoutSlideOrder;

            // registering a listener for the event 'request:formatting', that can be triggered
            // to force the slide format manager to format some slides specified by their IDs.
            self.listenTo(model, 'request:formatting', externalFormattingHandler);

            // starting the process to format slides, if there is no user input for a specified time
            registerIdleTimeChecker();
            idleTimeChecker(); // starting process

          //model.trigger('formatmanager:init');
        };

        /**
         * Checks whether a specified slide is currently formatted.
         *
         * @param {String} id
         *  The ID of the slide, that needs to be formatted.
         *
         * @returns {Boolean}
         *  Whether the slide with the specified id is currently formatted.
         */
        this.isRunningTask = function (id) {
            return runningTasks[id] === 1;
        };

        // initialization -----------------------------------------------------

        app.onInit(function () {
            model = app.getModel();
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            deregisterIdleTimeChecker();
            model = allUnformattedSlides = docOrder = layoutOrder = null;
        });

    } // class SlideFormatManager

    // constants --------------------------------------------------------------

    // export =================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: SlideFormatManager });
});
