/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/presentation/model/model', [
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/utils',
    'io.ox/office/presentation/model/listhandlermixin',
    'io.ox/office/presentation/model/modelattributesmixin',
    'io.ox/office/presentation/model/slideattributesmixin',
    'io.ox/office/presentation/model/slideoperationmixin',
    'io.ox/office/presentation/model/objectoperationmixin',
    'io.ox/office/presentation/model/pagehandlermixin',
    'io.ox/office/presentation/model/updatedocumentmixin',
    'io.ox/office/presentation/model/dynfontsizemixin',
    'io.ox/office/presentation/model/updatelistsmixin',
    'io.ox/office/presentation/model/dragtocopymixin',
    'io.ox/office/presentation/components/drawing/drawingresize',
    'io.ox/office/textframework/model/editor',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/view/selectionbox',
    'io.ox/office/presentation/utils/operations',
    'io.ox/office/presentation/utils/presentationutils',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/drawinglayer/view/imageutil',
    'io.ox/office/presentation/components/field/slidefield',
    'io.ox/office/presentation/model/numberformatter',
    'io.ox/office/presentation/model/slideformatmanager'
], function (KeyCodes, Utils, ListHandlerMixin, ModelAttributesMixin, SlideAttributesMixin, SlideOperationMixin, ObjectOperationMixin, PageHandlerMixin, UpdateDocumentMixin, DynFontSizeMixin, UpdateListsMixin, DragToCopyMixin, DrawingResize, Editor, DOM, Position, SelectionBox, Operations, PresentationUtils, DrawingFrame, Image, SlideField, NumberFormatter, SlideFormatManager) {

    'use strict';

    // class PresentationModel ================================================

    /**
     * Represents the document model of a presentation application.
     *
     * Triggers the events supported by the base class EditModel, and the
     * Triggers the events supported by the base class Editor, and the
     * following additional events:
     * - 'inserted:slide': A new slide was inserted.
     * - 'removed:slide': A slide was removed.
     * - 'removed:activeslide': The active slide was removed.
     * - 'moved:slide': A slide was moved.
     * - 'change:activeView:after': The active view was modified, triggering after handler.
     * - 'change:activeView': The active view was modified.
     * - 'change:slide:before': The active slide was modified, triggering before handler.
     * - 'change:slide': The active slide was modified.
     * - 'change:slidelayout': A layout or master slide was modified.
     * - 'slideModel:init': All slide related model values are ready to use (e.g. for initializing the slide pane).
     * - 'slidestate:formatted': At least one slide is completely formatted.  Parameter is an array of all slide IDs,
     *                          that are formatted.
     * - 'slidestate:update': The content of at least one slide was modified so that any preview items within
     *                          the slide pane need to be updated. Parameter is an array of all slide IDs, that
     *                          require an update in the slide pane.
     *                          Furthermore an object, that contains as keys all those slide IDs, for that the
     *                          slide attribute 'followMasterShapes' was modified. The value for each slide ID
     *                          is the Boolean, whether 'followMasterShapes' was set to true or false. Additional
     *                          parameter is an options object, that contains the property 'onlyDocumentSlides'.
     *                          If set to true, no master or layout slides were modified.
     *
     * @constructor
     *
     * @extends Editor
     * @extends ListHandlerMixin
     * @extends ModelAttributesMixin
     * @extends SlideAttributesMixin
     * @extends PageOperationMixin
     * @extends ObjectOperationMixin
     * @extends PageHandlerMixin
     * @extends UpdateDocumentMixin
     * @extends DynFontSizeMixin
     * @extends UpdateListsMixin
     *
     * @param {PresentationApplication} app
     *  The application containing this document model.
     */
    function PresentationModel(app) {

        var // self reference for local functions
            self = this,
            // the root element for the document contents
            pagediv = null,
            // the logical selection, synchronizes with browser DOM selection
            selection = null,
            // a counter for the 'standard' slides, used to generate unique keys for the slides (only used locally)
            mainSlideCounter = 1,
            // shortcuts for style sheet containers
            characterStyles = null, paragraphStyles = null, tableStyles = null, tableCellStyles = null, tableRowStyles = null, drawingStyles = null, slideStyles = null, pageStyles = null,
            // the model for all 'standard' slides. Those slides, that are not master or layout slides.
            allStandardSlides = {},
            // the model for the layout slides, an object with target string as key and target node as value
            allLayoutSlides = {},
            // the model for the master slides, an object with target string as key and target node as value
            allMasterSlides = {},
            // the model for standard/layout slides -> the key is the id of the standard slide, the value the key of the layout slide
            layoutConnection = {},
            // the model for layout/master slides -> the key is the id of the layout slide, the value the key of the master slide
            layoutMasterConnection = {},
            // a collector that combines the id of a slide with the type as key value pair
            idTypeConnection = {},
            // container with the slide IDs in the correct order (model)
            slideOrder = [],
            // container with the slide IDs of master and layout slides in the correct order (model)
            masterSlideOrder = [],
            // the id of the currently active slide (set in setActiveSlideId)
            activeSlideId = '',
            // the id of the active slide that was used before switching to master/layout view
            lastActiveStandardSlideId = '',
            // whether the master and layout slides are visible (true) or the 'standard' slides are visible (model) (set in setActiveView)
            isMasterView = false,
            // a collector for the slide IDs that need to be updated in the slide pane
            allModifiedSlideIDs = [],
            // the layer node for all layout slide nodes
            layoutSlideLayerNode = null,
            // the layer node for all master slide nodes
            masterSlideLayerNode = null,
            // a selection range in a master or layout slide
            masterLayoutSelection = null,
            // the currently used slidePaneWidth by the user in the browser, it's a temp value independent from the operation.
            localViewSlidePaneWidth = null,
            // debounced function for updating a slide and its content completely. It will be set after successful document load.
            implSlideChanged = $.noop,
            // the handler function that can be used to inform the slide pane about changes (that are not handled in 'operations:after')
            debouncedUpdateSlideStateHandler = $.noop,
            // the selection box object
            selectionBox = null,
            // the paragraph node that contains the selection
            selectedParagraph = $(),
            // a helper to keep track of already formatted slides (performance)
            slideFormatManager = null,
            // a marker to remember the last slide that should be activated
            lastActivationSlideId = null;

        // base constructor ---------------------------------------------------

        Editor.call(this, app, { slideMode: true });
        SlideOperationMixin.call(this);
        ObjectOperationMixin.call(this, app);
        ListHandlerMixin.call(this);
        ModelAttributesMixin.call(this);
        SlideAttributesMixin.call(this);
        PageHandlerMixin.call(this, app);
        UpdateDocumentMixin.call(this, app);
        DynFontSizeMixin.call(this, app);
        UpdateListsMixin.call(this, app);
        DragToCopyMixin.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Deleting the model of the presentation app completely. This is necessary, if a new
         * snapshot is applied.
         */
        function emptyPresentationModel() {

            allStandardSlides = {};
            allLayoutSlides = {};
            allMasterSlides = {};
            layoutConnection = {};
            layoutMasterConnection = {};
            idTypeConnection = {};
            slideOrder = [];
            masterSlideOrder = [];

            mainSlideCounter = 1;
            activeSlideId = '';
            isMasterView = false;

            // emptying the drawing place holder container
            drawingStyles.emptyPlaceHolderModel();

            // emptying the slide attributes model
            self.emptySlideAttributesModel();
        }

        /**
         * Registering one layout or master slide in the slide model.
         *
         * @param {jQuery} slide
         *      The jQuerified slide that will be saved inside the slide collector. The key
         *      inside this collector is the id, the value the jQuerified slide node.
         *
         * @param {String} id
         *      The id of the slide. For master and layout slides, this id is sent from the
         *      server. For 'standard' slides a locally generated id is used.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.type='']
         *      The type of the slide. This is required to distinguish master slides, layout
         *      slide and 'standard' slides. The latter are the default.
         *  @param {String} [options.target='']
         *      The target of the slide that will be registered. This is especially important
         *      for the connection from layout slides to master slides.
         *  @param {Number} [options.index=-1]
         *      The zero-based index describing the position of the slide inside its view. This
         *      can be the container for the standard slides or the container for the master
         *      layout slides.
         *  @param {Object} [options.attrs=null]
         *      The slide attributes.
         *  @param {Object} [options.documentReloaded=false]
         *      Whether the document was reloaded with a snapshot.
         *
         * @returns {Number}
         *  The number of slides in the standard or master/layout model.
         */
        function addIntoSlideModel(slide, id, options) {

            var // the type of the slide
                type = Utils.getStringOption(options, 'type', ''),
                // the target of the slide
                target = null,
                // the order index at which the slide shall be inserted
                index = Utils.getNumberOption(options, 'index', -1),
                // whether the slide is a standard slide
                isStandardSlide = false,
                // whether the model is modified
                insertedSlide = false,
                // an optional attribute object for the new slide
                attrs = Utils.getObjectOption(options, 'attrs', null),
                // the order index at which the slide shall be inserted
                documentReloaded = Utils.getBooleanOption(options, 'documentReloaded', false);

            if (type === PresentationModel.TYPE_SLIDE) {

                isStandardSlide = true;
                insertedSlide = true;

                // adding the slide into the model
                allStandardSlides[id] = slide;

                if (index > -1) {
                    slideOrder.splice(index, 0, id);
                } else {
                    slideOrder.push(id); // if index not specified, append slide to the end
                }

                // reading the required options
                target = Utils.getStringOption(options, 'target', '');

                // saving connection between standard slide and its layout slide
                if (target) { layoutConnection[id] = target; }

            } else if (type === PresentationModel.TYPE_MASTER_SLIDE) {

                insertedSlide = true;

                // saving the jQuerified slide in the model
                allMasterSlides[id] = slide;

                // adding the ID of the master slide into the sorted container
                if (index > -1) {
                    masterSlideOrder.splice(index, 0, id);
                } else {
                    masterSlideOrder.push(id); // if index not specified, append slide to the end
                }

            } else if (type === PresentationModel.TYPE_LAYOUT_SLIDE) {

                insertedSlide = true;

                // saving the jQuerified slide in the model
                allLayoutSlides[id] = slide;

                // reading the required options
                target = Utils.getStringOption(options, 'target', '');

                // saving connection between layout slide and its master slide
                if (target) { layoutMasterConnection[id] = target; }

                // adding the ID of the layout slide into the sorted container
                if (index > -1) {
                    masterSlideOrder.splice(index, 0, id);
                } else {
                    masterSlideOrder.push(id); // if index not specified, append slide to the end
                }

            } else {
                Utils.log('Warning: Unknown slide type: ' + type);
            }

            // registering the type of the slide for every id
            idTypeConnection[id] = type;

            // saving the slide attributes in slide model
            if (attrs) { self.saveSlideAttributes(id, attrs); }

            // saving the target also at the slide (required for generating undo operation and for snapshots)
            if (target) { slide.data('target', target); }  // -> this needs to be updated, if the model changes

            // informing the listeners, that a new slide was inserted (model was updated)
            if (insertedSlide) { self.trigger('inserted:slide', { id: id, isMasterView: isMasterView, documentReloaded: documentReloaded }); }

            return isStandardSlide ? slideOrder.length : masterSlideOrder.length;
        }

        /**
         * Removing a slide specified by its ID from the model.
         * Before calling this function it must be checked, if the specified slide is the
         * only slide in its view (self.isOnlySlideInView(id)) or if it is a used
         * master or layout slide (self.isUsedMasterOrLayoutSlide(id)).
         *
         * @param {String} id
         *      The id of the slide. For master and layout slides, this id is sent from the
         *      server. For 'standard' slides a locally generated id is used.
         *
         * @returns {Number}
         *  The number of slides in the standard or master/layout model.
         */
        function removeFromSlideModel(id) {

            // Required checks:
            // self.isUsedMasterOrLayoutSlide(id) -> not removing master/layout slides that are referenced
            // self.isOnlySlideInView(id)         -> not removing the last slide in its view

            var // whether the slide is a standard slide
                isStandardSlide = false,
                // whether the model is modified
                removedSlide = false;

            // if this was the active slide, it cannot be active anymore after removal
            if (isValidSlideId(id) && (id === activeSlideId)) {
                activeSlideId = '';
                // handling visibility of master and layout slide
                // -> this needs to be done, before the model is updated
                self.trigger('removed:activeslide', self.getSlideIdSet(id));
            }

            if (self.isStandardSlideId(id)) {

                isStandardSlide = true;
                removedSlide = true;

                // removing the slide from the model
                delete allStandardSlides[id];

                // removing the connection between standard slide and its layout slide
                delete layoutConnection[id];

                // removing the id from the slide order array
                slideOrder = _.without(slideOrder, id);

            } else if (self.isLayoutSlideId(id)) {

                removedSlide = true;

                // removing the slide from the model
                delete allLayoutSlides[id];

                // removing the connection between layout slide and its masterlayout slide
                delete layoutMasterConnection[id];

                // removing the id from the master slide order array
                masterSlideOrder = _.without(masterSlideOrder, id);

            } else if (self.isMasterSlideId(id)) {

                removedSlide = true;

                // removing the slide from the model
                delete allMasterSlides[id];

                // removing the id from the master slide order array
                masterSlideOrder = _.without(masterSlideOrder, id);
            }

            // removing registration of type for the slide
            delete idTypeConnection[id];

            // deleting the attributes in the attributes model
            self.deleteSlideAttributes(id);

            // informing the listeners, that a slide was removed (model was updated)
            if (removedSlide) { self.trigger('removed:slide', { id: id, isMasterView: isMasterView }); }

            return isStandardSlide ? slideOrder.length : masterSlideOrder.length;
        }

        /**
         * Changing the ID of a layout slide assigned to a document slide.
         *
         * @param {String} id
         *  The ID of the document slide that will get a new layout slide ID assigned.
         *
         * @param {String} layoutId
         *  The ID of the layout slide.
         *
         * @returns {Boolean}
         *  Whether the new layout ID was assigned to the specified document slide ID.
         */
        function changeLayoutIdInModel(id, layoutId) {

            var // whether the new layout ID was assigned to the document id
                changed = false;

            // update the model
            if (id && layoutId && _.isString(layoutConnection[id]) && self.isLayoutSlideId(layoutId)) {
                // changing connection between document slide and its layout slide
                layoutConnection[id] = layoutId;

                // saving the target also at the slide (required for generating undo operation and for snapshots)
                self.getSlideById(id).data('target', layoutId);

                // Trigger event about change of model (?)

                changed = true;
            }

            return changed;
        }

        /**
         * Changing the ID of a master slide assigned to a layout slide.
         *
         * @param {String} id
         *  The ID of the layout slide that will get a new master slide ID assigned.
         *
         * @param {String} masterId
         *  The ID of the master slide.
         *
         * @returns {Boolean}
         *  Whether the new master ID was assigned to the specified layout slide ID.
         */
        function changeMasterIdInModel(id, masterId) {

            var // whether the new master ID was assigned to the layout id
                changed = false;

            // update the model
            if (id && masterId && _.isString(layoutMasterConnection[id]) && self.isMasterSlideId(masterId)) {
                // changing connection between document slide and its layout slide
                layoutMasterConnection[id] = masterId;

                // saving the target also at the slide (required for generating undo operation and for snapshots)
                self.getSlideById(id).data('target', masterId);

                // Trigger event about change of model (?)

                changed = true;
            }

            return changed;
        }

        /**
         * Moving the ID in the sorted slide container from index 'from' to index 'to'.
         *
         * @param {Number} from
         *  The current index of the moved slide ID.
         *
         * @param {Number} to
         *  The new index of the moved slide ID.
         *
         * @param {Boolean} isStandardSlide
         *  Whether the moved slide is in the standard or the master/layout container.
         *
         * @returns {String|Null}
         *  The ID of the moved slide. Or null, if no slide was moved.
         */
        function moveSlideIdInModel(from, to, isStandardSlide) {

            var // the ID of the moved slide
                id = null,
                // the affected container
                container = isStandardSlide ? slideOrder : masterSlideOrder,
                // the length of the container
                max = container.length;

            if (_.isNumber(from) && _.isNumber(to) && to !== from && from < max && to < max) {

                id = container[from];

                // 1. step remove slide ID
                // 2. step insert slide ID
                container.splice(from, 1); // delete
                container.splice(to, 0, id); // insert

                // informing the listeners, that the slide order was changed (model was updated)
                self.trigger('moved:slide', { id: id, to: to, from: from, isMasterView: !isStandardSlide });

            //    Utils.log('Container: ' + JSON.stringify(container)); // TODO: Remove debug code
            }

            return id;
        }

        /**
         * Setting the id of the active slide (model).
         *
         * @param {String} id
         *  The zero-based index of the active slide inside the standard or  master/layout view.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.forceUpdate=false]
         *   Whether a view update is required, even if the index of the slide did not change.
         *
         * @returns {Boolean}
         *  Whether another slide was activated.
         */
        function setActiveSlideId(id, options) {

            var // whether the slide was modified
                modified = activeSlideId !== id,
                // whether an update is forced
                forceUpdate = Utils.getBooleanOption(options, 'forceUpdate', false),
                // whether another view was activated
                activated = false,
                // the id of the currently active slide
                oldId = activeSlideId,
                // collector for all modified slides
                viewCollector = {};

            if (isValidSlideId(id) || (id === '')) {

                lastActivationSlideId = id; // setting marker to remember this slide

                if (slideFormatManager.isUnformattedSlide(id)) { // is the slide already formatted -> if not, do not activate it
                    // the slide cannot be activated immediately
                    if (!slideFormatManager.isRunningTask(id)) {  // check, if it is currently formatted
                        slideFormatManager.forceSlideFormatting(id).done(function () {
                            // after formatting is done, activating the slide, if still required
                            if (lastActivationSlideId === id) { setActiveSlideId(id); }
                        });

                    }
                    return activated; // shortcut for unformatted slides
                }

                // modifying the model
                activeSlideId = id;

                if (modified || forceUpdate) {

                    // setting all IDs that need to be activated
                    if (self.isStandardSlideId(id)) {
                        viewCollector.newSlideId = id;
                        viewCollector.newLayoutId = self.getLayoutSlideId(viewCollector.newSlideId);
                        viewCollector.newMasterId = self.getMasterSlideId(viewCollector.newLayoutId);
                    } else if (self.isLayoutSlideId(id)) {
                        viewCollector.newLayoutId = id;
                        viewCollector.newMasterId = self.getMasterSlideId(viewCollector.newLayoutId);
                    } else if (self.isMasterSlideId(id)) {
                        viewCollector.newMasterId = id;
                    }

                    // setting all IDs that need to be deactivated
                    if (self.isStandardSlideId(oldId)) {
                        viewCollector.oldSlideId = oldId;
                        viewCollector.oldLayoutId = self.getLayoutSlideId(viewCollector.oldSlideId);
                        viewCollector.oldMasterId = self.getMasterSlideId(viewCollector.oldLayoutId);
                    } else if (self.isLayoutSlideId(oldId)) {
                        viewCollector.oldLayoutId = oldId;
                        viewCollector.oldMasterId = self.getMasterSlideId(viewCollector.oldLayoutId);
                    } else if (self.isMasterSlideId(oldId)) {
                        viewCollector.oldMasterId = oldId;
                    }

                    // add the currently active slide to the viewCollector
                    viewCollector.activeSlideId = self.getActiveSlideId();

                    // informing the listeners, that the slide need to be updated (model was updated)
                    self.trigger('change:slide:before', viewCollector);

                    // informing the listeners, that the slide need to be updated (model was updated)
                    self.trigger('change:slide', viewCollector);

                    // informing the listeners, that the scroll bar needs to be updated
                    self.trigger('update:verticalscrollbar', { pos: getActiveSlideIndex() });

                    // clearing all drawing selections (including removing of all handlers at drawing)
                    self.getSelection().clearAllDrawingSelections();

                    // updating the value for the active target (TODO)
                    self.setActiveTarget(self.isLayoutOrMasterId(activeSlideId) ? activeSlideId : '');
                    self.getSelection().setNewRootNode(self.getRootNode(activeSlideId));

                    if (id === '') {
                        self.getSelection().setEmptySelection();
                    } else {
                        // selecting the slide (TODO: Check if this can be avoided)
                        self.getSelection().setSlideSelection();
                    }

                    // a new slide was activated
                    activated = true;
                }
            }

            return activated;
        }

        /**
         * Setting the active view (model) and triggering the events 'change:activeView:after' and
         * 'change:activeView' event to notify observers.
         * The active view can be the view of the normal/standard slides or the view of the master/layout slides.
         *
         * @param {Boolean} isMaster
         *  Whether the view of the master/layout slides shall be activated.
         *
         * @returns {Boolean}
         *  Whether another slide view was activated.
         */
        function setActiveView(isMaster) {

            var // whether the slide view was modified
                modified = (isMaster !== isMasterView),
                // the number of slides in the new active view
                slideCount = 0;

            // updating the model
            if (modified) {

                // changing the model
                isMasterView = isMaster;

                slideCount = self.getActiveViewCount();

                // informing the listeners, that the activeView is changed (model was updated)
                self.trigger('change:activeView', { isMasterView: isMasterView, slideCount: slideCount });

                // informing the listeners, that the activeView is changed (model was updated)
                self.trigger('change:activeView:after', { isMasterView: isMasterView, slideCount: slideCount });
            }

            return modified;
        }

        /**
         * Provides a unique ID for the standard slides. This ID is used locally only. The master
         * and layout slides get their IDs from the server. The local ID for the standard slides
         * is used for the model only.
         *
         * @returns {String}
         *  A unique id.
         */
        function getNextSlideID() {
            return 'slide_' + mainSlideCounter++;
        }

        /**
         * Provides a unique ID for the custom layout slides. The local ID for the custom layout slides
         * is used for the model only.
         *
         * @returns {String}
         *  A unique id.
         */
        function getNextCustomLayoutId() {
            var highestId = _.first(_.clone(masterSlideOrder).sort(function (a, b) { return b - a; }));
            return parseInt(highestId, 10) + 1 + '';
        }

        /**
         * Getting the index of the active slide in the current container
         *
         * @returns {Number}
         *  The index of the active slide in the current container. If the
         *  active slide id could not be found in the array, '-1' is returned.
         */
        function getActiveSlideIndex() {

            var // the container that contains the order of the slides
                orderContainer = isMasterView ? masterSlideOrder : slideOrder;

            return _.indexOf(orderContainer, activeSlideId);
        }

        /**
         * Helper function to generate the debounced function for updating a slide
         * and all its place holder drawings completely. The generated function
         * should be called carefully, because of performance reasons. This function
         * needs to be created after successful loading.
         *
         * @returns {Function}
         *  The debounced function for updating a slide completely.
         */
        function getDebouncedUpdateSlideHandler() {

            var // all slide nodes that need to be updated
                slides = $();

            // direct callback: called every time when implSlideChanged() has been called
            function registerSlide(slide) {
                // store the new slide in the collection (jQuery keeps the collection unique)
                slides = slides.add(slide);
            }

            // deferred callback: called once, after current script ends
            function updateSlides() {

                _.each(slides, function (slide) {

                    // updating the slide, so that background drawings are handled correctly
                    self.getSlideStyles().updateElementFormatting(slide);

                    // updating all place holder drawings in the slide
                    self.getDrawingStyles().updatePlaceHolderDrawings(slide);

                });

                slides = $(); // reset collector
            }

            // create and return the deferred implTableChanged() method
            return self.createDebouncedMethod(registerSlide, updateSlides, { infoString: 'Presentation: implSlideChanged' });
        }

        /**
         * Helper function to keep the global 'mainSlideCounter' up-to-date. After loading the
         * document with fast load or from local storage, this number needs to be updated.
         * Then a new slide can be inserted from the client with a valid and unique id.
         *
         * @param {String} slideId
         *  A used ID of a slide.
         */
        function updateSlideIdCounter(slideId) {

            var // resulting array of the regular expression
                matches = /(\d+)$/.exec(slideId),
                // the number value at the end of the id
                number = 0;

            if (_.isArray(matches)) {
                number = parseInt(matches[1], 10);

                if (number >= mainSlideCounter) {
                    mainSlideCounter = number + 1;
                }
            }
        }

        /**
         * Adding all slides below a specified node into the presentation slide model.
         *
         * @param {jQuery} node
         *  The jQuerified container node for the slides.
         *
         * @param {String} selector
         *  The css selector to find the slides.
         *
         * @param {String} type
         *  The slide type.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.documentReloaded=false]
         *   Whether the document was reloaded with a new snapshot.
         */
        function addAllSlidesOfLayerIntoSlideModel(node, selector, type, options) {

            var // whether the document was reloaded with a snapshot
                documentReloaded = Utils.getBooleanOption(options, 'documentReloaded', false);

            _.each(node.find(selector), function (slideNode) {

                var // the jQuerified slide node
                    slide = $(slideNode),
                    // the slide id
                    slideId = slide.attr('data-container-id'),
                    // the target of the slide (set as attribute during fast load)
                    target = slide.attr('target'),
                    // the slide index
                    index = -1,
                    // the slide attributes (reading from data object)
                    slideAttrs = slide.data('attributes'),
                    // the supported attributes at a slide node
                    supportedAttrs = null;

                // Info: When loading with fast load, the list style attributes defined at slides, are assigned
                //       to the slide.data('attributes') object. This will not be the case, if the document
                //       is loaded without fast load, because 'listStyles' is not a supported family for the
                //       slide attributes.
                if (!documentReloaded && slideAttrs.listStyles) {
                    supportedAttrs = _.copy(slideAttrs, true);
                    delete supportedAttrs.listStyles;
                    if (_.isEmpty(supportedAttrs)) {
                        slide.removeData('attributes');
                    } else {
                        slide.data('attributes', supportedAttrs);
                    }
                }

                if (type === self.getStandardSlideType()) {
                    updateSlideIdCounter(slideId);
                } else if (type === self.getLayoutSlideType()) {
                    index = self.getLastPositionInModelBehindMaster(target);
                }

                if (documentReloaded) { target = slide.data('target'); }

                slide.removeAttr('target'); // removing fast load specific attribute that is not updated by the model

                addIntoSlideModel(slide, slideId, { type: type, target: target, attrs: slideAttrs, index: index, documentReloaded: documentReloaded });
            });
        }

        /**
         * The handler function for the event 'change:defaults'. Within the object
         * changedAttributes the default text list styles are defined, that will be
         * used in non-placeholder drawings, like text frames or shapes.
         *
         * @param {jQuery.Event} event
         *  The 'change:defaults' event.
         *
         * @param {Object} newAttributes
         *  An object with the new attributes (will be ignored).
         *
         * @param {Object} oldAttributes
         *  An object with the old attributes (will be ignored).
         *
         * @param {Object} changedAttributes
         *  An object with the changed attributes.
         */
        function registerDefaultTextListStyles(event, newAttributes, oldAttributes, changedAttributes) {

            if (changedAttributes && changedAttributes.defaultTextListStyles) {
                self.setDefaultTextListStyles(changedAttributes.defaultTextListStyles);
            }
        }

        /**
         * Receiving the layer node for the layout slides. If it does not exist, it is created.
         *
         * @returns {jQuery}
         *  The layout slide layer node
         */
        function getOrCreateLayoutSlideLayerNode() {

            // is there already a layout slide layer node
            if (layoutSlideLayerNode) { return layoutSlideLayerNode; }

            // creating the new node
            layoutSlideLayerNode = $('<div>').addClass(DOM.LAYOUTSLIDELAYER_CLASS);

            // inserting the new node before the page content node
            layoutSlideLayerNode.insertBefore(DOM.getPageContentNode(pagediv));

            return layoutSlideLayerNode;
        }

        /**
         * Receiving the layer node for the master slides. If it does not exist, it is created.
         *
         * @returns {jQuery}
         *  The master slide layer node
         */
        function getOrCreateMasterSlideLayerNode() {

            // is there already a layout slide layer node
            if (masterSlideLayerNode) { return masterSlideLayerNode; }

            masterSlideLayerNode = $('<div>').addClass(DOM.MASTERSLIDELAYER_CLASS);

            // inserting the new node before the page content node
            masterSlideLayerNode.insertBefore(DOM.getPageContentNode(pagediv));

            return masterSlideLayerNode;
        }

        /**
         * Provides the standard slide node corresponding to the specified id string.
         *
         * @param {String} id
         *  The slide id.
         *
         * @returns {jQuery|null}
         *  The standard slide node. Or null, if no standard slide node with the specified id exists.
         */
        function getStandardSlide(id) {
            return allStandardSlides[id] || null;
        }

        /**
         * Provides the layout slide node in the layout slide layer corresponding to the specified
         * id string.
         *
         * @param {String} id
         *  The slide id.
         *
         * @returns {jQuery|null}
         *  The slide node inside the layout layer node. Or null, if no layout slide node with the
         *  specified id exists.
         */
        function getLayoutSlide(id) {
            return allLayoutSlides[id] || null;
        }

        /**
         * Provides the master slide node in the master slide layer corresponding to the specified
         * id string.
         *
         * @param {String} id
         *  The id string.
         *
         * @returns {jQuery|null}
         *  The slide node inside the master layer node. Or null, if no master slide node with the
         *  specified id exists.
         */
        function getMasterSlide(id) {
            return allMasterSlides[id] || null;
        }

        /**
         * Getting the last top level position inside the slide, that has the current
         * selection.
         * Info: this function is DEPRECATED. Use getNextAvailablePositionInSlide().
         *
         * @returns {Array<Number>}
         *  The next available position at the root of the slide.
         */
        function getNextAvailablePositionInActiveSlide() {

            var // the active slide
                slide = self.getSlideById(activeSlideId),
                // the position of the active slide
                pos = self.getActiveSlidePosition();

            pos.push(slide.children(DrawingFrame.NODE_SELECTOR).length); // only drawings are relevant children

            return pos;
        }

        /**
         * Getting the last top level position inside any slide (master, layout or standard). If id is passed,
         * gets the position in slide with that id, otherwise in active slide.
         * Behaves as more general version of getNextAvailablePositionInActiveSlide() function.
         *
         * @param {String} [id]
         *  The slide id. If not specified, the active slide is used.
         *
         * @returns {Array<Number>}
         *  The next available position at the root of the slide.
         */
        function getNextAvailablePositionInSlide(id) {

            var // the slide ID
                localId = id || activeSlideId,
                // the slide node
                slide = self.getSlideById(id),
                // the logical position
                pos = self.isLayoutOrMasterId(localId) ? [0] : [_.indexOf(slideOrder, localId)];

            pos.push(slide.children(DrawingFrame.NODE_SELECTOR).length); // only drawings are relevant children

            return pos;
        }

        /**
         * Provides the content node of the layout slide node or master slide node in the layout slide
         * layer or master slide layer corresponding to the specified target string. This node can be
         * used for calculations of logical positions (not the slide node itself).
         *
         * @param {String} id
         *  The slide id.
         *
         * @returns {jQuery|null}
         *  The slide content node inside the layout or master slide layer node. This content node is
         *  the parent of the slide node. Or null, if no slide with the specified target exists.
         */
        function getLayoutOrMasterSlideContentNode(id) {

            var // the slide node, that is the child of the content node
                slide = self.getLayoutOrMasterSlide(id);

            return slide ? slide.parent() : null;
        }

        /**
         * Checking, whether a specified id is a valid and registered slide id.
         *
         * @param {String} id
         *  The slide id to be checked.
         *
         * @returns {Boolean}
         *  Whether the specified id is a valid value for a slide id.
         */
        function isValidSlideId(id) {
            return _.isString(idTypeConnection[id]);
        }

        /**
         * Forcing the update of the slides, that have a dependency to a specified target. The
         * target must be defined inside the data object. The event 'change:slidelayout' is
         * triggered to inform all listeners.
         *
         * @param {Object} data
         *  This object is generated inside the function 'savePlaceHolderAttributes' in the
         *  drawing styles. The object's keys are: 'target', 'placeHolderType', 'placeHolderIndex'
         *  and 'attrs'.
         */
        function triggerLayoutChangeEvent(data) {
            self.trigger('change:slidelayout', data);
        }

        /**
         * This function is called, if a node received new attributes. If this node is inside
         * a layout or master slide, it is necessary to update the collector for the attributes
         * of the drawings that are located inside a layout or master slide (the model is the object
         * 'placeHolderCollector'.
         *
         * Because this function is triggered after (!) the drawing received the new attributes,
         * these new attributes are already assigned to the drawing and saved in the data object
         * at the drawing node.
         *
         * Additionally all slides, that have the affected layout or master slide as target,
         * need to be repainted (or invalidated?). Therefore this function itself triggers the
         * event 'change:slidelayout'.
         *
         * @param {HTMLElement|jQuery} element
         *  The element node, that received the new attributes.
         *
         * @param {String} target
         *  The active target for the specified element.
         *
         * @param {String} family
         *  The family of the attribute set.
         *
         * @param {Object} attributes
         *  A map with formatting attribute values, mapped by the attribute
         *  names, and by attribute family names.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.immediateUpdate=true]
         *      Whether slides that are dependent from the layout or master slide change
         *      shall be updated immediately. Default is the immediate update.
         *  @param {String} [options.saveOldAttrs=false]
         *      Whether the old attributes shall be saved and returned in the return object.
         *
         * @returns {Object|Null}
         *  If the attributes are registered, an object with the complete registration
         *  information is returned. The object's keys are: 'target', 'placeHolderType',
         *  'placeHolderIndex', 'attrs' and 'oldAttrs'. If the attributes were not registered,
         *  because target, type or index are not defined, null is returned.
         */
        function registerPlaceHolderAttributes(element, target, family, attrs, options) {

            var // the drawing node containing the specified element
                drawing = null,
                // a paragraph node
                paragraph = null,
                // the explicit attributes at the drawing
                drawingAttrs = null,
                // the paragraph level
                paraLevel = 0,
                // the key in the list styles corresponding to the paragraph level
                paraKey = null,
                // an object containing the information about the saved values
                registrationObject = null,
                // whether drawing attributes shall be registered
                isDrawing = (family === 'drawing'),
                // whether an update of dependent slide shall be triggered immediately
                immediateUpdate = Utils.getBooleanOption(options, 'immediateUpdate', true),
                // whether an update of dependent slide shall be triggered immediately
                saveOldAttrs = Utils.getBooleanOption(options, 'saveOldAttrs', false);

            if (target && self.isLayoutOrMasterId(target)) {

                drawing = isDrawing ? $(element) : $(element).closest('div.drawing');

                if (drawing.length > 0 && PresentationUtils.isPlaceHolderDrawing(drawing)) {

                    if (isDrawing) {
                        drawingAttrs = attrs;
                    } else {

                        // only registration of character attributes for place holder drawings without complete selection (for example footer)
                        if (_.contains(PresentationModel.NOT_COMPLETE_SELECTION_PLACEHOLDER_TYPES, PresentationUtils.getPlaceHolderDrawingType(drawing))) { return null; }

                        paragraph = (family === 'paragraph') ? $(element) : $(element).closest(DOM.PARAGRAPH_NODE_SELECTOR);
                        paraLevel = paragraphStyles.getParagraphLevel(paragraph);
                        paraKey = paragraphStyles.generateKeyFromLevel(paraLevel);

                        drawingAttrs = { listStyle: {} };
                        drawingAttrs.listStyle[paraKey] = attrs;
                    }

                    registrationObject = self.getDrawingStyles().savePlaceHolderAttributes(target, drawingAttrs, drawing, { saveOldAttrs: saveOldAttrs });

                    if (immediateUpdate && registrationObject) {

                        // informing the listeners that place holder drawing styles were modified and that
                        // all affected drawings (that use the modified place holder type and index at a slide
                        // with the specified target) need to be updated.
                        // Therefore the registrationObject is 'complete', with type and index containing at
                        // least the default values.
                        triggerLayoutChangeEvent(registrationObject);
                    }
                }
            }

            return registrationObject;
        }

        /**
         * Inserting an image as slide background. This function generates the required
         * 'setAttributes' operation.
         *
         * @param {Object} imageHolder
         *  The object containing the url for the image.
         */
        function insertImageForSlideBackground(imageHolder) {

            var // the operation attributes
                attrs = { fill: { type: 'bitmap', color: null } },
                // the new operation
                newOperation = { name: Operations.SET_ATTRIBUTES, start: self.getActiveSlidePosition(), attrs: attrs };

            if (imageHolder.url) {
                attrs.fill.imageUrl = imageHolder.url;
            } else {
                attrs.fill.imageUrl = imageHolder;
            }

            self.extendPropertiesWithTarget(newOperation, self.getActiveTarget());
            self.applyOperations(newOperation);
        }

        /**
         * Getting the index for a master or a layout slide that can be used in operations to set the
         * position of a master or layout slide. For a layout slide this is a 0-based index, that describe
         * the position of a layout slide relative to its master slide.
         * For a master slide this is the 0-based index relative to all other master slides.
         * This index is different from the index in the global container 'masterSlideOrder' that contains
         * the sorted IDs of master and layout slides.
         *
         * @param {String} id
         *  The ID of the master or layout slide.
         *
         * @returns {Number}
         *  The 0-based index of the specified slide. If the index cannot be determined, -1 is returned.
         */
        function getOperationIndexForSlideById(id) {

            var // the index for a master or layout slide that describes its position in an operation
                index = -1;

            if (self.isMasterSlideId(id)) {

                index = self.getMasterSlideStartPosition(id);

            } else if (self.isLayoutSlideId(id)) {

                index = self.getLayoutSlideStartPosition(id);

            }

            return index;
        }

        /**
         * Function that is executed in the start handler of the selection box.
         *
         * @returns {Boolean}
         *  Whether the selection box can continue. If 'false' is returned, the selection box
         *  will be cancelled.
         */
        function selectionBoxStartHandler() {

            if (!isMasterView && self.isEmptySlideView()) {
                // Clicking on content root node creates a new slide, no selection box required
                self.insertSlide();
                return false;
            } else {
                self.getSelection().selectionStartHandler();
            }

            return true;
        }

        // private listener functions -----------------------------------------

        /**
         * Function that is executed after the 'import success' event of the application.
         */
        function importSuccessHandler() {

            // no functionality of page layout required
            self.deletePageLayout();
            // no functionality of drawing layer required
            self.disableDrawingLayer();
            // deleting place holder for header and footer
            DOM.getMarginalTemplateNode(pagediv).remove();

            // set the local slide pane width to the width previously saved in the document
            localViewSlidePaneWidth = self.getSlidePaneWidthInDocAttributes();

            // the active slide was already set in updateDocumentFormatting -> now the slide can be selected
            self.getSelection().setSlideSelection();

            // setting the handler function for updating lists after the document is imported successfully
            self.setUpdateListsDebounced(self.getDebouncedUpdateListsHandler());

            // registering the selection box handler at the application content root
            selectionBox = new SelectionBox(app, app.getView().getContentRootNode());
            selectionBox.registerSelectionBoxMode(PresentationModel.SET_SELECTION_MODE, { contentRootClassActive: PresentationModel.SET_SELECTION_MODE, selectionFilter: PresentationModel.MULTI_SELECTION_BOX_FILTER, startHandler: selectionBoxStartHandler, stopHandler: self.getSelection().selectAllDrawingsInSelectionBox });
            selectionBox.setActiveMode(PresentationModel.SET_SELECTION_MODE);

            // registering the vertical scroll handler to switch between the slides (not on touch devices)
            if (!Utils.TOUCHDEVICE) { app.getView().enableVerticalScrollBar(); }

            // register the listener for 'change:slidelayout'.
            // -> this update shall not be triggered during document load.
            self.listenTo(self, 'change:slidelayout', updateDrawingLayoutFormatting);

            // register the listener for 'change:activeView'.
            // -> this update shall not be triggered during document load.
            self.listenTo(self, 'change:activeView:after', updateActiveViewHandler);

            // register the listener for 'operations:after'.
            // -> this update shall not be triggered during document load.
            self.listenTo(self, 'operations:after', getOperationAfterHandler());

            // register the listener to handle the visibility of the page node
            // -> this node must be made invisible after removing the last document slide
            self.listenTo(self, 'removed:slide inserted:slide change:activeView', pageVisibilityHandler);

            // Creating handler for full update of changed slides
            // -> this can be used for example after changing the layout of a slide
            implSlideChanged = getDebouncedUpdateSlideHandler();

            // Creating the handler that informs the slide pane about changes (not handled by 'operations:after')
            debouncedUpdateSlideStateHandler = getDebouncedUpdateSlideStateHandler();

            // trigger asynchronously an event to initialize the SlidePane, after the controller updated the GUI
            app.getController().one('change:items', function () {
                self.executeDelayed(function () {
                    self.trigger('slideModel:init', { isMasterView: isMasterView, activeSlideId: self.getActiveSlideId(), width: localViewSlidePaneWidth, slideRatio: self.getSlideRatio() });
                }, 100, 'Presentation: Trigger slideModel:init');
            });

            // handling selection change events
            selection.on('change', selectionChangeHandler);
        }

        /**
         * Updating the model for the presentation application. This is necessary, after the document
         * was loaded with fast load or from local storage. The latter is not supported yet.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.deleteModel=false]
         *   Whether the existing model needs to be deleting before updating it.
         *  @param {Boolean} [options.documentReloaded=false]
         *   Whether the document was reloaded with a new snapshot.
         */
        function updatePresentationModel(options) {

            var // the drawing styles object
                drawingStyles = self.getDrawingStyles(),
                // whether the existing model needs to be removed
                deleteExistingModel = Utils.getBooleanOption(options, 'deleteModel', false),
                // whether the document was reloaded (snapshot was applied)
                documentReloaded = Utils.getBooleanOption(options, 'documentReloaded', false);

             // deleting the model is required, if a new snapshot is applied
            if (deleteExistingModel) { emptyPresentationModel(); }

            // setting the master and layout layer node
            layoutSlideLayerNode = pagediv.find('>' + DOM.LAYOUTSLIDELAYER_SELECTOR);
            masterSlideLayerNode = pagediv.find('>' + DOM.MASTERSLIDELAYER_SELECTOR);

            // filling the slide model
            addAllSlidesOfLayerIntoSlideModel(masterSlideLayerNode, '>' + DOM.SLIDECONTAINER_SELECTOR + '>' + DOM.SLIDE_SELECTOR, self.getMasterSlideType(), { documentReloaded: documentReloaded });
            addAllSlidesOfLayerIntoSlideModel(layoutSlideLayerNode, '>' + DOM.SLIDECONTAINER_SELECTOR + '>' + DOM.SLIDE_SELECTOR, self.getLayoutSlideType(), { documentReloaded: documentReloaded });
            addAllSlidesOfLayerIntoSlideModel(pagediv, '>' + DOM.PAGECONTENT_NODE_SELECTOR + '>' + DOM.SLIDE_SELECTOR, self.getStandardSlideType(), { documentReloaded: documentReloaded });

            // filling the drawing place holder model
            _.each(pagediv.find('.drawing'), function (drawing) {

                var // the target attribute of the drawing
                    target = null;

                drawing = $(drawing);
                target = drawing.attr('target');  // this attribute is set in fast load

                // removing the attributes from fast load, because they are not updated by the model
                drawing.removeAttr('target');

                // trying to get target also after loading using snapshot (load without fast load (and unit test))
                if (!target && deleteExistingModel && PresentationUtils.isPlaceHolderDrawing(drawing)) {
                    // getting the slide of this drawing to determine the target (if it is a master or layout slide)
                    target = self.getSlideId(drawing.closest(DOM.SLIDE_SELECTOR));
                    if (target && !self.isLayoutOrMasterId(target)) { target = null; }
                }

                if (target) {
                    drawingStyles.savePlaceHolderAttributes(target, drawing.data('attributes'), drawing);
                }
            });
        }

        /**
         * Presentation specific handler for the selection 'change' event.
         */
        function selectionChangeHandler() {
            self.handleListItemVisibility(); // handling the visibility of the list items in empty paragraphs
        }

        /**
         * Listener to the events 'undo:after' and 'redo:after'.
         *
         * @param {jQuery.Event} event
         *  A jQuery 'undo:after' and 'redo:after' event object.
         *
         * @param {Object[]} operations
         *  An array containing the undo or redo operations.
         */
        function undoRedoAfterHandler(event, operations) {

            var // finding first insert slide operation
                handleOperation = null,
                // the ID of the inserted slide
                slideId = null,
                // whether the new inserted slide is a layout or master slide
                isLayoutMasterSlide = false,
                // whether a change of the active view is required
                changeActiveView = false;

            // helper function to activate the slide, that is affected by the operation
            function switchToSlide(id) {
                isLayoutMasterSlide = self.isLayoutOrMasterId(id);
                changeActiveView = (isLayoutMasterSlide !== isMasterView);  // change of view required?
                if (changeActiveView) { setActiveView(isLayoutMasterSlide); }  // activating the new view, if required
                self.setActiveSlideId(id); // finally activating the new inserted slide
            }

            // handling insert slide operations
            function handleInsertSlideOperation() {
                slideId = getSlideIdFromOperation(handleOperation); // getting the slide ID from the operation
                switchToSlide(slideId);
            }

            // handling insert slide operations
            function handleDeleteSlideOperation() {
                isLayoutMasterSlide = _.isString(handleOperation.target);
                changeActiveView = (isLayoutMasterSlide !== isMasterView);  // change of view required?
                if (changeActiveView) { setActiveView(isLayoutMasterSlide); } // activating the new view, if required
                self.changeToSlideAtIndex(handleOperation.start[0]); // activating the slide in the correct view
            }

            // handling group/ungroup operations
            function handleGroupOperations() {
                var allGroupingPositions = []; // collector for all logical positions of drawings that will be selected
                slideId = getSlideIdFromOperation(handleOperation[0]); // getting the slide ID from the operation
                if (slideId !== self.getActiveSlideId()) { switchToSlide(slideId); }

                // there can only be more than one group and more than one ungroup operation
                if (handleOperation[0].name === Operations.GROUP) {
                    selection.clearMultiSelection(); // clearing the existing multi selection
                    if (handleOperation.length === 1) {
                        selection.setTextSelection(handleOperation[0].start, Position.increaseLastIndex(handleOperation[0].start)); // selecting the grouped drawing
                    } else {
                        // undo of several combined 'ungroup' operations
                        _.each(handleOperation, function (op) { allGroupingPositions.push(op.start); });
                        selection.setMultiDrawingSelectionByPosition(allGroupingPositions); // selecting all grouped drawings
                    }
                } else {
                    _.each(handleOperation, function (op) {
                        var slidePos = _.initial(op.start);
                        var fullDrawingPos = _.map(op.drawings, function (index) {
                            var newPos = _.clone(slidePos);
                            newPos.push(index);
                            return newPos;
                        });
                        allGroupingPositions = allGroupingPositions.concat(fullDrawingPos);
                    });
                    selection.setMultiDrawingSelectionByPosition(allGroupingPositions); // selecting all ungrouped drawings
                }
            }

            // handling move operations
            function handleMoveOperations() {
                // changing the active slide and selecting all moved drawings
                var allDrawingPositions = []; // collector for all logical positions of drawings that will be selected
                slideId = getSlideIdFromOperation(handleOperation[0]); // getting the slide ID from the operation
                if (slideId !== self.getActiveSlideId()) { switchToSlide(slideId); }

                if (handleOperation.length === 1) {
                    selection.setTextSelection(handleOperation[0].to, Position.increaseLastIndex(handleOperation[0].to)); // selecting the one drawing
                } else {
                    // undo of several combined 'ungroup' operations
                    _.each(handleOperation, function (op) { allDrawingPositions.push(op.to); });
                    selection.setMultiDrawingSelectionByPosition(allDrawingPositions); // selecting all moved drawings
                }
            }

            // handling insertDrawing operations (without any insertSlide operation)
            function handleInsertDrawingOperations() {

                var allDrawingPositions = []; // collector for all logical positions of drawings that will be selected
                slideId = getSlideIdFromOperation(handleOperation[0]); // getting the slide ID from the operation
                if (slideId !== self.getActiveSlideId()) { switchToSlide(slideId); }

                // -> setting multi selections, if more than one drawing was inserted
                if (handleOperation.length > 1) {
                    _.each(handleOperation, function (op) {
                        if (op.start.length === 2) { allDrawingPositions.push(op.start); } // collecting all top level drawings, no grouped drawings
                    });

                    if (allDrawingPositions.length > 0) {
                        self.getDrawingStyles().handleAllEmptyPlaceHolderDrawingsOnSlide(self.getSlideById(slideId)); // taking care of empty place holder drawings
                        selection.setMultiDrawingSelectionByPosition(allDrawingPositions); // selecting all inserted drawings
                    }
                }
            }

            // checking for special operations in the undo group
            // finding first insert slide operation
            handleOperation = _.find(operations, function (operation) { return _.contains(PresentationModel.INSERT_SLIDE_OPERATIONS, operation.name); });
            if (handleOperation) {
                handleInsertSlideOperation();
            } else {
                // finding first delete operation of a slide
                handleOperation = _.find(operations, function (operation) { return operation.name === Operations.DELETE && operation.start.length === 1; });
                if (handleOperation) {
                    handleDeleteSlideOperation();
                } else {
                    // finding group/ungroup operations
                    handleOperation = _.filter(operations, function (operation) { return (operation.name === Operations.GROUP || operation.name === Operations.UNGROUP); });
                    if (handleOperation && handleOperation.length > 0) {
                        handleGroupOperations();
                    } else {
                        // finding move operations
                        handleOperation = _.filter(operations, function (operation) { return (operation.name === Operations.MOVE); });
                        if (handleOperation && handleOperation.length > 0) {
                            handleMoveOperations();
                        } else {
                            // finding insertDrawing operations (without insertSlide)
                            handleOperation = _.filter(operations, function (operation) { return operation.name === Operations.INSERT_DRAWING; });
                            if (handleOperation && handleOperation.length > 0) {
                                handleInsertDrawingOperations();
                            }
                        }
                    }
                }
            }

        }

        /**
         * Listener function for the event 'change:slidelayout'. This function is called after modifications
         * of layout or master slide. Then it is necessary to update all slides, that use the modified slide
         * as target (or whose target uses the modified slide as target).
         *
         * Info: This function is critical for the performance. It might be necessary to update
         *       many elements. Therefore this should be done asynchronous (TODO).
         *
         * -> TODO: Shifting this into another class?
         *
         * @param {jQuery.Event} event
         *  The 'drawingUpdate:after' event
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.target='']
         *   The target of the layout or master slide that was modified
         *  @param {String} [options.placeHolderType='']
         *   The place holder type that was modified.
         *  @param {Number} [options.placeHolderIndex=null]
         *   The place holder type that was modified.
         *  @param {Object} [options.attrs=null]
         *   The set of modified attributes.
         */
        function updateDrawingLayoutFormatting(event, options) {

            var // the target of the modified master or layout slide
                target = Utils.getStringOption(options, 'target', ''),
                // the drawing styles object
                drawingStyles = self.getDrawingStyles(),
               // a container with the IDs of all slides affected by the modification of the master/layout slide
                allSlideIDs = self.getAllSlideIDsWithSpecificTargetAncestor(target),
                // the type of place holder that was modified
                type = allSlideIDs.length > 0 ? Utils.getStringOption(options, 'placeHolderType', '') : null,
                // the type of place holder that was modified
                index = allSlideIDs.length > 0 ? Utils.getNumberOption(options, 'placeHolderIndex', null) : null,
                // the changed attributes
                attrs = allSlideIDs.length > 0 ? Utils.getObjectOption(options, 'attrs', null) : null,
                // whether the modification was done on the master slide (-> update should be independent from index?)
                isMasterSlideChange = target && self.isMasterSlideId(target);

            // updating attributes and/or list styles of one drawing
            function updateOneDrawing(drawing) {

                // updating the drawing itself, if required
                if (attrs.drawing) { drawingStyles.updateElementFormatting($(drawing)); }

                // updating the paragraphs, if required
                if (attrs.listStyle) {
                    _.each($(drawing).find(DOM.PARAGRAPH_NODE_SELECTOR), function (para) {
                        // updating all paragraphs, that have a modified attribute set
                        if (paragraphStyles.generateKeyFromLevel(paragraphStyles.getParagraphLevel(para)) in attrs.listStyle) {
                            paragraphStyles.updateElementFormatting($(para));
                            self.updateListsDebounced($(para), { singleParagraph: true }); // only list update for the specified paragraph
                        }
                    });
                }
                self.updateDynFontSizeDebounced($(drawing));
            }

            if (target && type && _.isNumber(index) && attrs) {

                _.each(allSlideIDs, function (id) {

                    var // the list of all indizes that will be updated
                        allIndices = null,
                        // an object containing the conversion for place holder drawings on master slides
                        // -> 'ctrTitle' is 'title' on master, 'subTitle' is 'body' on master
                        masterTypes = _.invert(drawingStyles.getMasterSlideTypeConverter()),
                        // a type array, if more than one type needs to be adapted
                        typeArray = null;

                    if (isMasterSlideChange && masterTypes[type]) {

                        // not only searching for 'body', but also for 'subTitle' place holders
                        typeArray = [type, masterTypes[type]];

                        // updating all drawings, that have the specified place holder type
                        // -> using options.baseAttributes for the place holder attributes from layout slide and master slide
                        _.each(drawingStyles.getAllPlaceHolderDrawingsOnSlideBySelector(id, typeArray), function (drawing) {
                            updateOneDrawing(drawing);
                        });

                    } else {

                        allIndices = [index];

                        // changes in the master slide affect all 'unassigned placeholder' drawings
                        if (isMasterSlideChange) { allIndices.push(self.getDrawingStyles().getMsDefaultPlaceHolderIndex()); }

                        _.each(allIndices, function (oneIndex) {

                            // updating all drawings, that have the specified place holder type
                            // -> using options.baseAttributes for the place holder attributes from layout slide and master slide
                            _.each(drawingStyles.getAllPlaceHolderDrawingsOnSlideBySelector(id, type, oneIndex), function (drawing) {
                                updateOneDrawing(drawing);
                            });
                        });
                    }
                });
            }
        }

        /**
         * Listener function for the event 'change:activeView:after'. This function is called when the active view
         * (model) is changed, or to put it simply, when the view is switched between the 'normal' slides and
         * the master/layout slides. After that it is necessary to activate the appropriate slide in the view.
         *
         * @param {jQuery.Event} event
         *  The 'change:activeView:after' event
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.isMasterView=false]
         *   Whether the masterView was activated.
         */
        function updateActiveViewHandler(event, options) {

            var // the id of the slide that will be activated
                newSlideId = null,
                // whether the master/layout view will be activated or not
                showMaster = Utils.getBooleanOption(options, 'isMasterView', false);

            // activating a slide in the view
            if (showMaster) {
                lastActiveStandardSlideId = activeSlideId;
                newSlideId = self.getLayoutOrMasterSlideId(activeSlideId);
            } else {
                newSlideId = lastActiveStandardSlideId;
            }

            if (newSlideId === '' && !showMaster) {
                setActiveSlideId('', { forceUpdate: true }); // allowing the empty string for 'newSlideId', if this is not the master view
            } else {
                setActiveSlideId(newSlideId || self.getIdOfFirstSlideOfActiveView(), { forceUpdate: true });
            }
        }

        /**
         * Listener function for the event 'document:reloaded', that is triggered, if the user cancels a long
         * running action.
         */
        function documentReloadHandler() {
            // deleting and updating the complete presentation model
            updatePresentationModel({ deleteModel: true, documentReloaded: true });
        }

        /**
         * Handler for the event 'image:loaded'. It is possible, that the slide pane
         * needs to be informed about a loaded image (44971).
         *
         * @param {jQuery.Event} event
         *  The 'image:loaded' event
         *
         * @param {Node|jQuery} node
         *  The image drawing node.
         */
        function imageLoadedHandler(event, node) {

            var // the slide node
                slideNode = $(node).parent(),
                // the slide ID
                slideId = null;

            if (DOM.isSlideNode(slideNode)) {
                slideId = self.getSlideId(slideNode);
                if (!slideFormatManager.isUnformattedSlide(slideId)) {
                    debouncedUpdateSlideStateHandler(slideId); // registering the ID
                    // TODO: Also handling master/layout slide dependencies (?)
                }
            }

        }

//        /**
//         * Updating the top value of a drawing node. This is necessary, if a drawing node has its vertical
//         * text content aligned to bottom or centered and if this drawing has 'autoResizeHeight' set to true.
//         * In that case, the drawing needs to grow in top direction.
//         * -> Problem: Can this happen without an operation?
//         *
//         * @param {jQuery.Event} event
//         *  The 'drawingUpdate:after' event.
//         *
//         * @param {jQuery} node
//         *  The drawing node.
//         */
//        function updateVerticalDrawingPosition(event, node) {
//
//            var // the text frame node inside the drawing node
//                textFrame = null,
//                // the vertical text aligment property
//                verticalAlign = null,
//                // the height saved at the node
//                savedHeight = null,
//                // the new height of the drawing node
//                newHeight = null,
//                // the difference of old and new height
//                difference = null,
//                // the new value for the top offset
//                newTopOffset = null;
//
//            if (DrawingFrame.isAutoResizeHeightDrawingFrame(node)) {
//
//                // the text frame node inside the drawing node
//                textFrame = DrawingFrame.getTextFrameNode(node);
//                // the vertical text alignment property
//                verticalAlign = (textFrame && textFrame.length > 0) ? textFrame.attr(DrawingFrame.VERTICAL_ALIGNMENT_ATTRIBUTE) : '';
//
//                if (verticalAlign && (verticalAlign === 'bottom' || verticalAlign === 'centered')) {
//
//                    if (!_.isString(node.attr('verticalTop'))) { saveVerticalPositionAttributes(null, node); }
//
//                    savedHeight = Utils.getElementAttributeAsInteger(node, 'verticalHeight', 0);
//                    newHeight = node.height();
//
//                    if (savedHeight !== newHeight) {
//                        difference = savedHeight - newHeight;
//                        if (verticalAlign === 'centered') { difference = Utils.round(difference / 2, 1); }
//                        newTopOffset = Utils.getElementAttributeAsInteger(node, 'verticalTop', 0) + difference;
//                        node.css('top', newTopOffset + 'px');
//                        node.attr({ verticalTop: newTopOffset, verticalHeight: newHeight });
//                    }
//                }
//            }
//        }
//
//        /**
//         * Saving the vertical position and height of the text frame. This values can be used
//         * to calculate a possible vertical movement 'upwards' of the text frame. This is only
//         * necessary, if the vertical text alignment is set to 'bottom' or 'centered'.
//         * This values need to be set for example from 'handleAdditionalTextFrameSelection'
//         * -> never overwrite existing values
//         *
//         * @param {jQuery.Event} event
//         *  The 'drawingUpdate:after' event.
//         *
//         * @param {jQuery} node
//         *  The drawing node.
//         */
//        function saveVerticalPositionAttributes(event, node) {
//            if (DrawingFrame.isAutoResizeHeightDrawingFrame(node) && !_.isNumber(node.attr('verticalTop'))) {
//                node.attr({ verticalTop: Utils.convertCssLength($(node).css('top'), 'px', 1), verticalHeight: node.height() });
//            }
//        }

        /**
         * This function returns the ID of that slide, that was modified
         * by the specified operation.
         *
         * @param {Object} operation
         *  The object containing one operation.
         *
         * @returns {String}
         *  The ID of the slide, that was modified by the specified operation.
         *  If this ID cannot be determined, null is returned.
         */
        function getSlideIdFromOperation(operation) {

            var // the slide ID of the slide affected by this operation
                slideID = null,
                // the name of the operation
                name = operation.name;

            if (operation.id && _.contains([Operations.LAYOUT_SLIDE_INSERT, Operations.MASTER_SLIDE_INSERT], name)) {

                slideID = operation.id;

            } else if (operation.target && name !== Operations.SLIDE_INSERT) {

                // handling for content of master and layout slides -> the target is already the ID
                slideID = operation.target;

            } else {

                // handling for document slides -> the ID  is saved in the sorted 'slideOrder' collector
                // -> the start position of the operation can be used to determine the ID of the slide.
                if (operation.start) {
                    slideID = slideOrder[_.first(operation.start)];
                }
            }

            return slideID;
        }

        /**
         * Setting the visibility of the page node. This node must be made invisible,
         * if the last document nodes was deleted.
         *
         * @returns {Boolean}
         *  Whether the visibility of the page node has changed.
         */
        function pageVisibilityHandler() {

            // whether the visibility was changed
            var visibilityChanged = false;
            // whether the current visible view is empty
            var isEmptyView = self.isEmptySlideView();
            // whether the page node is hidden
            var isHiddenPage = pagediv.hasClass('hiddenpage');

            if (isEmptyView && !isHiddenPage) {
                pagediv.addClass('hiddenpage'); // do hide the page
                visibilityChanged = true;
            } else if (isHiddenPage && !isEmptyView) {
                pagediv.removeClass('hiddenpage'); // do hide the page
                visibilityChanged = true;
            }

            return visibilityChanged;
        }

        /**
         * A helper function to create the debounced handler for 'operations:after' event.
         * It contains a direct callback that collects modified slides and an asynchronous
         * callback that can be used to update the content of the slide pane.
         */
        function getOperationAfterHandler() {
            var
                // a copy of the collector for the event
                collectorCopy = null,

                // a copy of the collector of all indirectly affected normal slides for the event
                indirectSlidesCopy = null,

                // collecting all slide IDs for those operations that had attribute 'followMasterShapes'
                allFollowMasterShapeIDs = {},

                // a copy of the collector for the master shape following slides
                followMasterShapeCopy = null,

                // whether only document slides were modified, no layout or master slides
                onlyDocumentSlides = true,

                // a data object that will be written as soon as a document attribute operation takes place
                // that also features a `page` attribute and its `orientation` key.
                slideRatioChangeData = null;

            // This direct callback is triggered after every operation and therefore
            // very performance critical. It is used to collect the IDs of the modified
            // slides.
            // First parameter: The 'operations:after' event.
            // Second parameter: An array of operations.
            function registerSlide(event, operations) {

                if (operations && operations.length > 0) {
                    _.each(operations, function (op) {

                        var // the ID of the slide modified by the specified operation
                            slideID = getSlideIdFromOperation(op);

                        if (slideID && !_.contains(allModifiedSlideIDs, slideID)) { allModifiedSlideIDs.push(slideID); }

                        // check, whether this is a master or layout slide
                        if (self.isLayoutOrMasterId(slideID)) { onlyDocumentSlides = false; }

                        // special handling for 'followMasterShapes'
                        // -> collecting all attribute values, if this property was switched
                        if (op.name === Operations.SET_ATTRIBUTES) {
                            if (op.attrs && op.attrs.slide) {
                                if (op.attrs.slide.followMasterShapes === null) {

                                    allFollowMasterShapeIDs[slideID] = true;

                                } else if (_.isBoolean(op.attrs.slide.followMasterShapes)) {

                                    allFollowMasterShapeIDs[slideID] = op.attrs.slide.followMasterShapes;
                                }
                            }
                        } else if (op.name === Operations.SET_DOCUMENT_ATTRIBUTES) {
                            if (op.attrs && op.attrs.page && op.attrs.page.orientation) {
                                // attrs: {
                                //     page: {
                                //         width:        33866,
                                //         height:       19050,
                                //         orientation:  "landscape"
                                //     },
                                //     name: "setDocumentAttributes"
                                // }
                                slideRatioChangeData = {
                                    slideRatio: (op.attrs.page.width / op.attrs.page.height)
                                };
                            }
                        }
                    });
                }
            }

            // this indirect callback might update a large number of modified drawings
            // and is therefore also very performance critical.
            function updateSlidePreview() {

                var // whether slide IDs might be duplicated
                    avoidDuplicates = false,
                    // a collector for the indirectly affected normal slides
                    indirectStandardSlides = [];

                if (!onlyDocumentSlides && allModifiedSlideIDs.length > 0) {

                    // layout or master slides were modified -> update of dependent slides is also required.
                    // Info: In layout view, document slides do not need to be updated. This is done, when
                    //       when switching from layout view to document view.
                    //       -> but a remote client might need an update of the slides in the 'normal' view.
                    _.each(_.copy(allModifiedSlideIDs), function (id) {

                        var // whether the specified slide is a master slide
                            isMasterSlide = self.isMasterSlideId(id),
                            // all slides affected by the change of the layout or master slide with the specified ID
                            allAncestorSlides = isMasterSlide ? self.getAllSlideIDsWithSpecificTargetParent(id) : null;

                        if (allAncestorSlides && allAncestorSlides.length > 0) {
                            allModifiedSlideIDs = allModifiedSlideIDs.concat(allAncestorSlides);
                            avoidDuplicates = true;
                        }
                    });

                    // avoiding duplicates in master and layout slides
                    if (avoidDuplicates) { allModifiedSlideIDs = _.uniq(allModifiedSlideIDs); }

                    // collecting all affected 'normal' slides (these might be visible in a remote client)
                    _.each(allModifiedSlideIDs, function (layoutSlideId) {

                        var // a container for all normal slides that are dependent from the layout slide
                            allStandardSlides = null;

                        if (self.isLayoutSlideId(layoutSlideId)) {
                            allStandardSlides = self.getAllSlideIDsWithSpecificTargetParent(layoutSlideId);
                            if (allStandardSlides.length > 0) {
                                indirectStandardSlides = indirectStandardSlides.concat(allStandardSlides);
                            }
                        }

                        // making the list of IDs unique and sort it
                        if (allStandardSlides && allStandardSlides.length > 0) {
                            indirectStandardSlides = _.uniq(indirectStandardSlides).sort();
                        }
                    });
                }

                // the event gets copies of the collectors
                collectorCopy = _.copy(allModifiedSlideIDs);
                followMasterShapeCopy = _.copy(allFollowMasterShapeIDs);
                indirectSlidesCopy = _.copy(indirectStandardSlides);

                // resetting the containers
                allModifiedSlideIDs = [];
                allFollowMasterShapeIDs = {};

                // If 'onlyDocumentSlides' is set to true, only document slide need to be cloned, no layout or master slide!

                // TODO in listener: Iterating sliced over all slides, whose IDs are collected in 'allModifiedSlideIDs'.
                //                   self.iterateArraySliced(allModifiedSlideIDs, ...) {}
                // TODO in listener: Check, if a specified ID is again in the global collector using the function
                //                   'this.isModifiedSlideID(id)'. In this case the update of the slide in the slide
                //                   pane can be skipped.

                self.trigger('slidestate:update', collectorCopy, followMasterShapeCopy, indirectSlidesCopy, { onlyDocumentSlides: onlyDocumentSlides });

                if (slideRatioChangeData) {
                    self.trigger('slidesize:change', slideRatioChangeData);

                    slideRatioChangeData = null;
                }
                // resetting also the boolean that registers, if only document slide were modified (no master or layout slides).
                onlyDocumentSlides = true;
            }

            // create and return the deferred implTableChanged() method
            return self.createDebouncedMethod(registerSlide, updateSlidePreview, { delay: 500, infoString: 'Presentation: operationAfterHandler' });
        }

        /**
         * Helper function to inform the slide pane about changes, that were NOT
         * handled by 'operations:after' event. This are the insertion of template
         * placeholder texts in layout slides or the load of a (large) image.
         *
         * @returns {Function}
         *  The debounced function for triggering 'slidestate:update' debounced without
         *  'operations:after' event. The usage of this function should be the exception,
         *  the regular update happens via 'operations:after'.
         */
        function getDebouncedUpdateSlideStateHandler() {

            var // all slide nodes that need to be updated
                allSlideIDs = [];

            // direct callback: saving slide IDs
            function registerSlidesForSlidePaneUpdate(slideIDs) {

                if (_.isString(slideIDs)) {
                    allSlideIDs.push(slideIDs);
                } else {
                    allSlideIDs = allSlideIDs.concat(slideIDs);
                }
            }

            // deferred callback: called once, after current script ends
            function triggerSlidePaneUpdate() {

                self.trigger('slidestate:update', _.uniq(allSlideIDs));

                allSlideIDs = []; // reset collector
            }

            // create and return the deferred method
            return self.createDebouncedMethod(registerSlidesForSlidePaneUpdate, triggerSlidePaneUpdate, { delay: 1000, infoString: 'Presentation: debouncedUpdateSlideStateHandler' });
        }

        // public methods -----------------------------------------------------

        /**
         * Public helper function: Adding a slide in the model.
         * See description at private function 'addIntoSlideModel'.
         */
        this.addIntoSlideModel = function (slide, id, options) {
            return addIntoSlideModel(slide, id, options);
        };

        /**
         * Public helper function: Removing a slide from the model.
         * See description at private function 'removeFromSlideModel'.
         */
        this.removeFromSlideModel = function (id) {
            return removeFromSlideModel(id);
        };

        /**
         * Public helper function: Moving a slide inside the model.
         * See description at private function 'moveSlideIdInModel'.
         */
        this.moveSlideIdInModel = function (start, end, isStandardSlide) {
            return moveSlideIdInModel(start, end, isStandardSlide);
        };

        /**
         * Public helper function: Changing the layout id for a specified
         * document slide id.
         * See description at private function 'changeLayoutIdInModel'.
         */
        this.changeLayoutIdInModel = function (id, layoutId) {
            return changeLayoutIdInModel(id, layoutId);
        };

        /**
         * Public helper function: Changing the master id for a specified
         * layout slide id.
         * See description at private function 'changeMasterIdInModel'.
         */
        this.changeMasterIdInModel = function (id, layoutId) {
            return changeMasterIdInModel(id, layoutId);
        };

        /**
         * Public helper function: Provides a unique ID for the standard slides.
         * See description at private function 'getNextSlideID'.
         */
        this.getNextSlideID = function () {
            return getNextSlideID();
        };

        /**
         * Public helper function: Provides a unique ID for the custom layout slides.
         * See description at private function 'getNextCustomLayoutId'.
         */
        this.getNextCustomLayoutId = function () {
            return getNextCustomLayoutId();
        };

        /**
         * Public helper function: Receiving the layer node for the layout slides.
         * If it does not exist, it is created.
         * See description at private function 'getOrCreateLayoutSlideLayerNode'.
         */
        this.getOrCreateLayoutSlideLayerNode = function () {
            return getOrCreateLayoutSlideLayerNode();
        };

        /**
         * Public helper function: Receiving the layer node for the master slides.
         * If it does not exist, it is created.
         * See description at private function 'getOrCreateMasterSlideLayerNode'.
         */
        this.getOrCreateMasterSlideLayerNode = function () {
            return getOrCreateMasterSlideLayerNode();
        };

        /**
         * Receiving the layer node for the layout slides, if it exists.
         *
         * @returns {jQuery|Null}
         *  The layout slide layer node, or null if it does not exist.
         */
        this.getLayoutSlideLayerNode = function () {
            return layoutSlideLayerNode;
        };

        /**
         * Receiving the layer node for the master slides, if it exists.
         *
         * @returns {jQuery|Null}
         *  The master slide layer node, or null if it does not exist.
         */
        this.getMasterSlideLayerNode = function () {
            return masterSlideLayerNode;
        };

        /**
         * Public helper function: Setting the id of the active slide.
         * See description at private function 'setActiveSlideId'.
         */
        this.setActiveSlideId = function (id, options) {
            return setActiveSlideId(id, options);
        };

        /**
         * Public helper function: Getting the last top level position inside the slide,
         * that has the current selection.
         * See description at private function 'getNextAvailablePositionInActiveSlide'.
         */
        this.getNextAvailablePositionInActiveSlide = function () {
            return getNextAvailablePositionInActiveSlide();
        };

        /**
         * Public helper function: Getting the last top level position inside the slide,
         * that has the given id.
         * See description at private function 'getNextAvailablePositionInSlide'.
         */
        this.getNextAvailablePositionInSlide = function (id) {
            return getNextAvailablePositionInSlide(id);
        };

        /**
         * Public helper function: Triggering a full update of a slide. This is for
         * example required after changing the layout of a slide.
         * 'implSlideChanged' is a debounced function, with a direct callback that
         * registers the slides for update.
         *
         * @param {Node|jQuery} slide
         *  The slide node that will be registered for a deferred update.
         */
        this.implSlideChanged = function (slide) {
            return implSlideChanged(slide);
        };

        /**
         * Getting the id of the first slide in the currently active view.
         *
         * @returns {String}
         *  The id of the first slide of the currently active view.
         */
        this.getIdOfFirstSlideOfActiveView = function () {
            return self.getIdOfSlideOfActiveViewAtIndex(0);
        };

        /**
         * Getting the id of the last slide in the currently active view.
         *
         * @returns {String}
         *  The id of the last slide of the currently active view.
         */
        this.getIdOfLastSlideOfActiveView = function () {
            return isMasterView ? _.last(masterSlideOrder) : _.last(slideOrder);
        };

        /**
         * Getting the id of the specified slide in the currently active view. The
         * specification is done by the index of the slide (which is its logical
         * position).
         *
         * @param {Number} index
         *  The zero-based index of the slide inside the currently active view.
         *
         * @returns {String}
         *  The id of the first slide of the currently active view.
         */
        this.getIdOfSlideOfActiveViewAtIndex = function (index) {
            return isMasterView ? masterSlideOrder[index] : slideOrder[index];
        };

        /**
         * Getting the key, that is used to mark standard slides.
         *
         * @returns {String}
         *  The key, that is used to mark standard slides.
         */
        this.getStandardSlideType = function () {
            return PresentationModel.TYPE_SLIDE;
        };

        /**
         * Getting the key, that is used to mark master slides.
         *
         * @returns {String}
         *  The key, that is used to mark master slides.
         */
        this.getMasterSlideType = function () {
            return PresentationModel.TYPE_MASTER_SLIDE;
        };

        /**
         * Getting the key, that is used to mark layout slides.
         *
         * @returns {String}
         *  The key, that is used to mark layout slides.
         */
        this.getLayoutSlideType = function () {
            return PresentationModel.TYPE_LAYOUT_SLIDE;
        };

        /**
         * Getting an object that contains the default drawing attributes
         * for new inserted drawings.
         *
         * @returns {Object}
         *  An object containing the default drawing attributes for new inserted
         *  drawings.
         */
        this.getInsertDrawingAttibutes = function () {
            return _.clone(PresentationModel.INSERT_DRAWING_ATTRIBUTES);
        };

        /**
         * Getting the registered slide styles.
         *
         * @returns {Object}
         *  The container with the registered slide styles.
         */
        this.getSlideStyles = function () {
            return slideStyles;
        };

        /**
         * Getting the slide format manager of the model.
         *
         * @returns {Object}
         *  The slide format manager of the model.
         */
        this.getSlideFormatManager = function () {
            return slideFormatManager;
        };

        /**
         * Getting the id of the currently activated slide.
         *
         * @returns {String}
         *  The id of the currently active slide.
         */
        this.getActiveSlideId = function () {
            return activeSlideId;
        };

        /**
         * Getting the logical position of the active slide
         *
         * @returns {Array<Number>}
         *  The logical position of the active slide in its content container.
         */
        this.getActiveSlidePosition = function () {
            return isMasterView ? [0] : [getActiveSlideIndex()];
        };

        /**
         * Getting the index of the active slide in its current view.
         *
         * @returns {Number}
         *  The index of the active slide in its current view.
         */
        this.getActiveSlideIndex = function () {
            return getActiveSlideIndex();
        };

        /**
         * Getting the index of the layout slide relative to it's master slide.
         *
         * @param {String} masterId
         *  Id of the master slide to whom layout slide belongs.
         * @param {String} layoutId
         *  Id of the layout slide for which we want to get index.
         * @returns {Number|Null}
         *  The index of the layout slide relative to it's master slide.
         */
        this.getRelativeLayoutIndex = function (masterId, layoutId) {
            var masterIndex = _.indexOf(masterSlideOrder, masterId);
            var layoutIndex = _.indexOf(masterSlideOrder, layoutId);

            return (masterIndex > -1 && layoutIndex > -1) ? layoutIndex - masterIndex : null;
        };

        /**
         * Check, whether the master and layout slides are visible.
         *
         * @returns {Boolean}
         *  Whether the master and layout slides are visible.
         */
        this.isMasterView = function () {
            return isMasterView;
        };

        /**
         * Receiving the type of the slide for a specified id.
         *
         * @param {String} id
         *  The slide id.
         *
         * @returns {String|Null}
         *  The type of the slide. Or null, if no type could be determined.
         */
        this.getSlideType = function (id) {
            return idTypeConnection[id] || null;
        };

        /**
         * Receiving a complete copy of the id-type-connection model object.
         *
         * @returns {Object}
         *  A copy of the id-type-connection model object
         */
        this.getIdTypeConnectionModel = function () {
            return _.clone(idTypeConnection);
        };

        /**
         * Check, whether the specified id string is a valid id of a standard slide.
         *
         * @param {String} id
         *  The id string.
         *
         * @returns {Boolean}
         *  Whether the specified id string is a valid id for a standard slide.
         */
        this.isStandardSlideId = function (id) {
            return allStandardSlides[id] !== undefined;
        };

        /**
         * Check, whether the specified id string is a valid id of a layout slide.
         *
         * @param {String} id
         *  The id string.
         *
         * @returns {Boolean}
         *  Whether the specified id string is a valid id for a layout slide.
         */
        this.isLayoutSlideId = function (id) {
            return allLayoutSlides[id] !== undefined;
        };

        /**
         * Check, whether the specified id string is a valid id for a master slide.
         *
         * @param {String} id
         *  The id string.
         *
         * @returns {Boolean}
         *  Whether the specified id string is a valid id for a master slide.
         */
        this.isMasterSlideId = function (id) {
            return allMasterSlides[id] !== undefined;
        };

        /**
         * Check, whether the specified id string is a valid id of a layout or master slide.
         *
         * @param {String} id
         *  The id string.
         *
         * @returns {Boolean}
         *  Whether the specified id string is a valid id for a layout or master slide node.
         */
        this.isLayoutOrMasterId = function (id) {
            return self.isLayoutSlideId(id) || self.isMasterSlideId(id);
        };

        /**
         * Provides the layout slide node or master slide node in the layout slide layer or master slide
         * layer corresponding to the specified target string.
         *
         * @param {String} id
         *  The slide id.
         *
         * @returns {jQuery|null}
         *  The slide node inside the layout or master slide layer node. Or null, if no node with the
         *  specified target exists.
         */
        this.getLayoutOrMasterSlide = function (id) {
            return getLayoutSlide(id) || getMasterSlide(id) || null;
        };

        /**
         * Recieving the slide set consisting of document slide id, layout slide id and master slide
         * id for a specified id.
         * If the id is not specified, the active slide id is used.
         *
         * @param {String} [id]
         *  The slide id. If not specified, the id of the active slide is used.
         *
         * @returns {Object}
         *  An object containing the properties 'id', 'layoutId' and 'masterId', that represent the
         *  group of IDs for one specified id.
         */
        this.getSlideIdSet = function (id) {

            var // the IDs for document slide, layout slide and master slide
                slideId = null, layoutId = null, masterId = null;

            if (!id) { id = activeSlideId; }  // if not specified, the active slide is used

            // setting all IDs that need to be activated
            if (self.isStandardSlideId(id)) {
                slideId = id;
                layoutId = self.getLayoutSlideId(slideId);
                masterId = self.getMasterSlideId(layoutId);
            } else if (self.isLayoutSlideId(id)) {
                layoutId = id;
                masterId = self.getMasterSlideId(layoutId);
            } else if (self.isMasterSlideId(id)) {
                masterId = id;
            }

            return { id: slideId, layoutId: layoutId, masterId: masterId };
        };

        /**
         * Recieving the slide set consisting of document slide, layout slide and master slide for
         * a specified id.
         * If the id is not specified, the active slide id is used.
         *
         * @param {String} [id]
         *  The slide id. If not specified, the id of the active slide is used.
         *
         * @returns {Object}
         *  An object containing the properties 'slide', 'layoutSlide' and 'masterSlide'. The values
         *  are the jQuerified slides (or null, if the slide is not set or cannot be determined).
         */
        this.getSlideSet = function (id) {

            var // the grouped document slide, layout slide and master slide
                slide = null, layoutSlide = null, masterSlide = null,
                // the ID set for the specified slide id
                slideIdSet = null;

            if (!id) { id = activeSlideId; }  // if not specified, the active slide is used

            slideIdSet = self.getSlideIdSet(id);

            if (slideIdSet.id) { slide =  self.getSlideById(slideIdSet.id); }
            if (slideIdSet.layoutId) { layoutSlide =  self.getSlideById(slideIdSet.layoutId); }
            if (slideIdSet.masterId) { masterSlide =  self.getSlideById(slideIdSet.masterId); }

            return { slide: slide, layoutSlide: layoutSlide, masterSlide: masterSlide };
        };

        /**
         * Provides a jQuerified slide node specified by its id. Or null, if no slide is registered with
         * this id.
         *
         * @param {String} id
         *  The slide id.
         *
         * @returns {jQuery|null}
         *  The slide node specified by the id. Or null, if no node with the specified id exists.
         */
        this.getSlideById = function (id) {
            return getStandardSlide(id) || getLayoutSlide(id) || getMasterSlide(id) || null;
        };

        /**
         * Receiving all slides of the currently active view. This can be the standard view or the
         * master/layout view.
         *
         * @returns {Object|null}
         *  An object containing the IDs as key and the jQuerified slides as value for all slides
         *  of the currently active view.
         */
        this.getAllSlidesOfActiveView = function () {
            return isMasterView ? Utils.extendOptions(allLayoutSlides, allMasterSlides) : allStandardSlides;
        };

        /**
         * Receiving all slides from given group.
         *
         * @param {String} group
         *  Can be master, layout or standard group.
         * @returns {Object|null}
         *  An object containing the IDs as key and the jQuerified slides as value for all slides of one group.
         */
        this.getAllGroupSlides = function (group) {
            switch (group) {
                case 'master':
                    return _.clone(allMasterSlides);
                case 'layout':
                    return _.clone(allLayoutSlides);
                case 'standard':
                    return _.clone(allStandardSlides);
                default:
                    return null;
            }
        };

        /**
         * Receiving all slides from all views.
         *
         * @returns {Object|null}
         *  An object containing the IDs as key and the jQuerified slides as value for all slides.
         */
        this.getAllSlides = function () {
            return Utils.extendOptions(allStandardSlides, allLayoutSlides, allMasterSlides);
        };

        /**
         * Activating a slide with a specified id.
         *
         * @param {String} id
         *  The id of the slide, that shall be activated.
         *
         * @returns {Boolean}
         *  Whether the slide was changed.
         */
        this.changeToSlide = function (id) {

            var // whether the slide changed
                slideChange = false;

            if (id !== activeSlideId) { slideChange = setActiveSlideId(id); }

            return slideChange;
        };

        /**
         * Getter for the ID of the specified slide node.
         *
         * @param {Node|jQuery|Null} slideNode
         *  The DOM node to be checked. If this object is a jQuery collection, uses
         *  the first DOM node it contains. If missing or null, returns empty string.
         *
         * @returns {String}
         *  The ID of the specified slide. If this is no slide or the ID cannot be
         *  determined, an empty string is returned.
         */
        this.getSlideId = function (slideNode) {
            return DOM.isSlideNode(slideNode) ? DOM.getTargetContainerId(slideNode) : '';
        };

        /**
         * Check, whether the active slide view is empty.
         *
         * @returns {Boolean}
         *  Whether the active slide view is empty.
         */
        this.isEmptySlideView = function () {
            return (isMasterView ? masterSlideOrder.length === 0 : slideOrder.length === 0);
        };

        /**
         * Check, if the active slide is the first slide inside the standard or the master/layout view.
         *
         * @returns {Boolean}
         *  Whether the active slide is the first slide inside the standard or the master/layout view.
         */
        this.isFirstSlideActive = function () {
            return (isMasterView ? _.first(masterSlideOrder) : _.first(slideOrder)) === activeSlideId;
        };

        /**
         * Check, if the active slide is the last slide inside the standard or the master/layout view.
         *
         * @returns {Boolean}
         *  Whether the active slide is the last slide inside the standard or the master/layout view.
         */
        this.isLastSlideActive = function () {
            return (isMasterView ? _.last(masterSlideOrder) : _.last(slideOrder)) === activeSlideId;
        };

        /**
         * Check, if the specified slide ID is the first inside its container.
         *
         * @returns {Boolean}
         *  Whether the specified slide is the first slide inside its container.
         */
        this.isFirstSlideInContainer = function (id) {
            return (self.isLayoutOrMasterId(id) ? _.first(masterSlideOrder) : _.first(slideOrder)) === id;
        };

        /**
         * Check, if the specified slide ID is the last inside its container.
         *
         * @returns {Boolean}
         *  Whether the specified slide is the last slide inside its container.
         */
        this.isLastSlideInContainer = function (id) {
            return (self.isLayoutOrMasterId(id) ? _.last(masterSlideOrder) : _.last(slideOrder)) === id;
        };

        /**
         * Check, whether a slide specified by its id, is the only slide in its view. This can be the view
         * for the standard slides or the view for the master/layout slides.
         *
         * @param {String} [id]
         *  The id of the slide. If not specified, the active slide id is used.
         *
         * @returns {Boolean}
         *  Whether the specified slide is the only slide in its view.
         */
        this.isOnlySlideInView = function (id) {

            var // whether the specified slide is the only slide in the view (standard view or master/layout view)
                isOnlySlide = false,
                // the id of the slide
                localId = id || activeSlideId;

            if (self.isStandardSlideId(localId)) {
                isOnlySlide = (slideOrder.length === 1);
            } else if (self.isLayoutOrMasterId(localId)) {
                isOnlySlide = (masterSlideOrder.length === 1);
            }

            return isOnlySlide;
        };

        /**
         * Getting the index of the specified slide in the current container. If
         * no id is specified, the active slide is used.
         *
         * @param {String} [id]
         *  The id of the slide. If not specified, the active slide id is used.
         *
         * @returns {Number}
         *  The index of the specified or active slide in its container. If the
         *  slide id could not be found in the array, '-1' is returned.
         */
        this.getSortedSlideIndexById = function (id) {

            var // the id of the slide
                localId = id || activeSlideId,
                // the container that contains the order of the slides
                orderContainer = self.isLayoutOrMasterId(localId) ? masterSlideOrder : slideOrder;

            return _.indexOf(orderContainer, localId);
        };

        /**
         * Receiving the ID of a document slide from its logical position or from
         * its index in the sorted slide order container. This function CANNOT be
         * used for layout or master slides.
         *
         * @param {Number|Number[]} pos
         *  The index or the logical position specifying a document slide.
         *
         * @returns {String|Null}
         *  The ID of the document slide. Or null, if it cannot be determined.
         */
        this.getSlideIdByPosition = function (pos) {

            var // the index of the slide in the sorted
                slideIndex = _.isArray(pos) ? _.first(pos) : pos;

            return (_.isNumber(slideIndex) && slideOrder[slideIndex]) ? slideOrder[slideIndex] : null;
        };

        /**
         * Getting the logical position of a slide specified by its ID. If the ID is not
         * valid, null is returned.
         *
         * @param {String} id
         *  The ID of the slide
         *
         * @returns {Number[]|Null}
         *  The logical position of the specified slide. Or null, if it cannot be determined.
         */
        this.getSlidePositionById = function (id) {
            return self.isStandardSlideId(id) ? [_.indexOf(slideOrder, id)] : (self.isLayoutOrMasterId(id) ? [0] : null);
        };

        /**
         * Returns slide number for given slide id.
         * This function CANNOT be used for layout or master slides.
         *
         * @param {String} slideID
         *  The ID of the document slide
         *
         * @returns {Number}
         *  The number of the document slide. Or null, if it cannot be determined.
         */
        this.getSlideNumByID = function (slideID) {
            var index = _.indexOf(slideOrder, slideID);

            return index > -1 ? index + 1 : null;
        };

        /**
         * Whether the document contains more than one master slide.
         *
         * @returns {Boolean}
         *  Whether the document contains more than one master slide.
         */
        this.isMultiMasterSlideDocument = function () {
            return _.keys(allMasterSlides).length > 1;
        };

        /**
         * Collecting the position information for the master slides in the model container 'masterSlideOrder'.
         *
         * @returns {Object}
         *  An object that has as properties the IDs of the master slides and as values the index for each
         *  slide in the container 'masterSlideOrder'.
         *  Additionally a sorted array with all master slide indices is returned with the property 'order'.
         */
        this.getAllMasterSlideIndices = function () {

            var // an array with all master slide IDs
                allMasterSlideIDs = _.keys(allMasterSlides),
                // a sorted array with all master slide indices
                allMasterSlideIndices = [],
                // a helper object, that has as keys the master slide IDs and a special key 'order'
                infoObject = {};

            _.each(allMasterSlideIDs, function (id) {
                var // the index for the specified id
                    index = _.indexOf(masterSlideOrder, id);

                infoObject[id] = index;
                allMasterSlideIndices.push(index);
            });

            infoObject.order = allMasterSlideIndices.sort(function (a, b) { return a - b; });

            return infoObject;
        };

        /**
         * Getting a valid index to insert a slide as last slide behind a specified master slide. If -1
         * is returned, the slide can be added to the end of the container.
         *
         * @param {String} id
         *  The id of a master slide.
         *
         * @returns {Number}
         *  The index in the container 'masterSlideOrder' that can be used as final index behind the specified
         *  master slide (it is the position of the following master slide). If there is only one master slide
         *  -1 is returned. This means, that the new element can be added to the end of the container.
         */
        this.getLastPositionInModelBehindMaster = function (id) {

            var // a helper object with all informations about the master slide indices in the model 'masterSlideOrder'
                indexObject = self.getAllMasterSlideIndices(),
                // the new index
                newIndex = -1;

            if (indexObject.order.length > 1 && _.isNumber(indexObject[id])) {

                if (indexObject[id] !== _.last(indexObject.order)) {
                    newIndex = indexObject.order[_.indexOf(indexObject.order, indexObject[id]) + 1];
                }

            }

            return newIndex;
        };

        /**
         * Getting the number of layout slides that have the master specified by
         * its ID as target.
         *
         * @param {String} id
         *  The ID of the master slide
         *
         * @returns {Number}
         *  The number of layout slides belonging to the master. Or -1, if the
         *  specified ID is not the ID of a master slide.
         */
        this.getNumberOfLayoutSlidesOfSpecifiedMaster = function (id) {

            var // the number of layout slides
                count = -1,
                // a search index in the master slide container
                index = 0;

            if (self.isMasterSlideId(id)) {

                count = 0;
                index = _.indexOf(masterSlideOrder, id) + 1;

                while (index < masterSlideOrder.length && self.isLayoutSlideId(masterSlideOrder[index])) {
                    count++;
                    index++;
                }

            }

            return count;
        };

        /**
         * Check, whether a specified ID is the ID of a layout slide and if this layout slide is
         * used by at least one standard slide.
         * The model 'layoutConnection' can be used check, if the specified ID is at least once the value.
         *
         * @param {String} id
         *  The ID of the slide.
         *
         * @returns {Boolean}
         *  Whether the slide specified by the ID is a layout slide that is referenced by at least
         *  one standard slide.
         */
        this.isUsedLayoutSlide = function (id) {
            return self.isLayoutSlideId(id) && _.contains(_.values(layoutConnection), id);
        };

        /**
         * Check, whether a master slide specified by its id is used as target by at least one layout
         * slide. If this is not the case, the slide can be removed.
         *
         * @param {String} id
         *  The id of the slide.
         *
         * @returns {Boolean}
         *  Whether the specified slide is used as target by another slide.
         */
        this.isUsedMasterSlideByLayoutSlide = function (id) {
            return _.contains(_.values(layoutMasterConnection), id);
        };

        /**
         * Check, whether a master slide specified by its id is used by at least one document
         * slide (standard slide). This means, that at least one document slide is based on
         * a layout slide, that has the specified master slide as target.
         *
         * @param {String} id
         *  The id of the slide.
         *
         * @returns {Boolean}
         *  Whether the specified slide is used as target by another slide.
         */
        this.isUsedMasterSlideByStandardSlide = function (id) {

            var // whether the ID specifies a master slide, that is used by at least one document slide
                isUsed = false,
                // a collection of all layout slide, that have the specified master as target
                allLayoutSlides = null;

            if (self.isMasterSlideId(id)) {

                allLayoutSlides = self.getAllSlideIDsWithSpecificTargetParent(id);

                if (allLayoutSlides && allLayoutSlides.length > 0) {
                    isUsed = _.isString(_.find(allLayoutSlides, self.isUsedLayoutSlide));
                }
            }

            return isUsed;
        };

        /**
         * Check, whether a layout or master slide specified by its id is used as target by another slide.
         * If this is the case, the slide cannot be removed.
         *
         * @param {String} id
         *  The id of the slide.
         *
         * @returns {Boolean}
         *  Whether the specified slide is used as target by a standard (document) slide.
         */
        this.isUsedMasterOrLayoutSlide = function (id) {
            return self.isLayoutOrMasterId(id) && (self.isUsedLayoutSlide(id) || self.isUsedMasterSlideByStandardSlide(id));
        };

        /**
         * Receiving the layout slide id for a document slide specified by its id.
         *
         * @param {String} id
         *  The id of the document slide, whose layout slide id shall be determined.
         *
         * @returns {String|Null}
         *  The id of the layout slide that belongs to the specified slide. If no layout slide exists,
         *  null is returned.
         */
        this.getLayoutSlideId = function (id) {
            return layoutConnection[id] || null;
        };

        /**
         * Receiving the master slide id for a layout slide specified by its id.
         *
         * @param {String} id
         *  The id of the layout slide, whose master slide id shall be determined.
         *
         * @returns {String|Null}
         *  The id of the master slide that belongs to the specified layout slide. If no master slide
         *  exists, null is returned.
         */
        this.getMasterSlideId = function (id) {
            return layoutMasterConnection[id] || null;
        };

        /**
         * Receiving the layout slide id or master slide id for a slide specified by its id.
         *
         * @param {String} id
         *  The id of the slide, whose layout slide id or master slide id shall be determined.
         *
         * @returns {String|Null}
         *  The id of the layout or master slide that belongs to the specified slide. If no
         *  layout or master slide exists, null is returned.
         */
        this.getLayoutOrMasterSlideId = function (id) {
            return self.getLayoutSlideId(id) || self.getMasterSlideId(id);
        };

        /**
         * Receiving the model object for the connection between standard and layout slides.
         *
         * @returns {Object}
         *  The connection object for standard and layout slide IDs.
         */
        this.getLayoutConnection = function () {
            return _.copy(layoutConnection);
        };

        /**
         * Receiving the model object for the connection between layout and master slides.
         *
         * @returns {Object}
         *  The connection object for layout and master slide IDs.
         */
        this.getLayoutMasterConnection = function () {
            return _.copy(layoutMasterConnection);
        };

        /**
         * Check, whether a slide specified by its ID can be deleted. This is not the case, if:
         * - the slide is the last slide in its view.
         * - the slide is a master or layout slide, that is referenced by at least one document
         *   slide.
         *
         * @param {String} [id]
         *  The slide ID. If not specified, the ID of the active slide is used.
         *
         * @returns {Boolean}
         *  Whether the slide can be deleted.
         */
        this.isDeletableSlide = function (id) {
            return !self.isUsedMasterOrLayoutSlide(id ? id : activeSlideId);
        };

        /**
         * Check, whether a slide selection (in the slide pane) can be deleted. This is not the case, if:
         * - there are no slides
         * - the final master slide would be deleted
         * - a slide in the selection is a master or layout slide, that is referenced by at least one document
         *   slide.
         *
         * @param {Number[]} selection
         *  An Array containing the position as index from all selected slides.
         *
         * @returns {Boolean}
         *  Whether the slides can be deleted.
         */
        this.isSlideSelectionDeletable = function (selection) {

            // whether the selected slides can be deleted
            var deletable = true;
            // a container for all master slides that shall be deleted
            var masterSlideIds = null;

            // for master/layout slides
            if (self.isMasterView()) {

                masterSlideIds = [];

                _.each(selection, function (slideIndex) {
                    // when an element in the selection is not deletable
                    var slideId = self.getIdOfSlideOfActiveViewAtIndex(slideIndex);
                    if (self.isUsedMasterOrLayoutSlide(slideId)) { deletable = false; }
                    if (deletable && self.isMasterSlideId(slideId)) { masterSlideIds.push(slideId); }
                });

                if (deletable && (masterSlideIds.length === self.getMasterSlideCount())) { deletable = false; }  // never delete the final master slide
            }

            return deletable && !self.isEmptySlideView();
        };

        /**
         * Receiving the number of all normal (standard) slides.
         *
         * @returns {Number}
         *  The number of all normal (standard) slides
         */
        this.getStandardSlideCount = function () {
            return slideOrder.length;
        };

        /**
         * Receiving the sorted array of normal slide IDs.
         *
         * @param {Boolean} doCopy
         *  Whether a copy of the order container shall be returned. If not
         *  specified, no copy is made.
         *
         * @returns {String[]}
         *  The sorted array of normal slide IDs.
         */
        this.getStandardSlideOrder = function (doCopy) {
            return doCopy ? _.clone(slideOrder) : slideOrder;
        };

        /**
         * Checking, whether a specified id is a valid and registered slide id.
         *
         * @param {String} id
         *  The slide id to be checked.
         *
         * @returns {Boolean}
         *  Whether the specified id is a valid value for a slide id.
         */
        this.isValidSlideId = function (id) {
            return isValidSlideId(id);
        };

        /**
         * Receiving the slide id of the direct parent slide. This is the
         * layout slide for a standard slide and the master slide id for
         * a layout slide.
         *
         * @param {String} [id]
         *  The ID of the slide, whose direct parent slide shall be determined.
         *  If not specified, the id of the active slide is used.
         *
         * @returns {String|Null}
         *  The ID of the direct parent slide. If not specified, null is
         *  returned.
         */
        this.getParentSlideId = function (id) {

            var // the ID of the direct parent slide
                parentId = null;

            id = id || activeSlideId;

            if (self.isStandardSlideId(id)) {
                parentId = self.getLayoutSlideId(id);
            } else if (self.isLayoutSlideId(id)) {
                parentId = self.getMasterSlideId(id);
            }

            return parentId;
        };

        /**
         * Receiving the sorted array of master and layout slide IDs.
         *
         * @param {Boolean} doCopy
         *  Whether a copy of the order container shall be returned. If not
         *  specified, no copy is made.
         *
         * @returns {String[]}
         *  The sorted array of master and layout slide IDs.
         */
        this.getMasterSlideOrder = function (doCopy) {
            return doCopy ? _.clone(masterSlideOrder) : masterSlideOrder;
        };

        /**
         * Receiving the number of master and layout slides.
         *
         * @returns {Number}
         *  The number of all master and layout slides
         */
        this.getMasterSlideCount = function () {
            return masterSlideOrder.length;
        };

        /**
         * Receiving the number of slides in active view.
         *
         * @returns {Number}
         *  The number of slides in the active view.
         */
        this.getActiveViewCount = function () {

            var // the container that contains the order of the slides
                orderContainer = isMasterView ? masterSlideOrder : slideOrder;

            return orderContainer.length;
        };

        /**
         * Receiving the next valid slide (up or downwards). A valid slide is,
         * (A) a master slide, or (B) a normal/layout slide. By default the next valid
         * slide is a normal or layout slide. It returns -1 when there are no valid
         * slides in the given direction anymore.
         *
         * @param {String} id
         *  The id of the slide, whose next slide (normal/layout or master) shall be determined.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.masterIsValidSlide=false]
         *      Whether next valid slide should be a master slide or a normal/layout slide.
         *      Default are the normal and layout slides.
         *
         *  @param {Boolean} upwards
         *   Whether the next slide is upwards or downwards from the given slide ID.
         *   Default is upwards.
         *
         * @returns {Number}
         *  The index of the next valid slide. If the index is not valid
         *  (e.g. there is not normal or layout slide up/downwards anymore), -1 is returned.
         */
        this.getNextValidSlideIndex = function (id, options) {

            var // whether the next slide is upwards or downwards
                upwards = Utils.getBooleanOption(options, 'upwards', true),
                // whether a valid slide is a master or normal/layout slide
                masterIsValidSlide = Utils.getBooleanOption(options, 'masterIsValidSlide', false),
                // whether the slide id is in master view or not
                isMasterOrLayoutId = self.isLayoutOrMasterId(id),
                // container that contains the order of the slides
                container = isMasterOrLayoutId ? masterSlideOrder : slideOrder,
                // the index of the given slide id
                currentIndex = _.indexOf(container, id),
                // next slide index up or down for the given id
                nextSlideIndex = upwards ? currentIndex - 1 : currentIndex + 1;

            // returns whether the 'nextSlideIndex' is in a valid range or not
            function inRange(nextSlideIndex) {
                return (nextSlideIndex > -1 && nextSlideIndex < container.length);
            }
            // returns whether the slide at the 'nextSlideIndex' is a valid slide or not
            function isValidSlide(masterIsValidSlide, nextSlideIndex) {
                return (masterIsValidSlide ? !(self.isMasterSlideId(container[nextSlideIndex])) : self.isMasterSlideId(container[nextSlideIndex]));
            }

            // Find the next valid slide position:
            // - 1)  check if the index is in range to prevent undefined values for 'isValidSlide()' and prevent a infinite loop
            // -- 2A) masterIsValidSlide = true -> go to the next position until it's not a normal/layout slide or out of range
            // -- 2B) masterIsValidSlide = false -> go to the next position until it's not a master slide or out of range
            while (inRange(nextSlideIndex) && isValidSlide(masterIsValidSlide, nextSlideIndex)) {

                nextSlideIndex = upwards ? nextSlideIndex - 1 : nextSlideIndex + 1;
            }

            // check if the index is in a valid range, when not there is no next valid slide in that direction and -1 is returned
            return inRange(nextSlideIndex) ? nextSlideIndex : -1;
        };

        /**
         * Receiving a list of all slide IDs of those slides, that have as parent the specified target.
         * The target parameter can be the id of the corresponding layout slide or master slide.
         *
         * @param {String} target
         *  The id of the target slide. The target slide is a layout slide or a master slide.
         *
         * @returns {String[]}
         *  An array containing the IDs of those slides, that have the specified target as parent.
         */
        this.getAllSlideIDsWithSpecificTargetParent = function (target) {

            var // a collector for all affected slide IDs
                allSlideIDs = [];

            if (self.isLayoutSlideId(target)) {
                _.each(_.keys(layoutConnection), function (key) {
                    if (layoutConnection[key] === target) { allSlideIDs.push(key); }
                });
            } else if (self.isMasterSlideId(target)) {
                _.each(_.keys(layoutMasterConnection), function (key) {
                    if (layoutMasterConnection[key] === target) { allSlideIDs.push(key); }
                });
            }

            return allSlideIDs;
        };

        /**
         * Receiving a list of all slide IDs of those slides, that have as target the specified target
         * or whose target slide has the specified target.
         *
         * The target can be the id of the corresponding layout slide or master slide.
         *
         * @param {String} target
         *  The id of the target slide. The target slide is a layout slide or a master slide.
         *
         * @returns {String[]}
         *  An array containing the IDs of those slides, that have the specified target or whose
         *  target slide have the specified target.
         */
        this.getAllSlideIDsWithSpecificTargetAncestor = function (target) {

            var // a collector for all affected slide IDs
                allSlideIDs = self.getAllSlideIDsWithSpecificTargetParent(target);

            // additionally search all 'standard' slides affected by the changes of the layout slides
            if (self.isMasterSlideId(target)) {
                _.each(_.clone(allSlideIDs), function (layoutID) {
                    allSlideIDs = allSlideIDs.concat(self.getAllSlideIDsWithSpecificTargetParent(layoutID));
                });
            }

            return allSlideIDs;
        };

        /**
         * Switching the view for displaying the 'normal' slides or the master and
         * layout slides.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.showMaster=false]
         *      Whether the normal slides shall be displayed or the master and layout slides.
         *      Default are the normal slides.
         */
        this.selectSlideView = function (options) {

            var // whether the master/layout view will be activated or not
                showMaster = Utils.getBooleanOption(options, 'showMaster', false);

            // setting value in the model
            setActiveView(showMaster);
        };

        /**
         * Check, if a specified slide or the currently active slide displays content of layout or master slide.
         *
         * @param {String} [id]
         *  The id of a slide. If not specified, the id of the currently active slide is used.
         *
         * @returns {Boolean}
         *  Whether the specified or the active slide has the attribute 'followmastershapes' set or not.
         */
        this.isFollowMasterShapeSlide = function (id) {
            return (id || activeSlideId) ? (self.getSlideFamilyAttributeForSlide(id ? id : activeSlideId, 'slide', 'followMasterShapes') !== false) : true;
        };

        /**
         * Check, if a specified slide or the currently active slide is a 'hidden' slide.
         *
         * @param {String} [id]
         *  The id of a slide. If not specified, the id of the currently active slide is used.
         *
         * @returns {Boolean}
         *  Whether the specified or the active slide has the attribute 'hidden' set or not.
         */
        this.isHiddenSlide = function (id) {
            return (id || activeSlideId) ? (self.getSlideFamilyAttributeForSlide(id ? id : activeSlideId, 'slide', 'hidden') === true) : false;
        };

        /**
         * Generating an operation for the slide attribute 'followMasterShapes'.
         *
         * @param {Boolean} value
         *  The value for the attribute 'followMasterShapes'.
         */
        this.setFollowMasterShape = function (value) {

            var // the operations generator
                generator = null,
                // whether a master slide is active
                isMasterSlide = self.isMasterSlideId(self.getActiveSlideId()),
                // the options for the setAttributes operation
                operationOptions = null;

            // no 'followMasterShape' operation for master slides
            if (!isMasterSlide) {

                generator = self.createOperationsGenerator();
                operationOptions = { start: self.getActiveSlidePosition(), attrs: { slide: { followMasterShapes: value } } };

                // generate the 'setAttributes' operation
                generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions);

                // apply all collected operations
                this.applyOperations(generator);
            }
        };

        /**
         * Generating an operation for the slide attribute 'hidden'. This is only
         * possible in the normal view. Layout or master slides cannot be 'hidden'.
         *
         * @param {Boolean} value
         *  The value for the attribute 'hidden'.
         */
        this.setHiddenSlide = function (value) {

            var // the operations generator
                generator = null,
                // the options for the setAttributes operation
                operationOptions = null;

            if (!isMasterView) { // no valid operation in master view

                generator = self.createOperationsGenerator();
                operationOptions = { start: self.getActiveSlidePosition(), attrs: { slide: { hidden: value } } };

                // generate the 'setAttributes' operation
                generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions);

                // apply all collected operations
                this.applyOperations(generator);
            }
        };

        /**
         * Generating an operation to set the slidePaneWidth attribute.
         * Note: The 'localViewSlidePaneWidth' is always rounded to
         * thousandths (e.g. to 15.123), so the 'slidePaneWidth' in this operation is
         * also always rounded to thousandths.
         *
         * @param {Number} width
         *  The width for the slide pane in percent.
         */
        this.setSlidePaneWidthInDocAttributes = function () {
            self.applyOperations({ name: Operations.SET_DOCUMENT_ATTRIBUTES, attrs: { layout: { slidePaneWidth: localViewSlidePaneWidth } } });
        };

        /**
         * Setting the currently used slidePaneWidth by the user in the browser.
         *
         * @param {Number} width
         *  The width for the slide pane in percent. It is rounded to
         *  thousandths (e.g. to 15.123)
         */
        this.setLocalViewSlidePaneWidth = function (width) {
            localViewSlidePaneWidth = Utils.round(width, 0.001);
        };

        /**
         * Getting the currently used slidePaneWidth by the user in the browser.
         *
         * @returns {Number}
         *  The width for the slide pane in percent.
         */

        this.getSlidePaneWidth = function () {
            return localViewSlidePaneWidth;

        };

        /**
         * Getting the slidePaneWidth from operations.
         *
         * @returns {Number}
         *  The width for the slide pane width in percent.
         *  We get a value rounded to thousandths from the
         *  filter (e.g. to 15.123)
         */
        this.getSlidePaneWidthInDocAttributes = function () {
            return self.getLayoutAttributes().slidePaneWidth;
        };

        /**
         * Check, whether the active slide or a slide specified by an ID has a background set.
         *
         * @param {String} [id]
         *  The id of a slide. If not specified, the id of the currently active slide is used.
         *
         * @returns {Boolean}
         *  Whether the specified or the active slide has a defined background.
         */
        this.isSlideWithBackground = function (id) {

            var // the fill type slide attribute
                fillType = self.getSlideFamilyAttributeForSlide(id || self.getActiveSlideId(), 'fill', 'type');

            return fillType && fillType !== 'none';
        };

        /**
         * Deleting the slide background.
         */
        this.resetBackground = function () {

            var // the operations generator
                generator = self.createOperationsGenerator(),
                // the options for the setAttributes operation
                operationOptions = { start: self.getActiveSlidePosition(), attrs: self.getEmptySlideBackgroundAttributes() };

            // generate the 'setAttributes' operation
            generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions);

            // apply all collected operations
            this.applyOperations(generator);
        };

        /**
         * Setting the background color of the active slide.
         * Info: The filter does not like the color 'auto'. Therefore this is handled
         *       like removing an existing background (color and image). In pptx file
         *       the background color is also only removed, if 'auto' is selected by
         *       the user.
         *
         * @param {Object} color
         *  The color object.
         */
        this.setBackgroundColor = function (color) {

            var // the operations generator
                generator = self.createOperationsGenerator(),
                // the options for the setAttributes operation
                operationOptions = { start: self.getActiveSlidePosition(), attrs: { fill: { type: 'solid', color: color, imageUrl: null } } },
                // whether the color 'auto' was selected -> removing any background
                isAutoColor = (color && color.type && color.type === 'auto');

            if (isAutoColor) { operationOptions.attrs = self.getEmptySlideBackgroundAttributes(); }

            // generate the 'setAttributes' operation
            generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions);

            // apply all collected operations
            this.applyOperations(generator);
        };

        /**
         * Shows an insert image dialog from drive, local or URL dialog.
         *
         * @param {String} dialogType
         *  The dialog type. Types: 'drive', 'local' or 'url'. Default is 'drive'.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved or rejected after the dialog has
         *  been closed and the image has been inserted into the document.
         */
        this.showInsertImageDialogForSlideBackground = function (dialogType) {
            return Image.showInsertImageDialogByType(app, dialogType).then(function (imageFragment) {
                return insertImageForSlideBackground(imageFragment);
            });
        };

        /**
         * Getting the vertical text alignment of a currently selected text frame node.
         *
         * @returns {String}
         *  The vertical text alignment mode of the text inside a selected text frame node. If it cannot
         *  be determined, the default value 'top' is returned.
         */
        this.getVerticalAlignmentMode = function () {

            var // the selected text frame node (also finding text frames inside groups)
                drawingFrame = selection.getAnyTextFrameDrawing({ forceTextFrame: true }),
                // the text frame node inside the shape
                textFrame = DrawingFrame.getTextFrameNode(drawingFrame);

            return (textFrame && textFrame.length > 0 && textFrame.attr(DrawingFrame.VERTICAL_ALIGNMENT_ATTRIBUTE)) || 'top';
        };

        /**
         * Setting the vertical text alignment mode inside a text frame.
         *
         * @param {String} mode
         *  The vertical text alignment mode of the text inside a selected text frame node.
         */
        this.setVerticalAlignmentMode = function (state) {

            var // the operations generator
                generator = self.createOperationsGenerator(),
                // the options for the setAttributes operation
                operationOptions = {},
                // a container for the logical positions of all affected drawings
                allDrawings = selection.getAllDrawingsInSelection(DrawingFrame.isTextFrameShapeDrawingFrame);

            if (!allDrawings.length > 0) { return; }

            // collecting the attributes for the operation
            operationOptions.attrs =  {};
            operationOptions.attrs.shape = { anchor: state };

            // iterating over all drawings
            _.each(allDrawings, function (oneDrawing) {
                operationOptions.start = Position.getOxoPosition(app.getModel().getCurrentRootNode(), oneDrawing, 0);
                // generate the 'setAttributes' operation
                generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions);
            });

            // apply all collected operations
            self.applyOperations(generator);
        };

        /**
         * Getter method for root container node with passed target, or if target is not passed, default page node.
         *
         * @param {String} [target]
         *  ID of root container node
         *
         * @returns {jQuery}
         *  The root node specified by the target string, or the page node, if no target was specified.
         */
        this.getRootNode = function (target) {
            return (target && self.isLayoutOrMasterId(target)) ? getLayoutOrMasterSlideContentNode(target) : pagediv;
        };

        /**
         * Helper function to check the visibility of the page node. This node must be invisible, if there
         * is no remaining slide in the active view. But during loading the 'pageVisibilityHandler' must be called
         * with this public function.
         *
         * @returns {Boolean}
         *  Whether the visibility of the page node has changed.
         */
        this.checkPageVisibility = function () {
            pageVisibilityHandler();
        };

        /**
         * Getter method for currently active root container node.
         *
         * @param {String} [target]
         *  ID of root container node
         *
         * @returns {jQuery}
         *  The root node specified by the target string
         */
        this.getCurrentRootNode = function (target) {

            // target for operation - if exists
            var currTarget = target || self.getActiveTarget();

            // checking, if this is a target from a master or layout slide
            if (currTarget && self.isLayoutOrMasterId(currTarget)) { return getLayoutOrMasterSlideContentNode(currTarget); }

            // checking, if the target is a target for a comment. Then this is handled by the comment layer.
            if (currTarget && self.getCommentLayer().getCommentRootNode(currTarget)) { return self.getCommentLayer().getCommentRootNode(currTarget); }

            // returning the page node, if no other root node was found before
            return pagediv;
        };

        /**
         * Getter method to receive all available layout slide IDs.
         *
         * @returns {String[]}
         *  An array with all defined IDs of layout slides.
         */
        this.getAllLayoutSlideIds = function () {
            return _.keys(allLayoutSlides);
        };

        /**
         * Changing the active slide, if possible
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.next=true]
         *   Whether the next or the previous slide shall be selected.
         *
         * @returns {Boolean}
         *  Whether the slide was changed.
         */
        this.changeToNeighbourSlide = function (options) {

            var // whether the next slide shall be made visible
                next = Utils.getBooleanOption(options, 'next', true),
                // the container that contains the order of the slides
                orderContainer = isMasterView ? masterSlideOrder : slideOrder,
                // the index of the currently active slide inside the ordered container
                index = _.indexOf(orderContainer, activeSlideId),
                // the index of the slide that shall be activated
                newIndex = -1,
                // whether the slide changed
                slideChange = false;

            if (next) {
                if (index < (isMasterView ? masterSlideOrder.length : slideOrder.length)) { newIndex = index + 1; }
            } else {
                if (index > 0) { newIndex = index - 1; }
            }

            if (newIndex > -1) { slideChange = self.changeToSlide(self.getIdOfSlideOfActiveViewAtIndex(newIndex)); }

            return slideChange;
        };

        /**
         * Changing to a slide specified by its index in the order containers.
         * The view is not changed. If the index does not exist in the container,
         * the last slide in the view is selected.
         *
         * @param {Number} index
         *  The index in the order container.
         */
        this.changeToSlideAtIndex = function (index) {

            var // the container that contains the order of the slides
                orderContainer = isMasterView ? masterSlideOrder : slideOrder;

            if (orderContainer[index]) {
                self.changeToSlide(orderContainer[index]);
            } else {
                self.changeToLastSlideInView();
            }

        };

        /**
         * Activating the first slide in the current view.
         */
        this.changeToFirstSlideInView = function () {
            self.changeToSlide(isMasterView ? _.first(masterSlideOrder) : _.first(slideOrder));
        };

        /**
         * Activating the last slide in the current view.
         */
        this.changeToLastSlideInView = function () {
            self.changeToSlide(isMasterView ? _.last(masterSlideOrder) : _.last(slideOrder));
        };

        /**
         * Modifying the selection in that way, that only complete paragraphs are selected. The start
         * position has to be set to the beginning of its current paragraph and the end position has
         * to be set to the end of the current paragraph.
         *
         * This function is currently used for generating setAttribute operations, if the selection
         * is inside a layout or master slide. In this case a character attribute is assigned to the
         * complete paragraph.
         *
         * This automatic expansion of selection is NOT applied to header or footer place holder types.
         * In this case the selection is used as it is, so that only parts of the text inside a paragraph
         * can get a specific character attribute assigned
         * (see PresentationModel.NOT_COMPLETE_SELECTION_PLACEHOLDER_TYPES).
         */
        this.setMasterLayoutSelection = function () {

            var // the logical start position
                startPos = _.clone(selection.getStartPosition()),
                // the logical end position
                endPos = _.clone(selection.getEndPosition());

            // checking if the selection is inside a place holder drawing
            if (selection.isAdditionalTextframeSelection() && PresentationUtils.isPlaceHolderDrawing(selection.getSelectedTextFrameDrawing()) && !_.contains(PresentationModel.NOT_COMPLETE_SELECTION_PLACEHOLDER_TYPES, PresentationUtils.getPlaceHolderDrawingType(selection.getSelectedTextFrameDrawing()))) {

                // saving the current selection
                masterLayoutSelection = [_.clone(startPos), _.clone(endPos)];

                // expanding the selection
                startPos[startPos.length - 1] = 0;
                endPos[endPos.length - 1] = Position.getParagraphLength(self.getCurrentRootNode(), _.initial(endPos));
                selection.setTextSelection(startPos, endPos);
            }

        };

        /**
         * After a setAttributes operation on a master or layout slide, it is necessary to update
         * the styles defined at the drawing/slide.
         * After this 'styling' update, all slides that used this style need to be updated.
         *
         * @param {HTMLElement|jQuery} element
         *  The element node, that received the new attributes.
         *
         * @param {String} target
         *  The active target for the specified element.
         *
         * @param {String} family
         *  The family of the attribute set.
         *
         * @param {Object} attributes
         *  A map with formatting attribute values, mapped by the attribute
         *  names, and by attribute family names.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.immediateUpdate=true]
         *      Whether slides that are dependent from the layout or master slide change
         *      shall be updated immediately. Default is the immediate update.
         *  @param {String} [options.saveOldAttrs=false]
         *      Whether the old attributes shall be saved and returned in the return object.
         *
         * @returns {Object|Null}
         *  If the attributes are registered, an object with the complete registration
         *  information is returned. The object's keys are: 'target', 'placeHolderType',
         *  'placeHolderIndex', 'attrs' and 'oldAttrs'. If the attributes were not registered,
         *  because target, type or index are not defined, null is returned.
         */
        this.updateMasterLayoutStyles = function (element, target, family, attributes, options) {

            // restoring a previous selection (if user has edit privileges)
            if (masterLayoutSelection) {
                if (target && self.isLayoutOrMasterId(target) && self.getEditMode()) { selection.setTextSelection.apply(selection, masterLayoutSelection); }
                masterLayoutSelection = null;
            }

            return registerPlaceHolderAttributes(element, target, family, attributes, options);
        };

        /**
         * Updating the model for the 'slide' family.
         *
         * @param {Number[]} start
         *  The logical position of the slide, whose attributes will be updated.
         *
         * @param {Object} attrs
         *  The object containing the attributes.
         *
         * @param {String} [target]
         *  The target of the slide, if exists.
         *
         * @returns {Object|null}
         *  If the attributes are registered, an object with the complete registration
         *  information is returned. The object's keys are: 'id' and 'attrs'.
         *  If the attributes were not registered, null is returned.
         */
        this.updateSlideFamilyModel = function (start, attrs, target) {

            var // the slide id
                slideId = target ? target : slideOrder[_.first(start)];

            return self.saveSlideAttributes(slideId, attrs);
        };

        /**
         * Merging the place holder attributes into a specified attribute object.
         *
         * @param {HTMLElement|jQuery} element
         *  The element node, that received the new attributes.
         *
         * @param {String} family
         *  The family of the attribute set.
         *
         * @param {Object} attrs
         *  A map with formatting attribute values, mapped by the attribute
         *  names, and by attribute family names.
         *
         * @param {Object} placeHolderAttrs
         *  A map with formatting attribute values, mapped by the attribute
         *  names, and by attribute family names that is saved in the place holder model.
         *
         * @returns {Object}
         *  An object with the merged attributes of the specified attributes
         *  and the specified place holder attributes.
         */
        this.mergePlaceHolderAttributes = function (element, family, attrs, placeHolderAttrs) {

            var // a paragraph node
                paragraph = null,
                // the paragraph level
                paraLevel = 0,
                // the key in the list styles corresponding to the paragraph level
                paraKey = null,
                // whether drawing attributes shall be registered
                isDrawing = (family === 'drawing'),
                // the merged attributes object
                mergedAttrs = attrs;

            if (isDrawing) {
                if (placeHolderAttrs[family]) {
                    mergedAttrs = self.extendAttributes(placeHolderAttrs, attrs);
                }
            } else {
                if (placeHolderAttrs.listStyle) {
                    paragraph = (family === 'paragraph') ? $(element) : $(element).closest(DOM.PARAGRAPH_NODE_SELECTOR);
                    paraLevel = paragraphStyles.getParagraphLevel(paragraph);
                    paraKey = paragraphStyles.generateKeyFromLevel(paraLevel);

                    if (placeHolderAttrs.listStyle[paraKey] && placeHolderAttrs.listStyle[paraKey][family]) {
                        mergedAttrs = self.extendAttributes(placeHolderAttrs.listStyle[paraKey], attrs);
                    }
                }
            }

            return mergedAttrs;
        };

        /**
         * Public function to force an update of slides after a change of a master or layout
         * slide.
         *
         * @param {Object} data
         *  This object is generated inside the function 'savePlaceHolderAttributes' in the
         *  drawing styles. The object's keys are: 'target', 'placeHolderType', 'placeHolderIndex'
         *  and 'attrs'.
         */
        this.forceMasterLayoutUpdate = function (data) {
            triggerLayoutChangeEvent(data);
        };

        /**
         * Triggering the event 'slidestate:update' with one specified slide id. Typically this
         * event is only triggered after an 'operations:after' event. But there are some special
         * cases, in which the slide pane needs to be updated without an operation. For example,
         * if a template text is inserted into a place holder drawing on a layout slide (44991).
         *
         * @param {String} id
         *  The id of the slide that need to be updated in the slide pane.
         */
        this.forceTriggerOfSlideStateUpdateEvent = function (id) {
            debouncedUpdateSlideStateHandler(id);
        };

        /**
         * Returning the ratio of width divided by height of the page.
         *
         * @returns {Number}
         *  The ratio of width divided by height of the page. If it cannot be determined, 1 is returned.
         */
        this.getSlideRatio = function () {

            var // the attributes of the page
                pageAttributes = pageStyles.getElementAttributes(self.getNode()).page;

            return pageAttributes && pageAttributes.width && pageAttributes.height ? Utils.round(pageAttributes.width / pageAttributes.height, 0.0001) : 1;
        };

        /**
         * Returning the width and height of the slide in the document.
         *
         * @returns {Object}
         *  The width and height in Hmm from the slide in the document.
         */
        this.getSlideDocumentSize = function () {

            var // the attributes of the page
                pageAttributes = pageStyles.getElementAttributes(self.getNode()).page;

            return (pageAttributes && pageAttributes.width && pageAttributes.height) ? { width: pageAttributes.width, height: pageAttributes.height } : null;
        };

        /**
         * Check, whether at a specified position the list auto detection shall be executed. In
         * the text application there are not limitations yet.
         *
         * @returns {Boolean}
         *  Whether the list auto detection can be executed in the specified paragraph.
         */
        this.isForcedHardBreakPosition = function (paragraph) {
            return PresentationUtils.isTitlePlaceHolderDrawing($(paragraph).closest('div.drawing'));
        };

        /**
         * Check, whether the specified drawing is a drawing that contains place holder attributes. This means, that
         * the phType is set, or that the phIndex is specified. It is possible, that only the phIndex is specified. In
         * this case the phType is defaulted to 'body'.
         *
         * @param {HTMLElement|jQuery} drawing
         *  The drawing node.
         *
         * @returns {Boolean}
         *  Whether the specified drawing contains presentation place holder attributes.
         */
        this.isPlaceHolderDrawing = function (drawing) {
            return Boolean(PresentationUtils.isPlaceHolderDrawing(drawing));
        };

        /**
         * Returns the layout attributes of the document.
         *
         * @returns {Object}
         *  The layout attributes.
         */
        this.getLayoutAttributes = function () {
            return self.getDefaultAttributes('layout');
        };

        /**
         * Getting the start position of a layout slide relative to its master slide. This
         * is a zero-based number value, not a logical position. It can be used in the
         * operation 'moveLayoutSlide'. The first layout slide behind the master slide
         * has the value 0.
         *
         * @param {id} String
         *  The ID of the layout slide.
         *
         * @returns {Number}
         *  The zero based position of the layout slide behind its master slide. If it cannot
         *  be determined, -1 is returned.
         */
        this.getLayoutSlideStartPosition = function (id) {

            var // the zero based position of the layout slide relative to its master slide
                start = -1,
                // the ID of the master slide
                masterId = null;

            if (self.isLayoutSlideId(id)) {

                masterId = self.getMasterSlideId(id);

                if (masterId) {
                    start = self.getSortedSlideIndexById(id) - self.getSortedSlideIndexById(masterId) - 1; // zero-based value
                }

            }

            return start;
        };

        /**
         * Getting the start position of a master slide relative to all other master slides.
         * This is a zero-based number value, not a logical position. It can be used in the
         * operation 'insertMasterSlide'.
         *
         * @param {id} String
         *  The ID of the master slide.
         *
         * @returns {Number}
         *  The zero based position of the master slide relative to all other master slides.
         *  If it cannot be determined, -1 is returned.
         */
        this.getMasterSlideStartPosition = function (id) {

            var // the zero based position of the layout slide relative to its master slide
                start = -1,
                // a helper object with all informations about the master slide indices in the model 'masterSlideOrder'
                indexObject = null;

            if (self.isMasterSlideId(id)) {

                // a helper object with all informations about the master slide indices in the model 'masterSlideOrder'
                indexObject = self.getAllMasterSlideIndices();

                if (indexObject[id]) { start = _.indexOf(indexObject.order, indexObject[id]); }
            }

            return start;

        };

        /**
         * Check, whether a specified slide ID is in the container of the modified slide IDs.
         * Modified means, that the slide was modified since the last trigger of the event
         * 'slidestate:update'.
         *
         * @param {String} id
         *  The ID of that slide that is checked for modification.
         *
         * @returns {Boolean}
         *  Whether the specified id is in the collector of slide IDs of all modified slides.
         */
        this.isModifiedSlideID = function (id) {
            return _.contains(allModifiedSlideIDs, id);
        };

        /**
         * Getter for the selection box object, that can be used to create a specified
         * rectangle on the application content root node.
         *
         * @returns {Object}
         *  The selection box object.
         */
        this.getSelectionBox = function () {
            return selectionBox;
        };

        /**
         * Handling the page up and page down event, independent
         * from the current cursor position.
         *
         * @param {jQuery.Event} event
         *  A jQuery event object.
         *
         * @returns {Boolean}
         *  Whether the event was handled within this function.
         */
        this.handleSlideChangeByPageUpDown = function (event) {

            var // whether the event was handled within this function
                handledEvent = false;

            if (event.keyCode === KeyCodes.PAGE_UP) {
                if (!self.isFirstSlideActive()) {
                    self.changeToNeighbourSlide({ next: false });
                }
                handledEvent = true;
            } else if (event.keyCode === KeyCodes.PAGE_DOWN) {
                if (!self.isLastSlideActive()) {
                    self.changeToNeighbourSlide({ next: true });
                }
                handledEvent = true;
            }

            return handledEvent;
        };

        /**
         * Modifying the active silde, if the user uses a cursor key and the selection is
         * a slide selection ('isTopLevelTextCursor'). In this case the active slide needs
         * to be modified.
         *
         * @param {jQuery.Event} event
         *  A jQuery event object.
         */
        this.setActiveSlideByCursorEvent = function (event) {

            if (!self.getSelection().isTopLevelTextCursor() || !DOM.isCursorKey(event.keyCode)) { return; }

            if (event.keyCode === KeyCodes.LEFT_ARROW || event.keyCode === KeyCodes.UP_ARROW || event.keyCode === KeyCodes.PAGE_UP || event.keyCode === KeyCodes.HOME) {
                // switch to previous or first slide
                if (!self.isFirstSlideActive()) {
                    if (event.keyCode === KeyCodes.HOME) {
                        self.changeToFirstSlideInView();
                    } else {
                        self.changeToNeighbourSlide({ next: false });
                    }
                }
            } else if (event.keyCode === KeyCodes.RIGHT_ARROW || event.keyCode === KeyCodes.DOWN_ARROW || event.keyCode === KeyCodes.PAGE_DOWN || event.keyCode === KeyCodes.END) {
                // switch to next or last slide
                if (!self.isLastSlideActive()) {
                    if (event.keyCode === KeyCodes.END) {
                        self.changeToLastSlideInView();
                    } else {
                        self.changeToNeighbourSlide({ next: true });
                    }
                }
            }

        };

        /**
         * Returns whether a multi selection contains at least one place holder drawing.
         *
         * @returns {Boolean}
         *  Whether a multi selection contains at least one place holder drawing.
         */
        this.isAtLeastOnePlaceholderInMultiSelection = function () {

            var // a place holder drawing
                placeHolder = _.find(selection.getArrayOfSelectedDrawingNodes(), function (drawing) {
                    return PresentationUtils.isPlaceHolderDrawing(drawing);
                });

            return !!placeHolder;
        };

        /**
         * Getting the name for the 'insert'-operation for the specified node.
         * In the case of a slide, this can be one of:
         * - Operations.SLIDE_INSERT
         * - Operations.LAYOUT_SLIDE_INSERT
         * - Operations.MASTER_SLIDE_INSERT
         *
         * @param {Node|jQuery|Null} node
         *  The DOM node to be checked. If this object is a jQuery collection, uses
         *  the first DOM node it contains. If missing or null, returns empty string.
         *
         * @returns {Object|Null}
         *  An object containing required data for the insert operation of the slide:
         *   - 'name': The name of the required 'insert'-operation.
         *   - 'index': The index for a master or layout slide.
         */
        this.getInsertOperationNameForNode = function (node, options) {

            var // the name of the operation
                operationName = null,
                // the index value required for master and layout slides
                operationIndex = null,
                // the id of the slide
                id = self.getSlideId(node);

            if (self.isMasterSlideId(id)) {
                operationName = Operations.MASTER_SLIDE_INSERT;
                operationIndex = getOperationIndexForSlideById(id);
                if (options) { options.id = id; }  // saving also the slide ID at the options object
            } else if (self.isLayoutSlideId(id)) {
                operationName = Operations.LAYOUT_SLIDE_INSERT;
                operationIndex = getOperationIndexForSlideById(id);
                if (options) { options.id = id; }  // saving also the slide ID at the options object
            } else {
                operationName = Operations.SLIDE_INSERT;
            }

            return operationName ? { name: operationName, index: operationIndex } : null;
        };

        /**
         * Getting the default text for an empty text frame drawing.
         * This function is application specific.
         *
         * @param {HTMLElement|jQuery} drawing
         *  The drawing node.
         *
         * @returns {String|Null}
         *  The default text for a place holder drawing. Or null, if it cannot be determined.
         */
        this.getDefaultTextForTextFrame = function (drawing) {

            // drawing must be a place holder drawing.
            // -> text is dependent from the place holder type

            var // the template text
                templateText = null;

            if (drawing && PresentationUtils.isPlaceHolderDrawing(drawing)) {
                templateText = PresentationUtils.getPlaceHolderTemplateText(drawing);
            }

            return templateText;
        };

        /**
         * Changing the visibility of a slide, that is specified by a drawing that is
         * located in the slide. This is necessary for formatting reasons, where the
         * slide must not be invisible. In this case the drawing dimensions would not
         * be correct.
         *
         * @param {HTMLElement|jQuery} node
         *  The node in the affected slide or the slide itself.
         *
         * @param {Object} options
         *  Optional parameters:
         *  @param {Boolean} [options.makeVisible=true]
         *   Whether the slide shall be made visible or not.
         *
         * @returns {Boolean}
         *  Whether the visibility of the slide has changed.
         */
        this.handleContainerVisibility = function (node, options) {

            var // whether the visibility was changed
                visibilityChanged = false,
                // the slide node (containing the specified node)
                slide = DOM.isSlideNode(node) ? $(node) : $(node).closest(DOM.SLIDE_SELECTOR),
                // whether the container shall be made visible or not
                makeVisible = Utils.getBooleanOption(options, 'makeVisible', true),
                // activates the "invisiblewithreflow" which is only active if slide is invible
                // instead of display none, it uses visibility "hidden" to keep dom-layout
                invisibleWithReflow = Utils.getBooleanOption(options, 'invisibleWithReflow', true);

            if (slide && slide.length > 0) {

                if (makeVisible && slide.hasClass('invisibleslide')) {
                    slide.removeClass('invisibleslide'); // making slide visible for formatting
                    visibilityChanged = true;
                } else if (!makeVisible && !slide.hasClass('invisibleslide')) {
                    slide.addClass('invisibleslide'); // making slide invisible after formatting
                    visibilityChanged = true;
                }

                if (invisibleWithReflow & !slide.hasClass('invisiblewithreflow')) {
                    slide.addClass('invisiblewithreflow');
                } else if (!invisibleWithReflow && slide.hasClass('invisiblewithreflow')) {
                    slide.removeClass('invisiblewithreflow');
                }
            }

            return visibilityChanged;
        };

        /**
         * Getting the selection in the slide pane.
         *
         * @returns {Number[]}
         *  An Array containing the position as index from all selected slides in the slide pane.
         */
        this.getSlidePaneSelection = function () {
            return app.getView().getSlidePane().getSlidePaneSelection();
        };

        /**
         * Receiving some data from the model, that need to be restored after applying a snap shot.
         *
         * @returns {Object}
         *  An object containing some data from the model, that need to be restored after applying a snap shot.
         */
        this.getModelDataObject = function () {
            return { mainSlideCounter: mainSlideCounter, activeSlideId: activeSlideId, isMasterView: isMasterView };
        };

        /**
         * Setting some data into the model during applying a snapshot.
         *
         * @param {Object} [data]
         *  An optional object, that contains the values of some data from the model.
         */
        this.setModelDataObject = function (data) {

            if (data) {
                if (_.isNumber(data.mainSlideCounter)) { mainSlideCounter = data.mainSlideCounter; }
                if (_.isString(data.activeSlideId)) { activeSlideId = data.activeSlideId; }
                isMasterView = !!data.isMasterView;
            }

        };

        /**
         * Getting the paragraph that contains the current selection.
         *
         * @returns {jQuery}
         *  The paragraph that contains the current selection. If there is
         *  no paragraph, the jQuery collection contains no element.
         */
        this.getSelectedParagraph = function () {
            return selectedParagraph;
        };

        /**
         * Setting the paragraph that contains the current selection.
         *
         * @param {jQuery} [para]
         *  Setting the paragraph that contains the current selection. If the
         *  paragraph is not specified, the global 'selectedParagraph' is set
         *  to an empty jQuery collection.
         */
        this.setSelectedParagraph = function (para) {
            selectedParagraph = para || $();
        };

        /**
         * Tries to find layout id with queried slide type. If no id is found, returns first layout id from list.
         *
         * @param {String} type
         *  Queried type of slide.
         * @returns {String|null}
         */
        this.getLayoutIdFromType = function (type) {
            var layoutIDs = self.getAllLayoutSlideIds();
            var first = _.first(layoutIDs) || null;

            var result = _.find(layoutIDs, function (layoutID) {
                var slideType = self.getSlideFamilyAttributeForSlide(layoutID, 'slide', 'type');

                if (slideType === type) {
                    return layoutID;
                }
            });

            return result || first;
        };

        // registering the handler functions for the operations ---------------

        this.registerOperationHandler(Operations.MASTER_SLIDE_INSERT, self.insertMasterSlideHandler);
        this.registerOperationHandler(Operations.LAYOUT_SLIDE_INSERT, self.insertLayoutSlideHandler);
        this.registerOperationHandler(Operations.SLIDE_INSERT, self.insertSlideHandler);
        this.registerOperationHandler(Operations.DRAWING_INSERT, self.insertDrawingHandler);
        this.registerOperationHandler(Operations.TABLE_INSERT, self.insertTableHandler);
        this.registerOperationHandler(Operations.LAYOUT_SLIDE_MOVE, self.moveLayoutSlideHandler);
        this.registerOperationHandler(Operations.LAYOUT_CHANGE, self.changeLayoutHandler);
        this.registerOperationHandler(Operations.SLIDE_MOVE, self.moveSlideHandler);
        this.registerOperationHandler(Operations.MOVE, self.moveHandler);

        // initialization -----------------------------------------------------

        pagediv = self.getNode();
        selection = self.getSelection();

        // setting style collectors
        characterStyles = self.getStyleCollection('character');
        paragraphStyles = self.getStyleCollection('paragraph');
        tableStyles = self.getStyleCollection('table');
        tableRowStyles = self.getStyleCollection('row');
        tableCellStyles = self.getStyleCollection('cell');
        drawingStyles = self.getStyleCollection('drawing');
        slideStyles = self.getStyleCollection('slide');
        pageStyles = self.getStyleCollection('page');

        // the editor.js has an own drawing style object with own format handler -> that need to be overwritten
        self.setCharacterStyles(characterStyles);
        self.setParagraphStyles(paragraphStyles);
        self.setSlideStyles(slideStyles);
        self.setDrawingStyles(drawingStyles);
        self.setTableStyles(tableStyles);
        self.setTableRowStyles(tableRowStyles);
        self.setTableCellStyles(tableCellStyles);
        self.setPageStyles(pageStyles);

        // setting the handler function for moving and resizing drawings
        self.getSelection().setDrawingResizeHandler(DrawingResize.drawDrawingSelection);

        // setting the handler function for updating lists (this is used during 'updateDocumentFormatting')
        self.setUpdateLists(self.getUpdateListsHandler());

        // setting the manager that handles formatting of slides (for performance reasons)
        slideFormatManager = new SlideFormatManager(app);

        // setting handler for handling fields in presentation
        self.setFieldManager(new SlideField(app));

        // setting handler for handling fields in presentation
        self.setNumberFormatter(new NumberFormatter(this));

        self.waitForImportSuccess(importSuccessHandler);

        self.one('fastload:done', updatePresentationModel);

//        // register the listener for 'textframeHeight:update'.
//        self.listenTo(self, 'textframeHeight:update', updateVerticalDrawingPosition);
//
//        // register the listener for 'addtionalTextFrame:selected' of the selection
//        self.listenTo(selection, 'addtionalTextFrame:selected', saveVerticalPositionAttributes);

        // registering the default list styles
        self.listenTo(self, 'change:defaults', registerDefaultTextListStyles);

        // registering the handler for document reload event. This happens, if the user cancels a long running action
        self.listenTo(self, 'document:reloaded', documentReloadHandler);

        // registering the handler for the event 'image:loaded'
        self.listenTo(self, 'image:loaded', imageLoadedHandler);

        // registering handler for undo:after and redo:after
        self.getUndoManager().on('undo:after redo:after', undoRedoAfterHandler);

        // registering process mouse down handler at the page
        pagediv.on({ 'mousedown touchstart': self.getPageProcessMouseDownHandler() });
        if (self.getListenerList()) { self.getListenerList()['mousedown touchstart'] = self.getPageProcessMouseDownHandler(); }

        // destroy all class members on destruction
        this.registerDestructor(function () {

            // selection.destroy();
            if (selectionBox) { selectionBox.deRegisterSelectionBox(); }
            slideFormatManager.destroy();
            self = selectionBox = pagediv = slideFormatManager = null;
            characterStyles = paragraphStyles = tableStyles = tableCellStyles = tableRowStyles = drawingStyles = slideStyles = pageStyles = null;
        });

    } // class PresentationModel

    // constants --------------------------------------------------------------

    /**
     * The attributes that are used for new inserted drawings.
     */
    PresentationModel.INSERT_DRAWING_ATTRIBUTES = {
        left: 5000,
        top: 3000
    };

    /**
     * The type used to recognize master slides.
     */
    PresentationModel.TYPE_MASTER_SLIDE = 'masterSlide';

    /**
     * The type used to recognize layout slides.
     */
    PresentationModel.TYPE_LAYOUT_SLIDE = 'layoutSlide';

    /**
     * The type used to recognize 'standard' slides.
     */
    PresentationModel.TYPE_SLIDE = 'standardSlide';

    /**
     * A CSS filter to specify the event target for the selection box to
     * create multi drawing selections.
     */
    PresentationModel.SET_SELECTION_MODE = 'setselection';

    /**
     * A CSS filter to specify the event target for the selection box to
     * create multi drawing selections.
     */
    PresentationModel.MULTI_SELECTION_BOX_FILTER = DOM.SLIDE_SELECTOR + ', ' + DOM.APP_CONTENT_ROOT_SELECTOR;

    /**
     * A list of those place holder drawing types, for that the selection is NOT automatically expanded
     * to the complete paragraph, if character attributes are assigned.
     */
    PresentationModel.NOT_COMPLETE_SELECTION_PLACEHOLDER_TYPES = ['ftr'];

    /**
     * A list of all operations, that insert a slide
     */
    PresentationModel.INSERT_SLIDE_OPERATIONS = [Operations.SLIDE_INSERT, Operations.LAYOUT_SLIDE_INSERT, Operations.MASTER_SLIDE_INSERT];

    // exports ================================================================

    // derive this class from class EditModel
    return Editor.extend({ constructor: PresentationModel });

});
