/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/presentation/format/pagestyles', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/model/stylecollection'
], function (Utils, StyleCollection) {

    'use strict';

    var // definitions for page attributes
        DEFINITIONS = {

            /**
             * Total width of a single page, in 1/100 of millimeters.
             */
            width: { def: 28000 },

            /**
             * Total height of a single page, in 1/100 of millimeters.
             */
            height: { def: 21000 },

            /**
             * the actual paper format (portrait or landscape)
             */
            orientation: { def: 'landscape' },

            /**
             * Margin between left page border and editing area, in 1/100 of
             * millimeters.
             */
            marginLeft: { def: 0 },

            /**
             * Margin between right page border and editing area, in 1/100 of
             * millimeters.
             */
            marginRight: { def: 0 },

            /**
             * Margin between top page border and editing area, in 1/100 of
             * millimeters.
             */
            marginTop: { def: 0 },

            /**
             * Margin between bottom page border and editing area, in 1/100 of
             * millimeters.
             */
            marginBottom: { def: 0 },

            /**
             * Margin between top page border and top header border, in 1/100 of millimeters
             */
            marginHeader: { def: 0 },

            /**
             * Margin between bottom page border and bottom footer border, in 1/100 of millimeters
             */
            marginFooter: { def: 0 },

            /**
             * Whether to use separate header/footer settings for the first page.
             *
             */
            firstPage: { def: false },

            /**
             * Whether to use separate header/footer settings for even and odd pages, and mirror margin settings accordingly.
             */
            evenOddPages: { def: false }

        };

    // private global functions ===============================================

    /**
     * Will be called for every page whose attributes have been changed.
     *
     * @param {jQuery} page
     *  The page container element whose character attributes have been
     *  changed, as jQuery object.
     *
     * @param {Object} mergedAttributes
     *  A map of attribute maps (name/value pairs), keyed by attribute family,
     *  containing the effective attribute values merged from style sheets and
     *  explicit attributes.
     */
    function updatePageFormatting(page, mergedAttributes) {

        var // the page attributes of the passed attribute map
            pageAttributes = mergedAttributes.page,
            // effective page width (at least 2 cm)
            pageWidth = Math.max(pageAttributes.width, 2000),
            // effective page height (at least 2 cm)
            pageHeight = Math.max(pageAttributes.height, 2000);

        // Set CSS attributes.
        page.css({
            width: Utils.convertHmmToLength(pageWidth, 'px', 1),
            minHeight: Utils.convertHmmToLength(pageHeight, 'px', 1)
        });
    }

    // class PageStyles =======================================================

    /**
     * Contains the style sheets for page formatting attributes. The CSS
     * formatting will be read from and written to the page container elements.
     *
     * @constructor
     *
     * @extends StyleCollection
     *
     * @param {TextModel} docModel
     *  The text document model containing instance.
     */
    function PageStyles(docModel) {

        // base constructor ---------------------------------------------------

        StyleCollection.call(this, docModel, 'page', { formatHandler: updatePageFormatting });

        // initialization -----------------------------------------------------

        // register the attribute definitions for the style family
        docModel.registerAttributeDefinitions('page', DEFINITIONS);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = null;
        });

    } // class PageStyles

    // exports ================================================================

    // derive this class from class StyleCollection
    return StyleCollection.extend({ constructor: PageStyles });

});
