/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/presentation/components/drawing/drawingresize', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/tracking',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/presentation/utils/presentationutils'
], function (Utils, Tracking, AttributeUtils, DrawingFrame, Operations, DOM, Position, PresentationUtils) {

    'use strict';

    // the names of all tracking events but 'tracking:start'
    var TRACKING_EVENT_NAMES = 'tracking:move tracking:scroll tracking:end tracking:cancel';

    var ACTIVE_DRAGGING = 'activedragging';

    // static class DrawingSlideResize ========================================

    var DrawingSlideResize = {};

    // static methods ---------------------------------------------------------

    /**
     * Draws a selection box for the specified drawing node and registers
     * mouse handlers for moving and resizing.
     *
     * @param {PresentationApplication} app
     *  The application instance containing the drawing.
     *
     * @param {HTMLElement|jQuery} drawingNode
     *  The drawing node to be selected, as DOM node or jQuery object.
     */
    DrawingSlideResize.drawDrawingSelection = function (app, drawingNode) {

        var // the model object
            model = app.getModel(),
            // the selection object
            selection = model.getSelection(),
            // the current root node
            rootNode = model.getCurrentRootNode(),
            // the node containing the scroll bar
            scrollNode = app.getView().getContentRootNode(),
            // if it's text frame with table inside it
            containsTable = DrawingFrame.nodeContainsTable(drawingNode),
            // object containing information about movable and resizable
            options = { movable: true, resizable: true, rotatable: !containsTable },  // -> parameter?
            // the container element used to visualize the selection
            selectionBox = null,
            // saving the selected resize node
            resizeNode = null,
            //zoom factor in floating point notation
            zoomFactor,
            // target ID of currently active root node, if existing
            target = model.getActiveTarget(),
            // the resize nodes of the selection
            resizeHandles = null,
            // the rotate handle node of the selection
            rotateHandle = null,
            // an object collecting the attributes for an operation
            operationProperties = {},
            // whether this is a multi drawing selection
            isMultiSelection = false,
            // a collector for all move boxes and drawing informations
            allDrawings = null;

        // helper function to collect all move box information in multiple drawing selections
        function collectAndActivateAllMoveBoxes(options) {
            _.each(selection.getMultiSelection(), function (drawingSelection) {
                var drawingNode = selection.getDrawingNodeFromMultiSelection(drawingSelection);
                var drawingPos = selection.getStartPositionFromMultiSelection(drawingSelection);

                activateMoveBox(drawingNode, drawingPos, options);
            });
        }

        function activateMoveBox(drawingNode, drawingPos, options) {
            var evalRatio = options && options.resize || false; // don't evaluate lock ratio on move, just for resize

            var attrs = null;
            var lockResizeAspectRatio = false;

            drawingNode.addClass(ACTIVE_DRAGGING);

            if (evalRatio) {
                attrs = AttributeUtils.getExplicitAttributes(drawingNode);
                lockResizeAspectRatio = (attrs && attrs.drawing && attrs.drawing.noChangeAspect || Utils.SMALL_DEVICE) || false;
            }

            var oneMoveBox = cloneDiv(drawingNode);
            allDrawings.push({ drawingNode: drawingNode, moveBox: oneMoveBox, width: drawingNode.width(), height: drawingNode.height(), start: drawingPos, lockResizeAspectRatio: lockResizeAspectRatio });
        }

        function clearDrawing(drawing) {
            drawing.removeClass(ACTIVE_DRAGGING);
            drawing.find('>.copy').remove();
        }

        function cloneCanvas(canvas) {
            var img = document.createElement('img');
            img.src = canvas.toDataURL();
            img.width         = canvas.width;
            img.height        = canvas.height;
            img.className     = canvas.className;
            img.style.cssText = canvas.style.cssText;
            img.style.position = 'absolute';
            return img;
        }

        function cloneDiv(drawing) {
            drawing.find('>.copy').remove();

            var content = drawing.find('.content:not(.copy)');
            var clone = content.clone();
            drawing.append(clone);

            var orgCanvases = content.find('canvas').get();
            clone.find('canvas').get().forEach(function (canvas, index) {
                var img = cloneCanvas(orgCanvases[index]);
                $(canvas).replaceWith(img);
            });
            clone.css({
                pointerEvents: 'none',
                width: '100%',
                height: '100%',
                left: 0,
                top: 0
            });
            clone.addClass('copy');

            return clone;
        }

        // helper function to collect and prepare all drawings for rotation in multiple drawing selections
        function collectAndPrepareDrawingsForRotation() {
            _.each(selection.getMultiSelection(), function (drawingSelection) {
                var drawingNode = selection.getDrawingNodeFromMultiSelection(drawingSelection);
                var drawingPos;
                var startRotation;

                if (!DrawingFrame.nodeContainsTable(drawingNode)) {
                    drawingPos = selection.getStartPositionFromMultiSelection(drawingSelection);
                    startRotation = DrawingFrame.getDrawingRotationAngle(model, drawingNode);
                    allDrawings.push({ drawingNode: drawingNode, start: drawingPos, rotation: startRotation });
                }
            });
        }

        // helper function to convert degrees to radians. If no valid degree is passed, returns null.
        function convertDegreeToRadian(deg) {
            return _.isNumber(deg) && deg * Math.PI / 180 || null;
        }

        // helper function to normalize rotated drawings pixels shift for move. Returns coordinates object
        function getNormalizedMoveCoordinates(drawing, shiftXo, shiftYo) {
            var tempX = null;
            var tempY = null;
            var angle = DrawingFrame.getDrawingRotationAngle(model, drawing);
            var rad = convertDegreeToRadian(angle);

            if (rad) {
                tempX = shiftXo * Math.cos(rad) + shiftYo * Math.sin(rad);
                tempY = shiftYo * Math.cos(rad) - shiftXo * Math.sin(rad);
                shiftXo = tempX;
                shiftYo = tempY;
            }

            return { x: shiftXo, y: shiftYo };
        }

        // helper function to normalize drawing pixels delta for resize. Returns coordinates object
        function getNormalizedResizeDeltas(drawing, deltaXo, deltaYo, useX, useY, scaleX, scaleY) {
            var localX = null;
            var localY = null;
            var angle = DrawingFrame.getDrawingRotationAngle(model, drawing);
            var rad = convertDegreeToRadian(angle);

            if (rad) {
                localX = useX ? (deltaXo * Math.cos(rad) + deltaYo * Math.sin(rad)) * scaleX : 0;
                localY = useY ? (deltaYo * Math.cos(rad) - deltaXo * Math.sin(rad)) * scaleY : 0;
            } else {
                localX = useX ? deltaXo * scaleX : 0;
                localY = useY ? deltaYo * scaleY : 0;
            }

            return { x: localX, y: localY };
        }

        // helper function to normalize offset of rotated drawings. Returns object of top and left offset properties.
        function normalizeOffset(drawingNode, moveBox) {
            var moveBoxOffsetLeft = moveBox.offset().left;
            var moveBoxOffsetTop = moveBox.offset().top;
            var angle = DrawingFrame.getDrawingRotationAngle(model, drawingNode);
            var tLeft, tTop, dLeft, dTop;
            if (_.isNumber(angle)) {
                $(drawingNode).css({ transform: '' }); // reset to def to get values
                tLeft = $(drawingNode).offset().left;
                tTop = $(drawingNode).offset().top;
                $(drawingNode).css({ transform: 'rotate(' + angle + 'deg)' }); // return property to it's value

                dLeft = tLeft - $(drawingNode).offset().left;
                dTop = tTop - $(drawingNode).offset().top;

                moveBoxOffsetLeft += dLeft;
                moveBoxOffsetTop += dTop;
            }
            return { left: moveBoxOffsetLeft, top: moveBoxOffsetTop };
        }

        // helper function to normalize resize offset of rotated drawings. Returns object of top and left offset properties,
        // and sendMoveOperation flag, to send appropriate move operations if drawing is rotated and needs to be translated to the position.
        function normalizeResizeOffset(drawingNode, moveBox) {
            var moveBoxOffsetTop, moveBoxOffsetLeft;
            var sendMoveOperation = false;
            var angle = DrawingFrame.getDrawingRotationAngle(model, drawingNode);

            if (_.isNumber(angle)) {
                moveBox.css({ transform: 'rotate(' + (-angle) + 'deg)' });
                moveBoxOffsetLeft = moveBox.offset().left;
                moveBoxOffsetTop = moveBox.offset().top;
                moveBox.css({ transform: '' });

                sendMoveOperation = true;
            } else {
                moveBoxOffsetLeft = moveBox.offset().left;
                moveBoxOffsetTop = moveBox.offset().top;
            }

            return { left: moveBoxOffsetLeft, top: moveBoxOffsetTop, sendMoveOperation: sendMoveOperation };
        }

        function extendAndGenerateOp(generator, drawingNode, operationProperties, target) {
            PresentationUtils.extendPlaceholderProp(app, drawingNode, operationProperties);
            model.extendPropertiesWithTarget(operationProperties, target);
            generator.generateOperation(Operations.SET_ATTRIBUTES, operationProperties);

            model.updateDynFontSizeDebounced(drawingNode);
        }

        /**
         * Handling all tracking events while moving a drawing.
         */
        var drawingMoveHandler = (function () {

            var // the current position change (in px)
                shiftX = 0, shiftY = 0,
                // the current scroll position
                scrollX = 0, scrollY = 0,
                // the start scroll position
                startScrollX = 0, startScrollY = 0,
                // whether the move box was at least once painted
                moveBoxPainted = false;

            // initialing the moving of the drawing according to the passed tracking event
            function startMoveDrawing() {

                zoomFactor = app.getView().getZoomFactor() / 100;

                startScrollX = scrollNode.scrollLeft();
                startScrollY = scrollNode.scrollTop();

                moveBoxPainted = false;

                isMultiSelection = selection.isMultiSelectionSupported() && selection.isMultiSelection();

                allDrawings = []; // 'tracking:start' event is triggered twice consecutively, so we clean collection to prevent doubling

                if (isMultiSelection) {
                    collectAndActivateAllMoveBoxes();
                } else {
                    var drawingPos = Position.getOxoPosition(rootNode, drawingNode, 0);
                    activateMoveBox(drawingNode, drawingPos);
                }
            }

            // updating the drawing position according to the passed tracking event
            function updateMove(event) {
                // make move box visible when tracking position has moved
                DrawingFrame.toggleTracking(drawingNode, true);

                // reading scrollPosition again. Maybe it was not updated or not updated completely.
                scrollX = scrollNode.scrollLeft() - startScrollX;
                scrollY = scrollNode.scrollTop() - startScrollY;

                shiftX = (event.pageX - event.startX + scrollX) / zoomFactor;
                shiftY = (event.pageY - event.startY + scrollY) / zoomFactor;

                // only move the moveBox, if the mouse was really moved
                if ((_.isNumber(shiftX)) && (_.isNumber(shiftY)) && (shiftX !== 0) || (shiftY !== 0)) {

                    var updateMoveBox = function (oneDrawing) {
                        var normalizedCoords = getNormalizedMoveCoordinates(oneDrawing.drawingNode, shiftX, shiftY);
                        oneDrawing.moveBox.css({ cursor: event.cursor, left: normalizedCoords.x, top: normalizedCoords.y, width: oneDrawing.width, height: oneDrawing.height });
                    };

                    moveBoxPainted = true;

                    _.each(allDrawings, updateMoveBox);
                }
            }

            // updates scroll position according to the passed tracking event
            function updateMoveScroll(event) {

                // update scrollPosition with suggestion from event
                if (event.scrollX) {
                    scrollNode.scrollLeft(scrollNode.scrollLeft() + event.scrollX);
                }

                if (event.scrollY) {
                    scrollNode.scrollTop(scrollNode.scrollTop() + event.scrollY);
                }

                scrollX = scrollNode.scrollLeft() - startScrollX;
                scrollY = scrollNode.scrollTop() - startScrollY;
            }

            // handling the drawing position, when moving is stopped according to the passed tracking event
            function stopMoveDrawing(event) {

                var // the operations generator
                    generator = model.createOperationsGenerator(),
                    //
                    availablePos = null,
                    //
                    newPoses = null,
                    //
                    dragToCopy = event.ctrlKey,
                    //
                    finalizeDrawing = function (oneDrawing) {

                        var start = oneDrawing.start;

                        if (dragToCopy) {
                            if (!availablePos) {
                                availablePos = model.getNextAvailablePositionInActiveSlide();
                                newPoses = [];
                            } else {
                                availablePos[1]++;
                            }
                            start = _.clone(availablePos);
                            newPoses.push(start);

                            var node = oneDrawing.drawingNode;
                            if (DOM.isDrawingPlaceHolderNode(node)) { node = DOM.getDrawingPlaceHolderNode(node); }

                            // generate operations for the drawing (including its attributes)
                            generator.generateDrawingOperations(node, start, { ignoreFamily: 'changes', target: target });
                        }

                        var normalizedOffset = normalizeOffset(oneDrawing.drawingNode, oneDrawing.moveBox);

                        // setting the new attributes of the drawing
                        operationProperties = {
                            start: start,
                            attrs: { drawing: {
                                left: Utils.convertLengthToHmm((normalizedOffset.left - rootNode.offset().left) / zoomFactor, 'px'),
                                top: Utils.convertLengthToHmm((normalizedOffset.top - rootNode.offset().top) / zoomFactor, 'px') }
                            }
                        };
                        extendAndGenerateOp(generator, oneDrawing, operationProperties, target);
                    };

                if (moveBoxPainted && !($(drawingNode).parent().is('.notselectable'))) { // only move drawing, if it is selectable

                    _.each(allDrawings, finalizeDrawing);

                    // apply the operations (undo group is created automatically)
                    model.applyOperations(generator);

                    if (dragToCopy) {
                        model.getSelection().setMultiDrawingSelectionByPosition(newPoses);
                    }
                }
            }

            // finalizes the move tracking
            function finalizeMoveDrawing() {
                leaveTracking();
                drawingNode.off(TRACKING_EVENT_NAMES);

                // Resetting variables for new mouseup events without mousemove
                shiftX = shiftY = scrollX = scrollY = 0;

                isMultiSelection = false;
                _.each(allDrawings, function (oneDrawing) {
                    oneDrawing.moveBox.css({ left: 0, top: 0, width: 0, height: 0 }).toggle(false);
                    clearDrawing(oneDrawing.drawingNode);
                });
                allDrawings = [];
            }

            function updateMoveAlways(event) {
                var dragToCopy = event.ctrlKey;
                event.cursor = dragToCopy ? 'copy' : 'move';
                // model.getNode().toggleClass('drag-to-copy', dragToCopy);
            }

            // return the actual drawingMoveHandler() function
            return function (event) {

                // handling for text frame drawings, if the click happened inside the internal text frame element
                if ($(event.target).closest(DrawingFrame.TEXTFRAME_NODE_SELECTOR).length > 0) {

                    Tracking.disableTracking(drawingNode)
                        .off('tracking:start', trackingStartHandler)
                        .off(TRACKING_EVENT_NAMES);

                    return;
                }

                switch (event.type) {
                    case 'tracking:start':
                        startMoveDrawing(event);
                        break;
                    case 'tracking:move':
                        updateMove(event);
                        break;
                    case 'tracking:scroll':
                        updateMoveScroll(event);
                        break;
                    case 'tracking:end':
                        stopMoveDrawing(event);
                        finalizeMoveDrawing(event);
                        break;
                    case 'tracking:cancel':
                        finalizeMoveDrawing(event);
                        break;
                }

                updateMoveAlways(event);
            };

        }()); // end of drawingMoveHandler() local scope

        /**
         * Handles all tracking events while resizing a drawing.
         */
        var drawingResizeHandler = (function () {

            var // the size of the resized drawing (in px)
                finalWidth = 0, finalHeight = 0,
                // the current scroll position
                scrollX = 0, scrollY = 0,
                // the initial scroll position
                startScrollX = 0, startScrollY = 0,
                // whether resizing is available in horizontal/vertical direction
                useX = false, useY = false,
                // whether resizing is available in left or top direction
                useLeft = false, useTop = false,
                // correction factor for resizing to the left/top
                scaleX = 0, scaleY = 0;

            // initializes resizing the drawing according to the passed tracking event
            function startResizeDrawing(event) {

                var // evaluating which resize element is active
                    pos = $(event.target).attr('data-pos');

                // helper function to set position properties at move box(es)
                function setPositionAtMoveBox(oneMoveBox) {

                    if (useLeft) {
                        scaleX = -1;
                        oneMoveBox.css({ left: 'auto', right: 0 });
                    } else {
                        scaleX = 1;
                        oneMoveBox.css({ left: 0, right: 'auto' });
                    }

                    if (useTop) {
                        scaleY = -1;
                        oneMoveBox.css({ top: 'auto', bottom: 0 });
                    } else {
                        scaleY = 1;
                        oneMoveBox.css({ top: 0, bottom: 'auto' });
                    }

                }

                zoomFactor = app.getView().getZoomFactor() / 100;

                startScrollX = scrollNode.scrollLeft();
                startScrollY = scrollNode.scrollTop();

                // collecting information about the handle node
                useX = /[lr]/.test(pos);
                useY = /[tb]/.test(pos);

                useLeft = /[l]/.test(pos);
                useTop = /[t]/.test(pos);

                isMultiSelection = selection.isMultiSelectionSupported() && selection.isMultiSelection();

                if (isMultiSelection) {
                    collectAndActivateAllMoveBoxes({ resize: true });
                } else {
                    var drawingPos = Position.getOxoPosition(rootNode, drawingNode, 0);
                    activateMoveBox(drawingNode, drawingPos, { resize: true });
                }
                _.each(allDrawings, function (oneDrawing) {
                    setPositionAtMoveBox(oneDrawing.moveBox);
                    oneDrawing.drawingNode.addClass(ACTIVE_DRAGGING);
                });
            }

            // updates scroll position according to the passed tracking event
            function updateResizeScroll(event) {

                // update scrollPosition with suggestion from event
                scrollNode
                    .scrollLeft(scrollNode.scrollLeft() + event.scrollX)
                    .scrollTop(scrollNode.scrollTop() + event.scrollY);

                scrollX = scrollNode.scrollLeft() - startScrollX;
                scrollY = scrollNode.scrollTop() - startScrollY;
            }

            // updates resizing the drawing according to the passed tracking event
            function updateResize(event) {

                var // the horizontal shift
                    deltaX = (event.pageX / zoomFactor - event.startX / zoomFactor + scrollX),
                    // the vertical shift
                    deltaY = (event.pageY / zoomFactor - event.startY / zoomFactor + scrollY),
                    // the scaling factor for the width
                    scaleWidth = 1,
                    // the scaling factor for the height
                    scaleHeight = 1;

                // normalize and scale deltas of the dimensions
                var normalizedDeltas = getNormalizedResizeDeltas(drawingNode, deltaX, deltaY, useX, useY, scaleX, scaleY);

                // helper function to set the current width and height of each selected drawing
                function updateDrawingDimension(drawingNode, moveBox, oldWidth, oldHeight, lockAspectRatio) {

                    // update drawing size
                    finalWidth = Math.max(0, oldWidth + normalizedDeltas.x);
                    finalHeight = Math.max(0, oldHeight + normalizedDeltas.y);

                    // use the same scaling factor for vertical and horizontal resizing, if both are enabled
                    // -> the larger number wins
                    if (useX && useY && (lockAspectRatio || event.shiftKey)) {
                        scaleWidth = finalWidth / oldWidth;
                        scaleHeight = finalHeight / oldHeight;

                        if (scaleWidth > scaleHeight) {
                            finalHeight = scaleWidth * oldHeight;
                        } else {
                            finalWidth = scaleHeight * oldWidth;
                        }
                    }

                    // make move box visible when tracking position has moved
                    DrawingFrame.toggleTracking(drawingNode, true);
                    moveBox.css({ width: finalWidth, height: finalHeight });
                }

                _.each(allDrawings, function (oneDrawing) {
                    updateDrawingDimension(oneDrawing.drawingNode, oneDrawing.moveBox, oneDrawing.width, oneDrawing.height, oneDrawing.lockResizeAspectRatio);
                });
            }

            // stop resizing of drawing
            function stopResizeDrawing() {

                var // the operations generator
                    generator = model.createOperationsGenerator(),
                    // the affected drawings with auto resize functionality
                    allAutoResizeNodes = null;

                // helper function to generate the operation for resizing one drawing
                function generateResizeOperation(drawingNode, moveBox, start) {
                    var normalizedOffset = normalizeResizeOffset(drawingNode, moveBox);

                    if (moveBox.css('display') !== 'none' && moveBox.width() && moveBox.height()) { // generate operations only if moveBox has real w/h value, measured without outer
                        // setting the new attributes of the drawing
                        operationProperties = {
                            start: start,
                            attrs: { drawing: {
                                width: Utils.convertLengthToHmm(moveBox.outerWidth(true), 'px'), // 'outer' is required to avoid shrinking in unused dimension
                                height: Utils.convertLengthToHmm(moveBox.outerHeight(true), 'px') }
                            }
                        };

                        // evaluating useTop and useLeft to avoid sending of superfluous drawing information
                        if (useTop || normalizedOffset.sendMoveOperation) { operationProperties.attrs.drawing.top = Utils.convertLengthToHmm((normalizedOffset.top - rootNode.offset().top) / zoomFactor, 'px'); }
                        if (useLeft || normalizedOffset.sendMoveOperation) { operationProperties.attrs.drawing.left = Utils.convertLengthToHmm((normalizedOffset.left - rootNode.offset().left) / zoomFactor, 'px'); }

                        // after resizing an automatic resized text frame in vertical direction by the user,
                        // the automatic resize will be removed
                        if (useY) {
                            if (DrawingFrame.isAutoResizeHeightDrawingFrame(drawingNode)) {
                                operationProperties.attrs.shape = { autoResizeHeight: false };
                            } else if (DrawingFrame.isGroupDrawingFrame(drawingNode)) {
                                // disable autoResizeHeight for all children inside a resized group
                                allAutoResizeNodes = drawingNode.find(DrawingFrame.AUTORESIZEHEIGHT_SELECTOR);
                                if (allAutoResizeNodes.length > 0) {
                                    _.each(allAutoResizeNodes, function (resizeNode) {

                                        var // the logical positions of the children
                                            localDrawingPos = Position.getOxoPosition(rootNode, $(resizeNode).parent(), 0);

                                        generator.generateOperation(Operations.SET_ATTRIBUTES, {
                                            start: localDrawingPos,
                                            attrs: { shape: { autoResizeHeight: false } }
                                        });
                                    });
                                }
                            }
                        }

                        extendAndGenerateOp(generator, drawingNode, operationProperties, target);
                    }
                }

                _.each(allDrawings, function (oneDrawing) {
                    generateResizeOperation(oneDrawing.drawingNode, oneDrawing.moveBox, oneDrawing.start);
                });

                // Applying the operations (undo group is created automatically)
                model.applyOperations(generator);
            }

            // finalizes the resize tracking
            function finalizeResizeDrawing() {
                leaveTracking();
                resizeNode.off(TRACKING_EVENT_NAMES);
                app.getView().scrollToChildNode(drawingNode);

                isMultiSelection = false;
                _.each(allDrawings, function (oneDrawing) {
                    oneDrawing.moveBox.css({ left: 0, top: 0, width: 0, height: 0 }).toggle(false);
                    clearDrawing(oneDrawing.drawingNode);
                });
                allDrawings = [];
            }

            // return the actual drawingResizeHandler() function
            return function (event) {

                event.preventDefault();

                switch (event.type) {
                    case 'tracking:start':
                        startResizeDrawing(event);
                        break;
                    case 'tracking:move':
                        updateResize(event);
                        break;
                    case 'tracking:scroll':
                        updateResizeScroll(event);
                        break;
                    case 'tracking:end':
                        stopResizeDrawing(event);
                        finalizeResizeDrawing(event);
                        break;
                    case 'tracking:cancel':
                        finalizeResizeDrawing(event);
                        break;
                }

                // events must not bubble up to drawing node (move handling)
                event.stopPropagation();
            };

        }()); // end of drawingResizeHandler() local scope

        var drawingRotateHandler = (function () {
            // dimensions of the drawing
            var drawingWidth = 0, drawingHeight = 0;
            // position of the center rotation of the drawingWidth
            var centerX, centerY;
            // the current scroll position
            var scrollX = 0, scrollY = 0;
            // the initial scroll position
            var startScrollX = 0, startScrollY = 0;
            // starting angle before rotation
            var startAngle = 0;
            // final angle on finish of the rotation
            var rotationAngle = null;
            // flag for marking that the operation is generated
            var generatedOp = false;

            function startRotateDrawing() {
                var drawingOffset;
                zoomFactor = app.getView().getZoomFactor() / 100;
                startAngle = DrawingFrame.getDrawingRotationAngle(model, drawingNode);
                if (_.isNumber(startAngle)) {
                    $(drawingNode).css({ transform: '' }); // reset to def to get values
                    drawingOffset = drawingNode.offset();
                    $(drawingNode).css({ transform: 'rotate(' + startAngle + 'deg)' }); // return property to it's value
                } else {
                    startAngle = 0;
                    drawingOffset = drawingNode.offset();
                }

                drawingHeight = drawingNode.height() * zoomFactor;
                drawingWidth = drawingNode.width() * zoomFactor;
                centerX = drawingOffset.left + (drawingWidth / 2);
                centerY = drawingOffset.top + (drawingHeight / 2);

                isMultiSelection = selection.isMultiSelectionSupported() && selection.isMultiSelection();

                allDrawings = []; // 'tracking:start' event is triggered twice consecutively, so we clean collection to prevent doubling

                if (isMultiSelection) {
                    collectAndPrepareDrawingsForRotation();
                } else {
                    var drawingPos = Position.getOxoPosition(rootNode, drawingNode, 0);
                    allDrawings.push({ drawingNode: drawingNode, start: drawingPos, rotation: startAngle });

                }
                _.each(allDrawings, function (obj) {
                    obj.drawingNode.addClass('rotating-state');
                });
            }

            function updateRotate(event) {
                var radians = Math.atan2((event.pageX + scrollX - centerX), (event.pageY + scrollY - centerY));
                rotationAngle = (radians * (180 / Math.PI) * -1) + 180;
                // round to full degrees
                rotationAngle = parseInt(rotationAngle, 10);

                if (event.shiftKey) {
                    rotationAngle = Math.round(rotationAngle / 15) * 15; // make 15 deg step with shift key
                }
                if (isMultiSelection) {
                    _.each(allDrawings, function (obj) {
                        var startAngleTemp = obj.rotation;
                        obj.drawingNode.css('transform', 'rotate(' + (rotationAngle - startAngle + startAngleTemp) + 'deg)');
                    });
                } else {
                    drawingNode.css('transform', 'rotate(' + (rotationAngle) + 'deg)');
                }
            }

            function updateRotateScroll(event) {
                // update scrollPosition with suggestion from event
                scrollNode
                    .scrollLeft(scrollNode.scrollLeft() + event.scrollX)
                    .scrollTop(scrollNode.scrollTop() + event.scrollY);

                scrollX = scrollNode.scrollLeft() - startScrollX;
                scrollY = scrollNode.scrollTop() - startScrollY;
            }

            function stopRotateDrawing() {
                var generator = model.createOperationsGenerator();
                var operationProperties;

                if (_.isNumber(rotationAngle)) {
                    if (isMultiSelection) {
                        _.each(allDrawings, function (obj) {
                            var startAngleTemp = obj.rotation;

                            var tempRotationAngle = rotationAngle - startAngle + startAngleTemp;
                            // for 360 degrees apply 0 rotation
                            tempRotationAngle = tempRotationAngle === 360 ? 0 : tempRotationAngle;

                            // normalize negative and angles greater than 360
                            if (tempRotationAngle < 0) {
                                tempRotationAngle += 360;
                            }
                            if (tempRotationAngle > 360) {
                                tempRotationAngle -= 360;
                            }

                            operationProperties = {
                                start: obj.start,
                                attrs: { drawing: { rotation: tempRotationAngle } }
                            };
                            extendAndGenerateOp(generator, drawingNode, operationProperties, target);
                        });
                    } else {
                        // for 360 degrees apply 0 rotation
                        rotationAngle = rotationAngle === 360 ? 0 : rotationAngle;

                        operationProperties = {
                            start: Position.getOxoPosition(rootNode, drawingNode, 0),
                            attrs: { drawing: { rotation: rotationAngle } }
                        };
                        extendAndGenerateOp(generator, drawingNode, operationProperties, target);
                    }

                    // Applying the operations (undo group is created automatically)
                    model.applyOperations(generator);
                    generatedOp = true;
                }
            }

            function finalizeRotateDrawing() {
                if (isMultiSelection) {
                    _.each(allDrawings, function (obj) {
                        obj.drawingNode.removeClass('rotating-state');
                    });
                } else {
                    drawingNode.removeClass('rotating-state');
                }

                if (!generatedOp) { // reset to start angle if tracking was canceled
                    if (isMultiSelection) {
                        _.each(allDrawings, function (obj) {
                            var startAngleTemp = obj.rotation || 0;

                            obj.drawingNode.css('transform', 'rotate(' + startAngleTemp + 'deg)');
                        });
                    } else {
                        drawingNode.css('transform', 'rotate(' + startAngle + 'deg)');
                    }
                } else {
                    if (isMultiSelection) {
                        _.each(allDrawings, function (obj) {
                            DrawingFrame.updateResizersMousePointers(obj.drawingNode, DrawingFrame.getDrawingRotationAngle(model, obj.drawingNode));
                        });
                    } else {
                        DrawingFrame.updateResizersMousePointers(drawingNode, DrawingFrame.getDrawingRotationAngle(model, drawingNode));
                    }
                }

                leaveTracking();
                app.getView().scrollToChildNode(drawingNode);

                if (isMultiSelection) {
                    isMultiSelection = false;
                    allDrawings = [];
                }
                generatedOp = false;
            }

            // return the actual drawingRotateHandler() function
            return function (event) {

                event.preventDefault();

                switch (event.type) {
                    case 'tracking:start':
                        startRotateDrawing(event);
                        break;
                    case 'tracking:move':
                        updateRotate(event);
                        break;
                    case 'tracking:scroll':
                        updateRotateScroll(event);
                        break;
                    case 'tracking:end':
                        stopRotateDrawing(event);
                        finalizeRotateDrawing(event);
                        break;
                    case 'tracking:cancel':
                        finalizeRotateDrawing(event);
                        break;
                }

                // events must not bubble up to drawing node (move handling)
                event.stopPropagation();
            };
        }()); // end of drawingRotateHandler local scope

        /**
         * Leaving the tracking mode.
         */
        function leaveTracking() {
            DrawingFrame.toggleTracking(drawingNode, false);
        }

        /**
         * Handler for 'tracking:start' events for 'div.handle' and 'div.move'
         * nodes. This handler has to decide, which node got the event. For this
         * node all other tracking events have to be registered. Additionally the
         * handler is called with the 'tracking:start' event, so that the drawing
         * initialization happens.
         *
         * @param {jQuery.Event} event
         *  The 'tracking:start' event, that starts the moving or the resizing
         *  of a drawing.
         */
        function trackingStartHandler(event) {

            var pos = DrawingFrame.getResizerHandleType(event.target);
            if (_.isString(pos)) {
                resizeNode = $(event.target);
                resizeNode.off(TRACKING_EVENT_NAMES);
                resizeNode.on(TRACKING_EVENT_NAMES, drawingResizeHandler);
                drawingResizeHandler.call(this, event);
            } else if ($(event.target).hasClass('rotate-handle')) {
                drawingNode.off(TRACKING_EVENT_NAMES); // <- should not be necessary, but it is (more and more setAttributes OPs)
                drawingNode.on(TRACKING_EVENT_NAMES, drawingRotateHandler);
                drawingRotateHandler.call(this, event);
            } else {
                drawingNode.off(TRACKING_EVENT_NAMES); // <- should not be necessary, but it is (more and more setAttributes OPs)
                drawingNode.on(TRACKING_EVENT_NAMES, drawingMoveHandler);
                drawingMoveHandler.call(this, event);
            }
        }

        // starting code of static method drawDrawingSelection()
        drawingNode = $(drawingNode);

        if (drawingNode.length !== 1) {
            Utils.error('DrawingResize.drawDrawingSelection(): single drawing node expected');
            drawingNode = drawingNode.first();
        }

        if (drawingNode.hasClass('horizontal-line')) {
            options.movable = false;
        } else if (drawingNode.hasClass('grouped')) {
            // options.movable = options.resizable = false;
            drawingNode = drawingNode.closest(DrawingFrame.GROUPCONTENT_SELECTOR).parent();
        }

        if (!model.getEditMode()) {
            options.movable = options.resizable = options.rotatable = false;
        }

        options.scaleHandles = 100 / app.getView().getZoomFactor();

        // the container element used to visualize the selection
        selectionBox = DrawingFrame.drawSelection(drawingNode, options);

        DrawingFrame.updateResizersMousePointers(drawingNode, DrawingFrame.getDrawingRotationAngle(model, drawingNode));

        // set visible drawing anchor
        app.getView().setVisibleDrawingAnchor(drawingNode, AttributeUtils.getExplicitAttributes(drawingNode));

        if (options.movable) {
            Tracking.enableTracking(drawingNode, {
                autoScroll: true,
                borderNode: scrollNode,
                borderMargin: -30,
                borderSize: 60,
                minSpeed: 10,
                maxSpeed: 250,
                trackModifiers: true
            });
            drawingNode.on('tracking:start', trackingStartHandler);
        } else {
            Tracking.disableTracking(drawingNode);
            drawingNode.off('tracking:start', trackingStartHandler);
        }

        // initialize resize tracking
        resizeHandles = selectionBox.find('>.resizers>[data-pos]');
        if (options.resizable) {
            Tracking.enableTracking(resizeHandles, {
                autoScroll: true,
                borderNode: scrollNode,
                borderMargin: -30,
                borderSize: 60,
                minSpeed: 10,
                maxSpeed: 250
            });
            // use 'trackModifiers' = true if you want to have special behavior on ctrl-button!!!!
            resizeHandles.on('tracking:start', trackingStartHandler);
        } else {
            Tracking.disableTracking(resizeHandles);
            resizeHandles.off('tracking:start', trackingStartHandler);
        }

        // initialize rotate tracking
        rotateHandle = selectionBox.find('.rotate-handle');
        if (options.rotatable) {
            Tracking.enableTracking(rotateHandle, {
                autoScroll: true,
                borderNode: scrollNode,
                borderMargin: -30,
                borderSize: 60,
                minSpeed: 10,
                maxSpeed: 250
            });
            resizeHandles.on('tracking:start', trackingStartHandler);
        } else {
            Tracking.disableTracking(rotateHandle);
            resizeHandles.off('tracking:start', trackingStartHandler);
        }
    };

    // exports ================================================================

    return DrawingSlideResize;

});
