/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Sven Jacobi <sven.jacobi@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/presentation/app/application', [
    'io.ox/office/editframework/app/editapplication',
    'io.ox/office/presentation/model/model',
    'io.ox/office/presentation/view/view',
    'io.ox/office/presentation/app/controller',
    'io.ox/office/tk/utils'
], function (EditApplication, PresentationModel, PresentationView, PresentationController, Utils) {

    'use strict';

    // class PresentationApplication ==========================================

    /**
     * The OX Presentation application.
     *
     * @constructor
     *
     * @extends EditApplication
     *
     * @param {Object} launchOptions
     *  All options passed to the core launcher (the ox.launch() method).
     */
    var PresentationApplication = EditApplication.extend({ constructor: function (launchOptions) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        EditApplication.call(this, PresentationModel, PresentationView, PresentationController, launchOptions, {
            prepareFlushHandler: prepareFlushDocument,
            prepareRenameHandler: prepareRenameHandler,
            postProcessHandler: postProcessDocument,
            optimizeOperationsHandler: optimizeOperations,
            sendActionsDelay: 1000,
            isMultiSelectionApp: true,
            localStorageApp: false
        });

        // private methods ----------------------------------------------------

        /**
         * Optimizing the actions (operations) before sending them to the server.
         *
         * @param {Object[]} actionsBuffer
         *  An array with actions.
         *
         * @returns {Object[]}
         *  The optimized array with actions.
         */
        function optimizeOperations(actionsBuffer) {
            return self.getModel().optimizeActions(actionsBuffer);
        }

        /**
         * Post-processes the document contents and formatting, after all its
         * import operations have been applied.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  document has been post-processed successfully, or rejected when an
         *  error has occurred.
         */
        function postProcessDocument() {
            return self.getModel().updateDocumentFormatting();
        }

        /**
         * Preprocessing before the document will be flushed for downloading,
         * printing, sending as mail, or closing.
         */
        function prepareFlushDocument() {
            sendChangedViewAttributes();
            return self.getModel().getPreFlushDocumentPromise();
        }

        /**
         * Preparations before the rename requests will be sent to the
         * backend. Must call sendChangedViewAttributes to make sure that
         * view settings are not sent while closing the document.
         */
        function prepareRenameHandler() {
            sendChangedViewAttributes();
        }

        /**
         * Send changed view attributes from the slide pane with an operation,
         * so that the attributes are stored in the document. Only send changed
         * view attributes when the user has edit rights, has modified the
         * document locally, is not on mobile or tablets.
         */
        function sendChangedViewAttributes() {

            // don't save it on tablets/mobile, because the slide pane is not resizable there
            if (Utils.COMPACT_DEVICE) {
                return self;
            }

            // only send changed attributes when user has edit rights
            if (!self.getModel().getEditMode()) {
                return self;
            }

            // whether the document is modified
            var modified = self.isLocallyModified();

            // if file was not modified, check if the slide pane width has been changed compared to the document
            if (!modified) {
                if (self.getModel().getSlidePaneWidthInDocAttributes() !== self.getModel().getSlidePaneWidth()) {
                    modified = true;
                }
            }
            // if the file was not modified, nothing to do
            if (!modified) {
                return self;
            }
            // send the slidePaneWidth with an operation
            self.getModel().setSlidePaneWidthInDocAttributes();

        }

        // initialization -----------------------------------------------------

        // destroy all class members
        this.registerDestructor(function () {
            launchOptions = null;
        });

    } }); // class PresentationApplication

    // exports ================================================================

    return PresentationApplication;

});
