/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Edy Haryono <edy.haryono@open-xchange.com>
 */

define('io.ox/office/portal/view/templateview', [
    'io.ox/office/portal/portalutils',
    'io.ox/core/extensions',
    'io.ox/office/portal/action/templateactions'
], function (PortalUtils, Ext, TemplateActions) {

    'use strict';

    var templateView = Backbone.View.extend({

        tagName: 'div',

        className: 'office-portal-template',

        initialize: function () {

            var // the document view object
                self = this;

            // observer for the collection (when the collection is updated)
            this.listenTo(this.collection, 'update', function () {
                // re-render the recent list for the current app
                self.render(self.model.get('appBaseName'));
            });

            // observer for the collection (when only a model in the collection has been changed)
            this.listenTo(this.collection, 'change', function () {
                // re-render the recent list for the current app
                self.render(self.model.get('appBaseName'));
            });
        },

        /**
         * Renders templates section of supported applications and 'My Templates' section
         *
         * @param {String} appBaseName
         *  The base name of the current application.
         *
         * @returns {templateView}
         */
        render: function (appBaseName) {

            // render nothing if we are on Text app on Android with non-supported Chrome or other browser
            if ((_.browser.Android) && (appBaseName === 'text') && !PortalUtils.CHROME_ON_ANDROID) { return; }

            var self = this,
                templateWrapper = $('<div>').addClass('template-list f6-target');

            // empty the template list
            self.$el.empty();

            self.$el.append(templateWrapper);

            // draw create with blank template
            TemplateActions[appBaseName].invoke('draw', templateWrapper, Ext.Baton({ app: self.model, appBaseName: appBaseName }));

            // get templates to render from template collection for all sources (user, admin, global)
            // and render the templates for the active app type
            _.each(['user', 'admin', 'global'], function (templateSource) {

                // 'user' and 'admin' templates are used alike while rendering templates,
                // therefore is only differentiated between 'user' and 'global' here
                var type = ((templateSource === 'global') ? 'global' : 'user');

                _.each(this.collection.getTemplates(appBaseName, templateSource), function (template) {
                    // render
                    // Important: 'template.attributes' is cloned here, because it's changed when a document is opened otherwise (passed by reference).
                    // That would introduce a 'version' attribute in the baton that prevent displaying the most recent template thumbnail.
                    // (e.g. modify a template -> thumbnail is not updated)
                    TemplateActions[appBaseName + type].invoke('draw', templateWrapper, Ext.Baton({ data: _.clone(template.attributes) }));
                });
            }, this);

            return this;
        }

    });

    return templateView;

});
