/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

/* eslint no-new: 0 */

define('io.ox/office/portal/mainhelper', [
    'io.ox/core/commons',
    'io.ox/office/tk/io',
    'io.ox/office/tk/utils/driveutils',
    'io.ox/office/portal/portalutils',
    'io.ox/office/portal/collection/documentcollection',
    'io.ox/office/portal/collection/templatecollection',
    'io.ox/office/portal/view/documentview',
    'io.ox/office/portal/view/templateview',
    'io.ox/office/portal/view/toolbarview',
    'io.ox/office/portal/view/tooltip',
    'io.ox/office/baseframework/app/guidedtour',
    'io.ox/office/editframework/app/fileactions',
    'settings!io.ox/office',
    'gettext!io.ox/office/portal/main',
    'less!io.ox/office/portal/style'
], function (Commons, IO, DriveUtils, Utils, DocumentCollection, TemplateCollection, DocumentView, TemplateView, ToolbarView, ToolTip, GuidedTour, FileActions, Settings, gt) {

    'use strict';

    // static class MainHelper ================================================

    var // the exported MainHelper class
        MainHelper = {};

    // methods ----------------------------------------------------------------

    MainHelper.createLauncher = function (appBaseName, appTitle) {

        var // the module name of the portal application
            PORTAL_MODULE_NAME = Utils.getPortalModuleName(appBaseName),
            // the module name of the edit application
            EDIT_MODULE_NAME = Utils.getEditorModuleName(appBaseName),
            // the application instance
            app = ox.ui.createApp({ name: PORTAL_MODULE_NAME, title: appTitle, trackingId: 'io.ox/' + appBaseName + '-portal', appBaseName: appBaseName }),
            // the document collection for the recent documents list
            documentCollection = new DocumentCollection(),
            // the template collection for the listed templates
            templateCollection = new TemplateCollection();

        // load fileActions for the current App to use necessary action links for recent list/template context menus
        FileActions.create(appBaseName);

        app.setLauncher(function (options) {

            // create main window object
            var portalWindow = ox.ui.createWindow({
                name: PORTAL_MODULE_NAME,
                title: appTitle,
                chromeless: true
            });

            var rootNode = portalWindow.nodes.main,
                windowBodyNode = portalWindow.nodes.body,
                recentsNode = $('<div>').addClass('office-portal-recents').append($('<h2>').text(gt('Recent Documents'))),
                recentsList = $('<div>').addClass('office-portal-recents-list f6-target'),
                templatesNode = $('<div>').addClass('office-portal-templates').append($('<h2>').text(gt('New from Template'))),
                templateTypeWrapper = $('<div>').addClass('template-type-wrapper col-xs-12');

            // the bootstrap of the OX Documents app
            function initOXDocuments() {
                // TODO: refactor later: there are two 'on' listener for portalWindow in this class, merge them
                portalWindow.setTitle(appTitle);
                portalWindow.on('show', function () {
                    portalWindow.setTitle(appTitle);
                });

                // prefetch application source code
                IO.prefetchModuleSource(EDIT_MODULE_NAME);

                // create application views
                new ToolbarView({ el: windowBodyNode, model: app });
                new DocumentView({ collection: documentCollection, el: recentsList, model: app });
                new TemplateView({ collection: templateCollection, el: templateTypeWrapper, model: app });

                // append main nodes to the rootNode
                rootNode.empty().append(recentsNode.append(recentsList), templatesNode.append(templateTypeWrapper));

                // get data for the portal the first time and show it
                updatePortal();
            }

            function updatePortal() {

                // Remove the highlight class 'permanentHover' for all templates.
                // The highlight is set (e.g.) when the user opens a document from a template.
                // Therefore it must be removed later.
                portalWindow.nodes.main.find('.permanentHover').removeClass('permanentHover');

                if (DriveUtils.isGuest()) {
                    // guests & anonymous users
                    documentCollection.setData([]);
                    templateCollection.setData([]);
                } else {

                    // try to get recent files and templates
                    IO.sendRequest(IO.FILTER_MODULE_NAME, { action: 'gettemplateandrecentlist', type: appBaseName })
                    .done(function (data) {

                        // must be reset, because the order from the list is important and
                        // when no element is removed/added, but the order is, there would be no repaint
                        // (minor issue, reset also repaints, so we repaint two times (reset, setData) for the recent list)
                        documentCollection.reset();

                        // set data for the collection, what cause a 'update' that renders also the view
                        documentCollection.setData(data.recents || []);
                        templateCollection.setData(data.templates || []);

                        // to init the bootstrap tooltips in portal
                        $('[data-toggle="tooltip"]').tooltip();

                        // start the welcome tour for this application once
                        GuidedTour.runAppTour(PORTAL_MODULE_NAME, { once: true, auto: true });
                    }).fail(appFail);
                }
            }

            // yells if something went wrong...
            function appFail(result) {
                require(['io.ox/core/capabilities', 'io.ox/core/notifications']).done(function (Capabilities, Notifications) {
                    var errorMsg = '';
                    if (!Capabilities.has('rt')) {
                        errorMsg += gt('A server component is not working. Please contact the server administrator.');
                    } else {
                        errorMsg += (result && result.error) ? result.error + ':' : '';
                        errorMsg += gt('Application may not work as expected until this problem is solved.');
                    }
                    Notifications.yell('error', errorMsg);
                });
                portalWindow.nodes.main.children().hide();
                portalWindow.nodes.body.children().hide();
            }

            // refreshes Documents app only on visibility
            function refresh() {

                 //TODO for later - maybe we should refresh all portals not only the visible, but check if 'documentsToShow' in 'showOptimalCount' is called without an NAN error on global refresh
                if (!portalWindow.state.visible) { return; }
                updatePortal();
            }

            // save settings to the app object
            app.settings = Settings;

            portalWindow.addClass('io-ox-office-portal-main');

            // set it as the application window
            app.setWindow(portalWindow);

            // show portal window
            portalWindow.show();

            // Reinit app on portal window show. This is needed for this use case:
            // User opens a document from the recents list -> loses app focus,
            // and close the document -> gained focus to this app without an app 'resume' event).
            portalWindow.on('show', updatePortal);

            // refresh Documents portal on OX global refresh event
            ox.on('refresh^', refresh);

            // after the tooltip is inserted in the DOM, set the max-width for the tooltip, according to the rootNote width
            rootNode.on('inserted.bs.tooltip', function () {
                // '-90' to give some margin
                rootNode.find('.tooltip-inner').css('maxWidth', rootNode.outerWidth() - 90 + 'px');
            });

            // init OX Documents in folder support addition success.
            Commons.addFolderSupport(app, null, 'infostore', options.folder).always(initOXDocuments).fail(appFail);
        });

        // register as application tour (will create an entry in the settings menu to start the tour)
        if (!_.device('smartphone')) {
            GuidedTour.registerAppTour(PORTAL_MODULE_NAME, PORTAL_MODULE_NAME + '/fulltour');
        }

        return { getApp: app.getInstance };
    };

    // exports ================================================================

    return MainHelper;

});
