/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/operationspane', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/keycodes',
    'io.ox/office/baseframework/view/pane',
    'io.ox/office/editframework/app/rtconnection',
    'io.ox/office/editframework/view/popup/attributestooltip'
], function (Utils, Forms, KeyCodes, Pane, RTConnection, AttributesToolTip) {

    'use strict';

    // class OperationsPane ===================================================

    /**
     * A view pane attached to the bottom border of the application view that
     * contains the operations log, and other custom debug information defined
     * by the application.
     *
     * @constructor
     *
     * @extends Pane
     *
     * @param {EditView} docView
     *  The document view containing this pane instance.
     */
    function OperationsPane(docView) {

        // self reference
        var self = this;

        // the application instance
        var app = docView.getApp();

        // the document model instance
        var docModel = docView.getDocModel();

        // a tool-tip for details about attribute sets shown in this pane
        var attrsToolTip = new AttributesToolTip(docView);

        // the <table> element of the operations log
        var opTableNode = null;

        // the <table> element of the debug info table
        var infoTableNode = null;

        // number of operations already logged
        var operationsCount = 0;

        // operations not yet logged
        var pendingOperations = [];

        // background loop logging all pending operations
        var logOperationsTimer = null;

        // output elements for the debug info table (faster than DOM look-ups)
        var infoNodeMap = {};

        // base constructor ---------------------------------------------------

        Pane.call(this, docView, {
            position: 'bottom',
            classes: 'operations-pane noI18n',
            size: 300,
            resizable: true,
            minSize: 100,
            maxSize: 500
        });

        // private methods ----------------------------------------------------

        /**
         * Scrolls the operations output down to its end.
         */
        function scrollDownOperationsNode() {
            var scrollNode = opTableNode.parent();
            scrollNode.scrollTop(scrollNode[0].scrollHeight);
        }

        /**
         * Direct callback for logOperations(): called every time when
         * logOperations() is called.
         */
        function registerOperations(event, operations, external) {
            RTConnection.log('registerOperations entered');

            // append the new operations to the pending operations queue
            if (!app.isOperationsBlockActive()) {
                RTConnection.log('registerOperations appending operations');
                pendingOperations = pendingOperations.concat(operations.map(function (operation) {
                    if (app.isPrepareLosingEditRights()) {
                        RTConnection.log('registerOperations while preparelosingeditrights: ' + JSON.stringify(operation));
                    }
                    return { operation: operation, external: external };
                }));
            }

            // update the OSN in the info section
            self.setDebugInfoText('application', 'osn', docModel.getOperationStateNumber());
        }

        function generateOperationsMarkup(operations) {

            // the number of operations applied successfully by the document model
            var successCount = docModel.getOperationsCount();
            // bug 29601: number of operations to be skipped (that have been optimized away)
            var skipCount = 0;

            return operations.reduce(function (markup, entry) {

                var operation = _.clone(entry.operation);
                var name = operation.name || '';
                var osn = ('osn' in operation) ? operation.osn : '';
                var opl = ('opl' in operation) ? operation.opl : '';

                // bug 29601: skip operations, that were 'optimized' in 'sendActions' in the
                // optimizeOperationHandler. Otherwise the debug pane will be irritating, because
                // 'merged operations' are followed by the operations, that will not be sent to
                // the server, because they were removed by the 'optimizeOperationHandler'.
                if (!entry.external) {
                    if (_.isNumber(opl) && (opl > 1)) {
                        skipCount = opl - 1;
                    } else if (skipCount > 0) {
                        skipCount -= 1;
                        return markup;
                    }
                }

                delete operation.name;
                delete operation.osn;
                delete operation.opl;
                operationsCount += 1;

                var source = entry.external ? 'External' : 'Internal';
                var classes = source.toLowerCase() + ((successCount < operationsCount) ? ' error' : '');
                if (entry.removed) { classes += ' removed'; }

                markup += '<tr class="' + classes + '">';
                markup += '<td>' + operationsCount + '</td>';
                markup += '<td title="' + source + ' operation">' + source[0] + '</td>';
                markup += '<td style="text-align:right;">' + osn + '</td>';
                markup += '<td style="text-align:right;">' + opl + '</td>';
                markup += '<td>' + Utils.escapeHTML(name) + '</td>';
                markup += '<td title="' + Forms.createLoggingTooltipMarkup(operation) + '">' + Utils.escapeHTML(Utils.stringifyForDebug(operation)) + '</td>';
                markup += '</tr>';
                return markup;
            }, '');
        }

        /**
         * Starts the background loop that logs collected pending operations.
         */
        function startLogger() {

            // create a new background loop that processes all pending operations
            logOperationsTimer = self.repeatDelayed(function () {

                // the entire HTML mark-up to be appended to the operations log
                var markup = '';

                // generate HTML mark-up for at most 200ms
                var t0 = _.now();
                while ((pendingOperations.length > 0) && (_.now() - t0 < 200)) {
                    markup += generateOperationsMarkup(pendingOperations.splice(0, 20));
                }

                // append the mark-up, and scroll down to the last log entry
                if (markup.length > 0) {
                    opTableNode.append(markup);
                    scrollDownOperationsNode();
                }
            }, 1000);
        }

        /**
         * Stops the background loop that logs collected pending operations.
         */
        function abortLogger() {
            logOperationsTimer.abort();
            logOperationsTimer = null;
        }

        /**
         * Marks the last operations in the operations table with orange
         * background color. This means, that they were not executed, not send
         * to the server and they do not influence the OSN for the following
         * operations. This scenario happens for example, if a paste process is
         * cancelled and an existing selection was already removed before the
         * pasting started.
         *
         * @param {Number} number
         *  The number of operations that need to be marked with orange
         *  background color.
         */
        function operationsRemovedHandler(number) {

            // mark operations not yet logged
            var pendingCount = pendingOperations.length;
            for (var index = Math.max(0, pendingCount - number); index < pendingCount; index += 1) {
                pendingOperations[index].removed = true;
            }

            // mark operations already logged
            if (pendingCount < number) {
                opTableNode.find('tr').slice(number - pendingCount).addClass('removed');
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Adds a new header row to the information window.
         *
         * @param {String} header
         *  The name of the header row, used to identify data rows shown under
         *  the new header row, and as label text.
         *
         * @param {String} [tooltip]
         *  A tooltip for the entire header row.
         *
         * @returns {OperationsPane}
         *  A reference to this instance.
         */
        this.addDebugInfoHeader = function (header, tooltip) {
            var rowNode = $('<tr title="' + Utils.escapeHTML(tooltip) + '"><th colspan="2">' + Utils.capitalizeWords(header) + '</th></tr>');
            infoTableNode.append(rowNode);
            infoNodeMap[header] = rowNode;
            return this;
        };

        /**
         * Adds a new data row to the information window.
         *
         * @param {String} header
         *  The name of the header row to associate this data row to.
         *
         * @param {String} label
         *  The name of the data row, used to identify the row when inserting
         *  text contents, and as label text.
         *
         * @param {String} [tooltip]
         *  A tooltip for the first cell of the data row (the row label).
         *
         * @returns {OperationsPane}
         *  A reference to this instance.
         */
        this.addDebugInfoNode = function (header, label, tooltip) {
            var infoNode = $('<td>');
            var rowNode = $('<tr><td title="' + Utils.escapeHTML(tooltip) + '">' + label + '</td></tr>').append(infoNode);
            infoNodeMap[header].last().after(rowNode);
            infoNodeMap[header] = infoNodeMap[header].add(rowNode);
            infoNodeMap[header + ':' + label] = infoNode;
            return this;
        };

        /**
         * Changes the text shown in a specific data row of the information
         * window.
         *
         * @param {String} header
         *  The name of the header containing the data row to be changed.
         *
         * @param {String} label
         *  The name of the data row to be changed.
         *
         * @param {String} text
         *  The new text contents shown in the data row.
         *
         * @returns {OperationsPane}
         *  A reference to this instance.
         */
        this.setDebugInfoText = function (header, label, text) {
            infoNodeMap[header + ':' + label].text(text);
            return this;
        };

        /**
         * Changes the HTML mark-up shown in a specific data row of the
         * information window.
         *
         * @param {String} header
         *  The name of the header containing the data row to be changed.
         *
         * @param {String} label
         *  The name of the data row to be changed.
         *
         * @param {String} markup
         *  The new HTML mark-up shown in the data row.
         *
         * @returns {OperationsPane}
         *  A reference to this instance.
         */
        this.setDebugInfoMarkup = function (header, label, markup) {
            infoNodeMap[header + ':' + label].html(markup);
            return this;
        };

        /**
         * Registers a formatter callback function for specific attribute types
         * or values, shown in a tool-tip for attribute sets provided by this
         * instance. See method AttributesToolTip.registerFormatter() for more
         * details.
         *
         * @returns {OperationsPane}
         *  A reference to this instance.
         */
        this.registerAttributeFormatter = function (selector, formatter) {
            attrsToolTip.registerFormatter(selector, formatter);
            return this;
        };

        // initialization -----------------------------------------------------

        // create the operations log table, and the debug info table
        opTableNode = $('<table><tbody><tr><th></th><th title="Operation type">T</th><th title="Operation State Number" style="font-size:70%;">OSN</th><th title="Operation Length" style="font-size:70%;">OPL</th><th>Name</th><th>Properties</th></tr></tbody></table>');
        infoTableNode = $('<table style="table-layout:fixed;"><colgroup><col width="50"></col><col></col></colgroup></table>');

        // add the operations table and the info table to the debug pane
        [{ node: opTableNode, cls: 'ops' }, { node: infoTableNode, cls: 'info' }].forEach(function (entry) {
            var scrollNode = $('<div tabindex="1">').append(entry.node);
            self.getNode().append($('<div class="debug-table ' + entry.cls + '">').append(scrollNode));
        });

        // create the output nodes in the debug pane
        this.addDebugInfoHeader('application', 'Global application state')
            .addDebugInfoNode('application', 'state', 'Application state')
            .addDebugInfoNode('application', 'osn', 'Operation state number')
            .addDebugInfoNode('application', 'undo', 'Undo/redo stack state');

        // log the application state
        this.listenTo(app, 'docs:state', function (state) {
            self.setDebugInfoText('application', 'state', state);
        });

        // log the operation state number of the document
        this.listenTo(docModel, 'change:osn', function () {
            self.setDebugInfoText('application', 'osn', docModel.getOperationStateNumber());
        });

        // log size of undo/redo stacks
        this.listenTo(docModel.getUndoManager(), 'change:count', function (event, undoCount, redoCount) {
            self.setDebugInfoText('application', 'undo', 'undo=' + undoCount + ', redo=' + redoCount);
        });

        // simply collect all operations while importing the document
        this.listenTo(docModel, 'operations:after', registerOperations);

        // scroll to last operation when showing the debug pane
        this.on('pane:show', scrollDownOperationsNode);

        // keyboard shortcuts
        this.getNode().find('[tabindex]').on('keydown', function (event) {
            if (KeyCodes.matchKeyCode(event, 'A', { ctrlOrMeta: true })) {
                var docRange = window.document.createRange();
                docRange.setStart(event.delegateTarget, 0);
                docRange.setEnd(event.delegateTarget, 1);
                window.getSelection().removeAllRanges();
                window.getSelection().addRange(docRange);
                return false;
            }
            if (event.keyCode === KeyCodes.ESCAPE) {
                docView.grabFocus();
                return false;
            }
        });

        // show the formatting attributes tool-tip
        this.getNode().on('click', AttributesToolTip.NODE_SELECTOR, function (event) {
            attrsToolTip.initialize(event.currentTarget).show();
        });

        // log operations after importing the document
        this.waitForImport(function () {

            // mark some operations in the pane as removed
            this.listenTo(app, 'operations:remove', operationsRemovedHandler);

            // start logging with visible operations pane, pause logging in hidden operations pane
            this.on({ 'pane:show': startLogger, 'pane:hide': abortLogger });

            // start logging of initial import operations after a delay
            this.executeDelayed(function () {
                if (!logOperationsTimer && self.isVisible()) { startLogger(); }
            }, 3000);

        }, this);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            attrsToolTip.destroy();
            self = app = docView = docModel = attrsToolTip = null;
            opTableNode = infoTableNode = null;
            pendingOperations = logOperationsTimer = infoNodeMap = null;
        });

    } // class OperationsPane

    // exports ================================================================

    // derive this class from class Pane
    return Pane.extend({ constructor: OperationsPane });

});
