/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Carsten Driesner <carsten.driesner@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/utils/color', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/render/canvas'
], function (Utils, Canvas) {

    'use strict';

    // the gamma correction factors (intended to be similar to MSO)
    var INV_GAMMA = 2.27;
    var GAMMA = 1 / INV_GAMMA;

    // threshold luma for dark colors
    var Y_DARK = 0.4;

    // maps the lower-case names of all preset colors to the real mixed-case names and RGB color values
    var PRESET_COLOR_MAP = (function () {
        var map = {};
        _.each({
            aliceBlue: 'f0f8ff',         antiqueWhite: 'faebd7',      aqua: '00ffff',                  aquamarine: '7fffd4',
            azure: 'f0ffff',             beige: 'f5f5dc',             bisque: 'ffe4c4',                black: '000000',
            blanchedAlmond: 'ffebcd',    blue: '0000ff',              blueViolet: '8a2be2',            brown: 'a52a2a',
            burlyWood: 'deb887',         cadetBlue: '5f9ea0',         chartreuse: '7fff00',            chocolate: 'd2691e',
            coral: 'ff7f50',             cornflowerBlue: '6495ed',    cornsilk: 'fff8dc',              crimson: 'dc143c',
            cyan: '00ffff',              darkBlue: '00008b',          darkCyan: '008b8b',              darkGoldenrod: 'b8860b',
            darkGray: 'a9a9a9',          darkGreen: '006400',         darkKhaki: 'bdb76b',             darkMagenta: '8b008b',
            darkOliveGreen: '556b2f',    darkOrange: 'ff8c00',        darkOrchid: '9932cc',            darkRed: '8b0000',
            darkSalmon: 'e9967a',        darkSeaGreen: '8fbc8f',      darkSlateBlue: '483d8b',         darkSlateGray: '2f4f4f',
            darkTurquoise: '00ced1',     darkViolet: '9400d3',        deepPink: 'ff1493',              deepSkyBlue: '00bfff',
            dimGray: '696969',           dodgerBlue: '1e90ff',        firebrick: 'b22222',             floralWhite: 'fffaf0',
            forestGreen: '228b22',       fuchsia: 'ff00ff',           gainsboro: 'dcdcdc',             ghostWhite: 'f8f8ff',
            gold: 'ffd700',              goldenrod: 'daa520',         gray: '808080',                  green: '008000',
            greenYellow: 'adff2f',       honeydew: 'f0fff0',          hotPink: 'ff69b4',               indianRed: 'cd5c5c',
            indigo: '4b0082',            ivory: 'fffff0',             khaki: 'f0e68c',                 lavender: 'e6e6fa',
            lavenderBlush: 'fff0f5',     lawnGreen: '7cfc00',         lemonChiffon: 'fffacd',          lightBlue: 'add8e6',
            lightCoral: 'f08080',        lightCyan: 'e0ffff',         lightGoldenrodYellow: 'fafad2',  lightGray: 'd3d3d3',
            lightGreen: '90ee90',        lightPink: 'ffb6c1',         lightSalmon: 'ffa07a',           lightSeagreen: '20b2aa',
            lightSkyBlue: '87cefa',      lightSlateGray: '778899',    lightSteelBlue: 'b0c4de',        lightYellow: 'ffffe0',
            lime: '00ff00',              limeGreen: '32cd32',         linen: 'faf0e6',                 magenta: 'ff00ff',
            maroon: '800000',            mediumAquamarine: '66cdaa',  mediumBlue: '0000cd',            mediumOrchid: 'ba55d3',
            mediumPurple: '9370d8',      mediumSeagreen: '3cb371',    mediumSlateBlue: '7b68ee',       mediumSpringGreen: '00fa9a',
            mediumTurquoise: '48d1cc',   mediumVioletred: 'c71585',   midnightBlue: '191970',          mintCream: 'f5fffa',
            mistyRose: 'ffe4e1',         moccasin: 'ffe4b5',          navajoWhite: 'ffdead',           navy: '000080',
            oldLace: 'fdf5e6',           olive: '808000',             oliveDrab: '6b8e23',             orange: 'ffa500',
            orangeRed: 'ff4500',         orchid: 'da70d6',            paleGoldenrod: 'eee8aa',         paleGreen: '98fb98',
            paleTurquoise: 'afeeee',     paleVioletRed: 'd87093',     papayaWhip: 'ffefd5',            peachPuff: 'ffdab9',
            peru: 'cd853f',              pink: 'ffc0cb',              plum: 'dda0dd',                  powderBlue: 'b0e0e6',
            purple: '800080',            red: 'ff0000',               rosyBrown: 'bc8f8f',             royalBlue: '4169e1',
            saddleBrown: '8b4513',       salmon: 'fa8072',            sandyBrown: 'f4a460',            seaGreen: '2e8b57',
            seaShell: 'fff5ee',          sienna: 'a0522d',            silver: 'c0c0c0',                skyBlue: '87ceeb',
            slateBlue: '6a5acd',         slateGray: '708090',         snow: 'fffafa',                  springGreen: '00ff7f',
            steelBlue: '4682b4',         tan: 'd2b48c',               teal: '008080',                  thistle: 'd8bfd8',
            tomato: 'ff6347',            turquoise: '40e0d0',         violet: 'ee82ee',                wheat: 'f5deb3',
            white: 'ffffff',             whiteSmoke: 'f5f5f5',        yellow: 'ffff00',                yellowGreen: '9acd32'
        }, function (value, key) {
            map[key.toLowerCase()] = { name: key, rgb: value };
        });
        return map;
    }());

    // private static functions ===============================================

    /**
     * Returns the specified integer property from a JSON object, normalizes
     * the number, but does not restrict it any further.
     *
     * @param {Object} json
     *  An arbitrary JSON object.
     *
     * @param {String} prop
     *  The name of an integer property to be extracted from the JSON object.
     *  If the property does not exist, this function returns zero.
     *
     * @returns {Number}
     *  The normalized value of the specified property.
     */
    function getNum(json, prop) {
        return Utils.getIntegerOption(json, prop, 0) / 100000;
    }

    /**
     * Returns the specified integer property from a JSON object, normalizes
     * the number, and restricts it to non-negative numbers.
     *
     * @param {Object} json
     *  An arbitrary JSON object.
     *
     * @param {String} prop
     *  The name of an integer property to be extracted from the JSON object.
     *  If the property does not exist, this function returns zero.
     *
     * @returns {Number}
     *  The normalized value of the specified property, restricted to
     *  non-negative numbers.
     */
    function getPosNum(json, prop) {
        return Utils.getIntegerOption(json, prop, 0, 0) / 100000;
    }

    /**
     * Returns the specified integer property from a JSON object, normalizes
     * the number, and restricts it to the interval [0;1].
     *
     * @param {Object} json
     *  An arbitrary JSON object.
     *
     * @param {String} prop
     *  The name of an integer property to be extracted from the JSON object.
     *  If the property does not exist, this function returns zero.
     *
     * @returns {Number}
     *  The normalized value of the specified property, restricted to the
     *  interval [0;1].
     */
    function getLimNum(json, prop) {
        return Utils.getIntegerOption(json, prop, 0, 0, 100000) / 100000;
    }

    /**
     * Returns the RGB value of the specified system color.
     *
     * @param {String} sysColor
     *  The name of a system color, as used in CSS, and in document operations.
     *  The character case of the color name will be ignored.
     *
     * @returns {String}
     *  The RGB value of the system color, as 6-digit string.
     */
    var getSystemRGBColor = (function () {

        var canvas = new Canvas();
        canvas.initialize({ width: 3, height: 3 });

        // store only lower-case color names in the cache
        function getColorKey(sysColor) {
            return sysColor.toLowerCase();
        }

        // converts color names to RGB strings
        function getRGBColor(sysColor) {
            canvas.render(function (context) {
                context.setFillStyle(sysColor).drawRect(0, 0, 3, 3, 'fill');
            });
            return canvas.getPixelColor(1, 1).hex;
        }

        return _.memoize(getRGBColor, getColorKey);
    }());

    // class AlphaModel ======================================================

    /**
     * Base class for various color models, providing the alpha channel of a
     * color.
     *
     * @constructor
     *
     * @property {Number} a
     *  The value of the alpha channel (opacity), in the interval [0;1]. The
     *  value 0 represents full transparency, the value 1 represents full
     *  opacity.
     */
    var AlphaModel = _.makeExtendable(function (a) {

        // public properties
        this.a = a;

    }); // class AlphaModel

    // public methods ---------------------------------------------------------

    /**
     * Changes the specified color channel to the specified value. This method
     * is intended to be used by subclasses representing different color models
     * to manipulate the various supported color channels.
     *
     * @param {String} channel
     *  The name of the color channel. The supported channel names depend on
     *  the subclass representing a specific color model. The channel name 'a'
     *  will manipulate the alpha channel provided by this base class.
     *
     * @param {Number} value
     *  The new value of the color channel.
     *
     * @param {Number} [cycle]
     *  If specified, the size of the channel interval (the valid numbers for
     *  the channel). The channel value passed in the parameter 'value' will be
     *  adjusted to the channel size by calculating the modulo. If omitted, the
     *  passed value will be reduced to the interval [0,1].
     *
     * @returns {AlphaModel}
     *  A reference to this instance.
     */
    AlphaModel.prototype.set = function (channel, value, cycle) {
        this[channel] = (typeof cycle === 'number') ? Utils.mod(value, cycle) : Utils.minMax(value, 0, 1);
        return this;
    };

    /**
     * Modulates the specified color channel (multiplies the color channel with
     * the specified factor). This method is intended to be used by subclasses
     * representing different color models to manipulate the various supported
     * color channels.
     *
     * @param {String} channel
     *  The name of the color channel. The supported channel names depend on
     *  the subclass representing a specific color model. The channel name 'a'
     *  will manipulate the alpha channel provided by this base class.
     *
     * @param {Number} factor
     *  The modulation factor. MUST be non-negative.
     *
     * @param {Number} [cycle]
     *  If specified, the size of the channel interval (the valid numbers for
     *  the channel). The resulting channel value will be adjusted to the
     *  channel size by calculating the modulo. If omitted, the resulting value
     *  will be reduced to the interval [0,1].
     *
     * @returns {AlphaModel}
     *  A reference to this instance.
     */
    AlphaModel.prototype.modulate = function (channel, factor, cycle) {
        return this.set(channel, this[channel] * factor, cycle);
    };

    /**
     * Shifts the specified color channel (adds the specified value to the
     * color channel). This method is intended to be used by subclasses
     * representing different color models to manipulate the various supported
     * color channels.
     *
     * @param {String} channel
     *  The name of the color channel. The supported channel names depend on
     *  the subclass representing a specific color model. The channel name 'a'
     *  will manipulate the alpha channel provided by this base class.
     *
     * @param {Number} shift
     *  The value to be added to the color channel.
     *
     * @param {Number} [cycle]
     *  If specified, the size of the channel interval (the valid numbers for
     *  the channel). The resulting channel value will be adjusted to the
     *  channel size by calculating the modulo. If omitted, the resulting value
     *  will be reduced to the interval [0,1].
     *
     * @returns {AlphaModel}
     *  A reference to this instance.
     */
    AlphaModel.prototype.offset = function (channel, shift, cycle) {
        return this.set(channel, this[channel] + shift, cycle);
    };

    // class RGBModel =========================================================

    /**
     * Representation of a color using the RGB color model. This class adds the
     * color channels 'r', 'g', and 'b' which can be manipulated by the methods
     * of the base class AlphaModel.
     *
     * @constructor
     *
     * @extends AlphaModel
     *
     * @property {Number} r
     *  The value of the red channel, in the interval [0;1].
     *
     * @property {Number} g
     *  The value of the green channel, in the interval [0;1].
     *
     * @property {Number} b
     *  The value of the blue channel, in the interval [0;1].
     */
    var RGBModel = AlphaModel.extend({ constructor: function (r, g, b, a) {

        // base constructor
        AlphaModel.call(this, a);

        // public properties
        this.r = r;
        this.g = g;
        this.b = b;

    } }); // class RGBModel

    // static methods ---------------------------------------------------------

    /**
     * Parses the passed hexadecimal RGB string and creates an instance of the
     * class RGBModel.
     *
     * @param {String} rgb
     *  The string representation of a 6-digit hexadecimal number (an RGB color
     *  in the format 'RRGGBB'). If the string is not such a number, returns an
     *  RGB color model for the color black.
     *
     * @returns {RGBModel}
     *  The new RGB color model created from the passed string.
     */
    RGBModel.parseHex = function (rgb) {
        if (!/^[0-9a-f]{6}$/i.test(rgb)) { return new RGBModel(0, 0, 0, 1); }
        return new RGBModel(parseInt(rgb.substr(0, 2), 16) / 255, parseInt(rgb.substr(2, 2), 16) / 255, parseInt(rgb.substr(4, 2), 16) / 255, 1);
    };

    /**
     * Parses the passed JSON representation of an RGB color and creates an
     * instance of the class RGBModel.
     *
     * @param {Any} json
     *  The JSON representation of an RGB color, expected to be an object with
     *  the number properties 'r', 'g', and 'b'.
     *
     * @returns {RGBModel}
     *  The new RGB color model created from the passed JSON data.
     */
    RGBModel.parseJSON = function (json) {
        return new RGBModel(getLimNum(json, 'r'), getLimNum(json, 'g'), getLimNum(json, 'b'), 1);
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns a clone of this RGB color model.
     *
     * @returns {RGBModel}
     *  A clone of this RGB color model.
     */
    RGBModel.prototype.clone = function () {
        return new RGBModel(this.r, this.g, this.b, this.a);
    };

    /**
     * Returns the luma (the weighted lightness) of this RGB color.
     *
     * @returns {Number}
     *  The luma of this color, in the interval [0;1].
     */
    RGBModel.prototype.getLuma = function () {
        return 0.2126 * this.r + 0.7152 * this.g + 0.0722 * this.b;
    };

    /**
     * Returns a reference to this instance. Used to ease the conversion from
     * different color models to an RGB color model.
     *
     * @returns {RGBModel}
     *  A reference to this instance.
     */
    RGBModel.prototype.toRGB = function () {
        return this;
    };

    /**
     * Creates an instance of the class HSLModel representing the color of this
     * RGB color model.
     *
     * @returns {HSLModel}
     *  An instance of the class HSLModel representing the color of this RGB
     *  color model.
     */
    RGBModel.prototype.toHSL = function () {

        var max = Math.max(this.r, this.g, this.b),
            min = Math.min(this.r, this.g, this.b),
            diff = max - min,
            sum = max + min;

        var h = (diff === 0) ? 0 : (max === this.r) ? Utils.mod((this.g - this.b) / diff * 60, 360) : (max === this.g) ? ((this.b - this.r) / diff * 60 + 120) : ((this.r - this.g) / diff * 60 + 240);
        var s = ((sum === 0) || (sum === 2)) ? 0 : (sum < 1) ? (diff / sum) : (diff / (2 - sum));
        var l = sum / 2;

        return new HSLModel(h, s, l, this.a);
    };

    /**
     * Applies a gamma shift to this color model.
     *
     * @returns {RGBModel}
     *  A reference to this instance.
     */
    RGBModel.prototype.gamma = function () {
        this.r = Math.pow(this.r, GAMMA);
        this.g = Math.pow(this.g, GAMMA);
        this.b = Math.pow(this.b, GAMMA);
        return this;
    };

    /**
     * Applies an inverse gamma shift to this color model.
     *
     * @returns {RGBModel}
     *  A reference to this instance.
     */
    RGBModel.prototype.invGamma = function () {
        this.r = Math.pow(this.r, INV_GAMMA);
        this.g = Math.pow(this.g, INV_GAMMA);
        this.b = Math.pow(this.b, INV_GAMMA);
        return this;
    };

    /**
     * Changes this color model to its inverse RGB color.
     *
     * @returns {RGBModel}
     *  A reference to this instance.
     */
    RGBModel.prototype.inv = function () {
        this.r = 1 - this.r;
        this.g = 1 - this.g;
        this.b = 1 - this.b;
        return this;
    };

    /**
     * Changes this color model to its weighted grayscale.
     *
     * @returns {RGBModel}
     *  A reference to this instance.
     */
    RGBModel.prototype.gray = function () {
        this.r = this.g = this.b = this.getLuma();
        return this;
    };

    // class HSLModel =========================================================

    /**
     * Representation of a color using the HSL color model. This class adds the
     * color channels 'h', 's', and 'l' which can be manipulated by the methods
     * of the base class AlphaModel.
     *
     * @constructor
     *
     * @extends AlphaModel
     *
     * @property {Number} h
     *  The value of the hue channel, in the half-open interval [0;360). The
     *  value 0 represents red, 120 represents green, and 240 represents blue.
     *
     * @property {Number} s
     *  The value of the saturation channel, in the interval [0;1]. The value 0
     *  represents gray, the value 1 represents the fully saturated color.
     *
     * @property {Number} l
     *  The value of the luminance channel, in the interval [0;1]. The value 0
     *  represents black (regardless of the other values), and the value 1
     *  represents white.
     */
    var HSLModel = AlphaModel.extend({ constructor: function (h, s, l, a) {

        // base constructor
        AlphaModel.call(this, a);

        // public properties
        this.h = h;
        this.s = s;
        this.l = l;

    } }); // class HSLModel

    // static methods ---------------------------------------------------------

    HSLModel.parseJSON = function (json) {
        var h = Utils.mod(Utils.getIntegerOption(json, 'h', 0) / 1000, 360);
        return new HSLModel(h, getLimNum(json, 's'), getLimNum(json, 'l'), 1);
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns a clone of this HSL color model.
     *
     * @returns {HSLModel}
     *  A clone of this HSL color model.
     */
    HSLModel.prototype.clone = function () {
        return new HSLModel(this.h, this.s, this.l, this.a);
    };

    /**
     * Creates an instance of the class RGBModel representing the color of this
     * HSL color model.
     *
     * @returns {RGBModel}
     *  An instance of the class RGBModel representing the color of this HSL
     *  color model.
     */
    HSLModel.prototype.toRGB = function () {

        var h = this.h / 60,
            r = (Utils.minMax(Math.abs(h - 3) - 1, 0, 1) - 0.5) * this.s + 0.5,
            g = (Utils.minMax(2 - Math.abs(2 - h), 0, 1) - 0.5) * this.s + 0.5,
            b = (Utils.minMax(2 - Math.abs(4 - h), 0, 1) - 0.5) * this.s + 0.5,
            l = 0;

        if (this.l < 0.5) {
            l = 2 * this.l;
            r *= l;
            g *= l;
            b *= l;
        } else if (this.l > 0.5) {
            l = 2 - 2 * this.l;
            r = 1 - (1 - r) * l;
            g = 1 - (1 - g) * l;
            b = 1 - (1 - b) * l;
        }

        return new RGBModel(r, g, b, this.a);
    };

    /**
     * Returns a reference to this instance. Used to ease the conversion from
     * different color models to an HSL color model.
     *
     * @returns {HSLModel}
     *  A reference to this instance.
     */
    HSLModel.prototype.toHSL = function () {
        return this;
    };

    /**
     * Darkens this color with the provided shade value.
     *
     * @param {Number} shade
     *  The shade factor to darken this color, in the closed interval [0;1].
     *  The number 0 will change this color to black, the number 1 will not
     *  modify this color, all numbers inside the interval will darken this
     *  color accordingly.
     *
     * @returns {HSLModel}
     *  A reference to this instance.
     */
    HSLModel.prototype.shade = function (shade) {
        this.l *= shade;
        return this;
    };

    /**
     * Lightens this color with the provided tint value.
     *
     * @param {Number} tint
     *  The tint factor to lighten this color, in the closed interval [0;1].
     *  The number 0 will change this color to white, the number 1 will not
     *  modify this color, all numbers inside the interval will lighten this
     *  color accordingly.
     *
     * @returns {HSLModel}
     *  A reference to this instance.
     */
    HSLModel.prototype.tint = function (tint) {
        this.l = (this.l - 1) * tint + 1;
        return this;
    };

    /**
     * Changes this color model to its complement color (rotates the hue by
     * 180 degrees).
     *
     * @returns {HSLModel}
     *  A reference to this instance.
     */
    HSLModel.prototype.comp = function () {
        return this.offset('h', 180, 360);
    };

    // class ColorDescriptor ==================================================

    /**
     * A descriptor object with details about a resolved document color.
     *
     * @constructor
     *
     * @property {RGBModel} rgb
     *  The RGB color model of the color.
     *
     * @property {HSLModel} hsl
     *  The HSL color model of the color.
     *
     * @property {Number} a
     *  The value of the alpha channel (0 to 1). This property is provided for
     *  convenience, it is also contained in the RGB color model, and in the
     *  HSL color model.
     *
     * @property {Number} y
     *  The luma (weighted lightness) of the color (0 to 1).
     *
     * @property {Boolean} dark
     *  Whether the color is considered to be dark (the luma is less than a
     *  specific threshold value).
     *
     * @property {String} css
     *  The exact CSS representation of the color, as hexadecimal '#RRGGBB'
     *  representation for an opaque color, as function-style rgba() for a
     *  semi-transparent color, or as keyword 'transparent'.
     *
     * @property {String} hex
     *  The upper-case hexadecimal RRGGBB value, ignoring the actual opacity of
     *  the color.
     */
    function ColorDescriptor(model) {

        // both color models
        this.rgb = model.toRGB();
        this.hsl = model.toHSL();

        // alpha channel, and luma
        this.a = model.a;
        this.y = this.rgb.getLuma();
        this.dark = this.y <= Y_DARK;

        // CSS representation (double rounding to workaround rounding errors during RGB/HSL conversion)
        var r8 = Math.round(Math.round(this.rgb.r * 25500000) / 100000);
        var g8 = Math.round(Math.round(this.rgb.g * 25500000) / 100000);
        var b8 = Math.round(Math.round(this.rgb.b * 25500000) / 100000);

        // add plain hexadecimal color representation without alpha channel
        var hex = ('00000' + (r8 * 0x10000 + g8 * 0x100 + b8).toString(16)).substr(-6);
        this.hex = hex.toUpperCase();

        // add CSS color representation ('transparent' for full transparency, RGB hex
        // code for opaque colors, or 'rgba' function style for transparent colors)
        this.css = (this.a === 0) ? 'transparent' : (this.a === 1) ? ('#' + hex) : 'rgba(' + r8 + ',' + g8 + ',' + b8 + ',' + this.a + ')';

    } // class ColorDescriptor

    // color transformations ==================================================

    // all color transformation handlers, mapped by transformation name
    var TRANSFORMATIONS = {

        alpha:    function (model, transform) { return model.set('a', getNum(transform, 'value')); },
        alphaMod: function (model, transform) { return model.modulate('a', getPosNum(transform, 'value')); },
        alphaOff: function (model, transform) { return model.offset('a', getNum(transform, 'value')); },

        red:      function (model, transform) { return model.toRGB().set('r', getNum(transform, 'value')); },
        redMod:   function (model, transform) { return model.toRGB().modulate('r', getPosNum(transform, 'value')); },
        redOff:   function (model, transform) { return model.toRGB().offset('r', getNum(transform, 'value')); },

        green:    function (model, transform) { return model.toRGB().set('g', getNum(transform, 'value')); },
        greenMod: function (model, transform) { return model.toRGB().modulate('g', getPosNum(transform, 'value')); },
        greenOff: function (model, transform) { return model.toRGB().offset('g', getNum(transform, 'value')); },

        blue:     function (model, transform) { return model.toRGB().set('b', getNum(transform, 'value')); },
        blueMod:  function (model, transform) { return model.toRGB().modulate('b', getPosNum(transform, 'value')); },
        blueOff:  function (model, transform) { return model.toRGB().offset('b', getNum(transform, 'value')); },

        hue:      function (model, transform) { return model.toHSL().set('h', Utils.getIntegerOption(transform, 'value', 0) / 1000, 360); },
        hueMod:   function (model, transform) { return model.toHSL().modulate('h', getPosNum(transform, 'value'), 360); },
        hueOff:   function (model, transform) { return model.toHSL().offset('h', Utils.getIntegerOption(transform, 'value', 0) / 1000, 360); },

        sat:      function (model, transform) { return model.toHSL().set('s', getNum(transform, 'value')); },
        satMod:   function (model, transform) { return model.toHSL().modulate('s', getPosNum(transform, 'value')); },
        satOff:   function (model, transform) { return model.toHSL().offset('s', getNum(transform, 'value')); },

        lum:      function (model, transform) { return model.toHSL().set('l', getNum(transform, 'value')); },
        lumMod:   function (model, transform) { return model.toHSL().modulate('l', getPosNum(transform, 'value')); },
        lumOff:   function (model, transform) { return model.toHSL().offset('l', getNum(transform, 'value')); },

        shade:    function (model, transform) { return model.toHSL().shade(getLimNum(transform, 'value')); },
        tint:     function (model, transform) { return model.toHSL().tint(getLimNum(transform, 'value')); },

        gamma:    function (model) { return model.toRGB().gamma(); },
        invGamma: function (model) { return model.toRGB().invGamma(); },

        comp:     function (model) { return model.toHSL().comp(); },
        inv:      function (model) { return model.toRGB().inv(); },
        gray:     function (model) { return model.toRGB().gray(); }
    };

    // class Color ============================================================

    /**
     * Runtime representation of a color of any type supported in document
     * operations.
     *
     * @constructor
     *
     * @param {String} type
     *  The type of the color. Supports all types also supported in document
     *  operations:
     *  - 'auto': The automatic color.
     *  - 'system': Specific colors depending on the operating system.
     *  - 'preset': Predefined colors from a fixed color palette.
     *  - 'crgb': CRGB color model (color channels as percentage, no gamma).
     *  - 'rgb': RGB color model (color channels as 6-digit hexadecimal value).
     *  - 'hsl': HSL color model (hue, saturation, luminance).
     *  - 'scheme': Scheme color from current document theme.
     *
     * @param {String|Object} [value]
     *  The value of the color. Expected data type depends on the color type:
     *  - 'auto': Color value will be ignored and can be omitted.
     *  - 'system', 'preset', 'scheme': A string with the name of the color.
     *  - 'crgb': An object with the integer properties 'r', 'g', and 'b', each
     *      of them in the interval [0;100000] (representing 0% to 100%).
     *  - 'rgb': A string with a 6-digit hexadecimal character (RRGGBB).
     *  - 'hsl': An object with the integer properties 'h', 's', and 'l'. The
     *      property 'h' is expected to be in the interval [0;359999]
     *      (representing 1/1000 of degrees), the other properties have to be
     *      in the interval [0;100000] (representing 0% to 100%).
     *
     * @param {Array} [transforms]
     *  An array with color transformations to be applied to the base color
     *  described by the parameters 'type' and 'value'. Each array element must
     *  be an object with the string property 'type', and the optional integer
     *  property 'value'. Supports all color transformations that are supported
     *  in document operations.
     *
     * @param {String} [defRGB]
     *  A fall-back RGB color value (6-digit hexadecimal number as string). May
     *  be provided by document operations, and can be used in case the color
     *  cannot be resolved correctly (e.g. system colors).
     */
    function Color(type, value, transforms, defRGB) {

        // public methods -----------------------------------------------------

        /**
         * Returns the type of this color.
         *
         * @returns {String}
         *  The type of this color. See constructor parameter 'type' for
         *  details.
         */
        this.getType = function () {
            return type;
        };

        /**
         * Returns whether this color is the automatic color.
         *
         * @returns {Boolean}
         *  Whether this color is the automatic color.
         */
        this.isAuto = function () {
            return type === 'auto';
        };

        /**
         * Returns whether this color is equal to the passed color.
         *
         * @param {Color} color
         *  The color to be compared with this color.
         *
         * @returns {Boolean}
         *  Whether this color is equal to the passed color.
         */
        this.equals = function (color) {
            var jsonColor1 = this.toJSON(), jsonColor2 = color.toJSON();
            // ignore unused properties, e.g. 'fallbackValue'
            return (jsonColor1.type === jsonColor2.type) &&
                _.isEqual(jsonColor1.value, jsonColor2.value) &&
                _.isEqual(jsonColor1.transformations, jsonColor2.transformations);
        };

        /**
         * Returns a deep clone of this color.
         *
         * @returns {Color}
         *  A deep clone of this color.
         */
        this.clone = function () {
            return new Color(type, value, transforms, defRGB);
        };

        /**
         * Appends one or more color transformations to this color.
         *
         * @param {String|Object|Array} tType
         *  The name of a color transformation, a complete color transformation
         *  object, or an array of transformation objects (see constructor
         *  parameter 'transforms' for details).
         *
         * @param {Number} [tValue]
         *  The value for the color transformation. Can be omitted, if the
         *  specified transformations does not need a value (e.g. 'gray'). Will
         *  be ignored, if the first parameter is not a string.
         *
         * @returns {Color}
         *  A reference to this instance.
         */
        this.transform = function (tType, tValue) {
            if (_.isString(tType)) {
                var transform = { type: tType };
                if (_.isNumber(tValue)) { transform.value = tValue; }
                (transforms || (transforms = [])).push(transform);
            } else if (_.isArray(tType)) {
                transforms = transforms ? transforms.concat(tType) : tType.slice();
            } else if (_.isObject(tType)) {
                (transforms || (transforms = [])).push(tType);
            }
            return this;
        };

        /**
         * Returns a color descriptor with details about the color represented
         * by this instance.
         *
         * @param {String|Color} auto
         *  Additional information needed to resolve the automatic color. Can
         *  be a string for predefined automatic color types, or an instance of
         *  the class Color with a type other than the automatic color.
         *  The following color types are supported:
         *  - 'text': for text colors (mapped to black),
         *  - 'line': for line colors (mapped to black),
         *  - 'fill': for fill colors (mapped to full transparency).
         *
         * @param {Theme} [theme]
         *  A document theme object used to map scheme color names to color
         *  values. If missing, theme colors will not be resolved correctly.
         *
         * @returns {ColorDescriptor}
         *  A descriptor for the resulting color.
         */
        this.resolve = function (auto, theme) {

            // the color model containing the resolved color
            var model = null;
            // whether to apply the color transformations
            var applyTransforms = true;

            function createDefaultModel() {
                applyTransforms = false;
                return _.isString(defRGB) ? RGBModel.parseHex(defRGB) : new RGBModel(0, 0, 0, 1);
            }

            // resolve automatic color with explicit color passed in 'auto' parameter
            if ((type === 'auto') && (auto instanceof Color)) {
                if (transforms) { auto = auto.clone().transform(transforms); }
                return auto.resolve('fill', theme);
            }

            // create a color model for all supported color types
            switch (type) {

                // predefined system UI colors without fixed RGB values
                case 'system':
                    model = _.isString(value) ? RGBModel.parseHex(getSystemRGBColor(value)) : createDefaultModel();
                    break;

                // predefined colors with fixed RGB values
                case 'preset':
                    var preset = _.isString(value) ? PRESET_COLOR_MAP[value.toLowerCase()] : null;
                    model = preset ? RGBModel.parseHex(preset.rgb) : createDefaultModel();
                    break;

                // CRGB color model: each channel in the interval [0;100000]
                case 'crgb':
                    model = RGBModel.parseJSON(value).gamma();
                    break;

                // HSL color model (hue, saturation, luminance)
                case 'hsl':
                    model = HSLModel.parseJSON(value);
                    break;

                // simple hexadecimal RGB
                case 'rgb':
                    model = _.isString(value) ? RGBModel.parseHex(value) : createDefaultModel();
                    break;

                // scheme color
                case 'scheme':
                    if (theme && _.isString(value)) {
                        if (value === 'style') {
                            Utils.warn('Color.resolve(): scheme color type "style" not implemented');
                            model = createDefaultModel();
                        } else {
                            model = RGBModel.parseHex(theme.getSchemeColor(value, '000000'));
                        }
                    } else {
                        model = createDefaultModel();
                    }
                    break;

                // automatic color, effective color depends on passed 'auto' parameter
                case 'auto':
                    // resolve automatic color by color type name
                    switch (auto) {
                        case 'text':
                        case 'line':
                            model = new RGBModel(0, 0, 0, 1); // black
                            break;
                        case 'fill':
                            model = new RGBModel(0, 0, 0, 0); // full transparency
                            applyTransforms = false;
                            break;
                        default:
                            Utils.error('Color.resolve(): unknown color type "' + auto + '" for auto color');
                            model = createDefaultModel();
                    }
                    break;

                default:
                    Utils.error('Color.resolve(): unknown color type "' + type + '"');
                    model = createDefaultModel();
            }

            // apply all transformations
            if (applyTransforms && transforms) {
                transforms.forEach(function (transform) {
                    var type = Utils.getStringOption(transform, 'type', ''),
                        handler = TRANSFORMATIONS[type];
                    if (handler) {
                        model = handler(model, transform);
                    } else {
                        Utils.warn('Color.resolve(): unknown color transformation "' + type + '"');
                    }
                });
            }

            // create the color descriptor for the resulting color model
            return new ColorDescriptor(model);
        };

        /**
         * Returns a color descriptor with details about the color represented
         * by this instance, assuming it to be a text color on a specific
         * background style. The automatic color will be resolved to the color
         * black or white, depending on the lightness of the passed background
         * colors. All other colors will be resolved according to the method
         * Color.resolve().
         *
         * @param {Array<Color>} fillColors
         *  The source fill colors, from outermost to innermost level. The last
         *  explicit (non-automatic) fill color in the array will be used as
         *  effective fill color.
         *
         * @param {Theme} [theme]
         *  A document theme object used to map scheme color names to color
         *  values. If missing, theme colors will not be resolved correctly.
         *
         * @returns {ColorDescriptor}
         *  A descriptor for the resulting color.
         */
        this.resolveText = function (fillColors, theme) {

            // the resulting text color
            var textColor = this;
            // the effective fill color
            var fillColor = null;

            // resolve explicit colors directly
            if (this.isAuto()) {

                // find last non-transparent fill color in the passed array
                // TODO: merge semi-transparent colors?
                fillColors.forEach(function (currFillColor) {
                    if (!currFillColor.isAuto()) { fillColor = currFillColor; }
                });

                // resolve to black or white according to lightness of fill color (assume white for missing fill color)
                textColor = new Color('preset', (fillColor && fillColor.resolve('fill', theme).dark) ? 'white' : 'black');
            }

            return textColor.resolve('text', theme);
        };

        /**
         * Returns a color descriptor with details for a color that is a mix of
         * this color instance, and the passed color instance.
         *
         * @param {Color} color
         *  The other color to be mixed with this color.
         *
         * @param {Number} weighting
         *  Specifies how much of the other color will be mixed into this
         *  color. Values equal to or less than zero will result in an exact
         *  copy of THIS color. Values equal to or greater than one will result
         *  in an exact copy of the OTHER color. Values between zero and one
         *  result in a mixed color between this and the other color, where
         *  values close to zero result in colors similar to THIS color, and
         *  values close to one result in colors similar to the OTHER color.
         *
         * @param {String|Color} auto
         *  Additional information needed to resolve the automatic color. See
         *  method Color.resolve() for details.
         *
         * @param {Theme} [theme]
         *  A document theme object used to map scheme color names to color
         *  values. If missing, theme colors will not be resolved correctly.
         *
         * @returns {ColorDescriptor}
         *  A descriptor for the resulting color.
         */
        this.resolveMixed = function (color, weighting, auto, theme) {

            // resolve this color and the other color
            var desc1 = this.resolve(auto, theme);
            var desc2 = color.resolve(auto, theme);

            // early exit if passed weithing is out of the open interval (0,1)
            if (weighting <= 0) { return desc1; }
            if (weighting >= 1) { return desc2; }

            // use the RGBA color model to calculate a mixed color
            var r = desc1.rgb.r + (desc2.rgb.r - desc1.rgb.r) * weighting;
            var g = desc1.rgb.g + (desc2.rgb.g - desc1.rgb.g) * weighting;
            var b = desc1.rgb.b + (desc2.rgb.b - desc1.rgb.b) * weighting;
            var a = desc1.rgb.a + (desc2.rgb.a - desc1.rgb.a) * weighting;

            var rgb = new RGBModel(r, g, b, a);
            return new ColorDescriptor(rgb);
        };

        /**
         * Returns the JSON representation of this color, as used in document
         * operations.
         *
         * @returns {Object}
         *  The JSON representation of this color.
         */
        this.toJSON = function () {
            var json = { type: type };
            if (type !== 'auto') { json.value = value; }
            if (transforms && (transforms.length > 0)) { json.transformations = transforms.slice(); }
            if (_.isString(defRGB)) { json.fallbackValue = defRGB; }
            return json;
        };

        // initialization -----------------------------------------------------

        // create copies of input values to prevent modification from outside
        value = _.isString(value) ? value.toLowerCase() : _.isObject(value) ? _.clone(value) : null;
        transforms = _.isArray(transforms) ? transforms.slice() : null;

    } // class Color

    // static methods ---------------------------------------------------------

    /**
     * Parses the passed JSON representation of a color, and returns a new
     * instance of the class Color.
     *
     * @param {Any} json
     *  The JSON data to be parsed. Should be an object with the properties
     *  supported in document operations.
     *
     * @returns {Color}
     *  A new color instance.
     */
    Color.parseJSON = function (json) {
        return new Color(
            Utils.getStringOption(json, 'type', 'auto'),
            Utils.getOption(json, 'value', null),
            Utils.getArrayOption(json, 'transformations', null),
            Utils.getStringOption(json, 'fallbackValue', null)
        );
    };

    /**
     * Converts the passed CSS color string to an instance of the class Color.
     *
     * @param {String} [value]
     *  The CSS color value. The following CSS colors will be accepted:
     *  - Predefined color names, e.g. 'red'.
     *  - Hexadecimal colors (6-digit, and 3-digit), with leading hash sign.
     *  - RGB colors in function style, using the rgb() or rgba() function.
     *  - HSL colors in function style, using the hsl() or hsla() function.
     *  If missing, this method will return the value null.
     *
     * @param {Boolean} [fill=false]
     *  If set to true, the color is intended to be used as fill color. If the
     *  color is fully transparent, the automatic color will be returned,
     *  instead of an RGB color with the alpha channel set to 0.
     *
     * @returns {Color|Null}
     *  A new color instance, if the passed string could be converted to a
     *  color, otherwise null.
     */
    Color.parseCSS = function (value, fill) {

        // creates an instance of Color for full transparency
        function createTransparentColor(type, val) {
            return fill ? new Color('auto') : new Color(type, val).transform('alpha', 0);
        }

        // creates an instance of Color with optional alpha channel
        function createAlphaColor(type, val, alpha) {
            alpha = _.isString(alpha) ? parseFloat(alpha) : 1;
            if (alpha === 0) { return createTransparentColor(type, val); }
            var color = new Color(type, val);
            if (alpha < 1) { color.transform('alpha', Math.round(alpha * 100000)); }
            return color;
        }

        // no valid string passed
        if (!_.isString(value) || (value.length === 0)) { return null; }

        // CSS keyword 'transparent'
        if (value === 'transparent') { return createTransparentColor('rgb', '000000'); }

        // predefined color names
        var preset = PRESET_COLOR_MAP[value.toLowerCase()];
        if (preset) { return new Color('preset', preset.name); }

        // 6-digit hexadecimal color
        if (/^#[0-9a-f]{6}$/i.test(value)) {
            return new Color('rgb', value.slice(1));
        }

        // 3-digit hexadecimal color
        if (/^#[0-9a-f]{3}$/i.test(value)) {
            return new Color('rgb', value[1] + value[1] + value[2] + value[2] + value[3] + value[3]);
        }

        // RGB/RGBA color in function style
        var rgbParts = value.match(/^rgb\(\s*(\d+)\s*,\s*(\d+)\s*,\s*(\d+)\s*\)$/) || value.match(/^rgba\(\s*(\d+)\s*,\s*(\d+)\s*,\s*(\d+)\s*,\s*(1|0|0?\.\d+)\s*\)$/);
        if (rgbParts) {
            for (var i = 1; i <= 3; ++i) {
                rgbParts[i] = Math.min(255, parseInt(rgbParts[i], 10)).toString(16);
                if (rgbParts[i].length === 1) { rgbParts[i] = '0' + rgbParts[i]; }
            }
            return createAlphaColor('rgb', rgbParts[1] + rgbParts[2] + rgbParts[3], rgbParts[4]);
        }

        // HSL/HSLA color in function style
        var hslParts = value.match(/^hsl\(\s*(\d+)\s*,\s*(\d+)%\s*,\s*(\d+)%\s*\)$/) || value.match(/^hsla\(\s*(\d+)\s*,\s*(\d+)%\s*,\s*(\d+)%\s*,\s*(1|0|0?\.\d+)\s*\)$/);
        if (hslParts) {
            hslParts[1] = Utils.mod(parseInt(hslParts[1], 10), 360) * 1000;
            hslParts[2] = Math.min(100, parseInt(hslParts[2], 10)) * 1000;
            hslParts[3] = Math.min(100, parseInt(hslParts[3], 10)) * 1000;
            return createAlphaColor('hsl', { h: hslParts[1], s: hslParts[2], l: hslParts[3] }, hslParts[4]);
        }

        Utils.warn('Color.parseCSS(): unrecognized CSS color "' + value + '"');
        return null;
    };

    /**
     * Returns whether the passed JSON representations of two colors are equal.
     *
     * @param {Object} jsonColor1
     * The JSON representation of the first color to be compared to the other
     * color.
     *
     * @param {Object} jsonColor2
     * The JSON representation of the second color to be compared to the other
     * color.
     *
     * @returns {Boolean}
     *  Whether the JSON representations of two colors are equal.
     */
    Color.isEqual = function (jsonColor1, jsonColor2) {
        // Convert to Color instance, and use its euqals() method. This normalizes
        // missing properties (e.g. missing 'type' property defaults to type 'auto'),
        // and ignores unused properties (e.g. the 'fallbackValue' property).
        return Color.parseJSON(jsonColor1).equals(Color.parseJSON(jsonColor2));
    };

    // constants --------------------------------------------------------------

    /**
     * The JSON representation of the automatic color.
     *
     * @constant
     */
    Color.AUTO = { type: 'auto' };

    /**
     * The JSON representation of the hyperlink scheme color.
     *
     * @constant
     */
    Color.HYPERLINK = { type: 'scheme', value: 'hyperlink' };

    // exports ================================================================

    return Color;

});
