/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Malte Timmermann <malte.timmermann@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/model/undomanager', [
    'io.ox/office/tk/utils',
    'io.ox/office/baseframework/model/modelobject'
], function (Utils, ModelObject) {

    'use strict';

    // private global functions ===============================================

    /**
     * Helper function for the method UndoAction.appendOperations().
     */
    function insertOperations(array, operations, prepend) {

        // no insertion without actual operations
        if (!operations) { return array; }

        // callback functions and operations cannot be mixed
        if ((array.length > 0) && (_.isFunction(operations) !== _.isFunction(array[0]))) {
            Utils.error('UndoAction.appendOperations(): callback function and operation objects cannot be mixed!');
            return array;
        }

        // create a deep clone of the passed operations
        if (!_.isFunction(operations)) { operations = _.copy(operations, true); }

        // concatenate array according to the passed direction
        return prepend ? [].concat(operations, array) : array.concat(operations);
    }

    // class UndoAction =======================================================

    /**
     * An instance of UndoAction is used by the undo manager to store undo and
     * redo operations and to apply them in the document model.
     *
     * @constructor
     *
     * @param {Object|Array<Object>|Function} [undoOperations]
     *  One ore more operations that form a logical undo action. When the undo
     *  action is executed, the operations will be applied in the exact order
     *  as passed in this parameter; they will NOT be applied in reversed order
     *  as it would be the case if the operations have been added in multiple
     *  undo actions. The parameter may be a single operation object, an array
     *  of operation objects, an arbitrary callback function, or omitted.
     *
     * @param {Object|Array<Object>|Function} [redoOperations]
     *  One ore more operations that form a logical redo action. When the redo
     *  action is executed, the operations will be applied in the exact order
     *  as passed in this parameter. The parameter may be a single operation
     *  object, an array of operation objects, an arbitrary callback function,
     *  or omitted.
     *
     * @param {Object} [undoSelectionState]
     *  A descriptor of the initial selection state that will be stored in the
     *  undo action. If specified, the selection state will be restored after
     *  applying the undo operations of this undo action. See public methods
     *  EditModel.getSelectionState() and EditModel.setSelectionState() for
     *  details.
     *
     * @param {Object} [redoSelectionState]
     *  A descriptor of the resulting selection state that will be stored in
     *  the undo action. If specified, the selection state will be restored
     *  after applying the redo operations of this undo action. See public
     *  methods EditModel.getSelectionState() and EditModel.setSelectionState()
     *  for details.
     */
    function UndoAction(undoOperations, redoOperations, undoSelectionState, redoSelectionState) {

        this.undoOperations = [];
        this.redoOperations = [];
        this.undoSelectionState = undoSelectionState;
        this.redoSelectionState = redoSelectionState;
        this.options = null;

        this.appendOperations(undoOperations, redoOperations);

    } // class UndoAction

    // public methods ---------------------------------------------------------

    /**
     * Appends the passed undo and redo operations to the own operation arrays.
     * Undo operations will be inserted at the beginning of the undo operations
     * array, and redo operations will be appended to the redo operations
     * array.
     *
     * @param {Object|Array<Object>|Function} undoOperations
     *  One ore more operations that form a logical undo action. See the
     *  description of the constructor of this class for details.
     *
     * @param {Object|Array<Object>|Function} redoOperations
     *  One ore more operations that form a logical redo action. See the
     *  description of the constructor of this class for details.
     */
    UndoAction.prototype.appendOperations = function (undoOperations, redoOperations) {

        // add the undo operations at the beginning of the array, accept an arbitrary callback as undo operation
        this.undoOperations = insertOperations(this.undoOperations, undoOperations, true);

        // add the redo operations at the end of the array, accept an arbitrary callback as redo operation
        this.redoOperations = insertOperations(this.redoOperations, redoOperations, false);
    };

    // class UndoManager ======================================================

    /**
     * A collection of undo actions for a document model. Each undo action
     * contains an arbitrary number of document operations to undo, and to redo
     * a specific user action.
     *
     * Triggers the following events:
     * - 'undo:before': Before undo operations are about to be applied by the
     *      method UndoManager.undo().
     * - 'undo:after': After undo operations have been applied by the method
     *      UndoManager.undo().
     * - 'redo:before': Before redo operations are about to be applied by the
     *      method UndoManager.redo().
     * - 'redo:after': After redo operations have been applied by the method
     *      UndoManager.redo().
     * - 'undogroup:open': After a new undo group has been opened by the method
     *      UndoManager.enterUndoGroup().
     * - 'undogroup:close': After an undo group has been closed by the method
     *      UndoManager.enterUndoGroup().
     * - 'change:count': After the number of available undo/redo actions has
     *      changed. Event listeners receive the current number of available
     *      undo and redo actions.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {EditModel} docModel
     *  The document model containing this undo manager instance.
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {Function} [initOptions.mergeUndoActionHandler]
     *      A function that will be called while adding a new undo action onto
     *      the undo stack, and that can try to merge the operations of the new
     *      undo action into the operations of the last undo action already
     *      stored in the undo stack. Receives the following parameters:
     *      (1) {UndoAction} prevAction
     *          The last undo action stored in the undo stack.
     *      (2) {UndoAction} nextAction
     *          The new action about to be pushed onto the undo stack.
     *      Must return true, if the new undo action has been merged into the
     *      existing undo action, in that case the new undo action will be
     *      discarded. All other return values result in pushing the new undo
     *      action onto the undo stack.
     */
    function UndoManager(docModel, initOptions) {

        // self reference
        var self = this;

        // the application instance
        var app = docModel.getApp();

        // callback function that implements merging two actions to one action
        var mergeUndoActionHandler = Utils.getFunctionOption(initOptions, 'mergeUndoActionHandler');

        // all undo actions
        var actions = [];

        // index of the next redo action in the 'actions' array
        var currentAction = 0;

        // number of nested action groups
        var groupLevel = 0;

        // current undo action in grouped mode
        var groupAction = null;

        // whether undo or redo operations are currently processed
        var processing = false;

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel);

        // private methods ----------------------------------------------------

        /**
         * Triggers a 'change:count' event with the current number of available
         * undo actions, and redo actions.
         */
        function triggerChangeCountEvent() {
            self.trigger('change:count', self.getUndoCount(), self.getRedoCount());
        }

        /**
         * Pushes the passed new undo action onto the undo stack. Shortens the
         * undo stack if the current undo action is not the topmost action on
         * the stack.
         *
         * @param {UndoAction} newAction
         *  The new undo action to be pushed onto the undo stack.
         */
        function pushAction(newAction) {

            // last action on action stack
            var lastAction = null;

            // truncate main undo stack and push the new action
            if (currentAction < actions.length) {

                // remove undone actions, push new action without merging
                actions.splice(currentAction);

            } else if (mergeUndoActionHandler && (actions.length > 0)) {

                // custom callback function wants to merge operations of consecutive undo actions
                lastAction = _.last(actions);
                if (mergeUndoActionHandler.call(self, lastAction, newAction) === true) {
                    lastAction.redoSelectionState = newAction.redoSelectionState;
                    return;
                }
            }

            actions.push(newAction);
            currentAction = actions.length;
            triggerChangeCountEvent();
        }

        /**
         * Opens a new undo group. If no undo group is open yet, triggers an
         * 'undogroup:open' event.
         */
        function openUndoGroup() {

            // notify listeners when top-level undo group has been opened
            if (groupLevel === 0) {
                self.trigger('undogroup:open');
            }

            groupLevel += 1;
        }

        /**
         * Closes the most recent undo group. If no undo group is open anymore
         * afterwards, triggers an 'undogroup:close' event.
         *
         * @param {Object} options
         *  Additional options for undo/redo after
         */
        function closeUndoGroup(options) {

            // push existing group action to action stack on last group level
            groupLevel -= 1;
            if ((groupLevel === 0) && groupAction) {
                if (!app.isOperationsBlockActive()) { // the operations were not cancelled
                    groupAction.options = options;
                    pushAction(groupAction);
                }
                groupAction = null;
            }

            // notify listeners when top-level undo group has been closed
            if (groupLevel === 0) {
                self.trigger('undogroup:close');
            }
        }

        /**
         * Invokes the passed callback functions sequentially. Each callback
         * function may run asynchronous code and return a promise.
         *
         * @param {Array<Function>} callbacks
         *  The callback functions to be invoked.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after all callbacks have been
         *  invoked and have finished successfully; ot that will be rejected,
         *  if any callback function returns a rejected promise, or the boolean
         *  value false.
         */
        function invokeCallbacks(callbacks) {
            return callbacks.reduce(function (promise, callback) {
                return promise.then(function () {
                    var result = callback.call(self);
                    return (result === false) ? $.Deferred().reject() : result;
                });
            }, $.when());
        }

        /**
         * Applies the passed operations from an undo action.
         *
         * @param {Array<Object>|Array<Function>} operations
         *  The operations to be applied, either as array of operation objects,
         *  or as array of callback functions.
         *
         * @param {Object} [selectionState]
         *  The new selection state to be set at the document model.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the operations have been
         *  applied successfully (or when no callback function has returned a
         *  rejected promise); otherwise the promise will be rejected.
         */
        function applyOperations(operations, selectionState) {

            // invoke the callback functions, or apply the document operations
            // (bug 30954: always create a deep copy of the operations)
            var promise = _.isFunction(operations[0]) ? invokeCallbacks(operations) :
                docModel.applyOperations(_.copy(operations, true), { async: true });

            // restore the selection state, after all operations have been applied
            return promise.done(function () {
                if (_.isObject(selectionState)) {
                    docModel.setSelectionState(selectionState);
                }
            });
        }

        // methods ------------------------------------------------------------

        /**
         * Returns whether the undo manager is enabled (importing the document
         * has been finished, and the undo manager is currently not processing
         * any undo or redo operations).
         *
         * @returns {Boolean}
         *  Whether the undo manager is enabled.
         */
        this.isUndoEnabled = function () {
            return !processing && this.isImportFinished();
        };

        /**
         * Clears all undo actions from the stack of the undo manager.
         *
         * @returns {UndoManager}
         *  A reference to this instance.
         */
        this.clearUndoActions = function () {
            if (actions.length > 0) {
                actions = [];
                currentAction = 0;
                triggerChangeCountEvent();
            }
            return this;
        };

        /**
         * Clears all undo actions in the stack that are above the current
         * pointer position. This disables the redo stack.
         *
         * @returns {UndoManager}
         *  A reference to this instance.
         */
        this.disableRedo = function () {

            if (currentAction < actions.length) {
                actions.splice(currentAction);
            }
            return this;
        };

        /**
         * Opens an undo action group and executes the specified callback
         * function. All undo operations added by the callback function will be
         * collected in the action group and act like a single undo action.
         * Nested calls are supported.
         *
         * @param {Function} callback
         *  The callback function that will be executed while the undo action
         *  group is open. If the function returns a promise, the undo group
         *  remains opened asynchronously until this object will be resolved or
         *  rejected by the callback.
         *
         * @param {Object} [context]
         *  The context the callback function will be bound to.
         *
         * @param {Object} [options]
         *  Additional options that will be stored in the undo action generated
         *  by this method.
         *
         * @returns {Any}
         *  The return value of the callback function. In asynchronous mode,
         *  this will be the promise returned by the callback function.
         */
        this.enterUndoGroup = function (callback, context, options) {

            // open a new undo group
            openUndoGroup();

            // invoke the callback function, wait for its result
            var result = callback.call(context);
            this.waitForAny($.when(result), function () {
                closeUndoGroup(options);
            });

            return result;
        };

        /**
         * Creates a new undo action that will apply the passed undo and redo
         * operations in the document model.
         *
         * @constructor
         *
         * @param {Object|Array<Object>|Function} [undoOperations]
         *  One ore more operations that form a logical undo action. When the
         *  undo action is executed, the operations will be applied in the
         *  exact order as passed in this parameter; they will NOT be applied
         *  in reversed order (as would be the case if the operations had been
         *  added in multiple undo actions). The parameter may be a single
         *  operation object, an array of operation objects, an arbitrary
         *  callback function, or omitted. If the callback function returns the
         *  boolean value false, or a promise that will be rejected, undoing
         *  this action will be considered to be failed.
         *
         * @param {Object|Array<Object>|Function} [redoOperations]
         *  One ore more operations that form a logical redo action. When the
         *  redo action is executed, the operations will be applied in the
         *  exact order as passed in this parameter. The parameter may be a
         *  single operation object, an array of operation objects, an
         *  arbitrary callback function, or omitted. If the callback function
         *  returns the boolean value false, or a promise that will be
         *  rejected, redoing this action will be considered to be failed.
         *
         * @param {Object} [undoSelectionState]
         *  A descriptor of the initial selection state that will be stored in
         *  the undo action. If specified, the selection state will be restored
         *  after applying the undo operations of this undo action. See methods
         *  EditModel.getSelectionState() and EditModel.setSelectionState() for
         *  details. In grouped undo actions, the selection state of the first
         *  embedded undo action will be used to restore the selection state.
         *
         * @param {Object} [redoSelectionState]
         *  A descriptor of the final selection state that will be stored in
         *  the undo action. If specified, the selection state will be restored
         *  after applying the redo operations of this undo action. See methods
         *  EditModel.getSelectionState() and EditModel.setSelectionState() for
         *  details. In grouped undo actions, the selection state of the last
         *  embedded undo action will be used to restore the selection state.
         *
         * @returns {UndoManager}
         *  A reference to this instance.
         */
        this.addUndo = function (undoOperations, redoOperations, undoSelectionState, redoSelectionState) {

            // check that undo manager is valid and either undo or redo operations have been passed
            if (this.isUndoEnabled() && (_.isObject(undoOperations) || _.isObject(redoOperations))) {

                if (groupLevel > 0) {
                    // active group action: insert operations into its operation arrays
                    if (groupAction) {
                        groupAction.appendOperations(undoOperations, redoOperations);
                        // always use the last selection state (but stick to the first undo selection state)
                        groupAction.redoSelectionState = redoSelectionState;
                    } else {
                        groupAction = new UndoAction(undoOperations, redoOperations, undoSelectionState, redoSelectionState);
                    }
                } else {
                    // create and insert a new action
                    pushAction(new UndoAction(undoOperations, redoOperations, undoSelectionState, redoSelectionState));
                }
            }

            return this;
        };

        /**
         * Returns the number of undo actions available on the undo stack.
         *
         * @returns {Number}
         *  The number of undo actions available on the stack.
         */
        this.getUndoCount = function () {
            return currentAction;
        };

        /**
         * Applies the undo operations of the next undo action.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved or rejected after the undo action
         *  has been applied completely.
         */
        this.undo = function () {

            // nothing to do without an undo action on the stack
            if (currentAction === 0) { return $.when(); }

            this.trigger('undo:before');
            processing = true;

            // apply the undo operations of the next undo action
            currentAction -= 1;
            var undoAction = actions[currentAction];
            var operations = undoAction.undoOperations;
            var promise = applyOperations(operations, undoAction.undoSelectionState);

            // notify listeners after all operations have been applied
            this.waitForAny(promise, function () {
                processing = false;
                // do not pass callback functions to the listeners
                self.trigger('undo:after', _.isFunction(operations[0]) ? [] : operations, undoAction.options);
                triggerChangeCountEvent();
            });

            return promise;
        };

        /**
         * Returns the number of redo actions available on the redo stack.
         *
         * @returns {Number}
         *  The number of redo actions available on the stack.
         */
        this.getRedoCount = function () {
            return actions.length - currentAction;
        };

        /**
         * Applies the redo operations of the next undo action.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved or rejected after the redo action
         *  has been applied completely.
         */
        this.redo = function () {

            // nothing to do without an undo action on the stack
            if (currentAction === actions.length) { return $.when(); }

            this.trigger('redo:before');
            processing = true;

            // apply the redo operations of the next undo action
            var undoAction = actions[currentAction];
            var operations = undoAction.redoOperations;
            var promise = applyOperations(operations, undoAction.redoSelectionState);
            currentAction += 1;

            // notify listeners after all operations have been applied
            this.waitForAny(promise, function () {
                processing = false;
                // do not pass callback functions to the listeners
                self.trigger('redo:after', _.isFunction(operations[0]) ? [] : operations);
                triggerChangeCountEvent();
            });

            return promise;
        };

        // initialization -----------------------------------------------------

        // initial notification about undo stack size
        this.waitForImportSuccess(triggerChangeCountEvent);

        // destroy class members on destruction
        this.registerDestructor(function () {
            self = initOptions = null;
            app = docModel = mergeUndoActionHandler = actions = null;
        });

    } // class UndoManager

    // exports ================================================================

    return ModelObject.extend({ constructor: UndoManager });

});
