/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/model/stylecollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/editframework/utils/operations',
    'io.ox/office/editframework/utils/operationcontext',
    'io.ox/office/editframework/utils/attributeutils'
], function (Utils, TimerMixin, ModelObject, Operations, OperationContext, AttributeUtils) {

    'use strict';

    // class StyleCollection ==================================================

    /**
     * Collection for hierarchical style sheets of a specific attribute family.
     *
     * Attribute definitions that have been registered with the method
     * EditModel.registerAttributeDefinitions() may contain the following
     * additional properties:
     *  - {Function} [definition.format]
     *      A function that applies the passed attribute value to a DOM element
     *      (usually its CSS formatting). Will be called in the context of this
     *      style sheet collection. The function receives the DOM element as
     *      jQuery object in the first parameter, and the attribute value in
     *      the second parameter. An alternative way to update the element
     *      formatting using a complete map of all attribute values is to
     *      specify a global format handler (see options below).
     *  - {Function|Boolean} [definition.preview]
     *      A function that applies the passed attribute value to a DOM element
     *      used to show a preview of a style sheet in the GUI. Will be called
     *      in the context of this style sheet collection. The function
     *      receives the DOM element as jQuery object in the first parameter,
     *      and the attribute value in the second parameter. An alternative way
     *      to update the preview element formatting using a complete map of
     *      all attribute values is to specify a global preview handler (see
     *      options below). If the boolean value 'true' will be set instead of
     *      a function, the registered 'format' method will be called instead.
     *
     * Instances will trigger the following events:
     *  - 'insert:stylesheet': After a new style sheet has been inserted into
     *      this collection. The event handler receives the identifier of the
     *      new style sheet.
     *  - 'delete:stylesheet': After a style sheet has been deleted from this
     *      collection. The event handler receives the identifier of the
     *      deleted style sheet.
     *  - 'change:stylesheet': After the formatting attributes, or other
     *      properties of a style sheet in this collection have been changed.
     *      The event handler receives the identifier of the changed style
     *      sheet.
     *
     * @constructor
     *
     * @extends ModelObject
     * @extends TimerMixin
     *
     * @param {EditModel} docModel
     *  The document model containing this instance.
     *
     * @param {String} styleFamily
     *  The main attribute family represented by the style sheets contained by
     *  instances of the derived collection class. The style sheets and the DOM
     *  elements referring to the style sheets must support all attributes of
     *  this attribute family.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. All callback functions will be called in the
     *  context of this style sheet collection.
     *  @param {Boolean} [initOptions.styleSheetSupport=true]
     *      If set to false, this collection is not allowed to contain style
     *      sheets. It can only be used to format DOM elements with explicit
     *      formatting attributes. DEPRECATED!
     *  @param {String} [initOptions.families]
     *      The names of additional attribute families supported by the style
     *      sheets of this collection (space-separated). It is not needed to
     *      repeat the name of the style family (as passed to the 'styleFamily'
     *      parameter) here.
     *  @param {Object} [initOptions.parentResolvers]
     *      The parent style families whose associated style sheets can contain
     *      attributes of the family supported by this style sheet collection.
     *      The DOM elements referring to the style sheets of the specified
     *      style families must be ancestors of the DOM elements referring to
     *      the style sheets of this container. The passed object maps the
     *      attribute family to an ancestor element resolver function. The
     *      function receives the descendant DOM element as jQuery object in
     *      the first parameter, and returns the ancestor element of that DOM
     *      element which is associated to the parent style family used as map
     *      key.
     *  @param {Function} [initOptions.formatHandler]
     *      A callback function that will be called for every DOM element whose
     *      attributes have been changed, and that will format the DOM element
     *      according to the new attribute values. In difference to the
     *      individual formatter methods specified in the definitions for
     *      single attributes, this handler will be called once for a DOM
     *      element regardless of the number of changed attributes. Receives
     *      the following parameters:
     *      (1) {jQuery} element
     *          The element whose attributes have been changed.
     *      (2) {Object} mergedAttributes
     *          The effective attribute values merged from style sheets and
     *          explicit attributes, as map of attribute maps (name/value
     *          pairs), keyed by attribute family.
     *      (3) {Boolean} async
     *          If set to true, the format handler may execute asynchronously
     *          in a browser timeout loop. In this case, the handler MUST
     *          return a promise that will be resolved after execution.
     *  @param {Function} [initOptions.previewHandler]
     *      A callback function that will be called if a DOM element used in
     *      the GUI to show a preview of a style sheet will be formatted. In
     *      difference to the individual previewer methods specified in the
     *      definitions for single attributes, this handler will be called once
     *      for the DOM element regardless of the number of changed attributes.
     *      Receives the following parameters:
     *      (1) {jQuery} element
     *          The element whose attributes have been changed.
     *      (2) {Object} mergedAttributes
     *          The effective attribute values merged from style sheets and
     *          explicit attributes, as map of attribute maps (name/value
     *          pairs), keyed by attribute family.
     *      (3) {Boolean} async
     *          If set to true, the preview handler may execute asynchronously
     *          in a browser timeout loop. In this case, the handler MUST
     *          return a promise that will be resolved after execution.
     *  @param {Function} [initOptions.styleAttributesResolver]
     *      A function that extracts and returns specific attributes from the
     *      complete 'attributes' object of a style sheet. The attributes
     *      returned by this function may depend on a source element (e.g. on
     *      the position of this source element in its parents). The function
     *      receives the following parameters:
     *      (1) {Object} styleAttributes
     *          The complete original 'attributes' object of the style sheet.
     *          MUST NOT be changed by the callback function!
     *      (2) {jQuery} element
     *          The element referring to the style sheet.
     *      (3) {jQuery} [sourceNode]
     *          The descendant source node.
     *  @param {Function} [initOptions.elementAttributesResolver]
     *      A function that extracts and returns specific attributes from the
     *      explicit attribute map of an element. The attributes returned by
     *      this function may depend on a source element (e.g. on the position
     *      of this source element in its parents). The function receives the
     *      following parameters:
     *      (1) {Object} elementAttributes
     *          The original explicit element attributes of the style sheet, as
     *          map of attribute maps (name/value pairs), keyed by attribute
     *          family. MUST NOT be changed by the callback function!
     *      (2) {jQuery} element
     *          The element whose attributes have been passed in the first
     *          parameter.
     *      (3) {jQuery} [sourceNode]
     *          The descendant source node.
     */
    function StyleCollection(docModel, styleFamily, initOptions) {

        // self reference
        var self = this;

        // all registered style sheets, mapped by unique identifier
        var styleSheetMap = {};

        // the names of all supported attribute families, as flag set
        var supportedFamilySet = null;

        // identifier of the default style sheet
        var defaultStyleId = null;

        // whether this collection supports style sheets
        var styleSheetSupport = Utils.getBooleanOption(initOptions, 'styleSheetSupport', true);

        // the resolver functions for DOM ancestor elements, keyed by attribute family
        var parentResolvers = Utils.getObjectOption(initOptions, 'parentResolvers', {});

        // formatting handler, called after attributes of a model element have been changed
        var formatHandler = Utils.getFunctionOption(initOptions, 'formatHandler');

        // layout handler, that handles including attributes from master and layout slides into merged attributes
        var layoutHandler = Utils.getFunctionOption(initOptions, 'layoutHandler');

        // preview handler, called after attributes of a GUI preview element have been changed
        var previewHandler = Utils.getFunctionOption(initOptions, 'previewHandler');

        // custom resolver for style attributes depending on a context element
        var styleAttributesResolver = Utils.getFunctionOption(initOptions, 'styleAttributesResolver');

        // custom resolver for explicit element attributes depending on a context element
        var elementAttributesResolver = Utils.getFunctionOption(initOptions, 'elementAttributesResolver');

        // base constructors --------------------------------------------------

        ModelObject.call(this, docModel, { trigger: 'always' });
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Returns the identifier of the style sheet that will be effectively
         * used for the specified style sheet identifier. If the specified
         * style sheet does not exist, but this collection contains a default
         * style sheet, its unique identifier will be returned instead.
         * Otherwise, the passed style sheet identifier will be returned.
         *
         * @param {String} styleId
         *  The unique identifier of a style sheet.
         *
         * @returns {String}
         *  The passed style sheet identifier, if such a style sheet exists in
         *  this collection, otherwise the identifier of the default style
         *  sheet.
         */
        function getEffectiveStyleId(styleId) {
            return (!(styleId in styleSheetMap) && (defaultStyleId in styleSheetMap)) ? defaultStyleId : styleId;
        }

        /**
         * Returns whether the passed style sheet is a descendant of the other
         * passed style sheet.
         */
        function isDescendantStyleSheet(styleSheet, ancestorStyleSheet) {
            while (styleSheet) {
                if (styleSheet === ancestorStyleSheet) { return true; }
                styleSheet = styleSheetMap[styleSheet.parentId];
            }
            return false;
        }

        /**
         * Delete all cached merged attribute sets in all style sheets.
         */
        function deleteCachedAttributeSets() {
            _.each(styleSheetMap, function (styleSheet) {
                delete styleSheet.sparseAttributes;
                delete styleSheet.mergedAttributes;
            });
        }

        /**
         * Returns the complete attributes of an ancestor of the passed
         * element, if a parent style family has been registered and its
         * element resolver function returns a valid ancestor element, and
         * extends missing attributes by their current default values.
         *
         * @param {HTMLElement|jQuery} [element]
         *  An element whose ancestor will be searched and queried for its
         *  attributes. If this object is a jQuery collection, uses the first
         *  DOM node it contains. If missing, returns just the current default
         *  values of all supported attribute families.
         *
         * @returns {Object}
         *  The formatting attributes of an ancestor of the passed element, as
         *  map of attribute value maps (name/value pairs), keyed by attribute
         *  family. If no ancestor element has been found, returns the current
         *  default values for all supported attribute families.
         */
        function resolveBaseAttributes(element) {

            var // passed element, as jQuery object
                $element = $(element),
                // the matching parent element, its style family, the style sheet collection, and the attributes
                parentElement = null,
                parentFamily = null,
                parentStyleCollection = null,
                parentAttributes = null,
                // the resulting merged attributes of the ancestor element
                mergedAttributes = {};

            // collect attributes from ancestor element if specified (only one parent style family must match at a time)
            if ($element.length > 0) {

                // find a matching ancestor element and its style family (only one parent must match at a time)
                _.each(parentResolvers, function (elementResolver, family) {
                    if ($(parentElement).length === 0) {
                        parentElement = elementResolver.call(self, $element);
                        parentFamily = family;
                    }
                });

                // add the element attributes of the ancestor element (only the supported attribute families)
                if ($(parentElement).length > 0) {
                    parentStyleCollection = docModel.getStyleCollection(parentFamily);
                    parentAttributes = parentStyleCollection.getElementAttributes(parentElement, { special: true, sourceNode: $element });
                    docModel.extendAttributes(mergedAttributes, parentAttributes, { special: true });
                }
            }

            // remove attribute maps of unsupported families (this also removes the style identifier of the parent element)
            self.filterBySupportedFamilies(mergedAttributes);

            // add missing default attribute values of supported families not found in the parent element
            _.each(supportedFamilySet, function (flag, family) {
                if (!(family in mergedAttributes)) {
                    mergedAttributes[family] = docModel.getDefaultAttributes(family);
                }
            });

            return mergedAttributes;
        }

        /**
         * Returns the attributes from the specified style sheet and its parent
         * style sheets. Does not add default attribute values, or attributes
         * from ancestors of the passed element.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @param {HTMLElement|jQuery} [element]
         *  An element referring to a style sheet in this collection whose
         *  attributes will be extracted. Will be passed to a custom style
         *  attributes resolver (see the option 'styleAttributesResolver'
         *  passed to the constructor). If this object is a jQuery collection,
         *  uses the first DOM node it contains.
         *
         * @param {HTMLElement|jQuery} [sourceNode]
         *  The source DOM node corresponding to a child attribute family that
         *  has initiated the call to this method. Will be passed to a custom
         *  style attributes resolver (see the option 'styleAttributesResolver'
         *  passed to the constructor). If this object is a jQuery collection,
         *  uses the first DOM node it contains.
         *
         * @returns {Object}
         *  The formatting attributes contained in the style sheet and its
         *  ancestor style sheets, as map of attribute value maps (name/value
         *  pairs), keyed by attribute family.
         */
        function resolveStyleAttributeSet(styleId, element, sourceNode) {

            var // passed element, as jQuery object
                $element = $(element),
                // passed source node, as jQuery object
                $sourceNode = $(sourceNode),
                // the resulting merged attributes of the style sheet and its ancestors
                mergedAttributes = {};

            // collects style sheet attributes recursively through parents
            function collectStyleAttributes(styleSheet) {

                // exit recursive call chain if no more parent styles are available
                if (!styleSheet) { return; }

                // call recursively to get the attributes of the parent style sheets
                collectStyleAttributes(styleSheetMap[styleSheet.parentId]);

                // add all attributes of the current style sheet, mapped directly by attribute family
                docModel.extendAttributes(mergedAttributes, styleSheet.attributes);

                // try user-defined resolver for style attributes mapped in non-standard structures
                if (_.isFunction(styleAttributesResolver) && ($element.length > 0)) {
                    docModel.extendAttributes(mergedAttributes, styleAttributesResolver.call(self, styleSheet.attributes, $element, $sourceNode));
                }
            }

            // fall-back to default style sheet if passed identifier is invalid
            styleId = getEffectiveStyleId(styleId);

            // collect attributes from the style sheet and its parents
            collectStyleAttributes(styleSheetMap[styleId]);

            // remove attribute maps of unsupported families (and the style identifier)
            self.filterBySupportedFamilies(mergedAttributes);

            // restore identifiers of style sheet
            mergedAttributes.styleId = styleId;

            return mergedAttributes;
        }

        /**
         * Returns the explicit attributes from the specified element. Does not
         * add default attribute values. Uses a custom element attributes
         * resolver function if specified in the constructor of this style
         * sheet collection.
         *
         * @param {HTMLElement|jQuery} [element]
         *  The element whose explicit attributes will be extracted. Will be
         *  passed to a custom attributes resolver (see the option
         *  'elementAttributesResolver' passed to the constructor). If this
         *  object is a jQuery collection, uses the first DOM node it contains.
         *
         * @param {HTMLElement|jQuery} [sourceNode]
         *  The source DOM node corresponding to a child attribute family that
         *  has initiated the call to this method. Will be passed to a custom
         *  element attributes resolver (see the option
         *  'elementAttributesResolver' passed to the constructor). If this
         *  object is a jQuery collection, uses the first DOM node it contains.
         *
         * @returns {Object}
         *  The explicit formatting attributes contained in the element, as map
         *  of attribute value maps (name/value pairs), keyed by attribute
         *  family.
         */
        function resolveElementAttributeSet(element, sourceNode) {

            var // the resulting merged attributes of the element
                mergedAttributes = AttributeUtils.getExplicitAttributes(element, { direct: true });

            // call custom element attribute resolver
            if (_.isFunction(elementAttributesResolver)) {
                mergedAttributes = elementAttributesResolver.call(self, mergedAttributes, $(element), $(sourceNode));
            }

            return mergedAttributes;
        }

        /**
         * Updates the element formatting according to the passed attributes.
         *
         * @param {jQuery} element
         *  The element whose formatting will be updated, as jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A complete attribute set with all attribute values merged from
         *  style sheet and explicit attributes.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.preview=false]
         *      If set to true, a GUI element used to preview the formatting of
         *      a style sheet will be rendered. Executes the 'preview' callback
         *      functions instead of the regular 'format' callback functions
         *      registered in the attribute definitions, and calls the preview
         *      handler instead of the format handler registered at this class
         *      instance.
         *  @param {Boolean} [options.async=false]
         *      If set to true, the registered format or preview handler will
         *      be asked to execute asynchronously in a browser timeout loop.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the registered format or
         *  preview handler has finished. If the option 'async' is set, and the
         *  format handler supports asynchronous execution, the promise will be
         *  resolved sometime in the future. Otherwise, the returned promise is
         *  already resolved when this method returns.
         */
        function updateElementFormatting(element, mergedAttributes, options) {

            var // definitions of own attributes
                definitions = docModel.getAttributeDefinitions(styleFamily),
                // whether to render a GUI preview element
                preview = Utils.getBooleanOption(options, 'preview', false),
                // whether to call the real format handler or the preview handler
                currentFormatHandler = preview ? previewHandler : formatHandler,
                // the resulting promise returned from the format or preview handler
                resultPromise = null;

            if (!preview) {
                self.DBG_COUNT = (self.DBG_COUNT || 0) + 1;
            }

            // call single format handlers for all attributes of the own style family
            _.each(mergedAttributes[styleFamily], function (value, name) {
                var formatMethod = null;
                if (name in definitions) {
                    formatMethod = Utils.getFunctionOption(definitions[name], 'format');
                    if (preview && (Utils.getBooleanOption(definitions[name], 'preview', false) !== true)) {
                        formatMethod = Utils.getFunctionOption(definitions[name], 'preview');
                    }
                    if (_.isFunction(formatMethod)) {
                        formatMethod.call(self, element, value);
                    }
                }
            });

            // call format handler taking all attributes at once
            if (_.isFunction(currentFormatHandler)) {
                try {
                    resultPromise = currentFormatHandler.call(self, element, mergedAttributes, Utils.getBooleanOption(options, 'async', false));
                } catch (ex) {
                    Utils.exception(ex);
                    resultPromise = $.Deferred().reject();
                }
            }

            // wrap resultPromise in $.when() to convert plain values to a promise
            return $.when(resultPromise);
        }

        // protected methods --------------------------------------------------

        /**
         * Callback handler for the document operation 'insertStyleSheet'.
         * Inserts a new style sheet into this collection.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'insertStyleSheet' document operation.
         *
         * @param {Function} [callback]
         *  A callback the will be invoked with the new style sheet as first
         *  parameter. Used by internal code to create specialized style sheets
         *  with some non-standard properties.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyInsertStyleSheetOperation = function (context, callback) {

            // check that style sheet handling is enabled
            context.ensure(styleSheetSupport, 'creating style sheets not allowed for style family \'%s\'', styleFamily);

            // check that the passed operation represents the correct style family
            context.ensure(context.getStr('type') === styleFamily, 'invalid attribute family for style sheet');

            // the unique identifier of the style sheet
            var styleId = context.getStr('styleId');

            // TODO: disallow multiple insertion of style sheets? (predefined style sheets may be created on demand without checking existence)
            if ((styleId in styleSheetMap) && !styleSheetMap[styleId].dirty) {
                Utils.warn('StyleCollection.applyInsertStyleSheetOperation(): style sheet \'%s\' exists already', styleId);
            }

            // get or create a style sheet object, set identifier and user-defined name of the style sheet
            var styleSheet = styleSheetMap[styleId] || (styleSheetMap[styleId] = {});
            styleSheet.id = styleId;
            styleSheet.name = context.getOptStr('styleName', styleId);

            // set parent of the style sheet, check for cyclic references
            styleSheet.parentId = context.getOptStr('parent');
            if (isDescendantStyleSheet(styleSheetMap[styleSheet.parentId], styleSheet)) {
                Utils.warn('StyleCollection.applyInsertStyleSheetOperation(): cyclic reference, cannot set style sheet parent "' + styleSheet.parentId + '"');
                styleSheet.parentId = null;
            }

            // set style sheet options
            styleSheet.hidden = context.getOptBool('hidden');
            styleSheet.priority = context.getOptInt('uiPriority');
            styleSheet.custom = context.getOptBool('custom');

            // bug 27716: set first visible style as default style, if imported default style is hidden
            if (_.isString(defaultStyleId) && (defaultStyleId in styleSheetMap) && styleSheetMap[defaultStyleId].hidden && !styleSheet.hidden) {
                defaultStyleId = styleId;
            }

            // set default style sheet
            if (context.getOptBool('default')) {
                if (_.isNull(defaultStyleId)) {
                    defaultStyleId = styleId;
                } else if (defaultStyleId !== styleId) {
                    Utils.warn('StyleCollection.applyInsertStyleSheetOperation(): multiple default style sheets "' + defaultStyleId + '" and "' + styleId + '"');
                }
            }

            // store a deep clone of the passed attributes
            styleSheet.attributes = _.copy(context.getOptObj('attrs', {}), true);

            // delete all cached merged attributes (parent tree may have changed)
            deleteCachedAttributeSets();

            // invoke user callback
            if (_.isFunction(callback)) { callback.call(this, styleSheet); }

            // notify listeners
            this.trigger('insert:stylesheet', styleId);
        };

        /**
         * Callback handler for the document operation 'deleteStyleSheet'.
         * Removes an existing style sheet from this collection, and triggers a
         * 'delete:stylesheet' on success.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'deleteStyleSheet' document operation.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyDeleteStyleSheetOperation = function (context) {

            // check that the passed operation represents the correct style family
            context.ensure(context.getStr('type') === styleFamily, 'invalid attribute family for style sheet');

            // the unique identifier of the style sheet
            var styleId = context.getStr('styleId');
            context.ensure(styleId in styleSheetMap, 'style sheet \'%s\' does not exist', styleId);

            // default style sheet cannot be deleted
            context.ensure(styleId !== defaultStyleId, 'cannot remove default style sheet');

            // update parent of all style sheets referring to the removed style sheet
            _.each(styleSheetMap, function (childSheet) {
                if (styleId === childSheet.parentId) {
                    childSheet.parentId = styleSheetMap[styleId].parentId;
                }
            });

            // the style sheet to be deleted
            var styleSheet = styleSheetMap[styleId];

            // special behavior: built-in style sheets will not be deleted but invalidated
            if (styleSheet.builtIn) {
                styleSheet.dirty = true;
                return;
            }

            // remove the style sheet from the map, and delete all cached merged attributes
            // (parent tree may have changed)
            delete styleSheetMap[styleId];
            deleteCachedAttributeSets();

            // notify listeners
            this.trigger('delete:stylesheet', styleId);
        };

        /**
         * Callback handler for the document operation 'changeStyleSheet'.
         * Changes the properties, or formatting attributes of an existing
         * style sheet in this collection, and triggers a 'change:stylesheet'
         * on success.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'changeStyleSheet' document operation.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyChangeStyleSheetOperation = function (context) {

            // check that the passed operation represents the correct style family
            context.ensure(context.getStr('type') === styleFamily, 'invalid attribute family for style sheet');

            // the unique identifier of the style sheet
            var styleId = context.getStr('styleId');
            context.ensure(styleId in styleSheetMap, 'style sheet \'%s\' does not exist', styleId);

            // the style sheet to be changed
            var styleSheet = styleSheetMap[styleId];

            // change the readable name of the style sheet
            if (context.has('styleName')) { styleSheet.name = context.getStr('styleName'); }

            // change the formatting attributes
            if (context.has('attrs')) { styleSheet.attributes = context.getObj('attrs'); }

            // delete all cached merged attributes (parent tree, or descendant style sheets may have changed)
            deleteCachedAttributeSets();

            // notify listeners
            this.trigger('change:stylesheet', styleId);
        };

        // public methods ------------------------------------------------------------

        /**
         * Returns the style family name of this style sheet collection.
         *
         * @returns {String}
         *  The style family name of this style sheet collection.
         */
        this.getStyleFamily = function () {
            return styleFamily;
        };

        /**
         * Returns a set object with the names of all attribute families
         * supported by style sheets contained in this instance, including the
         * main style family.
         *
         * @returns {Object}
         *  A set object containing the names of all supported attribute
         *  families as property keys.
         */
        this.getSupportedFamilies = function () {
            return supportedFamilySet;
        };

        /**
         * Removes all attribute maps from the passed attribute set in-place
         * that are not mapped by an attribute family supported by this style
         * sheet collection. Additionally, all existing but empty attribute
         * maps will be removed.
         *
         * @param {Object} attributeSet
         *  (in/out parameter) The attribute set to be filtered. All properties
         *  of this object whose key is not a supported attribute family will
         *  be deleted.
         *
         * @returns {Object}
         *  The filtered attribute set passed to this method (has been modified
         *  in-place), for convenience.
         */
        this.filterBySupportedFamilies = function (attributeSet) {

            // remove all properties not keyed by a supported attribute family
            _.each(attributeSet, function (attributes, family) {
                if (!(family in supportedFamilySet) || !_.isObject(attributes) || _.isEmpty(attributes)) {
                    delete attributeSet[family];
                }
            });

            // return the modified attribute set for convenience
            return attributeSet;
        };

        /**
         * Removes all unsupported attributes from the passed attribute set
         * in-place. Additionally, all existing but empty attribute maps will
         * be removed.
         *
         * @param {Object} attributeSet
         *  (in/out parameter) The attribute set to be filtered.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.special=false]
         *      If set to true, special attributes (attributes that are marked
         *      with the 'special' flag in the attribute definition) are
         *      considered to be supported. Otherwise, special attributes will
         *      be filtered.
         *
         * @returns {Object}
         *  The filtered attribute set passed to this method (has been modified
         *  in-place), for convenience.
         */
        this.filterBySupportedAttributes = function (attributeSet, options) {

            _.each(attributeSet, function (attributes, family) {

                // attribute definitions of the current family
                var definitions = (family in supportedFamilySet) ? docModel.getAttributeDefinitions(family) : null;

                // process all attributes, if the current attribute family is supported
                if (definitions) {
                    _.each(attributes, function (value, name) {
                        if (!AttributeUtils.isRegisteredAttribute(definitions, name, options)) {
                            delete attributes[name];
                        }
                    });

                    if (_.isEmpty(attributes)) {
                        delete attributeSet[family];
                    }
                    return;
                }

                // delete the current property, unless it is a (supported) style sheet reference
                if (!styleSheetSupport || (family !== 'styleId')) {
                    delete attributeSet[family];
                }
            });

            // return the modified attribute set for convenience
            return attributeSet;
        };

        /**
         * Returns the identifier of the default style sheet.
         *
         * @returns {String|Null}
         *  The identifier of the default style sheet.
         */
        this.getDefaultStyleId = function () {
            return defaultStyleId;
        };

        /**
         * Inserts a new 'dirty' (predefined) style sheet into this collection.
         *
         * @param {String} styleId
         *  The unique identifier of of the new style sheet.
         *
         * @param {String} styleName
         *  The user-defined name of of the new style sheet.
         *
         * @param {Object} attributes
         *  The formatting attributes contained in the new style sheet. The
         *  structure of this object is dependent on the style family of this
         *  collection.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.parent]
         *      The identifier of of the parent style sheet the new style sheet
         *      will derive undefined attributes from.
         *  @param {String} [options.category]
         *      If specified, the name of a category that can be used to group
         *      style sheets into several sections in the user interface.
         *  @param {Number} [options.priority=0]
         *      The sorting priority of the style (the lower the value the
         *      higher the priority).
         *  @param {Boolean} [options.defStyle=false]
         *      True, if the new style sheet is the default style sheet of this
         *      style sheet collection. The default style will be used for all
         *      elements without explicit style sheet. Only the first style
         *      sheet that will be inserted with this flag will be permanently
         *      registered as default style sheet.
         *  @param {Boolean} [options.hidden=false]
         *      If set to true, the style sheet should be displayed in the user
         *      interface.
         *  @param {Boolean} [options.builtIn=false]
         *      Whether the style sheet is predefined (provided in a new empty
         *      document).
         *  @param {Boolean} [options.custom=false]
         *      Whether the style sheet has been customized (changed by the
         *      user, especially useful for built-in style sheets).
         *
         * @returns {Boolean}
         *  Whether the new style sheet has been created successfully.
         */
        this.createDirtyStyleSheet = function (styleId, styleName, attributeSet, options) {

            // the document operation that will be executed to insert the style sheet
            var operation = { type: styleFamily, styleId: styleId, styleName: styleName, attrs: attributeSet, dirty: true };

            // add parent style identifier
            var parentStyle = Utils.getStringOption(options, 'parent', null);
            if (parentStyle) { operation.parent = parentStyle; }

            // add other options supported by the operation
            operation.uiPriority = Utils.getIntegerOption(options, 'priority', 0);
            operation.hidden = Utils.getBooleanOption(options, 'hidden', false);
            operation.custom = Utils.getBooleanOption(options, 'custom', false);

            // apply the operation (should not throw, just in case)
            try {
                var context = new OperationContext(docModel, operation, false);
                this.applyInsertStyleSheetOperation(context, function (styleSheet) {
                    styleSheet.category = Utils.getStringOption(options, 'category', null);
                    styleSheet.builtIn = Utils.getBooleanOption(options, 'builtIn', false);
                    styleSheet.dirty = true;
                });
                return true;
            } catch (error) {
                Utils.exception(error);
                return false;
            }
        };

        /**
         * Returns whether this collection contains the specified style sheet.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @returns {Boolean}
         *  Whether this instance contains a style sheet with the passed
         *  identifier.
         */
        this.containsStyleSheet = function (styleId) {
            return styleId in styleSheetMap;
        };

        /**
         * Returns whether this collection contains a style sheet with the
         * specified name (not identifier).
         *
         * @param {String} styleName
         *  The name of the style sheet.
         *
         * @returns {Boolean}
         *  Whether this instance contains a style sheet with the passed name.
         */
        this.containsStyleSheetByName = function (styleName) {
            styleName = styleName.toLowerCase();
            return _.any(styleSheetMap, function (styleSheet) {
                return styleSheet.name.toLowerCase() === styleName;
            });
        };

        /**
         * Returns the style identifier of the first style sheet found with the
         * specified name.
         *
         * @param {String} styleName
         *  The name of the style sheet.
         *
         * @returns {String|Null}
         *  Whether this instance contains a style sheet with the passed name.
         */
        this.getStyleIdByName = function (styleName) {
            var resultId = null;
            styleName = styleName.toLowerCase();
            _.any(styleSheetMap, function (styleSheet, styleId) {
                if (styleSheet.name.toLowerCase() === styleName) {
                    resultId = styleId;
                    return true;
                }
            });
            return resultId;
        };

        /**
         * Returns the names of all style sheets in a map, keyed by their
         * unique identifiers.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.skipHidden=false]
         *      If set to true, the names of hidden style sheets will not be
         *      returned.
         *
         * @returns {Object}
         *  A map with all style sheet names, keyed by style sheet identifiers.
         */
        this.getStyleSheetNames = function (options) {

            var // whether to skip hidden style sheets
                skipHidden = Utils.getBooleanOption(options, 'skipHidden', false),
                // the result map
                names = {};

            _.each(styleSheetMap, function (styleSheet, styleId) {
                if (!skipHidden || !styleSheet.hidden) {
                    names[styleId] = styleSheet.name;
                }
            });
            return names;
        };

        /**
         * Returns the merged attributes from the specified style sheet and its
         * parent style sheets.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.skipDefaults=false]
         *      If set to true, the returned attribute set will NOT contain the
         *      default values of attributes not contained in the style sheet
         *      (or its ancestors). This means the attribute set returned from
         *      this method MAY NOT be complete.
         *
         * @returns {Object}
         *  A complete (except when using the option 'skipDefaults', see above)
         *  attribute set (a map of attribute maps with name/value pairs, keyed
         *  by the attribute families, additionally containing the style sheet
         *  reference in the 'styleId' property), containing values for all
         *  supported attributes, collected from the specified style sheet, and
         *  all its parent style sheets, up to the map of default attributes.
         *  For efficiency, a reference to the cached attribute set will be
         *  returned. ATTENTION! For efficiency, a reference to the cached
         *  attribute set will be returned. This object MUST NOT be changed!
         */
        this.getStyleAttributeSet = function (styleId, options) {

            // the style sheet entry
            var styleSheet = styleSheetMap[getEffectiveStyleId(styleId)];
            // whether to skip the default attribute values
            var skipDefaults = Utils.getBooleanOption(options, 'skipDefaults', false);
            // the cached incomplete style sheet attributes
            var sparseAttributeSet = styleSheet ? styleSheet.sparseAttributes : null;
            // the cached complete style sheet attributes
            var mergedAttributeSet = styleSheet ? styleSheet.mergedAttributes : null;

            // the effective style sheet attributes according to passed options
            var attributeSet = skipDefaults ? sparseAttributeSet : mergedAttributeSet;
            if (attributeSet) { return attributeSet; }

            // calculate the existing attributes of the style sheet, and its ancestors
            if (!sparseAttributeSet) {
                sparseAttributeSet = resolveStyleAttributeSet(styleId);
                if (styleSheet) { styleSheet.sparceAttributes = sparseAttributeSet; }
            }

            // return the incomplete attributes if requested
            if (skipDefaults) { return sparseAttributeSet; }

            // start with the attribute default values, extend with attributes of specified style sheet
            attributeSet = resolveBaseAttributes();
            docModel.extendAttributes(attributeSet, sparseAttributeSet);
            if (styleSheet) { styleSheet.mergedAttributes = attributeSet; }
            return attributeSet;
        };

        /**
         * Returns the merged attribute set of the default style sheet of this
         * collection.
         *
         * @returns {Object}
         *  The complete merged attribute set of the default style sheet of
         *  this collection.
         *  ATTENTION! For efficiency, a reference to the cached attribute set
         *  will be returned. This object MUST NOT be changed!
         */
        this.getDefaultStyleAttributeSet = function () {
            return this.getStyleAttributeSet(defaultStyleId || '');
        };

        /**
         * Returns the UI category for the specified style sheet.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @returns {String}
         *  The UI category. If the style sheet does not exist, returns null.
         */
        this.getUICategory = function (styleId) {
            return (styleId in styleSheetMap) ? styleSheetMap[styleId].category : null;
        };

        /**
         * Returns the UI priority for the specified style sheet.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @returns {Number}
         *  The UI priority. If the style sheet does not exist, returns 0.
         */
        this.getUIPriority = function (styleId) {
            return (styleId in styleSheetMap) ? styleSheetMap[styleId].priority : 0;
        };

        /**
         * Returns the identifier of the parent style sheet for the specified
         * style sheet.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @returns {String|Null}
         *  The parent id of the style sheet; or null, if no parent exists.
         */
        this.getParentId = function (styleId) {
            return (styleId in styleSheetMap) ? styleSheetMap[styleId].parentId : null;
        };

        /**
         * Returns whether the specified style sheet is hidden in the user
         * interface.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @returns {Boolean|Null}
         *  True, if the specified style sheet is hidden; false, if the style
         *  sheet is visible; or null, if the style sheet does not exist.
         */
        this.isHidden = function (styleId) {
            return (styleId in styleSheetMap) ? styleSheetMap[styleId].hidden : null;
        };

        /**
         * Returns the user defined name for the specified style sheet.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @returns {String|Null}
         *  The user defined name of the style sheet; or null, if the style
         *  sheet does not exist.
         */
        this.getName = function (styleId) {
            return (styleId in styleSheetMap) ? styleSheetMap[styleId].name : null;
        };

        /**
         * @param {String} styleId
         *
         * @returns {Boolean|Null}
         */
        this.isCustom = function (styleId) {
            return (styleId in styleSheetMap) ? styleSheetMap[styleId].custom : null;
        };

        /**
         * Returns whether the specified style sheet is dirty.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @returns {Boolean|Null}
         *  The dirty state of the style sheet; or null, if the style sheet
         *  does not exist.
         */
        this.isDirty = function (styleId) {
            return (styleId in styleSheetMap) ? styleSheetMap[styleId].dirty : null;
        };

        /**
         * Changes the dirty state of the specified style sheet.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @param {Boolean} dirty
         */
        this.setDirty = function (styleId, dirty) {
            if (styleId in styleSheetMap) {
                styleSheetMap[styleId].dirty = dirty;
            }
        };

        /**
         * Returns the explicit attribute set of the specified style sheet,
         * without resolving the parent style sheets, or converting to the
         * attributes of a specific attribute family.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @returns {Object}
         *  The explicit attribute set contained in the style sheet, as a deep
         *  clone.
         */
        this.getStyleSheetAttributeMap = function (styleId) {
            return (styleId in styleSheetMap) ? _.copy(styleSheetMap[styleId].attributes, true) : {};
        };

        /**
         * Builds a complete attribute set containing all formatting attributes
         * of the own style family, and of the additional attribute families
         * supported by the style sheets in this collection, all set to the
         * null value.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.style=false]
         *      If set to true, the resulting attribute set will contain an
         *      additional property 'styleId' (reference to the style sheet),
         *      also set to the value null.
         *
         * @returns {Object}
         *  A complete attribute set with null values for all attributes
         *  registered for all supported attribute families.
         */
        this.buildNullAttributeSet = function (options) {
            var attributeSet = docModel.buildNullAttributeSet(supportedFamilySet);
            if (styleSheetSupport && Utils.getBooleanOption(options, 'style', false)) {
                attributeSet.styleId = null;
            }
            return attributeSet;
        };

        /**
         * Returns the merged values of the formatting attributes from a style
         * sheet and explicit attributes.
         *
         * @param {Object} attributes
         *  An (incomplete) attribute set containing an optional style sheet
         *  identifier in the property 'styleId', and explicit attributes.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.special=false]
         *      If set to true, includes special attributes (attributes that
         *      are marked with the 'special' flag in the attribute definitions
         *      passed to the constructor) to the result set.
         *  @param {Boolean} [options.skipDefaults=false]
         *      If set to true, the returned attribute set will NOT contain the
         *      default values of attributes not contained in the style sheet
         *      (or its ancestors). This means the attribute set returned from
         *      this method MAY NOT be complete.
         *
         * @returns {Object}
         *  A complete (except when using the option 'skipDefaults', see above)
         *  merged attribute set with the values of all formatting attributes
         *  supported by this collection.
         */
        this.getMergedAttributes = function (attributes, options) {

            // the identifier of the style sheet
            var styleId = Utils.getStringOption(attributes, 'styleId', '');
            // resulting merged attributes (start with defaults and style sheet attributes)
            var mergedAttributes = _.copy(this.getStyleAttributeSet(styleId, options), true);

            // add the passed explicit attributes (protect effective style sheet identifier)
            styleId = mergedAttributes.styleId;
            docModel.extendAttributes(mergedAttributes, attributes, options);
            mergedAttributes.styleId = styleId;

            // filter by supported attributes, according to the passed options
            return this.filterBySupportedAttributes(mergedAttributes, options);
        };

        /**
         * Returns the values of the formatting attributes in the specified DOM
         * element.
         *
         * @param {HTMLElement|jQuery} element
         *  The element whose attributes will be returned. If this object is a
         *  jQuery collection, uses the first DOM node it contains.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {HTMLElement|jQuery} [options.sourceNode]
         *      A descendant of the passed element associated to a child
         *      attribute family. Will be passed to a style attribute resolver
         *      callback function where it might be needed to resolve the
         *      correct attributes according to the position of this source
         *      node.
         *  @param {Boolean} [options.special=false]
         *      If set to true, includes special attributes (attributes that
         *      are marked with the 'special' flag in the attribute definitions
         *      passed to the constructor) to the result map.
         *
         * @returns {Object}
         *  A complete merged attribute set with the values of all formatting
         *  attributes supported by this collection.
         */
        this.getElementAttributes = function (element, options) {

            var // the descendant source node
                sourceNode = Utils.getOption(options, 'sourceNode'),
                // the identifier of the style sheet referred by the element
                styleId = AttributeUtils.getElementStyleId(element),
                // resulting merged attributes (start with defaults, parent element, and style sheet attributes)
                mergedAttributes = resolveBaseAttributes(element);

            // add attributes of the style sheet and its parents
            docModel.extendAttributes(mergedAttributes, resolveStyleAttributeSet(styleId, element, sourceNode));

            // call the passed layout listener to add attributes from layout/master slide
            if (_.isFunction(layoutHandler)) { layoutHandler.call(self, $(element), mergedAttributes); }

            // add the explicit attributes of the element (protect effective style sheet identifier)
            styleId = mergedAttributes.styleId;
            docModel.extendAttributes(mergedAttributes, resolveElementAttributeSet(element, sourceNode), options);
            mergedAttributes.styleId = styleId;

            // filter by supported attributes, according to the passed options
            return this.filterBySupportedAttributes(mergedAttributes, options);
        };

        /**
         * Changes specific formatting attributes in the specified DOM element.
         *
         * @param {HTMLElement|jQuery} element
         *  The element whose attributes will be changed. If this object is a
         *  jQuery collection, uses the first DOM node it contains.
         *
         * @param {Object} attributes
         *  An (incomplete) attribute set with all formatting attributes to be
         *  changed. To clear an explicit attribute value (thus defaulting to
         *  the current style sheet), the respective value in this attribute
         *  set has to be set explicitly to the null value. Attributes missing
         *  in this attribute set will not be modified at the element.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.clear=false]
         *      If set to true, explicit element attributes that are equal to
         *      the attributes of the current style sheet will be removed from
         *      the element.
         *  @param {Boolean} [options.special=false]
         *      If set to true, allows to change special attributes (attributes
         *      that are marked with the 'special' flag in the attribute
         *      definitions passed to the constructor).
         *  @param {Boolean} [options.preview=false]
         *      If set to true, a GUI element used to preview the formatting of
         *      a style sheet will be rendered. Executes the 'preview' callback
         *      functions instead of the regular 'format' callback functions
         *      registered in the attribute definitions, and calls the preview
         *      handler instead of the format handler registered at this class
         *      instance.
         *  @param {Function} [options.changeListener]
         *      If specified, will be called if the attributes of the element
         *      have been changed. Will be called in the context of this style
         *      sheet collection. Receives the passed element as first
         *      parameter, the old explicit attribute set as second parameter,
         *      and the new explicit attribute set as third parameter.
         *
         * @returns {StyleCollection}
         *  A reference to this instance.
         */
        this.setElementAttributes = function (element, attributes, options) {

            var // whether to remove element attributes equal to style attributes
                clear = Utils.getBooleanOption(options, 'clear', false),
                // whether a preview element is rendered
                preview = Utils.getBooleanOption(options, 'preview', false),
                // change listener notified for changed attributes
                changeListener = Utils.getFunctionOption(options, 'changeListener'),
                // new style sheet identifier
                styleId = attributes.styleId,

                // the existing explicit element attributes
                oldElementAttributes = AttributeUtils.getExplicitAttributes(element, { direct: true }),
                // new explicit element attributes (clone, there may be multiple elements pointing to the same data object)
                newElementAttributes = _.copy(oldElementAttributes, true),
                // merged attribute values from style sheets and explicit attributes
                mergedAttributes = resolveBaseAttributes(element);

            // add/remove new style sheet identifier, or get current style sheet identifier
            if (_.isString(styleId)) {
                newElementAttributes.styleId = styleId;
            } else if (_.isNull(styleId)) {
                delete newElementAttributes.styleId;
            } else {
                styleId = AttributeUtils.getElementStyleId(element);
            }

            // collect all attributes of the new or current style sheet, and its parents
            docModel.extendAttributes(mergedAttributes, resolveStyleAttributeSet(styleId, element));

            // add or remove the passed explicit attributes
            attributes = _.copy(attributes, true);
            this.filterBySupportedFamilies(attributes);
            _.each(attributes, function (attributeValues, family) {

                var // definitions of own attributes
                    definitions = docModel.getAttributeDefinitions(family),
                    // add an attribute map for the current family
                    elementAttributeValues = null;

                if (_.isObject(attributeValues)) {

                    // add an attribute map for the current family
                    elementAttributeValues = newElementAttributes[family] || (newElementAttributes[family] = {});

                    // update the attribute map with the passed attributes
                    _.each(attributeValues, function (value, name) {
                        if (AttributeUtils.isRegisteredAttribute(definitions, name, options)) {
                            // check whether to clear the attribute
                            if (_.isNull(value) || (clear && _.isEqual(mergedAttributes[family][name], value))) {
                                delete elementAttributeValues[name];
                            } else {
                                elementAttributeValues[name] = value;
                            }
                        }
                    });

                    // remove empty attribute value maps completely
                    if (_.isEmpty(elementAttributeValues)) {
                        delete newElementAttributes[family];
                    }
                }
            });

            // check if any attributes have been changed
            if (preview || !_.isEqual(oldElementAttributes, newElementAttributes)) {

                // write back new explicit attributes to the element
                $(element).data('attributes', newElementAttributes);

                // Do really update the element formatting only if 'preview' is true or after the import of the
                // document is completed.
                if (preview || self.isImportFinished()) {

                    // call the passed layout listener to add attributes from layout/master slide
                    // -> this needs to be done, before explicit attributes
                    if (_.isFunction(layoutHandler)) { layoutHandler.call(self, $(element), mergedAttributes); }

                    // merge explicit attributes into style attributes, and update element formatting
                    docModel.extendAttributes(mergedAttributes, resolveElementAttributeSet(element), options);
                    updateElementFormatting($(element), mergedAttributes, { preview: preview });
                }

                // call the passed change listener
                if (_.isFunction(changeListener)) {
                    changeListener.call(this, element, oldElementAttributes, newElementAttributes);
                }
            }

            return this;
        };

        /**
         * Changes all given attributes in the style sheet, does not trigger
         * any listener or change anything in the DOM element is in use for
         * attributes: priority, dirty, builtIn.
         *
         * @param {String} styleId
         *
         * @param {Object} [options]
         *
         * @returns {StyleCollection}
         *  A reference to this instance.
         */
        this.setStyleOptions = function (styleId, options) {
            var styleSheet = styleSheetMap[styleId];
            if (styleSheet) {
                _.extend(styleSheet, options);
            }
            return this;
        };

        /**
         * Updates the CSS formatting in the specified DOM element, according
         * to its current attribute and style settings.
         *
         * @param {HTMLElement|jQuery} element
         *  The element to be updated. If this object is a jQuery collection,
         *  uses the first DOM node it contains.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Object} [options.baseAttributes]
         *      An attribute set with all attribute values of the ancestor of
         *      the passed element, merged from style sheet and explicit
         *      attributes, keyed by attribute family. Used internally while
         *      updating the formatting of all child elements of a node.
         *  @param {Boolean} [options.preview=false]
         *      If set to true, a GUI element used to preview the formatting of
         *      a style sheet will be rendered. Executes the 'preview' callback
         *      functions instead of the regular 'format' callback functions
         *      registered in the attribute definitions, and calls the preview
         *      handler instead of the format handler registered at this class
         *      instance.
         *  @param {Boolean} [options.async=false]
         *      If set to true, the registered format or preview handler will
         *      be asked to execute asynchronously in a browser timeout loop.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the registered format or
         *  preview handler has finished. If the option 'async' is set, and the
         *  format handler supports asynchronous execution, the promise will be
         *  resolved sometime in the future. Otherwise, the returned promise is
         *  already resolved when this method returns.
         */
        this.updateElementFormatting = function (element, options) {

            var // the identifier of the style sheet referred by the element
                styleId = AttributeUtils.getElementStyleId(element),
                // precalculated base attributes passed to this method
                baseAttributes = Utils.getObjectOption(options, 'baseAttributes'),
                // the merged attributes of the passed element
                mergedAttributes = null;

            if (_.isObject(baseAttributes)) {
                mergedAttributes = docModel.getDefaultAttributeSet(supportedFamilySet); // all families required
                docModel.extendAttributes(mergedAttributes, baseAttributes);
            } else {
                mergedAttributes = resolveBaseAttributes(element);
            }

            // call the passed layout listener to add attributes from layout/master slide
            // -> this needs to be done, before explicit attributes
            if (_.isFunction(layoutHandler)) { layoutHandler.call(self, $(element), mergedAttributes); }

            // add attributes of the style sheet and its parents, and the explicit attributes of the element
            docModel.extendAttributes(mergedAttributes, resolveStyleAttributeSet(styleId, element));
            docModel.extendAttributes(mergedAttributes, resolveElementAttributeSet(element), { special: true });

            // update the formatting of the element
            return updateElementFormatting($(element), mergedAttributes, options);
        };

        /**
         *
         * @param {String} styleId
         *
         * @param {String} possParentStyleId
         *
         * @returns {Boolean}
         */
        this.isChildOfOther = function (styleId, possParentStyleId) {
            if (styleId === possParentStyleId) {
                return true;
            }
            var parentId = self.getParentId(styleId);
            if (parentId) {
                return self.isChildOfOther(parentId, possParentStyleId);
            }
            return false;
        };

        // operation generators -----------------------------------------------

        /**
         * Generates an 'insertStyleSheet' operation, if the specified style
         * sheet exists in this style sheet collection, and is marked dirty.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {String} styleId
         *  The identifier of the style sheet.
         *
         * @returns {StyleCollection}
         *  A reference to this instance.
         */
        this.generateMissingStyleSheetOperations = function (generator, styleId) {

            // the style sheet descriptor
            var styleSheet = styleSheetMap[styleId];
            if (!styleSheet || (styleId === defaultStyleId) || !styleSheet.dirty) { return; }

            // the properties for the shyle sheet operations
            var properties = { type: styleFamily, styleId: styleId };

            // generate the 'deleteStyleSheet' operation for undo
            generator.generateOperation(Operations.DELETE_STYLESHEET, properties, { undo: true });

            // add more properties for style sheet insertion
            properties.styleName = styleSheet.name;
            properties.attrs = styleSheet.attributes;
            if (styleSheet.parentId) { properties.parent = styleSheet.parentId; }
            if (_.isNumber(styleSheet.priority)) { properties.uiPriority = styleSheet.priority; }
            if (styleSheet.hidden) { properties.hidden = true; }
            if (styleSheet.custom) { properties.custom = true; }

            // generate the 'insertStyleSheet' operation, remove the dirty flag from the style sheet
            generator.generateOperation(Operations.INSERT_STYLESHEET, properties);
            styleSheet.dirty = false;

            return this;
        };

        // initialization -----------------------------------------------------

        // add additional attribute families passed in the options
        supportedFamilySet = Utils.makeSet(Utils.getTokenListOption(initOptions, 'families', []));
        supportedFamilySet[styleFamily] = true;

        // forward other public methods to methods of EditModel
        ['getCssFontFamily', 'resolveColor', 'resolveTextColor', 'getCssColor', 'getCssTextColor', 'getCssBorderAttributes', 'getCssBorder', 'getCssTextDecoration'].forEach(function (method) {
            this[method] = docModel[method].bind(docModel);
        }, this);

        // destroy class members on destruction
        this.registerDestructor(function () {
            self = docModel = initOptions = styleSheetMap = supportedFamilySet = null;
            parentResolvers = formatHandler = previewHandler = styleAttributesResolver = elementAttributesResolver = null;
        });

    } // class StyleCollection

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: StyleCollection });

});
