/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/editframework/app/seleniumwrapper', [
    'io.ox/office/tk/utils'
], function (Utils) {

    'use strict';

    var innerMap = {};
    var SeleniumWrapper = {};

    function getErrorHandler(transactionID) {
        return function (err) {
            Utils.log('SeleniumWrapper errHandler', transactionID, err);
            innerMap[transactionID] = { state: false, result: JSON.stringify(err) };
        };
    }

    function getNormalDoneHandler(transactionID) {
        return function () {
            Utils.log('SeleniumWrapper doneHandler', transactionID);
            innerMap[transactionID] = { state: true, result: null };
        };
    }

    SeleniumWrapper.call = function (fnName, transactionID, options) {
        Utils.log('SeleniumWrapper.call', fnName, transactionID, options);

        if (SeleniumWrapper[fnName]) {

            try {
                var app = ox.ui.App.getCurrentApp();
                SeleniumWrapper[fnName](app, transactionID, options);
            } catch (e) {
                getErrorHandler(transactionID)('error while calling "' + fnName + '" !!! ' + JSON.stringify(e));
            }
        } else {
            getErrorHandler(transactionID)('no impl found for "' + fnName + '" !!!');
        }
    };

    SeleniumWrapper.get = function (transactionID) {
        var result = innerMap[transactionID];
        Utils.log('SeleniumWrapper.get', transactionID, result);
        delete innerMap[transactionID];
        return result;
    };

    // -------------------------------------------------------------------------

    SeleniumWrapper.getAllFilesInFolder = function (app, transactionID, options) {
        require(['io.ox/files/api']).done(function (FilesAPI) {
            FilesAPI.getAll(options.folderId, { cache: false, columns: '1,2,3,5,20,23,108,700,702,703,704,705,707,710,711' }).done(function (files) {
                innerMap[transactionID] = { state: true, result: files };
            }).fail(getErrorHandler(transactionID));
        }).fail(getErrorHandler(transactionID));
    };

    SeleniumWrapper.getAllSubfoldersInFolder = function (app, transactionID, options) {
        require(['io.ox/core/folder/api']).done(function (FolderAPI) {
            FolderAPI.list(options.folderId).done(function (folder) {
                innerMap[transactionID] = { state: true, result: folder };
            }).fail(getErrorHandler(transactionID));
        }).fail(getErrorHandler(transactionID));
    };

    SeleniumWrapper.oxLaunch = function (app, transactionID, options) {
        ox.launch(options.app, options).done(getNormalDoneHandler(transactionID)).fail(getErrorHandler(transactionID));
    };

    SeleniumWrapper.quitApplication = function (app, transactionID) {
        app.quit().done(getNormalDoneHandler(transactionID)).fail(getErrorHandler(transactionID));
    };

    SeleniumWrapper.reloadDocument = function (app, transactionID) {
        app.reloadDocument().done(getNormalDoneHandler(transactionID)).fail(getErrorHandler(transactionID));
    };

    SeleniumWrapper.applyOperations = function (app, transactionID, options) {
        app.getModel().createAndApplyOperations(function (generator) {
            generator.appendOperations(options.ops);
        }).done(getNormalDoneHandler(transactionID)).fail(getErrorHandler(transactionID));
    };

    SeleniumWrapper.setSettingsStates = function (app, transactionID, options) {
        require(['settings!io.ox/office']).done(function (OfficeSettings) {
            Object.keys(options).forEach(function (optionKey) {
                OfficeSettings.set(optionKey, options[optionKey]);
            });
            OfficeSettings.save();
            getNormalDoneHandler(transactionID)();
        });
    };

    SeleniumWrapper.clearTrashFolder = function (app, transactionID) {
        require(['io.ox/office/tk/utils/driveutils', 'io.ox/core/folder/api']).done(function (DriveUtils, FolderAPI) {
            FolderAPI.clear(DriveUtils.getStandardTrashFolderId()).done(getNormalDoneHandler(transactionID)).fail(getErrorHandler(transactionID));
        });
    };

    SeleniumWrapper.getAttributesFromTextApp = function (app, transactionID, options) {
        var family = options.family;
        var res = app.getModel().getAttributes(family);
        if (res !== null) {
            res = res[family];
        } else {
            res = null;
        }
        innerMap[transactionID] = { state: true, result: res };
    };

    SeleniumWrapper.getAttributesFromSpreadsheetApp = function (app, transactionID, options) {
        var family = options.family;
        var res = app.getView().getCellCollection().getCellAttributeSet(app.getView().getActiveCell());
        if (res !== null) {
            res = res[family];
        } else {
            res = null;
        }
        innerMap[transactionID] = { state: true, result: res };
    };

    SeleniumWrapper.getActiveCellEntry = function (app, transactionID) {
        var model = app.getView().getCellCollection().getCellModel(app.getView().getActiveCell());
        var res = {};
        if (model) {
            res.formula = model.f;
            res.display = model.d;
            res.value = model.v;
            res.url = model.url;
        }
        innerMap[transactionID] = { state: true, result: res };
    };

    SeleniumWrapper.scrollToCell = function (app, transactionID, options) {
        var SheetUtils = app.getSheetUtils();
        app.getView().getActiveGridPane().scrollToCell(SheetUtils.Address.parse(options.cellAddress));

        innerMap[transactionID] = { state: true };
    };

    SeleniumWrapper.setTextSelection = function (app, transactionID, options) {
        app.getModel().getSelection().setTextSelection(options.start, options.end);
        getNormalDoneHandler(transactionID)();
    };

    SeleniumWrapper.getSheetName = function (app, transactionID) {
        innerMap[transactionID] = { state: true, result: app.getView().getSheetName() };
    };

    SeleniumWrapper.shiftSeleniumLog = function (app, transactionID) {
        innerMap[transactionID] = { state: true, result: Utils.shiftSeleniumLog() };
    };

    SeleniumWrapper.getControllerUpdate = function (app, transactionID) {
        app.getController().getControllerUpdatePromise().done(getNormalDoneHandler(transactionID)).fail(getErrorHandler(transactionID));
    };

    SeleniumWrapper.typeTextInTextApp = function (app, transactionID, options) {
        function trigger(el, type, keyCode, charCode) {
            el.trigger(new $.Event(type, { which: keyCode, keyCode: keyCode, charCode: charCode }));
        }

        var done = getNormalDoneHandler(transactionID);

        var model = app.getModel();
        var page = model.getNode();
        var text = options.text;

        _.each(text, function (charCode) {
            var keyCode = null;
            if (charCode === ' ') {
                keyCode = 32;
            } else {
                keyCode = charCode.charCodeAt(0);
            }
            trigger(page, 'keydown',  0, 0);
            trigger(page, 'keypress', keyCode, charCode);
            trigger(page, 'keyup',    0, 0);
        });

        var textPromise = model.getInputTextPromise();
        if (textPromise) {
            textPromise.done(done);
        } else {
            done();
        }
    };

    /*
    SeleniumWrapper.waitForScheduler = function (app, transactionID) {
        require(['io.ox/office/tk/utils/scheduler']).done(function (Scheduler) {
            if (Scheduler.getA)
        });
    };
    */

    // exports ================================================================

    return SeleniumWrapper;

});
