/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define.async('io.ox/office/drawinglayer/view/control/shapetypepicker', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/control/radiolist',
    'io.ox/office/drawinglayer/view/drawinglabels',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/tk/io',
    'gettext!io.ox/office/drawinglayer/main'
], function (Utils, Forms, RadioList, Labels, DrawingFrame, IO, gt) {

    'use strict';

    // class ShapeTypePicker ==================================================

    var ALL_PRESETS = null;

    function getSection(id) {
        var unMatch = id.match(/callout|gear|leftrightcirculararrow/i);
        if (unMatch) {
            return null;
        }
        var match = id.match(/arrow|math|flowchart|star|actionbutton|ribbon/i);
        if (match) {
            return match[0].toString().toLowerCase();
        }
        return 'shape';
    }

    /**
     * @constructor
     *
     * @extends RadioList
     */
    function ShapeTypePicker(docView, initOptions) {

        // self reference
        var self = this;

        // the document model
        var docModel = docView.getDocModel();

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            tooltip: gt('Insert a shape'),
            label: gt('Shape'),
            icon: 'fa-forumbee',
            updateCaptionMode: 'none',
            toggle: true,
            itemDesign: 'grid',
            gridColumns: 16,
            smallerVersion: { hideLabel: true }
        }, initOptions));

        // private methods ----------------------------------------------------

        /**
         * hook on create items, only called once per type,
         * sets the button text and fill the chain-data for deferred rendering
         */
        function createItemHandler(event, buttonNode) {
            buttonNode.addClass('mini-caption').css({ minWidth: 43, padding: 0, lineHeight: 'normal' }).append('<img style = "margin: 2px;">');
        }

        /**
         * called initial and on theme-change
         * creates CanvasJS structure
         * and draws deferred all types in the chain to images
         */
        var updatePreviewShapes = this.createDebouncedMethod($.noop, function () {
            var boundRect = { width: 32, height: 32, left: 0, top: 0 };
            var canvas = DrawingFrame.initializeCanvas(null, boundRect, 1);

            var fillAttrs = { color: { type: 'preset', value: 'white' } };
            var lineAttrs = { color: { type: 'scheme', value: 'accent1' }, width: 53 };

            self.iterateArraySliced(self.getOptionButtons(), function (buttonNode) {
                var id = Forms.getButtonValue(buttonNode);
                var preset = ALL_PRESETS[id];
                var img = $(buttonNode).find('>img');

                var rect = boundRect;
                if (id.match(/leftright/i)) {
                    rect = { width: 32, height: 16, left: 0, top: 0 };
                    img.css({ marginTop: 10 });
                } else if (id.match(/updown/i)) {
                    rect = { width: 16, height: 32, left: 0, top: 0 };
                    img.css({ marginLeft: 10 });
                }
                canvas.render(function (context /*, width, height */) {
                    context.clearRect(-100, -100, 400, 400);
                    DrawingFrame.drawShape(docModel, context, rect, preset, fillAttrs, lineAttrs, {});
                });

                // update the image element
                img.attr({ src: canvas.getDataURL() });

            }).done(function () {
                canvas.destroy();
            });
        }, { delay: 50 });

        // initialization -----------------------------------------------------

        this.getMenuNode().addClass('shape-type-picker');

        self.createMenuSection('arrow', { label: gt('Arrows') });
        self.createMenuSection('math', { label: gt('Equation shapes') });
        self.createMenuSection('actionbutton', { label: gt('Action Buttons') });
        // self.createMenuSection('callout', { label: gt('Callouts') });
        self.createMenuSection('flowchart', { label: gt('Flowcharts') });
        self.createMenuSection('star', { label: gt('Stars') });
        self.createMenuSection('ribbon', { label: gt('Ribbon') });
        self.createMenuSection('shape', { label: gt('Basic shapes') });

        // create the (empty) option buttons
        self.getMenu().on('create:item', createItemHandler);
        _.each(ALL_PRESETS, function (preset, id) {
            var section = getSection(id);
            if (section) { self.createOptionButton(id, { section: section }); }
        });

        // lazy initialization of the charts before first usage of the menu
        this.getMenu().one('popup:beforeshow', updatePreviewShapes);

        // refresh preview charts in the drop-down menu after changed theme (color scheme)
        this.listenTo(docModel.getThemeCollection(), 'triggered', updatePreviewShapes);

        this.registerUpdateHandler(function (newState, lastState) {
            if (lastState !== newState) {
                Forms.selectNodes(self.getNode().find(Forms.BUTTON_SELECTOR), newState !== false);
            }
        });

        // destroy class members on destruction
        this.registerDestructor(function () {
            self = initOptions = docModel = docView = null;
        });

    } // class ShapeTypePicker

    ShapeTypePicker.getPresetShape = function (id) {
        return ALL_PRESETS[id];
    };

    // exports ================================================================

    return IO.loadJSON('io.ox/office/drawinglayer/view/presetGeometries').then(function (presets) {
        ALL_PRESETS = presets;
        return RadioList.extend({ constructor: ShapeTypePicker });
    });
});
