/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/drawinglayer/view/chartstyleutil', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/dateutils',
    'io.ox/office/editframework/utils/color',
    'gettext!io.ox/office/drawinglayer/main'
], function (Utils, DateUtils, Color, gt) {

    'use strict';

    var BACKGROUNDTRANSFORMATION = [{ type: 'satMod', value: 25000 }, { type: 'lumOff', value: 35000 }];

    var STYLESET = [
        { bg: { type: 'scheme', value: 'light1' }, bevelEnabled: false },
        { bg: { type: 'scheme', value: 'light1' }, bevelEnabled: false },
        { bg: { type: 'scheme', value: 'light1' }, bevelEnabled: false },
        { bg: { type: 'scheme', value: 'light1' }, bevelEnabled: true },
        { bg: null, bevelEnabled: false },
        { bg: { type: 'scheme', value: 'text1', transformations: [{ type: 'lumOff', value: 25000 }] }, bevelEnabled: true }
    ];

    var COLORSET = [];

    (function () {
        var allAccents = [];
        for (var i = 1; i <= 6; i++) {
            allAccents.push({ type: 'scheme', value: 'accent' + i });
        }

        function getAccentColorSetName(accent) {
            //#. predefined formatting style for chart objects: single accent color from the current color scheme
            //#. Example result: "Accent 1", "Accent 2"
            //#. %1$d is the index of the accent color
            //#, c-format
            return gt.pgettext('chart-format', 'Accent %1$d', _.noI18n(accent));
        }

        COLORSET.push({
            //#. predefined formatting style for chart objects: black/gray/white only
            name: gt.pgettext('chart-format', 'Grayscale'),
            colors: [{ type: 'rgb', value: '555555' }, { type: 'rgb', value: '9E9E9E' }, { type: 'rgb', value: '727272' }, { type: 'rgb', value: '464646' }, { type: 'rgb', value: '838383' }, { type: 'rgb', value: 'C1C1C1' }],
            type: 'group'
        });

        COLORSET.push({
             //#. predefined formatting style for chart objects: all accent colors from the current color scheme
            name: gt.pgettext('chart-format', 'Accents'),
            colors: allAccents,
            type: 'group'
        });

        _.each(allAccents, function (color, index) {
            COLORSET.push({
                name: getAccentColorSetName(index + 1),
                colors: [color],
                type: 'single'
            });
        });
    }());

    // static class ChartStyleUtil ============================================

    /**
     * an infoholder for the 48 chartstyles, a combination of colorpatterns, backgroundcolors and highlighting
     */
    var ChartStyleUtil = {};

    // public methods ---------------------------------------------------------

    /**
     * return a list of different combinations of backgrounds and highlights
     */
    ChartStyleUtil.getStyleSet = function () {
        return STYLESET;
    };

    /**
     * return a list of different combinations of grayscales and accents
     */
    ChartStyleUtil.getColorSet = function () {
        return COLORSET;
    };

    /**
     * returns the color-transformations for desaturate the first color in the used pattern
     */
    ChartStyleUtil.getBackgroundTransformation = function () {
        return BACKGROUNDTRANSFORMATION;
    };

    function getCorrectLineColor(docModel, chartModel) {
        var attrs = chartModel.getMergedAttributeSet(true);
        var bg = attrs.fill.color;
        if (Color.parseJSON(bg).isAuto()) {
            chartModel.updateBackgroundColor();
            attrs = chartModel.getMergedAttributeSet(true);
            bg = attrs.fill.color;
        }
        var colorDesc = docModel.resolveColor(Color.parseJSON(bg), 'line');
        if (!_.isString(bg.fallbackValue)) {
            bg.fallbackValue = colorDesc.hex;
        }
        return { type: 'preset', value: colorDesc.dark ? 'white' : 'black' };
    }

    /**
     * sets all character-model-properties to the target,
     * target is a CanvasJS-data-model
     * @param {Object} character character-model comes from operations
     * @param {Object} target target-object is a inner object of CanvasJS
     * @param {Object} prefix necessary if there are more than one character-holder on the target object
     */
    ChartStyleUtil.handleCharacterProps = function (chartModel, character, target, prefix) {

        var app = chartModel.getApp();
        var docModel = app.getModel();
        var docView = app.getView();

        var usePrefix;
        if (prefix && prefix.length) {
            usePrefix = prefix + 'Font';
        } else {
            usePrefix = 'font';
        }

        target[usePrefix + 'Color'] = docModel.getCssColor(character.color, getCorrectLineColor(docModel, chartModel));
        target[usePrefix + 'Family'] = docModel.getCssFontFamily(character.fontName);
        target[usePrefix + 'Size'] = docView.getZoomFactor() * Utils.convertLength(character.fontSize, 'pt', 'px');
        target[usePrefix + 'Weight'] = character.bold ? 'bold' : 'normal';
        target[usePrefix + 'Style'] = character.italic ? 'italic' : 'normal';
    };

    /**
     * sets all lineshape-model-properties to the target,
     * target is a CanvasJS-data-model
     * @app {BaseApplication}
     * @character {object} lineshape-model comes from operations
     * @target {object} target-object is a inner object of CanvasJS
     * @prefix {object} necessary for more than one character-holder on the target object
     */
    ChartStyleUtil.handleLineProps = function (chartModel, line, target, prefix) {

        var app = chartModel.getApp();
        var docModel = app.getModel();
        var docView = app.getView();
        var w = line.width ? line.width * (72 / 2540) : 1;
        var usePrefix = prefix;

        if (line.type === 'none') {
            target[usePrefix + 'Color'] = 'transparent';
        } else {
            target[usePrefix + 'Color'] = docModel.getCssColor(line.color, getCorrectLineColor(docModel, chartModel));
            target[usePrefix + 'Thickness'] = Math.max(1, Math.round(docView.getZoomFactor() * w));
        }
    };

    // exports ================================================================

    return ChartStyleUtil;

});
