/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Carsten Driesner <carsten.driesner@open-xchange.com>
 */

define('io.ox/office/baseframework/utils/errorcode', [
    'io.ox/office/tk/utils'
], function (Utils) {

    'use strict';

    // class ErrorCode ========================================================

    /**
     * Provides methods to check a server result for possible errors.
     *
     * @constructor
     *
     * @param {Object} [resultData]
     *  The result object directly provided by the server which contains error
     *  information that are used to initialize ErrorCode instance.
     */
    function ErrorCode(resultData) {

        var // the error description
            description = '',

            // the error code as string constant
            codeAsStringConstant = 'NO_ERROR',

            // error class
            errorClass = ErrorCode.ERRORCLASS_NO_ERROR,

            // source of the error
            errorSource = ErrorCode.ERRORSOURCE_CLIENT,

            // error context
            errorContext = null,

            // optional error context data
            errorContextData = null,

            // optional error value
            errorValue = null;

        // public methods -----------------------------------------------------

        /**
         * Checks if this instance represents an error state or not.
         *
         * @returns {Boolean}
         *  TRUE if the instance represents an error state, FALSE if not.
         */
        this.isError = function () {
            return errorClass >= ErrorCode.ERRORCLASS_ERROR;
        };

        /**
         * Checks if this instance represents a warning state or not.
         *
         * @returns {Boolean}
         *  TRUE if the instance represents warning state, FALSE if not.
         */
        this.isWarning = function () {
            return errorClass === ErrorCode.ERRORCLASS_WARNING;
        };

        /**
         * Provides a textual description what error has happened.
         *
         * @returns {String}
         *  A description of the error.
         */
        this.getDescription = function () {
            return description;
        };

        /**
         * Provides a unique string constant that specifies the error.
         *
         * @returns {String}
         *  A unique string constant for the error.
         */
        this.getCodeAsConstant = function () {
            return codeAsStringConstant;
        };

        /**
         * Provides the error class of the error, which describes the
         * severity of the error.
         *
         * @returns {Number}
         */
        this.getErrorClass = function () {
            return errorClass;
        };

        /**
         * Sets the error class of the error.
         *
         * @param {Number} newErrorClass
         *  The new error class.
         */
        this.setErrorClass = function (newErrorClass) {
            errorClass = newErrorClass;
        };

        /**
         * Provides the source of the error
         *
         * @returns {Number}
         */
        this.getErrorSource = function () {
            return errorSource;
        };

        /**
         * Provides the context of the error. A context describes the
         * specific area, where the error ocurred, e.g. load,
         * save, delete etc.
         *
         * @returns {String|null}
         *  The error context or null, if the context is unknown.
         */
        this.getErrorContext = function () {
            return errorContext;
        };

        /**
         * Provides an optional error value which describes in more
         * detail what caused the error. E.g. the illegal character
         * that resulted in a failed rename operation.
         *
         * @return {String}
         *  The error value. Can be an empty string if no error
         *  value was set.
         */
        this.getErrorValue = function () {
            return errorValue;
        };

        /**
         * Provides the textual representation of the error code object.
         *
         * @returns {String}
         *  The textual representation of the error code object.
         */
        this.getErrorText = function () {
            return 'Error: constant=' + codeAsStringConstant + ', description=' + description;
        };

        /**
         * Sets the source of the error code.
         *
         * @param {Number} source
         *  The source of the error code.
         */
        this.setErrorSource = function (source) {
            if (_.isNumber(source) && (source >= ErrorCode.ERRORSOURCE_SERVER) && (source <= ErrorCode.ERRORSOURCE_CLIENT)) {
                errorSource = source;
            }
        };

        /**
         * Sets the context of the error code.
         *
         * @param {String} context
         *  The new context of the error code.
         */
        this.setErrorContext = function (context) {
            errorContext = context;
        };

        /**
         * Gets the context data of this error code.
         *
         * @returns {Object}
         *  An optional object containing property/value pairs with
         *  additional data for this error.
         */
        this.getContextData = function () {
            return errorContextData;
        };

        /**
         * Sets the context data of this error code.
         *
         * @param {Object} contextData
         *  An object containing property/value pairs with additional data for
         *  this error.
         */
        this.setContextData = function (newContextData) {
            errorContextData = newContextData;
        };

        /**
         * Provides the error code content as property object, which can
         * be used to initialize a new ErrorCode object.
         *
         * @returns {Object}
         *  An object which can be used to initialize a new error code
         *  object.
         */
        this.getAsResult = function () {
            var result = { error: {} };

            result.error[ErrorCode.PROPERTY_ERRORDESCRIPTION] = description;
            result.error[ErrorCode.PROPERTY_ERRORCODEASSTRING] = codeAsStringConstant;
            result.error[ErrorCode.PROPERTY_ERRORCLASS] = errorClass;
            result.error[ErrorCode.PROPERTY_ERRORSOURCE] = errorSource;
            result.error[ErrorCode.PROPERTY_ERRORCONTEXTDATA] = errorContextData;
            result.error[ErrorCode.PROPERTY_ERRORVALUE] = errorValue;
            return result;
        };

        // initialization -----------------------------------------------------

        if (_.isObject(resultData)) {
            if (_.isObject(resultData.error)) {
                // check the provided resultData object for error information - this
                // is the default error format of a server answer
                description = Utils.getStringOption(resultData.error, ErrorCode.PROPERTY_ERRORDESCRIPTION, '');
                codeAsStringConstant = Utils.getStringOption(resultData.error, ErrorCode.PROPERTY_ERRORCODEASSTRING, 'NO_ERROR');
                errorClass = Utils.getNumberOption(resultData.error, ErrorCode.PROPERTY_ERRORCLASS, ErrorCode.ERRORCLASS_NO_ERROR);
                errorContextData = Utils.getObjectOption(resultData.error, ErrorCode.PROPERTY_ERRORCONTEXTDATA, {});
                errorValue = Utils.getStringOption(resultData.error, ErrorCode.PROPERTY_ERRORVALUE, '');
            } else if (_.isObject(resultData) && !_.isString(Utils.getStringOption(resultData, ErrorCode.PROPERTY_OX_ERRORID, null))) {
                // we also support to provide a error object directly - detect and ignore plain error data from backend
                description = Utils.getStringOption(resultData, ErrorCode.PROPERTY_ERRORDESCRIPTION, '');
                codeAsStringConstant = Utils.getStringOption(resultData, ErrorCode.PROPERTY_ERRORCODEASSTRING, 'NO_ERROR');
                errorClass = Utils.getNumberOption(resultData, ErrorCode.PROPERTY_ERRORCLASS, ErrorCode.ERRORCLASS_NO_ERROR);
                errorSource = Utils.getNumberOption(resultData, ErrorCode.PROPERTY_ERRORSOURCE, ErrorCode.ERRORSOURCE_CLIENT);
                errorContextData = Utils.getObjectOption(resultData, ErrorCode.PROPERTY_ERRORCONTEXTDATA, {});
                errorValue = Utils.getStringOption(resultData, ErrorCode.PROPERTY_ERRORVALUE, '');
            }
        }

    } // class ErrorCode

    // static methods ---------------------------------------------------------

    /**
     * Extracts the error information from result data received from the server
     * side.
     *
     * @param {Object} serverResult
     *  The result data sent by the server.
     *
     * @returns {Object|Null}
     *  The error information extracted from the server data or null if no
     *  error information could be extracted.
     */
    ErrorCode.extractErrorInformation = function (serverResult) {
        return (_.isObject(serverResult) && serverResult.error) ? { error: serverResult.error } : null;
    };

    /**
     * Extracts the error information from a real-time result received from the
     * server side.
     *
     * @param {Object} rtResult
     *  The result data sent by the server.
     *
     * @returns {Object|Null}
     *  An error object containing the error code of the real-time answer or
     *  null if no error information could be extracted.
     */
    ErrorCode.extractRealtimeError = function (rtResult) {
        return (_.isObject(rtResult) && rtResult.error && _.isNumber(rtResult.error.code) && _.isString(rtResult.error.prefix)) ? { code: rtResult.error.code, prefix: rtResult.error.prefix } : null;
    };

    /**
     * Checks, if an arbitrary value is a error code object.
     *
     * @param {Any} value
     *  The value to be checked.
     *
     * @returns {Boolean}
     *  Whether the value supports vital functions and looks like a error code
     *  object.
     */
    ErrorCode.isErrorCode = function (value) {
        return _.isObject(value) && _.isFunction(value.isError) && _.isFunction(value.getErrorText);
    };

    // constants --------------------------------------------------------------

    // the constant error code as string for no error
    ErrorCode.CONSTANT_NO_ERROR = 'NO_ERROR';

    // the property names
    ErrorCode.PROPERTY_ERRORDESCRIPTION = 'errorDescription';
    ErrorCode.PROPERTY_ERRORCODEASSTRING = 'error';
    ErrorCode.PROPERTY_ERRORCLASS = 'errorClass';
    ErrorCode.PROPERTY_ERRORSOURCE = 'errorSource';
    ErrorCode.PROPERTY_ERRORCONTEXTDATA = 'errorContextData';
    ErrorCode.PROPERTY_ERRORVALUE = 'errorValue';

    // normal error data sent by plain backend including RT
    ErrorCode.PROPERTY_OX_ERRORID = 'error_id';

    // the error classes describing the severity of an error
    ErrorCode.ERRORCLASS_NO_ERROR = 0;
    ErrorCode.ERRORCLASS_WARNING = 1;
    ErrorCode.ERRORCLASS_ERROR = 2;
    ErrorCode.ERRORCLASS_FATAL_ERROR = 3;

    // the source of the error (server or client)
    ErrorCode.ERRORSOURCE_SERVER = 1;
    ErrorCode.ERRORSOURCE_CLIENT = 2;

    // a general error
    ErrorCode.GENERAL_ERROR = { error: 'GENERAL_UNKNOWN_ERROR', errorClass: ErrorCode.ERRORCLASS_ERROR };

    // exports ================================================================

    return ErrorCode;
});
