/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/baseframework/app/extensionregistry', [
    'io.ox/core/capabilities',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/config'
], function (Capabilities, Utils, Config) {

    'use strict';

    // which file formats are editable (maps format identifier to booleans)
    var EDITABLE_FILE_FORMATS = {
        ooxml: true,
        odf: Config.getFlag('odfsupport', true)
    };

    // the extension suffix for files with an internal error
    var INTERNAL_ERROR_SUFFIX = '_ox';

    // file extension configuration
    var FILE_EXTENSION_CONFIGURATION = {

        text: {
            module: 'io.ox/office/text',
            requires: 'text',
            editable: !_.browser.Android || Utils.supportedChromeVersionOnAndroidForText(),
            extensions: {
                docx: { format: 'ooxml' },
                docm: { format: 'ooxml', macros: true },
                dotx: { format: 'ooxml', template: true },
                dotm: { format: 'ooxml', template: true, macros: true },
                odt:  { format: 'odf', macros: true },
                ott:  { format: 'odf', template: true, macros: true },
                doc:  { format: 'ooxml', macros: true, convert: true },
                dot:  { format: 'ooxml', template: true, templateLocked: true, macros: true, convert: true },
                rtf:  { format: 'ooxml', convert: true }
            }
        },

        spreadsheet: {
            module: 'io.ox/office/spreadsheet',
            requires: 'spreadsheet',
            editable: true,
            extensions: {
                xlsx: { format: 'ooxml' },
                xlsm: { format: 'ooxml', macros: true },
                xltx: { format: 'ooxml', template: true },
                xltm: { format: 'ooxml', template: true, macros: true },
                xlam: { macros: true }, // add-in
                xls:  { format: 'ooxml', macros: true, convert: true },
                xlt:  { format: 'ooxml', template: true, templateLocked: true, macros: true, convert: true },
                xla:  { macros: true }, // add-in
                xlsb: { format: 'ooxml', macros: true, convert: true },
                ods:  { format: 'odf', macros: true }, //spreadsheet
                ots:  { format: 'odf', template: true, macros: true } //spreadsheet
            }
        },

        presentation: {
            module: 'io.ox/office/presentation',
            requires: 'presentation',
            editable: !_.browser.Android || Utils.supportedChromeVersionOnAndroidForText(),
            extensions: {
                pptx: { format: 'ooxml' },
                pptm: { format: 'ooxml', macros: true },
                potx: { format: 'ooxml', template: true },
                potm: { format: 'ooxml', template: true, macros: true },
                ppsx: {}, // slide show
                ppsm: { macros: true }, // slide show
                ppam: { macros: true }, // add-in
                odp:  { format: 'odf', macros: true },
                otp:  { format: 'odf', template: true, macros: true },
                ppt:  { format: 'ooxml', macros: true, convert: true },
                pot:  { format: 'ooxml', template: true, templateLocked: true, macros: true, convert: true },
                pps:  { format: 'ooxml', macros: true, convert: true }, // slide show
                ppa:  { format: 'ooxml', macros: true, convert: true } // add-in
            }
        },

        preview: {
            extensions: {
                pdf:  {},
                odg:  { format: 'odf', macros: true }, // drawing
                otg:  { format: 'odf', template: true, macros: true }, // drawing
                odi:  { format: 'odf' }, // image
                oti:  { format: 'odf', template: true }, // image
                odc:  { format: 'odf' }, // chart
                otc:  { format: 'odf', template: true }, // chart
                odf:  { format: 'odf' }, // formula
                otf:  { format: 'odf', template: true }, // formula
                odm:  { format: 'odf' } // global text document
            }
        }
    };

    // extension to template extension map
    var EXT_TO_TEMPLATE_EXT_MAP = {
        docx: 'dotx',
        docm: 'dotx',
        dotm: 'dotx',
        xlsx: 'xltx',
        xlsm: 'xltx',
        xltm: 'xltx',
        pptx: 'potx',
        pptm: 'potx',
        potm: 'potx',
        odt: 'ott',
        ods: 'ots',
        odp: 'otp'
    };

    // all application configurations, mapped by application identifier
    var applicationMap = {};

    // all configurations, mapped by lower-case extension
    var fileExtensionMap = {};

    // whether OX Preview is available at all
    var previewAvailable = Capabilities.has('document_preview');

    // static class ExtensionRegistry =========================================

    /**
     * Configuration settings for all supported file extensions of all OX
     * Documents application.
     */
    var ExtensionRegistry = {};

    // static methods ---------------------------------------------------------

    /**
     * Returns whether the passed module name specifies an OX Documents
     * application that can edit documents on the current platform.
     *
     * @param {String} editModule
     *  The module name of the application.
     *
     * @returns {Boolean}
     *  Whether the application is able to edit documents.
     */
    ExtensionRegistry.supportsEditMode = function (editModule) {
        return Utils.getBooleanOption(applicationMap[editModule], 'editable', false);
    };

    /**
     * Returns the configuration settings of the file extension contained by
     * the passed file name.
     *
     * @param {String} fileName
     *  The file name (case-insensitive).
     *
     * @param {String} [editModule]
     *  If specified, must match the module name of the edit application
     *  registered for the extension of the passed file name. If omitted, the
     *  edit application module name for the extension will be ignored.
     *
     * @returns {Object|Null}
     *  The file extension settings; or null, if the extension is not known.
     */
    ExtensionRegistry.getExtensionSettings = function (fileName, editModule) {

        // the lower-case extension of the passed file name
        var extension = Utils.getFileExtension(fileName);
        // the extension settings object
        var extensionSettings = (extension in fileExtensionMap) ? fileExtensionMap[extension] : null;
        // the module name of the edit application for the extension
        var module = Utils.getStringOption(extensionSettings, 'module');

        // passed module name must match if specified
        return (_.isString(editModule) && (editModule !== module)) ? null : extensionSettings;
    };

    /**
     * Returns the type category of the file with the passed file name.
     *
     * @param {String} fileName
     *  The file name (case-insensitive).
     *
     * @returns {String}
     *  The type category of the file with the passed file name; or an empty
     *  string, if the file extension is not registered.
     *  - 'text': A text document.
     *  - 'spreadsheet': A spreadsheet document.
     *  - 'presentation': A presentation document.
     *  - 'preview': Another viewable document that is not part of the
     *      preceding file categories.
     */
    ExtensionRegistry.getFileCategory = function (fileName) {
        var extensionSettings = ExtensionRegistry.getExtensionSettings(fileName);
        return Utils.getStringOption(extensionSettings, 'type', '');
    };

    /**
     * Returns all extensions that can be viewed with OX Preview, as array.
     *
     * @returns {Array<String>}
     *  All extensions supported by OX Preview.
     */
    ExtensionRegistry.getViewableExtensions = function () {
        var extensions = [];
        _.each(fileExtensionMap, function (settings, extension) {
            if (settings.viewable) {
                extensions.push(extension);
            }
        });
        return extensions;
    };

    /**
     * Returns an array containing all file extensions that can be edited.
     *
     * @returns {Array<String>}
     *  An array containing all file extensions that can be edited.
     */
    ExtensionRegistry.getEditableExtensions = function () {
        var extensions = [];
        _.each(fileExtensionMap, function (settings, extension) {
            if (settings.editable) {
                extensions.push(extension);
            }
        });
        return extensions;
    };

    /**
     * Returns whether the file with the passed name is viewable by the OX
     * Preview application.
     *
     * @param {String} fileName
     *  The file name (case-insensitive).
     *
     * @returns {Boolean}
     *  Whether the specified file is viewable by the OX Preview application.
     */
    ExtensionRegistry.isViewable = function (fileName) {
        var extensionSettings = ExtensionRegistry.getExtensionSettings(fileName);
        return Utils.getBooleanOption(extensionSettings, 'viewable', false);
    };

    /**
     * Returns the module name of the edit application that is able to modify
     * the file with the passed file name.
     *
     * @param {String} fileName
     *  The file name (case-insensitive).
     *
     * @returns {String|Null}
     *  The module name of the edit application that is able to modify the file
     *  with the passed file name; or null, if no edit module exists for the
     *  file extension.
     */
    ExtensionRegistry.getEditModule = function (fileName) {
        var extensionSettings = ExtensionRegistry.getExtensionSettings(fileName);
        return Utils.getStringOption(extensionSettings, 'module', null);
    };

    /**
     * Returns whether the file with the passed name is editable by any of the
     * OX Documents edit applications.
     *
     * @param {String} fileName
     *  The file name (case-insensitive).
     *
     * @param {String} [editModule]
     *  If specified, must match the module name of the edit application
     *  registered for the extension of the passed file name. If omitted, the
     *  edit application module name for the extension will be ignored.
     *
     * @returns {Boolean}
     *  Whether the specified file is editable by one of the OX Documents edit
     *  applications.
     */
    ExtensionRegistry.isEditable = function (fileName, editModule) {
        var extensionSettings = ExtensionRegistry.getExtensionSettings(fileName, editModule);
        return Utils.getBooleanOption(extensionSettings, 'editable', false);
    };

    /**
     * Returns whether the file with the passed name is editable natively with
     * any of the OX Documents edit applications, according to the file format.
     *
     * @param {String} fileName
     *  The file name (case-insensitive).
     *
     * @param {String} [editModule]
     *  If specified, must match the module name of the edit application
     *  registered for the extension of the passed file name. If omitted, the
     *  edit application module name for the extension will be ignored.
     *
     * @returns {Boolean}
     *  Whether the specified file is editable natively with one of the OX
     *  Documents edit applications.
     */
    ExtensionRegistry.isNative = function (fileName, editModule) {
        var extensionSettings = ExtensionRegistry.getExtensionSettings(fileName, editModule);
        return Utils.getBooleanOption(extensionSettings, 'editable', false) && Utils.getBooleanOption(extensionSettings, 'native', false);
    };

    /**
     * Returns whether the file with the passed name is editable with any of
     * the OX Documents edit applications by conversion to a natively supported
     * file format, according to the file format.
     *
     * @param {String} fileName
     *  The file name (case-insensitive).
     *
     * @param {String} [editModule]
     *  If specified, must match the module name of the edit application
     *  registered for the extension of the passed file name. If omitted, the
     *  edit application module name for the extension will be ignored.
     *
     * @returns {Boolean}
     *  Whether the specified file is editable with one of the OX Documents
     *  edit applications by conversion to a native file format.
     */
    ExtensionRegistry.isConvertible = function (fileName, editModule) {
        var extensionSettings = ExtensionRegistry.getExtensionSettings(fileName, editModule);
        return Utils.getBooleanOption(extensionSettings, 'editable', false) && !Utils.getBooleanOption(extensionSettings, 'native', false);
    };

    /**
     * Returns the file format identifier of the file with the passed name.
     *
     * @param {String} fileName
     *  The file name (case-insensitive).
     *
     * @param {String} [editModule]
     *  If specified, must match the module name of the edit application
     *  registered for the extension of the passed file name. If omitted, the
     *  edit application module name for the extension will be ignored.
     *
     * @returns {String}
     *  The file format identifier of the file with the passed name:
     *  - 'ooxml' for a file that will be edited in the OfficeOpenXML file
     *      format (either natively, or by conversion from a file format not
     *      supported natively).
     *  - 'odf' for a file that will be edited in the OpenDocument file format.
     *  - An empty string for all other file names.
     */
    ExtensionRegistry.getFileFormat = function (fileName, editModule) {
        var extensionSettings = ExtensionRegistry.getExtensionSettings(fileName, editModule);
        return Utils.getStringOption(extensionSettings, 'format', '');
    };

    /**
     * Returns whether the file with the passed name is a template file.
     *
     * @param {String} fileName
     *  The file name (case-insensitive).
     *
     * @param {String} [editModule]
     *  If specified, must match the module name of the edit application
     *  registered for the extension of the passed file name. If omitted, the
     *  edit application module name for the extension will be ignored.
     *
     * @returns {Boolean}
     *  Whether the specified file is a document template file.
     */
    ExtensionRegistry.isTemplate = function (fileName, editModule) {
        var extensionSettings = ExtensionRegistry.getExtensionSettings(fileName, editModule);
        return Utils.getBooleanOption(extensionSettings, 'template', false);
    };

    /**
     * Returns whether the file with the passed name is a locked template file.
     * (is used for binary templates)
     *
     * @param {String} fileName
     *  The file name (case-insensitive).
     *
     * @param {String} [editModule]
     *  If specified, must match the module name of the edit application
     *  registered for the extension of the passed file name. If omitted, the
     *  edit application module name for the extension will be ignored.
     *
     * @returns {Boolean}
     *  Whether the specified file is a locked document template file.
     */
    ExtensionRegistry.isTemplateLocked = function (fileName, editModule) {
        var extensionSettings = ExtensionRegistry.getExtensionSettings(fileName, editModule);
        return Utils.getBooleanOption(extensionSettings, 'templateLocked', false);
    };

    /**
     * Returns whether the file with the passed name may contain macro scripts.
     *
     * @param {String} fileName
     *  The file name (case-insensitive).
     *
     * @param {String} [editModule]
     *  If specified, must match the module name of the edit application
     *  registered for the extension of the passed file name. If omitted, the
     *  edit application module name for the extension will be ignored.
     *
     * @returns {Boolean}
     *  Whether the specified file may contain macro scripts.
     */
    ExtensionRegistry.isScriptable = function (fileName, editModule) {
        var extensionSettings = ExtensionRegistry.getExtensionSettings(fileName, editModule);
        return Utils.getBooleanOption(extensionSettings, 'macros', false);
    };

    /**
     * Returns whether the file with the passed name contains the 'internal
     * error' extension suffix.
     *
     * @param {String} fileName
     *  The file name (case-insensitive).
     *
     * @returns {Boolean}
     *  Whether the file with the passed name contains the 'internal error'
     *  extension suffix.
     */
    ExtensionRegistry.isError = function (fileName) {
        var extension = Utils.getFileExtension(fileName);
        return extension.indexOf(INTERNAL_ERROR_SUFFIX, extension.length - INTERNAL_ERROR_SUFFIX.length) >= 0;
    };

    /**
     * Extends and returns the passed file name with the 'internal error'
     * extension suffix. If the file name already contains this suffix, it will
     * be returned as is.
     *
     * @param {String} fileName
     *  The file name (case-insensitive).
     *
     * @returns {String}
     *  The passed passed file name with the 'internal error' extension suffix.
     */
    ExtensionRegistry.createErrorFileName = function (fileName) {
        return ExtensionRegistry.isError(fileName) ? fileName : (fileName + INTERNAL_ERROR_SUFFIX);
    };

    /**
     * Provides the pre-defined template extension for a given file name.
     *
     * @param {String} fileName
     *  The file name (case-insensitive).
     *
     * @returns {String|Null}
     *  The template extension for the file name; or null if the format cannot
     *  be written to a template format.
     */
    ExtensionRegistry.getTemplateExtensionForFileName = function (fileName) {
        var extension = Utils.getFileExtension(fileName);
        extension = EXT_TO_TEMPLATE_EXT_MAP[extension];
        return _.isString(extension) ? extension : null;
    };

    // static initialization ==================================================

    // process the configuration of all modules
    _.each(FILE_EXTENSION_CONFIGURATION, function (moduleConfiguration, documentType) {

        // the capability required to edit the file
        var editable = Utils.getBooleanOption(moduleConfiguration, 'editable', false);
        // the capability required to edit the file
        var requires = Utils.getStringOption(moduleConfiguration, 'requires', null);
        // the module name of the edit application
        var module = Utils.getStringOption(moduleConfiguration, 'module', null);
        // whether the edit module is available at all
        var editAvailable = editable && _.isString(requires) && Capabilities.has(requires) && _.isString(module);

        // application configuration antry
        applicationMap[documentType] = { editable: editAvailable };

        // process all extensions registered for the module
        _.each(moduleConfiguration.extensions, function (extensionSettings, extension) {

            // the file format of files with the current extension
            var format = Utils.getStringOption(extensionSettings, 'format', '');
            // whether the file extension is supported natively
            var native = !Utils.getBooleanOption(extensionSettings, 'convert', false);

            if (extension in fileExtensionMap) {
                Utils.warn('ExtensionRegistry(): extension "' + extension + '" already registered');
            }

            // initialize all properties of the current extension
            var fileExtension = _.extend({
                type: documentType,
                viewable: previewAvailable,
                editable: editAvailable && Utils.getBooleanOption(EDITABLE_FILE_FORMATS, format, false),
                module: module,
                native: native
            }, extensionSettings);

            fileExtensionMap[extension] = fileExtension;

            // initialize all properties of the current extension with internal error
            // suffix (native only) - these files are not viewable in OX Preview
            if (native) {
                fileExtension = _.extend({}, fileExtension, { viewable: false });
                fileExtensionMap[extension + INTERNAL_ERROR_SUFFIX] = fileExtension;
            }
        });
    });

    // exports ================================================================

    return ExtensionRegistry;

});
