/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author York Richter <york.richter@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/baseframework/app/applicationlauncher', [
    'io.ox/core/extensions',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/config',
    'io.ox/office/tk/forms',
    'less!io.ox/office/baseframework/view/basestyle'
], function (ext, Utils, Config, Forms) {

    'use strict';

    // static class ApplicationLauncher =======================================

    var ApplicationLauncher = {};

    // static methods ---------------------------------------------------------

    /**
     * Returns the document type from the passed application module name.
     * The document type equals the last directory name of the module name.
     *
     * @param {String} moduleName
     *  The application module name.
     *
     * @returns {String}
     *  The document type of the passed module name.
     */
    ApplicationLauncher.getDocumentType = function (moduleName) {
        return moduleName.substr(moduleName.lastIndexOf('/') + 1);
    };

    /**
     * Returns the stand-alone mode from the server-side configuration flag
     * 'standalone', or the application's stand-alone URL hash fragment. This
     * function also checks if the application is loaded in an <iframe> and
     * gets the correct hash fragment from the it's source URL.
     *
     * @returns {Boolean}
     *  The state of the stand-alone mode.
     */
    ApplicationLauncher.getStandaloneMode = function () {
        if (Config.getFlag('standalone')) { return true; }
        var standalone = (window.self !== window.top) ? _.deserialize(window.self.location.hash.substr(1)).standalone : _.url.hash('standalone');
        return standalone ? (standalone.toLowerCase() === 'true' || standalone === '1') : false;
    };

    /**
     * Returns a running application instance for the specified file descriptor
     * if existing.
     *
     * @param {String} moduleName
     *  The application type identifier.
     *
     * @param {Object} fileDesc
     *  A file descriptor containing the file identifier, file name, and other
     *  important settings for a file.
     *
     * @returns {BaseApplication|Null}
     *  A running application instance for the specified file descriptor if
     *  existing; otherwise null.
     */
    ApplicationLauncher.getRunningApp = function (moduleName, fileDesc) {

        // find running application for the specified module type
        var runningApps = ox.ui.App.get(moduleName).filter(function (app) {
            return app.isEqualFileDescriptor && app.isEqualFileDescriptor(fileDesc);
        });

        // return running application if existing
        if (runningApps.length > 1) {
            Utils.warn('LaunchApplication.getRunningApp(): found multiple applications for the same file.');
        }
        return runningApps[0] || null;
    };

    /**
     * Creates a static launcher that has to be returned by the main module of
     * each application type. The launcher tries to find a running application
     * which is working on a file described in the launch options passed to the
     * launcher. If no such application exists, it creates and returns a new
     * application object.
     *
     * @param {String} moduleName
     *  The application type identifier.
     *
     * @param {Object} [appOptions]
     *  Optional parameters controlling the creation of the ox.ui.App base
     *  class instance:
     *  @param {String} [appOptions.icon]
     *      If specified, the CSS class name of a Bootstrap icon that will be
     *      shown in the top level launcher tab next to the application title.
     *  @param {String} [appOptions.trackingId = assigned moduleName]
     *      trackingId is a unique String as moduleName but better readable
     *      used by CoreMetrics' tracking
     *
     * @returns {Object}
     *  The launcher object expected by the method ox.launch().
     */
    ApplicationLauncher.createLauncher = function (moduleName, appOptions) {

        // hide core AppSuite topbar in standalone mode
        if (ApplicationLauncher.getStandaloneMode()) {
            ext.point('io.ox/core/topbar').disable('default');
        }

        // Bug 28664: remove all save points before they are checked to decide
        // whether to show the 'unsaved documents' dialog
        ext.point('io.ox/core/logout').extend({
            id: moduleName + '/logout/before',
            index: 'first', // run before the save points are checked by the core
            logout: function () {
                _.each(ox.ui.App.get(moduleName), function (app) {
                    if (!app.hasUnsavedChanges()) { app.removeRestorePoint(); }
                });
            }
        });

        // listen to user logout and notify all running applications
        ext.point('io.ox/core/logout').extend({
            id: moduleName + '/logout/quit',
            index: 'last', // run after the dialog (not called if logout was rejected)
            logout: function () {
                var promises = _.map(ox.ui.App.get(moduleName), function (app) {
                    // bug 33152: ignore the result of the quit handlers (rejected promise cancels logout)
                    var def = $.Deferred();
                    app.executeQuitHandlers('logout').always(function () { def.resolve(); });
                    return def.promise();
                });
                return $.when.apply($, promises);
            }
        });

        function getFullFileName(file) {
            return (file && (file['com.openexchange.file.sanitizedFilename'] || file.filename)) || '\xa0';
        }

        // callback method invoked from ox.launch()
        function getApp(launchOptions) {

            // Load from URL: different launch options are given, check if we have 'simple' file descriptor (file id, folder id)
            // in the URL hash params and build new simple launch options from it
            if (!launchOptions || launchOptions.id) {
                if (Config.getUrlFlag('convert')) {
                    launchOptions = {
                        action: 'convert',
                        file: null,
                        template: true,
                        keepFile: true,
                        target_filename: _.url.hash('destfilename'),
                        target_folder_id: _.url.hash('destfolderid'),
                        templateFile: { folder_id: launchOptions.folder, id: launchOptions.id }
                    };
                    _.url.hash('convert', null);
                    _.url.hash('destfilename', null);
                    _.url.hash('destfolderid', null);
                } else {
                    launchOptions = { action: 'load', file: { folder_id: launchOptions.folder, id: launchOptions.id, source: 'drive' } };
                }
                launchOptions.actionSource = 'url';
            }

            // get file descriptor from options
            var fileDesc = Utils.getObjectOption(launchOptions, 'file', null);

            // workaround for missing source info, if missing it must be drive
            if (fileDesc && !fileDesc.source) { fileDesc.source = 'drive'; }

            // find running application for the specified document
            var app = ApplicationLauncher.getRunningApp(moduleName, fileDesc);
            if (app) { return app; }

            // the icon shown in the top bar launcher
            var icon = Utils.getStringOption(appOptions, 'icon');
            icon = _.isString(icon) ? Forms.getIconClasses(icon) : '';

            // trackingId is a unique string as moduleName but better readable
            var trackingId = Utils.getStringOption(appOptions, 'trackingId', moduleName);

            // the name of the edited file, with and without extension
            var fullFileName = getFullFileName(launchOptions.file);
            var shortFileName = Utils.getFileBaseName(fullFileName);

            // no running application: create and initialize a new application object
            app = ox.ui.createApp({
                name: moduleName,
                title: shortFileName, // bug 38177: a title must be passed to always get the launcher button in the top bar
                closable: true,
                userContent: icon.length > 0,
                userContentIcon: icon,
                userContentClass: ApplicationLauncher.getDocumentType(moduleName) + '-launcher',
                trackingId: trackingId
            });

            // create the application window
            var appWindow = ox.ui.createWindow({
                name: moduleName,
                search: false,
                chromeless: true
            });
            app.setWindow(appWindow);

            appWindow.isBusy = function () {
                return !!this.nodes.blocker && this.nodes.blocker.is(':visible');
            };

            // override the busy() method of the application window to add Documents specific stuff
            appWindow.busy = _.wrap(appWindow.busy, function (busyMethod, pct, sub, callback, options) {

                // store current busy state
                var isAlreadyBusy = this.isBusy();

                // invoke core method
                return busyMethod.call(this, pct, sub, function () {

                    // the window blocker element (bound to 'this')
                    var blockerNode = this;
                    // the header/footer container nodes
                    var headerNode = blockerNode.find('.header');
                    var footerNode = blockerNode.find('.footer');

                    // initialization when busy mode starts
                    if (!isAlreadyBusy) {

                        // clean header/footer container nodes
                        headerNode.empty();
                        footerNode.empty();

                        // add special marker class for custom CSS formatting
                        blockerNode.addClass('io-ox-office-blocker').css('opacity', '');

                        // add file name to header area
                        if (Utils.getBooleanOption(options, 'showFileName', false)) {
                            headerNode.append($('<div class="filename clear-title">').text(_.noI18n(fullFileName)));
                        }

                        // on IE, remove stripes (they are displayed too nervous and flickering)
                        if (_.browser.IE) {
                            blockerNode.find('.progress-striped').removeClass('progress-striped');
                        }
                    }

                    // invoke the user-defined callback function
                    if (_.isFunction(callback)) {
                        return callback.call(appWindow, blockerNode, headerNode, footerNode, isAlreadyBusy);
                    }
                });
            });

            // bug 46405: override the launch method of the application instance to be able to fetch
            // the application mix-in class on demand, and to defer invocation of the original launch
            // method until the application has been extended with that mix-in class
            app.setLauncher(function () {

                // check that some launch options have been passed (action must always be part)
                if (!_.isObject(launchOptions) || !_.isString(launchOptions.action)) {
                    return $.Deferred();
                }

                // show a busy blocker screen to give the user a quick feedback that the app is starting
                if (launchOptions.hidden !== true) {
                    appWindow.show();
                    appWindow.setTitle(shortFileName);
                    appWindow.busy(null, null, null, { showFileName: true });
                }

                // load the application mix-in constructor for the current application type
                require([moduleName + '/app/application']).done(function (ApplicationClass) {

                    // the application may have been closed very quickly before RequireJS
                    // provides the application module
                    if (!app.cid) { return; }

                    // construct with mix-in application class (with the disadvantage that
                    // the application is not an instance of ApplicationClass)
                    ApplicationClass.call(app, launchOptions);
                });

                // bug 46405: do not return a pending promise (otherwise it is possible to open
                // multiple application instances for the same document)
                return $.when();
            });

            return app;
        }

        // ox.launch() expects an object with the method getApp()
        return { getApp: getApp };
    };

    // exports ================================================================

    return ApplicationLauncher;
});
