/* *
 *    OPEN-XCHANGE legal information
 *
 *    All intellctual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.documentconverter.client.osgi;

import java.net.URL;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.ajax.requesthandler.osgiservice.AJAXModuleActivator;
import com.openexchange.capabilities.CapabilityService;
import com.openexchange.config.ConfigurationService;
import com.openexchange.documentconverter.IDocumentConverter;
import com.openexchange.documentconverter.ManagerBasics;
import com.openexchange.documentconverter.NonNull;
import com.openexchange.documentconverter.client.impl.ClientConfig;
import com.openexchange.documentconverter.client.impl.ClientManager;
import com.openexchange.documentconverter.client.json.DocumentConverterActionFactory;
import com.openexchange.documentconverter.client.preview.DocumentConverterPreviewService;
import com.openexchange.file.storage.composition.IDBasedFileAccessFactory;
import com.openexchange.file.storage.composition.IDBasedFolderAccessFactory;
import com.openexchange.filemanagement.ManagedFileManagement;
import com.openexchange.guard.api.GuardApi;
import com.openexchange.imagetransformation.ImageServerTransformationService;
import com.openexchange.imagetransformation.ImageTransformationService;
import com.openexchange.preview.InternalPreviewService;

public class DocumentConverterClientActivator extends AJAXModuleActivator {

    /**
     * Initializes a new {@link DocumentConverterClientActivator}.
     */
    public DocumentConverterClientActivator() {
        super();
    }

    /* (non-Javadoc)
     * @see com.openexchange.osgi.DeferredActivator#getNeededServices()
     */
    @Override
    protected Class<?>[] getNeededServices() {
        return new Class<?>[] { ConfigurationService.class,
            CapabilityService.class,
            IDBasedFolderAccessFactory.class,
            IDBasedFileAccessFactory.class,
            ManagedFileManagement.class };
    }

    /* (non-Javadoc)
     * @see com.openexchange.osgi.HousekeepingActivator#getOptionalServices()
     */
    @Override
    protected Class<?>[] getOptionalServices() {
        return new Class<?>[] {
            IDocumentConverter.class,
            ImageTransformationService.class,
            GuardApi.class
        };
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void startBundle() throws Exception {
        try {
            if (log.isInfoEnabled()) {
                log.info("starting bundle: " + m_serviceName);
            }

            try {
                final ConfigurationService configService = getService(ConfigurationService.class);

                if (null != configService) {
                    final ClientConfig clientConfig = new ClientConfig();

                    // register DocumentConverter implementation service
                    implReadClientConfig(configService, clientConfig);

                    m_clientManager = ClientManager.getInstance(this, clientConfig, LoggerFactory.getLogger(ClientManager.class),
                        ManagerBasics.initUserLogging(ClientManager.class, clientConfig.CLIENT_LOGFILE, clientConfig.CLIENT_LOGLEVEL));

                    // register DocumentConverter related services
                    if (null != m_clientManager) {
                        log.info("Registering DocumentConverter services");
                        registerService(IDocumentConverter.class, m_clientManager, null);
                        registerService(ImageServerTransformationService.class, m_clientManager, null);
                    } else {
                        log.error("Could not register DocumentConverter services");
                    }

                    // register DocumentConverter related Json habdlers
                    registerModule(new DocumentConverterActionFactory(this), "oxodocumentconverter");

                    // register OfficePreviewService
                    log.info("Registering DocumentConverterPreviewService");
                    registerService(InternalPreviewService.class, new DocumentConverterPreviewService(this));
                }
            } catch (@SuppressWarnings("unused") Exception e) {
                if (log.isErrorEnabled()) {
                    log.error("starting bundle failed: " + m_serviceName);
                    log.error("Documentconverter manager could not be initialized");
                }
            }
        } catch (Throwable e) {
            e.printStackTrace();
        }

        // register information bean if/when ManagementService is available
        if (null != m_clientManager) {
            openTrackers();
        }
    }

    // - Implementation --------------------------------------------------------

    private static void implReadClientConfig(@NonNull ConfigurationService configService, @NonNull ClientConfig clientConfig) {
        final String emptyString = "";
        String curStringEntry = emptyString;
        final int emptyInt = Integer.MIN_VALUE;
        int curIntEntry = emptyInt;

        // REMOTEURL_DOCUMENTCONVERTER
        if (!StringUtils.isEmpty(curStringEntry = configService.getProperty("com.openexchange.documentconverter.client.remoteDocumentConverterUrl", emptyString))) {
            try {
                clientConfig.REMOTEURL_DOCUMENTCONVERTER = curStringEntry.startsWith("http") ? new URL(curStringEntry) : null;
            } catch (@SuppressWarnings("unused") Exception e) {
                if (log.isErrorEnabled()) {
                    log.error("com.openexchange.documentconverter.client.remoteDocumentConverterUrl is not a valid http URL: " + curStringEntry + " => Remote document conversion disabled!");
                }

                clientConfig.REMOTEURL_DOCUMENTCONVERTER = null;
            }
        } else if (log.isErrorEnabled()) {
            log.error("com.openexchange.documentconverter.client.remoteDocumentConverterUrl is not set at all => Documentconverter functionality will not be available!");
        }

        // MAX_DOCUMENT_SOURCEFILESIZE_MB
        if ((curIntEntry = configService.getIntProperty("com.openexchange.documentconverter.client.maxDocumentSourceFileSizeMB", emptyInt)) != emptyInt) {
            clientConfig.MAX_DOCUMENT_SOURCEFILESIZE_MB = curIntEntry;
        }

        // REMOTEURL_IMAGESERVER
        /* TODO (KA): enable reading of ImageServer Url for availability
        if (!StringUtils.isEmpty(curStringEntry = configService.getProperty("com.openexchange.documentconverter.client.remoteImageServerUrl", emptyString))) {
            try {
                clientConfig.REMOTEURL_IMAGESERVER = curStringEntry.startsWith("http") ? new URL(curStringEntry) : null;
            } catch (@SuppressWarnings("unused") Exception e) {
                if (log.isErrorEnabled()) {
                    log.error("com.openexchange.documentconverter.client.remoteImageServerUrl is not a valid http URL: " + curStringEntry + " => Remote image transformation disabled!");
                }

                clientConfig.REMOTEURL_IMAGESERVER = null;
            }
        } else if (log.isErrorEnabled()) {
            log.error("com.openexchange.documentconverter.client.remoteImageServerUrl is not set at all => ImageServer functionality will not be available!");
        }
        */

        // MAX_IMAGE_SOURCEFILESIZE_MB
        if ((curIntEntry = configService.getIntProperty("com.openexchange.documentconverter.client.maxImageSourceFileSizeMB", emptyInt)) != emptyInt) {
            clientConfig.MAX_IMAGE_SOURCEFILESIZE_MB = curIntEntry;
        }

        // PREVIEW_ENABLE_CACHE_LOOKUP
        clientConfig.PREVIEW_ENABLE_CACHE_LOOKUP = configService.getBoolProperty("com.openexchange.documentconverter.enableCacheLookup", false);

        // - Internally used config items ----------------------------------

        // ENGINE_LOGFILE
        if ((curStringEntry = configService.getProperty("com.openexchange.documentconverter.client.logFile", emptyString)) != emptyString) {
            clientConfig.CLIENT_LOGFILE = curStringEntry;
        }

        // ENGINE_LOGLEVEL
        if ((curStringEntry = configService.getProperty("com.openexchange.documentconverter.client.logLevel", emptyString)) != emptyString) {
            clientConfig.CLIENT_LOGLEVEL = curStringEntry;
        }

        // ENGINE_DEBUG
        clientConfig.CLIENT_DEBUG = configService.getBoolProperty("com.openexchange.documentconverter.client.debug", false);

    }

    // - Static Members --------------------------------------------------------

    protected static Logger log = LoggerFactory.getLogger(ClientManager.class);

    // - Members ---------------------------------------------------------------

    private final String m_serviceName = "com.openexchange.documentconverter.client";

    ClientManager m_clientManager = null;
}
