/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */


package com.openexchange.documentconverter;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLDecoder;
import java.util.HashMap;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ch.qos.logback.classic.Level;
import ch.qos.logback.classic.LoggerContext;
import ch.qos.logback.classic.encoder.PatternLayoutEncoder;
import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.rolling.RollingFileAppender;
import ch.qos.logback.core.rolling.TimeBasedRollingPolicy;


/**
 * {@link BasicManager}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.8.0
 */
/**
 * {@link ManagerBasics}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.8.0
 */
public abstract class ManagerBasics implements ILogProvider {

    final private static long HTTP_CONNECT_TIMEOUT_MS=60000;

    /**
     * Initializes a new {@link ManagerBasics}.
     */
    public ManagerBasics() {
        super();

        if (null == m_basicLogProvider) {
            m_basicLogProvider = this;
        }
    }

    public abstract void resetRemoteConnection(ServerType serverType);

    /* (non-Javadoc)
     * @see com.openexchange.documentconverter.ILogProvider#getSysLog()
     */
    @Override
    public Logger getSysLog() {
        return m_sysLog;
    }

    /* (non-Javadoc)
     * @see com.openexchange.documentconverter.ILogProvider#getUserLog()
     */
    @Override
    public Logger getUserLog() {
        return m_userLog;
    }

    /**
     * @param file
     * @param locale
     * @return
     */
    public static StringBuilder getFileHashBuilder(File file, String locale) {
        StringBuilder hashBuilder = null;

        if ((null != file) && file.canRead()) {
            hashBuilder = new StringBuilder();

            try {
                hashBuilder.append(Long.toString(file.length())).append('_').append(
                    Long.toString(FileUtils.checksumCRC32(file)));
            } catch (Exception e) {
                logExcpImpl(m_basicLogProvider, e);
            }

            // 'en*' is considered to be the default locale; add all other
            // locales to the
            // input file hash in order to be able to perform locale based
            // conversions
            if (!StringUtils.isEmpty(locale)) {
                hashBuilder.append('-').append(locale).append('-');
            }
        }

        return hashBuilder;
    }

    /**
     * @return
     */
    public static int getHTTPConnectTimeoutMilliseconds() {
        return ((int) HTTP_CONNECT_TIMEOUT_MS);
    }

    /**
     * @return
     */
    public static int getHTTPReadTimeoutMilliseconds() {
        return ((int) ((HTTP_CONNECT_TIMEOUT_MS * 210L) / 100L));
    }

    /**
     * @param prefix
     * @param fileName
     * @return
     */
    public static File createTempFile(String prefix, String fileName) {
        return createTempFile(FileUtils.getTempDirectoryPath(), prefix, fileName);
    }

    /**
     * @param prefix
     * @param fileName
     * @return
     */
    public static File createTempFile(@NonNull String tmpRoot, String prefix, String fileName) {
        File tmpFile = null;

        try {
            if ((null != fileName) && (fileName.length() > 0)) {
                final File scratchDir = new File(tmpRoot);
                final String dirPrefix = ((((null != prefix) && (prefix.length() > 0)) ? prefix : Long.valueOf(System.currentTimeMillis()).toString())) + "_";
                final long number = System.nanoTime();
                final long maxIterations = 10000;

                for (long i = number, end = number + maxIterations; i < end; ++i) {
                    final File tempDir = new File(scratchDir, dirPrefix + i);

                    if (tempDir.mkdirs()) {
                        tmpFile = new File(tempDir.getAbsolutePath(), fileName);
                        break;
                    }
                }
            } else {
                final File scratchDir = new File(tmpRoot);

                if ((scratchDir.isDirectory() && scratchDir.canWrite()) || mkdirs(scratchDir)) {
                    // create a temp. file with the name 'prefix*.tmp' within the configured documentconverter cache directory
                    tmpFile = File.createTempFile(prefix, ".tmp", scratchDir);
                }
            }
        } catch (IOException e) {
            logExcpImpl(m_basicLogProvider, e);
        }

        return tmpFile;
    }

    /**
     * @param dirFile
     * @return
     */
    public static boolean mkdirs(File directory) {
        boolean ret = false;

        if (null != directory) {
            if (!directory.exists()) {
                try {
                    FileUtils.forceMkdir(directory);
                } catch (IOException e) {
                    logExcpImpl(m_basicLogProvider, e);
                }

                FileUtils.waitFor(directory, 3);
            }

            ret = directory.canWrite();
        }

        return ret;
    }

    /**
     * @param filename
     * @return
     */
    public static byte[] getResourceBuffer(String filename) {
        ClassLoader loader = Thread.currentThread().getContextClassLoader();
        byte[] resourceBuffer = null;

        if (null == loader) {
            loader = ManagerBasics.class.getClassLoader();
        }

        final URL url = loader.getResource(filename);

        if (null != url) {
            URLConnection connection = null;

            try {
                connection = url.openConnection();

                if (null != connection) {
                    connection.connect();

                    try (InputStream resourceInputStm = connection.getInputStream()) {
                        if (null != resourceInputStm) {
                            resourceBuffer = IOUtils.toByteArray(resourceInputStm);
                        }
                    }
                }
            } catch (IOException e) {
                logExcpImpl(m_basicLogProvider, e);
            }
        }

        return resourceBuffer;
    }

    /**
     * @param jobProperties
     * @param deleteInputFile
     * @return
     */
    public static File getJobInputFile(@NonNull HashMap<String, Object> jobProperties, @NonNull boolean[] deleteInputFile) {
        File inputFile = (File) jobProperties.get(Properties.PROP_INPUT_FILE);

        deleteInputFile[0] = false;

        // if no input file is given, write content of possible InputStream
        // property to a temp file; the conversion is always done using a real
        // file as input source due to very bad performance with InputStream
        // reads via the UNO bridge (at least with latest LO readerengine)
        if (null == inputFile) {
            try (final InputStream inputStm = (InputStream) jobProperties.get(Properties.PROP_INPUT_STREAM)) {
                if (null != inputStm) {
                    if (null != (inputFile = ManagerBasics.createTempFile(FileUtils.getTempDirectoryPath(), "oxcs", null))) {
                        inputFile.deleteOnExit();

                        try {
                            FileUtils.copyInputStreamToFile(inputStm, inputFile);
                            IOUtils.closeQuietly(inputStm);

                            // set the just filled inputFile as new source at our
                            // job
                            // and remove the previously set input stream;
                            jobProperties.put(Properties.PROP_INPUT_FILE, inputFile);
                            jobProperties.remove(Properties.PROP_INPUT_STREAM);

                            // the temporarily created file needs to be deleted by
                            // the caller of this method later in time
                            deleteInputFile[0] = true;
                        } catch (IOException e) {
                            logExcpImpl(m_basicLogProvider, e);

                            FileUtils.deleteQuietly(inputFile);
                            inputFile = null;
                        }
                    }
                }
            } catch (IOException e) {
                logExcpImpl(m_basicLogProvider, e);
            }
        }

        // check for input type %PDF-x.x, since this input
        // type is explicitly checked at various locations
        if (null != inputFile) {
            try (final InputStream inputStm = FileUtils.openInputStream(inputFile)) {
                final byte[] buffer = new byte[256];

                if ((inputStm.read(buffer) > 0) && new String(buffer, "UTF-8").trim().toLowerCase().startsWith("%pdf-")) {
                    jobProperties.put(Properties.PROP_INPUT_TYPE, "pdf");
                }
            } catch (IOException e) {
                logExcpImpl(m_basicLogProvider, e);
            }
        }

        return inputFile;
    }

    // - Logging ---------------------------------------------------------------

    public static Logger initUserLogging(@NonNull Class<?> clazz, String userLogPath, String userLogLevel) {
        final File userLogDir = StringUtils.isEmpty(userLogPath) ? null : new File(userLogPath).getParentFile();
        ch.qos.logback.classic.Logger ret = null;

        // init the user log, if requested
        if ((null != userLogDir) && userLogDir.isDirectory() && userLogDir.canWrite()) {
            final LoggerContext loggerContext = new LoggerContext();

            ret = loggerContext.getLogger(clazz);

            if (null != ret) {
                final PatternLayoutEncoder layoutEncoder = new PatternLayoutEncoder();

                layoutEncoder.setPattern("[%date]-[%thread] %level: %msg%n");
                layoutEncoder.setContext(loggerContext);
                layoutEncoder.start();

                final RollingFileAppender<ILoggingEvent> fileAppender = new RollingFileAppender<>();
                final TimeBasedRollingPolicy<ILoggingEvent> rollingPolicy = new TimeBasedRollingPolicy<>();

                fileAppender.setFile(userLogPath);
                fileAppender.setEncoder(layoutEncoder);
                fileAppender.setContext(loggerContext);

                rollingPolicy.setParent(fileAppender);
                rollingPolicy.setContext(loggerContext);
                rollingPolicy.setCleanHistoryOnStart(false);
                rollingPolicy.setMaxHistory(6);
                rollingPolicy.setFileNamePattern(userLogPath + ".%d");
                rollingPolicy.start();

                fileAppender.setRollingPolicy(rollingPolicy);
                fileAppender.start();

                ret.addAppender(fileAppender);
                ret.setAdditive(false);

                Level logLevel = null;

                if (!StringUtils.isEmpty(userLogLevel)) {
                    if (userLogLevel.equals("TRACE")) {
                        logLevel = Level.TRACE;
                    } else if (userLogLevel.equals("DEBUG")) {
                        logLevel = Level.DEBUG;
                    } else if (userLogLevel.equals("INFO")) {
                        logLevel = Level.INFO;
                    } else if (userLogLevel.equals("WARN")) {
                        logLevel = Level.WARN;
                    } else if (userLogLevel.equals("ERROR")) {
                        logLevel = Level.ERROR;
                    }
                }

                if (null != logLevel) {
                    ret.setLevel(logLevel);
                }
            } else {
                System.err.println("Can't open log file: " + userLogPath);
            }
        }

        return ret;
    }

    /**
     * @param logProvider
     * @return
     */
    public static boolean isLogTraceImpl(ILogProvider logProvider) {
        final Logger sysLog = logProvider.getSysLog();
        final Logger userLog = logProvider.getUserLog();

        return (((null != sysLog) && sysLog.isTraceEnabled()) || ((null != userLog) && userLog.isTraceEnabled()));
    }

    /**
     * @return true, if the log level 'info' is enabled
     */
    public static boolean isLogDebugImpl(ILogProvider logProvider) {
        final Logger sysLog = logProvider.getSysLog();
        final Logger userLog = logProvider.getUserLog();

        return (((null != sysLog) && sysLog.isDebugEnabled()) || ((null != userLog) && userLog.isDebugEnabled()));
    }

    /**
     * @return true, if the log level 'info' is enabled
     */
    public static boolean isLogInfoImpl(ILogProvider logProvider) {
        final Logger sysLog = logProvider.getSysLog();
        final Logger userLog = logProvider.getUserLog();

        return (((null != sysLog) && sysLog.isInfoEnabled()) || ((null != userLog) && userLog.isInfoEnabled()));
    }

    /**
     * @return true, if the log level 'warn' is enabled
     */
    public static boolean isLogWarnImpl(ILogProvider logProvider) {
        final Logger sysLog = logProvider.getSysLog();
        final Logger userLog = logProvider.getUserLog();

        return (((null != sysLog) && sysLog.isWarnEnabled()) || ((null != userLog) && userLog.isWarnEnabled()));
    }

    /**
     * @return true, if the log level 'error' is enabled
     */
    public static boolean isLogErrorImpl(ILogProvider logProvider) {
        final Logger sysLog = logProvider.getSysLog();
        final Logger userLog = logProvider.getUserLog();

        return (((null != sysLog) && sysLog.isErrorEnabled()) || ((null != userLog) && userLog.isErrorEnabled()));
    }


    /**
     * @param e
     */
    public static void logExcpImpl(ILogProvider logProvider, Exception e) {
        if (isLogDebugImpl(logProvider)) {
            String message = (null != e) ? e.getMessage() : null;
            final String lowerMessage = (null != message) ? message.toLowerCase() : null;

            if ((e instanceof SocketException) || (e instanceof SocketTimeoutException) || ((e instanceof IOException) && (null != lowerMessage) && ((lowerMessage.contains("broken pipe") || lowerMessage.contains(("connection reset")) || lowerMessage.contains(("invalid stream")))))) {
                // ignore socket exceptions; these may occur, when a connection timeout has happened
            } else {
                if (null != message) {
                    if (message.length() > Properties.MAX_ERROR_MESSAGE_TOTAL_LENGTH) {
                        final String fillStr = "...";
                        final int beginLength = Properties.MAX_ERROR_MESSAGE_TOTAL_LENGTH - Properties.MAX_ERROR_MESSAGE_END_LENGTH - fillStr.length();
                        final int endLength = Math.min(message.length() - beginLength, Properties.MAX_ERROR_MESSAGE_END_LENGTH);
                        message = message.substring(0, beginLength) + ((endLength > 0) ? ("..." + message.substring(message.length() - endLength)) : "");
                    }
                } else {
                    message = "unknown";
                }

                logImpl(logProvider, LogType.LOGTYPE_ERROR, "Documentconverter caught an exception: \n" + message, null);
            }
        }
    }

    /**
     * @param logType
     * @param message
     * @param jobProperties
     * @param extraData
     */
    public static void logImpl(ILogProvider logProvider, final int logType, final String message, HashMap<String, Object> jobProperties, LogData... extraData) {
        final StringBuilder logStrBuilder = new StringBuilder(message);
        StringBuilder infoStrBuilder = null;
        String fileName = null;
        final Logger sysLog = logProvider.getSysLog();
        final Logger userLog = logProvider.getUserLog();

        if (null != jobProperties) {
            fileName = (String) jobProperties.get(Properties.PROP_INFO_FILENAME);
        }

        if (extraData.length > 0) {
            logStrBuilder.append(" (");

            for (int i = 0; i < extraData.length; ++i) {
                final LogData logData = extraData[i];

                if (null != logData) {
                    if (i > 0) {
                        logStrBuilder.append(", ");
                    }

                    logStrBuilder.append(logData.getKey());
                    logStrBuilder.append('=');

                    final String value = logData.getValue();
                    logStrBuilder.append((null != value) ? value : "null");
                }
            }
        }

        if (null != fileName) {
            try {
                fileName = URLDecoder.decode(fileName, "UTF-8");
            } catch (UnsupportedEncodingException e) {
                logExcpImpl(logProvider, e);
            }

            if (extraData.length > 0) {
                logStrBuilder.append(", filename=");
            } else {
                logStrBuilder.append(" (filename=");
            }

            logStrBuilder.append(fileName).append(')');
        } else if (extraData.length > 0) {
            logStrBuilder.append(')');

        }

        if ((null != jobProperties) && !jobProperties.isEmpty()) {
            for (String key : jobProperties.keySet()) {
                // log all info_ properties but info_Filename only in case of an error
                if (key.startsWith(Properties.INFO_PREFIX) && !key.equals(Properties.PROP_INFO_FILENAME)) {
                    final Object obj = jobProperties.get(key);

                    if (null != infoStrBuilder) {
                        infoStrBuilder.append(", ");
                    } else {
                        infoStrBuilder = new StringBuilder(" (jobproperties: ");
                    }

                    infoStrBuilder.append(key.substring(Properties.INFO_PREFIX.length())).append('=').append(
                            (null != obj) ? obj.toString() : "null");
                }
            }

            if (null != infoStrBuilder) {
                infoStrBuilder.append(')');
            }
        }

        if (null != infoStrBuilder) {
            logStrBuilder.append(infoStrBuilder);
        }

        final String logMessage = logStrBuilder.toString();

        if (logMessage.length() > 0) {
            if (LogType.LOGTYPE_INFO == logType) {
                if (null != sysLog) {
                    sysLog.info(logMessage);
                }

                if (null != userLog) {
                    userLog.info(logMessage);
                }
            } else if (LogType.LOGTYPE_WARN == logType) {
                if (null != sysLog) {
                    sysLog.warn(logMessage);
                }

                if (null != userLog) {
                    userLog.warn(logMessage);
                }
            } else if (LogType.LOGTYPE_ERROR == logType) {
                if (null != sysLog) {
                    sysLog.error(logMessage);
                }

                if (null != userLog) {
                    userLog.error(logMessage);
                }
            } else if (LogType.LOGTYPE_DEBUG == logType) {
                if (null != sysLog) {
                    sysLog.debug(logMessage);
                }

                if (null != userLog) {
                    userLog.debug(logMessage);
                }
            } else if (LogType.LOGTYPE_TRACE == logType) {
                if (null != sysLog) {
                    sysLog.trace(logMessage);
                }

                if (null != userLog) {
                    userLog.trace(logMessage);
                }
            }
        }
    }

    /**
     * @return
     */
    public static ILogProvider getBasicLogProvider() {
        return m_basicLogProvider;
    }

    // - Static members --------------------------------------------------------

    static ILogProvider m_basicLogProvider = null;

    static Logger m_sysLog = LoggerFactory.getLogger(ManagerBasics.class);

    static Logger m_userLog = null;
}
