/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.documentconverter;

import java.awt.image.BufferedImage;
import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import javax.imageio.ImageIO;
import org.apache.commons.lang.StringUtils;
import com.openexchange.imagetransformation.BasicTransformedImage;
import com.openexchange.imagetransformation.ImageTransformations;
import com.openexchange.imagetransformation.ScaleType;
import com.openexchange.imagetransformation.TransformedImage;

/**
 * {@link ImageServerTransformations}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.8.0
 */
public class ImageServerTransformations implements ImageTransformations {

    /**
     * Initializes a new {@link ImageServerTransformations}.
     * @param transformationsString
     */
    public ImageServerTransformations(@NonNull IImageTransformer transformer) {
        super();

        m_transformer = transformer;
    }

    /**
     * Initializes a new {@link ImageServerTransformations}.
     * @param transformer
     */
    public ImageServerTransformations(@NonNull IImageTransformer transformer, @NonNull InputStream imageStm, Object source) {
        super();

        m_transformer = transformer;

        initSource(source);
        initImageInput(imageStm);
    }


    /**
     * Initializes a new {@link ImageServerTransformations}.
     * @param manager
     */
    public ImageServerTransformations(@NonNull IImageTransformer transformer, @NonNull InputStream imageStm, @NonNull ArrayList<ImageServerTransformations.Transformation> transformations, Object source) {
        this(transformer, imageStm, source);

        for (ImageServerTransformations.Transformation curTransformation : transformations) {
            m_transformationList.add(curTransformation.createCopy());
        }
    }

    /****************************************************************************
     * Object                                                                   *
     ****************************************************************************/

    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        StringBuilder ret = new StringBuilder();

        for (final Transformation transformation : m_transformationList) {
            if (null != transformation) {
                // add '-' as delimiter between multiple transformations
                if (ret.length() > 0) {
                    ret.append('-');
                }

                ret.append(getLowerCaseAsciiTriple(transformation.m_transformationType.toString()));

                switch (transformation.m_transformationType) {
                    case SCALE: {
                        final Boolean imageShrinkOnly = (Boolean) transformation.m_transformationProperties.get(Properties.PROP_IMAGE_SHRINK_ONLY);

                        ret.append(transformation.m_transformationProperties.get(Properties.PROP_PIXEL_WIDTH)).
                            append('x').
                            append(transformation.m_transformationProperties.get(Properties.PROP_PIXEL_HEIGHT)).
                            append('-').
                            append(getLowerCaseAsciiTriple(transformation.m_transformationProperties.get(Properties.PROP_IMAGE_SCALE_TYPE).toString()));

                        if ((null != imageShrinkOnly) && imageShrinkOnly.booleanValue()) {
                            ret.append("_so");
                        }


                        break;
                    }

                    case CROP: {
                        ret.append(transformation.m_transformationProperties.get(Properties.PROP_PIXEL_X)).
                            append('.').
                            append(transformation.m_transformationProperties.get(Properties.PROP_PIXEL_Y)).
                            append('@').
                            append(transformation.m_transformationProperties.get(Properties.PROP_PIXEL_WIDTH)).
                            append('x').
                            append(transformation.m_transformationProperties.get(Properties.PROP_PIXEL_HEIGHT));

                        break;
                    }

                    case ROTATE:
                    case COMPRESS:
                    case NONE:
                    default: {
                        break;
                    }
                }
            }
        }

        return ret.toString();
    }

    // - ImageTransformations

    @Override
    public ImageTransformations scale(int maxWidth, int maxHeight, ScaleType scaleType) {
        return scale(maxWidth, maxHeight, scaleType, false);
    }

    /* (non-Javadoc)
     * @see com.openexchange.imagetransformation.ImageTransformations#scale(int, int, com.openexchange.imagetransformation.ScaleType)
     */
    @Override
    public ImageTransformations scale(int maxWidth, int maxHeight, ScaleType scaleType, boolean shrinkOnly) {
        final HashMap<String, Object> properties = new HashMap<String, Object>();

        properties.put(Properties.PROP_PIXEL_WIDTH, Integer.valueOf(maxWidth));
        properties.put(Properties.PROP_PIXEL_HEIGHT, Integer.valueOf(maxHeight));
        properties.put(Properties.PROP_IMAGE_SCALE_TYPE, scaleType);
        properties.put(Properties.PROP_IMAGE_SHRINK_ONLY, Boolean.valueOf(shrinkOnly));


        m_transformationList.add(new Transformation(TransformationType.SCALE, properties));

        return this;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imagetransformation.ImageTransformations#rotate()
     */
    @Override
    public ImageTransformations rotate() {
        m_transformationList.add(new Transformation(TransformationType.ROTATE, null));

        return this;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imagetransformation.ImageTransformations#crop(int, int, int, int)
     */
    @Override
    public ImageTransformations crop(int x, int y, int width, int height) {
        final HashMap<String, Object> properties = new HashMap<String, Object>();

        properties.put(Properties.PROP_PIXEL_X, Integer.valueOf(x));
        properties.put(Properties.PROP_PIXEL_Y, Integer.valueOf(y));
        properties.put(Properties.PROP_PIXEL_WIDTH, Integer.valueOf(width));
        properties.put(Properties.PROP_PIXEL_HEIGHT, Integer.valueOf(height));


        m_transformationList.add(new Transformation(TransformationType.SCALE, properties));

        return this;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imagetransformation.ImageTransformations#compress()
     */
    @Override
    public ImageTransformations compress() {
        m_transformationList.add(new Transformation(TransformationType.COMPRESS, null));

        return this;
    }

    /**
     * @return
     * @throws IOException
     */
    @Override
    public BufferedImage getImage() throws IOException {
        BufferedImage ret = null;

        try (final InputStream imageInputStm = new BufferedInputStream(getInputStream("png"))) {
            ret = ImageIO.read(imageInputStm);
        }

        return ret;
    }

    @Override
    public byte[] getBytes(String formatName) throws IOException {
        // TODO (KA): impl.
        return null;
    }

    /**
     * @param formatName
     * @return
     * @throws IOException
     */
    @Override
    public InputStream getInputStream(String formatName) throws IOException {
        final byte[] imageBytes = getBytes(formatName);
        InputStream ret = null;

        if (null != imageBytes) {
            ret = new ByteArrayInputStream(imageBytes);
        }

        return ret;
    }

    /**
     * @param formatName
     * @return
     * @throws IOException
     */
    @Override
    public BasicTransformedImage getTransformedImage(String formatName) throws IOException {
        return getFullTransformedImage(formatName);
    }

    /**
     * @param formatName
     * @return
     * @throws IOException
     */
    @Override
    public TransformedImage getFullTransformedImage(String formatName) throws IOException {
        return ((null != m_transformer) ? m_transformer.transformImage(m_imageInputStm, formatName, this) : null);
    }

    // - Implementation --------------------------------------------------------

    /**
     * @return
     */
    public int getTransformationCount() {
        return m_transformationList.size();
    }

    /**
     * @param pos
     * @return
     */
    public Transformation getTransformation(int pos) {
        return ((pos < m_transformationList.size()) ? m_transformationList.get(pos) : null);

    }

    // - Impl ------------------------------------------------------------------

    /**
     * @param imageStm
     * @return
     */
    protected boolean initImageInput(InputStream imageInputStm) {
        return (null != (m_imageInputStm = imageInputStm));
    }

    /**
     * @param imageStm
     * @return
     */
    protected boolean initSource(Object source) {
        m_source = source;

        return (null != m_source);
    }

    /**
     * @param tupleText
     * @param delim
     * @param tuple
     */
    protected static void parseIntTuple(String tupleText, String delim, int[] tuple) {
        if (!StringUtils.isEmpty(tupleText)) {
            final String[] tupleTexts = tupleText.split(delim);

            if (tupleTexts.length >= 1) {
                tuple[0] = Integer.parseInt(tupleTexts[0]);

                if (tupleTexts.length >= 2) {
                    tuple[1] = Integer.parseInt(tupleTexts[1]);
                }
            }

        }
    }

    /**
     * @param text
     * @return
     */
    protected static String getLowerCaseAsciiTriple(String text) {
        final StringBuilder ret = new StringBuilder();
        int givenCharCount = 0;

        if (!StringUtils.isEmpty(text)) {
            givenCharCount = text.length();
            ret.append(text.substring(0, Math.min(CHARTRIPLE_COUNT, givenCharCount)).toLowerCase());
        }

        // fill all missing characters with 'x'
        for (int i = 0, count = CHARTRIPLE_COUNT - givenCharCount; i < count; ++i) {
            ret.append('O');
        }

        ret.setLength(CHARTRIPLE_COUNT);

        return ret.toString();
    }

    // - Members ---------------------------------------------------------------

    protected IImageTransformer m_transformer = null;

    protected InputStream m_imageInputStm = null;

    protected ArrayList<Transformation> m_transformationList = new ArrayList<Transformation>();

    protected Object m_source = null;

    protected String m_targetFormat = null;

    protected TransformedImage m_transformedImage = null;

    // -Statics ----------------------------------------------------------------

    public final static int CHARTRIPLE_COUNT = 3;

    // - Inner types -----------------------------------------------------------

    /**
     * {@link Transformation}
     *
     * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
     * @since v7.8.0
     */
    public enum TransformationType {
        NONE,
        SCALE,
        ROTATE,
        CROP,
        COMPRESS
    }

    /**
     * {@link Transformation}
     *
     * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
     * @since v7.8.0
     */
    /**
     * {@link Transformation}
     *
     * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
     * @since v7.8.0
     */
    public class Transformation {

        /**
         * Initializes a new {@link Transformation}.
         */
        public Transformation(TransformationType transformationType, HashMap<String, Object> transformationProperties) {
            super();
            m_transformationType = transformationType;
            m_transformationProperties = transformationProperties;
        }

        /**
         * @return
         */
        public Transformation createCopy() {
            final HashMap<String, Object> copiedProperties = new HashMap<String, Object>(m_transformationProperties.size());

            copiedProperties.putAll(m_transformationProperties);

            return new Transformation(m_transformationType, copiedProperties);
        }

        /**
         * @return
         */
        public TransformationType getTransformationType() {
            return m_transformationType;
        }

        /**
         * @return
         */
        public Map<String, Object> getTransformationProperties() {
            return m_transformationProperties;
        }

        // - Members -----------------------------------------------------------

        /**
         * m_transformationType
         */
        public TransformationType m_transformationType = TransformationType.NONE;

        /**
         * m_transformationProperties
         */
        Map<String, Object> m_transformationProperties = new ConcurrentHashMap<String, Object>(6);
    }
}
