/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.documentconverter;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.RejectedExecutionHandler;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;

/**
 * {@link AsyncConverter}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.8.0
 */
/**
 * {@link AsyncExecutor}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.8.0
 */
public class AsyncExecutor {

    /**
     * {@link AsyncExecutorRunnable}
     *
     * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
     * @since v7.8.0
     */
    protected class AsyncExecutorRunnable implements Runnable {

        /**
         * Initializes a new {@link AsyncExecutorRunnable}.
         * @param jobType
         * @param jobProperties
         */
        protected AsyncExecutorRunnable(@NonNull String jobType, @NonNull HashMap<String, Object> jobProperties) {
            File inputFile = null;
            InputStream inputStm = null; // resource warning for inputStm checked
            File tempInputFile = null;

            m_jobType = jobType;
            m_jobProperties = new HashMap<>(jobProperties.size());
            m_resultProperties = new HashMap<>(8);

            m_jobProperties.putAll(jobProperties);

            // check for valid input source and set at async properties accordingly
            if ((null != (inputFile = (File) jobProperties.get(Properties.PROP_INPUT_FILE))) ||
                (null != (inputStm = (InputStream) jobProperties.get(Properties.PROP_INPUT_STREAM)))) {
                // file oder stream input source
                tempInputFile = ManagerBasics.createTempFile("oxasync", null);

                if (null != tempInputFile) {
                    tempInputFile.deleteOnExit();

                    try {
                        if (null != inputFile) {
                            FileUtils.copyFile(inputFile, tempInputFile);
                        } else {
                            FileUtils.copyInputStreamToFile(inputStm, tempInputFile);
                        }
                    } catch (IOException e) {
                        ManagerBasics.logExcpImpl(m_manager, e);
                    }

                    // replace the original input file with our temp. file
                    m_jobProperties.put(Properties.PROP_INPUT_FILE, tempInputFile);
                    m_jobProperties.remove(Properties.PROP_INPUT_STREAM);

                    m_validInput = true;
                }
            } else if (!StringUtils.isEmpty((String) jobProperties.get(Properties.PROP_REMOTE_CACHE_HASH))) {
                // remote hash input source
                m_validInput = true;
            }

            if (m_validInput) {
                // ensure, that we don't run into an async recursion
                m_jobProperties.remove(Properties.PROP_ASYNC);

                if (ManagerBasics.isLogDebugImpl(m_manager)) {
                    ManagerBasics.logImpl(m_manager, LogType.LOGTYPE_DEBUG, "Documentconverter scheduled asynchronous job", m_jobProperties);
                }
            } else if (ManagerBasics.isLogWarnImpl(m_manager)) {
                ManagerBasics.logImpl(m_manager, LogType.LOGTYPE_WARN, "Documentconverter is not able to execute async. job without file input, stream input or cached hash value set => please set source input property accordingly", null);
            }
        }

        /**
         * @return
         */
        protected boolean isValid() {
            return (m_jobType != null);
        }

        /**
         * @return
         */
        protected boolean isRunning() {
            return m_running;
        }

        protected boolean isValidInput() {
            return m_validInput;
        }

        /**
         *
         */
        protected void clear() {
            if (isValid() && !isRunning()) {
                File tempInputFile = (File) m_jobProperties.remove(Properties.PROP_INPUT_FILE);

                if (null != tempInputFile) {
                    tempInputFile.delete();
                    tempInputFile = null;

                    m_jobType = null;
                    m_jobProperties = m_resultProperties = null;
                }
            }

        }

        /* (non-Javadoc)
         * @see java.lang.Runnable#run()
         */
        @Override
        public void run() {
            m_running = true;

            if (isValidInput()) {
                if (ManagerBasics.isLogDebugImpl(m_manager)) {
                    ManagerBasics.logImpl(m_manager, LogType.LOGTYPE_DEBUG, "Documentconverter started asynchronous conversion",
                        m_jobProperties, new LogData("async_jobs_remaining", Integer.toString(getQueueSize())));
                }

                IOUtils.closeQuietly(m_documentConverter.convert(m_jobType, m_jobProperties, m_resultProperties));

                if (ManagerBasics.isLogDebugImpl(m_manager)) {
                    ManagerBasics.logImpl(m_manager, LogType.LOGTYPE_DEBUG, "Documentconverter finished asynchronous conversion",
                        m_jobProperties, new LogData("async_jobs_remaining", Integer.toString(getQueueSize())));
                }
            }

            m_running = false;
            clear();
        }

        // - Members -------------------------------------------------------

        protected String m_jobType = null;
        protected HashMap<String, Object> m_jobProperties = null;
        protected HashMap<String, Object> m_resultProperties = null;
        protected boolean m_validInput = false;
        protected volatile boolean m_running = false;
    }

    /**
     * Initializes a new {@link AsyncExecutor}.
     * @param manager
     * @param maxThreadCount
     */
    public AsyncExecutor(@NonNull IDocumentConverter converter, @NonNull ManagerBasics manager, int maxThreadCount) {
        super();

        if (maxThreadCount > 0) {
            m_documentConverter = converter;
            m_manager = manager;
            m_requestQueue = new ArrayBlockingQueue<>(MAX_REQUEST_QEUE_SIZE);
            m_requestExecutor = new ThreadPoolExecutor(maxThreadCount, maxThreadCount, 0, TimeUnit.MILLISECONDS, m_requestQueue);

            m_requestExecutor.setRejectedExecutionHandler(new RejectedExecutionHandler() {

                @Override
                public void rejectedExecution(Runnable runnable, ThreadPoolExecutor executor) {
                    if (null != runnable) {
                        ((AsyncExecutorRunnable) runnable).clear();
                    }
                }
            });
        }
    }

    public void terminate() {
        AsyncExecutor.AsyncExecutorRunnable curRunnable = null;

        isTerminated = true;

        while((curRunnable = (AsyncExecutor.AsyncExecutorRunnable) m_requestQueue.poll()) != null) {
            curRunnable.clear();
        }
    }

    /**
     * @param jobType
     * @param jobProperties
     * @param resultProperties
     */
    public void triggerExecution(String jobType, HashMap<String, Object> jobProperties, HashMap<String, Object> resultProperties) {
        if (!isTerminated && (null != m_requestExecutor) && (null != jobType) && (null != jobProperties) && (null != resultProperties)) {
            final AsyncExecutor.AsyncExecutorRunnable asyncRunnable = new AsyncExecutor.AsyncExecutorRunnable(jobType, jobProperties);

            if (asyncRunnable.isValid()) {
                m_requestExecutor.execute(asyncRunnable);
            }
        }
    }

    /**
     * @return The current size of the asynchronous queue,
     *  not counting the currently processed jobs
     */
    public int getQueueSize() {
        return ((null != m_requestQueue) ? m_requestQueue.size() : 0);
    }

    // - Members -----------------------------------------------------------

    static protected final int MAX_REQUEST_QEUE_SIZE = 1024;

    protected IDocumentConverter m_documentConverter = null;
    protected ManagerBasics m_manager = null;
    protected BlockingQueue<Runnable> m_requestQueue = null;
    protected ThreadPoolExecutor m_requestExecutor = null;
    protected boolean isTerminated = false;
}
