/* *
 *    OPEN-XCHANGE legal information
 *
 *    All intellctual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */


package com.openexchange.documentconverter.client.json;

import java.io.InputStream;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;
import javax.servlet.http.HttpServletRequest;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.ajax.fileholder.IFileHolder;
import com.openexchange.ajax.requesthandler.AJAXActionService;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.capabilities.CapabilityService;
import com.openexchange.capabilities.CapabilitySet;
import com.openexchange.config.ConfigurationService;
import com.openexchange.documentconverter.Feature;
import com.openexchange.documentconverter.JobError;
import com.openexchange.documentconverter.NonNull;
import com.openexchange.exception.OXException;
import com.openexchange.file.storage.FileStorageCapability;
import com.openexchange.file.storage.FileStorageFileAccess;
import com.openexchange.file.storage.composition.FolderID;
import com.openexchange.file.storage.composition.IDBasedFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccessFactory;
import com.openexchange.filemanagement.ManagedFile;
import com.openexchange.filemanagement.ManagedFileManagement;
import com.openexchange.groupware.attach.Attachments;
import com.openexchange.guard.api.GuardApi;
import com.openexchange.guard.api.GuardApis;
import com.openexchange.guard.api.Header;
import com.openexchange.mail.MailServletInterface;
import com.openexchange.mail.dataobjects.MailPart;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;
import com.openexchange.tools.session.ServerSession;

/**
 * {@link DocumentConverterAction}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
public class DocumentConverterAction implements AJAXActionService {

    /**
     * Initializes a new {@link DocumentConverterAction}.
     *
     * @param services
     */
    public DocumentConverterAction(ServiceLookup services) {
        super();

        m_services = services;

        if (null != m_services) {
            final ConfigurationService configService = m_services.getService(ConfigurationService.class);

            // read some config entries
            if (null != configService) {
                final int curIntEntry = configService.getIntProperty("com.openexchange.documentconverter.maxSourceFileSizeMB", Integer.MIN_VALUE);

                // ENGINE_MAX_SOURCEFILESIZE_MB
                if (curIntEntry != Integer.MIN_VALUE) {
                    m_maxSourceFileSizeMB = curIntEntry;
                }
            }

            m_managedFileManagement = m_services.getService(ManagedFileManagement.class);
        }
    }

    @Override
    public AJAXRequestResult perform(AJAXRequestData requestData, ServerSession session) {
        return null;
    }

    /**
     * @param e
     */
    void logException(Exception e) {
        final Logger LOG = LoggerFactory.getLogger(this.getClass());

        if (null != LOG) {
            LOG.error(((null != e) && (null != e.getMessage())) ? e.getMessage() : "Unknown exception occured");
        }
    }

    /**
     * @param session
     * @return
     */
    boolean isFeatureSupportedForUser(Session session, Feature feature) {
        boolean ret = false;

        if (null != session) {
            final CapabilityService capService = m_services.getService(CapabilityService.class);

            try {
                @SuppressWarnings("unused") final CapabilitySet capSet = (null != capService) ? capService.getCapabilities(session) : null;

                /* test prototype
                if ((null != capSet) && capSet.contains("capabilityTestString")) {
                    ret = true;
                }
                */
            } catch (final OXException e) {
                logException(e);
            }
        }

        return ret;
    }

    /**@param documentId
     * @param request
     * @param session
     * @param jobError
     * @return
     */
    protected ManagedFile getManagedFileForRequestDocument(@NonNull AJAXRequestData request, @NonNull ServerSession session, @NonNull JobError[] jobError) {
        final String documentId = getDocumentId(request);
        ManagedFile ret = null;

        try {
            ret = m_managedFileManagement.contains(documentId) ? m_managedFileManagement.getByID(documentId) : null;
        } catch (final OXException e) {
            logException(e);
        }

        if (null == ret) {
            final String fileId = request.getParameter("id");
            final String source = request.getParameter("source");
            final String attached = request.getParameter("attached");
            final String module = request.getParameter("module");
            final String folderId = request.getParameter("folder_id");
            final String version = request.getParameter("version");
            final String guardUrl = request.getParameter("guardUrl");
            final boolean mailCheck = ((attached != null) && (folderId != null) && (attached.length() > 0));
            final boolean taskCheck = mailCheck && (module != null);
            MailServletInterface mailInterface = null;
            InputStream documentStm = null;

            try {
                if ((null != source) && source.equals("mail") && mailCheck) {
                    mailInterface = MailServletInterface.getInstance(session);
                    final MailPart mailPart = mailInterface.getMessageAttachment(folderId, fileId, attached, false);

                    if (null != mailPart) {
                        documentStm = mailPart.getInputStream();
                    }
                } else if ((null != source) && (source.equals("tasks") || source.equals("calendar") || source.equals("contacts")) && taskCheck) {
                    documentStm = Attachments.getInstance().getAttachedFile(
                            session,
                            Integer.parseInt(folderId),
                            Integer.parseInt(attached),
                            Integer.parseInt(module),
                            Integer.parseInt(fileId),
                            session.getContext(),
                            session.getUser(),
                            session.getUserConfiguration());
                } else if ((null != source) && (source.equals("guard")) && (null != guardUrl) && (guardUrl.length() > 0)) {
                    final GuardApi guardApi = m_services.getOptionalService(GuardApi.class);

                    if (null != guardApi) {
                        final HttpServletRequest httpServletRequest = request.optHttpServletRequest();

                        if (null != httpServletRequest) {
                            final String userAgent = httpServletRequest.getHeader("User-Agent");

                            documentStm = guardApi.requestSessionSensitiveResource(
                                    parseQueryStringFromUrl(guardUrl),
                                    session, GuardApis.extractCookiesFrom(httpServletRequest, session),
                                    (null == userAgent) ?
                                            Collections.<Header> emptyList() :
                                                Collections.singletonList(new Header("User-Agent", userAgent)));
                        }
                    }
                } else {
                    final IDBasedFileAccessFactory fileFactory = m_services.getService(IDBasedFileAccessFactory.class);
                    final IDBasedFileAccess fileAccess = (null != fileFactory) ? fileFactory.createAccess(session) : null;

                    if (null != fileAccess) {
                        String versionToUse = version;

                        // #46555 map version String("null") to null
                        if ((null != versionToUse) && versionToUse.equalsIgnoreCase("null")) {
                            versionToUse = FileStorageFileAccess.CURRENT_VERSION;
                        }

                        try {
                            // #46555 check for versioning support
                            if (FileStorageFileAccess.CURRENT_VERSION != versionToUse) {
                                final FolderID folderIdentifier = new FolderID(folderId);

                                if (!fileAccess.supports(folderIdentifier.getService(), folderIdentifier.getAccountId(), FileStorageCapability.FILE_VERSIONS)) {
                                    versionToUse = FileStorageFileAccess.CURRENT_VERSION;
                                }
                            }
                        } catch (Exception e) {
                            logException(e);
                        }

                        documentStm = fileAccess.getDocument(fileId, versionToUse);
                    }
                }
            } catch (final OXException e) {
                logException(e);
            }

            if (null != documentStm) {
                try {
                    ret = m_managedFileManagement.createManagedFile(documentId, documentStm);
                } catch (final OXException e) {
                    logException(e);
                } finally {
                    IOUtils.closeQuietly(documentStm);
                    documentStm = null;
                }
            }

            IOUtils.closeQuietly(mailInterface);
        }

        return ret;
    }

    /**
     * @param request
     * @param fileHolder
     * @return
     */
    AJAXRequestResult getRequestResult(@NonNull AJAXRequestData request, IFileHolder fileHolder, JSONObject jsonObject, JobError jobError) {
        AJAXRequestResult ret = null;

        if ((jobError == JobError.NONE) && (null != fileHolder)) {
            request.setFormat("file");
            ret = new AJAXRequestResult(fileHolder, "file");
        } else {
            final JSONObject jsonResult = (null != jsonObject) ? jsonObject : new JSONObject();

            try {
                String errorCause = null;

                switch (jobError) {
                case NONE: {
                    break;
                }

                case TIMEOUT: {
                    errorCause = "timeout";
                    break;
                }

                case PASSWORD: {
                    errorCause = "passwordProtected";
                    break;
                }

                case NO_CONTENT: {
                    errorCause = "noContent";
                    break;
                }

                case MAX_SOURCESIZE: {
                    errorCause = "maxSourceSizeExceeded";
                    jsonResult.put("MaxSourceSizeMB", "" + m_maxSourceFileSizeMB);
                    break;
                }

                case GENERAL:
                default: {
                    errorCause = "filterError";
                    break;
                }
                }

                if (null != errorCause) {
                    jsonResult.put("cause", errorCause);
                }
            } catch (final JSONException e) {
                logException(e);
            }

            request.setFormat("json");
            ret = new AJAXRequestResult(jsonResult, "json");
        }

        return ret;
    }

    /**
     * Parses a query string from given URL.
     *
     * @param uri The URI string to be parsed
     * @return The parsed parameters or <code>null</code>
     */

    Map<String, String> parseQueryStringFromUrl(String uri) {
        if (StringUtils.isEmpty(uri)) {
            return null;
        }
        try {
            final String query = new java.net.URI(uri).getQuery();
            if (null != query) {
                return parseQueryString(query);
            }
        } catch (final Exception e) {
            logException(e);
        }
        return null;
    }

    String getDefaultEncoding() {
        final ConfigurationService configurationService = m_services.getOptionalService(ConfigurationService.class);
        return null == configurationService ? "UTF-8" : configurationService.getProperty("DefaultEncoding", "UTF-8").trim();
    }

    private static final java.util.regex.Pattern PATTERN_SPLIT = java.util.regex.Pattern.compile("&");

    /**
     * Parses given query string.
     *
     * @param queryStr The query string to be parsed
     * @return The parsed parameters
     */
    Map<String, String> parseQueryString(String queryStr) {
        final String[] pairs = PATTERN_SPLIT.split(queryStr, 0);
        final Map<String, String> map = new LinkedHashMap<>(4);
        for (String pair : pairs) {
            pair = pair.trim();
            if (pair.length() > 0) {
                // Look-up character '='
                final int pos = pair.indexOf('=');
                if (pos >= 0) {
                    map.put(pair.substring(0, pos), pair.substring(pos + 1));
                } else {
                    map.put(pair, "");
                }
            }
        }
        return map;
    }

    /**
     * @param request
     * @param fileHolder
     * @return
     */
    static AJAXRequestResult getRequestResult(@NonNull AJAXRequestData request, @NonNull IFileHolder fileHolder) {
        request.setFormat("file");
        return new AJAXRequestResult(fileHolder, "file");
    }

    /**
     * @param request
     * @return
     */
    static protected String getDocumentId(@NonNull AJAXRequestData request, String... additionalData) {
        final StringBuilder idBuilder = new StringBuilder();
        final String[][] allParamArrays = { idParamArray, additionalData };

        for (final String[] curParamArray : allParamArrays) {
            if (null != curParamArray) {
                final boolean isParam = curParamArray.equals(idParamArray);

                for (final String curParam : curParamArray) {
                    String curValue = curParam;

                    if (isParam) {
                        curValue = request.getParameter(curParam);

                        // check for non-unique CURRENT_VERSION file version
                        // and create unique version, if necessary
                        if (curParam.equalsIgnoreCase("version") && (FileStorageFileAccess.CURRENT_VERSION == curValue)) {
                            curValue = Long.toString(m_uniqueVersion.decrementAndGet());
                        }

                    }

                    idBuilder.append(StringUtils.isEmpty(curValue) ? "null" : curValue).append('_');
                }
            }
        }

        // remove last dash
        if (idBuilder.length() > 1) {
            idBuilder.setLength(idBuilder.length() - 1);
        }

        return idBuilder.toString();
    }

    /**
     * @param fileName
     * @param newExtension
     * @return
     */
    static protected String getReplacedExtensionFileName(String fileName, @NonNull String newExtension) {
        final int dotPos = StringUtils.isEmpty(fileName) ? -2 : fileName.lastIndexOf('.');
        StringBuffer newFileNameBuffer = new StringBuffer((null != fileName) ? (fileName.length() + 8) : 16);

        if (dotPos > -1) {
            newFileNameBuffer.append(fileName.substring(0, dotPos + 1)).append(newExtension);
        } else {
            newFileNameBuffer.append(StringUtils.isEmpty(fileName) ? "unknown" : fileName).append('.').append(newExtension);
        }

        return newFileNameBuffer.toString();
    }


    // - Statics ---------------------------------------------------------------

    static final private String[] idParamArray = { "uid", "id", "folder_id", "version", "source", "attached", "module" };

    // - Members ---------------------------------------------------------------

    static protected AtomicLong m_uniqueVersion = new AtomicLong(0);

    protected ServiceLookup m_services = null;

    protected ManagedFileManagement m_managedFileManagement = null;

    protected long m_maxSourceFileSizeMB = 32;
}

