/* *
 *    OPEN-XCHANGE legal information
 *
 *    All intellctual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */


package com.openexchange.documentconverter.client.json;

import java.io.InputStream;
import java.util.HashMap;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.ajax.container.DelegateFileHolder;
import com.openexchange.ajax.container.FileHolder;
import com.openexchange.ajax.fileholder.IFileHolder;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.documentconverter.IDocumentConverter;
import com.openexchange.documentconverter.JobError;
import com.openexchange.documentconverter.JobPriority;
import com.openexchange.documentconverter.NonNull;
import com.openexchange.documentconverter.Properties;
import com.openexchange.filemanagement.ManagedFile;
import com.openexchange.groupware.ldap.User;
import com.openexchange.server.ServiceLookup;
import com.openexchange.tools.session.ServerSession;

/**
 * {@link ConvertDocumentAction}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */

/*
 * MH/KA compilefix for buildsystem
 * @Action(method = RequestMethod.GET, name = "convertdocument", description = "Import a document either as HTML from an infostore file.",
 * parameters = {
 * @Parameter(name = "session", description = "A session ID previously obtained from the login module."),
 * @Parameter(name = "id", description = "Object ID of the requested infoitem."),
 * @Parameter(name = "folder_id", description = "Folder ID of the requested infoitem."),
 * @Parameter(name = "uid", description = "The unique id of the client application."),
 * @Parameter(name = "version", optional=true, description =
 * "If present, the infoitem data describes the given version. Otherwise the current version is returned."),
 * @Parameter(name = "convert_format", optional=true, description =
 * "If present, this parameter contains the format of the returned object. If this value is set to 'html', the document is returned in HTML format."
 * ),
 * @Parameter(name = "filename", optional=true, description =
 * "If present, this parameter contains the name of the infostore item for individual use by the backend.") }, responseDescription =
 * "Response with timestamp: An object containing the converted data of the requested infoitem. The data is given back as JSON object in HTML format."
 * )
 */
class ConvertDocumentAction extends DocumentConverterAction {

	/**
	 * Initializes a new {@link ConvertDocumentAction}.
	 *
	 * @param services
	 */
	public ConvertDocumentAction(ServiceLookup services) {
		super(services);
	}

	/*
	 * (non-Javadoc)
	 * @see com.openexchange.ajax.requesthandler.AJAXActionService#perform(com.openexchange.ajax.requesthandler.AJAXRequestData,
	 * com.openexchange.tools.session.ServerSession)
	 */
	@Override
	public AJAXRequestResult perform(AJAXRequestData request, ServerSession session) {
		AJAXRequestResult requestResult = new AJAXRequestResult(null, "json");

		if ((null != request) && (null != session)) {
			String convertAction = request.getParameter("convert_action");
			String priority = request.getParameter("convert_priority");

			priority = (null != priority) ? priority.toLowerCase() : "";
			final JobPriority jobPriority = priority.equals("background") ? JobPriority.BACKGROUND
					: (priority.equals("low") ? JobPriority.LOW
							: (priority.equals("medium") ? JobPriority.MEDIUM
									: (priority.equals("high") ? JobPriority.HIGH : JobPriority.highest())));

			if (null != convertAction) {
				convertAction = convertAction.toLowerCase();

				if (convertAction.equals("beginconvert") || convertAction.equals("getpage") || convertAction.equals("endconvert")) {
					requestResult = performConversionStep(request, session, convertAction, jobPriority);
				}
			}
		}

		return requestResult;
	}

	/**
	 * @param request
	 * @param session
	 * @return the AJAX request result
	 */
	@SuppressWarnings("resource")
	private AJAXRequestResult performConversionStep(AJAXRequestData request, ServerSession session, String convertAction, JobPriority convertPriority) {
		final IDocumentConverter manager = m_services.getService(IDocumentConverter.class);
		JobError jobError[] = { JobError.GENERAL };
		JSONObject jsonResult = new JSONObject();
		IFileHolder resultFileHolder = null;

		if (null != manager) {
			if (convertAction.equals("beginconvert")) {
				performBeginConvert(request, session, manager, convertPriority, jsonResult, jobError);
			} else if (convertAction.equals("getpage")) {
				resultFileHolder = performGetPage(request, session, manager, jsonResult, jobError);
			} else if (convertAction.equals("endconvert")) {
				performEndConvert(request, session, manager, jsonResult, jobError);
			}
		}

		return getRequestResult(request, resultFileHolder, jsonResult, jobError[0]);
	}

	/**
	 * @param request
	 * @param session
	 * @param manager
	 * @param convertPriority
	 * @param jsonResult
	 * @param jobError
	 */
	private void performBeginConvert(@NonNull AJAXRequestData request, @NonNull ServerSession session, @NonNull IDocumentConverter manager, @NonNull JobPriority convertPriority, @NonNull JSONObject jsonResult, @NonNull JobError[] jobError) {
		final HashMap<String, Object> jobProperties = new HashMap<>(8);
		final HashMap<String, Object> resultProperties = new HashMap<>(8);
		final ManagedFile managedFile = getManagedFileForRequestDocument(request, session, jobError);

		if (null != managedFile) {
			final String fileName = request.getParameter("filename");

			if (null != fileName) {
				jobProperties.put(Properties.PROP_INPUT_TYPE, FilenameUtils.getExtension(fileName).trim().toLowerCase());
				jobProperties.put(Properties.PROP_INFO_FILENAME, fileName);
			}

			// set the input file and other properties and start first conversion step
			jobProperties.put(Properties.PROP_INPUT_FILE, managedFile.getFile());
			jobProperties.put(Properties.PROP_PRIORITY, convertPriority);

			// set locale, if set and if different from 'en*'
			final User sessionUser = session.getUser();

			if (null != sessionUser) {
				final String prefLanguage = sessionUser.getPreferredLanguage();

				if ((null != prefLanguage) && (prefLanguage.length() > 0)) {
					jobProperties.put(Properties.PROP_LOCALE, prefLanguage);
				}
			}

			// delete initially created temp. file for document
			final String jobIdStr = manager.beginPageConversion("html", jobProperties, resultProperties);

			// set response parameters at JSON result
			int pageCount = resultProperties.containsKey(Properties.PROP_RESULT_PAGE_COUNT) ? ((Integer) resultProperties.get(Properties.PROP_RESULT_PAGE_COUNT)).intValue() : 0;

			if (resultProperties.containsKey(Properties.PROP_RESULT_ERROR_CODE)) {
				jobError[0] = JobError.fromErrorCode(((Integer) resultProperties.get(Properties.PROP_RESULT_ERROR_CODE)).intValue());
			}

			try {
				jsonResult.put("jobID", jobIdStr);
				jsonResult.put("pageCount", pageCount);
			} catch (JSONException e) {
				logException(e);
			}

			if ((JobError.NONE == jobError[0]) && ((jobIdStr == null) || (pageCount < 1))) {
				jobError[0] = JobError.GENERAL;
			}

			if (JobError.NONE != jobError[0]) {
				manager.endPageConversion(jobIdStr, jobProperties, resultProperties);
			}
		} else {
			jobError[0] = JobError.GENERAL;
		}
	}

	/**
	 * @param request
	 * @param session
	 * @param manager
	 * @param jsonResult
	 * @param jobError
	 * @return
	 */
	@SuppressWarnings("resource") // checked
	private IFileHolder performGetPage(@NonNull AJAXRequestData request, @NonNull ServerSession session, @NonNull IDocumentConverter manager, @NonNull JSONObject jsonResult, @NonNull JobError[] jobError) {
		IFileHolder fileHolderResult = null;
		final HashMap<String, Object> jobProperties = new HashMap<>(8);
		final HashMap<String, Object> resultProperties = new HashMap<>(8);
		final String jobIdStr = request.getParameter("job_id");
		final String pageNumberStr = request.getParameter("page_number");
		final int pageNumber = (null != pageNumberStr) ? Integer.parseInt(pageNumberStr) : 0;
		final String fileName = request.getParameter("filename");
		String targetFormat = request.getParameter("target_format");
		targetFormat = (null == targetFormat) ? "" : targetFormat.toLowerCase();
		final boolean isPngTarget = targetFormat.equals("png");
		final boolean isJpegTarget = !isPngTarget && (targetFormat.equalsIgnoreCase("jpg") || targetFormat.equalsIgnoreCase("jpeg"));
		final boolean isXMLTarget = !isPngTarget && !isJpegTarget;
		InputStream resultStm = null;

		if (null != fileName) {
			jobProperties.put(Properties.PROP_INFO_FILENAME, fileName);
			jobProperties.put(Properties.PROP_INPUT_TYPE, FilenameUtils.getExtension(fileName).trim().toLowerCase());
		}

		if ((jobIdStr != null) && (pageNumber > 0)) {

			if (isXMLTarget) {
				jobProperties.put(Properties.PROP_MIME_TYPE, "image/xml+svg");
				jobProperties.put(Properties.PROP_PIXEL_WIDTH, new Integer(-1));
				jobProperties.put(Properties.PROP_PIXEL_HEIGHT, new Integer(-1));
			} else {
				final String targetWidthStr = request.getParameter("target_width");
				final String targetHeightStr = request.getParameter("target_height");
				final String targetZoomStr = request.getParameter("target_zoom");

				jobProperties.put(Properties.PROP_MIME_TYPE, isPngTarget ? "image/png" : "image/jpeg");
				jobProperties.put(Properties.PROP_PIXEL_WIDTH, new Integer((null != targetWidthStr) ? targetWidthStr : "-1"));
				jobProperties.put(Properties.PROP_PIXEL_HEIGHT, new Integer((null != targetHeightStr) ? targetHeightStr : "-1"));
				jobProperties.put(Properties.PROP_PIXEL_ZOOM, new Double((null != targetZoomStr) ? targetZoomStr : "1.0"));
			}

			// get the input stream for the requested page
			resultStm = manager.getConversionPage(jobIdStr, pageNumber, jobProperties, resultProperties);
		}

		// prepare Ajax result object
		if (null != resultStm) {
			// check return type and result mime type and return either as file or as JSON object
			final String returnType = request.getParameter("returntype");

			if (((null != returnType) && returnType.equalsIgnoreCase("file")) || isPngTarget || isJpegTarget) {
				final FileHolder fileHolder = new FileHolder(resultStm, -1, (String) jobProperties.get(Properties.PROP_MIME_TYPE), null);

				// use a delegate file holder to ensure correct closing of the underlying stream
				fileHolderResult = new DelegateFileHolder(fileHolder).setStream(resultStm, -1);

				request.setFormat("file");
				request.setProperty("target_width", jobProperties.get(Properties.PROP_PIXEL_WIDTH));
				request.setProperty("target_height", jobProperties.get(Properties.PROP_PIXEL_HEIGHT));

				jobError[0] = JobError.NONE;
			} else {
				try {
					jsonResult.put("HTMLPages", IOUtils.toString(resultStm));
					jsonResult.put("MimeType", jobProperties.get(Properties.PROP_MIME_TYPE));
					jsonResult.put("Width", jobProperties.get(Properties.PROP_PIXEL_WIDTH));
					jsonResult.put("Height", jobProperties.get(Properties.PROP_PIXEL_HEIGHT));

					jobError[0] = JobError.NONE;
				} catch (Exception e) {
					logException(e);
				}
				finally {
					IOUtils.closeQuietly(resultStm);
				}
			}
		}

		return fileHolderResult;
	}

	/**
	 * @param request
	 * @param session
	 * @param manager
	 * @param jsonResult
	 * @param jobError
	 */
	static private void performEndConvert(@NonNull AJAXRequestData request, @NonNull ServerSession session, @NonNull IDocumentConverter manager, @NonNull JSONObject jsonResult, @NonNull JobError[] jobError) {
		final String jobIdStr = request.getParameter("job_id");
		final HashMap<String, Object> jobProperties = new HashMap<>(2);
		final HashMap<String, Object> resultProperties = new HashMap<>(8);

		if (jobIdStr != null) {
			manager.endPageConversion(jobIdStr, jobProperties, resultProperties);
			jobError[0] = JobError.NONE;
		}
	}
}
