/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.documentconverter;

import java.io.BufferedOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.Serializable;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.StringTokenizer;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * {@link TransferHelper}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.6.0
 */
public class TransferHelper {

    /**
     * Initializes a new {@link TransferHelper}.
     */
    public TransferHelper() {
        super();
    }

    /**
     * @param remoteUrl
     * @param transferObj
     */
    public static void postRequest(ManagerBasics manager, URL remoteUrl, ServerType serverType, TransferObject<? extends Serializable> transferObj, TransferResponseProcessor responseProcessor) {
        if ((null != remoteUrl) && (null != transferObj)) {
            HttpURLConnection connection = null;
            DataOutputStream dataOutputStm = null;
            InputStream inputStm = null;
            final String queryString = transferObj.getQuery();
            final String callCookie = transferObj.getCookie();

            try {
                if ((null != queryString) && (null != (connection = (HttpURLConnection) remoteUrl.openConnection()))) {
                    // initialize connection
                    connection.setRequestMethod("POST");
                    connection.setRequestProperty("Content-Type", "application/x-www-form-urlencoded");
                    connection.setRequestProperty("Content-Length", Integer.toString(queryString.length() + 1));
                    connection.setUseCaches(false);
                    connection.setDoOutput(true);
                    connection.setAllowUserInteraction(false);
                    connection.setConnectTimeout(ManagerBasics.getHTTPConnectTimeoutMilliseconds());
                    connection.setReadTimeout(ManagerBasics.getHTTPReadTimeoutMilliseconds());

                    // set Cookie if requested
                    if (null != transferObj.getCookie()) {
                        connection.setRequestProperty("Cookie", callCookie);
                    }

                    connection.connect();

                    // write to output stream
                    dataOutputStm = new DataOutputStream(connection.getOutputStream());
                    dataOutputStm.writeBytes("?");
                    dataOutputStm.writeBytes(queryString);

                    dataOutputStm.flush();
                    dataOutputStm.close();
                    dataOutputStm = null;

                    inputStm = connection.getInputStream();

                    if ((null != inputStm) && (null != responseProcessor)) {
                        final int responseCode = connection.getResponseCode();
                        final String converterCookie = getConverterCookie(connection);

                        responseProcessor.processResponse(responseCode, inputStm, converterCookie);
                    }
                } else if (LOG.isTraceEnabled()) {
					LOG.trace("Documentconverter could not open remote HTTPConnection: " + remoteUrl.toString());
                }
            } catch (Exception excp) {
                if (excp instanceof IOException) {
                    manager.resetRemoteConnection(serverType);
                } else {
					LOG.error(excp.getMessage());
                }
            } finally {
                IOUtils.closeQuietly(inputStm);
                IOUtils.closeQuietly(dataOutputStm);

                if (null != connection) {
                    connection.disconnect();
                }
            }
        }
    }

    /**
     * @param remoteUrl
     * @param transferObj
     */
    public static void postFormData(ManagerBasics manager, URL remoteUrl, ServerType serverType, TransferObject<? extends Serializable> transferObj, TransferResponseProcessor responseProcessor) {
        if ((null != remoteUrl) && (null != transferObj)) {
            HttpURLConnection connection = null;
            PrintWriter printWriter = null;
            BufferedOutputStream bufferedOutputStm = null;
            InputStream inputStm = null;

            try {
                if ((null != (connection = (HttpURLConnection) remoteUrl.openConnection()))) {
                    final String boundary = Long.toHexString(System.currentTimeMillis());
                    final String partSeparator = FDATA_BDRY + boundary + FDATA_CRLF;
                    final String callCookie = transferObj.getCookie();

                    // initialize connection
                    connection.setRequestMethod("POST");
                    connection.setRequestProperty("Content-Type", "multipart/form-data; boundary=" + boundary);
                    connection.setDoOutput(true);
                    connection.setUseCaches(false);
                    connection.setAllowUserInteraction(false);
                    connection.setConnectTimeout(ManagerBasics.getHTTPConnectTimeoutMilliseconds());
                    connection.setReadTimeout(ManagerBasics.getHTTPReadTimeoutMilliseconds());

                    // set Cookie if requested
                    if (null != transferObj.getCookie()) {
                        connection.setRequestProperty("Cookie", callCookie);
                    }

                    // perform the connection
                    connection.connect();

                    bufferedOutputStm = new BufferedOutputStream(connection.getOutputStream());
                    printWriter = new PrintWriter(new OutputStreamWriter(bufferedOutputStm, "UTF-8"), false);

                    writeParamMap(printWriter, transferObj, partSeparator);
                    printWriter.flush();

                    writeFileContent(printWriter, bufferedOutputStm, transferObj, partSeparator);
                    printWriter.flush();

                    // mark end of multipart/form-data
                    printWriter.append(FDATA_BDRY + boundary + FDATA_BDRY).append(FDATA_CRLF);

                    printWriter.flush();
                    printWriter.close();
                    printWriter = null;

                    bufferedOutputStm.flush();
                    bufferedOutputStm.close();
                    bufferedOutputStm = null;

                    inputStm = connection.getInputStream();

                    if ((null != inputStm) && (null != responseProcessor)) {
                        final int responseCode = connection.getResponseCode();
                        final String converterCookie = getConverterCookie(connection);

                        responseProcessor.processResponse(responseCode, inputStm, converterCookie);
                    }
                } else if (LOG.isTraceEnabled()) {
                    LOG.trace("Documentconverter could not open remote HTTPConnection: " + remoteUrl.toString());
                }
            } catch (Exception excp) {
                if (excp instanceof IOException) {
                    manager.resetRemoteConnection(serverType);
                } else {
					LOG.error(excp.getMessage());
                }
            } finally {
                IOUtils.closeQuietly(inputStm);
                IOUtils.closeQuietly(printWriter);
                IOUtils.closeQuietly(bufferedOutputStm);

                if (null != connection) {
                    connection.disconnect();
                }
            }
        }
    }

    // - Implementation --------------------------------------------------------

    /**
     * @param connection
     * @param responseProcessor
     */
    protected static String getConverterCookie(HttpURLConnection connection) {
        // extract cookies
        String converterCookie = null;
        int i = 0;

        while (true) {
            final String curHeaderFieldName = connection.getHeaderFieldKey(++i);

            if (null != curHeaderFieldName) {
                if (curHeaderFieldName.equalsIgnoreCase("set-cookie")) {
                    final String headerField = connection.getHeaderField(i);
                    final StringTokenizer tokenizer = new StringTokenizer(headerField, ";");

                    while (tokenizer.hasMoreTokens()) {
                        final String token = tokenizer.nextToken();
                        final int assignPos = token.indexOf('=');

                        if (-1 != assignPos) {
                            final String cookieKey = token.substring(0, assignPos);

                            if ((null != cookieKey) && cookieKey.trim().equalsIgnoreCase("jsessionid")) {
                                converterCookie = headerField;
                                break;
                            }
                        }
                    }

                    break;
                }
            } else {
                break;
            }
        }

        return converterCookie;
    }

    /**
     * @param printWriter
     * @param transferObject
     * @param partSeparator
     */
    private static void writeParamMap(PrintWriter printWriter, TransferObject<? extends Serializable> transferObject, String partSeparator) {
        if (null != transferObject.getParamMap()) {
            for (String key : transferObject.getParamMap().keySet()) {
                printWriter.append(partSeparator);
                printWriter.append("Content-Disposition: form-data; name=" + key).append(FDATA_CRLF).append(FDATA_CRLF);

                printWriter.append(transferObject.getParamMap().get(key)).append(FDATA_CRLF).flush();
            }
        }
    }

    /**
     * @param printWriter
     * @param objOutputStm
     * @param transferObject
     * @param partSeparator
     */
    private static void writeFileContent(PrintWriter printWriter, OutputStream outputStm, TransferObject<? extends Serializable> transferObject, String partSeparator) throws IOException {
        final Object serialObject = transferObject.getSerialObject();

        if (null != serialObject) {
            // write content part header
            printWriter.append(partSeparator);
            printWriter.append(
                "Content-Disposition: form-data; name=\"file\"; filename=" + ((null != transferObject.getFilename()) ? transferObject.getFilename() : "file.bin")).append(
                FDATA_CRLF);
            printWriter.append(
                "Content-Type: " + ((null != transferObject.getMimeType()) ? transferObject.getMimeType() : "application/octet-stream")).append(
                FDATA_CRLF);
            printWriter.append("Content-Transfer-Encoding: binary").append(FDATA_CRLF).append(FDATA_CRLF).flush();

            final ObjectOutputStream objectOutputStm = new ObjectOutputStream(new BufferedOutputStream(outputStm));
            objectOutputStm.writeObject(serialObject);
            objectOutputStm.flush();

            printWriter.append(FDATA_CRLF).flush();
        }
    }

    // - Members

    static private Logger LOG = LoggerFactory.getLogger(TransferHelper.class);

    static final protected String FDATA_BDRY = "--";

    static final protected String FDATA_CRLF = "\r\n";
}
