/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.realtime.doc;

import java.util.Date;

import com.openexchange.office.tools.doc.DocumentFormat;
import com.openexchange.office.tools.doc.DocumentMetaData;
import com.openexchange.office.tools.files.DocFileHelper;
import com.openexchange.office.tools.monitoring.CloseEvent;
import com.openexchange.office.tools.monitoring.CloseType;
import com.openexchange.office.tools.monitoring.DocumentEvent;
import com.openexchange.office.tools.monitoring.ErrorType;
import com.openexchange.office.tools.monitoring.OpenEvent;
import com.openexchange.office.tools.monitoring.OpenType;
import com.openexchange.office.tools.monitoring.OperationsEvent;
import com.openexchange.office.tools.monitoring.OperationsType;
import com.openexchange.office.tools.monitoring.SaveEvent;
import com.openexchange.office.tools.monitoring.SaveType;
import com.openexchange.office.tools.monitoring.Statistics;

/**
 * Helper class to log document events for Office documents monitoring.
 * {@link DocumentEventHelper}
 *
 * @author <a href="mailto:firstname.lastname@open-xchange.com">Carsten Driesner</a>
 * @since v7.8.0
 */
public class DocumentEventHelper {

    static private final long MAX_TIME_FOR_BINARY_CONVERSION_CONSIDERATION = 30000; // time in milliseconds

    /**
     * Adds a open error event to the monitoring data.
     *
     * @param fileHelper The current file helper instance.
     */
    public static void addOpenErrorEvent(final DocFileHelper fileHelper) {
        if (null != fileHelper) {
            Statistics.handleDocumentEvent(new DocumentEvent(fileHelper.getDocumentType(), ErrorType.LOAD));
        }
    }

    /**
     * Adds a open event to the monitoring data.
     *
     * @param fileHelper The current file helper instance.
     */
    public static void addOpenEvent(final DocFileHelper fileHelper, final DocumentMetaData docMetaData) {
        if (null != docMetaData) {
            Statistics.handleDocumentEvent(getOpenEvent(fileHelper, docMetaData));
        }
    }

    /**
     * Adds a save event to the monitoring data.
     *
     * @param fileHelper The current file helper instance.
     */
    public static void addSaveEvent(final DocFileHelper fileHelper, SaveType type) {
        if (null != fileHelper) {
            Statistics.handleDocumentEvent(new SaveEvent(fileHelper.getDocumentType(), type));
        }
    }

    /**
     * Adds a close event to the monitoring data.
     *
     * @param fileHelper The current file helper instance.
     */
    public static void addCloseEvent(final DocFileHelper fileHelper) {
        if (null != fileHelper) {
            Statistics.handleDocumentEvent(new CloseEvent(fileHelper.getDocumentType(), CloseType.CLOSE));
        }
    }

    /**
     * Adds a timeout on close event to the monitoring data.
     *
     * @param fileHelper The current file helper instance.
     */
    public static void addTimeoutEvent(final DocFileHelper fileHelper) {
        if (null != fileHelper) {
            Statistics.handleDocumentEvent(new CloseEvent(fileHelper.getDocumentType(), CloseType.TIMEOUT));
        }
    }

    /**
     * Adds a distributed document event to the monitoring data.
     *
     * @param fileHelper The current file helper instance.
     * @param otherClientCount The number of clients connected with this document.
     */
    public static void addDistributedEvent(final DocFileHelper fileHelper, int otherClientCount) {
        if ((null != fileHelper) && (otherClientCount > 0)) {
            Statistics.handleDocumentEvent(new OperationsEvent(
                fileHelper.getDocumentType(),
                OperationsType.DISTRIBUTED,
                otherClientCount));
        }
    }

    /**
     * Adds an incoming event to the monitoring data.
     *
     * @param fileHelper The current file helper instance.
     */
    public static void addIncomingEvent(final DocFileHelper fileHelper) {
        if (null != fileHelper) {
            Statistics.handleDocumentEvent(new OperationsEvent(fileHelper.getDocumentType(), OperationsType.INCOMING));
        }
    }

    /**
     * Provides an OpenEvent instance to be used for administrative purposes like VM surveillance.
     *
     * @return An OpenEvent instance describing the current document.
     */
    private static final OpenEvent getOpenEvent(final DocFileHelper fileHelper, final DocumentMetaData docMetaData) {
        OpenEvent openEvent = null;

        if (null != fileHelper) {
            final DocumentFormat documentFormat = fileHelper.getDocumentFormat();

            if (null != documentFormat) {
                OpenType documentOpenType = OpenType.NONE;
                final Date docCreated = docMetaData.getCreated();
                final long docExistsForMilliseconds = (null != docCreated) ? (System.currentTimeMillis() - docCreated.getTime()) : Long.MAX_VALUE;

                // mark document type as binary only if it has just been converted
                if (docMetaData.isBinaryConverted() && (docExistsForMilliseconds <= MAX_TIME_FOR_BINARY_CONVERSION_CONSIDERATION)) {
                    documentOpenType = OpenType.BINARY;
                } else {
                    switch (documentFormat) {
                    case DOCX:
                    case XLSX:
                    case PPTX: {
                        documentOpenType = OpenType.MSX;
                        break;
                    }

                    case ODT:
                    case ODS:
                    case ODP: {
                        documentOpenType = OpenType.ODF;
                        break;
                    }

                    case NONE:
                    case ODG:
                    default: {
                        documentOpenType = OpenType.NONE;
                        break;
                    }
                    }
                }

                if (OpenType.NONE != documentOpenType) {
                    openEvent = new OpenEvent(fileHelper.getDocumentType(), documentOpenType, docMetaData.getFileSize());
                }
            }
        }

        return openEvent;
    }
}
