/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.pptx.operations;

import javax.xml.bind.JAXBException;

import org.docx4j.dml.CTGroupShapeProperties;
import org.docx4j.dml.CTGroupTransform2D;
import org.docx4j.dml.CTNonVisualDrawingProps;
import org.docx4j.dml.CTPoint2D;
import org.docx4j.dml.CTPositiveSize2D;
import org.docx4j.jaxb.Context;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.exceptions.PartUnrecognisedException;
import org.docx4j.openpackaging.packages.PresentationMLPackage;
import org.docx4j.openpackaging.parts.PresentationML.MainPresentationPart;
import org.docx4j.openpackaging.parts.PresentationML.SlideLayoutPart;
import org.docx4j.openpackaging.parts.PresentationML.SlidePart;
import org.docx4j.openpackaging.parts.PresentationML.ViewPropertiesPart;
import org.docx4j.openpackaging.parts.relationships.Namespaces;
import org.docx4j.openpackaging.parts.relationships.RelationshipsPart;
import org.docx4j.openpackaging.parts.relationships.RelationshipsPart.AddPartBehaviour;
import org.docx4j.relationships.Relationship;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.pptx4j.Pptx4jException;
import org.pptx4j.pml.CTNormalViewPortion;
import org.pptx4j.pml.CTNormalViewProperties;
import org.pptx4j.pml.CommonSlideData;
import org.pptx4j.pml.GroupShape;
import org.pptx4j.pml.GroupShape.NvGrpSpPr;
import org.pptx4j.pml.ObjectFactory;
import org.pptx4j.pml.Sld;
import org.pptx4j.pml.ViewPr;

import com.openexchange.office.ooxml.components.Component;
import com.openexchange.office.ooxml.components.IParagraph;
import com.openexchange.office.ooxml.components.Component.SplitMode;
import com.openexchange.office.ooxml.components.Component.Type;
import com.openexchange.office.ooxml.operations.ApplyOperationHelper;
import com.openexchange.office.ooxml.pptx.PptxOperationDocument;
import com.openexchange.office.ooxml.pptx.components.SlideComponent;

public class PptxApplyOperationHelper extends ApplyOperationHelper {

    private final PptxOperationDocument operationDocument;
    private final PresentationMLPackage presentationMLPackage;

    public PptxApplyOperationHelper(PptxOperationDocument _operationDocument) {
        super();

        operationDocument = _operationDocument;
        presentationMLPackage = _operationDocument.getPackage();
    }

    @Override
    public PptxOperationDocument getOperationDocument() {
        return operationDocument;
    }

    public void insertParagraph(JSONArray start, JSONObject attrs)
        throws JAXBException, InvalidFormatException, PartUnrecognisedException, JSONException {

    	Component.getComponent(getOperationDocument().getRootComponent(), start, start.length()-1).insertChildComponent(operationDocument, start.getInt(start.length()-1), attrs, Component.Type.PARAGRAPH);
    }

    public void splitParagraph(JSONArray start)
        throws JSONException {

        ((IParagraph)Component.getComponent(getOperationDocument().getRootComponent(), start, start.length()-1)).splitParagraph(start.getInt(start.length()-1));
    }

    public void mergeParagraph(JSONArray start) {

    	((IParagraph)Component.getComponent(getOperationDocument().getRootComponent(), start, start.length())).mergeParagraph();
    }

    public void insertText(JSONArray start, String text, JSONObject attrs)
        throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException {

    	if(text.length()>0) {
    		final Component component = Component.getComponent(getOperationDocument().getRootComponent(), start, start.length()-1);
    		((IParagraph)component).insertText(operationDocument, start.getInt(start.length()-1), text, attrs);
    	}
    }

    public void delete(JSONArray start, JSONArray end)
        throws JSONException, InvalidFormatException, JAXBException, PartUnrecognisedException {

        if(start==null||start.length()==0)
            return;

        int startComponent = start.getInt(start.length()-1);
        int endComponent = startComponent;
        if(end!=null) {
            if(end.length()!=start.length())
                return;
            endComponent = end.getInt(end.length()-1);
        }
        Component component = Component.getComponent(getOperationDocument().getRootComponent(), start, start.length());
		component.splitStart(startComponent, SplitMode.DELETE);
        component.delete(getOperationDocument(), (endComponent-startComponent)+1);
    }

    public void setAttributes(JSONObject attrs, JSONArray start, JSONArray end)
        throws JAXBException, InvalidFormatException, PartUnrecognisedException, JSONException {

    	if(attrs==null) {
    		return;
    	}
        int startIndex = start.getInt(start.length()-1);
        int endIndex = startIndex;

        if(end!=null) {
            if(end.length()!=start.length())
                return;
            endIndex = end.getInt(end.length()-1);
        }
    	Component component = Component.getComponent(getOperationDocument().getRootComponent(), start, start.length());
		component.splitStart(startIndex, SplitMode.ATTRIBUTE);
        while(component!=null&&component.getComponentNumber()<=endIndex) {
        	if(component.getNextComponentNumber()>=endIndex+1) {
        		component.splitEnd(endIndex, SplitMode.ATTRIBUTE);
        	}
        	component.applyAttrsFromJSON(operationDocument, attrs);
            component = component.getNextComponent();
        }
    }

    public void insertSlide(JSONArray start, JSONObject attrs)
    	throws InvalidFormatException, Pptx4jException, JSONException {

    	final SlidePart newSlidePart = new SlidePart();
    	final Sld newSlide = new Sld();
    	final CommonSlideData commonSlideData = Context.getpmlObjectFactory().createCommonSlideData();
    	newSlide.setCSld(commonSlideData);
    	commonSlideData.setSpTree(createSpTree());
    	newSlidePart.setJaxbElement(newSlide);
    	final MainPresentationPart mainPresentationPart = presentationMLPackage.getMainPresentationPart();
    	mainPresentationPart.addSlide(start.getInt(0), newSlidePart);
    	final SlideLayoutPart slideLayoutPart = (SlideLayoutPart)getOperationDocument().getContextPart();
    	newSlidePart.addTargetPart(slideLayoutPart, AddPartBehaviour.REUSE_EXISTING);
    }

    public void deleteSlide(JSONArray start)
    	throws Pptx4jException, JSONException {

    	final MainPresentationPart mainPresentationPart = presentationMLPackage.getMainPresentationPart();
    	mainPresentationPart.removeSlide(start.getInt(0));
    }

    public GroupShape createSpTree() {
    	final ObjectFactory pmlFactory = Context.getpmlObjectFactory();
    	final org.docx4j.dml.ObjectFactory dmlFactory = Context.getDmlObjectFactory();
    	final GroupShape spTree = pmlFactory.createGroupShape();
    	final NvGrpSpPr nvGrpSpPr = pmlFactory.createGroupShapeNvGrpSpPr();
    	spTree.setNvGrpSpPr(nvGrpSpPr);
    	final CTNonVisualDrawingProps cNvPr = dmlFactory.createCTNonVisualDrawingProps();
    	cNvPr.setId(null);
    	cNvPr.setName("");
    	nvGrpSpPr.setCNvPr(cNvPr);
    	nvGrpSpPr.setCNvGrpSpPr(dmlFactory.createCTNonVisualGroupDrawingShapeProps());
    	nvGrpSpPr.setNvPr(pmlFactory.createNvPr());
    	final CTGroupShapeProperties grpSpPr = dmlFactory.createCTGroupShapeProperties();
    	final CTGroupTransform2D xFrm = dmlFactory.createCTGroupTransform2D();
    	final CTPoint2D off = dmlFactory.createCTPoint2D();
    	off.setX(0);
    	off.setY(0);
    	xFrm.setOff(off);
    	final CTPositiveSize2D ext = dmlFactory.createCTPositiveSize2D();
    	ext.setCx(0);
    	ext.setCy(0);
    	xFrm.setExt(ext);
    	final CTPoint2D chOff = dmlFactory.createCTPoint2D();
    	chOff.setX(0);
    	chOff.setY(0);
    	xFrm.setChOff(chOff);
    	final CTPositiveSize2D chExt = dmlFactory.createCTPositiveSize2D();
    	chExt.setCx(0);
    	chExt.setCy(0);
    	xFrm.setChExt(chExt);
    	grpSpPr.setXfrm(xFrm);
    	spTree.setGrpSpPr(grpSpPr);
    	return spTree;
    }

    public void insertDrawing(JSONArray start, String type, JSONObject attrs)
    	throws UnsupportedOperationException, InvalidFormatException, PartUnrecognisedException, JAXBException, JSONException {

    	Component.Type cType = Component.Type.AC_SHAPE;
    	if(type.equals("IMAGE")) {
    		cType = Component.Type.AC_IMAGE;
    	}
    	else if(type.equals("IMAGE")) {
    		cType = Component.Type.AC_GROUP;
    	}
    	Component.getComponent(getOperationDocument().getRootComponent(), start, start.length()-1).insertChildComponent(operationDocument, start.getInt(start.length()-1), attrs, cType);
    }

    public void setDocumentAttributes(JSONObject attrs)
    	throws InvalidFormatException {

    	final MainPresentationPart mainPresentationPart = presentationMLPackage.getMainPresentationPart();
        final RelationshipsPart mainPresentationRelations = mainPresentationPart.getRelationshipsPart();
    	final JSONObject layoutSettings = attrs.optJSONObject("layout");
    	if(layoutSettings!=null) {
    		ViewPropertiesPart viewPropertiesPart = null;    		
            final Relationship viewRel = mainPresentationRelations.getRelationshipByType(Namespaces.PRESENTATIONML_VIEW_PROPS);
            if(viewRel!=null) {
            	viewPropertiesPart = (ViewPropertiesPart)mainPresentationRelations.getPart(viewRel);
            }
            else {
            	viewPropertiesPart = new ViewPropertiesPart();
            	viewPropertiesPart.setJaxbElement(new ViewPr());
            	mainPresentationPart.addTargetPart(viewPropertiesPart);
            }
            final ViewPr viewPr = viewPropertiesPart.getJaxbElement();
            final Object slidePaneWidth = layoutSettings.opt("slidePaneWidth");
            if(slidePaneWidth instanceof Number) {
            	CTNormalViewProperties normalViewProperties = viewPr.getNormalViewPr();
            	if(normalViewProperties==null) {
            		normalViewProperties = Context.getpmlObjectFactory().createCTNormalViewProperties();
            		viewPr.setNormalViewPr(normalViewProperties);
            	}
            	CTNormalViewPortion leftViewPortion = normalViewProperties.getRestoredLeft();
            	if(leftViewPortion==null) {
            		leftViewPortion = Context.getpmlObjectFactory().createCTNormalViewPortion();
            		normalViewProperties.setRestoredLeft(leftViewPortion);
            	}
            	final Double sz = (((Number)slidePaneWidth).doubleValue() * 1000);
            	leftViewPortion.setSz(sz.intValue());
            }
    	}
    }

    public void insertHardBreak(JSONArray start, JSONObject attrs)
        throws JAXBException, InvalidFormatException, PartUnrecognisedException, JSONException {

    	Component.getComponent(getOperationDocument().getRootComponent(), start, start.length()-1)
			.insertChildComponent(operationDocument, start.getInt(start.length()-1), attrs, Type.HARDBREAK_DEFAULT);
    }

    public void changeLayout(JSONArray start)
    	throws InvalidFormatException {

    	final SlideLayoutPart slideLayoutPart = (SlideLayoutPart)operationDocument.getContextPart();
    	operationDocument.setContext(null);
    	final SlideComponent slideComponent = (SlideComponent)Component.getComponent(getOperationDocument().getRootComponent(), start, start.length());
    	final SlidePart slidePart = (SlidePart)slideComponent.getObject();
    	final RelationshipsPart slideRelationshipsPart = slidePart.getRelationshipsPart();
    	Relationship relationship = slideRelationshipsPart.getRelationshipByType(Namespaces.PRESENTATIONML_SLIDE_LAYOUT);
    	if(relationship!=null) {
    		slideRelationshipsPart.removeRelationship(relationship);
    	}
    	slidePart.addTargetPart(slideLayoutPart);
    }
}
