/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ods.dom;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.NavigableSet;
import java.util.Set;
import java.util.TreeSet;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.Names;
import org.odftoolkit.odfdom.component.Component;
import org.odftoolkit.odfdom.component.MapHelper;
import org.odftoolkit.odfdom.component.OdfOperationDocument;
import org.odftoolkit.odfdom.doc.OdfDocument;
import org.odftoolkit.odfdom.doc.OdfSpreadsheetDocument;
import org.odftoolkit.odfdom.dom.OdfDocumentNamespace;
import org.odftoolkit.odfdom.dom.attribute.style.StyleCellProtectAttribute;
import org.odftoolkit.odfdom.dom.attribute.style.StyleColumnWidthAttribute;
import org.odftoolkit.odfdom.dom.attribute.style.StyleRowHeightAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableDisplayAttribute;
import org.odftoolkit.odfdom.dom.element.OdfStyleBase;
import org.odftoolkit.odfdom.dom.element.OdfStylePropertiesBase;
import org.odftoolkit.odfdom.dom.element.draw.DrawFrameElement;
import org.odftoolkit.odfdom.dom.element.draw.DrawImageElement;
import org.odftoolkit.odfdom.dom.element.style.StyleChartPropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleDefaultStyleElement;
import org.odftoolkit.odfdom.dom.element.style.StyleGraphicPropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleHeaderFooterPropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleListLevelPropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StylePageLayoutPropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleParagraphPropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleRubyPropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleSectionPropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleStyleElement;
import org.odftoolkit.odfdom.dom.element.style.StyleTableCellPropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleTableColumnPropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleTablePropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleTableRowPropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleTextPropertiesElement;
import org.odftoolkit.odfdom.dom.style.OdfStyleFamily;
import org.odftoolkit.odfdom.dom.style.props.OdfStylePropertiesSet;
import org.odftoolkit.odfdom.incubator.doc.office.OdfOfficeAutomaticStyles;
import org.odftoolkit.odfdom.incubator.doc.office.OdfOfficeStyles;
import org.odftoolkit.odfdom.incubator.doc.office.OdfStylesBase;
import org.odftoolkit.odfdom.incubator.doc.style.OdfDefaultStyle;
import org.odftoolkit.odfdom.incubator.doc.style.OdfStyle;
import org.xml.sax.SAXException;

import com.openexchange.office.FilterException;
import com.openexchange.office.FilterException.ErrorCode;
import com.openexchange.office.odf.OdfOperationDoc;
import com.openexchange.office.ods.dom.SmlUtils.CellRef;
import com.openexchange.office.ods.dom.SmlUtils.CellRefRange;

/**
 *
 * @author sven.jacobi@open-xchange.com
 */

public class JsonOperationConsumer {

	private final OdfOperationDocument opsDoc;
    private final OdfSpreadsheetDocument doc;
    private final Styles styles;
    private final OfficeStyles officeStyles;
    private final Content content;
    private final Settings settings;

	public JsonOperationConsumer(OdfOperationDocument opsDoc)
		throws SAXException {

		this.opsDoc = opsDoc;
		doc = (OdfSpreadsheetDocument)opsDoc.getDocument();
		styles = (Styles)doc.getStylesDom();
		content = (Content)doc.getContentDom();
	    settings = (Settings)doc.getSettingsDom();
	    officeStyles = styles.getStyles();
	}

    public int applyOperations(JSONArray operations)
    	throws Exception {

        for (int i = 0; i < operations.length(); i++) {
        	OdfOperationDoc.setSuccessfulAppliedOperations(i);
            final JSONObject op = (JSONObject) operations.get(i);
            final String opName = op.getString("name");
            switch(opName)
            {
                case "fillCellRange": {
                    fillCellRange(op.getInt("sheet"), op.getJSONArray("start"), op.optJSONArray("end"), op.opt("value"), op.opt("result"), op.optJSONObject("attrs"), op.optInt("shared", -1), op.optJSONArray("ref"));
                    break;
                }
                case "setCellContents": {
                    setCellContents(op.getInt("sheet"), op.getJSONArray("start"), op.getJSONArray("contents"));
                    break;
                }
                case "clearCellRange": {
                    clearCellRange(op.getInt("sheet"), op.getJSONArray("start"), op.optJSONArray("end"), op.optBoolean("remove", false));
                    break;
                }
                case "mergeCells": {
                    mergeCells(op.getInt("sheet"), op.getJSONArray("start"), op.optJSONArray("end"), op.optBoolean("keepContent", false), op.optString("type", "merge"));
                    break;
                }
                case "insertRows": {
                    insertRows(op.getInt("sheet"), op.getInt("start"), op.optInt("end", -1));
                    break;
                }
                case "deleteRows": {
                    deleteRows(op.getInt("sheet"), op.getInt("start"), op.optInt("end", -1));
                    break;
                }
                case "insertColumns": {
                    insertColumns(op.getInt("sheet"), op.getInt("start"), op.optInt("end", -1));
                    break;
                }
                case "deleteColumns": {
                    deleteColumns(op.getInt("sheet"), op.getInt("start"), op.optInt("end", -1));
                    break;
                }
                case "insertSheet": {
                	insertSheet(op.getInt("sheet"), op.getString("sheetName"), op.optJSONObject("attrs"));
                    break;
                }
                case "moveSheet": {
                    content.moveSheet(op.getInt("sheet"), op.getInt("to"));
                    break;
                }
                case "deleteSheet": {
                    content.deleteSheet(op.getInt("sheet"));
                    break;
                }
                case "setSheetName": {
                	setSheetName(op.getInt("sheet"), op.getString("sheetName"));
                    break;
                }
                case "setColumnAttributes": {
                    setColumnAttributes(op.getInt("sheet"), op.getInt("start"), op.optInt("end", -1), op.getJSONObject("attrs"));
                    break;
                }
                case "setRowAttributes": {
                    setRowAttributes(op.getInt("sheet"), op.getInt("start"), op.optInt("end", -1), op.getJSONObject("attrs"));
                    break;
                }
                case "insertStyleSheet": {
                    insertStyleSheet(doc, op.getString("type"), op.getString("styleId"), op.optString("styleName"), op.optJSONObject("attrs"), op.optString("parent"), op.optBoolean("auto"), op.optBoolean("hidden"));
                    break;
                }
                case "insertDrawing": {
                    insertDrawing(op.getJSONArray("start"), op.getString("type"), op.optJSONObject("attrs"));
                    break;
                }
                case "deleteDrawing": {
                    deleteDrawing(op.getJSONArray("start"));
                    break;
                }
                case "setDrawingAttributes": {
                    applyDrawingAttributes(op.getJSONArray("start"), op.getJSONObject("attrs"));
                    break;
                }
                case "insertName": {
                    changeOrInsertName(op.optInt("sheet", -1), op.getString("exprName"), op.getString("formula"), null, op.optBoolean("isExpression", true), op.optBoolean("hidden", false), op.optString("ref"), true);
                    break;
                }
                case "changeName": {
                    changeOrInsertName(op.optInt("sheet", -1), op.getString("exprName"), op.optString("formula", null), op.optString("newName", null), op.opt("isExpression"), op.optBoolean("hidden", false), op.optString("ref", null), false);
                    break;
                }
                case "deleteName": {
                	deleteName(op.optInt("sheet", -1), op.getString("exprName"));
                	break;
                }
                case "insertTable": {
                    insertTable(op.getInt("sheet"), op.optString("table", null), op.getJSONArray("start"), op.getJSONArray("end"), op.optJSONObject("attrs"));
                    break;
                }
                case "changeTable": {
                    insertTable(op.getInt("sheet"), op.optString("table", null), op.optJSONArray("start"), op.optJSONArray("end"), op.optJSONObject("attrs"));
                    break;
                }
                case "deleteTable": {
                    deleteTable(op.getInt("sheet"), op.optString("table", null));
                    break;
                }
                case "changeTableColumn": {
                    changeTableColumn(op.getInt("sheet"), op.optString("table", null), op.getInt("col"), op.getJSONObject("attrs"));
                    break;
                }
                case "insertHyperlink": {
                	insertHyperlink(op.getInt("sheet"), op.getJSONArray("start"), op.optJSONArray("end"), op.getString("url"));
                	break;
                }
                case "deleteHyperlink": {
                	deleteHyperlink(op.getInt("sheet"), op.getJSONArray("start"), op.optJSONArray("end"));
                	break;
                }
                case "setDocumentAttributes": {
                	setDocumentAttributes(op.getJSONObject("attrs"));
                	break;
                }
                case "insertCondFormat": {
                	changeOrInsertCondFormat(true, op.getInt("sheet"), op.getInt("index"), op.getJSONArray("ranges"));
                	break;
                }
                case "changeCondFormat": {
                	changeOrInsertCondFormat(false, op.getInt("sheet"), op.getInt("index"), op.getJSONArray("ranges"));
                	break;
                }
                case "deleteCondFormat": {
                	deleteCondFormat(op.getInt("sheet"), op.getInt("index"));
                	break;
                }
                case "insertCondFormatRule": {
                	changeOrinsertCondFormatRule(true, op.getInt("sheet"), op.getInt("index"), op.getInt("ruleIndex"), op.optString("type", "formula"),
                		op.has("value1") ? op.get("value1") : "", op.optString("value2", ""), op.optInt("priority"), op.optBoolean("stop", false), op.has("attrs") ? op.getJSONObject("attrs") : new JSONObject());
                	break;
                }
                case "changeCondFormatRule": {
                	changeOrinsertCondFormatRule(false, op.getInt("sheet"), op.getInt("index"), op.optInt("ruleIndex"), op.optString("type", null),
                		op.opt("value1"), op.optString("value2", null), op.optInt("priority"), op.optBoolean("stop"), op.optJSONObject("attrs"));
                	break;
                }
                case "deleteCondFormatRule": {
                	deleteCondFormatRule(op.getInt("sheet"), op.getInt("index"), op.optInt("ruleIndex"));
                	break;
                }
                case "setSheetAttributes": {
                    setSheetAttributes(op.getInt("sheet"), op.getJSONObject("attrs"));
                    break;
                }
                case "createError": {
                    throw new FilterException("createError operation detected: " + opName, ErrorCode.UNSUPPORTED_OPERATION_USED);
                }
            }
/*
            else if (opName.equals("copySheet"))
            	applyOperationHelper.copySheet(op.getInt("sheet"), op.getInt("to"), op.getString("sheetName"));
            else if (opName.equals("setDrawingAttributes"))
                applyOperationHelper.applyDrawingAttributes(op.getJSONArray("start"), op.getJSONObject("attrs"));
            else if (opName.equals("insertChartDataSeries"))
                applyOperationHelper.insertChartDataSeries(op.getJSONArray("start"), op.getInt("series"), op.optJSONObject("attrs"));
            else if (opName.equals("insertChartDataSeries"))
                applyOperationHelper.insertChartDataSeries(op.getJSONArray("start"), op.getInt("series"), op.optJSONObject("attrs"));
            else if (opName.equals("setChartDataSeriesAttributes"))
                applyOperationHelper.setChartDataSeriesAttributes(op.getJSONArray("start"), op.getInt("series"), op.optJSONObject("attrs"));
            else if (opName.equals("deleteChartDataSeries"))
                applyOperationHelper.deleteChartDataSeries(op.getJSONArray("start"), op.getInt("series"));
            else if (opName.equals("setChartAxisAttributes"))
                applyOperationHelper.setChartAxisAttributes(op.getJSONArray("start"), op.getString("axis"), op.optJSONObject("attrs"));
            else if (opName.equals("setChartGridlineAttributes"))
                applyOperationHelper.setChartGridlineAttributes(op.getJSONArray("start"), op.getString("axis"), op.optJSONObject("attrs"));
            else if (opName.equals("setChartTitleAttributes"))
                applyOperationHelper.setChartTitleAttributes(op.getJSONArray("start"), op.getString("axis"), op.optJSONObject("attrs"));
            else if (opName.equals("setChartLegendAttributes"))
                applyOperationHelper.setChartLegendAttributes(op.getJSONArray("start"), op.optJSONObject("attrs"));
            else if (opName.equals("insertValidation"))
            	ValidationUtils.insertValidation(this, op);
            else if (opName.equals("changeValidation"))
            	ValidationUtils.changeValidation(this, op);
            else if (opName.equals("deleteValidation"))
            	ValidationUtils.deleteValidation(this, op.getInt("sheet"), op.getInt("index"));
            else
                logMessage("warn", "Ignoring unsupported operation: " + opName);
*/
        }
        OdfOperationDoc.setSuccessfulAppliedOperations(operations.length());
        return 1;
    }

    private void changeOrInsertCondFormat(boolean insert, int sheetIndex, int index, JSONArray ranges) {

    	final Sheet sheet = content.getSheets().get(sheetIndex);
    	final List<ConditionalFormat> conditionalFormats = sheet.getConditionalFormats(true).getConditionalFormatList();
    	final Ranges r = new Ranges(sheetIndex, ranges);
    	if(insert) {
    		conditionalFormats.add(index, new ConditionalFormat(r));
    	}
    	else {
    		conditionalFormats.get(index).setRanges(r);
    	}
    }

    private void deleteCondFormat(int sheetIndex, int index) {

    	content.getSheets().get(sheetIndex).getConditionalFormats(false).getConditionalFormatList().remove(index);
    }

    private void changeOrinsertCondFormatRule(boolean insert, int sheetIndex, int index, int ruleIndex, String type, Object value1, String value2, Integer priority, boolean stop, JSONObject attrs)
    	throws JSONException, SAXException {

    	final List<Condition> conditions = content.getSheets().get(sheetIndex).getConditionalFormats(false).getConditionalFormatList().get(index).getConditions();
    	if(insert) {
    		conditions.add(ruleIndex, new Condition());
    	}
    	conditions.get(ruleIndex).applyCondFormatRuleOperation(doc, type, value1, value2, priority, stop, attrs);
    }

    private void deleteCondFormatRule(int sheetIndex, int index, int ruleIndex) {

    	content.getSheets().get(sheetIndex).getConditionalFormats(false).getConditionalFormatList().get(index).getConditions().remove(ruleIndex);
    }

    private void setDocumentAttributes(JSONObject attrs) {

    	final JSONObject documentAttrs = attrs.optJSONObject("document");
    	if(documentAttrs!=null) {
    		final Object activeSheet = documentAttrs.opt("activeSheet");
    		if(activeSheet instanceof Integer) {
    			final String sheetName = content.getSheets().get(((Integer)activeSheet).intValue()).getSheetName();
    			if(sheetName!=null) {
    				settings.setActiveSheet(sheetName);
    			}
    		}
    	}
    }

    private void insertHyperlink(int sheet, JSONArray start, JSONArray optEnd, String url)
    	throws JSONException {

    	if(optEnd==null) {
    		optEnd = start;
    	}
    	final CellRefRange cellRefRange = new CellRefRange(new CellRef(start.getInt(0), start.getInt(1)), new CellRef(optEnd.getInt(0), optEnd.getInt(1)));
		final List<Hyperlink> hyperlinkList = content.getSheets().get(sheet).getHyperlinks();
        for(int i=hyperlinkList.size()-1;i>=0;i--) {
        	final Hyperlink hyperlink = hyperlinkList.get(i);
        	if((hyperlink.getCellRefRange()==null)||cellRefRange.contains(hyperlink.getCellRefRange())) {
        		hyperlinkList.remove(i);
        	}
        }
    	hyperlinkList.add(new Hyperlink(cellRefRange, url));
    }

    private void deleteHyperlink(int sheet, JSONArray start, JSONArray optEnd)
    	throws JSONException {

    	if(optEnd==null) {
    		optEnd = start;
    	}
    	final CellRefRange cellRefRange = new CellRefRange(new CellRef(start.getInt(0), start.getInt(1)), new CellRef(optEnd.getInt(0), optEnd.getInt(1)));
		final List<Hyperlink> hyperlinkList = content.getSheets().get(sheet).getHyperlinks();
		for(int i=hyperlinkList.size()-1;i>=0;i--) {
			final Hyperlink hyperlink = hyperlinkList.get(i);
			if((hyperlink.getCellRefRange()==null)||cellRefRange.intersects(hyperlink.getCellRefRange())) {
				hyperlinkList.remove(i);
			}
		}
    }

    public void insertDrawing(JSONArray start, String type, JSONObject attrs)
    	throws JSONException {

    	final Sheet sheet = content.getSheets().get(start.getInt(0));

    	// the default is two cell anchor ...
    	final DrawingAnchor drawingAnchor = new DrawingAnchor(0, 0);
    	Drawing drawing = null;
    	if(type.equals("image")) {
    		final ElementNS frameElement = new ElementNS(content, DrawFrameElement.ELEMENT_NAME.getUri(), DrawFrameElement.ELEMENT_NAME.getQName());
    		final ElementNS imageElement = new ElementNS(content, DrawImageElement.ELEMENT_NAME.getUri(), DrawImageElement.ELEMENT_NAME.getQName());
    		frameElement.appendChild(imageElement);
    		final Frame frame = new Frame(drawingAnchor, frameElement);
    		frame.setImageElement(imageElement);
    		drawing = frame;
    	}
    	else {
    		drawing = new Drawing(drawingAnchor);
    	}
		sheet.getDrawings().addDrawing(drawing, start.getInt(1));
		if(attrs!=null) {
			applyDrawingAttributes(start, attrs);
		}
    }

    public void deleteDrawing(JSONArray start)
    	throws JSONException {

    	final Sheet sheet = content.getSheets().get(start.getInt(0));
    	sheet.getDrawings().deleteDrawing(start.getInt(1));
    }

    private void changeOrInsertName(int sheet, String exprName, String formula, String newName, Object isExpression, boolean hidden, String ref, boolean insert) {

    	final NamedExpressions namedExpressions = sheet==-1?content.getNamedExpressions(true):content.getSheets().get(sheet).getNamedExpressions(true);
    	NamedExpression namedExpression;
    	if(insert) {
    		namedExpression = new NamedExpression(exprName);
    		if(isExpression==null) {
    			isExpression = Boolean.valueOf(true);
    		}
    	}
    	else {
    		namedExpression = namedExpressions.getExpressionList().remove(exprName);
        	if(newName!=null) {
        		namedExpression.setName(newName);
        	}
        	if(isExpression==null) {
        		isExpression = namedExpression.getExpression()!=null;
        	}
    	}
    	if(formula!=null) {
	    	if(((Boolean)isExpression).booleanValue()) {
	    		namedExpression.setExpression(formula);
	    	}
	    	else {
	    		namedExpression.setCellRangeAddress(getAddressWithoutBraces(formula));
	    	}
    	}
    	if(ref!=null) {
    		namedExpression.setBaseCellAddress(getAddressWithoutBraces(ref));
    	}
    	namedExpressions.getExpressionList().put(namedExpression.getName(), namedExpression);
    }

    private String getAddressWithoutBraces(String in) {
    	if(in==null||in.length()<2) {
    		return in;
    	}
    	if(in.charAt(0)=='['&&in.charAt(in.length()-1)==']') {
    		return in.substring(1, in.length()-1);
    	}
    	return in;
    }

    private void deleteName(int sheet, String exprName) {

    	final NamedExpressions namedExpressions = sheet==-1?content.getNamedExpressions(false):content.getSheets().get(sheet).getNamedExpressions(false);
    	namedExpressions.getExpressionList().remove(exprName);
    }

    private void insertTable(int sheetIndex, String table, JSONArray start, JSONArray end, JSONObject attrs)
    	throws JSONException {

    	final DatabaseRange databaseRange = content.getDatabaseRanges(true).getDatabaseRange(content, sheetIndex, table, true);
    	databaseRange.setDisplayFilterButtons(true);
    	final Range sheetRange = databaseRange.getRange();
    	final CellRefRange cellRefRange = sheetRange.getCellRefRange();
    	if(start!=null) {
    		final CellRef startRef = cellRefRange.getStart();
    		startRef.setColumn(start.getInt(0));
    		startRef.setRow(start.getInt(1));
    	}
    	if(end!=null) {
    		final CellRef endRef = cellRefRange.getEnd();
    		endRef.setColumn(end.getInt(0));
    		endRef.setRow(end.getInt(1));
    	}
    	sheetRange.setSheetIndex(sheetIndex);
    	if(attrs!=null) {
    		final JSONObject tableAttrs = attrs.optJSONObject("table");
    		if(tableAttrs!=null) {
    			Object containsHeader = tableAttrs.opt("headerRow");
    			if(containsHeader instanceof Boolean) {
    				databaseRange.setContainsHeader((Boolean)containsHeader);
    			}
    		}
    	}
    }

    private void deleteTable(int sheet, String table) {
    	content.getDatabaseRanges(false).deleteTable(content, sheet, table);
    }

    private void changeTableColumn(int sheet, String table, int col, JSONObject attrs) {
    	content.getDatabaseRanges(false).getDatabaseRange(content, sheet, table, false).changeTableColumn(content, col, attrs);
    }

    public void applyDrawingAttributes(JSONArray start, JSONObject attrs)
    	throws JSONException {

    	final int index = start.getInt(1);
    	final Sheet sheet = content.getSheets().get(start.getInt(0));
    	final Drawings drawings = sheet.getDrawings();
    	final JSONObject drawingAttrs = attrs.optJSONObject("drawing");
		final Drawing drawing = drawings.getDrawing(index);
    	if(drawingAttrs!=null) {
    		// first updating the drawingAnchor -> sheet / cell anchoring
    		final String anchorType = drawingAttrs.optString("anchorType", null);
    		if(anchorType!=null&&anchorType.equals("absolute")) {
				drawings.setDrawingAnchor(null, index);
    		}
    		else if(anchorType!=null||drawingAttrs.has("startCol")||drawingAttrs.has("startRow")) {
    			final DrawingAnchor oldDrawingAnchor = drawing.getAnchor();
    			int col = 0;
    			int row = 0;
    			if(oldDrawingAnchor!=null) {
    				col = oldDrawingAnchor.getColumn();
    				row = oldDrawingAnchor.getRow();
    			}
    			if(drawingAttrs.has("startCol")) {
    				col = drawingAttrs.getInt("startCol");
    			}
    			if(drawingAttrs.has("startRow")) {
    				row = drawingAttrs.getInt("startRow");
    			}
    			// ensure that the cell where the drawing is attached to is physically available ...
    			sheet.getRow(row, true, true, true).getCell(col, true, true, true);
   				drawings.setDrawingAnchor(new DrawingAnchor(col, row), index);
			}
    	}
    	drawing.applyAttributes(opsDoc, attrs);
    }

    public void mergeCells(int sheetIndex, JSONArray start, JSONArray optEnd, boolean keepContent, String type)
            throws JSONException {

        if(start.length()!=2)
            throw new RuntimeException("xlsx::ApplyoperationHelper::mergeCells: size of start parameter != 2");

        if(optEnd==null)
            optEnd = start;
        else if (optEnd.length()!=2)
            throw new RuntimeException("xlsx::ApplyoperationHelper::mergeCells: size of end parameter != 2");

        final Sheet sheet = content.getSheets().get(sheetIndex);

        // first we will remove each mergeCell that is covering our new mergeCellRange
        final SmlUtils.CellRefRange cellRefRange = new SmlUtils.CellRefRange(new SmlUtils.CellRef(start.getInt(0), start.getInt(1)), new SmlUtils.CellRef(optEnd.getInt(0), optEnd.getInt(1)));
        final List<MergeCell> mergeCellList = sheet.getMergeCells();
        for(int i=mergeCellList.size()-1;i>=0;i--) {
            final MergeCell mergeCell = mergeCellList.get(i);
            if(cellRefRange.intersects(mergeCell.getCellRefRange())) {
                mergeCellList.remove(i);
            }
        }
        if(type.equals("merge")) {
        	addMergeCellRange(sheet, cellRefRange);
        }
        else if(type.equals("horizontal")) {
            for(int row=start.getInt(1);row<=optEnd.getInt(1);row++) {
                addMergeCellRange(sheet, new CellRefRange(new CellRef(start.getInt(0), row), new CellRef(optEnd.getInt(0), row)));
            }
        }
        else if(type.equals("vertical")) {
            for(int column=start.getInt(0);column<=optEnd.getInt(0);column++) {
                addMergeCellRange(sheet, new CellRefRange(new CellRef(column, start.getInt(1)), new CellRef(column, optEnd.getInt(1))));
            }
        }
    }

    private void addMergeCellRange(Sheet sheet, CellRefRange mergeCellRange) {
        final List<MergeCell> mergeCellList = sheet.getMergeCells();
        mergeCellList.add(new MergeCell(mergeCellRange));
        // creating the last row (with ceiling cut)
        sheet.getRow(mergeCellRange.getEnd().getRow(), true, false, true);
        // creating the first row without repeatings (needed for covered cells, they are not allowed to be repeated)
        sheet.getRow(mergeCellRange.getStart().getRow(), true, true, true);

        final NavigableSet<Row> rowSelection = sheet.getRows().subSet(new Row(null, mergeCellRange.getStart().getRow()), true, new Row(null, mergeCellRange.getEnd().getRow()), true);
        final Iterator<Row> rowIter = rowSelection.iterator();
        boolean first = false;
        while(rowIter.hasNext()) {
        	final Row rowEntry = rowIter.next();
    		rowEntry.getCell(mergeCellRange.getEnd().getColumn(), true, false, true);
    		rowEntry.getCell(mergeCellRange.getStart().getColumn(), true, true, first);
    		first = false;
        }
    }

    public void insertRows(int sheetIndex, int start, int optEnd)
    	throws Exception {

    	final int count = getColumnRowCount(start, optEnd);
        content.getSheets().get(sheetIndex).insertRows(sheetIndex, start, count);
    }

    public void deleteRows(int sheetIndex, int start, int optEnd)
    	throws Exception {

    	final int count = getColumnRowCount(start, optEnd);
        content.getSheets().get(sheetIndex).deleteRows(sheetIndex, start, count);
    }

    public void insertColumns(int sheetIndex, int start, int optEnd)
    	throws Exception {

    	final int count = getColumnRowCount(start, optEnd);
        content.getSheets().get(sheetIndex).insertColumns(sheetIndex, start, count);
    }

    public void deleteColumns(int sheetIndex, int start, int optEnd)
    	throws Exception {

    	final int count = getColumnRowCount(start, optEnd);
        content.getSheets().get(sheetIndex).deleteColumns(sheetIndex, start, count);
    }

    private int getColumnRowCount(int start, int optEnd) {
        if(start<0) {
            throw new RuntimeException("ods::deleteColumns: start<0");
        }
        int count = 1;
        if(optEnd!=-1) {
            if(optEnd<start) {
                throw new RuntimeException("ods::deleteColumns: optEnd<start");
            }
            count = (optEnd-start)+1;
        }
        return count;
    }

    public void setCellContents(int sheetIndex, JSONArray start, JSONArray contents)
    	throws JSONException {

    	if(start.length()!=2)
            throw new RuntimeException("xlsx::ApplyoperationHelper::setCellContents: size of start parameter != 2");

        final Sheet sheet = content.getSheets().get(sheetIndex);

        int y = start.getInt(1);
        for(int i=0; i<contents.length(); i++) {
            final JSONArray cellArray = contents.getJSONArray(i);
            int x = start.getInt(0);
            for(int j=0; j<cellArray.length(); j++) {

                final Row row = sheet.getRow(y, true, true, true);
                final Cell cell = row.getCell(x, true, true, true);

                final Object cellObject = cellArray.get(j);
                if(cellObject instanceof JSONObject) {
	                final Object value = ((JSONObject)cellObject).opt("value");
	                if(value!=null) {
	                	cell.setCellContent(value);
	                }
	                final JSONObject attrs = ((JSONObject)cellObject).optJSONObject("attrs");
	                if(attrs!=null&&attrs.has("styleId")) {
	    				cell.setCellStyle(attrs.getString("styleId"));
	                }
                }
                x++;
            }
            y++;
        }
    }

    public void fillCellRange(int sheetIndex, JSONArray start, JSONArray optEnd, Object optValue, Object optResult, JSONObject optAttrs, int optShared, JSONArray optRef)
    	throws JSONException {

    	if(start.length()!=2)
            throw new RuntimeException("ods::JsonOperationConsumer::fillCellRange: size of start parameter != 2");

        if(optEnd==null)
            optEnd = start;
        else if (optEnd.length()!=2)
            throw new RuntimeException("ods::JsonOperationConsumer::fillCellRange: size of end parameter != 2");

        final Sheet sheet = content.getSheets().get(sheetIndex);

        // creating row entries...
        sheet.getRow(optEnd.getInt(1), true, false, true);
        sheet.getRow(start.getInt(1), true, true, false);

        final NavigableSet<Row> rowSelection = sheet.getRows().subSet(new Row(null, start.getInt(1)), true, new Row(null, optEnd.getInt(1)), true);
        final Iterator<Row> rowIter = rowSelection.iterator();
        while(rowIter.hasNext()) {
        	final Row row = rowIter.next();

        	// creating cell entries....
        	row.getCell(optEnd.getInt(0), true, false, true);
        	row.getCell(start.getInt(0), true, true, false);

        	final NavigableSet<Cell> cellSelection = row.getCells().subSet(new Cell(start.getInt(0)), true, new Cell(optEnd.getInt(0)), true);
        	final Iterator<Cell> cellIter = cellSelection.iterator();
        	while(cellIter.hasNext()) {
        		final Cell cell = cellIter.next();
        		if(optValue!=null) {
        			cell.setCellContent(optValue);
        		}
        		if(optAttrs!=null&&optAttrs.has("styleId")) {
    				cell.setCellStyle(optAttrs.getString("styleId"));
        		}
        	}
        }
    }

    public void clearCellRange(int sheetIndex, JSONArray start, JSONArray optEnd, boolean remove)
    	throws JSONException {

    	if(start.length()!=2)
            throw new RuntimeException("ods::JsonOperationConsumer::fillCellRange: size of start parameter != 2");

        if(optEnd==null)
            optEnd = start;
        else if (optEnd.length()!=2)
            throw new RuntimeException("ods::JsonOperationConsumer::fillCellRange: size of end parameter != 2");

    	content.getSheets().get(sheetIndex).clearCellRange(start.getInt(1), optEnd.getInt(1), start.getInt(0), optEnd.getInt(0), remove);
    }

    public void insertSheet(int sheetIndex, String sheetName, JSONObject attrs)
    	throws JSONException, SAXException {

    	content.insertSheet(sheetIndex, sheetName);
    	if(attrs!=null) {
    		setSheetAttributes(sheetIndex, attrs);
    	}
    }

    public void setSheetName(int sheetIndex, String sheetName) {
    	final Sheet _sheet = content.getSheets().get(sheetIndex);

        final String oldSheetName = _sheet.getSheetName();
        final String encodedOldName = encodeSheetName(oldSheetName);
        final String encodedNewName = encodeSheetName(sheetName);

        if(encodedOldName.length()<=0&&encodedNewName.length()<=0) {
            return;
        }
        _sheet.setSheetName(encodedNewName);

    	// taking care of viewSettings
        settings.setActiveSheet(updateSheetName(settings.getActiveSheet(), encodedOldName, encodedNewName));
    	final ConfigItemMapEntry globalViewSettings = settings.getGlobalViewSettings(false);
    	if(globalViewSettings!=null) {
    		final ConfigItemMapNamed viewTables = settings.getViewTables(globalViewSettings, false);
    		if(viewTables!=null) {
    			final HashMap<String, ConfigItemMapEntry> viewTableMap = viewTables.getItems();
    			final ConfigItemMapEntry oldViewSettings = viewTableMap.remove(encodedOldName);
    			if(oldViewSettings!=null) {
    				oldViewSettings.setName(encodedNewName);
    				viewTableMap.put(encodedNewName, oldViewSettings);
    			}
    		}
    	}

    	// defined names
    	final NamedExpressions namedExpressions = content.getNamedExpressions(false);
    	if(namedExpressions!=null) {
    		final Iterator<NamedExpression> namedExpressionIter = namedExpressions.getExpressionList().values().iterator();
    		while(namedExpressionIter.hasNext()) {
    			final NamedExpression namedExpression = namedExpressionIter.next();
    			namedExpression.setBaseCellAddress(updateSheetName(namedExpression.getBaseCellAddress(), encodedOldName, encodedNewName));
    			if(namedExpression.getExpression()!=null) {
        			namedExpression.setExpression(updateFormulaSheetName(namedExpression.getExpression(), encodedOldName, encodedNewName));
    			}
    			else {
    				namedExpression.setCellRangeAddress(updateSheetName(namedExpression.getCellRangeAddress(), encodedOldName, encodedNewName));
    			}
    		}
    	}
    	// sheetRefs of the whole document
    	for(Sheet sheet:content.getSheets()) {
    		final Iterator<Row> rowIter = sheet.getRows().iterator();
    		while(rowIter.hasNext()) {
    			final Row row = rowIter.next();
    			final Iterator<Cell> cellIter = row.getCells().iterator();
    			while(cellIter.hasNext()) {
    				final Cell cell = cellIter.next();
    				final Object cellContent = cell.getCellContent();
    				if(cellContent instanceof String) {
    					if(((String)cellContent).startsWith(Names.EQUATION)) {
    						cell.setCellContent(updateFormulaSheetName((String)cellContent, encodedOldName, encodedNewName));
    					}
    				}
    			}
    		}
    	}
    }

    public static int getNameOffset(String content, int startOffset) {

        boolean stringLiteral = false;
        boolean complexString = false;
        for(int i=startOffset; i<content.length();i++) {
            final char n = content.charAt(i);
            if(complexString) {
                if(n=='\'') {
                    if(i+1<content.length()&&content.charAt(i+1)=='\'') {
                        i++;
                    }
                    else {
                        complexString = false;
                    }
                }
            }
            else if(stringLiteral) {
                if(n=='"') {
                    if(i+1<content.length()&&content.charAt(i+1)=='"') {
                        i++;
                    }
                    else {
                        stringLiteral = false;
                    }
                }
            }
            else if(n=='\'') {
                complexString = true;
            }
            else if(n=='"') {
                stringLiteral = true;
            }
            else if(n=='[') {
                return i;
            }
        }
        return -1;
    }

    public static String updateSheetName(String content, String oldName, String newName) {
        String newContent = content;
        if(content!=null&&content.length()>oldName.length()) {
			newContent = content.replaceAll(oldName, newName);
        }
        return newContent;
    }

    private static String simpleCharPattern ="^[\\w.\\xa1-\\u2027\\u202a-\\uffff]+$";

    public static String updateFormulaSheetName(String content, String oldName, String newName) {
        String newContent = content;
        List<Integer> replaceOffsets = null;
        if(content!=null&&content.length()>oldName.length()) {
            for(int startOffset = 0; startOffset<content.length();) {
                final int nameOffset = getNameOffset(content, startOffset);
                if(nameOffset<0) {
                    break;
                }
                final int possibleLength = nameOffset - startOffset;
                if(possibleLength>=oldName.length()) {
                    boolean replace = false;
                    final boolean isComplex = content.charAt(nameOffset-1)=='\'';
                    final String subString = content.substring(nameOffset-oldName.length(), nameOffset);

                    final int mOffset = nameOffset - oldName.length();
                    if(isComplex&&oldName.charAt(0)=='\'') {
                        if(subString.equals(oldName)) {
                            replace = true;
                        }
                    }
                    else if(oldName.charAt(0)!='\'') {
                        if(subString.equals(oldName)) {
                            if(mOffset>startOffset) {
                                final String prev = content.substring(mOffset-1, mOffset);
                                if(!prev.matches(simpleCharPattern)) {
                                    replace = true;
                                }
                            }
                            else {
                                replace = true;
                            }
                        }
                    }
                    if(replace) {
                        if(replaceOffsets==null) {
                            replaceOffsets = new ArrayList<Integer>();
                        }
                        replaceOffsets.add(mOffset);
                    }
                }
                startOffset = nameOffset + 1;
            }
        }
        if(replaceOffsets!=null) {
            final StringBuffer buffer = new StringBuffer(content);
            for(int i=replaceOffsets.size()-1;i>=0;i--) {
                final int offset = replaceOffsets.get(i);
                buffer.replace(offset, offset + oldName.length(), newName);
            }
            newContent = buffer.toString();
        }
        return newContent;
    }

    public static String encodeSheetName(String sheetName) {
        final StringBuffer encodedName = new StringBuffer(sheetName.length());
        for(int i=0; i<sheetName.length();i++) {
            final char c = sheetName.charAt(i);
            encodedName.append(c);
            if(c=='\'') {
                encodedName.append(c);
            }
        }
        if(!sheetName.matches(simpleCharPattern)) {
            encodedName.insert(0, '\'');
            encodedName.append('\'');
        }
        return encodedName.toString();
    }

    public void setColumnAttributes(int sheetIndex, int start, int optEnd, JSONObject attrs)
    	throws JSONException, SAXException {

        if(optEnd==-1) {
            optEnd = start;
        }
        final Sheet sheet = content.getSheets().get(sheetIndex);
        // ... splitting up corresponding columns (repeated)..
        sheet.getColumn(optEnd, true, false, true);
        sheet.getColumn(start, true, true, false);
        final JSONObject columnAttrs = attrs.optJSONObject("column");
        if(columnAttrs!=null) {
	        final NavigableSet<Column> columns = sheet.getColumns().subSet(new Column(null, start), true, new Column(null, optEnd), true);
	        for(Column column:columns) {
	        	OdfStyle autoStyle = null;
	        	final Iterator<Entry<String, Object>> columnAttrsIter = columnAttrs.entrySet().iterator();
	        	while(columnAttrsIter.hasNext()) {
	        		final Entry<String, Object> columnAttrsEntry = columnAttrsIter.next();
    				final Object columnAttrsValue = columnAttrsEntry.getValue();
    				if(columnAttrsValue!=null) {
		        		switch(columnAttrsEntry.getKey()) {
		        			case "visible": {
	        					if(columnAttrsValue instanceof Boolean) {
	        						column.setVisibility(((Boolean)columnAttrsValue).booleanValue()?Visibility.VISIBLE:Visibility.COLLAPSE);
	        					}
	        					else {
	        						column.setVisibility(null);
	        					}
		        				break;
		        			}
		        			case "width": {
		        				if(autoStyle==null) {
		        					autoStyle = createUniqueAutoStyle(column);
		        				}
		        				if(columnAttrsValue instanceof Number) {
		        					 OdfStylePropertiesBase propsElement = autoStyle.getOrCreatePropertiesElement(OdfStylePropertiesSet.TableColumnProperties);
		        					 if(propsElement!=null) {
		        						 final StyleColumnWidthAttribute columnWidth = new StyleColumnWidthAttribute(content);
		        						 final Double width = new Double(((Number)columnAttrsValue).doubleValue() / 1000);
		        						 columnWidth.setValue(width.toString() + "cm");
		        						 propsElement.setOdfAttribute(columnWidth);
		        					 }
		        				}
		        				else {
		        					
		        				}
		        				break;
		        			}
		        		}
    				}
	        	}
	        }
        }
        // creating max repeated rows ...
        sheet.getRow(Sheet.getMaxRowCount()-1, true, false, false);
        final Iterator<Row> rowIter = sheet.getRows().iterator();
        while(rowIter.hasNext()) {
        	final Row row = rowIter.next();
            // splitting up repeated cells and allow to create a proper iterator over the NavigableSet
        	row.getCell(optEnd, true, false, true);
        	row.getCell(start, true, true, false);
        	final NavigableSet<Cell> cellSelection = row.getCells().subSet(new Cell(start), true, new Cell(optEnd), true);
        	final Iterator<Cell> cellIter = cellSelection.iterator();
        	while(cellIter.hasNext()) {
        		final Cell cell = cellIter.next();
    			if(attrs!=null&&attrs.has("styleId")) {
    				cell.setCellStyle(attrs.getString("styleId"));
    			}
        	}
        }
    }

    private OdfStyle createUniqueAutoStyle(IStyleAccess c)
    	throws SAXException {

    	final OdfOfficeAutomaticStyles autoStyles = ((Content)doc.getContentDom()).getAutomaticStyles();
    	OdfStyle oldStyle = null;
    	if(c.getStyleName()!=null&&!c.getStyleName().isEmpty()) {
    		oldStyle = autoStyles.getStyle(c.getStyleName(), c.getOdfStyleFamily());
    		final int count = officeStyles.getStyleUsage(c.getStyleFamily(), c.getStyleName());
    		if(count==1) {
    			return oldStyle;
    		}
    	}
    	OdfStyle newStyle;
    	if(oldStyle!=null) {
    		newStyle = autoStyles.makeStyleUnique(oldStyle);
    	}
    	else {
    		newStyle = autoStyles.newStyle(c.getOdfStyleFamily());
    	}
    	final String newStyleName = newStyle.getStyleNameAttribute();
    	final OfficeAutomaticStyle officeAutomaticStyle = new OfficeAutomaticStyle(newStyle, c.getStyleFamily(), newStyleName);
    	officeStyles.getAutomaticStyles(c.getStyleFamily(), true).put("newStyleName", officeAutomaticStyle);
    	c.replaceStyle(officeStyles, newStyleName);
    	return newStyle;
    }

    public void setRowAttributes(int sheetIndex, int start, int optEnd, JSONObject attrs)
    	throws JSONException, SAXException {

        if(optEnd==-1) {
            optEnd = start;
        }
        final Sheet sheet = content.getSheets().get(sheetIndex);
        // splitting up repeated rows and allow to create a proper iterator over the NavigableSet
        sheet.getRow(optEnd, true, false, true);
        sheet.getRow(start, true, true, false);
        final NavigableSet<Row> rows = sheet.getRows().subSet(new Row(null, start), true, new Row(null, optEnd), true);
        for(Row row:rows) {
        	OdfStyle autoStyle = null;
            final JSONObject rowAttrs = attrs.optJSONObject("row");
            if(rowAttrs!=null) {
	        	final Iterator<Entry<String, Object>> rowAttrsIter = rowAttrs.entrySet().iterator();
	        	while(rowAttrsIter.hasNext()) {
	        		final Entry<String, Object> rowAttrsEntry = rowAttrsIter.next();
    				final Object rowAttrsValue = rowAttrsEntry.getValue();
    				if(rowAttrsValue!=null) {
		        		switch(rowAttrsEntry.getKey()) {
		        			case "visible": {
	        					if(rowAttrsValue instanceof Boolean) {
		        					final Visibility oldVisibility = row.getVisibility();
		        					if(oldVisibility!=Visibility.FILTER) {
		        						row.setVisibility(((Boolean)rowAttrsValue).booleanValue()?Visibility.VISIBLE:Visibility.COLLAPSE);
		        					}
	        					}
	        					else {
	        						row.setVisibility(null);
	        					}
		        				break;
		        			}
		        			case "height": {
		        				if(autoStyle==null) {
		        					autoStyle = createUniqueAutoStyle(row);
		        				}
		        				if(rowAttrsValue instanceof Number) {
		        					 OdfStylePropertiesBase propsElement = autoStyle.getOrCreatePropertiesElement(OdfStylePropertiesSet.TableRowProperties);
		        					 if(propsElement!=null) {
		        						 final StyleRowHeightAttribute rowHeight = new StyleRowHeightAttribute(content);
		        						 final Double height = new Double(((Number)rowAttrsValue).doubleValue() / 1000);
		        						 rowHeight.setValue(height.toString() + "cm");
		        						 propsElement.setOdfAttribute(rowHeight);
		        					 }
		        				}
		        				else {
		        					
		        				}
		        				break;
		        			}
		        			case "filtered": {
		        				if(rowAttrsValue instanceof Boolean) {
		        					final Visibility oldVisibility = row.getVisibility();
		        					if(oldVisibility!=Visibility.COLLAPSE) {
		        						row.setVisibility(((Boolean)rowAttrsValue).booleanValue() ? Visibility.FILTER : Visibility.VISIBLE);
		        					}
		        				}
		        				else {
		        					row.setVisibility(Visibility.VISIBLE);
		        				}
		        			}
		        		}
    				}
	        	}
            }
        	// creating row entries...
        	row.getCell(sheet.getMaxColCount()-1, true, false, false);
        	final TreeSet<Cell> cells = row.getCells();
        	final Iterator<Cell> cellIter = cells.iterator();
        	while(cellIter.hasNext()) {
        		final Cell cell = cellIter.next();
    			if(attrs!=null&&attrs.has("styleId")) {
    				cell.setCellStyle(attrs.getString("styleId"));
    			}
        	}
        }
    }

    public void setSheetAttributes(int sheetIndex, JSONObject attrs)
    	throws JSONException, SAXException {

    	final JSONObject sheetProperties = attrs.optJSONObject("sheet");
    	if(sheetProperties!=null) {
	    	final Sheet sheet = content.getSheets().get(sheetIndex);
			final ConfigItemMapEntry globalViewSettings = settings.getGlobalViewSettings(true);
			final ConfigItemMapEntry sheetViewSettings = settings.getViewSettings(globalViewSettings, sheet.getSheetName(), true);

			JSONArray ranges = null;
        	OdfStyle autoStyle = null;
			int activeIndex = 0;

			for(Entry<String, Object> sheetPropEntry:sheetProperties.entrySet()) {
				switch(sheetPropEntry.getKey()) {
					case "visible" : {
						if(autoStyle==null) {
							autoStyle = createUniqueAutoStyle(sheet);
						}
						Boolean display = true;
        				if(sheetPropEntry.getValue() instanceof Boolean) {
        					display = (Boolean)sheetPropEntry.getValue();
        				}
        				OdfStylePropertiesBase propsElement = autoStyle.getOrCreatePropertiesElement(OdfStylePropertiesSet.TableProperties);
   					 	if(propsElement!=null) {
   					 		final TableDisplayAttribute displayAttr = new TableDisplayAttribute(content);
   					 		displayAttr.setBooleanValue(display);
   					 		propsElement.setOdfAttribute(displayAttr);
   					 	}
						break;
					}
					case "selectedRanges" : {
						ranges = (JSONArray)sheetPropEntry.getValue();
						break;
					}
					case "activeIndex" : {
						activeIndex = (Integer)sheetPropEntry.getValue();
						break;
					}
					case "zoom" : {
						sheetViewSettings.addConfigItem("ZoomValue", new Double((((Double)sheetPropEntry.getValue()) * 100)).intValue());
						break;
					}
					case "scrollLeft" : {
						sheetViewSettings.addConfigItem("PositionLeft", (Integer)sheetPropEntry.getValue());
						break;
					}
					case "scrollTop" : {
						sheetViewSettings.addConfigItem("PositionBottom", (Integer)sheetPropEntry.getValue());
						break;
					}
					case "scrollBottom" : {
						sheetViewSettings.addConfigItem("PositionTop", (Integer)sheetPropEntry.getValue());
						break;
					}
					case "scrollRight" : {
						sheetViewSettings.addConfigItem("PositionRight", (Integer)sheetPropEntry.getValue());
						break;
					}
					case "showGrid" : {
						sheetViewSettings.addConfigItem("ShowGrid", (Boolean)sheetPropEntry.getValue());
						break;
					}
				}
			}
			if(ranges!=null&&ranges.length()>=(activeIndex+1)) {
				final JSONObject range = ranges.getJSONObject(activeIndex);
				final JSONArray start = range.getJSONArray("start");
				sheetViewSettings.addConfigItem("CursorPositionX", start.getInt(0));
				sheetViewSettings.addConfigItem("CursorPositionY", start.getInt(1));
			}
    	}
    }

    static public void mapProperties(OdfStyleFamily styleFamily, JSONObject attrs, OdfStyleBase style, OdfDocument doc4Fonts)
    	throws JSONException {

        if (attrs != null && styleFamily != null && style != null) {
            Map<String, OdfStylePropertiesSet> familyProperties = Component.getAllOxStyleGroupingIdProperties(styleFamily);
            Set<String> propTypes = familyProperties.keySet();
            for (String type : propTypes) {
                if (type.equals("character") && attrs.hasAndNotNull("character")) {
                    JSONObject textProps = (JSONObject) attrs.opt("character");
                    if (textProps != null && textProps.length() > 0) {
                        OdfStylePropertiesBase propsElement = style.getOrCreatePropertiesElement(OdfStylePropertiesSet.TextProperties);
                        org.odftoolkit.odfdom.component.JsonOperationConsumer.mapCharacterProperties(textProps, (StyleTextPropertiesElement) propsElement, doc4Fonts);
                    }
                } else if (type.equals("paragraph") && attrs.hasAndNotNull("paragraph")) {
                    JSONObject paraProps = (JSONObject) attrs.opt("paragraph");
                    if (paraProps != null && paraProps.length() > 0) {
                        OdfStylePropertiesBase propsElement = style.getOrCreatePropertiesElement(OdfStylePropertiesSet.ParagraphProperties);
                        org.odftoolkit.odfdom.component.JsonOperationConsumer.mapParagraphProperties(paraProps, (StyleParagraphPropertiesElement) propsElement);
                    }
                } else if (type.equals("table")) {
                    if (attrs.hasAndNotNull("table")) {
                        JSONObject tableProps = (JSONObject) attrs.opt("table");
                        if (tableProps != null && tableProps.length() > 0) {
                            OdfStylePropertiesBase propsElement = style.getOrCreatePropertiesElement(OdfStylePropertiesSet.TableProperties);
                            org.odftoolkit.odfdom.component.JsonOperationConsumer.mapTableProperties(tableProps, (StyleTablePropertiesElement) propsElement);
                        }
                    } else if (attrs.hasAndNotNull("sheet")) {
                        // currently the sheet are handled different than the tableElement
                        JSONObject sheetProps = (JSONObject) attrs.opt("sheet");
                        if (sheetProps != null && sheetProps.length() > 0) {
                            OdfStylePropertiesBase propsElement = style.getOrCreatePropertiesElement(OdfStylePropertiesSet.TableProperties);
                            org.odftoolkit.odfdom.component.JsonOperationConsumer.mapTableProperties(sheetProps, (StyleTablePropertiesElement) propsElement);
                        }
                    } else {
                        // some default values have to be set (width 100% for MSO15)
                        OdfStylePropertiesBase propsElement = style.getOrCreatePropertiesElement(OdfStylePropertiesSet.TableProperties);
                        org.odftoolkit.odfdom.component.JsonOperationConsumer.mapTableProperties(null, (StyleTablePropertiesElement) propsElement);
                    }
                } else if (type.equals("row") && attrs.hasAndNotNull("row")) {
                    JSONObject props = (JSONObject) attrs.opt("row");
                    if (props != null && props.length() > 0) {
                        OdfStylePropertiesBase propsElement = style.getOrCreatePropertiesElement(OdfStylePropertiesSet.TableRowProperties);
                        org.odftoolkit.odfdom.component.JsonOperationConsumer.mapRowProperties(props, (StyleTableRowPropertiesElement) propsElement);
                    }
                } else if (type.equals("cell") && attrs.hasAndNotNull("cell")) {
                    JSONObject props = (JSONObject) attrs.opt("cell");
                    if (props != null && props.length() > 0) {
                        OdfStylePropertiesBase propsElement = style.getOrCreatePropertiesElement(OdfStylePropertiesSet.TableCellProperties);
                        mapCellProperties(props, (StyleTableCellPropertiesElement) propsElement, style);
                    }
                } else if (type.equals("column") && attrs.hasAndNotNull("column")) {
                    JSONObject props = (JSONObject) attrs.opt("column");
                    if (props != null && props.length() > 0) {
                        OdfStylePropertiesBase propsElement = style.getOrCreatePropertiesElement(OdfStylePropertiesSet.TableColumnProperties);
                        org.odftoolkit.odfdom.component.JsonOperationConsumer.mapColumnProperties(props, (StyleTableColumnPropertiesElement) propsElement);
                    }
                } else if (type.equals("list") && attrs.hasAndNotNull("list")) {
                    JSONObject props = (JSONObject) attrs.opt("list");
                    if (props != null && props.length() > 0) {
                        OdfStylePropertiesBase propsElement = style.getOrCreatePropertiesElement(OdfStylePropertiesSet.ListLevelProperties);
                        org.odftoolkit.odfdom.component.JsonOperationConsumer.mapListProperties(props, (StyleListLevelPropertiesElement) propsElement);
                    }
                } else if (type.equals("section") && attrs.hasAndNotNull("section")) {
                    JSONObject props = (JSONObject) attrs.opt("section");
                    if (props != null && props.length() > 0) {
                        OdfStylePropertiesBase propsElement = style.getOrCreatePropertiesElement(OdfStylePropertiesSet.SectionProperties);
                        org.odftoolkit.odfdom.component.JsonOperationConsumer.mapSectionProperties(props, (StyleSectionPropertiesElement) propsElement);
                    }

                } else if (type.equals("drawing") || type.equals("presentation")) {
                    if (attrs.has("drawing") || attrs.has("shape")|| attrs.has("line")|| attrs.has("fill") ) {
                        JSONObject allDrawingProperties = new JSONObject();
                        String subs[] = {"shape", "drawing"};
                        for( String sub : subs ) {
                            if(attrs.has(sub)) {
                                JSONObject subAttrs = attrs.getJSONObject(sub);
                                Iterator<String> keyIt = subAttrs.keys();
                                while( keyIt.hasNext()) {
                                    String key = keyIt.next();
                                    allDrawingProperties.put(key, subAttrs.get(key));
                                }
                            }
                        }
                        if( attrs.has("fill") ) {
                            allDrawingProperties.put("fill", attrs.getJSONObject("fill"));
                        }
                        if( attrs.has("line") ) {
                            allDrawingProperties.put("line", attrs.getJSONObject("line"));
                        }
                        if (allDrawingProperties.length() > 0) {
                            OdfStyleBase parentStyle = style.getParentStyle();
                            OdfStylePropertiesBase propsElement = style.getOrCreatePropertiesElement(OdfStylePropertiesSet.GraphicProperties);
                            org.odftoolkit.odfdom.component.JsonOperationConsumer.mapGraphicProperties(allDrawingProperties, (StyleGraphicPropertiesElement) propsElement, parentStyle);
                        }
                    }

// ToDo: How to differentiate Graphics from Drawings? ( see condition before - images have only GraphicProperties!)
//				} else if (type.equals("drawing")) {
//					JSONObject props = (JSONObject) attrs.opt("drawing");
//					if (props != null && props.length() > 0) {
//						OdfStylePropertiesBase propsElement = style.getOrCreatePropertiesElement(OdfStylePropertiesSet.DrawingPageProperties);
//						mapDrawingProperties(props, (StyleDrawingPagePropertiesElement) propsElement);
//					}
                } else if (type.equals("chart") || type.equals("chart")) {
                    JSONObject props = (JSONObject) attrs.opt("chart");
                    if (props != null && props.length() > 0) {
                        OdfStylePropertiesBase propsElement = style.getOrCreatePropertiesElement(OdfStylePropertiesSet.ChartProperties);
                        org.odftoolkit.odfdom.component.JsonOperationConsumer.mapChartProperties(props, (StyleChartPropertiesElement) propsElement);
                    }
                } else if (type.equals("page") || type.equals("page")) {
                    JSONObject props = (JSONObject) attrs.opt("page");
                    if (props != null && props.length() > 0) {
                        OdfStylePropertiesBase propsElement = style.getOrCreatePropertiesElement(OdfStylePropertiesSet.PageLayoutProperties);
                        org.odftoolkit.odfdom.component.JsonOperationConsumer.mapPageProperties(props, (StylePageLayoutPropertiesElement) propsElement);
                    }
                } else if (type.equals("ruby") || type.equals("ruby")) {
                    JSONObject props = (JSONObject) attrs.opt("ruby");
                    if (props != null && props.length() > 0) {
                        OdfStylePropertiesBase propsElement = style.getOrCreatePropertiesElement(OdfStylePropertiesSet.RubyProperties);
                        org.odftoolkit.odfdom.component.JsonOperationConsumer.mapRubyProperties(props, (StyleRubyPropertiesElement) propsElement);
                    }
                } else if (type.equals("headerFooter") || type.equals("headerFooter")) {
                    JSONObject props = (JSONObject) attrs.opt("headerFooter");
                    if (props != null && props.length() > 0) {
                        OdfStylePropertiesBase propsElement = style.getOrCreatePropertiesElement(OdfStylePropertiesSet.HeaderFooterProperties);
                        org.odftoolkit.odfdom.component.JsonOperationConsumer.mapHeaderFooterProperties(props, (StyleHeaderFooterPropertiesElement) propsElement);
                    }
                }
            }
        }
    }

    static public void mapCellProperties(JSONObject attrs, StyleTableCellPropertiesElement propertiesElement, OdfStyleBase style)
    	throws JSONException {

    	if (attrs!=null) {
    		String newCellProtect = null;
    		final Set<Entry<String, Object>> entrySet = attrs.entrySet();
            for (Entry<String, Object> entry : entrySet) {

            	final String key = entry.getKey();
            	final Object value = entry.getValue();

            	// No margin in ODF
            	org.odftoolkit.odfdom.component.JsonOperationConsumer.addBorderProperties(key, attrs, propertiesElement, null);
            	org.odftoolkit.odfdom.component.JsonOperationConsumer.addPaddingProperties(key, attrs, propertiesElement);
                if (key.equals("fillColor")) {
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "background-color");
                    } else {
                        JSONObject color = (JSONObject) value;
                        propertiesElement.setFoBackgroundColorAttribute(org.odftoolkit.odfdom.component.JsonOperationConsumer.getColor(color, MapHelper.TRANSPARENT));
                    }
                }
                else if( key.equals("alignVert")){
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "vertical-align");
                    } else {
                        String align = (String)value;
                        propertiesElement.setStyleVerticalAlignAttribute(align);
                    }
                }
                else if(key.equals("alignHor")){
                    OdfStylePropertiesBase propsElement = style.getOrCreatePropertiesElement(OdfStylePropertiesSet.ParagraphProperties);
                    if (value == null || value.equals(JSONObject.NULL)) {
                    	propsElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "text-align");
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "text-align-source");
                    } else {
                        String align = (String)value;
                        if(align.equals("right")) {
                        	align = "end";
                        }
                        else if(align.equals("left")) {
                        	align = "start";
                        }
                        propsElement.setAttributeNS(OdfDocumentNamespace.FO.getUri(), "fo:text-align", align);
                        propertiesElement.setAttributeNS(OdfDocumentNamespace.FO.getUri(), "style:text-align-source", "fix");
                    }
                }
                else if(key.equals("wrapText")) {
					if(value==null||!(Boolean)value) {
						propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "wrap-option");
					}
					else {
						propertiesElement.setAttributeNS(OdfDocumentNamespace.FO.getUri(), "fo:wrap-option", (Boolean)value ? "wrap" : "no-wrap");
					}
                }
                else if(key.equals("unlocked")) {
                	final String oldCellProtect = propertiesElement.getAttributeNS(StyleCellProtectAttribute.ATTRIBUTE_NAME.getUri(), StyleCellProtectAttribute.ATTRIBUTE_NAME.getLocalName());
                	if(newCellProtect==null) {
                		newCellProtect = "none";
                	}
                	if(value instanceof Boolean) {
                		if((Boolean)value) {
                			if(oldCellProtect.equals("hidden-and-protected")) {
                				newCellProtect = "formula-hidden";
                			}
                		}
                		else {
                			if(oldCellProtect.equals("formula-hidden")) {
                				newCellProtect = "hidden-and-protected";
                			}
                			else if(oldCellProtect.equals("hidden-and-protected")) {
                				newCellProtect = "hidden-and-protected";
                			}
                			else {
                				newCellProtect = "protected";
                			}
                		}
                	}
                	
                }
                else if(key.equals("hidden")) {
                	final String oldCellProtect = propertiesElement.getAttributeNS(StyleCellProtectAttribute.ATTRIBUTE_NAME.getUri(), StyleCellProtectAttribute.ATTRIBUTE_NAME.getLocalName());
                	if(newCellProtect==null) {
                		newCellProtect = "none";
                	}
                	if(value instanceof Boolean) {
                		if((Boolean)value) {
                			if(oldCellProtect.equals("hidden-and-protected")) {
                				newCellProtect = oldCellProtect;
                			}
                			else if(oldCellProtect.equals("protected")) {
                				newCellProtect = "hidden-and-protected";
                			}
                			else {
                				newCellProtect = "formula-hidden";
                			}
                		}
                		else {
                			if(oldCellProtect.equals("hidden-and-protected")) {
                				newCellProtect = "protected";
                			}
                		}
                	}
                }
            }
            if(newCellProtect!=null) {
            	propertiesElement.setAttributeNS(StyleCellProtectAttribute.ATTRIBUTE_NAME.getUri(), StyleCellProtectAttribute.ATTRIBUTE_NAME.getQName(), newCellProtect);
            }
        }
    }
    
//	/**
//	 * Adds a new none automatic style to the document. If the style is hidden a
//	 * default style, otherwise a template style will be added.
//	 * @param doc4Fonts  temporary font properties taken into adapter
//	 */
    public void insertStyleSheet(OdfDocument doc4Fonts, String type, String styleId, String styleName, JSONObject attrs, String parent, boolean auto, boolean hidden)
   		throws JSONException {

       if (attrs!=null) {
           final OdfStyleFamily styleFamily = Component.getFamily(type);
    	   final OdfStylesBase styles = auto ? content.getOrCreateAutomaticStyles() : doc4Fonts.getDocumentStyles();
           if(!auto&&hidden) { // DEFAULT STYLE
               // Removing any existing default style
               OdfDefaultStyle oldDefaultStyle = ((OdfOfficeStyles)styles).getDefaultStyle(styleFamily);
               if (oldDefaultStyle!=null) {
                   styles.removeChild(oldDefaultStyle);
               }
               // Adding new default style
               StyleDefaultStyleElement defaultStyleElement = ((OdfOfficeStyles)styles).newStyleDefaultStyleElement(styleFamily.getName());
               mapProperties(styleFamily, attrs, defaultStyleElement, doc4Fonts);
           }
           else { // TEMPLATE STYLE
               OdfStyle oldStyle = styles.getStyle(styleName, styleFamily);
               if (oldStyle!=null) {
                   styles.removeChild(oldStyle);
               } else {
                   /* Workaround for OOo applicatoins
                    * Applications from the OpenOffice family are exchanging any space in their names with an '_20_' string.
                    * Unfortunately they are using them equivalent in general, therefore the names 'Heading_20_2' and 'Heading 2' would result and into a name clash */
                   if (styleName != null && styleName.contains(" ")) {
                       String testName = styleName.replace(" ", "_20_");
                       oldStyle = styles.getStyle(testName, styleFamily);
                       if (oldStyle != null) {
                           styles.removeChild(oldStyle);
                       }
                   }
               }
               StyleStyleElement style = styles.newStyle(styleId, styleFamily);
               if (parent != null && !parent.isEmpty()) {
                   style.setStyleParentStyleNameAttribute(parent);
               }
               if (styleName != null && !styleName.isEmpty()) {
                   style.setStyleDisplayNameAttribute(styleName);
               }
               mapProperties(styleFamily, attrs, style, doc4Fonts);

               // apply numberFormat - uses exisiting 'Number' style or create a new one
               if(attrs.has("cell") && attrs.optJSONObject("cell").has("numberFormat")) {
                   JSONObject numFormat = attrs.optJSONObject("cell").optJSONObject("numberFormat");
                   if(numFormat!=null ) {
                       String code = numFormat.optString("code", "");
                       int id = numFormat.optInt("id", -1);
                       String dataStyleName = MapHelper.findOrCreateDataStyle(code, id, content);
                       style.setAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "style:data-style-name", dataStyleName);
                   } else {
                       style.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "data-style-name");
                   }
               }
           }
       }
   }    
}
