/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ods.dom;

import java.util.List;

import org.odftoolkit.odfdom.dom.OdfSchemaDocument;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Node;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;

import com.openexchange.office.FilterException;
import com.openexchange.office.FilterException.ErrorCode;

/**
 * @author sven.jacobi@open-xchange.com
 */
public class ContentHandler extends DOMBuilder {

    private static final Logger LOG = LoggerFactory.getLogger(ContentHandler.class);

    // the empty XML file to which nodes will be added
    private final Content contentDom;
    private OdfSchemaDocument mSchemaDoc = null;
    private int maxSheets;

    public ContentHandler(Node rootNode, XMLReader xmlReader)
    	throws SAXException {

    	super(rootNode, xmlReader, (Content)rootNode);

        // Initialize starting DOM node
        contentDom = (Content)rootNode;
        mSchemaDoc = contentDom.getDocument();
        mSchemaDoc.setContentDom(contentDom);
        Object oMaxSheets = contentDom.getDocument().getPackage().getRunTimeConfiguration().get("maxSheets");
        maxSheets = oMaxSheets instanceof Integer ? ((Integer)oMaxSheets).intValue() : 256;
    }

    public Content getContentDom() {
    	return contentDom;
    }

    @Override
    public SaxContextHandler startElement(Attributes attributes, String uri, String localName, String qName)
    	throws SAXException {

    	if(qName.equals("table:table")) {
    		if(maxSheets>0) {
    			if(contentDom.getSheets().size()>=maxSheets) {
		        	final FilterException filterException = new FilterException("", ErrorCode.COMPLEXITY_TOO_HIGH);
		        	filterException.setSubType("MAX_SHEET_COUNT_EXCEEDED");
		        	throw filterException;
    			}
    		}
    		String sheetStyle = null;
    		for(int i=0; i<attributes.getLength(); i++) {
				if(attributes.getLocalName(i).equals("style-name")) {
					sheetStyle = attributes.getValue(i);
					break;
				}
    		}
    		final Sheet sheet = new Sheet(contentDom, sheetStyle);
    		appendChild(ElementNS.addAttributes(getFileDom(), sheet, attributes));
    		contentDom.getSheets().add(sheet);
    		return new SheetHandler(this, sheet);
    	}
    	else if(qName.equals("table:named-expressions")) {
    		final NamedExpressions namedExpressions = contentDom.getNamedExpressions(true);
    		return new NamedExpressionsHandler(this, namedExpressions);
    	}
    	else if(qName.equals("table:database-ranges")) {
    		final DatabaseRanges databaseRanges = contentDom.getDatabaseRanges(true);
    		return new DatabaseRangesHandler(this, databaseRanges);
    	}
/*
    	else if(qName.equals("table:content-validations")) {
    		final ContentValidations contentValidations = contentDom.getContentValidations(true);
    		return new ContentValidationsHandler(this, contentValidations);
    	}
*/
    	// if we come to this postion, the element is inserted normally into the dom...
    	final SaxContextHandler newContext = super.startElement(attributes, uri, localName, qName);
    	if(qName.equals("office:automatic-styles")) {
    		// we want to have a special context for automatic styles
    		return new OfficeAutomaticStylesHandler(getCurrentNode(), newContext, this.getFileDom(), contentDom.getStyles());
    	}
    	return newContext;
    }

    @Override
    public void endElement(String localName, String qName)
    	throws SAXException {
    	super.endElement(localName, qName);

    	if(qName.equals("table:table")) {
    		final List<Sheet> sheets = contentDom.getSheets();
    		final int lastIndex = sheets.size()-1;
    		if(sheets.get(lastIndex).isLinkedTable()) {
        		sheets.remove(lastIndex);
    		}
    	}
    }
}
