/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.json.actions;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.documentation.RequestMethod;
import com.openexchange.documentation.annotations.Action;
import com.openexchange.documentation.annotations.Parameter;
import com.openexchange.groupware.upload.UploadFile;
import com.openexchange.office.json.tools.RequestDataHelper;
import com.openexchange.office.tools.StorageHelper;
import com.openexchange.office.tools.StreamInfo;
import com.openexchange.office.tools.files.DocFileHelper;
import com.openexchange.office.tools.resource.Resource;
import com.openexchange.office.tools.resource.ResourceManager;
import com.openexchange.server.ServiceLookup;
import com.openexchange.tools.encoding.Base64;
import com.openexchange.tools.session.ServerSession;

/**
 * {@link AddFileAction}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
@Action(method = RequestMethod.POST, name = "addfile", description = "Get the whole document as PDF file or get the substream of an infostore file in its naive format.", parameters = {
    @Parameter(name = "session", description = "A session ID previously obtained from the login module."),
    @Parameter(name = "add_fileid", description = "Object ID of the requested infoitem."),
    @Parameter(name = "add_folderid", description = "Folder ID of the requested infoitem."),
    @Parameter(name = "uid", description = "The unique id of the client application."),
    @Parameter(name = "version", optional=true, description = "If present, the infoitem data describes the given version. Otherwise the current version is returned."),
    @Parameter(name = "filename", optional=true, description = "If present, this parameter contains the name of the infostore item to be used as initial part of the filename for the returned file, other the filename is set to 'file'."),
    @Parameter(name = "add_filename", optional=true, description = "If present, this parameter contains the name of the just uploaded file to be set as filename for the new file content of the document. Otherwise a system time based name is generated.")
}, responseDescription = "Response with timestamp: A JSON object containing the actual file metadata and the fragment name of the just uploaded file within the document: 'id', 'folder_id', 'version' 'filename' and 'added_fragment'.")

public class AddFileAction extends DocumentFilterAction {

    private static final Log LOG = com.openexchange.log.Log.loggerFor(AddFileAction.class);

    static final protected String m_resourceIdKey = "added_fileid";
    static final protected String m_resourceNameKey = "added_filename";
    static final protected String m_connectionResourceIdKey = "resourceid";

    /**
     * Initializes a new {@link AddFileAction}.
     *
     * @param services
     * @param oqm
     */
    public AddFileAction(ServiceLookup services, ResourceManager resMgr) {
        super(services);
        resourceManager = resMgr;
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.ajax.requesthandler.AJAXActionService#perform(com.openexchange.ajax.requesthandler.AJAXRequestData,
     * com.openexchange.tools.session.ServerSession)
     */
    @Override
    public AJAXRequestResult perform(AJAXRequestData request, ServerSession session) {
        final RequestDataHelper requestDataHelper = new RequestDataHelper(request);
        final JSONObject jsonObj = new JSONObject();
        AJAXRequestResult ajaxResult = null;
        Integer crc32 = null;

        try {
            final String p = requestDataHelper.getParameter("add_crc32");
            if (p != null) {
                crc32 = Integer.valueOf((int) (Long.parseLong(p, 16)));
            }
        } catch (NumberFormatException e) {
            //
        }

        final String extension = requestDataHelper.getParameter("add_ext");
        final String data64 = requestDataHelper.getParameter("add_filedata");
        final String fileId = requestDataHelper.getParameter("add_fileid");
        final String folderId = requestDataHelper.getParameter("add_folderid");
        byte[] data = null;

        if ((null != fileId) && (null != folderId)) {
            final DocFileHelper fileHelper = new DocFileHelper(m_services);
            final StorageHelper storageHelper = new StorageHelper(m_services, session, folderId);
            final StreamInfo streamInfo = fileHelper.getDocumentStream(session, storageHelper, folderId, fileId);
            final InputStream inputStream = streamInfo.getDocumentStream();
            if (null != inputStream) {
                try {
                    data = IOUtils.toByteArray(inputStream);
                } catch (IOException e) {
                    LOG.error("Could not read file stream", e);
                } finally {
                    IOUtils.closeQuietly(inputStream);
                    try {
                        streamInfo.close();
                    } catch (IOException e) {
                        LOG.warn("Could not correctly close IDBasedFileAccess instance", e);
                    }
                }
            } else {
                try {
                    streamInfo.close();
                } catch (IOException e) {
                    LOG.warn("Could not correctly close IDBasedFileAccess instance", e);
                }
            }
        } else {
            if (null != data64) {

                final String searchPattern = "base64,";
                int pos = data64.indexOf(searchPattern);

                if ((-1 != pos) && ((pos += searchPattern.length()) < (data64.length() - 1))) {
                    data = Base64.decode(data64.substring(pos));
                }
            } else { // in IE9 data64 is not readable from request
                InputStream uploadStm = null;

                try {
                    if (request.hasUploads()) {
                        final UploadFile uploadFile = request.getFiles().get(0);
                        uploadStm = new FileInputStream(uploadFile.getTmpFile());
                    }
                } catch (final Exception e) {
                    LOG.error("Could not create file stream", e);
                }

                if (null != uploadStm) {
                    try {
                        data = IOUtils.toByteArray(uploadStm);
                    } catch (IOException e) {
                        LOG.error("Could not read file stream", e);
                    } finally {
                        IOUtils.closeQuietly(uploadStm);
                    }
                }
            }
        }

        if ((data != null) && (data.length > 0)) {

            if (crc32 == null)
                crc32 = Integer.valueOf(Resource.getCRC32(data));

            long uid = resourceManager.addResource(crc32.intValue(), data);

            try {
                String resname = null;
                switch (DocFileHelper.getDocumentFormat(request.getParameter("filename"))) {
                case DOCX: {
                    resname = "word/media/uid" + Long.toHexString(uid) + "." + extension;
                    break;
                }
                case PPTX: {
                    resname = "ppt/media/uid" + Long.toHexString(uid) + "." + extension;
                    break;
                }
                case ODT: case ODS: {
                    resname = "Pictures/uid" + Long.toHexString(uid) + "." + extension;
                    break;
                }
                case XLSX: {
                    resname = "xl/media/uid" + Long.toHexString(uid) + "." + extension;
                    break;
                }

                case ODG:
                case ODP:
                case NONE:
                default:
                    break;
                }

                if (resname != null) {
                    final Resource resource = resourceManager.getResource(uid);
                    String managedResourceId = null;

                    // add the managed file id to the result object
                    if ((null != resource) && (null != (managedResourceId = resource.getManagedId()))) {
                        jsonObj.put(m_resourceIdKey, managedResourceId);
                    }

                    // add the filename to the result object
                    jsonObj.put(m_resourceNameKey, resname);
                }
            } catch (Exception e1) {
                LOG.error("Could not add to resource manager", e1);
            }
        }

        // set resource id at connection object; return the
        // object with fragment name and id in case of success
        if (jsonObj.has(m_resourceIdKey)) {
            final JSONObject requestData = new JSONObject();

            try {
                requestData.put(m_connectionResourceIdKey, jsonObj.getString(m_resourceIdKey));

                JSONObject resultData = sendConnectionMessageSynchronously(request, session, "addResource", "resource", requestData);

                if ((null != resultData) && resultData.has(m_connectionResourceIdKey) && (resultData.getString(m_connectionResourceIdKey).length() > 0) && jsonObj.has(m_resourceNameKey)) {
                    resultData.put(m_resourceNameKey, jsonObj.get(m_resourceNameKey));
                } else {
                    resultData = null;
                }

                ajaxResult = getAjaxRequestResult(resultData);
            } catch (JSONException e) {
                LOG.error("Could not send resource to realtime session", e);
            }
        }

        return ajaxResult;
    }

    // - Members ---------------------------------------------------------------

    private ResourceManager resourceManager = null;
}
