/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.backup.restore;

import java.io.ByteArrayInputStream;
import java.io.InputStream;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.exception.OXException;
import com.openexchange.file.storage.DefaultFile;
import com.openexchange.file.storage.File;
import com.openexchange.file.storage.FileStorageFileAccess;
import com.openexchange.file.storage.FileStoragePermission;
import com.openexchange.file.storage.composition.IDBasedFileAccess;
import com.openexchange.file.storage.composition.IDBasedFolderAccess;
import com.openexchange.log.Log;
import com.openexchange.office.FilterException;
import com.openexchange.office.IExporter;
import com.openexchange.office.IImporter;
import com.openexchange.office.backup.manager.DocumentBackupController;
import com.openexchange.office.backup.manager.DocumentRestoreData;
import com.openexchange.office.doc.imexport.ImExportHelper;
import com.openexchange.office.doc.imexport.ImporterPropertyHelper;
import com.openexchange.office.tools.StorageHelper;
import com.openexchange.office.tools.directory.DocRestoreID;
import com.openexchange.office.tools.doc.OXDocument;
import com.openexchange.office.tools.doc.OXDocument.ResolvedStreamInfo;
import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.office.tools.error.ExceptionToErrorCode;
import com.openexchange.office.tools.error.FilterExceptionToErrorCode;
import com.openexchange.office.tools.files.DocFileHelper;
import com.openexchange.office.tools.files.FileHelper;
import com.openexchange.office.tools.files.FolderHelper;
import com.openexchange.office.tools.json.JSONHelper;
import com.openexchange.office.tools.logging.ELogLevel;
import com.openexchange.office.tools.logging.LogWithLevel;
import com.openexchange.office.tools.message.OperationHelper;
import com.openexchange.office.tools.monitoring.RestoreDocEvent;
import com.openexchange.office.tools.monitoring.RestoreDocEventType;
import com.openexchange.office.tools.monitoring.Statistics;
import com.openexchange.office.tools.resource.ResourceManager;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;

public class RestoreDocument implements IRestoreDocument {

    @SuppressWarnings("deprecation")
    static private final org.apache.commons.logging.Log LOG = Log.loggerFor(RestoreDocument.class);

    private final ServiceLookup services;

    private enum RestoreMessage {
        RESTORE_REQUESTED,
        RESTORE_SUCCESS,
        RESTORE_FAILED
    }

    public RestoreDocument(final ServiceLookup services) {
        this.services = services;
    }

    /**
     * Tries to save a new document using a base document stream, stored actions and provided actions by a client.
     * The code tries to find out where the client derivates from the stored actions and combine it to a new valid
     * operations stream. This finally will be used to store the restored document.
     *
     * @param id
     * @param folderId
     * @param targetFolderId
     * @param initialFileName
     * @return
     * @throws OXException
     */
    @Override
    public final RestoreData saveAsWithBackupDocumentAndOperations(final Session session, final String id, final String restoreID, final String targetFolderId, final String initialFileName, final JSONArray operations) {
        final DocumentBackupController backupController = services.getService(DocumentBackupController.class);

        ErrorCode errorCode = ErrorCode.NO_ERROR;
        RestoreData restoreData = new RestoreData(errorCode, null, null, null);

        // log request
        logRestoreMessage(RestoreMessage.RESTORE_REQUESTED, restoreID, targetFolderId, errorCode);

        // there are two different formats of the operations
        // 1. operations array:
        //    [{op1}, {op2}, {op3}, {op4}, ...]
        // 2. operations encapsulate in an array of operations
        //    [{operations: [{op1},{op2}, ...]}, {operations: [{op1}, {op2}, ....}]}
        // check if we have to convert the format -> we always want to provide type 1
        JSONArray actions = null;
        if ((null != operations) && (operations.length() > 0)) {
            try {
                final JSONObject obj = operations.getJSONObject(0);

                if (obj.optJSONArray("operations") != null) {
                    // we have format 2 must be converted
                    JSONArray realOpsArray = null;
                    actions = new JSONArray();

                    for (int i = 0; i < operations.length(); i++) {
                        final JSONObject opsObject = operations.getJSONObject(i);
                        realOpsArray = opsObject.getJSONArray("operations");
                        JSONHelper.appendArray(actions, realOpsArray);
                    }
                } else {
                    // easy way - no conversion
                    actions = operations;
                }
            } catch (JSONException e) {
                errorCode = ErrorCode.BACKUPDOCUMENT_CLIENTACTIONS_MALFORMED;
            }
        } else {
            // no client operations - make sure we always have an array
            // in this special case we have an empty one
            actions = new JSONArray();
        }

        IDBasedFolderAccess folderAccess;
        try{
            folderAccess = FolderHelper.getFolderAccess(services, session);
        }catch (OXException e){
            LOG.warn("RT connection: Filter exception caught trying to restore document. FolderHelper.getFolderAccess did not work correctly", e);
            errorCode = ErrorCode.GENERAL_UNKNOWN_ERROR;
            folderAccess = null;
        }

        if (errorCode.isNoError() && (null != backupController)) {
            // BE CAREFUL - we receive a plain id via string which is not encoded - make sure
            // to use the correct function to create the DocResourceID
            final DocRestoreID docRestoreID = DocRestoreID.create(restoreID);
            final DocumentRestoreData docRestoreData = backupController.getDocumentRestoreData(docRestoreID);

            if ((null != docRestoreData) && (docRestoreData.getErrorCode().isNoError())) {
                // in case of success we can try to restore the client document using the backup document stream
                // and all stored operations
                //final String baseVersion = docRestoreData.getBaseVersion();
                final int baseOSN = docRestoreData.getBaseOSN();
                final JSONArray docAppliedOps = docRestoreData.getOperations();
                final int currOSN = docRestoreData.getOSN();
                final String origFileName = docRestoreData.getFileName();
                final String toRestoreFileName = StringUtils.isEmpty(initialFileName) ? docRestoreData.getFileName() : initialFileName;
                final String mimeType = docRestoreData.getMimeType();
                final ResourceManager resManager = docRestoreData.getResourceManager();

                try {
                    if (null != resManager) {
                        resManager.lockResources(true);
                    }

                    // make sure that we always provide a JSONArray instance (will be empty, if we don't have store ops)
                    final JSONArray storedDocAppliedOps = (null == docAppliedOps) ? new JSONArray() : docAppliedOps;
                    // use restore document class/interface to merge the operations from document & client
                    final JSONArray operationsToBeStored = this.getRestoredOperations(baseOSN, currOSN, storedDocAppliedOps, actions);

                    final InputStream baseDocStream = docRestoreData.getInputStream();
                    restoreData = this.saveRestoredDocumentOperations(session, baseDocStream, resManager, origFileName, targetFolderId, toRestoreFileName, mimeType, operationsToBeStored);

                    // set error code from restore data
                    errorCode = restoreData.getErrorCode();
                } catch (RestoreException e) {
                    LOG.warn("RT connection: Restore exception caught trying to restore document with Hazelcast data", e);
                    errorCode = ErrorCode.BACKUPDOCUMENT_SYNC_NOT_POSSIBLE;
                } catch (FilterException e) {
                    LOG.warn("RT connection: Filter exception caught trying to restore document with Hazelcast data", e);
                    errorCode = FilterExceptionToErrorCode.map(e, ErrorCode.SAVEDOCUMENT_FAILED_ERROR);
                } catch (Exception e) {
                    LOG.warn("RT connection: Exception caught trying to restore document with Hazelcast data", e);
                    errorCode = ErrorCode.GENERAL_UNKNOWN_ERROR;
                } finally {
                    if (null != resManager) {
                        resManager.lockResources(false);
                    }
                }
            } else {
                // We have no data found using our local DocumentBackupManager nor using the
                // global Hazelcast map. As a fall-back try to use the document file as the base.
                final ResourceManager resourceManager = new ResourceManager(services);
                final DocFileHelper docFileHelper = new DocFileHelper(services);
                final String folderId = FolderHelper.getFolderId(folderAccess, id);
                final StorageHelper storageHelper = new StorageHelper(services, session, folderId);
                final OXDocument oxDoc = new OXDocument(session, services, folderId, id, docFileHelper, false, resourceManager, storageHelper, false, null);

                // check error state provided by OXDocuments - a non-existent file
                // would be detected here
                if (oxDoc.getLastError().isNoError()) {
                    try {

                        final JSONObject importerProps = new JSONObject();
                        final IImporter importer = ImExportHelper.getImporter(services, docFileHelper);

                        ImporterPropertyHelper.setMemoryImporterProperties(importerProps);
                        final JSONObject operationsObject = oxDoc.getOperations(importer, importerProps, null);
                        if (null != operationsObject) {
                            final JSONArray docOperations = operationsObject.getJSONArray(OperationHelper.KEY_OPERATIONS);
                            final int baseOSN = (oxDoc.getUniqueDocumentId() != -1) ? oxDoc.getUniqueDocumentId() : operationsObject.getJSONArray(OperationHelper.KEY_OPERATIONS).length();
                            final int currOSN = OperationHelper.getFollowUpOSN(operationsObject);

                            // use restore document class/interface to merge the operations from document & client
                            // If we use a stored document, we cannot really merge but we have to hope that the
                            // client starts exactly with the next follow-up operation state number!!
                            final JSONArray operationsToBeStored = this.getRestoredOperations(baseOSN, currOSN, docOperations, actions);

                            final String mimeType = oxDoc.getMetaData().getFileMIMEType();
                            final ResourceManager resManager = new ResourceManager(services);
                            final InputStream docStream = new ByteArrayInputStream(oxDoc.getDocumentBuffer());
                            restoreData = saveRestoredDocumentOperations(session, docStream, resManager, oxDoc.getMetaData().getFileName(), targetFolderId, initialFileName, mimeType, operationsToBeStored);
                            errorCode = restoreData.getErrorCode();
                        } else {
                            errorCode = ErrorCode.BACKUPDOCUMENT_SYNC_NOT_POSSIBLE;
                        }
                    } catch (final FilterException e) {
                        LOG.warn("RT connection: Filter exception caught trying to restore document", e);
                        errorCode = ErrorCode.BACKUPDOCUMENT_SYNC_NOT_POSSIBLE;
                    } catch (final JSONException e) {
                        LOG.warn("RT connection: JSON exception caught trying to restore document", e);
                        errorCode = ErrorCode.BACKUPDOCUMENT_SYNC_NOT_POSSIBLE;
                    } catch (RestoreException e) {
                        LOG.warn("RT connection: Restore exception caught trying to restore document", e);
                        errorCode = ErrorCode.BACKUPDOCUMENT_SYNC_NOT_POSSIBLE;
                    }
                } else {
                    errorCode = oxDoc.getLastError();
                }
            }
        } else if (errorCode.isNoError()) {
            errorCode = ErrorCode.BACKUPDOCUMENT_SERVICE_NOT_AVAILABLE;
        }

        if (errorCode.isError()) {
            logRestoreMessage(RestoreMessage.RESTORE_FAILED, restoreID, targetFolderId, errorCode);
            Statistics.handleRestoreDocEvent(new RestoreDocEvent(RestoreDocEventType.RESTORED_ERROR));
        } else {
            logRestoreMessage(RestoreMessage.RESTORE_SUCCESS, restoreID, targetFolderId, errorCode);
            Statistics.handleRestoreDocEvent(new RestoreDocEvent(RestoreDocEventType.RESTORED_SUCCESS));
        }

        // set the latest error code
        restoreData.setErrorCode(errorCode);

        return restoreData;
    }

    /**
     * Creates the restore document file using the base document stream and the operations
     * for the restoring.
     *
     * @param session the session of the user requesting the restore of the document
     * @param baseDocStream the document stream of the existing base document
     * @param resManager the resource manager containing the needed image data
     * @param origFileName the original file name of the document file
     * @param targetFolderId the folder where the restore document should be written to
     * @param toRestoreFileName the file name of the restored document file
     * @param mimeType the mime type of the document
     * @param operationsToBeStored the operations to be applied for restoring the document content
     * @return an restore data instance with error code and other information
     */
    private RestoreData saveRestoredDocumentOperations(final Session session, final InputStream baseDocStream, final ResourceManager resManager, final String origFileName, String targetFolderId, final String toRestoreFileName, final String mimeType, final JSONArray operationsToBeStored) {
        RestoreData result = new RestoreData(ErrorCode.NO_ERROR, null, null, null);
        String restoredFileName = null;
        String restoredFileId = null;
        ErrorCode errorCode = ErrorCode.NO_ERROR;

        if (null != baseDocStream) {

            try {
                final IDBasedFolderAccess folderAccess = FolderHelper.getFolderAccess(services, session);

                if (!FolderHelper.providesFolderPermissions(folderAccess, targetFolderId, false, true, true)) {
                    // Fallback if we are not allowed to create a new file within the original folder
                    // Use the documents folder of the user as 2nd target folder
                    targetFolderId = DocFileHelper.getUserDocumentsFolderId(session, services);
                }

                int[] folderPermissions = FolderHelper.getFolderPermissions(FolderHelper.getFolderAccess(services, session), targetFolderId);

                if (null != folderPermissions) {
                    boolean canCreate = folderPermissions[FolderHelper.FOLDER_PERMISSIONS_FOLDER] >= FileStoragePermission.CREATE_OBJECTS_IN_FOLDER;
                    boolean canWrite = folderPermissions[FolderHelper.FOLDER_PERMISSIONS_WRITE] > FileStoragePermission.NO_PERMISSIONS;

                    // check permissions and bail out with error code
                    if (!canCreate) { result = new RestoreData(ErrorCode.GENERAL_PERMISSION_CREATE_MISSING_ERROR, restoredFileName, restoredFileId, targetFolderId); return result; }
                    if (!canWrite) { result = new RestoreData(ErrorCode.GENERAL_PERMISSION_WRITE_MISSING_ERROR, restoredFileName, restoredFileId, targetFolderId); return result; }

                    final StorageHelper storageHelper = new StorageHelper(services, session, targetFolderId);
                    final OXDocument oxDoc = new OXDocument(session, services, baseDocStream, origFileName, mimeType, resManager, storageHelper);
                    final IExporter exporter = ImExportHelper.getExporterService(services, DocFileHelper.getDocumentFormat(origFileName));

                    final ResolvedStreamInfo resolvedStreamInfo = oxDoc.getResolvedDocumentStream(exporter, resManager, operationsToBeStored, FileStorageFileAccess.CURRENT_VERSION, true, false);
                    if (null != resolvedStreamInfo) {
                        final WriteInfo info = writeRestoreDocumentFile(session, resolvedStreamInfo.resolvedStream, origFileName, toRestoreFileName, targetFolderId, mimeType);
                        errorCode = info.getErrorCode();
                        restoredFileName = info.getRestoredFileName();
                        restoredFileId = info.getRestoredFileId();

                        if (errorCode.isNoError()) {
                            errorCode = resolvedStreamInfo.errorCode;
                            // map the partial save to a backup specific error code to enable
                            // better error messages on the client side
                            if (resolvedStreamInfo.errorCode.getCode() == ErrorCode.SAVEDOCUMENT_FAILED_FILTER_OPERATION_ERROR.getCode()) {
                                errorCode = ErrorCode.BACKUPDOCUMENT_RESTORE_DOCUMENT_PARTIAL_WRITTEN;
                            }
                        }
                    } else {
                        errorCode = (null != resolvedStreamInfo) ? resolvedStreamInfo.errorCode : ErrorCode.BACKUPDOCUMENT_SYNC_NOT_POSSIBLE;
                    }
                } else {
                    errorCode = ErrorCode.GENERAL_PERMISSION_CREATE_MISSING_ERROR;
                }
            } catch (OXException e) {
                LOG.warn("RT connection: Exception caught trying to store restored document", e);
                errorCode = ExceptionToErrorCode.map(e, ErrorCode.GENERAL_UNKNOWN_ERROR, false);
            }

            result = new RestoreData(errorCode, restoredFileName, restoredFileId, targetFolderId);
        }

        return result;
    }

    /**
     * Creates a JSONArray with operations retrieved from the storedOperations and clientOperations
     * at the point where clientOperations derived from the storedOperations.
     *
     * @param baseOSN the operation state number of the first stored operation
     * @param currOSN the operation state number following the last stored operation
     * @param
     */
    @Override
    public JSONArray getRestoredOperations(int baseOSN, int currOSN, final JSONArray storedOperations, final JSONArray clientOperations) throws RestoreException {
        Validate.notNull(storedOperations);
        Validate.notNull(clientOperations);

        JSONArray operationsToBeRestored = null;

        try {
            if (JSONHelper.isNullOrEmpty(clientOperations)) {
                // in case of no client operations we just store the latest
                // changed document using the stored base stream and the stored
                // applied operations
                operationsToBeRestored = storedOperations;
            } else {
                // in case we have client operations we try to find out at
                // what point we have to apply them
                final JSONObject firstClientOp = clientOperations.getJSONObject(0);
                if (!JSONHelper.isNullOrEmpty(storedOperations)) {
                    final JSONObject firstBackupOp = storedOperations.getJSONObject(0);
                    final int firstClientOSN = firstClientOp.getInt("osn");
                    final int firstBackupOSN = firstBackupOp.getInt("osn");

                    if ((firstClientOSN >= baseOSN) && (firstClientOSN <= currOSN)) {
                        if (firstClientOSN == currOSN) {
                            // append client operations to the backup document operations
                            operationsToBeRestored = JSONHelper.appendArray(storedOperations, clientOperations);
                        } else {
                            operationsToBeRestored = new JSONArray();

                            if (firstClientOSN > firstBackupOSN) {
                                // copy the backup document operation up-to the branch point &
                                // append the client operations
                                int osn = 0;
                                int opl = 0;
                                int index = 0;
                                while ((index < storedOperations.length()) && (osn < firstClientOSN)) {
                                    final JSONObject op = storedOperations.getJSONObject(index++);
                                    osn = op.getInt("osn");
                                    opl = op.getInt("opl");
                                    operationsToBeRestored.put(op);
                                    osn += opl;
                                }
                            }

                            operationsToBeRestored = JSONHelper.appendArray(operationsToBeRestored, clientOperations);
                        }
                    } else {
                        // logical problem - shouldn't happen (the base stream of the document
                        // is newer than the changed document or we have a hole between client
                        // operations and changed doc operations.
                        // BAIL OUT and provide an error message!
                        throw new RestoreException("Restore Document. Logical problem detected, base document newer than client operations!", RestoreException.ErrorCode.SYNC_NO_MERGE_POINT_ERROR);
                    }
                } else {
                    // no stored operations but client operations -> try to
                    // make a restore with these operations
                    operationsToBeRestored = clientOperations;
                }
            }
        } catch (final JSONException e) {
            throw new RestoreException("Restore Document - JSON exception caught!", RestoreException.ErrorCode.SYNC_NOT_POSSIBLE_ERROR);
        }

        return operationsToBeRestored;
    }

    /**
     * Writes the document stream to the storage.
     *
     * @param session the session of the client which requests restore document
     * @param docStream the final document stream to be restored
     * @param origFileName the original file name
     * @param toRestoreFileName the file name of the restored file
     * @param targetFolderId the folder id where the restored file should be written
     * @param mimeType the mime type of the document
     * @return
     * @throws OXException
     */
    private WriteInfo writeRestoreDocumentFile(final Session session, final InputStream docStream, final String origFileName, final String toRestoreFileName, final String targetFolderId, final String mimeType) throws OXException {
        ErrorCode errorCode = ErrorCode.NO_ERROR;

        final String orgExt = FileHelper.getExtension(origFileName);
        final String restoredExt = FileHelper.getExtension(toRestoreFileName);
        String restoreFileName = toRestoreFileName;

        // make sure that the new file name has an extension
        if (StringUtils.isEmpty(restoredExt)) {
            final StringBuffer strBuf = new StringBuffer(toRestoreFileName);
            strBuf.append(".");
            strBuf.append(orgExt);
            restoreFileName = strBuf.toString();
        }

        final IDBasedFileAccess fileAccess = FileHelper.getFileAccess(this.services, session);
        final File file = new DefaultFile();
        file.setFileName(restoreFileName);
        file.setFolderId(targetFolderId);
        file.setFileMIMEType(mimeType);
        file.setTitle(restoreFileName);

        errorCode = FileHelper.createFileAndWriteStream(services, session, fileAccess, file, docStream);

        // retrieve the file name and file id and set it
        final String restoredFileName = file.getFileName();
        final String restoredFileId = file.getId();

        return new WriteInfo(errorCode, restoredFileName, restoredFileId);
    }

    /**
     * Writes a message to the log using log-level INFO.
     *
     * @param msgType
     * @param restoreId
     * @param targetFolderId
     * @param errorCode
     */
    private void logRestoreMessage(final RestoreMessage msgType, final String restoreId, final String targetFolderId, final ErrorCode errorCode) {
        final String logRestoreId = (StringUtils.isEmpty(restoreId) ? "unknown" : restoreId);
        final String logTargetFolderId = (StringUtils.isEmpty(targetFolderId) ? "unknown" : targetFolderId);

        ELogLevel logLevel = ELogLevel.E_INFO;
        String logMessage = null;
        switch (msgType) {
            case RESTORE_REQUESTED:
                logMessage = "RT connection: Restore document requested for: " + logRestoreId + ", target folder: " + logTargetFolderId;
            break;

            case RESTORE_SUCCESS:
                logMessage = "RT connection: Restore document was successful for: " + logRestoreId + ", target folder: " + logTargetFolderId;
            break;

            case RESTORE_FAILED:
                logMessage = "RT connection: Restore document failed for: " + logRestoreId + ", target folder: " + logTargetFolderId + ", with error: " + errorCode.toString();
                logLevel = ELogLevel.E_WARNING;
            break;

            default:
                logMessage = "RT connection: unknown message type for 'Restore Document', please check implementation!";
                logLevel = ELogLevel.E_ERROR;
                break;
        }

        LogWithLevel.log(LOG, logLevel, logMessage, null);
    }
}
