package com.openexchange.office.tools.memory;

import java.lang.management.ManagementFactory;
import java.lang.management.MemoryMXBean;
import java.lang.management.MemoryNotificationInfo;
import java.lang.management.MemoryPoolMXBean;
import java.lang.management.MemoryType;
import java.lang.management.MemoryUsage;

import javax.management.Notification;
import javax.management.NotificationEmitter;
import javax.management.NotificationListener;

import com.openexchange.java.ConcurrentList;

public class MemoryObserver implements NotificationListener
{
    private static final MemoryPoolMXBean thresholdMemPool = getThresholdMemPool();
    private static final MemoryObserver memoryObserver = new MemoryObserver();
    private final ConcurrentList<MemoryListener> registeredListeners = new ConcurrentList<MemoryListener>();

    public interface MemoryListener {
        void memoryTresholdExceeded(long usedMemory, long maxMemory);
    }

    private MemoryObserver() {
        MemoryMXBean mbean = ManagementFactory.getMemoryMXBean();
        NotificationEmitter emitter = (NotificationEmitter) mbean;
        emitter.addNotificationListener(this, null, null);
    }

    public static synchronized MemoryObserver getMemoryObserver() {
        return memoryObserver;
    }

    public boolean addListener(MemoryListener listener) {
        return registeredListeners.add(listener);
    }

    public boolean removeListener(MemoryListener listener) {
        return registeredListeners.remove(listener);
    }

    public static void setPercentageUsageThreshold(double percentage) {
        long maxMemory = thresholdMemPool.getUsage().getMax();
        long threshold = (long)(maxMemory * percentage);
        thresholdMemPool.setUsageThreshold(threshold);
    }

    public static boolean isUsageThresholdExceeded() {
        return thresholdMemPool.isUsageThresholdExceeded();
    }

    private static MemoryPoolMXBean getThresholdMemPool() {
        MemoryPoolMXBean thresholdMemPool = null;
        for (MemoryPoolMXBean pool : ManagementFactory.getMemoryPoolMXBeans()) {
            if (pool.getType() == MemoryType.HEAP && pool.isUsageThresholdSupported()) {
                thresholdMemPool = pool;
                break;
            }
        }
        return thresholdMemPool;
    }

    /**
     * Invoked when a JMX notification occurs. The registered listeners are
     * called to react on exceeding the memory threshold.
     */
    @Override
    public void handleNotification(Notification n, Object jb) {
        if (n.getType().equals(MemoryNotificationInfo.MEMORY_THRESHOLD_EXCEEDED)) {
            final java.util.List<MemoryListener> listeners = registeredListeners.getSnapshot();
            final MemoryUsage memUsage = thresholdMemPool.getUsage();
            long maxMemory = memUsage.getMax();
            long usedMemory = memUsage.getUsed();

            for (MemoryListener listener : listeners) {
                try {
                    listener.memoryTresholdExceeded(usedMemory, maxMemory);
                } catch (Throwable e) {
                    // nothing can be done
                }
            }
        }
    }
}
