/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.tools.error;

import org.json.JSONException;
import org.json.JSONObject;


/**
 * An enumeration to provide specific error information from the server to
 * the client side. The enumeration provides method to convert to JSON object
 * or to be created from a JSON object.
 *
 * {@link ErrorCode}
 *
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 */
public class ErrorCode implements Cloneable {

	public static final ErrorCode NO_ERROR = new ErrorCode(0, "NO_ERROR", "No error", 0);
	public static final ErrorCode GENERAL_UNKNOWN_ERROR = new ErrorCode(1, "GENERAL_UNKNOWN_ERROR", "An unknown error has occurred.", 2);
    public static final ErrorCode GENERAL_ARGUMENTS_ERROR = new ErrorCode(10, "GENERAL_ARGUMENTS_ERROR", "The request have been called with missing or wrong arguments", 2);
    public static final ErrorCode GENERAL_FILE_NOT_FOUND_ERROR = new ErrorCode(11, "GENERAL_FILE_NOT_FOUND_ERROR", "The document file couldn't be found", 2);
    public static final ErrorCode GENERAL_QUOTA_REACHED_ERROR = new ErrorCode(12, "GENERAL_QUOTA_REACHED_ERROR", "The quota for the user has been reached. Storing additonal data is not possible.", 2);
    public static final ErrorCode GENERAL_PERMISSION_CREATE_MISSING_ERROR = new ErrorCode(13, "GENERAL_PERMISSION_CREATE_MISSING_ERROR", "The permission to create a file is not available.", 2);
    public static final ErrorCode GENERAL_PERMISSION_READ_MISSING_ERROR = new ErrorCode(14, "GENERAL_PERMISSION_READ_MISSING_ERROR", "The permission to read a file is not available.", 2);
    public static final ErrorCode GENERAL_PERMISSION_WRITE_MISSING_ERROR = new ErrorCode(15, "GENERAL_PERMISSION_WRITE_MISSING_ERROR", "The permission to write a file is not available.", 2);
    public static final ErrorCode GENERAL_MEMORY_TOO_LOW_ERROR = new ErrorCode(16, "GENERAL_MEMORY_TOO_LOW_ERROR", "The memory of the system is not sufficient to process the request.", 2);
    public static final ErrorCode GENERAL_SYSTEM_BUSY_ERROR = new ErrorCode(17, "GENERAL_SYSTEM_BUSY_ERROR", "The system is too busy to process the current request.", 2);
    public static final ErrorCode GENERAL_FILE_LOCKED_ERROR = new ErrorCode(18, "GENERAL_FILE_LOCKED_ERROR", "The document is locked and cannot be modified.", 2);
    public static final ErrorCode GENERAL_FILE_CORRUPT_ERROR = new ErrorCode(19, "GENERAL_FILE_CORRUPT_ERROR", "The document is corrupt and cannot read.", 2);
    public static final ErrorCode GENERAL_SERVER_COMPONENT_NOT_WORKING_ERROR = new ErrorCode(20, "GENERAL_SERVER_COMPONENT_NOT_WORKING_ERROR", "A server component is not working correctly, therefore the request couldn't be processed.", 2);
    public static final ErrorCode GENERAL_BACKGROUNDSAVE_IN_PROGRESS_ERROR = new ErrorCode(20, "GENERAL_BACKGROUNDSAVE_IN_PROGRESS_ERROR", "The document is stored by a background save process. Editing is not allowed.", 2);
    public static final ErrorCode CREATEDOCUMENT_CONVERSION_FAILED_ERROR = new ErrorCode(200, "CREATEDOCUMENT_CONVERSION_FAILED_ERROR", "The document couldn't be created because of an conversion error!", 2);
    public static final ErrorCode CREATEDOCUMENT_CANNOT_READ_TEMPLATEFILE_ERROR = new ErrorCode(201, "CREATEDOCUMENT_CANNOT_READ_TEMPLATEFILE_ERROR", "The document couldn't be created because the template file couldn't be read!", 2);
    public static final ErrorCode CREATEDOCUMENT_CANNOT_READ_DEFAULTTEMPLATEFILE_ERROR = new ErrorCode(202, "CREATEDOCUMENT_CANNOT_READ_DEFAULTTEMPLATEFILE_ERROR", "The document coulnd't be created because the default template file couldn't be read!", 2);
    public static final ErrorCode RENAMEDOCUMENT_FAILED_ERROR = new ErrorCode(300, "RENAMEDOCUMENT_FAILED_ERROR", "Renaming the document failed!", 2);
    public static final ErrorCode RENAMEDOCUMENT_VALIDATION_FAILED_CHARACTERS_ERROR = new ErrorCode(301, "RENAMEDOCUMENT_VALIDATION_FAILED_CHARACTERS_ERROR", "The document couldn't be renamed as the validation of the new file name failed. File name contains invalid characters!", 2);
    public static final ErrorCode RENAMEDOCUMENT_NOT_SUPPORTED_ERROR = new ErrorCode(302, "RENAMEDOCUMENT_NOT_SUPPORTED_ERROR", "The document cannot be renamed, because the storage doesn't support this function.", 2);
    public static final ErrorCode RENAMEDOCUMENT_SAVE_IN_PROGRESS_WARNING = new ErrorCode(303, "RENAMEDOCUMENT_SAVE_IN_PROGRESS_WARNING", "The document must be saved before it can be renamed. Wait for completion of the asynchronous rename request.", 1);
    public static final ErrorCode RENAMEDOCUMENT_NOT_POSSIBLE_IN_PROGRESS_ERROR = new ErrorCode(304, "RENAMEDOCUMENT_NOT_POSSIBLE_IN_PROGRESS_ERROR", "Rename currently not possible, a rename request is in progress.", 2);
    public static final ErrorCode LOADDOCUMENT_FAILED_ERROR = new ErrorCode(400, "LOADDOCUMENT_FAILED_ERROR", "Loading the document failed!", 2);
    public static final ErrorCode LOADDOCUMENT_CANNOT_RETRIEVE_OPERATIONS_ERROR = new ErrorCode(401, "LOADDOCUMENT_CANNOT_RETRIEVE_OPERATIONS_ERROR", "Loading the document failed because operations couldn't be generated!", 2);
    public static final ErrorCode LOADDOCUMENT_CANNOT_READ_PASSWORD_PROTECTED_ERROR = new ErrorCode(402, "LOADDOCUMENT_CANNOT_READ_PASSWORD_PROTECTED_ERROR", "Loading the document failed because it's password protected!", 2);
    public static final ErrorCode LOADDOCUMENT_COMPLEXITY_TOO_HIGH_ERROR = new ErrorCode(403, "LOADDOCUMENT_COMPLEXITY_TOO_HIGH_ERROR", "Loading the document failed because the complexity of the content is too high!", 2);
    public static final ErrorCode LOADDOCUMENT_CALCENGINE_NOT_WORKING_ERROR = new ErrorCode(405, "LOADDOCUMENT_CALCENGINE_NOT_WORKING_ERROR", "Loading the document failed because the server-side engine used for calculating formulas is not available.", 2);
    public static final ErrorCode LOADDOCUMENT_TIMEOUT_RETRIEVING_STREAM_ERROR = new ErrorCode(406, "LOADDOCUMENT_TIMEOUT_RETRIEVING_STREAM_ERROR", "Loading the document failed, because the document stream couldn't retrieve in time.", 2);
    public static final ErrorCode LOADDOCUMENT_COMPLEXITY_TOO_HIGH_SHEET_COUNT_ERROR = new ErrorCode(407, "LOADDOCUMENT_COMPLEXITY_TOO_HIGH_SHEET_COUNT_ERROR", "Loading the document failed, because the number of sheets is too high!", 2);
    public static final ErrorCode LOADDOCUMENT_STRICT_OOXML_NOT_SUPPORTED_ERROR = new ErrorCode(408, "LOADDOCUMENT_STRICT_OOXML_NOT_SUPPORTED_ERROR", "Loading the document failed, because strict mode for OOXML is not supported!", 2);
    public static final ErrorCode LOADDOCUMENT_FEATURE_NOT_SUPPORTED_ERROR = new ErrorCode(409, "LOADDOCUMENT_FEATURE_NOT_SUPPORTED_ERROR", "Loading the document failed, the document contains a not supported feature!", 2);
    public static final ErrorCode LOADDOCUMENT_FEATURE_NOT_SUPPORTED_FRAME_ATTACHED_TO_FRAME_ERROR = new ErrorCode(410, "LOADDOCUMENT_FEATURE_NOT_SUPPORTED_FRAME_ATTACHED_TO_FRAME_ERROR", "Loading the document failed, the document contains frame attached to frame which is not supported!", 2);
    public static final ErrorCode LOADDOCUMENT_FILE_HAS_CHANGED_ERROR = new ErrorCode(411, "LOADDOCUMENT_FILE_HAS_CHANGED_ERROR", "Loading the document failed, because the document content has changed! Please try again later.", 2);
    public static final ErrorCode HANGUP_INVALID_OPERATIONS_SENT_ERROR = new ErrorCode(500, "HANGUP_INVALID_OPERATIONS_SENT_ERROR", "Client sent invalid operations and therefore lost edit rights!", 2);
    public static final ErrorCode HANGUP_NO_EDIT_RIGHTS_ERROR = new ErrorCode(501, "HANGUP_NO_EDIT_RIGHTS_ERROR", "Client sent operations without having edit rights. Operations are ignored and client must switch to read-only mode!", 2);
    public static final ErrorCode HANGUP_INVALID_OSN_DETECTED_ERROR = new ErrorCode(502, "HANGUP_INVALID_OSN_DETECTED_ERROR", "Client has invalid OSN after losing edit rights. Must switch to read-only mode.", 2);
    public static final ErrorCode HANGUP_CALCENGINE_NOT_RESPONDING_ERROR = new ErrorCode(503, "HANGUP_CALCENGINE_NOT_RESPONDING_ERROR", "Calcengine is not responding. Clients must be set to read-only to resync again.", 2);
    public static final ErrorCode SAVEDOCUMENT_FAILED_ERROR = new ErrorCode(601, "SAVEDOCUMENT_FAILED_ERROR", "Saving document failed", 2);
    public static final ErrorCode SAVEDOCUMENT_FAILED_NOBACKUP_ERROR = new ErrorCode(602, "SAVEDOCUMENT_FAILED_NOBACKUP_ERROR", "Saving document failed - no backup file could be written.", 2);
    public static final ErrorCode SAVEDOCUMENT_FAILED_OPERATIONS_ERROR = new ErrorCode(603, "SAVEDOCUMENT_FAILED_OPERATIONS_ERROR", "Saving document failed - the filter was not able to apply the operations to the document.", 2);
    public static final ErrorCode SAVEDOCUMENT_FAILED_NO_FILTER_ERROR = new ErrorCode(605, "SAVEDOCUMENT_FAILED_NO_FILTER_ERROR", "Saving document failed - document export filter not available.", 2);
    public static final ErrorCode SAVEDOCUMENT_FAILED_NO_RESOURCEMANAGER_ERROR = new ErrorCode(606, "SAVEDOCUMENT_FAILED_NO_RESOURCEMANAGER_ERROR", "Saving document failed - no resource manager available for filter.", 2);
    public static final ErrorCode SAVEDOCUMENT_FAILED_NO_DOCUMENTSTREAM_ERROR = new ErrorCode(607, "SAVEDOCUMENT_FAILED_NO_DOCUMENTSTREAM_ERROR", "Saving document failed - no document stream available.", 2);
    public static final ErrorCode SAVEDOCUMENT_FAILED_FILTER_OPERATION_ERROR = new ErrorCode(608, "SAVEDOCUMENT_FAILED_FILTER_OPERATION_ERROR", "Saving document failed - The document is inconsistent and is saved as _ox file!", 2);
    public static final ErrorCode SAVEDOCUMENT_SAVE_IN_PROGRESS_ERROR = new ErrorCode(609, "SAVEDOCUMENT_SAVE_IN_PROGRESS_ERROR", "Saving document not possible, because a save request is still in progress", 2);
    public static final ErrorCode SAVEDOCUMENT_PARTIAL_CHANGES_SAVED_ERROR = new ErrorCode(610, "SAVEDOCUMENT_PARTIAL_CHANGES_SAVED_ERROR", "Saving document was not completely successful, only a subset of changes could be saved", 2);
    public static final ErrorCode SAVEDOCUMENT_FILE_HAS_CHANGED_ERROR = new ErrorCode(611, "SAVEDOCUMENT_FILE_HAS_CHANGED_ERROR", "Saving document failed, because the document content has changed!", 2);
    public static final ErrorCode SAVEDOCUMENT_BACKUPFILE_CREATE_FAILED_ERROR = new ErrorCode(650, "SAVEDOCUMENT_BACKUPFILE_CREATE_FAILED_ERROR", "Saving document succeeded, but couldn't create backup file!", 1);
    public static final ErrorCode SAVEDOCUMENT_BACKUPFILE_CREATE_PERMISSION_MISSING_ERROR = new ErrorCode(651, "SAVEDOCUMENT_BACKUPFILE_CREATE_PERMISSION_MISSING_ERROR", "Saving document succeeded, but couldn't create backup file due to missing permissions!", 1);
    public static final ErrorCode SAVEDOCUMENT_BACKUPFILE_READ_OR_WRITE_PERMISSION_MISSING_ERROR = new ErrorCode(652, "SAVEDOCUMENT_BACKUPFILE_READ_OR_WRITE_PERMISSION_MISSING_ERROR", "Saving document succeeded, but couldn't read/write backup file due to missing permissions!", 1);
    public static final ErrorCode SAVEDOCUMENT_BACKUPFILE_QUOTA_REACHED_ERROR = new ErrorCode(653, "SAVEDOCUMENT_BACKUPFILE_QUOTA_REACHED_ERROR", "Saving document succeeded, but couldn't create backup file due to exceeded quota!", 1);
    public static final ErrorCode SAVEDOCUMENT_BACKUPFILE_IS_LOCKED_ERROR = new ErrorCode(654, "SAVEDOCUMENT_BACKUPFILE_IS_LOCKED_ERROR", "Saving document succeeded, but couldn't create backup file due to a lock protection!", 1);
    public static final ErrorCode COPYDOCUMENT_FAILED_ERROR = new ErrorCode(700, "COPYDOCUMENT_FAILED_ERROR", "Copying the document failed due to unknown reasons.", 2);
    public static final ErrorCode COPYDOCUMENT_USERFOLDER_UNKOWN_ERROR = new ErrorCode(701, "COPYDOCUMENT_USERFOLDER_UNKOWN_ERROR", "Copying the document failed, because the user's folder is unknown.", 2);
    public static final ErrorCode COPYDOCUMENT_FILENAME_UNKNOWN_ERROR = new ErrorCode(702, "COPYDOCUMENT_FILENAME_UNKNOWN_ERROR", "Copying the document failed, because the file name is unknown", 2);
    public static final ErrorCode COPYDOCUMENT_TEMPLATE_FORMAT_UNKOWN_ERROR = new ErrorCode(703, "COPYDOCUMENT_TEMPLATE_FORMAT_UNKOWN_ERROR", "Copying the document failed, because the template format couldn't be determined.", 2);
    public static final ErrorCode GETTEMPLATELIST_UNKNOWN_ERROR = new ErrorCode(800, "GETTEMPLATELIST_UNKNOWN_ERROR", "Retrieving the template list failed.", 2);
    public static final ErrorCode GETTEMPLATELIST_USERFOLDER_UNKNOWN_ERROR = new ErrorCode(801, "GETTEMPLATELIST_USERFOLDER_UNKNOWN_ERROR", "The user folder is unknown, therefore the user template list cannot be retrieved.", 2);
    public static final ErrorCode GETTEMPLATELIST_GLOBALFOLDER_IO_ERROR = new ErrorCode(802, "GETTEMPLATELIST_GLOBALFOLDER_IO_ERROR", "There was an IO exception accessing the local file system.", 2);
    public static final ErrorCode LOCALFILE_FILE_NOT_FOUND_ERROR = new ErrorCode(900, "LOCALFILE_FILE_NOT_FOUND_ERROR", "The local file cannot be found.", 2);
    public static final ErrorCode LOCALFILE_CANNOT_READ_FILE_ERROR = new ErrorCode(901, "LOCALFILE_CANNOT_READ_FILE_ERROR", "The local file cannot be read.", 2);
    public static final ErrorCode LOCALFILE_INVALID_ID_ERROR = new ErrorCode(902, "LOCALFILE_INVALID_ID_ERROR", "The file id provided is invalid.", 2);
    public static final ErrorCode GETUSERINFO_UNKNOWN_ERROR = new ErrorCode(1000, "GETUSERINFO_UNKNOWN_ERROR", "Retrieving the user info failed.", 2);
    public static final ErrorCode BACKUPFILE_WONT_WRITE_WARNING = new ErrorCode(1100, "BACKUPFILE_WONT_WRITE_WARNING", "A backup file won't be writte due to insufficient permissions.", 1);
    public static final ErrorCode SWITCHEDITRIGHTS_IN_PROGRESS_WARNING = new ErrorCode(1201, "SWITCHEDITRIGHTS_IN_PROGRESS_WARNING", "Switching edit rights not possible. A switch is currently in progress.", 1);
    public static final ErrorCode SWITCHEDITRIGHTS_NOT_POSSIBLE_SAVE_IN_PROGRESS_WARNING = new ErrorCode(1202, "SWITCHEDITRIGHTS_NOT_POSSIBLE_SAVE_IN_PROGRESS_WARNING", "Switching edit rights not possible. A save request is currently in progress.", 1);
    public static final ErrorCode SWITCHEDITRIGHTS_NEEDS_MORE_TIME_WARNING = new ErrorCode(1203, "SWITCHEDITRIGHTS_NEEDS_MORE_TIME_WARNING", "Switching edit rights needs more time", 1);
    public static final ErrorCode BACKUPDOCUMENT_SERVICE_NOT_AVAILABLE = new ErrorCode(1300, "BACKUPDOCUMENT_SERVICE_NOT_AVAILABLE", "The document cannot be saved, because the backup service is not available.", 2);
    public static final ErrorCode BACKUPDOCUMENT_BASEDOCSTREAM_NOT_FOUND = new ErrorCode(1301, "BACKUPDOCUMENT_BASEDOCSTREAM_NOT_FOUND", "The document cannot be saved, because the backup document stream cannot be found.", 2);
    public static final ErrorCode BACKUPDOCUMENT_CLIENTACTIONS_MALFORMED = new ErrorCode(1302, "BACKUPDOCUMENT_CLIENTACTIONS_MALFORMED", "The actions provided by the client are malformed and cannot be correctly interpreted as JSONArray", 2);
    public static final ErrorCode BACKUPDOCUMENT_SYNC_NOT_POSSIBLE = new ErrorCode(1303, "BACKUPDOCUMENT_SYNC_NOT_POSSIBLE", "The client operations cannot be synced with the document actions. Create backup is not possible!", 2);
	public static final ErrorCode BACKUPDOCUMENT_RESTORE_DOCUMENT_WRITTEN = new ErrorCode(1304, "BACKUPDOCUMENT_RESTORE_DOCUMENT_WRITTEN", "The document could only be written by restore document.", 2);
	public static final ErrorCode BACKUPDOCUMENT_RESTORE_DOCUMENT_PARTIAL_WRITTEN = new ErrorCode(1305, "BACKUPDOCUMENT_RESTORE_DOCUMENT_PARTIAL_WRITTEN", "The document could only be partially restored.", 2);
    public static final ErrorCode PRESENTER_MAX_PARTICIPANTS_FOR_PRESENTATION_REACHED_ERROR = new ErrorCode(1400, "PRESENTER_MAX_PARTICIPANTS_FOR_PRESENTATION_REACHED_ERROR", "The maximum number of participants for this presentation has been reached. Joining is not possible.", 2);

    public final static int CODE_NO_ERROR          = 0;
    public final static String CONSTANT_NO_ERROR   = "NO_ERROR";
    public final static String CONSTANT_NO_ERROR_DESCRIPTION = "No error";

    // must be adapted
    private final static int ERRORCLASS_COUNT      = 4;
    public final static int ERRORCLASS_NO_ERROR    = 0;
    public final static int ERRORCLASS_WARNING     = 1;
    public final static int ERRORCLASS_ERROR       = 2;
    public final static int ERRORCLASS_FATAL_ERROR = 3;


    private final int m_code;
    private final String m_codeAsStringConstant;
    private final String m_description;
    // can be changed by client code to change severity dependent on the context
    private int m_errorClass;

    /**
     * Creates a ErrorCode instance with the code, a unique constant string for the
     * code and a description.
     *
     * @param code
     *  The error code.
     *
     * @param codeAsString
     *  The code as a unique constant string.
     *
     * @param description
     *  The description of the error.
     */
    private ErrorCode(int code, final String codeAsString, final String description, int errorClass) {
        this.m_code = code;
        this.m_codeAsStringConstant = codeAsString;
        this.m_description = description;
        this.m_errorClass = errorClass;
    }

    /**
     * Determines if this instance represents an error or not.
     *
     * @return
     *  TRUE if the instance represents an error and FALSE if not.
     */
    public boolean isError() {
        return (m_errorClass >= ERRORCLASS_ERROR);
    }

    /**
     * Determines if this instance represents no error, means
     * it could be a warning or no error at all.
     *
     * @return
     *  TRUE if this instance represents no error or a warning, otherwise
     *  FALSE is returned.
     */
    public boolean isNoError() {
        return (m_errorClass <= ERRORCLASS_WARNING);
    }

    /**
     * Determines if this instance represents a warning or not.
     *
     * @return
     *  TRUE if the instance represents a warning and FALSE if not.
     */
    public boolean isWarning() {
        return (m_errorClass == ERRORCLASS_WARNING);
    }

    /**
     * Provides a description of the error.
     *
     * @return
     *  The description.
     */
    public String getDescription() {
       return m_description;
    }

    /**
     * Provides a unique number which describes the error case.
     *
     * @return
     *  The error code.
     */
    public int getCode() {
       return m_code;
    }

    /**
     * Provides a unique constant string for the error case.
     * @return
     */
    public String getCodeAsStringConstant() {
        return m_codeAsStringConstant;
    }

    /**
     * Provides the error class of the error.
     *
     * @return
     *  The error class of the error, which describes how severe the
     *  error condition must be treated by the code.
     */
    public int getErrorClass() {
        return m_errorClass;
    }

    /**
     * Creates a new error class instance which contains the same values
     * as this instance.
     */
    @Override
    public ErrorCode clone() {
        return new ErrorCode(this.m_code, this.m_codeAsStringConstant, this.m_description, this.m_errorClass);
    }

    /**
     * Sets the error class of the error. Sometimes an error has a
     * different impact and therefore the error class must be adapted.
     * Use this method with care as it can have a deep impact on the
     * error processing.
     *
     * @param errorClass
     */
    public static ErrorCode setErrorClass(final ErrorCode errorCode, int errorClass) {
        ErrorCode clone = errorCode.clone();
        if ((errorClass >= 0) && (errorClass < ERRORCLASS_COUNT)) {
            clone.m_errorClass = errorClass;
        }
        return clone;
    }

    /**
     * Provides a string representation of the error code instance.
     *
     * @return
     *  The string representation of the error code instance.
     */
    @Override
    public String toString() {
      return m_code + ": " + m_description;
    }

    /**
     * Provides a JSONObject which contains the state of the error
     * code instance.
     *
     * @return
     *  A JSONObject which contains the state of the error code
     *  instance or an empty object in case of an error.
     */
    public JSONObject getAsJSON() {
        JSONObject jsonObject = new JSONObject();
        try {
            jsonObject.put("errorClass", this.getErrorClass());
            jsonObject.put("errorCode", this.getCode());
            jsonObject.put("error", this.getCodeAsStringConstant());
            jsonObject.put("errorDescription", this.getDescription());
        } catch (final JSONException e) {
            //
        }
        return jsonObject;
    }

    /**
     * Creates a JSONObject that can be directly used as a result object
     * for server answers.
     *
     * @return
     *  A JSONObject which can be used as a server answer or an empty
     *  object in case of an error.
     */
    public JSONObject getAsJSONResultObject() {
    	JSONObject resultObject = new JSONObject();
    	JSONObject errorObject = this.getAsJSON();
    	try {
    		resultObject.put("error", errorObject);
    	} catch (final JSONException e) {
    		//
    	}
    	return resultObject;
    }

    /**
     * Extracts the error code from a JSON object that is the full representation
     * of a ErrorCode instance (created by getAsJSON()).
     *
     * @param jsonObject
     *  The JSON object where the error code shall be extracted.
     *
     * @param defaultValue
     *  The default value to be used if the error code cannot be extracted.
     *
     * @return
     *  The error code or the default value if the code couldn't be extracted
     *  from the JSONObject.
     */
    static public int getErrorCodeFromJSON(JSONObject jsonObject, int defaultValue) {
        int result = defaultValue;
        if (null != jsonObject) {
            result = jsonObject.optInt("errorCode", defaultValue);
        }
        return result;
    }

    /**
     * Extracts the error constant string from a JSON object that contains the
     * full representation of a ErrorCode instance (created by getAsJSON()).
     *
     * @param jsonObject
     *  The JSON object where the error constant string shall be extracted.
     *
     * @param defaultValue
     *  The default value to be used if the error code cannot be extracted.
     *
     * @return
     *  The error constant string or the default value if the code couldn't
     *  be extracted from the JSONObject.
     */
    static public String getErrorConstantFromJSON(JSONObject jsonObject, String defaultValue) {
        String result = defaultValue;
        if (null != jsonObject) {
            result = jsonObject.optString("error", defaultValue);
        }
        return result;
    }

    /**
     * Extracts the error description from a JSON object that contains the
     * full representation of a ErrorCode instance (created by getAsJSON()).
     *
     * @param jsonObject
     *  The JSON object where the error description string shall be extracted.
     *
     * @param defaultValue
     *  The default value to be used if the error description cannot be extracted.
     *
     * @return
     *  The error description or the default value if the string couldn't be
     *  extracted from the JSONObject.
     */
    static public String getErrorDescriptionFromJSON(JSONObject jsonObject, String defaultValue) {
        String result = defaultValue;
        if (null != jsonObject) {
            result = jsonObject.optString("errorDescription", defaultValue);
        }
        return result;
    }
 }
