/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.realtime.impl;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import org.apache.commons.lang.time.StopWatch;
import com.openexchange.office.tools.EasyStopWatch;
import com.openexchange.server.ServiceLookup;
import com.openexchange.timer.ScheduledTimerTask;
import com.openexchange.timer.TimerService;

/**
 * Manages the background processing of the connection instances which are
 * stored and maintained by connection pools.
 *
 * {@link ConnectionBackgroundProcessorManager}
 *
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 * @since v7.6.2
 */
public class ConnectionBackgroundProcessorManager implements IConnectionHandler {
    private static final org.apache.commons.logging.Log LOG = com.openexchange.log.LogFactory.getLog(ConnectionBackgroundProcessorManager.class);
    private static final int MIN_PERIODIC_TIMERVALUE = 60 * 1000; // time in milliseconds
    private static final int TIME_TO_SKIP_NEXT_ACTIVATION = (MIN_PERIODIC_TIMERVALUE / 2) + (MIN_PERIODIC_TIMERVALUE / 4);

    private final TimerService m_enumTimerService;
    private final ServiceLookup m_services;
    private final ArrayList<IConnectionPool> m_pools = new ArrayList<IConnectionPool>();
    private ScheduledTimerTask m_enumTimerTask = null;
    private IConnectionProcessor m_processor = null;
    private final StopWatch m_stopWatch = new StopWatch();
    private int m_skipNextActivations = 0;

    /**
     * Internal class to encapsulate the timer based processing of the
     * connections.
     *
     * {@link ProcessingRunnable}
     *
     * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
     * @since v7.6.2
     */
    private static class ProcessingRunnable implements Runnable {

        private final WeakReference<IConnectionHandler> m_handler;

        public ProcessingRunnable(final IConnectionHandler handler) {
            super();
            m_handler = new WeakReference<IConnectionHandler>(handler);
        }

        @Override
        public void run() {
            IConnectionHandler handler = m_handler.get();
            if (null != handler) {
                handler.processCollection();
            }
        }
    }

    /**
     * Initializes a new {@link ConnectionBackgroundProcessorManager}.
     * @param services
     */
    public ConnectionBackgroundProcessorManager(final ServiceLookup services) {
        this.m_services = services;
        m_enumTimerService = m_services.getService(TimerService.class);
    }

    /**
     * Adds a connection pool to be processed by the connection processor.
     *
     * @param pool
     *  A pool to be processed by the connection processor.
     */
    public synchronized void addConnectionPool(IConnectionPool pool) {
        if (null != pool) {
            m_pools.add(pool);
        }
    }

    /**
     * Removes a connection pool from the connection processor.
     *
     * @param pool
     *  The pool to be removed.
     */
    public synchronized void removeConnectionPool(IConnectionPool pool) {
        m_pools.remove(pool);
    }

    /**
     * Register the connection processor.
     *
     * @param processor
     */
    public synchronized void registerProcessor(IConnectionProcessor processor) {
        m_processor = processor;
    }

    /**
     * Unregister the connection processor.
     */
    public synchronized void unregisterProcessor() {
        m_processor = null;
    }

    /**
     * Starts the connection processor timer.
     */
    @Override
    public void start() {
        if (m_enumTimerService != null) {
            m_stopWatch.start();
            m_enumTimerTask = m_enumTimerService.scheduleAtFixedRate(new ProcessingRunnable(this), MIN_PERIODIC_TIMERVALUE, MIN_PERIODIC_TIMERVALUE);
        } else {
            m_enumTimerTask = null;
        }
    }

    /**
     * Stops a running connection processor timer.
     */
    @Override
    public void stop() {
        if (m_enumTimerTask != null) {
            m_stopWatch.stop();
            m_enumTimerTask.cancel();
            m_enumTimerTask = null;
        }
    }

    /**
     * Calls a possible collection processor, which do special
     * background processing using the open Connection instances.
     */
    @Override
    public void processCollection() {
        IConnectionProcessor processor = m_processor;
        long start = m_stopWatch.getTime();

        if (m_skipNextActivations > 0) {
            --m_skipNextActivations;
            LOG.debug("RT connection: FSS Pool skip activation - last run needed too much time.");
            return;
        }

        try {
            if (null != processor) {
                final EasyStopWatch stopWatch = new EasyStopWatch();

                final Iterator<IConnectionPool> poolIter = m_pools.iterator();
                while (poolIter.hasNext()) {
                    final IConnectionPool pool = poolIter.next();
                    boolean contEnumeration = true;
                    List<WeakReference<Connection>> connections = pool.getConnections();

                    LOG.debug("RT connection: FSS Pool for [" + pool.getName() + "] contains " + connections.size() + " document(s).");
                    connections = processor.getConnectionsToBeProcessed(connections);
                    final Iterator<WeakReference<Connection>> iter = connections.iterator();
                    if (iter.hasNext()) {
                        int numOfDocumentsSaved = 0;
                        // process only connection pool which has active connections
                        stopWatch.start();
                        processor.processStart(pool.getName());
                        while (iter.hasNext() && contEnumeration) {
                            final Connection connection = iter.next().get();

                            if ((null != connection) && (!connection.isProcessed())) {
                                contEnumeration = processor.process(connection, (MIN_PERIODIC_TIMERVALUE / 2));
                                if (contEnumeration) {
                                    // document is processed, if contEnumeration is true
                                    connection.setProcessedState(true);
                                    numOfDocumentsSaved++;
                                }
                            }
                        }
                        processor.processEnd();

                        final long elapsedTime =  stopWatch.stop();
                        final long timePerDoc = (elapsedTime / Math.max(numOfDocumentsSaved, 1));
                        LOG.debug("RT connection: FSS Pool [" + pool.getName() + "] contains " + connections.size() + " document(s) to be saved - " + numOfDocumentsSaved + " document(s) were saved in " + elapsedTime + "ms, time per document = " + timePerDoc + "ms");

                        if (!iter.hasNext()) {
                            // reset processed state to enable processing again
                            connections = pool.getConnections();
                            final Iterator<WeakReference<Connection>> resetIter = connections.iterator();
                            while (resetIter.hasNext()) {
                                final Connection connection = resetIter.next().get();
                                if (null != connection) {
                                    connection.setProcessedState(false);
                                }
                            }
                        }
                    }
                }
            }
        } catch (Exception e) {
            LOG.error("RT connection: Exception caught while FSS processes connections", e);
        } finally {
            long end = m_stopWatch.getTime();
            long diff = Math.abs(end - start);
            if (diff >= TIME_TO_SKIP_NEXT_ACTIVATION) {
                m_skipNextActivations = (int)Math.max(1, Math.min(Math.floor(diff / MIN_PERIODIC_TIMERVALUE), 3));
                LOG.debug("RT connection: FSS Pool run needed much time. Skipping next " + m_skipNextActivations + " activation(s).");
            }
        }
    }

    /**
     * Shuts down the connection processor manager.
     */
    @Override
    public void shutdown() {
        if (m_enumTimerService != null) {
            m_enumTimerService.purge();
        }
        stop();
    }

}
