/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.drawingml;

import java.math.BigDecimal;
import java.util.Iterator;
import java.util.List;
import java.util.Map.Entry;
import java.util.logging.Logger;

import org.docx4j.dml.BaseStyles;
import org.docx4j.dml.CTBackgroundFillStyleList;
import org.docx4j.dml.CTBlip;
import org.docx4j.dml.CTBlipFillProperties;
import org.docx4j.dml.CTColor;
import org.docx4j.dml.CTFillStyleList;
import org.docx4j.dml.CTGradientFillProperties;
import org.docx4j.dml.CTGradientStop;
import org.docx4j.dml.CTGradientStopList;
import org.docx4j.dml.CTHslColor;
import org.docx4j.dml.CTLineProperties;
import org.docx4j.dml.CTLineStyleList;
import org.docx4j.dml.CTNoFillProperties;
import org.docx4j.dml.CTNonVisualDrawingProps;
import org.docx4j.dml.CTPatternFillProperties;
import org.docx4j.dml.CTPoint2D;
import org.docx4j.dml.CTPositiveSize2D;
import org.docx4j.dml.CTPresetColor;
import org.docx4j.dml.CTPresetGeometry2D;
import org.docx4j.dml.CTPresetLineDashProperties;
import org.docx4j.dml.CTRelativeRect;
import org.docx4j.dml.CTSRgbColor;
import org.docx4j.dml.CTScRgbColor;
import org.docx4j.dml.CTSchemeColor;
import org.docx4j.dml.CTShapeProperties;
import org.docx4j.dml.CTSolidColorFillProperties;
import org.docx4j.dml.CTStyleMatrix;
import org.docx4j.dml.CTSystemColor;
import org.docx4j.dml.CTTextAutonumberBullet;
import org.docx4j.dml.CTTextBlipBullet;
import org.docx4j.dml.CTTextBody;
import org.docx4j.dml.CTTextBodyProperties;
import org.docx4j.dml.CTTextCharBullet;
import org.docx4j.dml.CTTextCharacterProperties;
import org.docx4j.dml.CTTextListStyle;
import org.docx4j.dml.CTTextParagraph;
import org.docx4j.dml.CTTextParagraphProperties;
import org.docx4j.dml.CTTextShapeAutofit;
import org.docx4j.dml.CTTextSpacing;
import org.docx4j.dml.CTTextSpacingPercent;
import org.docx4j.dml.CTTextSpacingPoint;
import org.docx4j.dml.CTTransform2D;
import org.docx4j.dml.IColorChoice;
import org.docx4j.dml.IFillProperties;
import org.docx4j.dml.INonVisualDrawingProperties;
import org.docx4j.dml.IShapeStyle;
import org.docx4j.dml.IStyleMatrixReference;
import org.docx4j.dml.ITransform2D;
import org.docx4j.dml.STPresetColorVal;
import org.docx4j.dml.STPresetLineDashVal;
import org.docx4j.dml.STSchemeColorVal;
import org.docx4j.dml.STShapeType;
import org.docx4j.dml.STTextAlignType;
import org.docx4j.dml.STTextAnchoringType;
import org.docx4j.dml.STTextAutonumberScheme;
import org.docx4j.dml.STTextHorzOverflowType;
import org.docx4j.dml.STTextUnderlineType;
import org.docx4j.dml.STTextVertOverflowType;
import org.docx4j.dml.STTextWrappingType;
import org.docx4j.dml.TextFont;
import org.docx4j.dml.Theme;
import org.docx4j.jaxb.Context;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.exceptions.PartUnrecognisedException;
import org.docx4j.openpackaging.parts.Part;
import org.docx4j.relationships.Relationship;
import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.office.ooxml.OperationDocument;
import com.openexchange.office.ooxml.tools.Commons;
import com.openexchange.office.ooxml.tools.json.JSONHelper;

final public class DMLHelper {

    public static void createJsonFromPositiveSize2D(JSONObject drawingProperties, CTPositiveSize2D positiveSize2D)
        throws JSONException {

        if(positiveSize2D!=null) {
            drawingProperties.put("width", Commons.coordinateTo100TH_MM(positiveSize2D.getCx()));
            drawingProperties.put("height", Commons.coordinateTo100TH_MM(positiveSize2D.getCy()));
        }
    }

    public static JSONObject createJsonFromBlipFillProperties(Part contextPart, JSONObject attrs, CTBlipFillProperties blipFillProperties)
        throws JSONException {

        if(blipFillProperties!=null) {
            attrs.put("imageUrl", getBlipUrl(contextPart, blipFillProperties.getBlip()));

            final CTRelativeRect srcRect = blipFillProperties.getSrcRect();
            if (srcRect!=null) {
                final int l = srcRect.getL() / 1000;
                final int t = srcRect.getT() / 1000;
                final int r = srcRect.getR() / 1000;
                final int b = srcRect.getB() / 1000;

                // add possible crop properties
                if (l != 0)
                    attrs.put("cropLeft", l);
                if (t != 0)
                    attrs.put("cropTop", t);
                if (r != 0)
                    attrs.put("cropRight", r);
                if (b != 0)
                    attrs.put("cropBottom", b);
            }
        }
        return attrs;
    }

    public static void createJsonFromTransform2D(JSONObject attrs, ITransform2D xFrm)
    	throws JSONException {

    	if(xFrm!=null) {
			final JSONObject initialDrawingAttrs = attrs.optJSONObject("drawing");
			final JSONObject drawingAttrs = initialDrawingAttrs!=null ? initialDrawingAttrs : new JSONObject();

            final CTPoint2D point2D = xFrm.getOff();
            if(point2D!=null) {
                if(point2D.getX()!=0) {
                    if(!drawingAttrs.hasAndNotNull("left")) {
                    	drawingAttrs.put("left", Commons.coordinateTo100TH_MM(point2D.getX()));
                    }
                }
                if(point2D.getY()!=0) {
                    if(!drawingAttrs.hasAndNotNull("top")) {
                    	drawingAttrs.put("top", Commons.coordinateTo100TH_MM(point2D.getY()));
                    }
                }
            }
            final CTPositiveSize2D positiveSize2D = xFrm.getExt();
            if(positiveSize2D!=null) {
                if(!drawingAttrs.hasAndNotNull("width")) {
                	drawingAttrs.put("width", Commons.coordinateTo100TH_MM(positiveSize2D.getCx()));
                }
                if(!drawingAttrs.hasAndNotNull("height")) {
                	drawingAttrs.put("height", Commons.coordinateTo100TH_MM(positiveSize2D.getCy()));
                }
            }
            if(xFrm.isFlipH()) {
                if(!drawingAttrs.hasAndNotNull("flipH")) {
                	drawingAttrs.put("flipH", true );
                }
            }
            if(xFrm.isFlipV()) {
                if(!drawingAttrs.hasAndNotNull("flipV")) {
                	drawingAttrs.put("flipV", true );
                }
            }
	        if(initialDrawingAttrs==null&&!drawingAttrs.isEmpty()) {
				attrs.put("drawing", drawingAttrs);
			}
        }
    }

    public static void applyTransform2DFromJson(ITransform2D transform2D, JSONObject attrs) {

    	final JSONObject drawingAttrs = attrs.optJSONObject("drawing");
    	if(drawingAttrs!=null) {
			final Object width = drawingAttrs.opt("width");
			if(width!=null) {
	            if(width instanceof Integer) {
	                CTPositiveSize2D extend = transform2D.getExt();
	                if(extend==null) {
	                	extend = Context.getDmlObjectFactory().createCTPositiveSize2D();
	                	transform2D.setExt(extend);
	                }
	                extend.setCx(((Integer)width).intValue()*360);
	            }
			}
			final Object height = drawingAttrs.opt("height");
			if(height!=null) {
				if(height instanceof Integer) {
	                CTPositiveSize2D extend = transform2D.getExt();
	                if(extend==null) {
	                	extend = Context.getDmlObjectFactory().createCTPositiveSize2D();
	                	transform2D.setExt(extend);
	                }
	                extend.setCy(((Integer)height).intValue()*360);
				}
			}
			if(width==JSONObject.NULL&&height==JSONObject.NULL) {
				transform2D.setExt(null);
			}
			final Object left = drawingAttrs.opt("left");
			if(left!=null) {
			    if(left instanceof Integer) {
			        CTPoint2D pos = transform2D.getOff();
			        if(pos==null) {
			            pos = Context.getDmlObjectFactory().createCTPoint2D();
			            transform2D.setOff(pos);
			        }
			        pos.setX(((Integer)left).longValue()*360);
			    }
			}
            final Object top = drawingAttrs.opt("top");
            if(top!=null) {
                if(top instanceof Integer) {
                    CTPoint2D pos = transform2D.getOff();
                    if(pos==null) {
                        pos = Context.getDmlObjectFactory().createCTPoint2D();
                        transform2D.setOff(pos);
                    }
                    pos.setY(((Integer)top).longValue()*360);
                }
            }
            if(left==JSONObject.NULL&&top==JSONObject.NULL) {
            	transform2D.setOff(null);
            }
			final Object flipH = drawingAttrs.opt("flipH");
			if(flipH!=null) {
				if(flipH instanceof Boolean) {
	                transform2D.setFlipH((Boolean)flipH);
				}
				else {
					transform2D.setFlipH(null);
				}
			}
			final Object flipV = drawingAttrs.opt("flipV");
			if(flipV!=null) {
				if(flipV instanceof Boolean) {
	                transform2D.setFlipV((Boolean)flipV);
				}
				else {
					transform2D.setFlipV(null);
				}
			}
    	}
    }

    public static void applyFillPropertiesFromJsonColor(IFillProperties fillProperties, Object jsonColor)
    	throws JSONException {

    	fillProperties.setNoFill(null);
		fillProperties.setBlipFill(null);
		fillProperties.setGradFill(null);
		fillProperties.setPattFill(null);
		fillProperties.setGrpFill(null);

		if(jsonColor instanceof JSONObject) {
			final CTSolidColorFillProperties fillColor = Context.getDmlObjectFactory().createCTSolidColorFillProperties();
			fillProperties.setSolidFill(fillColor);
			createSolidColorFillPropertiesFromJson(fillColor, (JSONObject)jsonColor);
		}
		else if (jsonColor==JSONObject.NULL) {
			fillProperties.setSolidFill(null);
		}
    }

    public static void applyFillPropertiesFromJson(IFillProperties fillProperties, JSONObject attrs, OperationDocument operationDocument, Part part)
    	throws JSONException, InvalidFormatException, PartUnrecognisedException {

    	final JSONObject fillAttrs = attrs.optJSONObject("fill");
    	if(fillAttrs!=null) {

    		// applying fill type first
    		final Object fillType = fillAttrs.opt("type");
    		if(fillType instanceof String) {
    			if(fillType.equals("none")) {
    				fillProperties.setNoFill(Context.getDmlObjectFactory().createCTNoFillProperties());
    			}
    			else {
        	    	fillProperties.setNoFill(null);
    			}
    			if(fillType.equals("solid")) {
    				if(fillProperties.getSolidFill()==null) {
    					fillProperties.setSolidFill(Context.getDmlObjectFactory().createCTSolidColorFillProperties());
    				}
    			}
    			else {
    				fillProperties.setSolidFill(null);
    			}
    			if(fillType.equals("bitmap")) {
    				if(fillProperties.getBlipFill()==null) {
            			fillProperties.setBlipFill(Context.getDmlObjectFactory().createCTBlipFillProperties());
    				}
    			}
    			else {
    				fillProperties.setBlipFill(null);
    			}
    			fillProperties.setGradFill(null);
    			fillProperties.setPattFill(null);
    			fillProperties.setGrpFill(null);
    		}
    		else if(fillType==JSONObject.NULL) {
    	    	fillProperties.setNoFill(null);
    			fillProperties.setSolidFill(null);
    			fillProperties.setGradFill(null);
    			fillProperties.setPattFill(null);
    			fillProperties.setBlipFill(null);
    			fillProperties.setGrpFill(null);
    		}
    		if(fillProperties.getSolidFill()!=null) {
				final JSONObject color = fillAttrs.optJSONObject("color");
				if(color!=null) {
					createSolidColorFillPropertiesFromJson(fillProperties.getSolidFill(), color);
				}
    		}
    		else if(fillProperties.getBlipFill()!=null) {
    			final String imageUrl = fillAttrs.optString("imageUrl", null);
    			final CTBlipFillProperties blipFillProperties = fillProperties.getBlipFill();
    			if(imageUrl!=null && operationDocument!=null && part!=null) {
    				blipFillProperties.setBlip(createCTBlip(operationDocument, part, imageUrl));
        			blipFillProperties.setStretch(Context.getDmlObjectFactory().createCTStretchInfoProperties());
    			}
    		}
    		else if(fillProperties.getGradFill()!=null) {

    		}
    	}
    }

    // styleMatrixReference can be zero, if available then the color from the styleMatrix is used
    private static void createJsonFromFillStyle(Object source, IStyleMatrixReference styleMatrixReference, JSONObject dest, Part part)
        throws JSONException {

        if(source instanceof CTSolidColorFillProperties) {
            dest.put("type", "solid");
            createJsonColorFromSolidColorFillProperties(dest, (CTSolidColorFillProperties)source);
        }
        else if(source instanceof CTNoFillProperties) {
            dest.put("type", "none");
        }
        else if(source instanceof CTGradientFillProperties) {
            // mapping gradient to solid fill...
            dest.put("type", "solid");
            final CTGradientStopList gradientStopList = ((CTGradientFillProperties)source).getGsLst();
            if(gradientStopList!=null) {
                final List<CTGradientStop> list = gradientStopList.getGs();
                if(!list.isEmpty()) {
                    final CTGradientStop gradientStop = list.get(0);
                    createJsonColorFromSolidColorFillProperties(dest, gradientStop);
                }
            }
        }
        else if(source instanceof CTPatternFillProperties) {
            // mapping pattern fill to solid fill
            dest.put("type", "solid");
            final CTColor fgFill = ((CTPatternFillProperties)source).getFgClr();
            if(fgFill!=null) {
                createJsonColorFromSolidColorFillProperties(dest, fgFill);
            }
        }
        else if(source instanceof CTBlipFillProperties) {
            dest.put("type", "bitmap");
            createJsonFromBlipFillProperties(part, dest, (CTBlipFillProperties)source);
        }
        if(styleMatrixReference!=null) {
            createJsonColorFromSolidColorFillProperties(dest, styleMatrixReference);
        }
    }

    // styleMatrixReference can be zero, if available then the color from the styleMatrix is used
    private static void createJsonFromLineStyle(CTLineProperties lineProperties, IStyleMatrixReference styleMatrixReference, JSONObject dest, Part part)
        throws JSONException {

        if(lineProperties.getNoFill()!=null) {
            createJsonFromFillStyle(lineProperties.getNoFill(), styleMatrixReference, dest, part);
        }
        else if(lineProperties.getSolidFill()!=null) {
            createJsonFromFillStyle(lineProperties.getSolidFill(), styleMatrixReference, dest, part);
        }
        else if(lineProperties.getGradFill()!=null) {
            createJsonFromFillStyle(lineProperties.getGradFill(), styleMatrixReference, dest, part);
        }
        else if(lineProperties.getPattFill()!=null) {
            createJsonFromFillStyle(lineProperties.getPattFill(), styleMatrixReference, dest, part);
        }
        if(lineProperties.getW()!=null) {
            dest.put("width", lineWidthToThmm(lineProperties.getW()));
        }
        if(lineProperties.getPrstDash()!=null) {
            switch(lineProperties.getPrstDash().getVal()) {
            case SYS_DOT:
            case DOT:
                dest.put("style", "dotted");
                break;
            case LG_DASH:
            case SYS_DASH:
            case DASH:
                dest.put("style", "dashed");
                break;
            case LG_DASH_DOT:
            case SYS_DASH_DOT:
            case DASH_DOT:
                dest.put("style", "dashDot");
                break;
            case LG_DASH_DOT_DOT:
            case SYS_DASH_DOT_DOT:
                dest.put("style", "dashDotDot");
                break;
            case SOLID:
            default:
                break;
            }
        }
    }

    private static void createJsonFromFillStyleMatrix(Theme theme, IStyleMatrixReference fillStyleReference, JSONObject dest, Part part)
        throws JSONException {

        if(fillStyleReference==null||theme==null) {
            return;
        }
        final BaseStyles themeElements = theme.getThemeElements();
        if(themeElements==null) {
            return;
        }
        final CTStyleMatrix fmtScheme = themeElements.getFmtScheme();
        if(fmtScheme==null) {
            return;
        }
        final long idx = fillStyleReference.getIdx();
        if(idx==0||idx==1000) {
            dest.put("type", "none");
            return;
        }
        else {
            List<Object> fillMatrix = null;
            if(idx>=1&&idx<1000) {
                final CTFillStyleList fillStyleList = fmtScheme.getFillStyleLst();
                if(fillStyleList!=null) {
                    fillMatrix = fillStyleList.getContent();
                }
            }
            else if(idx>1000) {
                final CTBackgroundFillStyleList bgFillStyleList = fmtScheme.getBgFillStyleLst();
                if(bgFillStyleList!=null) {
                    fillMatrix = bgFillStyleList.getContent();
                }
            }
            if(fillMatrix!=null) {
                final int index = (int)(idx>1000?idx-1001:idx-1);
                if(index<fillMatrix.size()) {
                    createJsonFromFillStyle(fillMatrix.get(index), fillStyleReference, dest, part);
                }
            }
        }
        return;
    }

    private static void createJsonFromLineStyleMatrix(Theme theme, IStyleMatrixReference lineStyleReference, JSONObject dest, Part part)
        throws JSONException {

        if(lineStyleReference==null||theme==null) {
            return;
        }
        final BaseStyles themeElements = theme.getThemeElements();
        if(themeElements==null) {
            return;
        }
        final CTStyleMatrix fmtScheme = themeElements.getFmtScheme();
        if(fmtScheme==null) {
            return;
        }
       final int idx = (int)lineStyleReference.getIdx();
       final CTLineStyleList lineStyleList = fmtScheme.getLnStyleLst();
       if(lineStyleList!=null) {
           final List<CTLineProperties> lineMatrix = lineStyleList.getLn();
           if(idx>=0&&idx<lineMatrix.size()) {
               createJsonFromLineStyle(lineMatrix.get(idx), lineStyleReference, dest, part);
           }
       }
       return;
    }

    public static void createJsonFromShapeProperties(JSONObject attrs, CTShapeProperties shapeProperties, Theme theme, IShapeStyle shapeStyle, Part part, boolean createTransform)
            throws JSONException {

        if(shapeProperties!=null) {
            if(createTransform) {
                createJsonFromTransform2D(attrs, shapeProperties.getXfrm());
            }
            if(shapeProperties.getPrstGeom()!=null) {
            	DMLGeometry.createJsonFromPresetGeometry(attrs, shapeProperties.getPrstGeom());
            }
            else if(shapeProperties.getCustGeom()!=null) {
            	DMLGeometry.createJsonFromCustomGeometry(attrs, shapeProperties.getCustGeom());
            }
        }
        createJsonFromLineProperties(attrs, theme, shapeStyle!=null ? shapeStyle.getLnRef() : null, shapeProperties!=null ? shapeProperties.getLn(): null, part);
        createJsonFromFillProperties(attrs, shapeProperties, theme, shapeStyle!=null ? shapeStyle.getFillRef() : null, part);
    }

    public static void createJsonFromFillProperties(JSONObject attrs, IFillProperties fillProperties, Theme theme, IStyleMatrixReference fillStyleReference, Part part)
        throws JSONException {

        final JSONObject initialFillAttrs = attrs.optJSONObject("fill");
        final JSONObject fillAttrs = initialFillAttrs!=null ? initialFillAttrs : new JSONObject(1);

        if(fillStyleReference!=null) {
            // try to apply the style first...
            createJsonFromFillStyleMatrix(theme, fillStyleReference, fillAttrs, part);
        }

        if(fillProperties!=null) {

        	// then each hard applied fill attribute
        	if(fillProperties.getNoFill()!=null) {
        	    createJsonFromFillStyle(fillProperties.getNoFill(), null, fillAttrs, part);
	    	}
	    	else if(fillProperties.getSolidFill()!=null) {
	    	    createJsonFromFillStyle(fillProperties.getSolidFill(), null, fillAttrs, part);
            }
	    	else if(fillProperties.getBlipFill()!=null) {
	    	    createJsonFromFillStyle(fillProperties.getBlipFill(), null, fillAttrs, part);
	    	}
	    	else if(fillProperties.getGradFill()!=null) {
	    	    createJsonFromFillStyle(fillProperties.getGradFill(), null, fillAttrs, part);
	    	}
	    	else if(fillProperties.getPattFill()!=null) {
	    	    createJsonFromFillStyle(fillProperties.getPattFill(), null, fillAttrs, part);
	    	}
        	/*
        	if(shapeProperties.getGrpFill()!=null) {

        	}
        	*/
        }
        if(initialFillAttrs==null&&!fillAttrs.isEmpty()) {
            attrs.put("fill", fillAttrs);
        }
    }

	public static CTShapeProperties applyShapePropertiesFromJson(CTShapeProperties shapeProperties, JSONObject attrs, OperationDocument operationDocument, Part part)
		throws JSONException, InvalidFormatException, PartUnrecognisedException {

		if(shapeProperties==null) {
			shapeProperties = Context.getDmlObjectFactory().createCTShapeProperties();
		}
		CTTransform2D transform2d = shapeProperties.getXfrm();
		if(transform2d==null) {
		    transform2d = Context.getDmlObjectFactory().createCTTransform2D();
		    shapeProperties.setXfrm(transform2d);
		}
		applyTransform2DFromJson(transform2d, attrs);
		if(transform2d.getExt()==null&&transform2d.getOff()==null) {
			shapeProperties.setXfrm(null);
		}
		shapeProperties.setLn(applyLinePropertiesFromJson(shapeProperties.getLn(), attrs));
		applyFillPropertiesFromJson(shapeProperties, attrs, operationDocument, part);

		final JSONObject geometryAttrs = attrs.optJSONObject("geometry");
		if(geometryAttrs!=null) {
			DMLGeometry.applyGeometryPropertiesFromJson(geometryAttrs, shapeProperties);
		}
		return shapeProperties;
	}

    public static void createJsonFromLineProperties(JSONObject attrs, Theme theme, IStyleMatrixReference lineStyleReference, CTLineProperties lineProperties, Part part)
        throws JSONException {

        final JSONObject initialLineAttrs = attrs.optJSONObject("line");
        final JSONObject lineAttrs = initialLineAttrs!=null ? initialLineAttrs : new JSONObject(2);

        if(lineStyleReference!=null) {
            // try to apply the style first...
            createJsonFromLineStyleMatrix(theme, lineStyleReference, lineAttrs, part);
        }
    	if(lineProperties!=null) {
    	    createJsonFromLineStyle(lineProperties, null, lineAttrs, part);
    	}

    	if(initialLineAttrs==null&&!lineAttrs.isEmpty()) {
            attrs.put("line", lineAttrs);
        }
    }

	public static CTLineProperties applyLinePropertiesFromJson(CTLineProperties lineProperties, JSONObject attrs) throws JSONException {

		final JSONObject lineAttrs = attrs.optJSONObject("line");
		if(lineAttrs!=null) {
			if(lineProperties==null) {
				lineProperties = Context.getDmlObjectFactory().createCTLineProperties();
			}
			final Object typeO = lineAttrs.opt("type");
			if(null!=typeO) {
				final String type = typeO instanceof String ? (String)typeO : "none";
                final CTSolidColorFillProperties oldSolidFill = lineProperties.getSolidFill();
			    lineProperties.setNoFill(null);
                lineProperties.setSolidFill(null);
                lineProperties.setGradFill(null);
                lineProperties.setPattFill(null);
                if(type.equals("none")) {
                    lineProperties.setNoFill(Context.getDmlObjectFactory().createCTNoFillProperties());
                }
                else if(type.equals("solid")) {
                    if(oldSolidFill!=null) {
                        lineProperties.setSolidFill(oldSolidFill);
                    } else {
                        lineProperties.setSolidFill(Context.getDmlObjectFactory().createCTSolidColorFillProperties());
                    }
                }
			}
			final Object color = lineAttrs.opt("color");
			if(color!=null) {
			    // applying color only possible if fill type is solid
			    final CTSolidColorFillProperties solidFill = lineProperties.getSolidFill();
			    if(solidFill!=null) {
			        if(color instanceof JSONObject) {
			            createSolidColorFillPropertiesFromJson(solidFill, (JSONObject)color);
	                }
			        else {
                        final CTPresetColor presetColor = Context.getDmlObjectFactory().createCTPresetColor();
                        presetColor.setVal(STPresetColorVal.BLACK);
                        solidFill.setPrstClr(presetColor);
			        }
			    }
			}
			final Object width = lineAttrs.opt("width");
			if(width!=null) {
			    if(width instanceof Number) {
                    lineProperties.setW(thmmToLineWidth(lineAttrs.getLong("width")));
			    }
			    else {
			        lineProperties.setW(null);
			    }
			}
			final Object style = lineAttrs.opt("style");
			if(style!=null) {
			    CTPresetLineDashProperties presetDash = null;
			    if(style instanceof String) {
			        if(((String)style).equals("dotted")) {
			            presetDash = Context.getDmlObjectFactory().createCTPresetLineDashProperties();
			            presetDash.setVal(STPresetLineDashVal.DOT);
			        }
			        else if (((String)style).equals("dashed")) {
                        presetDash = Context.getDmlObjectFactory().createCTPresetLineDashProperties();
                        presetDash.setVal(STPresetLineDashVal.DASH);
			        }
			        else if (((String)style).equals("dashDot")) {
                        presetDash = Context.getDmlObjectFactory().createCTPresetLineDashProperties();
                        presetDash.setVal(STPresetLineDashVal.DASH_DOT);
			        }
			        else if (((String)style).equals("dashDotDot")) {
                        presetDash = Context.getDmlObjectFactory().createCTPresetLineDashProperties();
                        presetDash.setVal(STPresetLineDashVal.LG_DASH_DOT_DOT);
			        }
			    }
		        lineProperties.setPrstDash(presetDash);
			}
		}
		return lineProperties;
	}

    public static void createJsonColorFromSolidColorFillProperties(JSONObject dest, IColorChoice fillProperties)
        throws JSONException {

        if(fillProperties!=null) {
            if (fillProperties.getScrgbClr()!= null) {
                createJsonFromScrgbClr(dest, fillProperties.getScrgbClr());
            }
            else if(fillProperties.getSrgbClr()!= null) {
                createJsonFromSrgbClr(dest, fillProperties.getSrgbClr());
            }
            else if (fillProperties.getHslClr()!= null) {
                createJsonFromHslClr(dest, fillProperties.getHslClr());
            }
            else if (fillProperties.getSchemeClr()!= null) {
                createJsonFromSchemeClr(dest, fillProperties.getSchemeClr());
            }
            else if (fillProperties.getSysClr()!= null) {
                createJsonFromSysClr(dest, fillProperties.getSysClr());
            }
            else if (fillProperties.getPrstClr()!= null) {
                createJsonFromPrstClr(dest, fillProperties.getPrstClr());
            }
        }
    }

	public static boolean createSolidColorFillPropertiesFromJson(IColorChoice docx, JSONObject json) throws JSONException {
		docx.setSysClr(null);
		docx.setScrgbClr(null);
		docx.setSrgbClr(null);
		docx.setHslClr(null);
		docx.setPrstClr(null);
		docx.setSchemeClr(null);


		final String type = json.getString("type");

		if (type.equals("auto")) {
			return false;
		}else if (type.equals("scheme")) {
			final CTSchemeColor schemeClr = new CTSchemeColor();
			createSchemeColorFromJson(schemeClr, json);
			docx.setSchemeClr(schemeClr);
		} else if (type.equals("rgb")) {
			final CTSRgbColor srgbClr = new CTSRgbColor();
			createSRgbColorFromJson(srgbClr, json);
			docx.setSrgbClr(srgbClr);
		} else {
			Logger.getAnonymousLogger().warning("colortype is not implemented " + type);
			// TODO: other colors?
		}
		return true;
	}

    public static void createJsonFromScrgbClr(JSONObject dest, CTScRgbColor color)
        throws JSONException {

        final byte[] value = new byte[3];
        value[0] = JSONHelper.gammaChannel(color.getR());
        value[1] = JSONHelper.gammaChannel(color.getG());
        value[2] = JSONHelper.gammaChannel(color.getB());
        dest.put("color", JSONHelper.makeColor("rgb", Commons.bytesToHexString(value, 0, 3), color.getEGColorTransform()));
    }

    public static void createJsonFromSrgbClr(JSONObject dest, CTSRgbColor color)
        throws JSONException {

        final byte[] value = color.getVal();
        dest.put("color", JSONHelper.makeColor("rgb", Commons.bytesToHexString(value, 0, 3), color.getEGColorTransform()));
    }

    public static void createJsonFromHslClr(JSONObject dest, CTHslColor color)
        throws JSONException {

        dest.put("color", JSONHelper.makeColor("hsl", JSONHelper.makeJSON("h", color.getHue(), "s", color.getSat(), "l", color.getLum()), color.getEGColorTransform()));
    }

    public static void createJsonFromSchemeClr(JSONObject dest, CTSchemeColor color)
        throws JSONException {

        String value = null;
        final STSchemeColorVal sourveVal = color.getVal();
        if (sourveVal != null) {
            switch (sourveVal) {
			case ACCENT_1:
			case ACCENT_2:
			case ACCENT_3:
			case ACCENT_4:
			case ACCENT_5:
			case ACCENT_6:
				value = sourveVal.value();
				break;
			case BG_1:
				value = "background1";
				break;
			case BG_2:
				value = "background2";
				break;
			case DK_1:
				value = "dark1";
				break;
			case DK_2:
				value = "dark2";
				break;
			case TX_1:
				value = "text1";
				break;
			case TX_2:
				value = "text2";
				break;
			case LT_1:
				value = "light1";
				break;
			case LT_2:
				value = "light2";
				break;
			case HLINK:
				value = "hyperlink";
				break;
			case FOL_HLINK:
				value = "followedHyperlink";
				break;
			case PH_CLR:
			    //nothing can be interpreted as AUTO color
				return;
			}
        }
        dest.put("color", JSONHelper.makeColor("scheme", value, color.getEGColorTransform()));
    }

	public static void createSchemeColorFromJson(CTSchemeColor docx, JSONObject json) throws JSONException {
		final String value = json.getString("value");
		final STSchemeColorVal colorVal;
		if (value.equals("background1")) {
			colorVal = STSchemeColorVal.BG_1;
		} else if (value.equals("background2")) {
			colorVal = STSchemeColorVal.BG_2;
		} else if (value.equals("dark1")) {
			colorVal = STSchemeColorVal.DK_1;
		} else if (value.equals("dark2")) {
			colorVal = STSchemeColorVal.DK_2;
		} else if (value.equals("text1")) {
			colorVal = STSchemeColorVal.TX_1;
		} else if (value.equals("text2")) {
			colorVal = STSchemeColorVal.TX_2;
		} else if (value.equals("light1")) {
			colorVal = STSchemeColorVal.LT_1;
		} else if (value.equals("light2")) {
			colorVal = STSchemeColorVal.LT_2;
		} else if (value.equals("hyperlink")) {
			colorVal = STSchemeColorVal.HLINK;
		} else if (value.equals("followedHyperlink")) {
			colorVal = STSchemeColorVal.FOL_HLINK;
		} else {
			STSchemeColorVal tmpVal;
			try {
				tmpVal = STSchemeColorVal.fromValue(value);
			} catch (final IllegalArgumentException e) {
				Logger.getAnonymousLogger().warning("could not find STSchemeColorVal for " + value);
				tmpVal = STSchemeColorVal.TX_1;
			}
			colorVal = tmpVal;
		}
		docx.setVal(colorVal);
		JSONHelper.saveTransformations(docx.getEGColorTransform(), json);
	}

	public static void createSRgbColorFromJson(CTSRgbColor docx, JSONObject json) throws JSONException {
		docx.setVal(Commons.hexStringToBytes(json.getString("value")));
		JSONHelper.saveTransformations(docx.getEGColorTransform(), json);
	}

    public static void createJsonFromSysClr(JSONObject dest, CTSystemColor color)
        throws JSONException {

        dest.put("color", JSONHelper.makeColor("system", color.getVal(), color.getEGColorTransform()));
    }

    public static void createJsonFromPrstClr(JSONObject dest, CTPresetColor color)
    	throws JSONException {

        dest.put("color", JSONHelper.makeColor("preset", color.getVal().value(), color.getEGColorTransform()));
    }

    public static CTNonVisualDrawingProps createNonVisualDrawingProps(String name) {
        final CTNonVisualDrawingProps nonVisualDrawingProps = Context.getDmlObjectFactory().createCTNonVisualDrawingProps();
        nonVisualDrawingProps.setName(name);
        return nonVisualDrawingProps;
    }

    public static void createJsonFromNonVisualDrawingProperties(JSONObject attrs, CTNonVisualDrawingProps nonVisualDrawingProps)
        throws JSONException {

        if(nonVisualDrawingProps==null) {
            return;
        }
		final JSONObject initialDrawingAttrs = attrs.optJSONObject("drawing");
		final JSONObject drawingAttrs = initialDrawingAttrs!=null ? initialDrawingAttrs : new JSONObject();

        final String name = nonVisualDrawingProps.getName();
        if(name!=null&&!name.isEmpty()) {
            drawingAttrs.put("name", name);
        }
        final String description = nonVisualDrawingProps.getDescr();
        if(description!=null&&!description.isEmpty()) {
            drawingAttrs.put("description", description);
        }
        if(nonVisualDrawingProps.isHidden()) {
        	drawingAttrs.put("hidden", true);
        }
        if(initialDrawingAttrs==null&&!drawingAttrs.isEmpty()) {
			attrs.put("drawing", drawingAttrs);
		}
    }

    public static void applyNonVisualDrawingProperties(INonVisualDrawingProperties iNonVisualDrawingProperties, JSONObject drawingProperties) {

    	if(drawingProperties==null) {
    		return;
    	}
    	final Object optName = drawingProperties.opt("name");
        final Object optDescription = drawingProperties.opt("description");
        if(optName!=null||optDescription!=null) {
        	CTNonVisualDrawingProps nonVisualDrawingProperties = iNonVisualDrawingProperties.getCNvPr();
        	if(nonVisualDrawingProperties==null) {
        		nonVisualDrawingProperties = Context.getDmlObjectFactory().createCTNonVisualDrawingProps();
        		iNonVisualDrawingProperties.setCNvPr(nonVisualDrawingProperties);
        	}
        	if(optName!=null) {
	            if(optName instanceof String) {
	            	nonVisualDrawingProperties.setName((String)optName);
	            }
	            else {
	                // name is a required xml attribute... so we are setting a dummy
	            	nonVisualDrawingProperties.setName("...");
	            }
        	}
        	if(optDescription!=null) {
	            if(optDescription instanceof String) {
	            	nonVisualDrawingProperties.setDescr((String)optDescription);
	            }
	            else {
	            	nonVisualDrawingProperties.setDescr(null);
	            }
	        }
        }
    }

    public static CTTransform2D createTransform2D(int x100thmm, int y100thmm, int width100thmm, int height100thmm) {
        final org.docx4j.dml.ObjectFactory dmlObjectFactory = Context.getDmlObjectFactory();
        final CTTransform2D transform2D = dmlObjectFactory.createCTTransform2D();
        final CTPoint2D point2D = dmlObjectFactory.createCTPoint2D();
        point2D.setX(Commons.coordinateFrom100TH_MM(x100thmm));
        point2D.setY(Commons.coordinateFrom100TH_MM(y100thmm));
        transform2D.setOff(point2D);
        final CTPositiveSize2D positiveSize2D = dmlObjectFactory.createCTPositiveSize2D();
        positiveSize2D.setCx(Commons.coordinateFrom100TH_MM(width100thmm));
        positiveSize2D.setCy(Commons.coordinateFrom100TH_MM(height100thmm));
        return transform2D;
    }

    public static CTPresetGeometry2D createPresetGeometry2D(STShapeType presetType) {
        final CTPresetGeometry2D presetGeometry2D = Context.getDmlObjectFactory().createCTPresetGeometry2D();
        presetGeometry2D.setPrst(presetType);
        return presetGeometry2D;
    }

    public static CTBlipFillProperties createBlipFillProperties(CTBlipFillProperties sourceBlipFill, OperationDocument operationDocument, Part part, String imageUrl)
    	throws InvalidFormatException, PartUnrecognisedException {

        final org.docx4j.dml.ObjectFactory dmlObjectFactory = Context.getDmlObjectFactory();
        final CTBlipFillProperties blipFillProperties = sourceBlipFill != null ? sourceBlipFill : dmlObjectFactory.createCTBlipFillProperties();
        blipFillProperties.setBlip(createCTBlip(operationDocument, part, imageUrl));
        blipFillProperties.setStretch(dmlObjectFactory.createCTStretchInfoProperties());
        return blipFillProperties;
    }

    public static CTBlip createCTBlip(OperationDocument operationDocument, Part part, String imageUrl)
    	throws InvalidFormatException, PartUnrecognisedException {

    	final CTBlip blip = Context.getDmlObjectFactory().createCTBlip();
        final Relationship relationship = Commons.createGraphicRelation(operationDocument, part, imageUrl);
        if(relationship!=null) {
            final boolean linked = relationship.getTargetMode() != null && relationship.getTargetMode().equals("External");
            if(linked) {
                blip.setLink(relationship.getId());
            }
            else {
                blip.setEmbed(relationship.getId());
            }
        }
        return blip;
    }

	public static void handleStandardTypes(CTTextBody txPr) {
		CTTextBodyProperties body = txPr.getBodyPr();
		if(body==null){
			body = new CTTextBodyProperties();
			txPr.setBodyPr(body);
		}

		CTTextListStyle lst2 = txPr.getLstStyle();
		if (lst2 == null) {
			lst2 = new CTTextListStyle();
			txPr.setLstStyle(lst2);
		}
	}

	/**
     * @param part (partName Uri should be /word/document.xml)
     * @return the imageUrl in following form: word/media/xxxx.png
     */
    public static String getBlipUrl(Part part, CTBlip blip) {
        String imageUrl = "";
        if(blip!=null) {
            imageUrl = Commons.getUrl(part, blip.getEmbed().length() > 0 ? blip.getEmbed() : blip.getLink() );
        }
        return imageUrl;
    }

    public static float fontSizeToPt(Integer sz) {
		if (sz == null) {
			return 10;
		}
		return ((float) sz) / 100f;
	}

    public static Integer ptTofontSize(double pt) {
		return (int) (pt * 100);
	}

    public static int lineWidthToThmm(Integer w) {
        if (w == null) {
            return 1;
        }
        return (int)((w/360f)+0.5);
    }

    private static Integer thmmToLineWidth(Long w) {
    	return (int)(w*360);
	}

	public static JSONObject createJsonFromProperties(CTTextBody props) throws JSONException {
    	JSONObject character = null;
    	if (props != null) {

    		final List<Object> paras = props.getContent();
    		if (paras != null && paras.size() > 0) {
    			final CTTextParagraph para = (CTTextParagraph)paras.get(0);

    			final CTTextParagraphProperties prop = para.getPPr(false);
    			if (prop != null) {
    				final CTTextCharacterProperties defPr = prop.getDefRPr(false);
    				if (defPr != null) {

    					// color
    					if (defPr.getSolidFill() != null) {
    					    character = new JSONObject();
    						createJsonColorFromSolidColorFillProperties(character, defPr.getSolidFill());
    					} else if (defPr.getNoFill() != null) {
    						character = new JSONObject();
    						Logger.getAnonymousLogger().warning("CtTextbody has a no fill property, result is set to null");
    					}
    					if (character == null) {
    						character = new JSONObject();
    					}

    					// font-family
    					final TextFont latn = defPr.getLatin();
    					if (latn != null) {
    						final String ff = latn.getTypeface();
    						if (ff != null && !ff.equals("+mn-lt")) {
    							character.put("fontName", ff);
    						}

    					}

    					final float size = fontSizeToPt(defPr.getSz());
    					character.put("fontSize", size);
    					final Boolean bold = defPr.isB();
    					final Boolean italic = defPr.isI();
    					if (bold != null && bold.booleanValue()) {
    						character.put("bold", true);
    					}
    					if (italic != null && italic.booleanValue()) {
    						character.put("italic", true);
    					}
    				}

    			}
    		}
    	}
    	return character;
    }

    public static void createJsonFromTextParagraphProperties(JSONObject attrs, CTTextParagraphProperties textParagraphProperties, Part part)
    	throws JSONException {

    	if(textParagraphProperties!=null) {
			final JSONObject initialParagraphAttrs = attrs.optJSONObject("paragraph");
			final JSONObject paragraphAttrs = initialParagraphAttrs!=null ? initialParagraphAttrs : new JSONObject();

			createJsonFromTextCharacterProperties(attrs, textParagraphProperties.getDefRPr(false), part);

			final Integer level = textParagraphProperties.getLvl();
			if(level!=null) {
				paragraphAttrs.put("level", level.intValue());
			}
			final JSONObject jsonLineHeight = createJsonFromCTTextSpacing(textParagraphProperties.getLnSpc());
			if(jsonLineHeight!=null) {
				paragraphAttrs.put("lineHeight", jsonLineHeight);
			}
			final JSONObject jsonSpacingBefore = createJsonFromCTTextSpacing(textParagraphProperties.getSpcBef());
			if(jsonSpacingBefore!=null) {
				paragraphAttrs.put("spacingBefore", jsonSpacingBefore);
			}
			final JSONObject jsonSpacingAfter = createJsonFromCTTextSpacing(textParagraphProperties.getSpcAft());
			if(jsonSpacingAfter!=null) {
				paragraphAttrs.put("spacingAfter", jsonSpacingAfter);
			}
			final Integer defTabSize = textParagraphProperties.getDefTabSz();
			if(defTabSize!=null) {
				paragraphAttrs.put("defaultTabSize", lineWidthToThmm(defTabSize.intValue()));
			}

			// bullet
			final JSONObject jsonBullet = new JSONObject(2);
			if(textParagraphProperties.getBuNone()!=null) {
				jsonBullet.put("type", "none");
			}
			else if(textParagraphProperties.getBuAutoNum()!=null) {
				final CTTextAutonumberBullet buAutoNumber = textParagraphProperties.getBuAutoNum();
				jsonBullet.put("type", "numbering");
				if(buAutoNumber.getStartAt()!=null) {
					jsonBullet.put("startAt", buAutoNumber.getStartAt());
				}
				if(buAutoNumber.getType()!=null) {
					jsonBullet.put("numType", buAutoNumber.getType().value());
				}
			}
			else if(textParagraphProperties.getBuChar()!=null) {
				final CTTextCharBullet buCharacter = textParagraphProperties.getBuChar();
				final String buChar = buCharacter.getChar();
				jsonBullet.put("type", "character");
				if(buChar!=null&&buChar.length()==1) {
					jsonBullet.put("character", buChar);
				}
			}
			else if(textParagraphProperties.getBuBlip()!=null) {
				final CTTextBlipBullet buBlip = textParagraphProperties.getBuBlip();
				jsonBullet.put("type", "bitmap");
				jsonBullet.put("imageUrl", getBlipUrl(part, buBlip.getBlip()));
			}
			if(!jsonBullet.isEmpty()) {
				paragraphAttrs.put("bullet", jsonBullet);
			}

			// bullet size
			if(textParagraphProperties.getBuSzTx()!=null) {
				final JSONObject buSize = new JSONObject(1);
				buSize.put("type", "followText");
				paragraphAttrs.put("bulletSize", buSize);
			}
			else if(textParagraphProperties.getBuSzPct()!=null) {
				final JSONObject buSize = new JSONObject(2);
				buSize.put("type", "percent");
				final Integer size = textParagraphProperties.getBuSzPct().getVal();
				if(size!=null) {
					buSize.put("size", (int)((size.intValue() + 500) / 1000));
				}
				paragraphAttrs.put("bulletSize", buSize);
			}
			else if(textParagraphProperties.getBuSzPts()!=null) {
				final JSONObject buSize = new JSONObject(2);
				buSize.put("type", "point");
				final Integer size = textParagraphProperties.getBuSzPts().getVal();
				if(size!=null) {
					buSize.put("size", (int)((size.intValue() + 500) / 100));
				}
				paragraphAttrs.put("bulletSize", buSize);
			}

			// bullet color
			if(textParagraphProperties.getBuClrTx()!=null) {
				final JSONObject bulletColor = new JSONObject(1);
				bulletColor.put("followText", true);
				paragraphAttrs.put("bulletColor", bulletColor);
			}
			else if(textParagraphProperties.getBuClr()!=null) {
				final JSONObject bulletColor = new JSONObject(2);
				bulletColor.put("followText", false);
				createJsonColorFromSolidColorFillProperties(bulletColor, textParagraphProperties.getBuClr());
				paragraphAttrs.put("bulletColor", bulletColor);
			}

			// bullet font
			if(textParagraphProperties.getBuFontTx()!=null) {
				final JSONObject bulletFont = new JSONObject(1);
				bulletFont.put("followText", true);
				paragraphAttrs.put("bulletFont", bulletFont);
			}
			else if(textParagraphProperties.getBuFont()!=null) {
				final JSONObject bulletFont = new JSONObject(2);
				bulletFont.put("followText", false);
				final String bulletFontName = textParagraphProperties.getBuFont().getTypeface();
				if(bulletFontName!=null) {
					bulletFont.put("name", bulletFontName);
				}
				paragraphAttrs.put("bulletFont", bulletFont);
			}

			final STTextAlignType paragraphAlign = textParagraphProperties.getAlgn();
			if(paragraphAlign!=null) {
				String a = null;
				switch(paragraphAlign) {
					case CTR: {
						a = "center";
						break;
					}
	
					case THAI_DIST:
					case DIST:
					case JUST:
					case JUST_LOW: {
						a = "justify";
						break;
					}
					case L: {
						a = "left";
						break;
					}
					case R: {
						a = "right";
						break;
					}
				}
				if(a!=null) {
					paragraphAttrs.put("alignment", a);
				}
			}
			final Integer marL = textParagraphProperties.getMarL();
			if(marL!=null) {
				paragraphAttrs.put("indentLeft", lineWidthToThmm(marL.intValue()));
			}
			final Integer marR = textParagraphProperties.getMarR();
			if(marR!=null) {
				paragraphAttrs.put("indentRight", lineWidthToThmm(marR.intValue()));
			}
			final Integer indent = textParagraphProperties.getIndent();
			if(indent!=null) {
				paragraphAttrs.put("indentFirstLine", lineWidthToThmm(indent.intValue()));
			}
	        if(initialParagraphAttrs==null&&!paragraphAttrs.isEmpty()) {
				attrs.put("paragraph", paragraphAttrs);
			}
    	}
    }

    public static void applyTextParagraphPropertiesFromJson(CTTextParagraphProperties paragraphProperties, JSONObject attrs, OperationDocument operationDocument, Part part)
    	throws JSONException, InvalidFormatException, PartUnrecognisedException {

    	if(attrs==null) {
    		return;
    	}
        final JSONObject paragraphAttrs = attrs.optJSONObject("paragraph");

        // change of paragraph level has to be applied first
        if(paragraphAttrs!=null) {
        	if(paragraphAttrs.has("level")) {
        		final Object level = paragraphAttrs.get("level");
        		if(level instanceof Integer) {
        			paragraphProperties.setLvl((Integer)level);
        		}
        		else if(level==JSONObject.NULL) {
        			paragraphProperties.setLvl(null);
        		}
        	}
        }
    	if(attrs.hasAndNotNull("character")) {
    		applyTextCharacterPropertiesFromJson(paragraphProperties.getDefRPr(true), attrs);
    	}
        if(paragraphAttrs==null) {
        	return;
        }
        final Iterator<Entry<String, Object>> entries = paragraphAttrs.entrySet().iterator();
        while(entries.hasNext()) {
        	final Entry<String, Object> entry = entries.next();
        	final Object value = entry.getValue();
        	switch(entry.getKey()) {
        		case "lineHeight": {
        			paragraphProperties.setLnSpc(createCTTextSpacingFromJson(value));
        			break;
        		}
        		case "spacingBefore": {
        			paragraphProperties.setSpcBef(createCTTextSpacingFromJson(value));
        			break;
        		}
        		case "spacingAfter": {
        			paragraphProperties.setSpcAft(createCTTextSpacingFromJson(value));
        			break;
        		}
        		case "defaultTabSize": {
        			if(value instanceof Number) {
        				paragraphProperties.setDefTabSz(thmmToLineWidth(((Number)value).longValue()));
        			}
        			else if(value==JSONObject.NULL) {
        				paragraphProperties.setDefTabSz(null);
        			}
        			break;
        		}
        		case "bullet": {
        			createBulletFromJson(paragraphProperties, value, operationDocument, part);
        			break;
        		}
        		case "bulletSize": {
        			createBulletSizeFromJson(paragraphProperties, value);
        			break;
        		}
        		case "bulletColor": {
        			createBulletColorFromJson(paragraphProperties, value);
        			break;
        		}
        		case "bulletFont": {
        			createBulletFontFromJson(paragraphProperties, value);
        			break;
        		}
        		case "alignment": {
        			if(value instanceof String) {
        				if(((String)value).equals("center")) {
        					paragraphProperties.setAlgn(STTextAlignType.CTR);
        				}
        				else if(((String)value).equals("justify")) {
        					paragraphProperties.setAlgn(STTextAlignType.JUST);
        				}
        				else if(((String)value).equals("right")) {
        					paragraphProperties.setAlgn(STTextAlignType.R);
        				}
        				else {
        					paragraphProperties.setAlgn(STTextAlignType.L);
        				}
        			}
        			else if(value==JSONObject.NULL) {
        				paragraphProperties.setAlgn(null);
        			}
        			break;
        		}
        		case "indentLeft": {
        			if(value instanceof Number) {
        				paragraphProperties.setMarL(thmmToLineWidth(((Number)value).longValue()));
        			}
        			else if(value==JSONObject.NULL) {
        				paragraphProperties.setMarL(null);
        			}
        			break;
        		}
        		case "indentRight": {
        			if(value instanceof Number) {
        				paragraphProperties.setMarR(thmmToLineWidth(((Number)value).longValue()));
        			}
        			else if(value==JSONObject.NULL) {
        				paragraphProperties.setMarR(null);
        			}
        			break;
        		}
        		case "indentFirstLine": {
        			if(value instanceof Number) {
        				paragraphProperties.setIndent(thmmToLineWidth(((Number)value).longValue()));
        			}
        			else if(value==JSONObject.NULL) {
        				paragraphProperties.setIndent(null);
        			}
        			break;
        		}
        	}
        }
    }

    // applying the properties to the listStyle
	public static void applyTextParagraphPropertiesFromJson(CTTextBody textBody, int level, JSONObject attrs, OperationDocument operationDocument, Part part)
		throws JSONException, InvalidFormatException, PartUnrecognisedException {

		if(attrs==null) {
			return;
		}
		CTTextListStyle textListStyle = textBody.getLstStyle();
		if(textListStyle==null) {
			textListStyle = Context.getDmlObjectFactory().createCTTextListStyle();
			textBody.setLstStyle(textListStyle);
		}
		applyTextParagraphPropertiesFromJson(textListStyle.getPPr(level, true), attrs, operationDocument, part);
	}

    public static JSONObject createJsonFromCTTextSpacing(CTTextSpacing textSpacing)
    	throws JSONException {

    	if(textSpacing==null) {
    		return null;
    	}

    	final JSONObject jsonLineHeight = new JSONObject(2);
		if(textSpacing.getSpcPct()!=null) {
			final CTTextSpacingPercent spacing = textSpacing.getSpcPct();
			jsonLineHeight.put("type", "percent");
			jsonLineHeight.put("value", (int)(spacing.getVal() / 1000));
		}
		else if(textSpacing.getSpcPts()!=null) {
			final CTTextSpacingPoint spacing = textSpacing.getSpcPts();
			jsonLineHeight.put("type", "fixed");
			jsonLineHeight.put("value", (int)((spacing.getVal() * 2.54) / 72));
		}
		return jsonLineHeight;
    }

    public static CTTextSpacing createCTTextSpacingFromJson(Object attrs) {
    	if(attrs instanceof JSONObject) {
    		final String type = ((JSONObject)attrs).optString("type", null);
    		final Object value = ((JSONObject)attrs).opt("value");
    		if(type!=null&&value instanceof Integer) {
        		final CTTextSpacing spacing = Context.getDmlObjectFactory().createCTTextSpacing();
        		if(type.equals("percent")) {
        			final CTTextSpacingPercent textSpacingPercent = Context.getDmlObjectFactory().createCTTextSpacingPercent();
        			spacing.setSpcPct(textSpacingPercent);
        			textSpacingPercent.setVal(((Integer)value).intValue()*1000);
        			return spacing;
        		}
        		else if(type.equals("fixed")) {
        			final CTTextSpacingPoint textSpacingFixed = Context.getDmlObjectFactory().createCTTextSpacingPoint();
        			spacing.setSpcPts(textSpacingFixed);
        			textSpacingFixed.setVal((int)((((Integer)value).intValue() * 72.0) / 2.54));
        			return spacing;
        		}
        		return spacing;
    		}
    	}
    	return null;
    }

    public static void createBulletFromJson(CTTextParagraphProperties paragraphProperties, Object bulletAttrs, OperationDocument operationDocument, Part part)
    	throws InvalidFormatException, PartUnrecognisedException {

    	paragraphProperties.setBuNone(null);
    	paragraphProperties.setBuAutoNum(null);
    	paragraphProperties.setBuChar(null);
    	paragraphProperties.setBuBlip(null);
    	if(bulletAttrs instanceof JSONObject) {
    		final JSONObject bullet = (JSONObject)bulletAttrs;
    		final String buType = bullet.optString("type", "");
    		if(buType.equals("none")) {
    			paragraphProperties.setBuNone(Context.getDmlObjectFactory().createCTTextNoBullet());
    		}
    		else if(buType.equals("numbering")) {
    			paragraphProperties.setBuAutoNum(Context.getDmlObjectFactory().createCTTextAutonumberBullet());
    			final Object startAt = bullet.opt("startAt");
    			if(startAt instanceof Integer) {
    				paragraphProperties.getBuAutoNum().setStartAt((Integer)startAt);
    			}
    			final Object numType = bullet.opt("numType");
    			if(numType instanceof String) {
    				paragraphProperties.getBuAutoNum().setType(STTextAutonumberScheme.fromValue((String)numType));
    			}
    		}
    		else if(buType.equals("character")) {
    			paragraphProperties.setBuChar(Context.getDmlObjectFactory().createCTTextCharBullet());
    			final String buChar = bullet.optString("character", "");
    			if(buChar.length()==1) {
    				paragraphProperties.getBuChar().setChar(buChar);
    			}
    		}
    		else if(buType.equals("bitmap")) {
    			paragraphProperties.setBuBlip(Context.getDmlObjectFactory().createCTTextBlipBullet());
    			final String imageUrl = bullet.optString("imageUrl", "");
    			if(!imageUrl.isEmpty()) {
    				paragraphProperties.getBuBlip().setBlip(createCTBlip(operationDocument, part, imageUrl));
    			}
    		}
    	}
    }

    public static void createBulletSizeFromJson(CTTextParagraphProperties paragraphProperties, Object buSize) {
    	paragraphProperties.setBuSzTx(null);
    	paragraphProperties.setBuSzPct(null);
    	paragraphProperties.setBuSzPts(null);
    	if(buSize instanceof JSONObject) {
    		final String type = ((JSONObject)buSize).optString("type", "");
    		if(type.equals("followText")) {
    			paragraphProperties.setBuSzTx(Context.getDmlObjectFactory().createCTTextBulletSizeFollowText());
    		}
    		else if(type.equals("percent")) {
    			final Object value = ((JSONObject)buSize).opt("size");
    			if(value instanceof Number) {
	    			paragraphProperties.setBuSzPct(Context.getDmlObjectFactory().createCTTextBulletSizePercent());
	    			paragraphProperties.getBuSzPct().setVal((int)(((Number)value).doubleValue() * 1000));
    			}
    		}
    		else if(type.equals("point")) {
    			final Object value = ((JSONObject)buSize).opt("size");
    			if(value instanceof Number) {
    				paragraphProperties.setBuSzPts(Context.getDmlObjectFactory().createCTTextBulletSizePoint());
	    			paragraphProperties.getBuSzPts().setVal((int)(((Number)value).doubleValue() * 100));
    			}
    		}
    	}
    }

    public static void createBulletColorFromJson(CTTextParagraphProperties paragraphProperties, Object buColor)
    	throws JSONException {

    	paragraphProperties.setBuClr(null);
    	paragraphProperties.setBuClrTx(null);
    	if(buColor instanceof JSONObject) {
    		final boolean followText = ((JSONObject)buColor).optBoolean("followText", false);
    		if(followText) { 
    			paragraphProperties.setBuClrTx(Context.getDmlObjectFactory().createCTTextBulletColorFollowText());
    		}
    		else {
    			final Object color = ((JSONObject)buColor).opt("color");
    			if(color instanceof JSONObject) {
	    			paragraphProperties.setBuClr(Context.getDmlObjectFactory().createCTColor());
	    			createSolidColorFillPropertiesFromJson(paragraphProperties.getBuClr(), (JSONObject)color);
    			}
    		}
    	}
    }

    public static void createBulletFontFromJson(CTTextParagraphProperties paragraphProperties, Object buFont) {
    
    	paragraphProperties.setBuFont(null);
    	paragraphProperties.setBuFontTx(null);
    	if(buFont instanceof JSONObject) {
    		final boolean followText = ((JSONObject)buFont).optBoolean("followText", false);
    		if(followText) {
    			paragraphProperties.setBuFontTx(Context.getDmlObjectFactory().createCTTextBulletTypefaceFollowText());
    		}
    		else {
    			final String name = ((JSONObject)buFont).optString("name", "");
    			paragraphProperties.setBuFont(Context.getDmlObjectFactory().createTextFont());
    			paragraphProperties.getBuFont().setTypeface(name);
    		}
    	}
    }

    public static void createJsonFromTextCharacterProperties(JSONObject attrs, CTTextCharacterProperties textCharacterProperties, Part part)
    	throws JSONException {

    	if(textCharacterProperties!=null) {
			final JSONObject initialCharacterAttrs = attrs.optJSONObject("character");
			final JSONObject characterAttrs = initialCharacterAttrs!=null ? initialCharacterAttrs : new JSONObject();

			final Integer fontSize = textCharacterProperties.getSz();
			if(fontSize!=null) {
				characterAttrs.put("fontSize", new BigDecimal(fontSize.floatValue() / 100.0f).setScale(2, BigDecimal.ROUND_HALF_UP).floatValue());
			}
			final TextFont latinFont = textCharacterProperties.getLatin();
			if(latinFont!=null&&latinFont.getTypeface()!=null) {
				characterAttrs.put("fontName", latinFont.getTypeface());
			}
        	if(textCharacterProperties.getNoFill()!=null) {
        		// TODO: we need a fill object instead of a plain color
        		// createJsonFromFillStyle(textCharacterProperties.getNoFill(), null, characterAttrs, part);
	    	}
	    	else if(textCharacterProperties.getSolidFill()!=null) {
	    		createJsonColorFromSolidColorFillProperties(characterAttrs, textCharacterProperties.getSolidFill());
            }
	    	else if(textCharacterProperties.getBlipFill()!=null) {
	    		// TODO: createJsonFromFillStyle(textCharacterProperties.getBlipFill(), null, characterAttrs, part);
	    	}
	    	else if(textCharacterProperties.getGradFill()!=null) {
	    	    // TODO: createJsonFromFillStyle(textCharacterProperties.getGradFill(), null, characterAttrs, part);
	    	}
	    	else if(textCharacterProperties.getPattFill()!=null) {
	    	    // TODO: createJsonFromFillStyle(textCharacterProperties.getPattFill(), null, characterAttrs, part);
	    	}
			final Boolean bold = textCharacterProperties.isB();
			if(bold!=null) {
				characterAttrs.put("bold", bold.booleanValue());
			}
			final Boolean italic = textCharacterProperties.isI();
			if(italic!=null) {
				characterAttrs.put("italic", italic.booleanValue());
			}
			final STTextUnderlineType underline = textCharacterProperties.getU();
			if(underline!=null) {
				switch(underline) {
				case DASH:
				case DASH_HEAVY:
				case DASH_LONG:
				case DASH_LONG_HEAVY:
				case DBL:
				case DOTTED:
				case DOTTED_HEAVY:
				case DOT_DASH:
				case DOT_DASH_HEAVY:
				case DOT_DOT_DASH:
				case DOT_DOT_DASH_HEAVY:
				case HEAVY:
				case SNG:
				case WAVY:
				case WAVY_DBL:
				case WAVY_HEAVY:
				case WORDS:
					characterAttrs.put("underline", true);
					break;
				default:
				case NONE:
					characterAttrs.put("underline", false);
					break;
				}
			}
	        if(initialCharacterAttrs==null&&!characterAttrs.isEmpty()) {
				attrs.put("character", characterAttrs);
			}
    	}
    }

    public static void applyTextCharacterPropertiesFromJson(CTTextCharacterProperties characterProperties, JSONObject attrs)
    	throws JSONException {

    	if(attrs==null) {
    		return;
    	}
        final JSONObject characterAttrs = attrs.optJSONObject("character");
        if(characterAttrs==null) {
        	return;
        }
        final Iterator<Entry<String, Object>> entries = characterAttrs.entrySet().iterator();
        while(entries.hasNext()) {
        	final Entry<String, Object> entry = entries.next();
        	final Object value = entry.getValue();
        	switch(entry.getKey()) {
        		case "color" : {
                	applyFillPropertiesFromJsonColor(characterProperties, value);
        			break;
        		}
        		case "fontSize": {
        			if(value instanceof Number) {
        				characterProperties.setSz(new Float(((Number)value).floatValue()*100.0f).intValue());
        			}
        			else if(value==JSONObject.NULL) {
        				characterProperties.setSz(null);
        			}
        			break;
        		}
        		case "fontName": {
        			if(value instanceof String) {
        				characterProperties.setLatin(Context.getDmlObjectFactory().createTextFont());
        				characterProperties.getLatin().setTypeface((String)value);
        			}
        			else if(value==JSONObject.NULL) {
        				characterProperties.setLatin(null);
        			}
        			break;
        		}
        		case "bold": {
        			if(value instanceof Boolean) {
        				characterProperties.setB((Boolean)value);
        			}
        			else if(value==JSONObject.NULL) {
        				characterProperties.setB(null);
        			}
        			break;
        		}
        		case "italic": {
        			if(value instanceof Boolean) {
        				characterProperties.setI((Boolean)value);
        			}
        			else if(value==JSONObject.NULL) {
        				characterProperties.setI(null);
        			}
        			break;
        		}
        		case "underline": {
        			if(value instanceof Boolean) {
        				characterProperties.setU(((Boolean)value).booleanValue() ? STTextUnderlineType.SNG : STTextUnderlineType.NONE);
        			}
        			else if(value==JSONObject.NULL) {
        				characterProperties.setU(null);
        			}
        			break;
        		}
        	}
        }
    }

    // applying the properties to the listStyle
	public static void applyTextCharacterPropertiesFromJson(CTTextBody textBody, int level, JSONObject attrs)
		throws JSONException {

		if(attrs==null||attrs.optJSONObject("character")==null) {
			return;
		}
		CTTextListStyle textListStyle = textBody.getLstStyle();
		if(textListStyle==null) {
			textListStyle = Context.getDmlObjectFactory().createCTTextListStyle();
			textBody.setLstStyle(textListStyle);
		}
		applyTextCharacterPropertiesFromJson(textListStyle.getPPr(level, true).getDefRPr(true), attrs);
	}

	public static JSONObject createJsonFromTextListStyle(CTTextListStyle textListStyle, Part part)
		throws JSONException {

		if(textListStyle==null) {
			return null;
		}
		final JSONObject style = new JSONObject();
		createTextParagraphProperties(textListStyle.getDefPPr(), style, "default", part);
		createTextParagraphProperties(textListStyle.getLvl1PPr(), style, "l1", part);
		createTextParagraphProperties(textListStyle.getLvl2PPr(), style, "l2", part);
		createTextParagraphProperties(textListStyle.getLvl3PPr(), style, "l3", part);
		createTextParagraphProperties(textListStyle.getLvl4PPr(), style, "l4", part);
		createTextParagraphProperties(textListStyle.getLvl5PPr(), style, "l5", part);
		createTextParagraphProperties(textListStyle.getLvl6PPr(), style, "l6", part);
		createTextParagraphProperties(textListStyle.getLvl7PPr(), style, "l7", part);
		createTextParagraphProperties(textListStyle.getLvl8PPr(), style, "l8", part);
		createTextParagraphProperties(textListStyle.getLvl9PPr(), style, "l9", part);
		return style;
	}

	private static JSONObject createTextParagraphProperties(CTTextParagraphProperties textParagraphProperties, JSONObject destStyle, String level, Part part)
		throws JSONException {

		if(textParagraphProperties==null) {
			return null;
		}

		final JSONObject properties = new JSONObject(2);
		DMLHelper.createJsonFromTextParagraphProperties(properties, textParagraphProperties, part);
		if(!properties.isEmpty()) {
			destStyle.put(level, properties);
		}
		return null;
	}

	public static JSONObject createJsonFromTextBodyProperties(JSONObject attrs, CTTextBodyProperties textBodyProperties)
		throws JSONException {

		if(textBodyProperties!=null) {
			final JSONObject initialTextBodyAttrs = attrs.optJSONObject("shape");
			final JSONObject textBodyAttrs = initialTextBodyAttrs!=null ? initialTextBodyAttrs : new JSONObject();

			textBodyProperties.getSpAutoFit();
			// top is default
			final STTextAnchoringType anchor = textBodyProperties.getAnchor();
			if(anchor!=null) {
				String a = null;
				switch(anchor) {
					case B: {
						a = "bottom";
						break;
					}
					case CTR: {
						a = "centered";
						break;
					}
					case DIST: {
						a = "distributed";
						break;
					}
					case JUST: {
						a = "justified";
						break;
					}
					case T: {
						a = "top";
						break;
					}
				}
				if(a!=null) {
					textBodyAttrs.put("anchor", a);
				}

				// false is default
				final Boolean anchorCentered = textBodyProperties.isAnchorCtr();
				if(anchorCentered!=null) {
					textBodyAttrs.put("anchorCentered", anchorCentered.booleanValue());
				}
	
				// default 2,5mm
				final Integer marginLeft = textBodyProperties.getLIns();
				if(marginLeft!=null) {
					textBodyAttrs.put("paddingLeft", marginLeft.intValue() / 360);
				}
	
				// default 2,5mm
				final Integer marginRight = textBodyProperties.getRIns();
				if(marginRight!=null) {
					textBodyAttrs.put("paddingRight", marginRight.intValue() / 360);
				}
	
				// default 1,25mm
				final Integer marginTop = textBodyProperties.getTIns();
				if(marginTop!=null) {
					textBodyAttrs.put("paddingTop", marginTop.intValue() / 360);
				}
	
				// default 1,25mm
				final Integer marginBottom = textBodyProperties.getBIns();
				if(marginBottom!=null) {
					textBodyAttrs.put("paddingBottom", marginBottom.intValue() / 360);
				}
				
				// default is square
	 			final STTextWrappingType wrap = textBodyProperties.getWrap();
				if(wrap!=null) {
					textBodyAttrs.put("wordWrap", wrap==STTextWrappingType.NONE ? false : true);
				}

				// default is overflow
				final STTextHorzOverflowType horzOverflow = textBodyProperties.getHorzOverflow();
				if(horzOverflow!=null) {
					textBodyAttrs.put("horzOverflow", horzOverflow==STTextHorzOverflowType.CLIP ? "clip" : "overflow");
				}

				final CTTextShapeAutofit spAutoFit = textBodyProperties.getSpAutoFit();
                if(spAutoFit!=null) {
                	textBodyAttrs.put("autoResizeHeight", true);
                }

				// default is overflow
				final STTextVertOverflowType vertOverflow = textBodyProperties.getVertOverflow();
				if(vertOverflow!=null) {
					String o = null;
					switch(vertOverflow) {
						case CLIP: {
							o = "clip";
							break;
						}
						case ELLIPSIS: {
							o = "ellipsis";
							break;
						}
						case OVERFLOW : {
							o = "overflow";
							break;
						}
					}
					if(o!=null) {
						textBodyAttrs.put("vertOverflow", o);
					}
				}
			}
	        if(initialTextBodyAttrs==null&&!textBodyAttrs.isEmpty()) {
				attrs.put("shape", textBodyAttrs);
			}
		}
		return attrs;
	}

	public static void applyTextBodyPropertiesFromJson(CTTextBodyProperties bodyPr, JSONObject attrs) {

		if(bodyPr==null) {
			return;
		}
		final JSONObject shapeAttrs = attrs.optJSONObject("shape");
	    if(shapeAttrs==null) {
	    	return;
	    }
	    final Iterator<Entry<String, Object>> iter = shapeAttrs.entrySet().iterator();
	    while(iter.hasNext()) {
	    	final Entry<String, Object> entry = iter.next();
	    	final Object value = entry.getValue();
	    	switch(entry.getKey()) {
	    		case "paddingLeft" : {
	                if(value instanceof Number) {
	                    bodyPr.setLIns(Integer.valueOf(((Number)value).intValue()*360));
	                }
	                else {
	                    bodyPr.setLIns(null);
	                }
	    			break;
	    		}
	    		case "paddingRight" : {
	                if(value instanceof Number) {
	                    bodyPr.setRIns(Integer.valueOf(((Number)value).intValue()*360));
	                }
	                else {
	                    bodyPr.setRIns(null);
	                }
	    			break;
	    		}
	    		case "paddingTop" : {
	                if(value instanceof Number) {
	                    bodyPr.setTIns(Integer.valueOf(((Number)value).intValue()*360));
	                }
	                else {
	                    bodyPr.setTIns(null);
	                }
	    			break;
	    		}
	    		case "paddingBottom" : {
	                if(value instanceof Number) {
	                    bodyPr.setBIns(Integer.valueOf(((Number)value).intValue()*360));
	                }
	                else {
	                    bodyPr.setBIns(null);
	                }
	    			break;
	    		}
	    		case "autoResizeHeight" : {
	                if(value instanceof Boolean) {
	                    bodyPr.setSpAutoFit(((Boolean)value).booleanValue()?Context.getDmlObjectFactory().createCTTextShapeAutofit():null);
	                }
	                else {
	                    bodyPr.setSpAutoFit(null);
	                }
	    			break;
	    		}
	    		case "anchor" : {
	    			STTextAnchoringType anchor = null;
	    			if(value instanceof String) {
	    				switch((String)value) {
	    					case "top" : {
	    						anchor = STTextAnchoringType.T;
	    						break;
	    					}
	    					case "centered" : {
	    						anchor = STTextAnchoringType.CTR;
	    						break;
	    					}
	    					case "distributed" : {
	    						anchor = STTextAnchoringType.DIST;
	    						break;
	    					}
	    					case "justified" : {
	    						anchor = STTextAnchoringType.JUST;
	    						break;
	    					}
	    					case "bottom" : {
	    						anchor = STTextAnchoringType.B;
	    						break;
	    					}
	    				}
	    			}
	    			bodyPr.setAnchor(anchor);
	    			break;
	    		}
	    		case "anchorCentered" : {
	    			Boolean anchorCentered = null;
	    			if(value instanceof Boolean) {
	    				anchorCentered = (Boolean)value;
	    			}
	    			bodyPr.setAnchorCtr(anchorCentered);
	    			break;
	    		}
	    		case "wordWrap" : {
	    			STTextWrappingType wordWrap = null;
	    			if(value instanceof Boolean) {
	    				wordWrap = ((Boolean)value).booleanValue() ? STTextWrappingType.SQUARE : STTextWrappingType.NONE;
	    			}
	    			bodyPr.setWrap(wordWrap);
	    			break;
	    		}
	    		case "horzOverflow" : {
	    			STTextHorzOverflowType horzOverflowType = null;
	    			if(value instanceof String) {
	    				if(((String)value).equals("clip")) {
	    					horzOverflowType = STTextHorzOverflowType.CLIP;
	    				}
	    				else if(((String)value).equals("overflow")) {
	    					horzOverflowType = STTextHorzOverflowType.OVERFLOW;
	    				}
	    			}
	    			bodyPr.setHorzOverflow(horzOverflowType);
	    			break;
	    		}
	    		case "vertOverflow" : {
	    			STTextVertOverflowType vertOverflowType = null;
	    			if(value instanceof String) {
	    				if(((String)value).equals("clip")) {
	    					vertOverflowType = STTextVertOverflowType.CLIP;
	    				}
	    				else if(((String)value).equals("overflow")) {
	    					vertOverflowType = STTextVertOverflowType.OVERFLOW;
	    				}
	    				else if(((String)value).equals("ellipsis")) {
	    					vertOverflowType = STTextVertOverflowType.ELLIPSIS;
	    				}
	    			}
	    			bodyPr.setVertOverflow(vertOverflowType);
	    			break;
	    		}
	    	}
	    }
	}
}
