/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.xlsx.tools;

import java.util.Iterator;

import org.json.JSONException;
import org.json.JSONObject;
import org.xlsx4j.sml.CTStylesheet;
import org.xlsx4j.sml.Col;
import org.xlsx4j.sml.Worksheet;

import com.openexchange.office.FilterException;
import com.openexchange.office.ooxml.xlsx.XlsxOperationDocument;

public class ColumnUtils {

	public static Double columnWidthToCharacter(XlsxOperationDocument operationDocument, Object width) {
		if(width instanceof Integer) {
            final int defaultFontPixelWidth = operationDocument.getDefaultFontSize() == 12.0 ? 8 : 7;
            final double maxDigitWidthIn100thmm = (defaultFontPixelWidth*2540)/96.0;
            return new Double((Integer)width / maxDigitWidthIn100thmm);
		}
		return null;
	}

    public static void applyColumnProperties(XlsxOperationDocument operationDocument, JSONObject columnProperties, Col col, Worksheet worksheet, CTStylesheet stylesheet)
        throws FilterException, JSONException {

        // Column properties
        final JSONObject jsonColumnProperties = columnProperties.optJSONObject("column");
        if(jsonColumnProperties!=null) {
            Iterator<String> keys = jsonColumnProperties.keys();
            while(keys.hasNext()) {
                String attr = keys.next();
                Object value = jsonColumnProperties.get(attr);
                if(attr.equals("width")) {
                	col.setWidth(columnWidthToCharacter(operationDocument, value));
                }
                else if(attr.equals("visible")) {
                    Boolean hidden = null;
                    if(value instanceof Boolean) {
                        hidden = !(Boolean)value;
                    }
                    col.setHidden(hidden);
                }
                else if(attr.equals("customWidth")) {
                    Boolean customWidth = null;
                    if(value instanceof Boolean) {
                        customWidth = (Boolean)value;
                    }
                    col.setCustomWidth(customWidth);
                }
                else if(attr.equals("isBestFit")) {
                    Boolean isBestFit = null;
                    if(value instanceof Boolean) {
                        isBestFit = (Boolean)value;
                    }
                    col.setBestFit(isBestFit);
                }
            }
        }
        col.setStyle(CellUtils.applyAutoStyle(col.getStyle(), columnProperties));

        // Excel 2013 is having problems with empty columns
        if(col.getWidth()==null) {
            col.setWidth(operationDocument.getDefaultColumnWidth(worksheet));
        }
    }

    public static double excelColumnWidthTo100thmm(XlsxOperationDocument operationDocument, double columnWidth) {
        final int defaultFontPixelWidth = operationDocument.getDefaultFontSize() == 12.0 ? 8 : 7;
        final double maxDigitWidthIn100thmm = (defaultFontPixelWidth*2540)/96.0;         // (for a 11pt calibri at 96dpi (7 pixel))
        long m1000thmm = Math.round((columnWidth * maxDigitWidthIn100thmm + 0.5)*100);
        return m1000thmm / 100.0d;
    }

    public static JSONObject createColumnProperties(XlsxOperationDocument operationDocument, Col col)
        throws JSONException {

        final JSONObject attrs = new JSONObject();
        final JSONObject columnProperties = new JSONObject();

        final Double columnWidth = col.getWidth();
        if(columnWidth!=null) {
            columnProperties.put("width", excelColumnWidthTo100thmm(operationDocument, columnWidth));
        }
        if(col.isHidden()) {
            columnProperties.put("visible", false);
        }
        if(col.isCustomWidth()) {
            columnProperties.put("customWidth", true);
        }
        if(!col.isBestFit()) {
            columnProperties.put("autoFit", false);
        }
        attrs.put("column", columnProperties);

        long s = col.getStyle();
        if(s>0) {
            attrs.put("styleId", "a" + Long.toString(s));
        }
        if(attrs.length()==0) {
            return null;
        }
        return attrs;
    }
}
