/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.pptx.tools;

import org.docx4j.dml.CTGroupShapeProperties;
import org.docx4j.dml.CTGroupTransform2D;
import org.docx4j.dml.CTNonVisualDrawingProps;
import org.docx4j.dml.CTPoint2D;
import org.docx4j.dml.CTPositiveSize2D;
import org.docx4j.dml.CTPresetGeometry2D;
import org.docx4j.dml.CTShapeProperties;
import org.docx4j.dml.CTTextBody;
import org.docx4j.dml.ITransform2D;
import org.docx4j.dml.STShapeType;
import org.docx4j.jaxb.Context;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.exceptions.PartUnrecognisedException;
import org.json.JSONException;
import org.json.JSONObject;
import org.pptx4j.pml.CTPlaceholder;
import org.pptx4j.pml.GroupShape;
import org.pptx4j.pml.GroupShape.NvGrpSpPr;
import org.pptx4j.pml.NvPr;
import org.pptx4j.pml.ObjectFactory;
import org.pptx4j.pml.Pic;
import org.pptx4j.pml.Pic.NvPicPr;
import org.pptx4j.pml.STPlaceholderSize;
import org.pptx4j.pml.STPlaceholderType;
import org.pptx4j.pml.Shape;
import org.pptx4j.pml.Shape.NvSpPr;

import com.openexchange.office.ooxml.OperationDocument;
import com.openexchange.office.ooxml.components.Component;
import com.openexchange.office.ooxml.drawingml.DMLHelper;
import com.openexchange.office.ooxml.operations.CreateOperationHelper;
import com.openexchange.office.ooxml.pptx.components.SlideComponent;

final public class PMLShapeHelper {

	public static void applyAttrsFromJSON(OperationDocument operationDocument, JSONObject attrs, Shape shape)
		throws JSONException, InvalidFormatException, PartUnrecognisedException {

		final JSONObject presentationAttrs = attrs.optJSONObject("presentation");
		if(presentationAttrs!=null) {
			final Object phType = presentationAttrs.opt("phType");
			if(phType instanceof String) {
				final CTPlaceholder placeholder = getCTPlaceholder(shape, true);
				placeholder.setType(STPlaceholderType.fromValue((String)phType));
			}
			else if(phType==JSONObject.NULL) {
				final CTPlaceholder placeholder = getCTPlaceholder(shape, false);
				if(placeholder!=null) {
					placeholder.setType(null);
				}
			}
 			final Object phSize = presentationAttrs.opt("phSize");
 			if(phSize instanceof String) {
				final CTPlaceholder placeholder = getCTPlaceholder(shape, true);
				placeholder.setSz(STPlaceholderSize.fromValue((String)phSize));
 			}
 			else if(phSize==JSONObject.NULL) {
				final CTPlaceholder placeholder = getCTPlaceholder(shape, false);
				if(placeholder!=null) {
					placeholder.setSz(null);
				}
 			}
			final Object phIndex = presentationAttrs.opt("phIndex");
			if(phIndex instanceof Integer) {
				final CTPlaceholder placeholder = getCTPlaceholder(shape, true);
				placeholder.setIdx(((Integer)phIndex).longValue());
			}
			else if(phIndex==JSONObject.NULL) {
				final CTPlaceholder placeholder = getCTPlaceholder(shape, false);
				if(placeholder!=null) {
					placeholder.setIdx(null);
				}
			}
		}
		com.openexchange.office.ooxml.drawingml.Shape.applyAttrsFromJSON(operationDocument, attrs, shape);
	}

	private static CTPlaceholder getCTPlaceholder(Shape shape, boolean forceCreate) {
		NvSpPr nvSpPr = shape.getNvSpPr();
		if(nvSpPr==null) {
			if(!forceCreate) {
				return null;
			}
			nvSpPr = Context.getpmlObjectFactory().createShapeNvSpPr();
			shape.setNvSpPr(nvSpPr);
		}
		NvPr nvPr = nvSpPr.getNvPr();
		if(nvPr==null) {
			if(!forceCreate) {
				return null;
			}
			nvPr = Context.getpmlObjectFactory().createNvPr();
			nvSpPr.setNvPr(nvPr);
		}
		CTPlaceholder placeholder = nvPr.getPh();
		if(placeholder==null) {
			if(!forceCreate) {
				return null;
			}
			placeholder = Context.getpmlObjectFactory().createCTPlaceholder();
			nvPr.setPh(placeholder);
		}
		return placeholder;
	}

	public static JSONObject createJSONAttrs(CreateOperationHelper createOperationHelper, JSONObject attrs, Shape shape)
		throws JSONException {

		final JSONObject initialPresentationAttrs = attrs.optJSONObject("presentation");
		final JSONObject presentationAttrs = initialPresentationAttrs!=null ? initialPresentationAttrs : new JSONObject();

		final NvSpPr nvSpPr = shape.getNvSpPr();
		if(nvSpPr!=null) {
			final NvPr NvPr = nvSpPr.getNvPr();
			if(NvPr!=null) {
				final CTPlaceholder placeholder = NvPr.getPh();
				if(placeholder!=null) {
					if(placeholder.getType()!=STPlaceholderType.OBJ) {
						presentationAttrs.put("phType", placeholder.getType().value());
					}
					if(placeholder.getSz()!=STPlaceholderSize.FULL) {
						presentationAttrs.put("phSize", placeholder.getSz().value());
					}
					if(placeholder.getIdx()>0) {
						presentationAttrs.put("phIndex", placeholder.getIdx());
					}
				}
			}
		}
        if(initialPresentationAttrs==null&&!presentationAttrs.isEmpty()) {
			attrs.put("presentation", presentationAttrs);
		}
        final CTTextBody txBody = shape.getTxBody();
        if(txBody!=null) {
        	final JSONObject style = DMLHelper.createJsonFromTextListStyle(txBody.getLstStyle(), createOperationHelper.getOperationDocument().getContextPart());
        	if(style!=null&&!style.isEmpty()) {
        		attrs.put("listStyle", style);
        	}
        }
        com.openexchange.office.ooxml.drawingml.Shape.createJSONAttrs(createOperationHelper.getOperationDocument(), attrs, shape);
        return attrs;
	}

	public static SlideComponent getSlideComponent(Component component) {
		while(!(component instanceof SlideComponent)) {
			component = component.getParentComponent();
		}
		return (SlideComponent)component;
	}

	public static Shape createShape() {
		final ObjectFactory pmlObjectFactory = Context.getpmlObjectFactory();
		final org.docx4j.dml.ObjectFactory dmlObjectFactory = Context.getDmlObjectFactory();

		final Shape shape = pmlObjectFactory.createShape();
		final NvSpPr nvSpPr = pmlObjectFactory.createShapeNvSpPr();
		nvSpPr.setCNvPr(createNonVisualDrawingProps());
		nvSpPr.setCNvSpPr(dmlObjectFactory.createCTNonVisualDrawingShapeProps());
		nvSpPr.setNvPr(pmlObjectFactory.createNvPr());
		shape.setNvSpPr(nvSpPr);
		return shape;
	}

	public static GroupShape createGroupShape() {
		final ObjectFactory pmlObjectFactory = Context.getpmlObjectFactory();
		final org.docx4j.dml.ObjectFactory dmlObjectFactory = Context.getDmlObjectFactory();

		final GroupShape groupShape = pmlObjectFactory.createGroupShape();
		final NvGrpSpPr nvSpPr = pmlObjectFactory.createGroupShapeNvGrpSpPr();
		nvSpPr.setCNvPr(createNonVisualDrawingProps());
		nvSpPr.setCNvGrpSpPr(dmlObjectFactory.createCTNonVisualGroupDrawingShapeProps());
		nvSpPr.setNvPr(pmlObjectFactory.createNvPr());
		groupShape.setNvGrpSpPr(nvSpPr);
		final CTGroupShapeProperties spPr = dmlObjectFactory.createCTGroupShapeProperties();
		spPr.setXfrm((CTGroupTransform2D)createXFrm(dmlObjectFactory.createCTGroupTransform2D()));
		groupShape.setGrpSpPr(spPr);
		return groupShape;
	}

	public static Pic createImage() {
		final ObjectFactory pmlObjectFactory = Context.getpmlObjectFactory();
		final org.docx4j.dml.ObjectFactory dmlObjectFactory = Context.getDmlObjectFactory();

		final Pic pic = Context.getpmlObjectFactory().createPic();
		final NvPicPr nvSpPr = pmlObjectFactory.createPicNvPicPr();
		nvSpPr.setCNvPr(createNonVisualDrawingProps());
		nvSpPr.setCNvPicPr(dmlObjectFactory.createCTNonVisualPictureProperties());
		nvSpPr.getCNvPicPr().setPicLocks(dmlObjectFactory.createCTPictureLocking());
		nvSpPr.getCNvPicPr().getPicLocks().setNoChangeAspect(Boolean.TRUE);
		nvSpPr.setNvPr(pmlObjectFactory.createNvPr());
		pic.setNvPicPr(nvSpPr);
		final CTShapeProperties spPr = dmlObjectFactory.createCTShapeProperties();
		final CTPresetGeometry2D presetGeometry = dmlObjectFactory.createCTPresetGeometry2D();
		presetGeometry.setPrst(STShapeType.RECT);
		presetGeometry.setAvLst(dmlObjectFactory.createCTGeomGuideList());
		spPr.setPrstGeom(presetGeometry);
		pic.setSpPr(spPr);
		pic.setBlipFill(dmlObjectFactory.createCTBlipFillProperties());
		pic.getBlipFill().setStretch(dmlObjectFactory.createCTStretchInfoProperties());
		pic.getBlipFill().getStretch().setFillRect(dmlObjectFactory.createCTRelativeRect());
		return pic;
	}

	public static CTNonVisualDrawingProps createNonVisualDrawingProps() {
		final CTNonVisualDrawingProps nonVisualDrawingProps = Context.getDmlObjectFactory().createCTNonVisualDrawingProps();
		nonVisualDrawingProps.setId(null);
		nonVisualDrawingProps.setName("");
		return nonVisualDrawingProps;
	}

	public static ITransform2D createXFrm(ITransform2D xFrm) {
		final CTPoint2D off = Context.getDmlObjectFactory().createCTPoint2D();
		off.setX(0);
		off.setY(0);
		xFrm.setOff(off);
		final CTPositiveSize2D ext = Context.getDmlObjectFactory().createCTPositiveSize2D();
		ext.setCx(0);
		ext.setCy(0);
		xFrm.setExt(ext);
		return xFrm;
	}
}
