/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ods.dom;

import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.TreeSet;

import org.apache.commons.lang.mutable.MutableInt;
import org.apache.xerces.dom.ElementNSImpl;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.IElementWriter;
import org.odftoolkit.odfdom.Names;
import org.odftoolkit.odfdom.component.Component;
import org.odftoolkit.odfdom.component.OdfOperationDocument;
import org.odftoolkit.odfdom.doc.OdfDocument;
import org.odftoolkit.odfdom.doc.OdfSpreadsheetDocument;
import org.odftoolkit.odfdom.doc.api.OperationRequest;
import org.odftoolkit.odfdom.dom.OdfDocumentNamespace;
import org.odftoolkit.odfdom.dom.OdfStylesDom;
import org.odftoolkit.odfdom.dom.attribute.table.TableFieldNumberAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableOperatorAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableProtectedAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableValueAttribute;
import org.odftoolkit.odfdom.dom.element.OdfStylableElement;
import org.odftoolkit.odfdom.dom.element.OdfStyleBase;
import org.odftoolkit.odfdom.dom.style.OdfStyleFamily;
import org.odftoolkit.odfdom.dom.style.props.OdfStylePropertiesSet;
import org.odftoolkit.odfdom.incubator.doc.office.OdfOfficeAutomaticStyles;
import org.odftoolkit.odfdom.incubator.doc.office.OdfOfficeStyles;
import org.odftoolkit.odfdom.incubator.doc.office.OdfStylesBase;
import org.odftoolkit.odfdom.incubator.doc.style.OdfDefaultStyle;
import org.odftoolkit.odfdom.incubator.doc.style.OdfStyle;
import org.odftoolkit.odfdom.pkg.OdfFileDom;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Attr;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import com.openexchange.office.ods.dom.SmlUtils.CellRefRange;

/**
 *
 * @author sven.jacobi@open-xchange.com
 */
public class JsonOperationProducer {

    private static final Logger LOG = LoggerFactory.getLogger(JsonOperationProducer.class);

    private JSONArray operationQueue;

    private final OdfSpreadsheetDocument doc;
    private final Styles styles;
    private final Content content;
    private final Settings settings;

    private Map<String, Boolean> knownStyles = new HashMap<String, Boolean>();

    public JsonOperationProducer(OdfOperationDocument opsDoc)
    	throws SAXException {

    	doc = (OdfSpreadsheetDocument)opsDoc.getDocument();
        styles = (Styles)doc.getStylesDom();
        content = (Content)doc.getContentDom();
        settings = (Settings)doc.getSettingsDom();
    }

    public JSONObject getDocumentOperations()
    	throws JSONException, SAXException {

    	operationQueue = new JSONArray();
    	createSetDocumentattributesOperation();
    	createInsertStyleOperations(styles.getOfficeStyles(), false);
    	createInsertStyleOperations(content.getAutomaticStyles(), true);
    	for(int i=0; i<content.getSheets().size();i++) {
    		createSheetOperations(content.getSheets().get(i), i);
    	}
    	createNamedExpressions(content.getNamedExpressions(false), null);
    	createDatabaseRanges(content.getDatabaseRanges(false));
        final JSONObject operations = new JSONObject(1);
        operations.put("operations", operationQueue);
        return operations;
    }

    private void createSetDocumentattributesOperation()
       	throws JSONException {

    	final JSONObject documentPropsObject = new JSONObject(2);
    	final JSONObject docPropsJson = new JSONObject(2);
        docPropsJson.putOpt("fileFormat", "odf");
    	docPropsJson.put("cols", content.getMaxColumnCount());
    	final int activeSheet = content.getSheetIndex(settings.getActiveSheet());
    	if(activeSheet>=0) {
    		docPropsJson.put("activeSheet", activeSheet);
    	}
        documentPropsObject.put("document", docPropsJson);
        final JSONObject characterProps = new JSONObject(1);
        characterProps.put("fontSize", 10);
        documentPropsObject.put("character", characterProps);
        final JSONObject insertComponentObject = new JSONObject(2);
        insertComponentObject.put("name", "setDocumentAttributes");
        insertComponentObject.put("attrs", documentPropsObject);
        operationQueue.put(insertComponentObject);
    }

    private void createInsertStyleOperations(OdfStylesBase stylesBase, boolean autoStyle)
    	throws JSONException {

        if (stylesBase!=null) {
/*
        	for (OdfStyle style : stylesBase.getStylesForFamily(OdfStyleFamily.Paragraph)) {
            	triggerStyleHierarchyOps(stylesBase, OdfStyleFamily.Paragraph, style, autoStyle);
            }
            for (OdfStyle style : stylesBase.getStylesForFamily(OdfStyleFamily.Text)) {
                triggerStyleHierarchyOps(stylesBase, OdfStyleFamily.Text, style, autoStyle);
            }
*/
            for (OdfStyle style : stylesBase.getStylesForFamily(OdfStyleFamily.Graphic)) {
                triggerStyleHierarchyOps(stylesBase, OdfStyleFamily.Graphic, style, autoStyle);
            }
            //always generate graphic default style
            if(stylesBase instanceof OdfOfficeStyles) {
            	triggerDefaultStyleOp(OdfStyleFamily.Graphic, ((OdfOfficeStyles)stylesBase).getDefaultStyle(OdfStyleFamily.Graphic));
            }

//		for(OdfStyle style : officeStyles.getStylesForFamily(OdfStyleFamily.Table)){
//			mJsonOperationProducer.triggerStyleHierarchyOps(officeStyles, OdfStyleFamily.Table, style);
//		}
//		for(OdfStyle style : officeStyles.getStylesForFamily(OdfStyleFamily.TableRow)){
//			mJsonOperationProducer.triggerStyleHierarchyOps(officeStyles, OdfStyleFamily.TableRow, style);
//		}
//		for(OdfStyle style : officeStyles.getStylesForFamily(OdfStyleFamily.TableColumn)){
//			mJsonOperationProducer.triggerStyleHierarchyOps(officeStyles, OdfStyleFamily.TableColumn, style);
//		}
            for (OdfStyle style : stylesBase.getStylesForFamily(OdfStyleFamily.TableCell)) {
                triggerStyleHierarchyOps(stylesBase, OdfStyleFamily.TableCell, style, autoStyle);
            }
//		for(OdfStyle style : officeStyles.getStylesForFamily(OdfStyleFamily.Section)){
//			mJsonOperationProducer.triggerStyleHierarchyOps(officeStyles, OdfStyleFamily.Section, style);
//		}
//		for(OdfStyle style : officeStyles.getStylesForFamily(OdfStyleFamily.List)){
//			mJsonOperationProducer.triggerStyleHierarchyOps(officeStyles, OdfStyleFamily.List, style);
//		}
        }
    }

    private void createConditionalFormats(ConditionalFormats conditionalFormats, Integer sheetIndex)
    	throws JSONException, SAXException {

    	if(conditionalFormats!=null) {
    		final List<ConditionalFormat> conditionalFormatList = conditionalFormats.getConditionalFormatList();
    		for(int i=0; i<conditionalFormatList.size(); i++) {
    			final ConditionalFormat conditionalFormat = conditionalFormatList.get(i);
    			final JSONObject insertComponentObject = new JSONObject(4);
    			insertComponentObject.put("name", "insertCondFormat");
    			insertComponentObject.put("sheet", sheetIndex);
    			insertComponentObject.put("index", i);
    			insertComponentObject.put("ranges", conditionalFormat.getRanges().getJSON());
    			operationQueue.put(insertComponentObject);
    			final List<Condition> conditionList = conditionalFormat.getConditions();
    			for(int j=0; j<conditionList.size(); j++) {
    				operationQueue.put(conditionList.get(j).createCondFormatRuleOperation(doc, sheetIndex, i, j));
    			}
    		}
    	}
    }

    private void createNamedExpressions(NamedExpressions namedExpressions, Integer sheetIndex)
    	throws JSONException {

    	if(namedExpressions!=null) {
    		final Iterator<Entry<String, NamedExpression>> namedExpressionIter = namedExpressions.getExpressionList().entrySet().iterator();
    		while(namedExpressionIter.hasNext()) {
    			final NamedExpression namedExpression = namedExpressionIter.next().getValue();
	        	final JSONObject insertComponentObject = new JSONObject(3);
	        	insertComponentObject.put("name", "insertName");
	        	insertComponentObject.put("exprName", namedExpression.getName());
	        	if(namedExpression.getExpression()!=null) {
	        		insertComponentObject.put("formula", namedExpression.getExpression());
	        	}
	        	else {
	        		String cellRange = namedExpression.getCellRangeAddress();
	        		if(cellRange==null) {
	        			cellRange = namedExpression.getBaseCellAddress();
	        		}
	        		if(cellRange==null) {
	        			cellRange = "";
	        		}
        			insertComponentObject.put("formula", cellRange);
        			insertComponentObject.put("isExpression", false);
    			}
	        	if(namedExpression.getBaseCellAddress()!=null) {
	        		insertComponentObject.put("ref", namedExpression.getBaseCellAddress());
	        	}
        		if(sheetIndex!=null) {
        			insertComponentObject.put("sheet", sheetIndex);
        		}
	        	operationQueue.put(insertComponentObject);
    		}
    	}
    }

    private void createDatabaseRanges(DatabaseRanges databaseRanges)
    	throws JSONException {

    	if(databaseRanges==null) {
    		return;
    	}
    	final Iterator<Entry<String, DatabaseRange>> databaseRangeIter = databaseRanges.getDatabaseRangeList().entrySet().iterator();
    	while(databaseRangeIter.hasNext()) {
    		final DatabaseRange databaseRange = databaseRangeIter.next().getValue();
    		if(databaseRange.getName().startsWith("__Anonymous_Sheet_DB__") && databaseRange.isDisplayFilterButtons()) {
    			final Range sheetRange = databaseRange.getRange();
    			if(sheetRange.getSheetIndex()>=0) {
	    			JSONObject insertComponentObject = new JSONObject(5);
	    			insertComponentObject.put("name", "insertTable");
	    			final int sheetIndex = sheetRange.getSheetIndex();
	    			insertComponentObject.put("sheet", sheetIndex);
	    			final CellRefRange cellRefRange = sheetRange.getCellRefRange();
	    			final JSONArray start = cellRefRange.getStart().getJSONArray();
	    			final JSONArray end = cellRefRange.getEnd().getJSONArray();
	    			insertComponentObject.put("start", start);
	    			insertComponentObject.put("end", end);
	    			final JSONObject tableAttrs = databaseRange.createTableAttrs();
	    			if(tableAttrs!=null) {
		    			final JSONObject attrs = new JSONObject(1);
		    			attrs.put("table", tableAttrs);
		    			insertComponentObject.put("attrs", attrs);
	    			}
	    			operationQueue.put(insertComponentObject);

	    			final MutableInt column = new MutableInt(-1);
	    			final JSONArray entries = new JSONArray();
	    			final List<ElementNSImpl> databaseRangeChildList = databaseRange.getChilds();
	    			for(ElementNSImpl child:databaseRangeChildList) {
	    				createDatabaseColumnChanges(sheetIndex, child, column, entries);
	    			}
	    			if(column.intValue()>=0&&column.intValue()<cellRefRange.getColumns()&&entries.length()>0) {
		    			insertComponentObject = new JSONObject(4);
		    			insertComponentObject.put("name", "changeTableColumn");
		    			insertComponentObject.put("sheet", sheetIndex);
	                    insertComponentObject.put("col", column.intValue());
	                    final JSONObject attrs = new JSONObject(1);
	                    final JSONObject filterAttrs = new JSONObject(2);
	                    filterAttrs.put("type", "discrete");
	                    filterAttrs.put("entries", entries);
	                    attrs.put("filter", filterAttrs);
	                    insertComponentObject.put("attrs", attrs);
		    			operationQueue.put(insertComponentObject);
	    			}
    			}
    		}
    	}
    }

    private void createDatabaseColumnChanges(int sheetIndex, Node node, MutableInt column, JSONArray entries)
    	throws JSONException {

    	if(node instanceof ElementNSImpl) {
    		final String elementName = ((ElementNSImpl)node).getLocalName();
    		if(elementName.equals("filter-condition")) {
                final String fieldNumber = ((ElementNSImpl) node).getAttributeNS(TableFieldNumberAttribute.ATTRIBUTE_NAME.getUri(), "field-number");
                if(!fieldNumber.isEmpty()&&column.intValue()==-1) {
                	column.setValue(Integer.parseInt(fieldNumber));
                }
                final String value = ((ElementNSImpl) node).getAttributeNS(TableValueAttribute.ATTRIBUTE_NAME.getUri(), "value");
                final String operator = ((ElementNSImpl) node).getAttributeNS(TableOperatorAttribute.ATTRIBUTE_NAME.getUri(), "operator");
                if(operator.equals("=")&&!value.isEmpty()) {
                	convertColumnChangeValue(entries, value);
                }
    		}
    		else if(elementName.equals("filter-set-item")) {
                final String value = ((ElementNSImpl) node).getAttributeNS(TableValueAttribute.ATTRIBUTE_NAME.getUri(), "value");
                if(!value.isEmpty()) {
                	convertColumnChangeValue(entries, value);
                }
    		}
    		final NodeList childNodes = ((ElementNSImpl)node).getChildNodes();
    		for(int i=0; i<childNodes.getLength(); i++) {
	    		createDatabaseColumnChanges(sheetIndex, childNodes.item(i), column, entries);
    		}
    	}
    }

    private void convertColumnChangeValue(JSONArray entries, String value)
    	throws JSONException {

    	try {
    		entries.put(Integer.parseInt(value));
    	}
    	catch (NumberFormatException e) {};
    	try {
    		entries.put(Double.parseDouble(value));
    	}
    	catch (NumberFormatException e) {};

    	entries.put(value);
    }

    private void createSheetOperations(Sheet sheet, int sheetIndex)
    	throws JSONException, SAXException {

    	final JSONObject insertComponentObject = new JSONObject(3);
        insertComponentObject.put("name", "insertSheet");
        insertComponentObject.put("sheet", sheetIndex);
        String tableName = sheet.getSheetName();
        if(tableName==null||tableName.isEmpty()) {
        	tableName = "table" + String.valueOf(sheetIndex+1);
        }
        insertComponentObject.put("sheetName", tableName);

        // creating and writing default attr styles...
        final JSONObject attrs = new JSONObject();
    	final TreeSet<Row> rows = sheet.getRows();
		int emptyRows = 1048576;
		if(!rows.isEmpty()) {
	    	final Row lastRow = rows.last();
    		emptyRows -= lastRow.getRow() + lastRow.getRepeated();
		}
    	Row emptyRow = null;
    	if(emptyRows>0) {
			emptyRow = new Row(null, 1048576 - emptyRows);
			emptyRow.setRepeated(emptyRows);
		}
        final AttrsHash<Row> defaultRowAttrs = getDefaultAttrs(sheet.getRows(), emptyRow);
        if(defaultRowAttrs!=null) {
        	defaultRowAttrs.getObject().createAttributes(attrs, doc);
    	}

        final JSONObject sheetProperties = createSheetProperties(sheet);
        if(sheetProperties!=null&&!sheetProperties.isEmpty()) {
        	attrs.put("sheet", sheetProperties);
        }
    	if(!attrs.isEmpty()) {
    		insertComponentObject.put("attrs", attrs);
    	}
        operationQueue.put(insertComponentObject);
        createColumnOperations(sheet.getColumns(), sheetIndex);
        createRowOperations(sheet, sheetIndex, defaultRowAttrs);
        createMergeOperations(sheetIndex, sheet.getMergeCells());
        createHyperlinkOperations(sheetIndex, sheet.getHyperlinks());
        createSheetDrawingOperations(sheet, sheetIndex);
        createConditionalFormats(sheet.getConditionalFormats(false), sheetIndex);
        createNamedExpressions(sheet.getNamedExpressions(false), sheetIndex);
    }

    private JSONObject createSheetProperties(Sheet sheet)
    	throws JSONException, SAXException {

    	final JSONObject sheetProperties = new JSONObject(10);
        final String sheetProtection = sheet.getAttributeNS(TableProtectedAttribute.ATTRIBUTE_NAME.getUri(), TableProtectedAttribute.ATTRIBUTE_NAME.getLocalName());
        if(sheetProtection.equals("true")) {
        	sheetProperties.put("locked", true);
        }
        final String sheetStyle = sheet.getStyleName();
        if(sheetStyle!=null&&!sheetStyle.isEmpty()) {
        	final OdfOfficeAutomaticStyles autoStyles = ((Content)doc.getContentDom()).getAutomaticStyles();
        	final OdfStyle odfStyle = autoStyles.getStyle(sheetStyle, OdfStyleFamily.Table);
        	if(odfStyle!=null) {
        		final Element tableProperties = odfStyle.getChildElement(OdfDocumentNamespace.STYLE.getUri(), "table-properties");
        		if(tableProperties!=null) {
        			final String display = tableProperties.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "display");
	        		if(display.equals("false")) {
	        			sheetProperties.put("visible", false);
	        		}
        		}
        	}
        }
        final ConfigItemMapEntry globalViewSettings = settings.getGlobalViewSettings(false);
        if(globalViewSettings!=null) {
			final ConfigItemMapEntry sheetViewSettings = settings.getViewSettings(globalViewSettings, sheet.getSheetName(), false);
			final Integer cursorPositionX = getConfigValueInt(globalViewSettings, sheetViewSettings, "CursorPositionX");
			final Integer cursorPositionY = getConfigValueInt(globalViewSettings, sheetViewSettings, "CursorPositionY");
			if(cursorPositionX!=null&&cursorPositionY!=null) {
				final SmlUtils.CellRef cellRef = new SmlUtils.CellRef(cursorPositionX, cursorPositionY);
    			final JSONArray ranges = new JSONArray();
    			final JSONObject range = new JSONObject(2);
	            range.put("start", cellRef.getJSONArray());
				range.put("end", cellRef.getJSONArray());
				ranges.put(range);
				sheetProperties.put("selectedRanges", ranges);
				sheetProperties.put("activeIndex", 0);
			}
			final Integer zoomValue = getConfigValueInt(globalViewSettings, sheetViewSettings, "ZoomValue");
			if(zoomValue!=null) {
				sheetProperties.put("zoom", zoomValue / 100.0);
			}
			final Integer positionX1 = getConfigValueInt(globalViewSettings, sheetViewSettings, "PositionLeft");
			if(positionX1!=null) {
				sheetProperties.put("scrollLeft", positionX1);
			}
			final Integer positionY1 = getConfigValueInt(globalViewSettings, sheetViewSettings, "PositionTop");
			if(positionY1!=null) {
				sheetProperties.put("scrollBottom", positionY1);
			}
			final Integer positionX2 = getConfigValueInt(globalViewSettings, sheetViewSettings, "PositionRight");
			if(positionX2!=null) {
				sheetProperties.put("scrollRight", positionX2);
			}
			final Integer positionY2 = getConfigValueInt(globalViewSettings, sheetViewSettings, "PositionBottom");
			if(positionY2!=null) {
				sheetProperties.put("scrollTop", positionY2);
			}
			final String showGrid = getConfigValue(globalViewSettings, sheetViewSettings, "ShowGrid");
			if(showGrid!=null&&showGrid.equals("false")) {
				sheetProperties.put("showGrid", false);
			}
        }
        return sheetProperties;
    }

    private Integer getConfigValueInt(ConfigItemMapEntry global, ConfigItemMapEntry local, String name) {
    	Integer val = null;
    	final String stringVal = getConfigValue(global, local, name);
    	if(stringVal!=null) {
    		try {
    			val = Integer.valueOf(stringVal);
    		}
    		catch(NumberFormatException e) {
    			return null;
    		}
    	}
    	return val;
    }
    private String getConfigValue(ConfigItemMapEntry global, ConfigItemMapEntry local, String name) {
    	String val = null;
    	if(local!=null) {
    		final IElementWriter item = local.getItems().get(name);
    		if(item instanceof ConfigItem) {
    			val = ((ConfigItem)item).getValue();
    		}
    	}
    	if(val==null&&global!=null) {
    		final IElementWriter item = global.getItems().get(name);
    		if(item instanceof ConfigItem) {
    			val = ((ConfigItem)item).getValue();
    		}
    	}
    	return val;
    }

    private <T> AttrsHash<T> getDefaultAttrs(TreeSet<? extends IAttrs<T>> objs, IAttrs<T> empty) {
    	// retrieving the default style (most used styles)
    	AttrsHash<T> maxUsedAttrs = null;
    	final HashMap<AttrsHash<T>, AttrsHash<T>> attrMap = new HashMap<AttrsHash<T>, AttrsHash<T>>();
    	if(empty!=null) {
    		final AttrsHash<T> attrHash = empty.createAttrHash();
    		if(attrHash!=null) {
    			attrMap.put(attrHash, attrHash);
    			maxUsedAttrs = attrHash;
    		}
    	}
    	for(IAttrs<T> obj:objs) {

    		final AttrsHash<T> attrHash = obj.createAttrHash();
    		AttrsHash<T> hashEntry = attrMap.get(attrHash);
    		if(hashEntry==null) {
    			attrMap.put(attrHash, attrHash);
    			hashEntry = attrHash;
    		}
    		else {
    			hashEntry.setCount(hashEntry.getCount()+attrHash.getCount());
    		}
			if(maxUsedAttrs==null) {
				maxUsedAttrs = hashEntry;
			}
			else if(hashEntry.getCount()>maxUsedAttrs.getCount()) {
				maxUsedAttrs = hashEntry;
			}
    	}
    	return maxUsedAttrs;
    }

    private void createMergeOperations(int sheetIndex, List<MergeCell> mergeCells)
    	throws JSONException {

    	for(MergeCell mergeCell:mergeCells) {
	    	final JSONObject addMergeCellsObject = new JSONObject(5);
	        addMergeCellsObject.put("name", "mergeCells");
	        addMergeCellsObject.put("sheet", sheetIndex);
	        addMergeCellsObject.put("start", mergeCell.getCellRefRange().getStart().getJSONArray());
	        addMergeCellsObject.put("end", mergeCell.getCellRefRange().getEnd().getJSONArray());
	        addMergeCellsObject.putOpt("type", "merge");
	        operationQueue.put(addMergeCellsObject);
    	}
    }

    private void createHyperlinkOperations(int sheetIndex, List<Hyperlink> hyperlinks)
    	throws JSONException {

    	for(Hyperlink hyperlink:hyperlinks) {
	    	final JSONObject insertHyperlinkObject = new JSONObject(5);
	    	insertHyperlinkObject.put("name", "insertHyperlink");
	    	insertHyperlinkObject.put("sheet", sheetIndex);
	    	final CellRefRange cellRefRange = hyperlink.getCellRefRange();
	    	insertHyperlinkObject.put("start", cellRefRange.getStart().getJSONArray());
	    	if(!cellRefRange.getStart().equals(cellRefRange.getEnd())) {
	    		insertHyperlinkObject.put("end", cellRefRange.getEnd().getJSONArray());
	    	}
	    	insertHyperlinkObject.put("url", hyperlink.getUrl());
	        operationQueue.put(insertHyperlinkObject);
    	}
    }

    private void createSheetDrawingOperations(Sheet sheet, int sheetIndex)
    	throws JSONException, SAXException {

    	final Drawings drawings = sheet.getDrawings();
    	for(int i=0; i<drawings.getCount(); i++) {
    		final Drawing drawing = drawings.getDrawing(i);
    		final JSONObject setInsertDrawingOperation = new JSONObject();
    		setInsertDrawingOperation.put("name", "insertDrawing");
    		final JSONArray start = new JSONArray(2);
    		start.put(sheetIndex);
    		start.put(i);
    		setInsertDrawingOperation.put("start", start);
    		setInsertDrawingOperation.put("type", drawing.getType());
    		final JSONObject attrs = drawing.createAttributes(sheet, null);
    		if(!attrs.isEmpty()) {
    			setInsertDrawingOperation.put("attrs", attrs);
    		}
    		operationQueue.put(setInsertDrawingOperation);
    	}
    }

    private void createColumnOperations(TreeSet<Column> columns, int sheetIndex)
    	throws JSONException, SAXException {

    	for(Column column:columns) {
    		final JSONObject attrs = new JSONObject();
			column.createAttributes(attrs, doc);
			if(!attrs.isEmpty()) {
		        final int start = column.getMin();
		        final int end = column.getMax();
		        final JSONObject setColumnAttributesOperation = new JSONObject(5);
		        setColumnAttributesOperation.put("name", "setColumnAttributes");
		        setColumnAttributesOperation.put("sheet", sheetIndex);
		        setColumnAttributesOperation.put("start", start);
		        if(end>start) {
		        	setColumnAttributesOperation.put("end", end);
		        }
		        setColumnAttributesOperation.put("attrs", attrs);
		        operationQueue.put(setColumnAttributesOperation);
			}
    	}
    }

    public void createRowOperations(Sheet sheet, int sheetIndex, AttrsHash<Row> defaultRowAttr)
    	throws JSONException, SAXException {

    	final TreeSet<Row> rows = sheet.getRows();
    	for(Row row:rows) {
    		// writing only row attributes for non default rows
    		if(!row.createAttrHash().equals(defaultRowAttr)) {
    			final JSONObject attrs = new JSONObject();
	    		row.createAttributes(attrs, doc);
	    		if(!attrs.isEmpty()) {
	    	        final JSONObject setRowAttributesOperation = new JSONObject(5);
	    	        final int start = row.getRow();
	    	        final int end = (start + row.getRepeated())-1;
	    	        setRowAttributesOperation.put("name", "setRowAttributes");
	    	        setRowAttributesOperation.put("sheet", sheetIndex);
	    	        setRowAttributesOperation.put("start", start);
	    	        if(end!=start) {
	    	        	setRowAttributesOperation.put("end", end);
	    	        }
	    	        if(attrs!=null)	{
	    	        	setRowAttributesOperation.put("attrs", attrs);
	    	        }
	    	        operationQueue.put(setRowAttributesOperation);
	    		}
    		}
    		if(row.getDefaultCellStyle()!=null&&!row.getDefaultCellStyle().isEmpty()) {
    			// the optimized version as we do have a default cell style within the row,
    			// so column ranges are not needed to get the cell style
    			for(Cell cell:row.getCells()) {
    				createCellOperations(sheet, row, cell, cell.getColumn(), cell.getRepeated(), row.getDefaultCellStyle(), sheetIndex);
    			}
    		}
    		else {

    			final TreeSet<Column> columns = sheet.getColumns();
        		if(!columns.isEmpty()) {
        			for(Cell cell:row.getCells()) {
        				if(cell.getCellStyle()!=null&&!cell.getCellStyle().isEmpty()) {
            				createCellOperations(sheet, row, cell, cell.getColumn(), cell.getRepeated(), null, sheetIndex);
        				}
        				else {
        					String defaultCellStyle = null;
        					// no row and no cell style ... the column default cell style is necessary
        					final int max = (cell.getColumn()+cell.getRepeated())-1;
        					for(int min = cell.getColumn(); min<=max; ) {
        						final Column col = columns.floor(new Column(null, min));
	        					if(col.getMax()<min) {
	        						// there are no more columns, the last used default cell style is used
	        						createCellOperations(sheet, row, cell, min, (max-min)+1, defaultCellStyle, sheetIndex);
	        						break;
	        					}
	        					else {
	        						if(col.getDefaultCellStyle()!=null&&!col.getDefaultCellStyle().isEmpty()) {
	        							defaultCellStyle = col.getDefaultCellStyle();
	        						}
	        						if(col.getMax()>=max) {
		        						createCellOperations(sheet, row, cell, min, (max-min)+1, defaultCellStyle, sheetIndex);
		        						break;
	        						}
	        						else {
	        							createCellOperations(sheet, row, cell, min, (col.getMax()-min)+1, defaultCellStyle, sheetIndex);
	        							min = col.getMax()+1;
	        						}
	        					}
        					}
        				}
        			}
        		}
        		else {
        			for(Cell cell:row.getCells()) {
        				createCellOperations(sheet, row, cell, cell.getColumn(), cell.getRepeated(), null,  sheetIndex);
        			}
        		}
    		}
    	}
    }

    public void createCellOperations(Sheet sheet, Row row, Cell cell, int column, int repeated, String defaultCellStyle, int sheetIndex)
    		throws JSONException, SAXException {

		if(repeated>1||row.getRepeated()>1) {
			createFillCellRange(sheet, row, cell, defaultCellStyle, sheetIndex, row.getRow(), column, repeated);
		}
		else {
			JSONArray contents = operationQueue.optJSONArray(operationQueue.length()-1);
			if(contents==null) {
				// check if the last operation is a setCellContentsOperation that can be reused ...
				// we only check for the "contents" array, this should be sufficient
    	        final JSONObject createSetCellContentOperation = new JSONObject(5);
    	        createSetCellContentOperation.put("name", "setCellContents");
    	        createSetCellContentOperation.put("sheet", sheetIndex);
    	        createSetCellContentOperation.put("start", createPosition(column, row.getRow()));

    	        contents = new JSONArray(1);
    	        contents.put(new JSONArray());
    	        createSetCellContentOperation.put("contents", contents);
    	        operationQueue.put(createSetCellContentOperation);
			}
	        final JSONArray cellArray = contents.getJSONArray(contents.length()-1);
        	final JSONObject cellObject = new JSONObject();
        	final Object cellValue = cell.getCellContent();
        	if(cellValue!=null) {
        		cellObject.put("value", cellValue);
        	}
        	final JSONObject cellAttrs = cell.createCellAttributes(sheet, defaultCellStyle, doc);
        	if(cellAttrs!=null) {
        		cellObject.put("attrs", cellAttrs);
        	}
        	cellArray.put(cellObject);
		}
    }

    public void createFillCellRange(Sheet sheet, Row row, Cell cell, String defaultCellStyle, int sheetIndex, int rowNumber, int columnNumber, int repeated)
    	throws JSONException, SAXException {

        final JSONObject createFillCellRangeOperation = new JSONObject(5);
        createFillCellRangeOperation.put("name", "fillCellRange");
        createFillCellRangeOperation.put("sheet", sheetIndex);
        createFillCellRangeOperation.put("start", createPosition(columnNumber, rowNumber));
        createFillCellRangeOperation.put("end", createPosition((columnNumber+repeated)-1, (rowNumber+row.getRepeated())-1));
        final Object cellValue = cell.getCellContent();
        if(cellValue!=null) {
        	createFillCellRangeOperation.put("value", cellValue);
        }
        final JSONObject cellAttrs = cell.createCellAttributes(sheet, defaultCellStyle, doc);
        if(cellAttrs!=null) {
        	createFillCellRangeOperation.put("attrs", cellAttrs);
        }
        operationQueue.put(createFillCellRangeOperation);
    }

    public static JSONArray createPosition(int columnNumber, int rowNumber) {

    	final JSONArray position = new JSONArray(2);
    	position.put(columnNumber);
    	position.put(rowNumber);
    	return position;
    }

    private void triggerStyleHierarchyOps(OdfStylesBase stylesBase, OdfStyleFamily styleFamily, OdfStyleBase style, boolean isAutoStyle)
    	throws JSONException {

        if (style != null) {

            if (!(style instanceof OdfDefaultStyle)) {
                if (!knownStyles.containsKey(((OdfStyle) style).getStyleNameAttribute())) {
                    List<OdfStyleBase> parents = new LinkedList<OdfStyleBase>();
                    OdfStyleBase parent = style;

                    // Collecting hierachy, to go back through the style hierarchy from the end, to be able to neglect empty styles and adjust parent style attribute
                    while (parent != null
                        && (parent instanceof OdfDefaultStyle || !knownStyles.containsKey(((OdfStyle) parent).getStyleNameAttribute()))) {
                        if (parent instanceof OdfDefaultStyle && stylesBase instanceof OdfOfficeStyles) {
                            triggerDefaultStyleOp(styleFamily, ((OdfOfficeStyles)stylesBase).getDefaultStyle(styleFamily));
                            // NEXT: there is no style above a default in the style hierarchy
                            break;
                        } else if (parent != null) {
                            parents.add(parent);

                            // NEXT: get the next parent style and if the style parent name is the OX DEFAULT NAME remove it
                            Attr parentStyleName = parent.getAttributeNodeNS(OdfDocumentNamespace.STYLE.getUri(), "parent-style-name");
                            if (parentStyleName != null && parentStyleName.getValue().equals(Names.OX_DEFAULT_STYLE_PREFIX + Component.getFamilyID(styleFamily) + Names.OX_DEFAULT_STYLE_SUFFIX)) {
                                parent.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "parent-style-name");
                                triggerDefaultStyleOp(styleFamily, style.getParentStyle());
                                break;
                            } else {
                                parent = parent.getParentStyle();
                            }
                        }

                        // trigger operation only for those style not already existing
                        // check if the named style already exists
                    }

                    String lastWrittenStyleName = null; // Only write out parents with mapped styles
                    boolean skippedEmptyParent = false;
                    // Intermediate ODF properties
                    Map<String, Map<String, String>> allOdfProps = new HashMap<String, Map<String, String>>();
                    // The property groups for this component, e.g. cell, paragraph, text for a cell with properties
                    Map<String, OdfStylePropertiesSet> familyPropertyGroups = Component.getAllOxStyleGroupingIdProperties(styleFamily);
                    // Mapped properties
                    Map<String, Object> mappedFormatting = null;

                    // addChild named style to operation
                    String styleName;

                    // due to inheritance the top ancestor style have to be propagated first
                    for (int i = parents.size() - 1; i >= 0; i--) {
                        style = parents.get(i);
                        if(style instanceof OdfDefaultStyle) {
                        	continue;
                        }
                        styleName = ((OdfStyle) style).getStyleNameAttribute();
                        // get all ODF properties from this style
                        MapHelper.getStyleProperties(style, familyPropertyGroups, allOdfProps);
                        // mapping the ODF attribute style props to our component properties
                        mappedFormatting = MapHelper.mapStyleProperties(familyPropertyGroups, allOdfProps);
                        if(isAutoStyle) {
                        	MapHelper.putNumberFormat(mappedFormatting, null, (OdfStyle)style, stylesBase, styles.getOfficeStyles());
                        }
                        else {
                        	MapHelper.putNumberFormat(mappedFormatting, null, (OdfStyle)style, null, stylesBase);
                        }

                        // No OdfStyle, as the parent still might be a default style without name
                        OdfStyleBase parentStyle = style.getParentStyle();
                        String parentStyleName = null;

                        // Default styles do not have a name
                        if (parentStyle != null && !(parentStyle instanceof OdfDefaultStyle)) {
                            parentStyleName = ((OdfStyle) parentStyle).getStyleNameAttribute();
                        }
                        String nextStyle = ((OdfStyle) style).getStyleNextStyleNameAttribute();
                        Integer outlineLevel = ((OdfStyle) style).getStyleDefaultOutlineLevelAttribute();
                        // Do not trigger operations to create empty styles
                        if (skippedEmptyParent) {
                            parentStyleName = lastWrittenStyleName;
                        }
                        String familyId = Component.getFamilyID(styleFamily);
                        if (parentStyleName != null && !parentStyleName.isEmpty()) {
                            insertStyleSheet(styleName, familyId, ((OdfStyle) style).getStyleDisplayNameAttribute(), mappedFormatting, parentStyleName, nextStyle, outlineLevel, false, false, isAutoStyle);
                        } else if(isAutoStyle) {	// the autostyle without parent
                        	insertStyleSheet(styleName, familyId, ((OdfStyle) style).getStyleDisplayNameAttribute(), mappedFormatting, null, nextStyle, outlineLevel, false, true, isAutoStyle);
                    	}
                    	else {
                    		insertStyleSheet(styleName, familyId, ((OdfStyle) style).getStyleDisplayNameAttribute(), mappedFormatting, Names.OX_DEFAULT_STYLE_PREFIX + familyId + Names.OX_DEFAULT_STYLE_SUFFIX, nextStyle, outlineLevel, false, false, isAutoStyle);
                        }

                        lastWrittenStyleName = styleName;
                        mappedFormatting.clear();
                        allOdfProps.clear();
                        // addChild named style to known styles, so it will be only executed once
                        knownStyles.put(styleName, Boolean.TRUE);
                    }
                }
            } else {
                // DEFAULT STYLE PARENT
                // Default styles will receive a name and will be referenced by the root style (the default style for the web editor)
                if (styleFamily.equals(OdfStyleFamily.Paragraph)) {
                    triggerDefaultStyleOp(styleFamily, style);
                } else {
                    triggerDefaultStyleOp(styleFamily, style);
                }
            }
        }
    }

    /**
     * Tests first if the default style was already added to the document, than
     * triggers a insertStylesheet operation
     */
    public Integer triggerDefaultStyleOp(OdfStyleFamily styleFamily, OdfStyleBase style)
    	throws JSONException {

    	Integer defaultTabStopWidth = null;
        // Intermediate ODF properties
        Map<String, Map<String, String>> allOdfProps = new HashMap<String, Map<String, String>>();
        // The property groups for this component, e.g. cell, paragraph, text for a cell with properties
        Map<String, OdfStylePropertiesSet> familyPropertyGroups = Component.getAllOxStyleGroupingIdProperties(styleFamily);
        // Mapped properties
        Map<String, Object> mappedFormatting = null;

        // addChild named style to operation
        String styleName;

        if (style instanceof OdfDefaultStyle && !knownStyles.containsKey(Names.OX_DEFAULT_STYLE_PREFIX + Component.getFamilyID(styleFamily) + Names.OX_DEFAULT_STYLE_SUFFIX)) {
            // get all ODF properties from this style
            MapHelper.getStyleProperties(style, familyPropertyGroups, allOdfProps);
            // mapping the ODF attribute style props to our component properties
            mappedFormatting = MapHelper.mapStyleProperties(familyPropertyGroups, allOdfProps);
            // Tabulator default size is an attribute in the default style, will be received from static mapping functions
            if (mappedFormatting.containsKey("paragraph")) {
                JSONObject paraProps = (JSONObject) mappedFormatting.get("paragraph");
                if (paraProps.has("document")) {
                    JSONObject documentProps = paraProps.optJSONObject("document");
                    defaultTabStopWidth = documentProps.optInt("defaultTabStop");
                }
            }
            String familyId = Component.getFamilyID(styleFamily);
            // Do not trigger operations to create empty styles
            if (!mappedFormatting.isEmpty()) {
                String displayName = "Default " + Component.getFamilyDisplayName(styleFamily) + " Style";
                insertStyleSheet(Names.OX_DEFAULT_STYLE_PREFIX + familyId + Names.OX_DEFAULT_STYLE_SUFFIX, familyId, displayName, mappedFormatting, null, null, null, true, true, false);
            }
            // addChild named style to known styles, so it will be only executed once
            styleName = Names.OX_DEFAULT_STYLE_PREFIX + Component.getFamilyID(styleFamily) + Names.OX_DEFAULT_STYLE_SUFFIX;
            knownStyles.put(styleName, Boolean.TRUE);
        }
        return defaultTabStopWidth;
    }

    public void insertStyleSheet(String styleId, String familyID, String displayName, Map<String, Object> componentProps, String parentStyle, String nextStyleId, Integer outlineLevel, boolean isDefaultStyle, boolean isHidden, boolean isAutoStyle)
    	throws JSONException {

        final JSONObject insertComponentObject = new JSONObject();
        insertComponentObject.put("name", "insertStyleSheet");
        if (styleId != null && !styleId.isEmpty()) {
            insertComponentObject.put("styleId", styleId);
        }
        insertComponentObject.put("type", familyID);
        if (displayName != null && !displayName.isEmpty()) {
            insertComponentObject.put("styleName", displayName);
        }
        if (familyID.equals("table")) {
            final JSONObject tableStyleAttrs = new JSONObject();
            tableStyleAttrs.put("wholeTable", componentProps);
            insertComponentObject.put("attrs", tableStyleAttrs);

        } else {
            insertComponentObject.put("attrs", componentProps);
        }
        if (parentStyle != null && !parentStyle.isEmpty()) {
            insertComponentObject.put("parent", parentStyle);
        }
        if (isDefaultStyle) {
            insertComponentObject.put("default", isDefaultStyle);
        }
        if (isAutoStyle) {
        	insertComponentObject.put("auto", true);
        }
        else if (isHidden) {
            insertComponentObject.put("hidden", isHidden);
        }
        if (outlineLevel != null || nextStyleId != null) {
            JSONObject paraProps;
            if (componentProps.containsKey("paragraph")) {
                paraProps = (JSONObject) componentProps.get("paragraph");
            } else {
                paraProps = new JSONObject();
                componentProps.put("paragraph", paraProps);
            }
            if (outlineLevel != null) {
                paraProps.put("outlineLevel", outlineLevel - 1);
            }
            if (nextStyleId != null && !nextStyleId.isEmpty()) {
                paraProps.put("nextStyleId", nextStyleId);
            }
            componentProps.put("paragraph", paraProps);
            insertComponentObject.put("attrs", componentProps);
        }
        operationQueue.put(insertComponentObject);
    }

    
    /**
     * Maps the styles of the given stylable ODF element to operations. In case
     * the ODF element uses automatic styles, all styles will be returned as
     * property map. In case the ODF element uses template styles, an operation
     * for each style is being triggered, which not have been triggered by now.
     *
     * @return the mapped automatic style grouped by property set
     */
    public static Map<String, Object> getHardStyles(OdfStylableElement styleElement, OdfStylesDom stylesDom)
    	throws SAXException, JSONException {

    	// Hard formatted properties (automatic styles)
        Map<String, Object> allHardFormatting = null;
        // AUTOMATIC STYLE HANDLING
        if (styleElement.hasAutomaticStyle()) {
            allHardFormatting = getAutomaticStyleHierarchyProps(styleElement, stylesDom);
        }
        return allHardFormatting;
    }
    
    static Map<String, Object> getAutomaticStyleHierarchyProps(OdfStylableElement styleElement, OdfFileDom stylesDom)
    	throws SAXException, JSONException {

    	// Hard formatted properties (automatic styles)
        Map<String, Object> allHardFormatting = null;
        Map<String, Map<String, String>> allOdfProps = null;
        // AUTOMATIC STYLE HANDLING
        if (styleElement.hasAutomaticStyle()) {
            OdfStyleBase style = styleElement.getAutomaticStyle();

            // all ODF properties
            allOdfProps = new HashMap<String, Map<String, String>>();
            List<OdfStyleBase> parents = new LinkedList<OdfStyleBase>();
            parents.add(style);
            OdfStyleBase parent = style.getParentStyle();
            // if automatic style inheritance is possible
            while (parent != null) {
                Node n = parent.getParentNode();
                // if it is no longer an automatic style (template or default style)
                if (n instanceof OdfOfficeStyles) {
                    break;
                }
                parents.add(parent);
                parent = parent.getParentStyle();
            }
            // due to inheritance the top ancestor style have to be propagated first
            boolean numberFormatInserted = false;
            OdfOfficeStyles officeStyles = ((OdfDocument)stylesDom.getDocument()).getStylesDom().getOfficeStyles();
            OdfOfficeAutomaticStyles automaticStyles = ((OdfDocument)stylesDom.getDocument()).getContentDom().getAutomaticStyles();
            for (int i = parents.size() - 1; i >= 0; i--) {
                OdfStyleBase styleBase = parents.get(i);
                MapHelper.getStyleProperties(styleBase, Component.getAllOxStyleGroupingIdProperties(styleElement), allOdfProps);                
                numberFormatInserted |= MapHelper.putNumberFormat(null, allOdfProps, (OdfStyle)styleBase, automaticStyles, officeStyles);
            }
            allHardFormatting = MapHelper.mapStyleProperties(Component.getAllOxStyleGroupingIdProperties(styleElement), allOdfProps);
            if(numberFormatInserted) {
                Map<String, String> cellProps = allOdfProps.get("cell");
                String formatString = cellProps.get("numberformat_code");

                JSONObject jsonCellProps = null;
                if(allHardFormatting.containsKey("cell")) {
                    jsonCellProps = (JSONObject)allHardFormatting.get("cell");
                } else {
                    jsonCellProps = new JSONObject();
                }
                JSONObject numberObject = new JSONObject();
                if(formatString.equals("M/D/YYYY")) {
                	numberObject.put("id", 14);
                }
                else {
                	numberObject.put("code", formatString);
                }
                jsonCellProps.put("numberFormat", numberObject);
                allHardFormatting.put("cell", jsonCellProps);
            }
        }
        return allHardFormatting;
    }
}
