/**
 * **********************************************************************
 *
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER
 *
 * Copyright 2008, 2010 Oracle and/or its affiliates. All rights reserved.
 *
 * Use is subject to license terms.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at http://www.apache.org/licenses/LICENSE-2.0. You can also
 * obtain a copy of the License at http://odftoolkit.org/docs/license.txt
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************************
 */
package org.odftoolkit.odfdom.dom;

import static org.odftoolkit.odfdom.component.PageArea.FOOTER_DEFAULT;
import static org.odftoolkit.odfdom.component.PageArea.FOOTER_EVEN;
import static org.odftoolkit.odfdom.component.PageArea.FOOTER_FIRST;
import static org.odftoolkit.odfdom.component.PageArea.HEADER_DEFAULT;
import static org.odftoolkit.odfdom.component.PageArea.HEADER_EVEN;
import static org.odftoolkit.odfdom.component.PageArea.HEADER_FIRST;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.component.Component;
import org.odftoolkit.odfdom.component.JsonOperationProducer;
import org.odftoolkit.odfdom.component.OdfOperationDocument;
import org.odftoolkit.odfdom.component.PageArea;
import org.odftoolkit.odfdom.dom.element.office.OfficeBodyElement;
import org.odftoolkit.odfdom.dom.element.office.OfficeMasterStylesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleFooterElement;
import org.odftoolkit.odfdom.dom.element.style.StyleFooterLeftElement;
import org.odftoolkit.odfdom.dom.element.style.StyleHeaderElement;
import org.odftoolkit.odfdom.dom.element.style.StyleHeaderLeftElement;
import org.odftoolkit.odfdom.dom.element.style.StyleMasterPageElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableElement;
import org.odftoolkit.odfdom.dom.style.OdfStyleFamily;
import org.odftoolkit.odfdom.incubator.doc.office.OdfOfficeAutomaticStyles;
import org.odftoolkit.odfdom.incubator.doc.office.OdfOfficeMasterStyles;
import org.odftoolkit.odfdom.incubator.doc.office.OdfOfficeStyles;
import org.odftoolkit.odfdom.incubator.doc.office.OdfStylesBase;
import org.odftoolkit.odfdom.incubator.doc.style.OdfStyle;
import org.odftoolkit.odfdom.incubator.doc.style.OdfStylePageLayout;
import org.odftoolkit.odfdom.pkg.OdfElement;
import org.odftoolkit.odfdom.pkg.OdfFileDom;
import org.odftoolkit.odfdom.pkg.OdfName;
import org.odftoolkit.odfdom.pkg.OdfPackage;
import org.odftoolkit.odfdom.pkg.OdfPackageDocument;
import org.odftoolkit.odfdom.pkg.OdfValidationException;
import org.odftoolkit.odfdom.pkg.OdfXMLFactory;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXException;

/**
 * A document in ODF is from the package view a directory with a media type. If
 * the media type represents a document described by the ODF 1.2 Schema, certain
 * files are assumed within: content.xml, styles.xml, metadata.xml and
 * settings.xml.
 *
 * The class represents such a document, providing easier access to its XML
 * files.
 */
public abstract class OdfSchemaDocument extends OdfPackageDocument {

    private OdfContentDom mContentDom;
    private OdfStylesDom mStylesDom;
    private OdfMetaDom mMetaDom;
    private OdfSettingsDom mSettingsDom;
    private OdfOfficeStyles mDocumentStyles;
    private JsonOperationProducer mJsonOperationQueue;

    /**
     * Creates a new OdfSchemaDocument.
     *
     * @param pkg - the ODF Package that contains the document. A baseURL is
     * being generated based on its location.
     * @param internalPath - the directory path within the package from where
     * the document should be loaded.
     * @param mediaTypeString - media type of stream. If unknown null can be
     * used.
     */
    protected OdfSchemaDocument(OdfPackage pkg, String internalPath, String mediaTypeString) {
        super(pkg, internalPath, mediaTypeString);
        ErrorHandler errorHandler = pkg.getErrorHandler();
        if (errorHandler != null) {
            if (pkg.getFileEntry(internalPath + "content.xml") == null && pkg.getFileEntry(internalPath + "styles.xml") == null) {
                try {
                    String baseURI = pkg.getBaseURI();
                    if (baseURI == null) {
                        baseURI = internalPath;
                    } else {
                        if (!internalPath.equals(ROOT_DOCUMENT_PATH)) {
                            baseURI = "/" + internalPath;
                        }
                    }
                    errorHandler.error(new OdfValidationException(OdfSchemaConstraint.DOCUMENT_WITHOUT_CONTENT_NOR_STYLES_XML, baseURI));
                } catch (SAXException ex) {
                    Logger.getLogger(OdfPackage.class.getName()).log(Level.SEVERE, null, ex);
                }
            }
            InputStream mimetypeStream = pkg.getInputStream(OdfPackage.OdfFile.MEDIA_TYPE.getPath(), true);
            if (internalPath.equals(ROOT_DOCUMENT_PATH) && mimetypeStream == null) {
                try {
                    errorHandler.error(new OdfValidationException(OdfSchemaConstraint.PACKAGE_SHALL_CONTAIN_MIMETYPE, pkg.getBaseURI()));
                } catch (SAXException ex) {
                    Logger.getLogger(OdfPackage.class.getName()).log(Level.SEVERE, null, ex);
                }
            }
        }
    }

    /**
     * This enum contains all possible standardized XML ODF files of the
     * OpenDocument document.
     */
    public static enum OdfXMLFile {

        /**
         * The XML file containing the content of an ODF document as specified
         * by the ODF 1.2 specification part 1.
         */
        CONTENT("content.xml"),
        /**
         * The XML file containing a predifined set of metadata related to an
         * ODF document as specified by the ODF 1.2 specification part 1.
         */
        META("meta.xml"),
        /**
         * The XML file containing the settings of an ODF document as specified
         * by the ODF 1.2 specification part 1.
         */
        SETTINGS("settings.xml"),
        /**
         * The XML file containing the styles of an ODF document as specified by
         * the ODF 1.2 specification part 1.
         */
        STYLES("styles.xml");
        private final String mFileName;

        /**
         * @return the file name of xml files contained in odf packages.
         */
        public String getFileName() {
            return mFileName;
        }

        OdfXMLFile(String fileName) {
            this.mFileName = fileName;
        }
    }

    /**
     * @return JSONObject of operations
     */
    // ToDo OX - JSONObject is to be considered..
    public JsonOperationProducer getJsonOperationQueue() {
        return mJsonOperationQueue;
    }

    public void setJsonOperationQueue(JsonOperationProducer queue) {
        mJsonOperationQueue = queue;
    }

    public JSONObject getOperations(OdfOperationDocument operationDoc)
    	throws SAXException, JSONException {

    	JSONObject ops = null;
        JsonOperationProducer queue = getJsonOperationQueue();
        if (queue == null) {
            try {
                this.getStylesDom();
                this.getContentDom();
                queue = getJsonOperationQueue();
            } catch (SAXException ex) {
                Logger.getLogger(OdfSchemaDocument.class.getName()).log(Level.SEVERE, null, ex);
                throw ex;
            }
        }
        if (queue != null) {
            ops = queue.getDocumentOperations();
        }
        return ops;
    }

    /**
     * The component tree is a high level abstraction of components (table,
     * paragraph, character, etc.) from the XML implementation details of the
     * document.
     */
    private Component mRootComponent;

    /**
     * Returns the component tree of the document. The component tree is a high
     * level abstraction of components (table, paragraph, character, etc.) from
     * the XML implementation details of the document.
     *
     * The DOM of the content.xml will be created if not done before.
     */
    public Component getRootComponent() {
        if (mRootComponent == null) {
            try {
                // Access the DOM of the content.xml so the XML is parsed once!!
                this.getContentDom();
            } catch (Exception ex) {
                Logger.getLogger(OdfSchemaDocument.class.getName()).log(Level.SEVERE, null, ex);
            }
        }
        return mRootComponent;
    }

    /**
     * Returns the component tree of the document. The component tree is a high
     * level abstraction of components (table, paragraph, character, etc.) from
     * the XML implementation details of the document.
     *
     * The DOM of the content.xml will be created if not done before.
     *
     * @param masterStyleName the name of the master style
     * @param localName the local name of the header or footer XML element
     * @return the header or footer element belonging to the given master page
     * style
     */
    public OdfElement getRootComponentElement(String masterStyleName, PageArea pageArea, boolean createIfNotExisting) {
        OdfElement targetElement = null;
        try {
            OdfStylesDom stylesDom = getStylesDom();
            if (stylesDom != null) {
                OdfOfficeMasterStyles masterStyles = stylesDom.getMasterStyles();
                if (masterStyles == null) {
                    masterStyles = stylesDom.getOrCreateMasterStyles();
                }
                StyleMasterPageElement masterPage = masterStyles.getOrCreateMasterPage(masterStyleName);
                if (pageArea.getPageAreaName().contains("header")) {
                    String localName = null;
                    if (pageArea.equals(HEADER_FIRST)) {
                        // header-first
                        localName = HEADER_FIRST.getLocalName();
                        // targetElement =  OdfXMLFactory.newOdfElement(stylesDom, STYLE_FIRST_PAGE);
                    } else if (pageArea.equals(HEADER_EVEN)) {
                        localName = HEADER_EVEN.getLocalName();
                    } else {
                        localName = HEADER_DEFAULT.getLocalName();
                    }
                    targetElement = (OdfElement) masterPage.getChildElement(StyleHeaderElement.ELEMENT_NAME.getUri(), localName, 0);

                    OdfOfficeAutomaticStyles autoStyles = stylesDom.getOrCreateAutomaticStyles();
                    String pageLayoutName = masterPage.getStylePageLayoutNameAttribute();
                    if (targetElement == null && createIfNotExisting) { // create a new page area
                        if (pageArea.equals(HEADER_FIRST)) {
                            // header-first
                            targetElement = OdfXMLFactory.newOdfElement(stylesDom, STYLE_HEADER_FIRST);
                        } else if (pageArea.equals(HEADER_EVEN)) {
                            targetElement = new StyleHeaderLeftElement(stylesDom);
                            OdfStylePageLayout pageLayout = null;
                            if(pageLayoutName != null && !pageLayoutName.isEmpty()){
                                pageLayout = autoStyles.getOrCreatePageLayout(pageLayoutName);
                            }else{
                                pageLayout = autoStyles.newPageLayout();
                            }
                            pageLayout.setStylePageUsageAttribute("right");

                        } else {
                            targetElement = new StyleHeaderElement(stylesDom);
                        }
                        masterPage.appendChild(targetElement);
                    }
                } else {
                    String localName = null;
                    if (pageArea.equals(FOOTER_FIRST)) {
                        // header-first
                        localName = FOOTER_FIRST.getLocalName();
                        // targetElement =  OdfXMLFactory.newOdfElement(stylesDom, STYLE_FIRST_PAGE);
                    } else if (pageArea.equals(FOOTER_EVEN)) {
                        localName = FOOTER_EVEN.getLocalName();
                    } else {
                        localName = FOOTER_DEFAULT.getLocalName();
                    }
                    targetElement = (OdfElement) masterPage.getChildElement(StyleFooterElement.ELEMENT_NAME.getUri(), localName, 0);

                    OdfOfficeAutomaticStyles autoStyles = stylesDom.getOrCreateAutomaticStyles();
                    String pageLayoutName = masterPage.getStylePageLayoutNameAttribute();
                    if (targetElement == null && createIfNotExisting) { // create a new page area
                        if (pageArea.equals(FOOTER_FIRST)) {
                            // header-first
                            targetElement = OdfXMLFactory.newOdfElement(stylesDom, STYLE_FOOTER_FIRST);
                        } else if (pageArea.equals(FOOTER_EVEN)) {
                            targetElement = new StyleFooterLeftElement(stylesDom);
                            OdfStylePageLayout pageLayout = autoStyles.getOrCreatePageLayout(pageLayoutName);
                            pageLayout.setStylePageUsageAttribute("right");

                        } else {
                            targetElement = new StyleFooterElement(stylesDom);
                        }
                        masterPage.appendChild(targetElement);
                    }
                }
            }
        } catch (Exception ex) {
            Logger.getLogger(OdfSchemaDocument.class.getName()).log(Level.SEVERE, null, ex);
        }
        return targetElement;
    }
    private static final OdfName STYLE_HEADER_FIRST = OdfName.newName(OdfDocumentNamespace.STYLE, "header-first");
    private static final OdfName STYLE_FOOTER_FIRST = OdfName.newName(OdfDocumentNamespace.STYLE, "footer-first");

    /**
     * ToDo API - Would be nice if we could keep this package layer, as the root
     * component should only be set internally
     */
    public void setRootComponent(Component rootComponent) {
        mRootComponent = rootComponent;
    }

    /**
     * Gets the ODF content.xml file as stream.
     *
     * @return - a stream of the ODF content 'content.xml' file
     * @throws java.lang.Exception - if the stream can not be extracted
     */
    public InputStream getContentStream() throws Exception {
        String path = getXMLFilePath(OdfXMLFile.CONTENT);
        return mPackage.getInputStream(path);
    }

    /**
     * Gets the ODF style.xml file as stream.
     *
     * @return - a stream of the ODF style 'styles.xml' file
     * @throws java.lang.Exception - if the stream can not be extracted
     */
    public InputStream getStylesStream() throws Exception {
        return mPackage.getInputStream(getXMLFilePath(OdfXMLFile.STYLES));
    }

    /**
     * Gets the ODF settings.xml file as stream.
     *
     * @return - a stream of the ODF settings 'setting.xml' file
     * @throws java.lang.Exception - if the stream can not be extracted
     */
    public InputStream getSettingsStream() throws Exception {
        return mPackage.getInputStream(getXMLFilePath(OdfXMLFile.SETTINGS));
    }

    /**
     * Gets the ODF metadata.xml file as stream.
     *
     * @return - a stream of the ODF metadata 'meta.xml' file
     * @throws java.lang.Exception - if the stream can not be extracted
     */
    public InputStream getMetaStream() throws Exception {
        return mPackage.getInputStream(getXMLFilePath(OdfXMLFile.META));
    }

    /**
     * Get the relative path for an embedded ODF document including its file
     * name.
     *
     * @param file represents one of the standardized XML ODF files.
     * @return path to embedded ODF XML file relative to ODF package root.
     */
    protected String getXMLFilePath(OdfXMLFile file) {
        return file.mFileName;
    }

    /**
     * Get the URI, where this ODF document is stored.
     *
     * @return the URI to the ODF document. Returns null if document is not
     * stored yet.
     */
    public String getBaseURI() {
        return mPackage.getBaseURI();

    }

    /**
     * Return the ODF type-based content DOM of the content.xml
     *
     * @return ODF type-based content DOM or null if no content.xml exists.
     * @throws Exception if content DOM could not be initialized
     */
    public OdfContentDom getContentDom() throws SAXException {
        if (mContentDom == null) {
            mContentDom = (OdfContentDom) getFileDom(OdfXMLFile.CONTENT);
        }
        return mContentDom;
    }

    /**
     * Return the ODF type-based styles DOM of the styles.xml
     *
     * @return ODF type-based styles DOM or null if no styles.xml exists.
     * @throws Exception if styles DOM could not be initialized
     */
    public OdfStylesDom getStylesDom() throws SAXException {
        if (mStylesDom == null) {
            mStylesDom = (OdfStylesDom) getFileDom(OdfXMLFile.STYLES);
        }
        return mStylesDom;
    }

    /**
     * Return the ODF type-based metadata DOM of the meta.xml
     *
     * @return ODF type-based meta DOM or null if no meta.xml exists.
     * @throws Exception if meta DOM could not be initialized
     */
    public OdfMetaDom getMetaDom() throws SAXException {
        if (mMetaDom == null) {
            mMetaDom = (OdfMetaDom) getFileDom(OdfXMLFile.META);
        }
        return mMetaDom;
    }

    /**
     * Return the ODF type-based settings DOM of the settings.xml
     *
     * @return ODF type-based settings DOM or null if no settings.xml exists.
     * @throws Exception if settings DOM could not be initialized
     */
    public OdfSettingsDom getSettingsDom() throws SAXException {
        if (mSettingsDom == null) {
            mSettingsDom = (OdfSettingsDom) getFileDom(OdfXMLFile.SETTINGS);
        }
        return mSettingsDom;
    }

    /**
     * Sets the ODF type-based content DOM of the content.xml
     *
     * @param contentDom ODF type-based content DOM or null if no content.xml
     * exists.
     */
    public void setContentDom(OdfContentDom contentDom) {
        mContentDom = contentDom;
    }

    /**
     * Sets the ODF type-based styles DOM of the styles.xml
     *
     * @param stylesDom ODF type-based styles DOM or null if no styles.xml
     * exists.
     */
    public void setStylesDom(OdfStylesDom stylesDom) {
        mStylesDom = stylesDom;
    }

    /**
     * Sets the ODF type-based meta DOM of the meta.xml
     *
     * @param metaDom ODF type-based meta DOM or null if no meta.xml exists.
     */
    public void setMetaDom(OdfMetaDom metaDom) {
        mMetaDom = metaDom;
    }

    /**
     * Sets the ODF type-based settings DOM of the settings.xml
     *
     * @param settingsDom ODF type-based settings DOM or null if no settings.xml
     * exists.
     */
    public void setSettingsDom(OdfSettingsDom settingsDom) {
        mSettingsDom = settingsDom;
    }

    /**
     *
     * @return the office:styles element from the styles dom or null if there is
     * no such element.
     */
    public OdfOfficeStyles getDocumentStyles() {
        if (mDocumentStyles == null) {
            try {
                OdfFileDom stylesDom = getStylesDom();
                if (stylesDom != null) {
                    mDocumentStyles = OdfElement.findFirstChildNode(OdfOfficeStyles.class, stylesDom.getFirstChild());
                    if (mDocumentStyles == null) {
                        this.getOrCreateDocumentStyles();
                    }
                } else {
                    // ToDo: the below - as styles.xml is mandatory in ODF
                    // give an invalid ODF message
                    // create new styles.xml file
                    // call this method again
                }
            } catch (Exception ex) {
                Logger.getLogger(OdfSchemaDocument.class.getName()).log(Level.SEVERE, null, ex);
            }
        }
        return mDocumentStyles;
    }

    /**
     * return the office:master-styles element of this document.
     *
     * @return the office:master-styles element
     */
    public OdfOfficeMasterStyles getOfficeMasterStyles() {
        try {
            OdfStylesDom stylesDom = getStylesDom();
            if (stylesDom != null) {
                return stylesDom.getMasterStyles();
            }
        } catch (Exception ex) {
            Logger.getLogger(OdfSchemaDocument.class.getName()).log(Level.SEVERE, null, ex);
        }
        return null;
    }

    /**
     *
     * @return the office:styles element from the styles dom. If there is not
     * yet such an element, it is created.
     */
    public OdfOfficeStyles getOrCreateDocumentStyles() {
        if (mDocumentStyles == null) {
            try {
                OdfStylesDom stylesDom = getStylesDom();
                if (stylesDom != null) {
                    Node parent = stylesDom.getFirstChild();
                    if (parent != null) {
                        mDocumentStyles = OdfElement.findFirstChildNode(OdfOfficeStyles.class, parent);
                        if (mDocumentStyles == null) {
                            mDocumentStyles = stylesDom.newOdfElement(OdfOfficeStyles.class);
                            parent.insertBefore(mDocumentStyles, stylesDom.getOrCreateAutomaticStyles());
                        }
                    }
                } else {
                    // give an invalid ODF message
                    // create new styles.xml file
                    // call this method again
                    // ToDo: do the above instead
                    return null;
                }
            } catch (Exception ex) {
                Logger.getLogger(OdfSchemaDocument.class.getName()).log(Level.SEVERE, null, ex);
            }
        }
        return mDocumentStyles;
    }

    public OdfStyle getStyleByName(OdfStyleFamily styleFamily, String styleName)
    	throws SAXException {

    	OdfStyle odfStyle = getStyleByName(getStylesDom(), styleFamily, styleName);
    	if(odfStyle==null) {
        	final OdfContentDom odfContentDom = getContentDom();
        	if(odfContentDom!=null) {
        		odfStyle = getStyleByName(odfContentDom.getAutomaticStyles(), styleFamily, styleName);
        	}
    	}
    	return odfStyle;
    }

    private OdfStyle getStyleByName(OdfStylesDom odfStylesDom, OdfStyleFamily styleFamily, String styleName) {
    	OdfStyle odfStyle = null;
    	if(odfStylesDom!=null) {
    		odfStyle = getStyleByName(odfStylesDom.getOfficeStyles(), styleFamily, styleName);
    		if(odfStyle==null) {
    			odfStyle = getStyleByName(odfStylesDom.getAutomaticStyles(), styleFamily, styleName);
    		}
    	}
    	return odfStyle;
    }

    private OdfStyle getStyleByName(OdfStylesBase odfStylesBase, OdfStyleFamily styleFamily, String styleName) {
    	OdfStyle odfStyle = null;
		if(odfStylesBase!=null) {
			odfStyle = odfStylesBase.getStyle(styleName, styleFamily);
		}
    	return odfStyle;
    }

    public OdfStyle getStyleByDisplayName(OdfStyleFamily styleFamily, String styleDisplayName)
    	throws SAXException {

    	OdfStyle odfStyle = getStyleByDisplayName(getStylesDom(), styleFamily, styleDisplayName);
    	if(odfStyle==null) {
    		final OdfContentDom odfContentDom = getContentDom();
    		if(odfContentDom!=null) {
    			odfStyle = getStyleByDisplayName(odfContentDom.getAutomaticStyles(), styleFamily, styleDisplayName);
    		}
    	}
    	return odfStyle;
    }

    private OdfStyle getStyleByDisplayName(OdfStylesDom odfStylesDom, OdfStyleFamily styleFamily, String styleName) {
    	OdfStyle odfStyle = null;
    	if(odfStylesDom!=null) {
    		odfStyle = getStyleByDisplayName(odfStylesDom.getOfficeStyles(), styleFamily, styleName);
    		if(odfStyle==null) {
    			odfStyle = getStyleByDisplayName(odfStylesDom.getAutomaticStyles(), styleFamily, styleName);
    		}
    	}
    	return odfStyle;
    }

    private OdfStyle getStyleByDisplayName(OdfStylesBase odfStylesBase, OdfStyleFamily styleFamily, String styleName) {
		if(odfStylesBase!=null) {
			for(OdfStyle odfStyle:odfStylesBase.getStylesForFamily(styleFamily)) {
				final String displayName = odfStyle.getStyleDisplayNameAttribute();
				if(displayName!=null&&displayName.equals(styleName)) {
					return odfStyle;
				}
			}
		}
    	return null;
    }

    /**
     * Return a list of table features in this document.
     *
     * @return a list of table features in this document.
     */
    // ToDo: Instead of a method to receive all possible feature/components on the document, there might be a generic or one each element?
    public List<TableTableElement> getTables() {
        List<TableTableElement> tableList = new ArrayList<TableTableElement>();
        try {
            // find tables from content.xml
            OfficeBodyElement officeBody = OdfElement.findFirstChildNode(OfficeBodyElement.class, getContentDom().getRootElement());
            OdfElement contentRoot = OdfElement.findFirstChildNode(OdfElement.class, officeBody);
            tableList = fillTableList(contentRoot, tableList);

            // find tables from styles.xml (header & footer)
            Map<String, StyleMasterPageElement> masterPages = getMasterPages();
            StyleMasterPageElement defaultMasterPage = masterPages.get("Standard");
            if (defaultMasterPage != null) {
                tableList = fillTableList(defaultMasterPage, tableList);
            }
        } catch (Exception ex) {
            Logger.getLogger(OdfSchemaDocument.class.getName()).log(Level.SEVERE, null, ex);
        }
        return tableList;
    }

    // Only tables being on root level are being considered
    private List<TableTableElement> fillTableList(Element startElement, List<TableTableElement> tableList) {
        NodeList childList = startElement.getChildNodes();
        for (int i = 0;
            i < childList.getLength();
            i++) {
            Node childNode = childList.item(i);
            if (childNode instanceof Element) {
                if (childNode instanceof TableTableElement) {
                    tableList.add((TableTableElement) childList.item(i));
                } else {
                    fillTableList((Element) childNode, tableList);
                }
            }
        }
        return tableList;
    }

    /**
     * ToDo: Instead of adding all elements using an index to the document, we
     * might add a pattern to the code generation to create a HashMap either on
     * demand (whenever such a structure is required from the user) or by
     * default
     *
     * @deprecated This method will be moved to the generated sources as soon
     * code generation was improved!
     *
     */
    @Deprecated
    public Map<String, StyleMasterPageElement> getMasterPages() throws Exception {

        // get original values:
        OdfStylesDom stylesDoc = getStylesDom();
        OfficeMasterStylesElement masterStyles = OdfElement.findFirstChildNode(OfficeMasterStylesElement.class, stylesDoc.getRootElement());
        Map<String, StyleMasterPageElement> masterPages = null;
        if (masterStyles != null) {
            NodeList lstMasterPages = stylesDoc.getElementsByTagNameNS(OdfDocumentNamespace.STYLE.getUri(), "master-page");
            if (lstMasterPages != null && lstMasterPages.getLength() > 0) {
                masterPages = new HashMap();
                for (int i = 0; i < lstMasterPages.getLength(); i++) {
                    StyleMasterPageElement masterPage = (StyleMasterPageElement) lstMasterPages.item(i); //Take the node from the list
                    //ToDo: Drop Attribute Suffix for methods returning String values and NOT Attributes
                    String styleName = masterPage.getStyleNameAttribute();
                    masterPages.put(styleName, masterPage);
                }
            }
        }
        return masterPages;
    }

    /**
     * Close the OdfPackage and release all temporary created data. Acter
     * execution of this method, this class is no longer usable. Do this as the
     * last action to free resources. Closing an already closed document has no
     * effect. Note that this will not close any cached documents.
     */
    @Override
    public void close() {
        mContentDom = null;
        mStylesDom = null;
        mMetaDom = null;
        mSettingsDom = null;
        mDocumentStyles = null;
        super.close();
    }

    public OdfFileDom getFileDom(OdfXMLFile file) throws SAXException {
        return getFileDom(getXMLFilePath(file));
    }
}
