/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.json.actions;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.documentation.RequestMethod;
import com.openexchange.documentation.annotations.Action;
import com.openexchange.documentation.annotations.Parameter;
import com.openexchange.documentconverter.Feature;
import com.openexchange.documentconverter.IManager;
import com.openexchange.documentconverter.Properties;
import com.openexchange.exception.OXException;
import com.openexchange.file.storage.DefaultFile;
import com.openexchange.file.storage.File;
import com.openexchange.file.storage.File.Field;
import com.openexchange.file.storage.FileStorageFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccessFactory;
import com.openexchange.file.storage.composition.IDBasedFolderAccess;
import com.openexchange.file.storage.composition.IDBasedFolderAccessFactory;
import com.openexchange.groupware.infostore.InfostoreExceptionCodes;
import com.openexchange.office.DocumentProperties;
import com.openexchange.office.FilterException;
import com.openexchange.office.IImporter;
import com.openexchange.office.doc.imexport.ImExportHelper;
import com.openexchange.office.json.tools.fields.DynamicFields;
import com.openexchange.office.tools.MimeTypeHelper;
import com.openexchange.office.tools.StorageHelper;
import com.openexchange.office.tools.StreamInfo;
import com.openexchange.office.tools.doc.DocumentFormatHelper;
import com.openexchange.office.tools.doc.DocumentType;
import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.office.tools.error.ExceptionToErrorCode;
import com.openexchange.office.tools.files.DocFileHelper;
import com.openexchange.office.tools.files.FileDescriptor;
import com.openexchange.office.tools.files.FileHelper;
import com.openexchange.office.tools.files.FolderHelper;
import com.openexchange.office.tools.files.LocalFileMappingManager;
import com.openexchange.office.tools.message.MessagePropertyKey;
import com.openexchange.office.tools.monitoring.CreateEvent;
import com.openexchange.office.tools.monitoring.DocumentEvent;
import com.openexchange.office.tools.monitoring.ErrorType;
import com.openexchange.office.tools.monitoring.Statistics;
import com.openexchange.server.ServiceLookup;
import com.openexchange.tools.session.ServerSession;
import com.openexchange.tx.TransactionAwares;

public class CreateFromTemplateAction extends DocumentFilterAction {

    private static final Log LOG = com.openexchange.log.Log.loggerFor(CreateFromTemplateAction.class);
    private static final String UNNAMED = "unnamed";

    private LocalFileMappingManager m_localFileMapper = null;

    public CreateFromTemplateAction(ServiceLookup services, LocalFileMappingManager localFileMapper) {
        super(services);
        m_localFileMapper = localFileMapper;
    }

    @Action(method = RequestMethod.GET, name = "createfromtemplate", description = "Request to create a new file based on a specific template.", parameters = {
        @Parameter(name = "session", description = "A session ID previously obtained from the login module."),
        @Parameter(name = "file_id", description = "The id of the file to be used as a template."),
        @Parameter(name = "target_filename", description = "The initial file name of the new file to be created"),
        @Parameter(name = "target_folder_id", description = "Folder ID of the requested infoitem."),
        @Parameter(name = "uid", description = "The unique id of the client application.")
    }, responseDescription = "Response with timestamp: A JSON object containing the id and folder_id of the newly created document.")

    @Override
    public AJAXRequestResult perform(AJAXRequestData request, ServerSession session) throws OXException {
        AJAXRequestResult requestResult = null;
        DocumentEvent documentEvent = null;

        if ((null != request) && (null != session)) {
            final IDBasedFileAccessFactory fileFactory = m_services.getService(IDBasedFileAccessFactory.class);
            final IDBasedFileAccess fileAccess = (null != fileFactory) ? fileFactory.createAccess(session) : null;

            DocumentType eventDocumentType = DocumentType.NONE;
            InputStream inputStm = null;
            boolean rollback = false;

            try {
                final File file = new DefaultFile();
                final String fileId = request.getParameter("file_id");

                String targetFolderId = request.getParameter("target_folder_id");
                if (StringUtils.isNotBlank(targetFolderId)) {
                    final IDBasedFolderAccessFactory folderFactory = m_services.getService(IDBasedFolderAccessFactory.class);
                    final IDBasedFolderAccess folderAccess = (null != folderFactory) ? folderFactory.createAccess(session) : null;
                    if (!FolderHelper.folderHasCreateAccess(folderAccess, targetFolderId)) {
                        targetFolderId = null;
                    }
                }
                if (StringUtils.isBlank(targetFolderId)) {
                    targetFolderId = DocFileHelper.getUserDocumentsFolderId(session, m_services);
                }

                if ((null != fileId) && (null != fileAccess) && (null != targetFolderId)) {
                    // Create document based on template file. It could be that we have to
                    // convert the template document to the related ODF document format.
                    boolean bLocalFile = fileId.startsWith("template:");
                    final String version = FileStorageFileAccess.CURRENT_VERSION;
                    String fullFilePath = null;
                    String extensionType = null;
                    String mimeType = null;
                    File templateFile = null;
                    String templateFileName = null;

                    if (bLocalFile) {
                        // a local file must be treated differently
                        fullFilePath = m_localFileMapper.getPathFromId(fileId);
                        if (null == fullFilePath) {
                            throw new OXException();
                        }

                        String fileName = FileHelper.getFileName(fullFilePath);
                        extensionType = FileHelper.getExtension(fileName);
                        mimeType = MimeTypeHelper.getMimeTypeFromTemplateExtension(extensionType);
                    } else {
                        templateFile = fileAccess.getFileMetadata(fileId, version);
                        templateFileName = templateFile.getFileName();
                        extensionType = FileHelper.getExtension(templateFileName);
                        mimeType = templateFile.getFileMIMEType();
                    }

                    // we can continue if we have a path or not local file
                    if ((bLocalFile && (null != fullFilePath)) || !bLocalFile) {
                        HashMap<String, String> conversionFormat = DocumentFormatHelper.getConversionFormatInfo(mimeType, extensionType);

                        if (conversionFormat != null) {
                            // conversion already sets the correct target document format
                            final IManager dcManager = m_services.getService(IManager.class);
                            ErrorCode errorCode = ErrorCode.NO_ERROR;

                            if ((null != dcManager) && dcManager.hasFeature(Feature.DOCUMENTCONVERTER)) {
                                StreamInfo streamInfo = null;
                                InputStream documentInputStm = null;

                                if (bLocalFile) {
                                    // local file can directly read from the local file system
                                    documentInputStm = DocFileHelper.getDocumentStreamFromLocalFile(fullFilePath);
                                } else {
                                    // OX Drive file must be loaded using Files API
                                    AJAXRequestData loadRequest = new AJAXRequestData();
                                    loadRequest.putParameter("id", fileId);
                                    loadRequest.putParameter("folder_id", targetFolderId);
                                    loadRequest.putParameter("version", version);
                                    loadRequest.putParameter("fileName", templateFileName);
                                    final StorageHelper storageHelper = new StorageHelper(m_services, session, targetFolderId);
                                    streamInfo = DocFileHelper.getDocumentStream(m_services, session, loadRequest, storageHelper);
                                    documentInputStm = streamInfo.getDocumentStream();
                                    errorCode = streamInfo.getErrorCode();
                                }

                                if (null != documentInputStm) {
                                    HashMap<String, Object> jobProperties = new HashMap<String, Object>(4);
                                    HashMap<String, Object> resultProperties = new HashMap<String, Object>(8);
                                    final String filterShortName = conversionFormat.get(Properties.PROP_FILTER_SHORT_NAME);

                                    jobProperties.put(Properties.PROP_INPUT_STREAM, documentInputStm);
                                    jobProperties.put(Properties.PROP_FILTER_SHORT_NAME, filterShortName);

                                    if (null != templateFileName) {
                                        jobProperties.put(Properties.PROP_INFO_FILENAME, templateFileName);
                                    }

                                    inputStm = dcManager.convert(filterShortName, jobProperties, resultProperties);
                                    IOUtils.closeQuietly(documentInputStm);

                                    if (null != streamInfo) {
                                        streamInfo.close();
                                    }

                                    // set new mime type and extensionType
                                    mimeType = conversionFormat.get(Properties.PROP_MIME_TYPE);
                                    extensionType = conversionFormat.get(Properties.PROP_INPUT_TYPE);
                                } else {
                                    // no document stream -> provide error information
                                    documentEvent = new DocumentEvent(eventDocumentType, ErrorType.CREATE);
                                    Statistics.handleDocumentEvent(documentEvent);

                                    if (null != streamInfo) {
                                        streamInfo.close();
                                    }

                                    return new AJAXRequestResult(createErrorObject(errorCode));
                                }
                            }

                            eventDocumentType = DocumentType.valueOf(conversionFormat.get("DocumentType"));
                        } else {
                            // we have to set the correct target document format (created from the template)
                            HashMap<String, String> targetFormat = DocumentFormatHelper.getTemplateTargetFormatInfo(mimeType, extensionType);
                            if (bLocalFile) {
                                inputStm = DocFileHelper.getDocumentStreamFromLocalFile(fullFilePath);
                            } else {
                                inputStm = fileAccess.getDocument(fileId, version);
                            }

                            // set new mime type and extensionType for the target document file
                            if (null != targetFormat) {
                                mimeType = targetFormat.get(Properties.PROP_MIME_TYPE);
                                extensionType = targetFormat.get(Properties.PROP_INPUT_TYPE);
                            }
                        }

                        // create a new file store item with default content
                        file.setId(FileStorageFileAccess.NEW);
                        file.setFolderId(targetFolderId);

                        final String initialFileName = request.getParameter("target_filename");
                        final String createFileName = ((null == initialFileName) || (initialFileName.length() < 1)) ? UNNAMED : initialFileName;
                        file.setFileName(createFileName + "." + extensionType);
                        file.setFileMIMEType(mimeType);

                        if (null != inputStm) {
                            final IImporter importer = ImExportHelper.getImporterService(m_services, DocFileHelper.getDocumentFormat("." + extensionType));
                            if(importer!=null) {
                                try {
                                    final DocumentProperties documentProperties = new DocumentProperties();
                                    documentProperties.put(DocumentProperties.PROP_SAVE_TEMPLATE_DOCUMENT, false);
                                    final InputStream documentStm = importer.getDefaultDocument(inputStm, documentProperties);
                                    if (documentStm != inputStm) {
                                        // Close resource input stream if default document stream
                                        // was created by the import getDefaultDocument function. Don't
                                        // do that if inputStm refers to resourceInputStream
                                        IOUtils.closeQuietly(inputStm);
                                    }
                                    inputStm = documentStm;
                                }
                                catch(FilterException e) {
                                    IOUtils.closeQuietly(inputStm);
                                    inputStm = null;
                                }
                            }
                        }

                        if (null != inputStm) {
                            final JSONObject result = new JSONObject(2);

                            try
                            {
                                inputStm = DynamicFields.insertOptionalFields(inputStm, extensionType, request, result, session, m_services);
                            }
                            catch (Exception e)
                            {
                                ErrorCode errorCode = ErrorCode.GENERAL_UNKNOWN_ERROR;
                                LOG.error("Handling optional fields is broken " + errorCode.getDescription(), e);
                                try
                                {
                                    requestResult = new AJAXRequestResult(createErrorObject(errorCode));
                                    documentEvent = new DocumentEvent(eventDocumentType, ErrorType.CREATE);
                                }
                                catch (final JSONException je)
                                {
                                    LOG.warn("Couldn't create JSON object while creating new document from template");
                                }
                            }

                            fileAccess.startTransaction();
                            rollback = true;
                            try
                            {
                                LOG.trace("RT connection: [CreateFromTemplate] saveDocument using meta data: " + ((null != file) ? file.toString() : "null"));
                                fileAccess.saveDocument(file, inputStm, FileStorageFileAccess.DISTANT_FUTURE, new ArrayList<Field>());
                            }
                            catch (OXException e)
                            {
                                //if 'only' validation of the name fails, we try it again with 'unnamend'
                                if (e.getCode() == InfostoreExceptionCodes.VALIDATION_FAILED_CHARACTERS.getNumber() || e.getCode() == InfostoreExceptionCodes.VALIDATION_FAILED_SLASH.getNumber()){
                                    file.setFileName(UNNAMED + "." + extensionType);
                                    LOG.trace("RT connection: [CreateFromTemplate] saveDocument using meta data: " + ((null != file) ? file.toString() : "null"));
                                    fileAccess.saveDocument(file, inputStm, FileStorageFileAccess.DISTANT_FUTURE, new ArrayList<Field>());
                                } else {
                                    throw e;
                                }
                            }
                            fileAccess.commit();
                            rollback = false;

                            // set the file descriptor as a separate object
                            result.put(MessagePropertyKey.KEY_FILEDESCRIPTOR, FileDescriptor.createJSONObject(file, null));
                            // we need to provide if the file has been converted or not
                            result.put("converted", (conversionFormat != null));

                            requestResult = new AJAXRequestResult(result);
                            documentEvent = new CreateEvent(eventDocumentType);
                        } else {
                            // error case: we don't have a input stream
                            ErrorCode errorCode = null;
                            if (null != conversionFormat) {
                                errorCode = ErrorCode.CREATEDOCUMENT_CONVERSION_FAILED_ERROR;
                            } else {
                                errorCode = ErrorCode.CREATEDOCUMENT_CANNOT_READ_TEMPLATEFILE_ERROR;
                            }
                            JSONObject result = errorCode.getAsJSON();
                            requestResult = new AJAXRequestResult(result);
                            documentEvent = new DocumentEvent(eventDocumentType, ErrorType.CREATE);
                            LOG.warn(errorCode.getDescription());
                        }
                    } else {
                        // no path found for the provided id
                        ErrorCode errorCode = ErrorCode.LOCALFILE_INVALID_ID_ERROR;
                        LOG.error(errorCode.getDescription());
                        try {
                            requestResult = new AJAXRequestResult(createErrorObject(errorCode));
                            documentEvent = new DocumentEvent(eventDocumentType, ErrorType.CREATE);
                        } catch (final JSONException e) {
                            LOG.warn("Couldn't create JSON object while creating new document from template");
                        }
                    }
                } else {
                    // required arguments are null or not provided
                    ErrorCode errorCode = ErrorCode.GENERAL_ARGUMENTS_ERROR;
                    LOG.error(errorCode.getDescription());
                    try {
                        requestResult = new AJAXRequestResult(createErrorObject(errorCode));
                        documentEvent = new DocumentEvent(eventDocumentType, ErrorType.CREATE);
                    } catch (final JSONException e) {
                        LOG.warn("Couldn't create JSON object while creating new document from template");
                    }
                }
            } catch (final Exception e) {
                ErrorCode errorCode = ErrorCode.GENERAL_PERMISSION_CREATE_MISSING_ERROR;
                if (e instanceof OXException) {
                    errorCode = ExceptionToErrorCode.map((OXException)e, errorCode, false);
                }

                LOG.error(errorCode.getDescription(), e);
                try {
                    requestResult = new AJAXRequestResult(createErrorObject(errorCode));
                    documentEvent = new DocumentEvent(eventDocumentType, ErrorType.CREATE);
                } catch (final JSONException je) {
                    LOG.warn("Couldn't create JSON object while creating new document from template");
                }
            } finally {
                // Roll-back (if needed) and finish
                if (rollback) {
                    TransactionAwares.rollbackSafe(fileAccess);
                }

                TransactionAwares.finishSafe(fileAccess);
                IOUtils.closeQuietly(inputStm);
            }
        }

        // update statistics with a possibly created event
        Statistics.handleDocumentEvent(documentEvent);

        return requestResult;
    }

    /**
     * Create the necessary JSON error object in case of an error.
     *
     * @param errorCode
     *  A valid ErrorCode.
     *
     * @return
     *  A JSON object containing the error information to be sent to the
     *  client.
     *
     * @throws JSONException
     */
    private JSONObject createErrorObject(ErrorCode errorCode) throws JSONException {
        final JSONObject jsonObj = new JSONObject(3);
        jsonObj.put("errorCode", errorCode.getCode());
        jsonObj.put("error", errorCode.getCodeAsStringConstant());
        jsonObj.put("errorDescription", errorCode.getDescription());
        return jsonObj;
    }
}
