/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.calcengine.worker;

import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openxchange.office_communication.configuration.configitems.CalcEngineConfig;
import com.openxchange.office_communication.tools.exec.ProcessUtils;
import com.openxchange.office_communication.tools.logging.v2.LogConst;

//=============================================================================
public class MemoryWatchDog implements Runnable
{
	//-------------------------------------------------------------------------
	private static final Logger LOG = LoggerFactory.getLogger(MemoryWatchDog.class);

	//-------------------------------------------------------------------------
	public static final String LOG_CATEGORY = "::" + LogConst.WORKER + "::" + LogConst.WATCHDOG;

	//-------------------------------------------------------------------------
	public static final long POLL_TIME_IN_MS = 60000;
	
	//-------------------------------------------------------------------------
	public MemoryWatchDog ()
	    throws Exception
	{}

	//-------------------------------------------------------------------------
	public /*no synchronized*/ void start ()
	    throws Exception
	{
		Thread aRun = null;

		synchronized (this)
		{
			if (m_aRun != null)
				return;
			
			final CalcEngineConfig aCfg          = CalcEngineConfig.access();
			final Long             nMemLimitInMB = aCfg.getWorkerMaxMemInMB();

			if (nMemLimitInMB < 1)
			{
				LOG.info(LOG_CATEGORY+" no memory limit configured. Watch dog will not be started.");
				return;
			}
			
			m_aRun = new Thread (this);
			aRun   = m_aRun;
		}
		
		LOG.info(LOG_CATEGORY+" start ...");
		aRun.start ();
	}

	//-------------------------------------------------------------------------
	public /*no synchronized*/ void stop ()
	    throws Exception
	{
		Thread aRun = null;

		synchronized (this)
		{
			if (m_aRun == null)
				return;
			
			aRun   = m_aRun;
			m_aRun = null;
		}
		
		LOG.info(LOG_CATEGORY+" stop ...");
		aRun.interrupt();
	}

	//-------------------------------------------------------------------------
	@Override
	public void run ()
	{
		try
		{
			final CalcEngineConfig aCfg          = CalcEngineConfig.access();
			final Long             nMemLimitInMB = aCfg.getWorkerMaxMemInMB();
			final Long             nMemLimitInB  = aCfg.getWorkerMaxMemInB();
			final int              nPID          = ProcessUtils.getPidOfThis();
			final Object           aSleep        = new Object ();

			Validate.isTrue(nPID> 0, "Could not retrieve PID.");
			
			LOG.info(LOG_CATEGORY+" watch for mem limit '"+nMemLimitInMB+"' MB");
			while (true)
			{
				synchronized (aSleep)
				{
					aSleep.wait(POLL_TIME_IN_MS);
				}
				
				final Long nRSS = ProcessUtils.getProcessResidentSetSizeInByte(nPID);
				if (nRSS > nMemLimitInB)
				{
					// Log on both channels: LOG framework and system.err.
					// In some environments system.err is intercepted and also logged ...
					// where the LOG framework isnt used real ...
					
					final String sMsg = LOG_CATEGORY+" process violates configured memory limit of '"+nMemLimitInMB+"' MB and will be killed.";
					LOG.error          (sMsg);
					System.err.println (sMsg);
					
					// wait some time till log message is processed !
					
					synchronized(aSleep)
					{
						aSleep.wait(250);
					}
					System.exit(-1);
				}
			}
		}
		catch (final Throwable ex)
		{
			// interrupted
			// dont log it
		}
	}
	
	//-------------------------------------------------------------------------
	private Thread m_aRun = null;
}
