package com.openexchange.office.calcengine.client;

import java.io.File;
import java.io.InputStream;
import java.util.Properties;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;

import com.openexchange.config.ConfigurationService;
import com.openexchange.office.tools.IPUtils;
import com.openexchange.server.ServiceLookup;
import com.openxchange.office_communication.configuration.configitems.ELoadBalancerMode;
import com.openxchange.office_communication.configuration.configitems.ELoadBalancerType;


//=============================================================================
public class CalcEngineConfig
{
    //-------------------------------------------------------------------------
    public static final String PROP_PREFIX = "calcengine";
    
    //-------------------------------------------------------------------------
    public static final String PROP_PLACEHOLDER = "%%s%%";

    //-------------------------------------------------------------------------
    public static final int INVALID_SERVER_NUMBER = 0;
    
    //-------------------------------------------------------------------------
    public static final int INVALID_RESTARTTRIGGER_MAXDOC = -1;
    
    //-------------------------------------------------------------------------
    public static final int MAX_SERVER_COUNT = 25;

    //-------------------------------------------------------------------------
    public static final long   DEFAULT_MAX_MEM_MB = 2048;
    public static final String DEFAULT_REQUEST_TIMEOUT = "55000";
    public static final String DEFAULT_WORKER_INIT_TIMEOUT = "20000";
    
    //-------------------------------------------------------------------------
    public static final ECalcEngineMode DEFAULT_MODE = ECalcEngineMode.E_INLINE;
    
    //-------------------------------------------------------------------------
    /** define the mode how calc engine has to be used.
     *  possible modes are : 
     *  - inline
     *  - local
     *  
     *  @see {@link ECalcEngineMode.from}
     */
    public static final String PROP_MODE = PROP_PREFIX+".mode";

    //-------------------------------------------------------------------------
    /** enable/disable the simulator on top of any real mode
     *  set it to TRUE or FALSE.
     */
    public static final String PROP_SIMULATOR = PROP_PREFIX+".simulator";

    //-------------------------------------------------------------------------
    /** if set it enable recording of document data at runtime.
     */
    public static final String PROP_RECORDING_PATH = PROP_PREFIX+".recording-path";

    //-------------------------------------------------------------------------
    /** enable/disable the performance tests on top of any real mode
     *  set it to TRUE or FALSE.
     */
    public static final String PROP_PERFORMANCE_TESTS                = PROP_PREFIX+".performance-tests";
    public static final String PROP_PERFORMANCE_TESTS_JMS_STATISTICS = PROP_PREFIX+".performance-tests.jms-statistics";

    //-------------------------------------------------------------------------
    public static final String PROP_REQUEST_TIMEOUT = PROP_PREFIX+".request.timeout";
    
    //-------------------------------------------------------------------------
    public static final String PROP_RESTORE_PATH = PROP_PREFIX+".restore-documents.path";
    
    //-------------------------------------------------------------------------
	public static final String PROP_LOADBALANCER_MODE = PROP_PREFIX+".load-balancer.mode";
	public static final String PROP_LOADBALANCER_TYPE = PROP_PREFIX+".load-balancer.type";
    
    //-------------------------------------------------------------------------
	public static final String PROP_QUEUE_IN = PROP_PREFIX+".queue.in";

	//-------------------------------------------------------------------------
    public static final String PROP_WORKER_BIN                    = PROP_PREFIX+".worker.bin"           ;
    public static final String PROP_WORKER_PORT_RANGE_MIN         = PROP_PREFIX+".worker.port_range.min";
    public static final String PROP_WORKER_PORT_RANGE_MAX         = PROP_PREFIX+".worker.port_range.max";
    public static final String PROP_WORKER_MAXMEM                 = PROP_PREFIX+".worker.memory.max-mb" ;
    public static final String PROP_WORKER_INIT_TIMEOUT           = PROP_PREFIX+".worker.init.timeout"  ;

    public static final String PROP_WORKER_RESTART_RESERVE_COUNT  = PROP_PREFIX+".worker.restart.reserve-count";
    public static final String PROP_WORKER_RESTART_TRIGGER_MAXDOC = PROP_PREFIX+".worker.restart.trigger.max-doc-count";

    //-------------------------------------------------------------------------
    /**
	calcengine.server.1.host                 = ces-01.oxhh.int
	calcengine.server.1.port                 = 4711
    
	calcengine.server.2.host                 = ces-01.oxhh.int
	calcengine.server.2.port                 = 4712
     */
    public static final String PROP_SERVER_X_HOST = PROP_PREFIX+".server."+PROP_PLACEHOLDER+".host";
    public static final String PROP_SERVER_X_PORT = PROP_PREFIX+".server."+PROP_PLACEHOLDER+".port";
    
    //-------------------------------------------------------------------------
    /** Use factory method {@link get()} to create new instances.
     *  We want to force using our singleton .-)
     */
    private CalcEngineConfig ()
        throws Exception
    {}
    
    //-------------------------------------------------------------------------
    public synchronized static CalcEngineConfig get ()
        throws Exception
    {
        if (m_gSingleton == null)
            m_gSingleton = new CalcEngineConfig ();
        return m_gSingleton;
    }

    //-------------------------------------------------------------------------
    public ECalcEngineMode getMode ()
        throws Exception
    {
        final String          sMode = mem_CfgSrv ().getProperty(PROP_MODE);
        final ECalcEngineMode eMode = ECalcEngineMode.fromString(sMode, DEFAULT_MODE);
        return eMode;
    }

    //-------------------------------------------------------------------------
    public ECalcEngineMode getDetailMode ()
        throws Exception
    {
    	ECalcEngineMode eMode = getMode ();
    	
    	if (
    		(eMode == ECalcEngineMode.E_INLINE   ) ||
    		(eMode == ECalcEngineMode.E_LOCAL    ) ||
    		(eMode == ECalcEngineMode.E_SIMULATOR)
    	   )
    	{
    		return eMode;
    	}
    	
		final boolean bIsServer = isTHISHostServer ();

    	if (
    		(eMode == ECalcEngineMode.E_HTTP) &&
    		(bIsServer                      )
    	   )
    	{
    		return ECalcEngineMode.E_HTTP_SERVER;
    	}

    	if (
    		(eMode == ECalcEngineMode.E_HTTP) &&
    		( ! bIsServer                   )
    	   )
    	{
    		return ECalcEngineMode.E_HTTP_CLIENT;
    	}

    	if (
    		(eMode == ECalcEngineMode.E_JMS) &&
    		(bIsServer                     )
    	   )
    	{
    		return ECalcEngineMode.E_JMS_SERVER;
    	}

    	if (
    		(eMode == ECalcEngineMode.E_JMS) &&
    		( ! bIsServer                  )
    	   )
    	{
    		return ECalcEngineMode.E_JMS_CLIENT;
    	}
    	
    	return DEFAULT_MODE;
    }

    //-------------------------------------------------------------------------
    public boolean isSimulatorOn ()
        throws Exception
    {
        try
        {
            final String  sSimuMode = mem_CfgSrv ().getProperty(PROP_SIMULATOR);
        	final boolean bSimuMode = Boolean.parseBoolean(sSimuMode);
        	return bSimuMode;
        }
        catch (Throwable ex)
        {
        	// NO LOGGING - NO HANDLING !
        	// This value is a 'hidden feature' useful for development and QA only.
        	// Don't disturb official product ...
        }

        return false;
    }

    //-------------------------------------------------------------------------
    public boolean isPerformanceMeasurementOn ()
        throws Exception
    {
        try
        {
            final String  sMode = mem_CfgSrv ().getProperty(PROP_PERFORMANCE_TESTS);
        	final boolean bMode = Boolean.parseBoolean(sMode);
        	return bMode;
        }
        catch (Throwable ex)
        {
        	// NO LOGGING - NO HANDLING !
        	// This value is a 'hidden feature' useful for development and QA only.
        	// Don't disturb official product ...
        }

        return false;
    }

    //-------------------------------------------------------------------------
    public int getRequestTimeout ()
        throws Exception
    {
        final String sTimeout = mem_CfgSrv ().getProperty(PROP_REQUEST_TIMEOUT, DEFAULT_REQUEST_TIMEOUT);
        final int    nTimeout = Integer.parseInt(sTimeout);
        Validate.isTrue(nTimeout>=0, "Invalid configuration value : "+PROP_REQUEST_TIMEOUT+" needs to >= 0.");
        return nTimeout;
    }

    //-------------------------------------------------------------------------
    public int getWorkerInitTimeout ()
        throws Exception
    {
        final String sTimeout = mem_CfgSrv ().getProperty(PROP_WORKER_INIT_TIMEOUT, DEFAULT_WORKER_INIT_TIMEOUT);
        final int    nTimeout = Integer.parseInt(sTimeout);
        Validate.isTrue(nTimeout>=0, "Invalid configuration value : "+PROP_WORKER_INIT_TIMEOUT+" needs to >= 0.");
        return nTimeout;
    }

    //-------------------------------------------------------------------------
    public String getWorkerBin ()
        throws Exception
    {
        String sJar = mem_CfgSrv ().getProperty(PROP_WORKER_BIN);
		if (StringUtils.isEmpty(sJar))
			sJar = impl_guessWorkerJar ();
        Validate.notEmpty(sJar, "Invalid configuration value : "+PROP_WORKER_BIN+" is missing or empty.");
        return sJar;
    }

    //-------------------------------------------------------------------------
    public long getWorkerMaxMemInKB ()
        throws Exception
    {
        final String sMemInMB = mem_CfgSrv ().getProperty(PROP_WORKER_MAXMEM);
              long   nMemInMB = 0;
        try
        {
        	nMemInMB = Long.parseLong(sMemInMB);
        }
        catch (final NumberFormatException ex)
        {
        	nMemInMB = DEFAULT_MAX_MEM_MB;
        }
        
        final long nMemInKB = nMemInMB * 1024;
        return nMemInKB;
    }

    //-------------------------------------------------------------------------
    public int getWorkerRestartReserveCount ()
        throws Exception
    {
        final String sReserveCount = mem_CfgSrv ().getProperty(PROP_WORKER_RESTART_RESERVE_COUNT);
              int    nReserveCount = 0;

		try
		{
			nReserveCount = Integer.parseInt(sReserveCount);
		}
		catch (final NumberFormatException ex)
		{
			nReserveCount = 0;
		}
		  
		return nReserveCount;
    }

    //-------------------------------------------------------------------------
    /** return INVALID_RESTARTTRIGGER_MAXDOC means : feature is disabled !
     */
    public int getWorkerRestartTriggerMaxDocCount ()
        throws Exception
    {
        final String sMaxDocs = mem_CfgSrv ().getProperty(PROP_WORKER_RESTART_TRIGGER_MAXDOC);
              int    nMaxDocs = INVALID_RESTARTTRIGGER_MAXDOC;

		try
		{
			nMaxDocs = Integer.parseInt(sMaxDocs);
		}
		catch (final NumberFormatException ex)
		{
			nMaxDocs = INVALID_RESTARTTRIGGER_MAXDOC;
		}
		  
		return nMaxDocs;
    }

    //-------------------------------------------------------------------------
    public int getServerCount ()
        throws Exception
    {
    	final ConfigurationService aCfg   = mem_CfgSrv ();
    	      int                  nNr    = 1;
    	      int                  nCount = 0;
    	while (nNr <= MAX_SERVER_COUNT)
    	{
    		final String sKey  = StringUtils.replace(PROP_SERVER_X_HOST, PROP_PLACEHOLDER, Integer.toString(nNr));
        	final String sHost = aCfg.getProperty(sKey);	
        	
        	if (StringUtils.isEmpty(sHost))
        		break;

        	nNr   ++;
    		nCount++;
    	}
    	
    	return nCount;
    }

    //-------------------------------------------------------------------------
    public boolean isTHISHostServer ()
        throws Exception
    {
    	final int     nServerNr = getServerNumberForTHISHost ();
    	final boolean bIsServer = nServerNr != INVALID_SERVER_NUMBER;
    	return bIsServer;
    }
    
    //-------------------------------------------------------------------------
    public int getServerNumberForTHISHost ()
        throws Exception
    {
    	final int nServerCount = getServerCount();

    	// a) no server configured at all
    	
    	if (nServerCount < 1)
    		return INVALID_SERVER_NUMBER;
    	
    	// b) look if any configured server match to any IP of THIS machine
    	
    	int nNr = 1;
    	while (nNr<=nServerCount)
    	{
    		final String sHost = getServerHost    (nNr);
    		final String sIP   = IPUtils.detectIP (sHost);
    		
    		if (IPUtils.isMyIP(sIP))
    			return nNr;
    		
    		nNr++;
    	}
    	
    	return INVALID_SERVER_NUMBER;
    }

    //-------------------------------------------------------------------------
    public String getServerHost (final int nNr)
        throws Exception
    {
    	final String sKey  = StringUtils.replace(PROP_SERVER_X_HOST, PROP_PLACEHOLDER, Integer.toString(nNr));
    	final String sHost = mem_CfgSrv ().getProperty(sKey);
        Validate.notEmpty(sHost, "Invalid configuration value : "+sKey+" is missing or empty.");
        return sHost;
    }

    //-------------------------------------------------------------------------
    public int getServerPort (final int nNr)
        throws Exception
    {
    	final String sKey  = StringUtils.replace(PROP_SERVER_X_PORT, PROP_PLACEHOLDER, Integer.toString(nNr));
    	final String sPort = mem_CfgSrv ().getProperty(sKey);
        Validate.notEmpty(sPort, "Invalid configuration value : "+sKey+" is missing or empty.");
        final int    nPort = Integer.parseInt(sPort);
        Validate.isTrue(nPort>0, "Invalid configuration value : "+sKey+" needs to > 0.");
        return nPort;
    }

    //-------------------------------------------------------------------------
    public int getWorkerPortRangeMin ()
        throws Exception
    {
    	final String sPort = mem_CfgSrv ().getProperty(PROP_WORKER_PORT_RANGE_MIN);
        Validate.notEmpty(sPort, "Invalid configuration value : "+PROP_WORKER_PORT_RANGE_MIN+" is missing or empty.");
        final int    nPort = Integer.parseInt(sPort);
        Validate.isTrue(nPort>0, "Invalid configuration value : "+PROP_WORKER_PORT_RANGE_MIN+" needs to > 0.");
        return nPort;
    }

    //-------------------------------------------------------------------------
    public int getWorkerPortRangeMax ()
        throws Exception
    {
    	final String sPort = mem_CfgSrv ().getProperty(PROP_WORKER_PORT_RANGE_MAX);
        Validate.notEmpty(sPort, "Invalid configuration value : "+PROP_WORKER_PORT_RANGE_MAX+" is missing or empty.");
        final int    nPort = Integer.parseInt(sPort);
        Validate.isTrue(nPort>0, "Invalid configuration value : "+PROP_WORKER_PORT_RANGE_MAX+" needs to > 0.");
        return nPort;
    }

    //-------------------------------------------------------------------------
    public boolean isRecordingOn ()
        throws Exception
    {
        final String  sRecordingPath = getRecordingPath ();
        final boolean bIsOn          = ! StringUtils.isEmpty(sRecordingPath);
        return bIsOn;
    }

    //-------------------------------------------------------------------------
    public String getRecordingPath ()
        throws Exception
    {
        final String sPath = mem_CfgSrv ().getProperty(PROP_RECORDING_PATH);
        return sPath;
    }
    //-------------------------------------------------------------------------
    public boolean isRestoreDocumentsEnabled ()
        throws Exception
    {
        final String  sPath      = getRestoreDocumentsPath ();
        final boolean bIsEnabled = ! StringUtils.isEmpty(sPath);
    	return bIsEnabled;
    }

    //-------------------------------------------------------------------------
    public String getRestoreDocumentsPath ()
        throws Exception
    {
    	final String VAR_TEMP = "${sys:java.io.tmpdir}";
        String sPath = mem_CfgSrv ().getProperty(PROP_RESTORE_PATH);
        
        if (StringUtils.contains(sPath, VAR_TEMP))
        {
        	final String sTemp = FileUtils.getTempDirectoryPath();
        	sPath = StringUtils.replace(sPath, VAR_TEMP, sTemp);
        }
        
    	return sPath;
    }

    //-------------------------------------------------------------------------
    public String getInQueue ()
        throws Exception
    {
    	final String sQueue = mem_CfgSrv().getProperty(PROP_QUEUE_IN, "calcengine_in");
    	return sQueue;
    }
    
    //-------------------------------------------------------------------------
    public ELoadBalancerMode getLoadBalancerMode ()
        throws Exception
    {
    	final String            sMode = mem_CfgSrv().getProperty(PROP_LOADBALANCER_MODE, ELoadBalancerMode.STR_ROUND_ROBIN);
    	final ELoadBalancerMode eMode = ELoadBalancerMode.fromString(sMode);
    	return eMode;
    }
    
    //-------------------------------------------------------------------------
    public ELoadBalancerType getLoadBalancerType ()
        throws Exception
    {
    	final String            sType = mem_CfgSrv().getProperty(PROP_LOADBALANCER_TYPE, ELoadBalancerType.STR_SERVER_DYNAMIC_ROUTER);
    	final ELoadBalancerType eType = ELoadBalancerType.fromString(sType);
    	return eType;
    }

    //-------------------------------------------------------------------------
    public boolean isJmsStatisticsEnabled ()
        throws Exception
    {
    	final String  sEnabled = mem_CfgSrv().getProperty(PROP_PERFORMANCE_TESTS_JMS_STATISTICS, "false");
    	final boolean bEnabled = Boolean.parseBoolean(sEnabled);
    	return bEnabled;
    }

    //-------------------------------------------------------------------------
    private static String impl_guessWorkerJar ()
    	throws Exception
    {
    	File aCheck = null;

    	// a) check if we are in production environment
    	//    where path is well known and fix

    	aCheck = new File ("/opt/open-xchange/sbin/calcengineworker");
    	if (aCheck.isFile())
    		return aCheck.getAbsolutePath();

    	// b) check if we are in special eclipse environment.
    	//    Our own build-in-eclipse.xml of worker project
    	//    help us a little bit. It generate a configuration
    	//    where some interesting values are inside ...
    	//    e.g. the path and name of the JAR .-)
    	
    	final String sTempDir          = System.getProperty("java.io.tmpdir");
    	final File   aTempDir          = new File (sTempDir);
    	final File   aEclipseWorkerEnv = new File (aTempDir, "com_openexchange_office_calcengine_worker.properties");
    	
    	if ( ! aEclipseWorkerEnv.isFile())
    		return null;

    	final InputStream aStream = FileUtils.openInputStream(aEclipseWorkerEnv);
    	try
    	{
    		final Properties aPropFile = new Properties ();
    		aPropFile.load(aStream);

    		final String sJar   = aPropFile.getProperty("worker.jar");
    		             aCheck = new File (sJar);
        	if (aCheck.isFile())
        		return aCheck.getAbsolutePath();
    	}
    	finally
    	{
    		IOUtils.closeQuietly(aStream);
    	}
    	
    	return null;
    }

    //-------------------------------------------------------------------------
    private ConfigurationService mem_CfgSrv ()
        throws Exception
    {
        if (m_aCfgSrv == null)
        {
        	final CalcEngineContext aCEContext = CalcEngineContext.get();
        	final ServiceLookup     aSmgr      = aCEContext.accessServiceLookup();
                                    m_aCfgSrv  = aSmgr.getService(ConfigurationService.class);
        }
        return m_aCfgSrv;
    }

    //-------------------------------------------------------------------------
    private static CalcEngineConfig m_gSingleton = null;
    
    //-------------------------------------------------------------------------
    private ConfigurationService m_aCfgSrv = null;
}
