/*
*
*    OPEN-XCHANGE legal information
*
*    All intellectual property rights in the Software are protected by
*    international copyright laws.
*
*
*    In some countries OX, OX Open-Xchange, open xchange and OXtender
*    as well as the corresponding Logos OX Open-Xchange and OX are registered
*    trademarks.
*    The use of the Logos is not covered by the GNU General Public License.
*    Instead, you are allowed to use these Logos according to the terms and
*    conditions of the Creative Commons License, Version 2.5, Attribution,
*    Non-commercial, ShareAlike, and the interpretation of the term
*    Non-commercial applicable to the aforementioned license is published
*    on the web site http://www.open-xchange.com/EN/legal/index.html.
*
*    Please make sure that third-party modules and libraries are used
*    according to their respective licenses.
*
*    Any modifications to this package must retain all copyright notices
*    of the original copyright holder(s) for the original code used.
*
*    After any such modifications, the original and derivative code shall remain
*    under the copyright of the copyright holder(s) and/or original author(s)per
*    the Attribution and Assignment Agreement that can be located at
*    http://www.open-xchange.com/EN/developer/. The contributing author shall be
*    given Attribution for the derivative code and a license granting use.
*
*     Copyright (C) 2016 OX Software GmbH
*     Mail: info@open-xchange.com
*
*
*     This program is free software; you can redistribute it and/or modify it
*     under the terms of the GNU General Public License, Version 2 as published
*     by the Free Software Foundation.
*
*     This program is distributed in the hope that it will be useful, but
*     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
*     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
*     for more details.
*
*     You should have received a copy of the GNU General Public License along
*     with this program; if not, write to the Free Software Foundation, Inc., 59
*     Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
*/
package com.openexchange.office.backup.manager.impl;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintStream;
import java.nio.charset.Charset;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.json.JSONTokener;

import com.openexchange.exception.OXException;
import com.openexchange.log.Log;
import com.openexchange.office.backup.distributed.DistributedDocumentResourceManager;
import com.openexchange.office.backup.distributed.DistributedDocumentStreamManager;
import com.openexchange.office.backup.manager.DocumentBackupController;
import com.openexchange.office.backup.manager.DocumentRestoreData;
import com.openexchange.office.hazelcast.access.HazelcastAccess;
import com.openexchange.office.hazelcast.doc.DocumentDirectory;
import com.openexchange.office.hazelcast.doc.DocumentResourcesDirectory;
import com.openexchange.office.tools.directory.DocRestoreID;
import com.openexchange.office.tools.directory.DocumentState;
import com.openexchange.office.tools.directory.IStreamIDCollection;
import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.office.tools.json.JSONHelper;
import com.openexchange.office.tools.message.MessageChunk;
import com.openexchange.office.tools.message.MessagePropertyKey;
import com.openexchange.office.tools.monitoring.RestoreDocEvent;
import com.openexchange.office.tools.monitoring.RestoreDocEventType;
import com.openexchange.office.tools.monitoring.Statistics;
import com.openexchange.office.tools.resource.Resource;
import com.openexchange.office.tools.resource.ResourceManager;
import com.openexchange.server.ServiceLookup;

/**
 * Central class to control storing and retrieving document restore data
 * to enable a user to request a restore of his/her private document content.
 *
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 * @since 7.8.1
 *
 */
public class DocumentBackupManager implements DocumentBackupController, StoreOperationData {

	@SuppressWarnings("deprecation")
    static private final org.apache.commons.logging.Log LOG = Log.loggerFor(DocumentBackupManager.class);

	private static final int OSN_UNDEFINED = -1;
    private static final AtomicReference<DocumentBackupManager> REF = new AtomicReference<DocumentBackupManager>();
	private ResourceManager resourceManager;
	private DocumentDirectory documentDir;
	private final AsyncBackupWorker backupWriter;
	private final Object sync = new Object();
	private final DistributedDocumentStreamManager distributedDocumentStreamManager;
	private final DistributedDocumentResourceManager distributedDocumentResourceManager;
	private final CleanupBackupDocTask cleanupTask;
	private final ServiceLookup services;
	private final String uniqueInstanceID;

	/**
	 * Creates a DocumentBackupManager instance.
	 *
	 * @param serviceLookup
	 */
	private DocumentBackupManager(final ServiceLookup serviceLookup, final DocumentDirectory docDir, final DocumentResourcesDirectory docResDir, final String uniqueInstanceID) {
	    distributedDocumentStreamManager = new DistributedDocumentStreamManager(serviceLookup, uniqueInstanceID);
	    distributedDocumentResourceManager = new DistributedDocumentResourceManager(serviceLookup, docResDir, uniqueInstanceID);
	    cleanupTask = new CleanupBackupDocTask(serviceLookup, this, docDir);
		resourceManager = new ResourceManager(serviceLookup);
		documentDir = docDir;
		backupWriter = new AsyncBackupWorker(this);
		services = serviceLookup;
		this.uniqueInstanceID = uniqueInstanceID;
    }

	/**
	 * Initializes the singleton.
	 *
	 * @param serviceLookup
	 */
	public static void init(final ServiceLookup serviceLookup, final DocumentDirectory docDir, final DocumentResourcesDirectory docResDir) {
	    String uuid = null;
	    try {
	        uuid = HazelcastAccess.getLocalMember().getUuid();
	    } catch (OXException e) {
	        LOG.error("RT connection: Please check package installation. Retrieving the Hazelcast local member not possible!",  e);
	    }

		REF.compareAndSet(null, new DocumentBackupManager(serviceLookup, docDir, docResDir, uuid));
	}

	/**
	 * Disposes the singleton, which resets memory consuming members.
	 */
	public static void destroy() {
		final DocumentBackupManager docDataManager = REF.getAndSet(null);

		if (null != docDataManager) {
			docDataManager.dispose();
		}
	}

	/**
	 * Returns the one and only DocumentDataManager instance.
	 *
	 * @return the one and only DocumentDataManager instance or null, if not
	 * initialized.
	 */
	public static final DocumentBackupManager get() {
		return REF.get();
	}

	/**
	 * Registers a new document state in the global document state directory
	 * managed by Hazelcast.
	 *
	 * @param docResId a unique document resource id which references the document
	 * @param osn the current operation state number
	 * @param version the current version of the document, can be null
	 * @param docContent the document content
	 * @return
	 */
	@Override
	public boolean registerDocument(final DocRestoreID docRestoreId, final String fileName, final String mimeType, final int osn, final String version, final byte[] docContent) {
		Validate.notNull(docRestoreId);
		Validate.notNull(docContent);

		try {
			DocumentState storedDocState;
			synchronized(sync) {
                storedDocState = documentDir.get(docRestoreId);
			}

			if (null == storedDocState) {
				storedDocState = new DocumentState(docRestoreId, uniqueInstanceID, -1, version);
				storedDocState.setFileName(fileName);
				storedDocState.setMimeType(mimeType);

				// store the document content using the distributed document stream manager
				final String managedDocFileId = distributedDocumentStreamManager.addDocumentStreams(docRestoreId, docContent);

                // store the backup managed file in the global hazelcast structures
                storedDocState.setManagedDocFileId(managedDocFileId);

                Statistics.handleRestoreDocEvent(new RestoreDocEvent(RestoreDocEventType.NEW_DOC));
			} else {
			    // overwrite a possible document stream with the latest
				String managedDocFileId = distributedDocumentStreamManager.addDocumentStreams(docRestoreId, docContent);
                // make sure that no resources are held by the old document entry, too.
				distributedDocumentResourceManager.removeDocResources(docRestoreId);

				// overwrite base/current osn and version
				storedDocState.setBaseOSN(osn);
				storedDocState.setBaseVersion(version);
				storedDocState.setCurrentOSN(osn);
				storedDocState.setCurrentVersion(version);

				// reset operations managed file
				storedDocState.setManagedOpsFileId(null);

                // store the backup managed file in the global hazelcast structures
                storedDocState.setManagedDocFileId(managedDocFileId);
                storedDocState.setFileName(fileName);
                storedDocState.setMimeType(mimeType);
			}

			// register this node instance as responsible and set document as active with current time stamp
			storedDocState.setActive(true);
			storedDocState.setTimeStamp(new Date().getTime());
			storedDocState.setUniqueInstanceId(this.uniqueInstanceID);

			synchronized(sync) {
				if (null != documentDir) {
                    documentDir.set(docRestoreId, storedDocState);
				}
			}

			return true;
		} catch (OXException e) {
			LOG.warn("RT connection: Exception while creating global backup structure for document", e);
		}

		return false;
	}

    @Override
    public boolean updateDocumentInitState(final DocRestoreID docRestoreId, final int osn) {
        try {
            DocumentState storedDocState;
            synchronized(sync) {
                storedDocState = documentDir.get(docRestoreId);
            }

            if (null == storedDocState) {
                // shouldn't happen - we want to set the initial osn if not set
                return false;
            } else if (storedDocState.getBaseOSN() == OSN_UNDEFINED) {
                storedDocState.setBaseOSN(osn);

                synchronized(sync) {
                    if (null != documentDir) {
                        documentDir.set(docRestoreId, storedDocState);
                    }
                }
            }

            return true;
        } catch (OXException e) {
            LOG.warn("RT connection: Exception while creating global backup structure for document", e);
        }

        return false;
    }

    @Override
    public boolean deregisterDocument(final DocRestoreID docRestoreId) {
        try {
            DocumentState storedDocState;
            synchronized(sync) {
                storedDocState = documentDir.get(docRestoreId);
            }

            if (null == storedDocState) {
                // shouldn't happen - we want to deregister a document
                return true;
            } else {
                // deregister means that we want to set the hazelcast data
                // to non-active, which starts the automatic cleanup
                // timeout process
                storedDocState.setActive(false);
                storedDocState.setTimeStamp(new Date().getTime());
                storedDocState.setUniqueInstanceId(this.uniqueInstanceID);

                synchronized(sync) {
                    if (null != documentDir) {
                        documentDir.set(docRestoreId, storedDocState);
                    }
                }
            }

            return true;
        } catch (OXException e) {
            LOG.warn("RT connection: Exception while creating global backup structure for document", e);
        }

        return false;
    }

	/**
	 * @param docResId
	 * @return
	 */
    @Override
	public void removeDocument(final DocRestoreID docRestoreId) {
        Validate.notNull(docRestoreId);

        synchronized(sync) {

			try {
                documentDir.remove(docRestoreId);
                Statistics.handleRestoreDocEvent(new RestoreDocEvent(RestoreDocEventType.REMOVED_DOC));
    		} catch (OXException e) {
                LOG.warn("RT connection: Exception caught trying to remove timed out hazeclast document structure, docResId=" + docRestoreId, e);
    		}
		}

        // remove distributed managed files for document & resources
		distributedDocumentStreamManager.removeDocumentStreams(docRestoreId);
		distributedDocumentResourceManager.removeDocResources(docRestoreId);
	}

	/**
	 *
	 * @param docResId
	 * @param actionChunk
	 * @param newOSN
	 * @return
	 * @throws Exception
	 */
	@Override
	public boolean addOperations(final DocRestoreID docRestoreId, final MessageChunk actionChunk, int newOSN) throws Exception {
		if ((null != actionChunk) && actionChunk.getOperations().hasAndNotNull(MessagePropertyKey.KEY_ACTIONS)) {
            backupWriter.appendOperations(docRestoreId, actionChunk, newOSN);
            return true;
		}

		return false;
	}

	/**
	 * @param docResId
	 * @param resource
	 * @return
	 */
	@Override
    public boolean addResource(final DocRestoreID docRestoreId, final Resource resource) {
        Validate.notNull(docRestoreId);

        distributedDocumentResourceManager.addResourceRef(docRestoreId, resource);

        return true;
    }

	/**
	 *
	 */
    @Override
    public DocumentRestoreData getDocumentRestoreData(DocRestoreID docRestoreId) {
        Validate.notNull(docRestoreId);

        final DocumentRestoreData docResourceData = new DocumentRestoreData();

        try {
            DocumentState storedDocState;
            synchronized(sync) {
                storedDocState = documentDir.get(docRestoreId);
            }

            if (null != storedDocState) {
                final String managedDocFileId = storedDocState.getManagedDocFileId();
                final String managedOpsFileId = storedDocState.getManagedOpsFileId();
                final Set<String> docResources = distributedDocumentResourceManager.getDocResources(docRestoreId);

                // set the base version/osn to have synchronization data
                docResourceData.setBaseOSN(storedDocState.getBaseOSN());
                docResourceData.setBaseVersion(storedDocState.getBaseVersion());
                docResourceData.setOSN(storedDocState.getCurrentOSN());
                docResourceData.setFileName(storedDocState.getFileName());
                docResourceData.setMimeType(storedDocState.getMimeType());

                if (StringUtils.isNotEmpty(managedDocFileId)) {
                    final InputStream docInputStream = distributedDocumentStreamManager.getStream(managedDocFileId);

                    // retrieve input stream from managed document resource
                    if (null != docInputStream) {
                        docResourceData.setInputStream(docInputStream);
                    } else {
                        LOG.warn("RT connection: couldn't retrieve backup document stream from managed file: " + managedDocFileId);
                        docResourceData.setErrorCode(ErrorCode.BACKUPDOCUMENT_BASEDOCSTREAM_NOT_FOUND);
                    }

                    // we always need a resource manager, but it can contain no resources
                    final ResourceManager resManager = new ResourceManager(services);
                    if (null != docResources) {
                        // add referenced resources to the new resource manager
                        if (docResources.size() > 0) {
                            for (final String managedResId : docResources) {
                                final Resource managedResource = resourceManager.createManagedResource(managedResId);
                                resManager.addResource(managedResource);
                            }
                        }
                    }
                    docResourceData.setResourceManager(resManager);

                    // retrieve operations from the managed resource
                    if (StringUtils.isNotEmpty(managedOpsFileId)) {

                        InputStream opsInputStream = null;
                        try {
                            opsInputStream = distributedDocumentStreamManager.getStream(managedOpsFileId);

                            // retrieve input stream from managed document resource
                            if (null != opsInputStream) {
                                final JSONArray operationsArray = new JSONArray();
                                final BufferedReader in = new BufferedReader(new InputStreamReader(opsInputStream, "UTF-8"));

                                // create a JSONArray containing all the operations
                                String line;
                                while((line = in.readLine()) != null) {
                                    try {
                                        final JSONTokener tokener = new JSONTokener(line);
                                        final JSONArray array = new JSONArray(tokener);
                                        int i = 0;
                                        while (i < array.length()) {
                                            final JSONObject opsObject = array.getJSONObject(i);
                                            if (null != opsObject) {
                                                final JSONArray operations = opsObject.getJSONArray("operations");
                                                if (null != operations) {
                                                    JSONHelper.appendArray(operationsArray, operations);
                                                }
                                            }
                                            i++;
                                        }
                                    } catch (JSONException e) {
                                        LOG.warn("RT connection: Exception caught while trying to parse JSON operation from backup data", e);
                                    }
                                }

                                docResourceData.setOperations(operationsArray);
                            } else {
                                LOG.warn("RT connection: couldn't retrieve backup document stream from managed file: " + managedDocFileId);
                                docResourceData.setErrorCode(ErrorCode.BACKUPDOCUMENT_BASEDOCSTREAM_NOT_FOUND);
                            }
                        } catch (Exception e) {
                            LOG.warn("RT connection: Exception caught while trying to read operations JSON from backup data", e);
                        } finally {
                            IOUtils.closeQuietly(opsInputStream);
                        }
                    }
                } else {
                    LOG.warn("RT connection: couldn't find backup managed file id for document: " + docRestoreId);
                }
            } else {
                LOG.warn("RT connection: couldn't find global document state for document: " + docRestoreId);
            }

        } catch (OXException e) {
            LOG.warn("RT connection: Exception while retrieving global backup structure for document", e);
        }

        return docResourceData;
    }

	/**
	 * @param docResId
	 * @param operations
	 * @return
	 */
	@Override
	public boolean storePendingOperationsToDocumentState(final DocRestoreID docRestoreId, final MessageChunk operations, int newOSN) {
		boolean result = false;
        JSONArray allActions = null;

        try {
        	allActions = operations.getOperations().getJSONArray(MessagePropertyKey.KEY_ACTIONS);
        } catch (JSONException e) {
        	return false;
        }

		DocumentState storedDocState = null;
        synchronized(sync) {
        	try {
                storedDocState = documentDir.get(docRestoreId);
        	} catch (OXException e) {
                LOG.warn("RT connection: Exception caught while trying to retrieve document data from Hazelcast", e);
        	}
		}

		if (null != storedDocState) {
			storedDocState.setCurrentOSN(newOSN);

			String managedOpsFileId = storedDocState.getManagedOpsFileId();

			if (null != managedOpsFileId) {
				// append new operations to the managed operations file
			    final InputStream opsStream = distributedDocumentStreamManager.getStream(managedOpsFileId);

                if (null != opsStream) {
                    ByteArrayOutputStream outStream = new ByteArrayOutputStream();
                    final PrintStream writer = new PrintStream(outStream);

                    try {
                        // copy the content of the original operation stream
                        IOUtils.copy(opsStream, outStream);

                        // append new operations using newline as separator
	                    writer.println();
	                    outStream.write(allActions.toString().getBytes(Charset.forName("UTF-8")));
                        IOUtils.closeQuietly(outStream);

	                    // overwrite old managed resoure file
                        final byte[] newOpsFile = outStream.toByteArray();
                        managedOpsFileId = distributedDocumentStreamManager.addOperationStream(docRestoreId, newOpsFile);

	                    // store the backup managed file via distributed managed file management
	                    storedDocState.setManagedOpsFileId(managedOpsFileId);
                    } catch (Exception e) {
                        LOG.warn("RT connection: Couldn't update document operation stream", e);
                        return false;
                    } finally {
                        IOUtils.closeQuietly(outStream);
                    }
                } else {
                    LOG.warn("RT connection: couldn't retrieve backup operations stream from managed file: " + managedOpsFileId);
                }
			} else {
			    // create a new operations file
                ByteArrayOutputStream outStream = new ByteArrayOutputStream();
                try {
                    outStream.write(allActions.toString().getBytes(Charset.forName("UTF-8")));
                    IOUtils.closeQuietly(outStream);

                    final byte[] newOpsFile = outStream.toByteArray();
                    managedOpsFileId = distributedDocumentStreamManager.addOperationStream(docRestoreId, newOpsFile);

                    // store the backup managed file via distributed managed file management
                    storedDocState.setManagedOpsFileId(managedOpsFileId);
                } catch (Exception e) {
                    LOG.warn("RT connection: Couldn't create document operation stream", e);
                    return false;
                } finally {
                    IOUtils.closeQuietly(outStream);
                }
			}
		} else
			return false; // no backup data set - bail out with false

		synchronized(sync) {
			if (null != documentDir) {
				try {
                    documentDir.set(docRestoreId, storedDocState);
                    result = true;
				} catch (OXException e) {
	                LOG.warn("RT connection: Exception caught while trying to set updated document data to Hazelcast", e);
				}
			}
		}

        return result;
	}

	@Override
    public Set<DocRestoreID> getDocuments() {
	    final Set<IStreamIDCollection> docCollection = distributedDocumentStreamManager.getStreamIDCollectionCollection();
	    final Set<DocRestoreID> docResCollection = new HashSet<DocRestoreID>();

	    if (null != docCollection) {
	        for (IStreamIDCollection collection : docCollection) {
	            final DocRestoreID id = collection.getUniqueCollectionID();

                docResCollection.add(id);
	        }
	    }

	    return docResCollection;
	}

	@Override
    public String getUniqueInstanceID() {
        return this.uniqueInstanceID;
    }

	/**
	 * Removes references to memory consuming objects.
	 */
	private void dispose() {
		synchronized(sync) {
			try {
				resourceManager = null;
                cleanupTask.dispose();
				documentDir = null;
				distributedDocumentResourceManager.dispose();
				distributedDocumentStreamManager.dispose();
            } catch (Throwable e) {
				// do nothing - we just remove references and call methods
                LOG.warn("RT connection: Exception caught while reset members for dispose", e);
			}
		}
	}

}
