/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

(function () {

    'use strict';

    // helper function that does nothing but returns the calling context
    function noop() { return this; }

    // class CoreApplication ==================================================

    /**
     * Replacement for a core application for unit tests. Mimics the public API
     * of an application object as returned by the method ox.ui.createApp().
     *
     * @constructor
     */
    function CoreApplication(options) {

        var appWindow = null,
            launcher = null,
            self = this;

        this.on = this.one = this.off = this.listenTo = noop;
        this.registerGlobalEventHandler = this.registerWindowResizeHandler = noop;

        this.getName = _.constant(options.name);
        this.setTitle = _.noop;

        this.getWindow = function () { return appWindow; };
        this.getWindowNode = function () { return appWindow.nodes.main; };
        this.setWindow = function (appWin) { appWindow = appWin; };

        this.setLauncher = function (callback) { launcher = callback; };
        this.launch = function () { return $.when(launcher()); };

        this.getInstance = function () { return self; };

        this.setQuit = _.noop;
        this.quit = _.noop;

    } // class CoreApplication

    // static methods ---------------------------------------------------------

    /**
     * Returns a new instance of the class CoreApplication.
     */
    CoreApplication.createApp = function (options) {
        return new CoreApplication(options);
    };

    /**
     * Returns a replacement for a core application window for unit tests.
     * Mimics the public API of an application window object as returned by the
     * method ox.ui.createWindow().
     */
    CoreApplication.createWindow = function () {
        var appWin = $({});
        appWin.nodes = { main: $('<div>'), outer: $('<div>') };
        appWin.state = { visible: true };
        appWin.setTitle = _.constant(appWin);
        appWin.busy = _.constant(appWin);
        appWin.idle = _.constant(appWin);
        return appWin;
    };

    /**
     * Launches an application instance of the specified module type (one of
     * the Documents editor application types).
     *
     * @returns {CoreApplication}
     *  An application instance. The passed Application class constructor has
     *  been mixed into this instance.
     */
    CoreApplication.launchApp = function (module, Application) {
        var app = new CoreApplication(module);
        Application.call(app, { action: 'new' });
        app.launch();
        return app;
    };

    // class TestView =========================================================

    /**
     * A simple test view class for unit tests. Mimics the common public API of
     * a Documents view instance.
     *
     * @constructor
     */
    function TestView(docModel) {

        var rootNode = $('<div>');

        this.on = this.one = this.off = this.listenTo = noop;
        this.getDocModel = _.constant(docModel);
        this.getContentRootNode = _.constant(rootNode);
        this.getHiddenRootNode = _.constant(rootNode);
        this.isVisible = _.constant(true);
        this.createClipboardNode = function () { return $('<div class="clipboard">'); };
        this.grabFocus = noop;
        this.setVisibleDrawingAnchor = noop;
        this.getZoomFactor = _.constant(1);

    } // class TestView

    // class TestApplication ==================================================

    /**
     * A simple test application class for unit tests. Mimics the common public
     * API of a Documents application instance.
     *
     * @constructor
     */
    function TestApplication(appBaseName, fileFormat, ModelClass, initOptions) {

        var appWin = null,
            docModel = null,
            docView = null,
            initDef = $.Deferred(),
            importDef = $.Deferred();

        CoreApplication.call(this, { name: 'io.ox/office/' + appBaseName });

        this.getModel = function () { return docModel; };
        this.getView = function () { return docView; };

        this.registerDestructor = noop;
        this.onInit = function (callback, context) { initDef.done(callback.bind(context)); };
        this.isImportStarted = _.constant(true);
        this.getImportStartPromise = _.constant(initDef.promise());
        this.isImportFinished = _.constant(true);
        this.getImportFinishPromise = _.constant(importDef.promise());
        this.getFileFormat = _.constant(fileFormat);
        this.isOOXML = _.constant(fileFormat === 'ooxml');
        this.isODF = _.constant(fileFormat === 'odf');
        this.getUserSettingsValue = _.constant(null);
        this.destroyImageNodes = noop;
        this.addAuthors = noop;
        this.isInternalError = _.noop;

        this.setLauncher(function () {
            appWin = CoreApplication.createWindow();
            this.setWindow(appWin);
            return require(['io.ox/office/tk/object/timermixin']).then(function (TimerMixin) {
                TimerMixin.call(this);
                docModel = new ModelClass(this);
                docView = new TestView(docModel);
                initDef.resolve();
                var initHandler = initOptions && initOptions.initHandler;
                return $.when(initHandler ? initHandler() : null).then(function () {
                    var operations = initOptions && initOptions.operations;
                    if (_.isArray(operations)) {
                        operations.forEach(docModel.invokeOperationHandler.bind(docModel));
                    }
                    _.defer(function () {
                        return importDef.resolve();
                    });
                });
            }.bind(this));
        }.bind(this));

    } // class TestApplication

    // Text ===================================================================

    /**
     * Creates a test application with a real text document model, and applies
     * the passed document operations in order top initialize the document
     * model.
     *
     * @returns {jQuery.Promise}
     *  A promise that will be resolved with the application instance.
     */
    function createTextApp(fileFormat, operations) {
        return require(['io.ox/office/text/model/model']).then(function (TextModel) {

            // create a dummy and empty test application
            var app = new TestApplication('text', fileFormat, TextModel, { operations: operations });

            // launch the application (create model/view/controller, apply operations)
            return app.launch().then(function () {
                // extend text app with needed methods
                app.stopOperationDistribution = function () {};
                app.getView().recalculateDocumentMargin = function () {};
                app.getView().hideFieldFormatPopup = function () {};
                app.getView().isSearchActive = function () {};

                // formatting the document (is asynchronous in large documents)
                return app.getModel().updateDocumentFormatting();
            }).then(function () {
                // set edit mode to true, so that operations generated in tests are applied and not silently skipped
                app.getModel().setEditMode(true);
                return app;
            });
        });
    }

    // Spreadsheet ============================================================

    /**
     * Creates a test application with a real spreadsheet document model, and
     * applies the passed document operations in order top initialize the
     * document model.
     *
     * @returns {jQuery.Promise}
     *  A promise that will be resolved with the application instance.
     */
    function createSpreadsheetApp(fileFormat, operations) {
        return require(['io.ox/office/spreadsheet/model/model']).then(function (SpreadsheetModel) {

            // create a dummy and empty test application
            var app = new TestApplication('spreadsheet', fileFormat, SpreadsheetModel, {
                initHandler: function () { return app.getModel().initFormulaResource(); },
                operations: operations
            });

            // launch the application (create model/view/controller, apply operations)
            return app.launch().then(_.constant(app));
        });
    }

    // Presentation ===========================================================

    /**
     * Creates a test application with a real presentation document model, and applies
     * the passed document operations in order top initialize the document model.
     *
     * @returns {jQuery.Promise}
     *  A promise that will be resolved with the application instance.
     */
    function createPresentationApp(fileFormat, operations) {
        return require(['io.ox/office/presentation/model/model']).then(function (PresentationModel) {

            // create a dummy and empty test application
            var app = new TestApplication('presentation', fileFormat, PresentationModel, { operations: operations });

            // launch the application (create model/view/controller, apply operations)
            return app.launch().then(function () {
                // extend text app with needed methods
                // formatting the document (is asynchronous in large documents)
                return app.getModel().updateDocumentFormatting();
            }).then(function () {
                // set edit mode to true, so that operations generated in tests are applied and not silently skipped
                app.getModel().setEditMode(true);
                return app;
            });
        });
    }

    // window.ox Replacement ==================================================

    /**
     * Replacement for the global 'window.ox' object for unit tests.
     */
    window.ox = {

        // standard properties
        base: 'v=7.x.x.20150511.070854',
        language: 'de_DE',

        // event interface
        on: noop,
        one: noop,
        off: noop,

        // standard contents of the 'ox.ui' object
        ui: {
            createApp: CoreApplication.createApp,
            createWindow: CoreApplication.createWindow,

            App: {
                getCurrentApp: _.constant(null),
                getCurrentWindow: _.constant(null)
            }
        },

        // special additions for the unit tests
        test: {
            text: {
                createApp: createTextApp
            },
            spreadsheet: {
                createApp: createSpreadsheetApp
            },
            presentation: {
                createApp: createPresentationApp
            }
        }
    };

    // ========================================================================
}());
