/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/tk/object/arraytemplate'
], function (ArrayTemplate) {

    'use strict';

    // static class ArrayTemplate =============================================

    describe('Toolkit module ArrayTemplate', function () {

        it('should exist', function () {
            expect(ArrayTemplate).to.be.an('object');
        });

        // static methods -----------------------------------------------------

        describe('method "create"', function () {
            it('should exist', function () {
                expect(ArrayTemplate).itself.to.respondTo('create');
            });
            it('should create a subclass of array', function () {
                var ArrayClass = ArrayTemplate.create(Object);
                expect(ArrayClass).to.be.a('function');
                expect(ArrayClass).to.have.property('prototype');
                expect(ArrayClass.prototype).to.have.property('constructor', ArrayClass);
                expect(ArrayClass).to.have.property('ElementType', Object);
                expect(ArrayClass).to.have.property('ArrayType', ArrayClass);
                expect(ArrayClass).to.have.property('__super__', Array.prototype);
            });
        });
    });

    // generated array class ==================================================

    describe('ArrayTemplate generated subclass', function () {

        // private helpers ----------------------------------------------------

        function Element(n) { this.n = n; }
        Element.compare = function (e1, e2) { return e1.n - e2.n; };
        Element.prototype.clone = function () { return new Element(this.n); };
        Element.prototype.equals = function (e) { return this.n === e.n; };
        Element.prototype.add = function (m) { return new Element(this.n + m); };
        Element.prototype.cloneOdd = function () { return (this.n % 2 > 0) ? this.clone() : null; };
        Element.prototype.toString = function () { return 'e' + this.n; };
        Element.prototype.toJSON = function () { return { n: this.n }; };

        var ArrayClass = ArrayTemplate.create(Element);

        var e1 = new Element(1),
            e2 = new Element(2),
            e3 = new Element(3),
            a0 = new ArrayClass();

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create an array', function () {
                expect(a0).to.be.an.instanceof(ArrayClass);
                expect(a0).to.have.property('length', 0);
                expect(a0).to.respondTo('push');
            });
            it('should insert single elements', function () {
                var a1 = new ArrayClass(e1);
                expect(a1).to.have.length(1);
                expect(a1[0]).to.equal(e1);
                var a2 = new ArrayClass(e1, e2, e3);
                expect(a2).to.have.length(3);
                expect(a2[0]).to.equal(e1);
                expect(a2[1]).to.equal(e2);
                expect(a2[2]).to.equal(e3);
            });
            it('should insert plain JS arrays of elements', function () {
                var a1 = new ArrayClass([e1]);
                expect(a1).to.have.length(1);
                expect(a1[0]).to.equal(e1);
                var a2 = new ArrayClass([e1, e2], [], [e3]);
                expect(a2).to.have.length(3);
                expect(a2[0]).to.equal(e1);
                expect(a2[1]).to.equal(e2);
                expect(a2[2]).to.equal(e3);
            });
            it('should copy-construct arrays of same class', function () {
                var a1 = new ArrayClass(e1, e2), a2 = new ArrayClass(a1);
                expect(a2).to.have.length(2);
                expect(a2[0]).to.equal(e1);
                expect(a2[1]).to.equal(e2);
                var a3 = new ArrayClass(a1, a0, a2);
                expect(a3).to.have.length(4);
                expect(a3[0]).to.equal(e1);
                expect(a3[1]).to.equal(e2);
                expect(a3[2]).to.equal(e1);
                expect(a3[3]).to.equal(e2);
            });
        });

        // static methods -----------------------------------------------------

        describe('method "get"', function () {
            it('should exist', function () {
                expect(ArrayClass).itself.to.respondTo('get');
            });
            it('should return typed array directly', function () {
                var a1 = new ArrayClass(e1, e2, e3);
                expect(ArrayClass.get(a1)).to.equal(a1);
            });
            it('should create an array from a plain JS array', function () {
                var a1 = [e1, e2, e3],
                    a2 = ArrayClass.get(a1);
                expect(a2).to.be.an.instanceof(ArrayClass);
                expect(a2).to.have.length(3);
                expect(a2[0]).to.equal(a1[0]);
                expect(a2[1]).to.equal(a1[1]);
                expect(a2[2]).to.equal(a1[2]);
            });
            it('should create an array from an element', function () {
                var a1 = ArrayClass.get(e1);
                expect(a1).to.be.an.instanceof(ArrayClass);
                expect(a1).to.have.length(1);
                expect(a1[0]).to.equal(e1);
            });
            it('should accept empty arrays', function () {
                expect(ArrayClass.get(a0)).to.equal(a0);
                var a1 = ArrayClass.get([]);
                expect(a1).to.be.an.instanceof(ArrayClass);
                expect(a1).to.have.length(0);
            });
        });

        describe('method "forEach"', function () {
            it('should exist', function () {
                expect(ArrayClass).itself.to.respondTo('forEach');
            });
            it('should visit array elements', function () {
                var a1 = new ArrayClass(e1, e2, e3), spy = sinon.spy();
                ArrayClass.forEach(a1, spy, a0);
                sinon.assert.callCount(spy, 3);
                sinon.assert.alwaysCalledOn(spy, a0);
                sinon.assert.calledWithExactly(spy.getCall(0), e1, 0, a1);
                sinon.assert.calledWithExactly(spy.getCall(1), e2, 1, a1);
                sinon.assert.calledWithExactly(spy.getCall(2), e3, 2, a1);
            });
            it('should visit plain JS array elements', function () {
                var a1 = [e1, e2, e3], spy = sinon.spy();
                ArrayClass.forEach(a1, spy, a0);
                sinon.assert.callCount(spy, 3);
                sinon.assert.alwaysCalledOn(spy, a0);
                sinon.assert.calledWithExactly(spy.getCall(0), e1, 0, a1);
                sinon.assert.calledWithExactly(spy.getCall(1), e2, 1, a1);
                sinon.assert.calledWithExactly(spy.getCall(2), e3, 2, a1);
            });
            it('should visit single elements', function () {
                var spy = sinon.spy();
                ArrayClass.forEach(e1, spy, a0);
                sinon.assert.callCount(spy, 1);
                sinon.assert.alwaysCalledOn(spy, a0);
                sinon.assert.calledWithExactly(spy, e1, 0, e1);
            });
            it('should accept empty arrays', function () {
                var spy = sinon.spy();
                ArrayClass.forEach(a0, spy, a0);
                ArrayClass.forEach([], spy, a0);
                expect(spy.called).to.equal(false);
            });
        });

        describe('method "some"', function () {
            it('should exist', function () {
                expect(ArrayClass).itself.to.respondTo('some');
            });
            it('should visit array elements', function () {
                var a1 = new ArrayClass(e1, e2, e3), stub1 = sinon.stub().returns(false);
                expect(ArrayClass.some(a1, stub1, a0)).to.equal(false);
                sinon.assert.callCount(stub1, 3);
                sinon.assert.alwaysCalledOn(stub1, a0);
                sinon.assert.calledWithExactly(stub1.getCall(0), e1, 0, a1);
                sinon.assert.calledWithExactly(stub1.getCall(1), e2, 1, a1);
                sinon.assert.calledWithExactly(stub1.getCall(2), e3, 2, a1);
                var stub2 = sinon.stub().returns(false);
                stub2.onSecondCall().returns(true);
                expect(ArrayClass.some(a1, stub2, a0)).to.equal(true);
                sinon.assert.callCount(stub2, 2);
                sinon.assert.alwaysCalledOn(stub2, a0);
                sinon.assert.calledWithExactly(stub2.getCall(0), e1, 0, a1);
                sinon.assert.calledWithExactly(stub2.getCall(1), e2, 1, a1);
            });
            it('should visit plain JS array elements', function () {
                var a1 = [e1, e2, e3], stub1 = sinon.stub().returns(false);
                expect(ArrayClass.some(a1, stub1, a0)).to.equal(false);
                sinon.assert.callCount(stub1, 3);
                sinon.assert.alwaysCalledOn(stub1, a0);
                sinon.assert.calledWithExactly(stub1.getCall(0), e1, 0, a1);
                sinon.assert.calledWithExactly(stub1.getCall(1), e2, 1, a1);
                sinon.assert.calledWithExactly(stub1.getCall(2), e3, 2, a1);
                var stub2 = sinon.stub().returns(false);
                stub2.onSecondCall().returns(true);
                expect(ArrayClass.some(a1, stub2, a0)).to.equal(true);
                sinon.assert.callCount(stub2, 2);
                sinon.assert.alwaysCalledOn(stub2, a0);
                sinon.assert.calledWithExactly(stub2.getCall(0), e1, 0, a1);
                sinon.assert.calledWithExactly(stub2.getCall(1), e2, 1, a1);
            });
            it('should visit single elements', function () {
                var stub1 = sinon.stub().returns(false);
                expect(ArrayClass.some(e1, stub1, a0)).to.equal(false);
                sinon.assert.callCount(stub1, 1);
                sinon.assert.alwaysCalledOn(stub1, a0);
                sinon.assert.calledWithExactly(stub1, e1, 0, e1);
                var stub2 = sinon.stub().returns(true);
                expect(ArrayClass.some(e1, stub2, a0)).to.equal(true);
                sinon.assert.callCount(stub2, 1);
                sinon.assert.alwaysCalledOn(stub2, a0);
                sinon.assert.calledWithExactly(stub2, e1, 0, e1);
            });
            it('should accept empty arrays', function () {
                var spy = sinon.spy();
                expect(ArrayClass.some(a0, spy, a0)).to.equal(false);
                expect(ArrayClass.some([], spy, a0)).to.equal(false);
                sinon.assert.callCount(spy, 0);
            });
        });

        describe('method "every"', function () {
            it('should exist', function () {
                expect(ArrayClass).itself.to.respondTo('every');
            });
            it('should visit array elements', function () {
                var a1 = new ArrayClass(e1, e2, e3), stub1 = sinon.stub().returns(true);
                expect(ArrayClass.every(a1, stub1, a0)).to.equal(true);
                sinon.assert.callCount(stub1, 3);
                sinon.assert.alwaysCalledOn(stub1, a0);
                sinon.assert.calledWithExactly(stub1.getCall(0), e1, 0, a1);
                sinon.assert.calledWithExactly(stub1.getCall(1), e2, 1, a1);
                sinon.assert.calledWithExactly(stub1.getCall(2), e3, 2, a1);
                var stub2 = sinon.stub().returns(true);
                stub2.onSecondCall().returns(false);
                expect(ArrayClass.every(a1, stub2, a0)).to.equal(false);
                sinon.assert.callCount(stub2, 2);
                sinon.assert.alwaysCalledOn(stub2, a0);
                sinon.assert.calledWithExactly(stub2.getCall(0), e1, 0, a1);
                sinon.assert.calledWithExactly(stub2.getCall(1), e2, 1, a1);
            });
            it('should visit plain JS array elements', function () {
                var a1 = [e1, e2, e3], stub1 = sinon.stub().returns(true);
                expect(ArrayClass.every(a1, stub1, a0)).to.equal(true);
                sinon.assert.callCount(stub1, 3);
                sinon.assert.alwaysCalledOn(stub1, a0);
                sinon.assert.calledWithExactly(stub1.getCall(0), e1, 0, a1);
                sinon.assert.calledWithExactly(stub1.getCall(1), e2, 1, a1);
                sinon.assert.calledWithExactly(stub1.getCall(2), e3, 2, a1);
                var stub2 = sinon.stub().returns(true);
                stub2.onSecondCall().returns(false);
                expect(ArrayClass.every(a1, stub2, a0)).to.equal(false);
                sinon.assert.callCount(stub2, 2);
                sinon.assert.alwaysCalledOn(stub2, a0);
                sinon.assert.calledWithExactly(stub2.getCall(0), e1, 0, a1);
                sinon.assert.calledWithExactly(stub2.getCall(1), e2, 1, a1);
            });
            it('should visit single elements', function () {
                var stub1 = sinon.stub().returns(true);
                expect(ArrayClass.every(e1, stub1, a0)).to.equal(true);
                sinon.assert.callCount(stub1, 1);
                sinon.assert.alwaysCalledOn(stub1, a0);
                sinon.assert.calledWithExactly(stub1, e1, 0, e1);
                var stub2 = sinon.stub().returns(false);
                expect(ArrayClass.every(e1, stub2, a0)).to.equal(false);
                sinon.assert.callCount(stub2, 1);
                sinon.assert.alwaysCalledOn(stub2, a0);
                sinon.assert.calledWithExactly(stub2, e1, 0, e1);
            });
            it('should accept empty arrays', function () {
                var spy = sinon.spy();
                expect(ArrayClass.every(a0, spy, a0)).to.equal(true);
                expect(ArrayClass.every([], spy, a0)).to.equal(true);
                sinon.assert.callCount(spy, 0);
            });
        });

        describe('method "filter"', function () {
            it('should exist', function () {
                expect(ArrayClass).itself.to.respondTo('filter');
            });
            it('should build an array from a typed array', function () {
                var a1 = new ArrayClass(e1, e2, e3),
                    stub = sinon.stub().returns(true);
                stub.onSecondCall().returns(false);
                var a2 = ArrayClass.filter(a1, stub, a0);
                expect(a2).to.be.an.instanceof(ArrayClass);
                expect(a2).to.have.length(2);
                expect(a2[0]).to.equal(e1);
                expect(a2[1]).to.equal(e3);
                sinon.assert.callCount(stub, 3);
                sinon.assert.alwaysCalledOn(stub, a0);
                sinon.assert.calledWithExactly(stub.getCall(0), e1, 0, a1);
                sinon.assert.calledWithExactly(stub.getCall(1), e2, 1, a1);
                sinon.assert.calledWithExactly(stub.getCall(2), e3, 2, a1);
            });
            it('should build an array from a plain JS array', function () {
                var a1 = [e1, e2, e3],
                    stub = sinon.stub().returns(true);
                stub.onSecondCall().returns(false);
                var a2 = ArrayClass.filter(a1, stub, a0);
                expect(a2).to.be.an.instanceof(ArrayClass);
                expect(a2).to.have.length(2);
                expect(a2[0]).to.equal(e1);
                expect(a2[1]).to.equal(e3);
                sinon.assert.callCount(stub, 3);
                sinon.assert.alwaysCalledOn(stub, a0);
                sinon.assert.calledWithExactly(stub.getCall(0), e1, 0, a1);
                sinon.assert.calledWithExactly(stub.getCall(1), e2, 1, a1);
                sinon.assert.calledWithExactly(stub.getCall(2), e3, 2, a1);
            });
            it('should build an array from a single element', function () {
                var stub = sinon.stub().returns(true);
                var a1 = ArrayClass.filter(e1, stub, a0);
                expect(a1).to.be.an.instanceof(ArrayClass);
                expect(a1).to.have.length(1);
                expect(a1[0]).to.equal(e1);
                sinon.assert.callCount(stub, 1);
                sinon.assert.alwaysCalledOn(stub, a0);
                sinon.assert.calledWithExactly(stub, e1, 0, e1);
            });
            it('should accept empty arrays', function () {
                var spy = sinon.spy(),
                    a1 = ArrayClass.filter(a0, spy, a0);
                expect(a1).to.be.an.instanceof(ArrayClass);
                expect(a1).to.be.empty;
                var a2 = ArrayClass.filter([], spy, a0);
                expect(a2).to.be.an.instanceof(ArrayClass);
                expect(a2).to.be.empty;
                sinon.assert.callCount(spy, 0);
            });
        });

        describe('method "reject"', function () {
            it('should exist', function () {
                expect(ArrayClass).itself.to.respondTo('reject');
            });
            it('should build an array from a typed array', function () {
                var a1 = new ArrayClass(e1, e2, e3),
                    stub = sinon.stub().returns(true);
                stub.onSecondCall().returns(false);
                var a2 = ArrayClass.reject(a1, stub, a0);
                expect(a2).to.be.an.instanceof(ArrayClass);
                expect(a2).to.have.length(1);
                expect(a2[0]).to.equal(e2);
                sinon.assert.callCount(stub, 3);
                sinon.assert.alwaysCalledOn(stub, a0);
                sinon.assert.calledWithExactly(stub.getCall(0), e1, 0, a1);
                sinon.assert.calledWithExactly(stub.getCall(1), e2, 1, a1);
                sinon.assert.calledWithExactly(stub.getCall(2), e3, 2, a1);
            });
            it('should build an array from a plain JS array', function () {
                var a1 = [e1, e2, e3],
                    stub = sinon.stub().returns(true);
                stub.onSecondCall().returns(false);
                var a2 = ArrayClass.reject(a1, stub, a0);
                expect(a2).to.be.an.instanceof(ArrayClass);
                expect(a2).to.have.length(1);
                expect(a2[0]).to.equal(e2);
                sinon.assert.callCount(stub, 3);
                sinon.assert.alwaysCalledOn(stub, a0);
                sinon.assert.calledWithExactly(stub.getCall(0), e1, 0, a1);
                sinon.assert.calledWithExactly(stub.getCall(1), e2, 1, a1);
                sinon.assert.calledWithExactly(stub.getCall(2), e3, 2, a1);
            });
            it('should build an array from a single element', function () {
                var stub = sinon.stub().returns(false);
                var a1 = ArrayClass.reject(e1, stub, a0);
                expect(a1).to.be.an.instanceof(ArrayClass);
                expect(a1).to.have.length(1);
                expect(a1[0]).to.equal(e1);
                sinon.assert.callCount(stub, 1);
                sinon.assert.alwaysCalledOn(stub, a0);
                sinon.assert.calledWithExactly(stub, e1, 0, e1);
            });
            it('should accept empty arrays', function () {
                var spy = sinon.spy(),
                    a1 = ArrayClass.reject(a0, spy, a0);
                expect(a1).to.be.an.instanceof(ArrayClass);
                expect(a1).to.be.empty;
                var a2 = ArrayClass.reject([], spy, a0);
                expect(a2).to.be.an.instanceof(ArrayClass);
                expect(a2).to.be.empty;
                sinon.assert.callCount(spy, 0);
            });
        });

        describe('method "map"', function () {
            it('should exist', function () {
                expect(ArrayClass).itself.to.respondTo('map');
            });
            it('should build an array from a typed array', function () {
                var a1 = new ArrayClass(e1, e2, e3),
                    stub = sinon.stub().returns(e2);
                stub.onSecondCall().returns(e3);
                var a2 = ArrayClass.map(a1, stub, a0);
                expect(a2).to.be.an.instanceof(ArrayClass);
                expect(a2).to.have.length(3);
                expect(a2[0]).to.equal(e2);
                expect(a2[1]).to.equal(e3);
                expect(a2[2]).to.equal(e2);
                sinon.assert.callCount(stub, 3);
                sinon.assert.alwaysCalledOn(stub, a0);
                sinon.assert.calledWithExactly(stub.getCall(0), e1, 0, a1);
                sinon.assert.calledWithExactly(stub.getCall(1), e2, 1, a1);
                sinon.assert.calledWithExactly(stub.getCall(2), e3, 2, a1);
            });
            it('should build an array from a plain JS array', function () {
                var a1 = [e1, e2, e3],
                    stub = sinon.stub().returns(e2);
                stub.onSecondCall().returns(e3);
                var a2 = ArrayClass.map(a1, stub, a0);
                expect(a2).to.be.an.instanceof(ArrayClass);
                expect(a2).to.have.length(3);
                expect(a2[0]).to.equal(e2);
                expect(a2[1]).to.equal(e3);
                expect(a2[2]).to.equal(e2);
                sinon.assert.callCount(stub, 3);
                sinon.assert.alwaysCalledOn(stub, a0);
                sinon.assert.calledWithExactly(stub.getCall(0), e1, 0, a1);
                sinon.assert.calledWithExactly(stub.getCall(1), e2, 1, a1);
                sinon.assert.calledWithExactly(stub.getCall(2), e3, 2, a1);
            });
            it('should build an array from a single element', function () {
                var stub = sinon.stub().returns(e2);
                var a1 = ArrayClass.map(e1, stub, a0);
                expect(a1).to.be.an.instanceof(ArrayClass);
                expect(a1).to.have.length(1);
                expect(a1[0]).to.equal(e2);
                sinon.assert.callCount(stub, 1);
                sinon.assert.alwaysCalledOn(stub, a0);
                sinon.assert.calledWithExactly(stub, e1, 0, e1);
            });
            it('should accept empty arrays', function () {
                var spy = sinon.spy(),
                    a1 = ArrayClass.map(a0, spy, a0);
                expect(a1).to.be.an.instanceof(ArrayClass);
                expect(a1).to.be.empty;
                var a2 = ArrayClass.map([], spy, a0);
                expect(a2).to.be.an.instanceof(ArrayClass);
                expect(a2).to.be.empty;
                sinon.assert.callCount(spy, 0);
            });
            it('should append arrays returned from callback', function () {
                var a1 = new ArrayClass(e1, e2, e3),
                    stub = sinon.stub().returns([]);
                stub.onFirstCall().returns(a1);
                stub.onThirdCall().returns([e1, e2]);
                var a2 = ArrayClass.map(a1, stub, a0);
                expect(a2).to.be.an.instanceof(ArrayClass);
                expect(a2).to.have.length(5);
                expect(a2[0]).to.equal(e1);
                expect(a2[1]).to.equal(e2);
                expect(a2[2]).to.equal(e3);
                expect(a2[3]).to.equal(e1);
                expect(a2[4]).to.equal(e2);
            });
            it('should filter falsy return values', function () {
                var a1 = new ArrayClass(e1, e2, e3),
                    stub = sinon.stub().returns(e2);
                stub.onFirstCall().returns(null);
                stub.onThirdCall().returns(false);
                var a2 = ArrayClass.map(a1, stub, a0);
                expect(a2).to.be.an.instanceof(ArrayClass);
                expect(a2).to.have.length(1);
                expect(a2[0]).to.equal(e2);
            });
        });

        describe('method "invoke"', function () {
            it('should exist', function () {
                expect(ArrayClass).itself.to.respondTo('invoke');
            });
            it('should build an array from a typed array', function () {
                var a1 = new ArrayClass(e1, e2, e3),
                    a2 = ArrayClass.invoke(a1, 'clone');
                expect(a2).to.be.an.instanceof(ArrayClass);
                expect(a2).to.have.length(3);
                expect(a2).to.not.equal(a1);
                expect(a2[0]).to.not.equal(a1[0]);
                expect(a2).to.deep.equal(a1);
                var a3 = ArrayClass.invoke(a1, 'add', 1);
                expect(a3).to.have.length(3);
                expect(a3[0]).to.have.property('n', 2);
                expect(a3[1]).to.have.property('n', 3);
                expect(a3[2]).to.have.property('n', 4);
            });
            it('should build an array from a plain JS array', function () {
                var a1 = [e1, e2, e3],
                    a2 = ArrayClass.invoke(a1, 'clone');
                expect(a2).to.be.an.instanceof(ArrayClass);
                expect(a2).to.have.length(3);
                expect(a2).to.not.equal(a1);
                expect(a2[0]).to.not.equal(a1[0]);
                expect(a2[0]).to.deep.equal(a1[0]);
                expect(a2[1]).to.deep.equal(a1[1]);
                expect(a2[2]).to.deep.equal(a1[2]);
                var a3 = ArrayClass.invoke(a1, 'add', 1);
                expect(a3).to.have.length(3);
                expect(a3[0]).to.have.property('n', 2);
                expect(a3[1]).to.have.property('n', 3);
                expect(a3[2]).to.have.property('n', 4);
            });
            it('should build an array from a single element', function () {
                var a1 = ArrayClass.invoke(e1, 'clone');
                expect(a1).to.be.an.instanceof(ArrayClass);
                expect(a1).to.have.length(1);
                expect(a1[0]).to.not.equal(e1);
                expect(a1[0]).to.deep.equal(e1);
                var a2 = ArrayClass.invoke(e1, 'add', 1);
                expect(a2).to.have.length(1);
                expect(a2[0]).to.have.property('n', 2);
            });
            it('should accept empty arrays', function () {
                var a1 = ArrayClass.invoke(a0, 'clone');
                expect(a1).to.be.an.instanceof(ArrayClass);
                expect(a1).to.be.empty;
                var a2 = ArrayClass.invoke([], 'clone');
                expect(a2).to.be.an.instanceof(ArrayClass);
                expect(a2).to.be.empty;
            });
            it('should append arrays returned from callback', function () {
                var a1 = new ArrayClass(e1, e2, e3),
                    a2 = new ArrayClass({ test: _.constant(a1) }, { test: _.constant([]) }, { test: _.constant([e1, e2]) }),
                    a3 = ArrayClass.invoke(a2, 'test');
                expect(a3).to.be.an.instanceof(ArrayClass);
                expect(a3).to.have.length(5);
                expect(a3[0]).to.equal(e1);
                expect(a3[1]).to.equal(e2);
                expect(a3[2]).to.equal(e3);
                expect(a3[3]).to.equal(e1);
                expect(a3[4]).to.equal(e2);
            });
            it('should filter falsy return values', function () {
                var a1 = new ArrayClass(e1, e2, e3);
                var a2 = ArrayClass.invoke(a1, 'cloneOdd');
                expect(a2).to.be.an.instanceof(ArrayClass);
                expect(a2).to.have.length(2);
                expect(a2[0]).to.deep.equal(e1);
                expect(a2[1]).to.deep.equal(e3);
            });
        });

        describe('method "group"', function () {
            it('should exist', function () {
                expect(ArrayClass).itself.to.respondTo('group');
            });
            function keyer(element, index) { return 'e' + (index % 2); }
            it('should build a group from a typed array', function () {
                var a1 = new ArrayClass(e1, e2, e3, e1, e2, e3),
                    spy = sinon.spy(keyer),
                    group = ArrayClass.group(a1, spy, a0);
                expect(group).to.be.an('object');
                expect(group).to.have.property('e0');
                expect(group).to.have.property('e1');
                expect(group.e0).to.be.an.instanceof(ArrayClass);
                expect(group.e0).to.deep.equal(new ArrayClass(e1, e3, e2));
                expect(group.e1).to.be.an.instanceof(ArrayClass);
                expect(group.e1).to.deep.equal(new ArrayClass(e2, e1, e3));
                sinon.assert.callCount(spy, 6);
                sinon.assert.alwaysCalledOn(spy, a0);
                sinon.assert.calledWithExactly(spy.getCall(0), e1, 0, a1);
                sinon.assert.calledWithExactly(spy.getCall(1), e2, 1, a1);
                sinon.assert.calledWithExactly(spy.getCall(2), e3, 2, a1);
                sinon.assert.calledWithExactly(spy.getCall(3), e1, 3, a1);
                sinon.assert.calledWithExactly(spy.getCall(4), e2, 4, a1);
                sinon.assert.calledWithExactly(spy.getCall(5), e3, 5, a1);
            });
            it('should build a group from a plain JS array', function () {
                var a1 = [e1, e2, e3, e1, e2, e3],
                    spy = sinon.spy(keyer),
                    group = ArrayClass.group(a1, spy, a0);
                expect(group).to.be.an('object');
                expect(group).to.have.property('e0');
                expect(group).to.have.property('e1');
                expect(group.e0).to.be.an.instanceof(ArrayClass);
                expect(group.e0).to.deep.equal(new ArrayClass(e1, e3, e2));
                expect(group.e1).to.be.an.instanceof(ArrayClass);
                expect(group.e1).to.deep.equal(new ArrayClass(e2, e1, e3));
                sinon.assert.callCount(spy, 6);
                sinon.assert.alwaysCalledOn(spy, a0);
                sinon.assert.calledWithExactly(spy.getCall(0), e1, 0, a1);
                sinon.assert.calledWithExactly(spy.getCall(1), e2, 1, a1);
                sinon.assert.calledWithExactly(spy.getCall(2), e3, 2, a1);
                sinon.assert.calledWithExactly(spy.getCall(3), e1, 3, a1);
                sinon.assert.calledWithExactly(spy.getCall(4), e2, 4, a1);
                sinon.assert.calledWithExactly(spy.getCall(5), e3, 5, a1);
            });
            it('should build an array from a single element', function () {
                var spy = sinon.spy(keyer),
                    group = ArrayClass.group(e1, spy, a0);
                expect(group).to.be.an('object');
                expect(group).to.have.property('e0');
                expect(group.e0).to.be.an.instanceof(ArrayClass);
                expect(group.e0).to.deep.equal(new ArrayClass(e1));
                sinon.assert.callCount(spy, 1);
                sinon.assert.alwaysCalledOn(spy, a0);
                sinon.assert.calledWithExactly(spy, e1, 0, e1);
            });
            it('should accept empty arrays', function () {
                var spy = sinon.spy(keyer),
                    group1 = ArrayClass.group(a0, spy, a0);
                expect(group1).to.be.an('object');
                expect(group1).to.be.empty;
                var group2 = ArrayClass.group([], spy, a0);
                expect(group2).to.be.an('object');
                expect(group2).to.be.empty;
                sinon.assert.callCount(spy, 0);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "empty"', function () {
            it('should exist', function () {
                expect(ArrayClass).to.respondTo('empty');
            });
            it('should return true for empty arrays', function () {
                expect(a0.empty()).to.equal(true);
            });
            it('should return true for empty arrays', function () {
                expect(new ArrayClass(e1, e2, e3).empty()).to.equal(false);
            });
        });

        describe('method "first"', function () {
            it('should exist', function () {
                expect(ArrayClass).to.respondTo('first');
            });
            it('should return the first array element', function () {
                expect(new ArrayClass(e1, e2, e3).first()).to.equal(e1);
            });
            it('should return null for empty arrays', function () {
                expect(a0.first()).to.equal(null);
            });
        });

        describe('method "last"', function () {
            it('should exist', function () {
                expect(ArrayClass).to.respondTo('last');
            });
            it('should return the last array element', function () {
                expect(new ArrayClass(e1, e2, e3).last()).to.equal(e3);
            });
            it('should return null for empty arrays', function () {
                expect(a0.last()).to.equal(null);
            });
        });

        describe('method "clear"', function () {
            it('should exist', function () {
                expect(ArrayClass).to.respondTo('clear');
            });
            it('should clear the array', function () {
                var a1 = new ArrayClass(e1, e2, e3);
                a1.clear();
                expect(a1).to.have.length(0);
            });
            it('should return itself', function () {
                var a1 = new ArrayClass(e1, e2, e3);
                expect(a1.clear()).to.equal(a1);
            });
        });

        describe('method "clone"', function () {
            it('should exist', function () {
                expect(ArrayClass).to.respondTo('clone');
            });
            var a1 = new ArrayClass(e1, e2, e3);
            it('should create a shallow clone', function () {
                var spy = sinon.spy(e1, 'clone'), a2 = a1.clone();
                sinon.assert.callCount(spy, 0);
                e1.clone.restore();
                expect(a2).to.be.an.instanceof(ArrayClass);
                expect(a2).to.have.length(3);
                expect(a2).to.not.equal(a1);
                expect(a2[0]).to.equal(a1[0]);
                expect(a2[1]).to.equal(a1[1]);
                expect(a2[2]).to.equal(a1[2]);
            });
            it('should create a deep clone', function () {
                var spy = sinon.spy(e1, 'clone'), a2 = a1.clone(true);
                sinon.assert.calledWithExactly(spy, true);
                e1.clone.restore();
                expect(a2).to.be.an.instanceof(ArrayClass);
                expect(a2).to.have.length(3);
                expect(a2).to.not.equal(a1);
                expect(a2).to.deep.equal(a1);
                expect(a2[0]).to.not.equal(a1[0]);
                expect(a2[1]).to.not.equal(a1[1]);
                expect(a2[2]).to.not.equal(a1[2]);
            });
        });

        describe('method "equals"', function () {
            it('should exist', function () {
                expect(ArrayClass).to.respondTo('equals');
            });
            var a1 = new ArrayClass(e1, e2, e3);
            it('should check shallow equality', function () {
                var a2 = a1.clone(), spy = sinon.spy(e1, 'equals');
                expect(a1.equals(a2)).to.equal(true);
                sinon.assert.callCount(spy, 0);
                a2.pop();
                expect(a1.equals(a2)).to.equal(false);
                sinon.assert.callCount(spy, 0);
                a2.push(e1.clone());
                expect(a1.equals(a2)).to.equal(false);
                sinon.assert.callCount(spy, 0);
                e1.equals.restore();
            });
            it('should check deep equality', function () {
                var a2 = a1.clone(), a3 = a1.clone(true), spy = sinon.spy(e1, 'equals');
                expect(a1.equals(a2, true)).to.equal(true);
                expect(a1.equals(a3, true)).to.equal(true);
                sinon.assert.callCount(spy, 2);
                e1.equals.restore();
            });
        });

        describe('method "append"', function () {
            it('should exist', function () {
                expect(ArrayClass).to.respondTo('append');
            });
            it('should append typed arrays', function () {
                var a1 = new ArrayClass(e1);
                a1.append(new ArrayClass(e2));
                expect(a1).to.have.length(2);
                expect(a1[0]).to.equal(e1);
                expect(a1[1]).to.equal(e2);
                a1.append(a0, new ArrayClass(e3, e1));
                expect(a1).to.have.length(4);
                expect(a1[0]).to.equal(e1);
                expect(a1[1]).to.equal(e2);
                expect(a1[2]).to.equal(e3);
                expect(a1[3]).to.equal(e1);
            });
            it('should append itself', function () {
                var a1 = new ArrayClass(e1, e2);
                a1.append(a1);
                expect(a1).to.have.length(4);
                expect(a1[0]).to.equal(e1);
                expect(a1[1]).to.equal(e2);
                expect(a1[2]).to.equal(e1);
                expect(a1[3]).to.equal(e2);
            });
            it('should append plain JS arrays', function () {
                var a1 = new ArrayClass(e1);
                a1.append([e2]);
                expect(a1).to.have.length(2);
                expect(a1[0]).to.equal(e1);
                expect(a1[1]).to.equal(e2);
                a1.append([], [e3, e1]);
                expect(a1).to.have.length(4);
                expect(a1[0]).to.equal(e1);
                expect(a1[1]).to.equal(e2);
                expect(a1[2]).to.equal(e3);
                expect(a1[3]).to.equal(e1);
            });
            it('should append single elements', function () {
                var a1 = new ArrayClass(e1);
                a1.append(e2);
                expect(a1).to.have.length(2);
                expect(a1[0]).to.equal(e1);
                expect(a1[1]).to.equal(e2);
                a1.append(e3, e1);
                expect(a1).to.have.length(4);
                expect(a1[0]).to.equal(e1);
                expect(a1[1]).to.equal(e2);
                expect(a1[2]).to.equal(e3);
                expect(a1[3]).to.equal(e1);
            });
        });

        describe('method "concat"', function () {
            it('should exist', function () {
                expect(ArrayClass).to.respondTo('concat');
            });
            it('should concatenate typed arrays', function () {
                var a1 = new ArrayClass(e1);
                var a2 = a1.concat(new ArrayClass(e2));
                expect(a1).to.have.length(1);
                expect(a2).to.have.length(2);
                expect(a2[0]).to.equal(e1);
                expect(a2[1]).to.equal(e2);
                var a3 = a2.concat(a0, new ArrayClass(e3, e1));
                expect(a3).to.have.length(4);
                expect(a3[0]).to.equal(e1);
                expect(a3[1]).to.equal(e2);
                expect(a3[2]).to.equal(e3);
                expect(a3[3]).to.equal(e1);
            });
            it('should concatenate to itself', function () {
                var a1 = new ArrayClass(e1, e2);
                var a2 = a1.concat(a1);
                expect(a1).to.have.length(2);
                expect(a2).to.have.length(4);
                expect(a2[0]).to.equal(e1);
                expect(a2[1]).to.equal(e2);
                expect(a2[2]).to.equal(e1);
                expect(a2[3]).to.equal(e2);
            });
            it('should concatenate plain JS arrays', function () {
                var a1 = new ArrayClass(e1);
                var a2 = a1.concat([e2]);
                expect(a1).to.have.length(1);
                expect(a2).to.have.length(2);
                expect(a2[0]).to.equal(e1);
                expect(a2[1]).to.equal(e2);
                var a3 = a2.concat([], [e3, e1]);
                expect(a3).to.have.length(4);
                expect(a3[0]).to.equal(e1);
                expect(a3[1]).to.equal(e2);
                expect(a3[2]).to.equal(e3);
                expect(a3[3]).to.equal(e1);
            });
            it('should append single elements', function () {
                var a1 = new ArrayClass(e1);
                var a2 = a1.concat(e2);
                expect(a1).to.have.length(1);
                expect(a2).to.have.length(2);
                expect(a2[0]).to.equal(e1);
                expect(a2[1]).to.equal(e2);
                var a3 = a2.concat(e3, e1);
                expect(a3).to.have.length(4);
                expect(a3[0]).to.equal(e1);
                expect(a3[1]).to.equal(e2);
                expect(a3[2]).to.equal(e3);
                expect(a3[3]).to.equal(e1);
            });
        });

        describe('method "slice"', function () {
            it('should exist', function () {
                expect(ArrayClass).to.respondTo('slice');
            });
            it('should return a sub-array', function () {
                var a1 = new ArrayClass(e1, e2, e3);
                var a2 = a1.slice(1, 3);
                expect(a2).to.be.an.instanceof(ArrayClass);
                expect(a2).to.have.length(2);
                expect(a2[0]).to.equal(e2);
                expect(a2[1]).to.equal(e3);
            });
        });

        describe('method "filter"', function () {
            it('should exist', function () {
                expect(ArrayClass).to.respondTo('filter');
            });
            it('should filter an array', function () {
                var a1 = new ArrayClass(e1, e2, e3),
                    stub = sinon.stub().returns(true);
                stub.onSecondCall().returns(false);
                var a2 = a1.filter(stub, a0);
                expect(a2).to.be.an.instanceof(ArrayClass);
                expect(a2).to.have.length(2);
                expect(a2[0]).to.equal(e1);
                expect(a2[1]).to.equal(e3);
                sinon.assert.callCount(stub, 3);
                sinon.assert.alwaysCalledOn(stub, a0);
                sinon.assert.calledWithExactly(stub.getCall(0), e1, 0, a1);
                sinon.assert.calledWithExactly(stub.getCall(1), e2, 1, a1);
                sinon.assert.calledWithExactly(stub.getCall(2), e3, 2, a1);
            });
            it('should accept empty arrays', function () {
                var spy = sinon.spy(),
                    a1 = a0.filter(spy, a0);
                expect(a1).to.be.an.instanceof(ArrayClass);
                expect(a1).to.be.empty;
                sinon.assert.callCount(spy, 0);
            });
        });

        describe('method "reject"', function () {
            it('should exist', function () {
                expect(ArrayClass).to.respondTo('reject');
            });
            it('should filter an array', function () {
                var a1 = new ArrayClass(e1, e2, e3),
                    stub = sinon.stub().returns(true);
                stub.onSecondCall().returns(false);
                var a2 = a1.reject(stub, a0);
                expect(a2).to.be.an.instanceof(ArrayClass);
                expect(a2).to.have.length(1);
                expect(a2[0]).to.equal(e2);
                sinon.assert.callCount(stub, 3);
                sinon.assert.alwaysCalledOn(stub, a0);
                sinon.assert.calledWithExactly(stub.getCall(0), e1, 0, a1);
                sinon.assert.calledWithExactly(stub.getCall(1), e2, 1, a1);
                sinon.assert.calledWithExactly(stub.getCall(2), e3, 2, a1);
            });
            it('should accept empty arrays', function () {
                var spy = sinon.spy(),
                    a1 = a0.reject(spy, a0);
                expect(a1).to.be.an.instanceof(ArrayClass);
                expect(a1).to.be.empty;
                sinon.assert.callCount(spy, 0);
            });
        });

        describe('method "group"', function () {
            it('should exist', function () {
                expect(ArrayClass).to.respondTo('group');
            });
            function keyer(element, index) { return 'e' + (index % 2); }
            it('should build a group from an array', function () {
                var a1 = new ArrayClass(e1, e2, e3, e1, e2, e3),
                    spy = sinon.spy(keyer),
                    group = a1.group(spy, a0);
                expect(group).to.be.an('object');
                expect(group).to.have.property('e0');
                expect(group).to.have.property('e1');
                expect(group.e0).to.be.an.instanceof(ArrayClass);
                expect(group.e0).to.deep.equal(new ArrayClass(e1, e3, e2));
                expect(group.e1).to.be.an.instanceof(ArrayClass);
                expect(group.e1).to.deep.equal(new ArrayClass(e2, e1, e3));
                sinon.assert.callCount(spy, 6);
                sinon.assert.alwaysCalledOn(spy, a0);
                sinon.assert.calledWithExactly(spy.getCall(0), e1, 0, a1);
                sinon.assert.calledWithExactly(spy.getCall(1), e2, 1, a1);
                sinon.assert.calledWithExactly(spy.getCall(2), e3, 2, a1);
                sinon.assert.calledWithExactly(spy.getCall(3), e1, 3, a1);
                sinon.assert.calledWithExactly(spy.getCall(4), e2, 4, a1);
                sinon.assert.calledWithExactly(spy.getCall(5), e3, 5, a1);
            });
            it('should accept empty arrays', function () {
                var spy = sinon.spy(keyer),
                    group = a0.group(spy, a0);
                expect(group).to.be.an('object');
                expect(group).to.be.empty;
                sinon.assert.callCount(spy, 0);
            });
        });

        describe('method "find"', function () {
            it('should exist', function () {
                expect(ArrayClass).to.respondTo('find');
            });
            it('should return the first matching element', function () {
                var a1 = new ArrayClass(e1, e2, e3, e2, e1),
                    stub = sinon.stub().returns(false);
                stub.onSecondCall().returns(true);
                expect(a1.find(stub, a0)).to.equal(e2);
                sinon.assert.callCount(stub, 2);
                sinon.assert.alwaysCalledOn(stub, a0);
                sinon.assert.calledWithExactly(stub.getCall(0), e1, 0, a1);
                sinon.assert.calledWithExactly(stub.getCall(1), e2, 1, a1);
            });
            it('should return null if no element matches', function () {
                var a1 = new ArrayClass(e1, e2, e3, e2, e1),
                    stub = sinon.stub().returns(false);
                expect(a1.find(stub, a0)).to.equal(null);
                sinon.assert.callCount(stub, 5);
                sinon.assert.alwaysCalledOn(stub, a0);
            });
            it('should accept empty arrays', function () {
                var spy = sinon.spy();
                expect(a0.find(spy, a0)).to.equal(null);
                sinon.assert.callCount(spy, 0);
            });
        });

        describe('method "sort"', function () {
            it('should exist', function () {
                expect(ArrayClass).to.respondTo('sort');
            });
            it('should sort with default element comparator', function () {
                var a1 = new ArrayClass(e3, e1, e2);
                expect(a1.sort()).to.equal(a1);
                expect(a1).to.have.length(3);
                expect(a1[0]).to.equal(e1);
                expect(a1[1]).to.equal(e2);
                expect(a1[2]).to.equal(e3);
            });
            it('should sort with custom sort order', function () {
                var a1 = new ArrayClass(e3, e1, e2);
                a1.sort(function (el1, el2) { return el2.n - el1.n; });
                expect(a1).to.have.length(3);
                expect(a1[0]).to.equal(e3);
                expect(a1[1]).to.equal(e2);
                expect(a1[2]).to.equal(e1);
            });
            it('should sort without default element comparator', function () {
                var ArrayClass2 = ArrayTemplate.create(Number);
                var a1 = new ArrayClass2(10, 2, 1);
                a1.sort();
                expect(a1[0]).to.equal(1);
                expect(a1[1]).to.equal(10);
                expect(a1[2]).to.equal(2);
                a1.sort(function (el1, el2) { return el1 - el2; });
                expect(a1[0]).to.equal(1);
                expect(a1[1]).to.equal(2);
                expect(a1[2]).to.equal(10);
            });
        });

        describe('method "sortBy"', function () {
            it('should exist', function () {
                expect(ArrayClass).to.respondTo('sortBy');
            });
            it('should sort with callback values', function () {
                function sorter(e) { return -e.n; }
                var a1 = new ArrayClass(e3, e1, e2), spy = sinon.spy(sorter);
                expect(a1.sortBy(spy, a0)).to.equal(a1);
                sinon.assert.alwaysCalledOn(spy, a0);
                expect(a1).to.have.length(3);
                expect(a1[0]).to.equal(e3);
                expect(a1[1]).to.equal(e2);
                expect(a1[2]).to.equal(e1);
            });
            it('should sort with property name', function () {
                var a1 = new ArrayClass(e3, e1, e2);
                expect(a1.sortBy('n')).to.equal(a1);
                expect(a1).to.have.length(3);
                expect(a1[0]).to.equal(e1);
                expect(a1[1]).to.equal(e2);
                expect(a1[2]).to.equal(e3);
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(ArrayClass).to.respondTo('toString');
            });
            var a1 = new ArrayClass(e1, e2, e3);
            it('should stringify an array', function () {
                expect(a1.toString()).to.equal('e1,e2,e3');
            });
            it('should use the separator string', function () {
                expect(a1.toString(' + ')).to.equal('e1 + e2 + e3');
            });
            it('should stringify implicitly', function () {
                expect('[' + a1 + ']').to.equal('[e1,e2,e3]');
            });
        });

        describe('method "toJSON"', function () {
            it('should exist', function () {
                expect(ArrayClass).to.respondTo('toJSON');
            });
            var a1 = new ArrayClass(e1, e2, e3);
            it('should convert to JSON data', function () {
                expect(a1.toJSON()).to.deep.equal([{ n: 1 }, { n: 2 }, { n: 3 }]);
            });
            it('should stringify to JSON', function () {
                expect(JSON.stringify(a1)).to.equal('[{"n":1},{"n":2},{"n":3}]');
            });
        });
    });

    // ========================================================================
});
