/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/tk/locale/parser'
], function (Parser) {

    'use strict';

    // static class Parser ====================================================

    describe('Toolkit module Parser', function () {

        it('should exist', function () {
            expect(Parser).to.be.an('object');
        });

        // private helpers ----------------------------------------------------

        function utcDate(y, m, d) {
            return new Date(Date.UTC(y, m, d));
        }

        function utcTime(h, m, s, ms) {
            return new Date(Date.UTC(1970, 0, 1, h, m, s, ms || 0));
        }

        var CENTURY = Math.floor(new Date().getFullYear() / 100) * 100;

        // static methods -----------------------------------------------------

        describe('method "numberToString"', function () {
            it('should exist', function () {
                expect(Parser).itself.to.respondTo('numberToString');
            });
            it('should convert numbers to string using local decimal separator', function () {
                expect(Parser.numberToString(0)).to.equal('0');
                expect(Parser.numberToString(42)).to.equal('42');
                expect(Parser.numberToString(-1)).to.equal('-1');
                expect(Parser.numberToString(0.5)).to.equal('0,5');
                expect(Parser.numberToString(-12.5)).to.equal('-12,5');
            });
        });

        describe('method "stringToNumber"', function () {
            it('should exist', function () {
                expect(Parser).itself.to.respondTo('stringToNumber');
            });
            it('should convert strings to number using local decimal separator', function () {
                expect(Parser.stringToNumber('0')).to.equal(0);
                expect(Parser.stringToNumber('42')).to.equal(42);
                expect(Parser.stringToNumber('-1')).to.equal(-1);
                expect(Parser.stringToNumber('0,5')).to.equal(0.5);
                expect(Parser.stringToNumber('-12,5')).to.equal(-12.5);
            });
            it('should return NaN for invalid strings', function () {
                expect(Parser.stringToNumber('')).to.be.nan;
                expect(Parser.stringToNumber('a')).to.be.nan;
            });
        });

        describe('method "parseLeadingNumber"', function () {
            it('should exist', function () {
                expect(Parser).itself.to.respondTo('parseLeadingNumber');
            });
            it('should parse numbers using local decimal separator', function () {
                expect(Parser.parseLeadingNumber('42abc')).to.deep.equal({ number: 42, text: '42', sign: '', remaining: 'abc' });
                expect(Parser.parseLeadingNumber('42,abc')).to.deep.equal({ number: 42, text: '42,', sign: '', remaining: 'abc' });
                expect(Parser.parseLeadingNumber('42,5abc')).to.deep.equal({ number: 42.5, text: '42,5', sign: '', remaining: 'abc' });
                expect(Parser.parseLeadingNumber('0,5abc')).to.deep.equal({ number: 0.5, text: '0,5', sign: '', remaining: 'abc' });
                expect(Parser.parseLeadingNumber(',5abc')).to.deep.equal({ number: 0.5, text: ',5', sign: '', remaining: 'abc' });
                expect(Parser.parseLeadingNumber(',5,5')).to.deep.equal({ number: 0.5, text: ',5', sign: '', remaining: ',5' });
                expect(Parser.parseLeadingNumber('42,5E3abc')).to.deep.equal({ number: 42500, text: '42,5E3', sign: '', remaining: 'abc' });
                expect(Parser.parseLeadingNumber('42,5e0abc')).to.deep.equal({ number: 42.5, text: '42,5e0', sign: '', remaining: 'abc' });
                expect(Parser.parseLeadingNumber('42,5e+03abc')).to.deep.equal({ number: 42500, text: '42,5e+03', sign: '', remaining: 'abc' });
                expect(Parser.parseLeadingNumber('42,5e-03abc')).to.deep.equal({ number: 0.0425, text: '42,5e-03', sign: '', remaining: 'abc' });
                expect(Parser.parseLeadingNumber('42,5e3,5')).to.deep.equal({ number: 42500, text: '42,5e3', sign: '', remaining: ',5' });
                expect(Parser.parseLeadingNumber('42e3,5')).to.deep.equal({ number: 42000, text: '42e3', sign: '', remaining: ',5' });
                expect(Parser.parseLeadingNumber('42.5')).to.deep.equal({ number: 42, text: '42', sign: '', remaining: '.5' });
            });
            it('should reject invalid text', function () {
                expect(Parser.parseLeadingNumber('abc')).to.equal(null);
                expect(Parser.parseLeadingNumber(',')).to.equal(null);
                expect(Parser.parseLeadingNumber('e3')).to.equal(null);
                expect(Parser.parseLeadingNumber(',e3')).to.equal(null);
            });
            it('should handle leading sign according to option', function () {
                expect(Parser.parseLeadingNumber('+42abc')).to.equal(null);
                expect(Parser.parseLeadingNumber('+42abc', { sign: true })).to.deep.equal({ number: 42, text: '+42', sign: '+', remaining: 'abc' });
                expect(Parser.parseLeadingNumber('-42abc')).to.equal(null);
                expect(Parser.parseLeadingNumber('-42abc', { sign: true })).to.deep.equal({ number: -42, text: '-42', sign: '-', remaining: 'abc' });
            });
            it('should use custom decimal separator', function () {
                expect(Parser.parseLeadingNumber('42#5abc')).to.deep.equal({ number: 42, text: '42', sign: '', remaining: '#5abc' });
                expect(Parser.parseLeadingNumber('42#5abc', { dec: '#' })).to.deep.equal({ number: 42.5, text: '42#5', sign: '', remaining: 'abc' });
                expect(Parser.parseLeadingNumber('42,5abc', { dec: '#' })).to.deep.equal({ number: 42, text: '42', sign: '', remaining: ',5abc' });
                expect(Parser.parseLeadingNumber('+42#5abc', { dec: '#' })).to.equal(null);
                expect(Parser.parseLeadingNumber('+42#5abc', { dec: '#', sign: true })).to.deep.equal({ number: 42.5, text: '+42#5', sign: '+', remaining: 'abc' });
                expect(Parser.parseLeadingNumber('-42#5abc', { dec: '#', sign: true })).to.deep.equal({ number: -42.5, text: '-42#5', sign: '-', remaining: 'abc' });
            });
            it('should reject remaining garbage after number', function () {
                expect(Parser.parseLeadingNumber('42abc', { complete: true })).to.equal(null);
            });
        });

        describe('method "parseLeadingDate"', function () {
            it('should exist', function () {
                expect(Parser).itself.to.respondTo('parseLeadingDate');
            });
            it('should parse valid dates', function () {
                expect(Parser.parseLeadingDate('21/3/1999')).to.deep.equal({ D: 21, M: 2, Y: 1999, date: utcDate(1999, 2, 21), text: '21/3/1999', remaining: '' });
                expect(Parser.parseLeadingDate('21/3/99')).to.deep.equal({ D: 21, M: 2, Y: 99, date: utcDate(CENTURY + 99, 2, 21), text: '21/3/99', remaining: '' });
            });
            // TODO: more
        });

        describe('method "parseLeadingTime"', function () {
            it('should exist', function () {
                expect(Parser).itself.to.respondTo('parseLeadingTime');
            });
            it('should parse valid times', function () {
                expect(Parser.parseLeadingTime('0:0:0')).to.deep.equal({ h: 0, m: 0, s: 0, ms: null, time: utcTime(0, 0, 0), text: '0:0:0', remaining: '' });
                expect(Parser.parseLeadingTime('23:59:59')).to.deep.equal({ h: 23, m: 59, s: 59, ms: null, time: utcTime(23, 59, 59), text: '23:59:59', remaining: '' });
            });
            // TODO: more
        });

        describe('method "parseFormatCode"', function () {
            it('should exist', function () {
                expect(Parser).itself.to.respondTo('parseFormatCode');
            });
            it('should cache the parsed format codes', function () {
                var format = Parser.parseFormatCode('General');
                expect(Parser.parseFormatCode('General')).to.equal(format);
            });
            it('should recognize "standard" category', function () {
                expect(Parser.parseFormatCode('General')).to.have.a.property('category', 'standard');
            });
            it('should recognize "number" category', function () {
                expect(Parser.parseFormatCode('0')).to.have.a.property('category', 'number');
                expect(Parser.parseFormatCode('0.00')).to.have.a.property('category', 'number');
                expect(Parser.parseFormatCode('#,##0')).to.have.a.property('category', 'number');
                expect(Parser.parseFormatCode('#,##0.00')).to.have.a.property('category', 'number');
                expect(Parser.parseFormatCode('#,##0.00;[Red]-#,##0.00')).to.have.a.property('category', 'number');
                expect(Parser.parseFormatCode('0;@')).to.have.a.property('category', 'number');
            });
            it('should recognize "scientific" category', function () {
                expect(Parser.parseFormatCode('0E+00')).to.have.a.property('category', 'scientific');
                expect(Parser.parseFormatCode('0.00E+00')).to.have.a.property('category', 'scientific');
                expect(Parser.parseFormatCode('0.00E-00')).to.have.a.property('category', 'scientific');
                expect(Parser.parseFormatCode('#,##0E+00')).to.have.a.property('category', 'scientific');
                expect(Parser.parseFormatCode('#,##0.00E+00')).to.have.a.property('category', 'scientific');
                expect(Parser.parseFormatCode('#,##0.00E+00;[Red]-#,##0.00E+00')).to.have.a.property('category', 'scientific');
                expect(Parser.parseFormatCode('0E+00;@')).to.have.a.property('category', 'scientific');
            });
            it('should recognize "percent" category', function () {
                expect(Parser.parseFormatCode('0%')).to.have.a.property('category', 'percent');
                expect(Parser.parseFormatCode('0.00%')).to.have.a.property('category', 'percent');
                expect(Parser.parseFormatCode('0.00%;[Red]-0.00%')).to.have.a.property('category', 'percent');
                expect(Parser.parseFormatCode('0%;@')).to.have.a.property('category', 'percent');
            });
            it('should recognize "fraction" category', function () {
                expect(Parser.parseFormatCode('# ?/?')).to.have.a.property('category', 'fraction');
                expect(Parser.parseFormatCode('# ??/??')).to.have.a.property('category', 'fraction');
                expect(Parser.parseFormatCode('# ??/10')).to.have.a.property('category', 'fraction');
                expect(Parser.parseFormatCode('# ??/16')).to.have.a.property('category', 'fraction');
            });
            it('should recognize "currency" category', function () {
                expect(Parser.parseFormatCode('$#,##0.00')).to.have.a.property('category', 'currency');
                expect(Parser.parseFormatCode('#,##0.00 \u20ac')).to.have.a.property('category', 'currency');
                expect(Parser.parseFormatCode('[$$-409]#,##0.00')).to.have.a.property('category', 'currency');
                expect(Parser.parseFormatCode('#,##0.00 [$\u20ac-407]')).to.have.a.property('category', 'currency');
            });
            it('should recognize "date" category', function () {
                expect(Parser.parseFormatCode('MM/DD/YYYY')).to.have.a.property('category', 'date');
                expect(Parser.parseFormatCode('D. MMM. YYYY')).to.have.a.property('category', 'date');
            });
            it('should recognize "time" category', function () {
                expect(Parser.parseFormatCode('hh:mm:ss')).to.have.a.property('category', 'time');
                expect(Parser.parseFormatCode('hh:mm:ss AM/PM')).to.have.a.property('category', 'time');
                expect(Parser.parseFormatCode('[hh]:mm:ss')).to.have.a.property('category', 'time');
                expect(Parser.parseFormatCode('mm:ss.00')).to.have.a.property('category', 'time');
            });
            it('should recognize "datetime" category', function () {
                expect(Parser.parseFormatCode('MM/DD/YYYY hh:mm:ss')).to.have.a.property('category', 'datetime');
                expect(Parser.parseFormatCode('DD. MMM. hh:mm AM/PM')).to.have.a.property('category', 'datetime');
            });
            it('should recognize "text" category', function () {
                expect(Parser.parseFormatCode('@')).to.have.a.property('category', 'text');
            });
            it('should recognize "custom" category', function () {
                expect(Parser.parseFormatCode('-General')).to.have.a.property('category', 'custom');
                expect(Parser.parseFormatCode('General;-General')).to.have.a.property('category', 'custom');
                expect(Parser.parseFormatCode('General;-0')).to.have.a.property('category', 'custom');
                expect(Parser.parseFormatCode('0;0E+00')).to.have.a.property('category', 'custom');
                expect(Parser.parseFormatCode('0;0%')).to.have.a.property('category', 'custom');
                expect(Parser.parseFormatCode('0E+00%')).to.have.a.property('category', 'custom');
                expect(Parser.parseFormatCode('$0%')).to.have.a.property('category', 'custom');
            });
        });

        describe('method "generateCJKDateTimeFormat"', function () {
            it('should exist', function () {
                expect(Parser).itself.to.respondTo('generateCJKDateTimeFormat');
            });
            it('should convert the {H} meta token', function () {
                expect(Parser.generateCJKDateTimeFormat('{H}:mm:ss', 'trad')).to.equal('h:mm:ss');
                expect(Parser.generateCJKDateTimeFormat('{H}:mm:ss', 'trad', { ampm: true })).to.equal('AM/PMh:mm:ss');
                expect(Parser.generateCJKDateTimeFormat('{H}:mm:ss', 'trad', { long: true })).to.equal('hh:mm:ss');
                expect(Parser.generateCJKDateTimeFormat('{H}:mm:ss', 'trad', { ampm: true, long: true })).to.equal('AM/PMhh:mm:ss');
            });
            it('should convert the meta tokens for all scripts', function () {
                expect(Parser.generateCJKDateTimeFormat('YYYY{YEAR}MM{MONTH}DD{DAY} {H}{HOUR}:mm{MINUTE}:ss{SECOND}', 'trad')).to.equal('YYYY\u5e74MM\u6708DD\u65e5 h\u6642:mm\u5206:ss\u79d2');
                expect(Parser.generateCJKDateTimeFormat('YYYY{YEAR}MM{MONTH}DD{DAY} {H}{HOUR}:mm{MINUTE}:ss{SECOND}', 'simp')).to.equal('YYYY\u5e74MM\u6708DD\u65e5 h\u65f6:mm\u5206:ss\u79d2');
                expect(Parser.generateCJKDateTimeFormat('YYYY{YEAR}MM{MONTH}DD{DAY} {H}{HOUR}:mm{MINUTE}:ss{SECOND}', 'hang')).to.equal('YYYY\ub144MM\uc6d4DD\uc77c h\uc2dc:mm\ubd84:ss\ucd08');
            });
        });
    });

    // ========================================================================
});
