/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/matrix',
    'io.ox/office/spreadsheet/model/formula/operand',
    'io.ox/office/spreadsheet/model/formula/formulacontext',
    'io.ox/office/spreadsheet/model/formula/impl/referencefuncs'
], function (SheetUtils, Matrix, Operand, FormulaContext, ReferenceFuncs) {

    'use strict';

    var ErrorCode = SheetUtils.ErrorCode;
    var Address = SheetUtils.Address;
    var Range3D = SheetUtils.Range3D;
    var Range3DArray = SheetUtils.Range3DArray;

    // initialize the formula context
    var context = null;
    before(function (done) {
        ox.test.spreadsheet.createApp('ooxml').done(function (app) {
            context = new FormulaContext(app.getModel(), 0, new Address(4, 5));
            done();
        });
    });

    // module ReferenceFuncs ==================================================

    describe('Spreadsheet module ReferenceFuncs', function () {

        it('should exist', function () {
            expect(ReferenceFuncs).to.be.an('object');
        });

        // function implementations -------------------------------------------

        describe('function "COLUMN"', function () {
            var COLUMN = ReferenceFuncs.COLUMN;
            it('should exist', function () {
                expect(COLUMN).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(COLUMN).itself.to.respondTo('resolve');
            });
            it('should return the column index', function () {
                expect(COLUMN.resolve.call(context, Range3D.create(1, 2, 3, 4, 0, 0))).to.equal(2);
                expect(COLUMN.resolve.call(context)).to.equal(5);
            });
        });

        describe('function "COLUMNS"', function () {

            var mat = new Matrix([[1, 2, 3], [4, 5, 6]]);
            var r1 = Range3D.create(1, 1, 2, 3, 0, 0);
            var r2 = Range3D.create(1, 1, 2, 3, 0, 1);

            var COLUMNS = ReferenceFuncs.COLUMNS;
            it('should exist', function () {
                expect(COLUMNS).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(COLUMNS).itself.to.respondTo('resolve');
            });
            it('should return the width of the operand', function () {
                expect(COLUMNS.resolve.call(context, new Operand(context, 42))).to.equal(1);
                expect(COLUMNS.resolve.call(context, new Operand(context, mat))).to.equal(3);
                expect(COLUMNS.resolve.call(context, new Operand(context, r1))).to.equal(2);
            });
            it('should throw errors for complex ranges', function () {
                expect(COLUMNS.resolve.bind(context, new Operand(context, r2))).to['throw'](ErrorCode).that.equals(ErrorCode.VALUE);
                expect(COLUMNS.resolve.bind(context, new Operand(context, new Range3DArray(r1, r1)))).to['throw'](ErrorCode).that.equals(ErrorCode.REF);
            });
        });

        describe('function "HLOOKUP"', function () {
            var HLOOKUP = ReferenceFuncs.HLOOKUP;
            it('should exist', function () {
                expect(HLOOKUP).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(HLOOKUP).itself.to.respondTo('resolve');
            });

            var stringMatrix = new Matrix([['Achsen', 'Getriebe', 'Schrauben'], [4, 4, 9], [5, 7, 10], [6, 8, 11]]);
            var numberMatrix = new Matrix([[1, 2, 3], ['a', 'b', 'c'], ['d', 'e', 'f']]);

            it('should return the result of the horizontal search', function () {
                expect(HLOOKUP.resolve.call(context, 'Achsen', stringMatrix, 2, true)).to.equal(4);
                expect(HLOOKUP.resolve.call(context, 'Getriebe', stringMatrix, 3, false)).to.equal(7);
                expect(HLOOKUP.resolve.call(context, 'B', stringMatrix, 3, true)).to.equal(5);
                expect(HLOOKUP.resolve.call(context, 'Schrauben', stringMatrix, 4)).to.equal(11);

                expect(HLOOKUP.resolve.call(context, 3, numberMatrix, 2, true)).to.equal('c');
                expect(HLOOKUP.resolve.call(context, 2.5, numberMatrix, 2, true)).to.equal('b');
                expect(HLOOKUP.resolve.call(context, 4, numberMatrix, 2, true)).to.equal('c');

                expect(HLOOKUP.resolve.bind(context, 'A', stringMatrix, 3, true)).to['throw'](ErrorCode).that.equals(ErrorCode.NA);
                expect(HLOOKUP.resolve.bind(context, 'Achsen', stringMatrix, 0, true)).to['throw'](ErrorCode).that.equals(ErrorCode.VALUE);
                expect(HLOOKUP.resolve.bind(context, 'Achsen', stringMatrix, 100, true)).to['throw'](ErrorCode).that.equals(ErrorCode.REF);
            });

            it('should return the result of the horizontal search with wildcards', function () {
                expect(HLOOKUP.resolve.call(context, 'Schra?ben', stringMatrix, 4)).to.equal(8);
                expect(HLOOKUP.resolve.call(context, 'Schra?ben', stringMatrix, 4, false)).to.equal(11);

                expect(HLOOKUP.resolve.call(context, 'G*e', stringMatrix, 4, false)).to.equal(8);
                expect(HLOOKUP.resolve.bind(context, 'G?e', stringMatrix, 4, false)).to['throw'](ErrorCode).that.equals(ErrorCode.NA);
            });
        });

        describe('function "LOOKUP"', function () {
            var LOOKUP = ReferenceFuncs.LOOKUP;
            it('should exist', function () {
                expect(LOOKUP).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(LOOKUP).itself.to.respondTo('resolve');
            });

            var rateData = [4.14, 4.19, 5.17, 5.77, 6.39];
            var colorData = ['Rot', 'Orange', 'Gelb', 'Gruen', 'Blau'];

            var rateMatrix = new Matrix([rateData]);
            var colorMatrix = new Matrix([colorData]);
            var matrix = new Matrix([rateData, colorData]);

            it('should return the result of the search', function () {
                expect(LOOKUP.resolve.call(context, 4.19, rateMatrix, colorMatrix)).to.equal('Orange');
                expect(LOOKUP.resolve.call(context, 5.75, rateMatrix, colorMatrix)).to.equal('Gelb');
                expect(LOOKUP.resolve.call(context, 7.66, rateMatrix, colorMatrix)).to.equal('Blau');

                expect(LOOKUP.resolve.call(context, 4.19, matrix)).to.equal('Orange');

                expect(LOOKUP.resolve.bind(context, 0, rateMatrix, colorMatrix)).to['throw'](ErrorCode).that.equals(ErrorCode.NA);

            });

            it('should return the result of the search in unsorted matrix', function () {
                var unsorted = new Matrix([[4.14, 4.19, 15.17, 5.77, 6.39], colorData]);
                expect(LOOKUP.resolve.call(context, 7.66, unsorted)).to.equal('Orange');
            });

            var rateMatrix2 = new Matrix([['rate', null, 4.14, 4.19, 5.17]]);
            var colorMatrix2 = new Matrix([['color', null, 'Rot', 'Orange', 'Gelb']]);

            it('should return the result of the search with confusing types', function () {
                expect(LOOKUP.resolve.call(context, 4.19, rateMatrix2, colorMatrix2)).to.equal('Orange');
                expect(LOOKUP.resolve.call(context, 5.75, rateMatrix2, colorMatrix2)).to.equal('Gelb');
                expect(LOOKUP.resolve.call(context, 7.66, rateMatrix2, colorMatrix2)).to.equal('Gelb');

                expect(LOOKUP.resolve.call(context, 4.19, new Matrix([['rate', null, 4.14, 4.19]]), new Matrix([['color', null, 'Rot', 'Orange']]))).to.equal('Orange');
                expect(LOOKUP.resolve.call(context, 4.19, new Matrix([['rate', null, 4.14]]), new Matrix([['color', null, 'Rot']]))).to.equal('Rot');

                expect(LOOKUP.resolve.bind(context, 4.19, new Matrix([['rate', null]]), new Matrix([['color', null]]))).to['throw'](ErrorCode).that.equals(ErrorCode.NA);
                expect(LOOKUP.resolve.bind(context, 4.19, new Matrix([['rate']]), new Matrix([['color']]))).to['throw'](ErrorCode).that.equals(ErrorCode.NA);
            });
        });

        describe('function "MATCH"', function () {
            var MATCH = ReferenceFuncs.MATCH;
            it('should exist', function () {
                expect(MATCH).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(MATCH).itself.to.respondTo('resolve');
            });

            var count = new Matrix([[25, 38, 40, 41]]);

            it('should return the result of the match', function () {
                expect(MATCH.resolve.call(context, 39, count, 1)).to.equal(2);
                expect(MATCH.resolve.call(context, 41, count, 0)).to.equal(4);
                expect(MATCH.resolve.call(context, 41, count, 1)).to.equal(4);

                expect(MATCH.resolve.bind(context, 39, count, 0)).to['throw'](ErrorCode).that.equals(ErrorCode.NA);
                expect(MATCH.resolve.bind(context, 39, count, -1)).to['throw'](ErrorCode).that.equals(ErrorCode.NA);
                expect(MATCH.resolve.bind(context, 41, count, -1)).to['throw'](ErrorCode).that.equals(ErrorCode.NA);
            });
        });

        describe('function "ROW"', function () {
            var ROW = ReferenceFuncs.ROW;
            it('should exist', function () {
                expect(ROW).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(ROW).itself.to.respondTo('resolve');
            });
            it('should return the row index', function () {
                expect(ROW.resolve.call(context, Range3D.create(1, 2, 3, 4, 0, 0))).to.equal(3);
                expect(ROW.resolve.call(context)).to.equal(6);
            });
        });

        describe('function "ROWS"', function () {

            var mat = new Matrix([[1, 2, 3], [4, 5, 6]]);
            var r1 = Range3D.create(1, 1, 2, 3, 0, 0);
            var r2 = Range3D.create(1, 1, 2, 3, 0, 1);

            var ROWS = ReferenceFuncs.ROWS;
            it('should exist', function () {
                expect(ROWS).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(ROWS).itself.to.respondTo('resolve');
            });
            it('should return the width of the operand', function () {
                expect(ROWS.resolve.call(context, new Operand(context, 42))).to.equal(1);
                expect(ROWS.resolve.call(context, new Operand(context, mat))).to.equal(2);
                expect(ROWS.resolve.call(context, new Operand(context, r1))).to.equal(3);
            });
            it('should throw errors for complex ranges', function () {
                expect(ROWS.resolve.bind(context, new Operand(context, r2))).to['throw'](ErrorCode).that.equals(ErrorCode.VALUE);
                expect(ROWS.resolve.bind(context, new Operand(context, new Range3DArray(r1, r1)))).to['throw'](ErrorCode).that.equals(ErrorCode.REF);
            });
        });

        describe('function "VLOOKUP"', function () {
            var VLOOKUP = ReferenceFuncs.VLOOKUP;
            it('should exist', function () {
                expect(VLOOKUP).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(VLOOKUP).itself.to.respondTo('resolve');
            });

            var stringMatrix = new Matrix([['Achsen', 4, 5, 6], ['Getriebe', 4, 7, 8], ['Schrauben', 9, 10, 11]]);
            var numberMatrix = new Matrix([[1, 'a', 'd'], [2, 'b', 'e'], [3, 'c', 'f']]);

            it('should return the result of the vertical search', function () {
                expect(VLOOKUP.resolve.call(context, 'Achsen', stringMatrix, 2, true)).to.equal(4);
                expect(VLOOKUP.resolve.call(context, 'Getriebe', stringMatrix, 3, false)).to.equal(7);
                expect(VLOOKUP.resolve.call(context, 'B', stringMatrix, 3, true)).to.equal(5);
                expect(VLOOKUP.resolve.call(context, 'Schrauben', stringMatrix, 4)).to.equal(11);

                expect(VLOOKUP.resolve.call(context, 3,  numberMatrix, 2, true)).to.equal('c');
                expect(VLOOKUP.resolve.call(context, 2.5,  numberMatrix, 2, true)).to.equal('b');
                expect(VLOOKUP.resolve.call(context, 4,  numberMatrix, 2, true)).to.equal('c');

                expect(VLOOKUP.resolve.bind(context, 'A', stringMatrix, 3, true)).to['throw'](ErrorCode).that.equals(ErrorCode.NA);
                expect(VLOOKUP.resolve.bind(context, 'Achsen', stringMatrix, 0, true)).to['throw'](ErrorCode).that.equals(ErrorCode.VALUE);
                expect(VLOOKUP.resolve.bind(context, 'Achsen', stringMatrix, 100, true)).to['throw'](ErrorCode).that.equals(ErrorCode.REF);
            });

            it('should return the result of the vertical search with wildcards', function () {
                expect(VLOOKUP.resolve.call(context, 'Schra?ben', stringMatrix, 4)).to.equal(8);
                expect(VLOOKUP.resolve.call(context, 'Schra?ben', stringMatrix, 4, false)).to.equal(11);

                expect(VLOOKUP.resolve.call(context, 'G*e', stringMatrix, 4, false)).to.equal(8);
                expect(VLOOKUP.resolve.bind(context, 'G?e', stringMatrix, 4, false)).to['throw'](ErrorCode).that.equals(ErrorCode.NA);
            });
        });

    });

    // ========================================================================
});
