/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/cellref',
    'io.ox/office/spreadsheet/model/formula/sheetref',
    'io.ox/office/spreadsheet/model/formula/resource',
    'io.ox/office/spreadsheet/model/formula/grammarconfig'
], function (SheetUtils, CellRef, SheetRef, FormulaResource, GrammarConfig) {

    'use strict';

    // class GrammarConfig ====================================================

    describe('Spreadsheet class GrammarConfig', function () {

        it('should exist', function () {
            expect(GrammarConfig).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        // convenience shortcuts
        var ErrorCode = SheetUtils.ErrorCode;

        // create a CellRef instance on-the-fly from $1.$1 notation (column/row offsets)
        function cell(text) {
            var m = /^(\$)?(-?\d+)\.(\$)?(-?\d+)$/i.exec(text);
            return new CellRef(parseInt(m[2], 10), parseInt(m[4], 10), !!m[1], !!m[3]);
        }

        // create a SheetRef instance on-the-fly
        function sheet(sheet, abs) {
            return new SheetRef(sheet, !_.isBoolean(abs) || abs);
        }

        var ooxResource = new FormulaResource('ooxml');
        var odfResource = new FormulaResource('odf');

        var ooxOpConfig = null;
        var ooxUiConfig = null;
        var odfOpConfig = null;
        var odfUiConfig = null;

        // maximum column/row index for this test
        var MAXCOL = 255;
        var MAXROW = 65535;

        // simple mock of a document model, used for sheet names, and maximum column/row checks
        var docModel = {
            getSheetName: function (index) {
                return ['Sheet1', 'Sheet2', 'Sheet3', 'Sheet 4'][index] || null;
            },
            isValidAddress: function (address) {
                return (address[0] >= 0) && (address[0] <= MAXCOL) && (address[1] >= 0) && (address[1] <= MAXROW);
            },
            isColRange: function (range) {
                return (range.start[1] === 0) && (range.end[1] === MAXROW);
            },
            isRowRange: function (range) {
                return (range.start[0] === 0) && (range.end[0] === MAXCOL);
            }
        };

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create a formula grammar configuration', function () {
                ooxOpConfig = new GrammarConfig(ooxResource, 'op');
                expect(ooxOpConfig).to.be.an('object');
            });
        });

        // static methods -----------------------------------------------------

        describe('method "create"', function () {
            it('should exist', function () {
                expect(GrammarConfig).itself.to.respondTo('create');
            });
            it('should create a new configuration', function () {
                ooxUiConfig = GrammarConfig.create(ooxResource, 'ui');
                expect(ooxUiConfig).to.be.an['instanceof'](GrammarConfig);
                expect(ooxUiConfig).to.not.equal(ooxOpConfig);
                odfOpConfig = GrammarConfig.create(odfResource, 'op');
                expect(odfOpConfig).to.be.an['instanceof'](GrammarConfig);
                expect(odfOpConfig).to.not.equal(ooxOpConfig);
                expect(odfOpConfig).to.not.equal(ooxUiConfig);
                odfUiConfig = GrammarConfig.create(odfResource, 'ui');
                expect(odfUiConfig).to.be.an['instanceof'](GrammarConfig);
                expect(odfUiConfig).to.not.equal(ooxOpConfig);
                expect(odfUiConfig).to.not.equal(ooxUiConfig);
                expect(odfUiConfig).to.not.equal(odfOpConfig);
            });
            it('should return an existing configuration', function () {
                var config = GrammarConfig.create(ooxResource, 'ui');
                expect(config).to.equal(ooxUiConfig);
            });
        });

        // constants ----------------------------------------------------------

        describe('constant "GRAMMAR"', function () {
            it('should exist', function () {
                expect(ooxOpConfig).to.have.a.property('GRAMMAR', 'op');
                expect(ooxUiConfig).to.have.a.property('GRAMMAR', 'ui');
                expect(odfOpConfig).to.have.a.property('GRAMMAR', 'op');
                expect(odfUiConfig).to.have.a.property('GRAMMAR', 'ui');
            });
        });

        describe('constant "REF_SYNTAX"', function () {
            it('should exist', function () {
                expect(ooxOpConfig).to.have.a.property('REF_SYNTAX', 'ooxml');
                expect(ooxUiConfig).to.have.a.property('REF_SYNTAX', 'ooxml');
                expect(odfOpConfig).to.have.a.property('REF_SYNTAX', 'of');
                expect(odfUiConfig).to.have.a.property('REF_SYNTAX', 'odfui');
            });
        });

        describe('constant "DEC"', function () {
            it('should exist', function () {
                expect(ooxOpConfig).to.have.a.property('DEC', '.');
                expect(ooxUiConfig).to.have.a.property('DEC', ',');
                expect(odfOpConfig).to.have.a.property('DEC', '.');
                expect(odfUiConfig).to.have.a.property('DEC', ',');
            });
        });

        describe('constant "SEP"', function () {
            it('should exist', function () {
                expect(ooxOpConfig).to.have.a.property('SEP', ',');
                expect(ooxUiConfig).to.have.a.property('SEP', ';');
                expect(odfOpConfig).to.have.a.property('SEP', ',');
                expect(odfUiConfig).to.have.a.property('SEP', ';');
            });
        });

        describe('constant "MAT_OPEN"', function () {
            it('should exist', function () {
                expect(ooxOpConfig).to.have.a.property('MAT_OPEN', '{');
                expect(ooxUiConfig).to.have.a.property('MAT_OPEN', '{');
                expect(odfOpConfig).to.have.a.property('MAT_OPEN', '{');
                expect(odfUiConfig).to.have.a.property('MAT_OPEN', '{');
            });
        });

        describe('constant "MAT_ROW"', function () {
            it('should exist', function () {
                expect(ooxOpConfig).to.have.a.property('MAT_ROW', ';');
                expect(ooxUiConfig).to.have.a.property('MAT_ROW', '|');
                expect(odfOpConfig).to.have.a.property('MAT_ROW', '|');
                expect(odfUiConfig).to.have.a.property('MAT_ROW', '|');
            });
        });

        describe('constant "MAT_COL"', function () {
            it('should exist', function () {
                expect(ooxOpConfig).to.have.a.property('MAT_COL', ',');
                expect(ooxUiConfig).to.have.a.property('MAT_COL', ';');
                expect(odfOpConfig).to.have.a.property('MAT_COL', ';');
                expect(odfUiConfig).to.have.a.property('MAT_COL', ';');
            });
        });

        describe('constant "MAT_CLOSE"', function () {
            it('should exist', function () {
                expect(ooxOpConfig).to.have.a.property('MAT_CLOSE', '}');
                expect(ooxUiConfig).to.have.a.property('MAT_CLOSE', '}');
                expect(odfOpConfig).to.have.a.property('MAT_CLOSE', '}');
                expect(odfUiConfig).to.have.a.property('MAT_CLOSE', '}');
            });
        });

        describe('constant "RE"', function () {
            it('should exist', function () {
                expect(ooxOpConfig).to.have.a.property('RE').that.is.an('object');
                expect(ooxUiConfig).to.have.a.property('RE').that.is.an('object');
                expect(odfOpConfig).to.have.a.property('RE').that.is.an('object');
                expect(odfUiConfig).to.have.a.property('RE').that.is.an('object');
            });
            it('should contain regular expressions', function () {
                expect(_.every(ooxOpConfig.RE, _.isRegExp)).to.equal(true);
                expect(_.every(ooxUiConfig.RE, _.isRegExp)).to.equal(true);
                expect(_.every(odfOpConfig.RE, _.isRegExp)).to.equal(true);
                expect(_.every(odfUiConfig.RE, _.isRegExp)).to.equal(true);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "isReservedSymbol"', function () {
            it('should exist', function () {
                expect(ooxOpConfig).to.respondTo('isReservedSymbol');
                expect(ooxUiConfig).to.respondTo('isReservedSymbol');
            });
            it('should recognize boolean literals', function () {
                expect(ooxOpConfig.isReservedSymbol(docModel, 'FALSE')).to.equal(true);
                expect(ooxUiConfig.isReservedSymbol(docModel, 'FALSE')).to.equal(false);
                expect(ooxOpConfig.isReservedSymbol(docModel, 'TRUE')).to.equal(true);
                expect(ooxUiConfig.isReservedSymbol(docModel, 'TRUE')).to.equal(false);
                expect(ooxOpConfig.isReservedSymbol(docModel, 'FALSCH')).to.equal(false);
                expect(ooxUiConfig.isReservedSymbol(docModel, 'FALSCH')).to.equal(true);
                expect(ooxOpConfig.isReservedSymbol(docModel, 'WAHR')).to.equal(false);
                expect(ooxUiConfig.isReservedSymbol(docModel, 'WAHR')).to.equal(true);
            });
            it('should recgonize A1 references', function () {
                expect(ooxOpConfig.isReservedSymbol(docModel, 'A1')).to.equal(true);
                expect(ooxUiConfig.isReservedSymbol(docModel, 'A1')).to.equal(true);
                expect(ooxOpConfig.isReservedSymbol(docModel, 'iv65536')).to.equal(true);
                expect(ooxUiConfig.isReservedSymbol(docModel, 'iv65536')).to.equal(true);
                expect(ooxOpConfig.isReservedSymbol(docModel, 'IW65536')).to.equal(false);
                expect(ooxUiConfig.isReservedSymbol(docModel, 'IW65536')).to.equal(false);
                expect(ooxOpConfig.isReservedSymbol(docModel, 'IV65537')).to.equal(false);
                expect(ooxUiConfig.isReservedSymbol(docModel, 'IV65537')).to.equal(false);
            });
            it('should recgonize R1C1 references', function () {
                expect(ooxOpConfig.isReservedSymbol(docModel, 'R1C1')).to.equal(true);
                expect(ooxUiConfig.isReservedSymbol(docModel, 'R1C1')).to.equal(true);
                expect(ooxOpConfig.isReservedSymbol(docModel, 'R65536C256')).to.equal(true);
                expect(ooxUiConfig.isReservedSymbol(docModel, 'R65536C256')).to.equal(true);
                expect(ooxOpConfig.isReservedSymbol(docModel, 'R65537C256')).to.equal(false);
                expect(ooxUiConfig.isReservedSymbol(docModel, 'R65537C256')).to.equal(false);
                expect(ooxOpConfig.isReservedSymbol(docModel, 'R65536C257')).to.equal(false);
                expect(ooxUiConfig.isReservedSymbol(docModel, 'R65536C257')).to.equal(false);
                expect(ooxOpConfig.isReservedSymbol(docModel, 'R1C')).to.equal(true);
                expect(ooxUiConfig.isReservedSymbol(docModel, 'R1C')).to.equal(true);
                expect(ooxOpConfig.isReservedSymbol(docModel, 'RC1')).to.equal(true);
                expect(ooxUiConfig.isReservedSymbol(docModel, 'RC1')).to.equal(true);
                expect(ooxOpConfig.isReservedSymbol(docModel, 'R1')).to.equal(true);
                expect(ooxUiConfig.isReservedSymbol(docModel, 'R1')).to.equal(true);
                expect(ooxOpConfig.isReservedSymbol(docModel, 'C1')).to.equal(true);
                expect(ooxUiConfig.isReservedSymbol(docModel, 'C1')).to.equal(true);
                expect(ooxOpConfig.isReservedSymbol(docModel, 'R')).to.equal(true);
                expect(ooxUiConfig.isReservedSymbol(docModel, 'R')).to.equal(true);
                expect(ooxOpConfig.isReservedSymbol(docModel, 'C')).to.equal(true);
                expect(ooxUiConfig.isReservedSymbol(docModel, 'C')).to.equal(true);
                expect(ooxOpConfig.isReservedSymbol(docModel, 'RC')).to.equal(true);
                expect(ooxUiConfig.isReservedSymbol(docModel, 'RC')).to.equal(true);
                // localized R1C1 references (not used in ODF)
                expect(ooxOpConfig.isReservedSymbol(docModel, 'Z1S1')).to.equal(false);
                expect(ooxUiConfig.isReservedSymbol(docModel, 'Z1S1')).to.equal(true);
                expect(odfOpConfig.isReservedSymbol(docModel, 'Z1S1')).to.equal(false);
                expect(odfUiConfig.isReservedSymbol(docModel, 'Z1S1')).to.equal(false);
            });
            it('should return false for other strings', function () {
                expect(ooxOpConfig.isReservedSymbol(docModel, 'A')).to.equal(false);
                expect(ooxUiConfig.isReservedSymbol(docModel, 'A')).to.equal(false);
                expect(ooxOpConfig.isReservedSymbol(docModel, '1')).to.equal(false);
                expect(ooxUiConfig.isReservedSymbol(docModel, '1')).to.equal(false);
                expect(ooxOpConfig.isReservedSymbol(docModel, ' A1')).to.equal(false);
                expect(ooxUiConfig.isReservedSymbol(docModel, ' A1')).to.equal(false);
                expect(ooxOpConfig.isReservedSymbol(docModel, 'A 1')).to.equal(false);
                expect(ooxUiConfig.isReservedSymbol(docModel, 'A 1')).to.equal(false);
                expect(ooxOpConfig.isReservedSymbol(docModel, 'R1 C1')).to.equal(false);
                expect(ooxUiConfig.isReservedSymbol(docModel, 'R1 C1')).to.equal(false);
                expect(ooxOpConfig.isReservedSymbol(docModel, 'C1R1')).to.equal(false);
                expect(ooxUiConfig.isReservedSymbol(docModel, 'C1R1')).to.equal(false);
                expect(ooxOpConfig.isReservedSymbol(docModel, 'S1Z1')).to.equal(false);
                expect(ooxUiConfig.isReservedSymbol(docModel, 'S1Z1')).to.equal(false);
                expect(ooxOpConfig.isReservedSymbol(docModel, '')).to.equal(false);
                expect(ooxUiConfig.isReservedSymbol(docModel, '')).to.equal(false);
                expect(ooxOpConfig.isReservedSymbol(docModel, ' TRUE')).to.equal(false);
                expect(ooxUiConfig.isReservedSymbol(docModel, ' WAHR')).to.equal(false);
            });
        });

        describe('method "isSimpleSheetName"', function () {
            it('should exist', function () {
                expect(ooxOpConfig).to.respondTo('isSimpleSheetName');
                expect(ooxUiConfig).to.respondTo('isSimpleSheetName');
            });
            it('should recognize simple sheet names', function () {
                // simple sheet name
                expect(ooxOpConfig.isSimpleSheetName(docModel, 'Sheet1')).to.equal(true);
                expect(ooxUiConfig.isSimpleSheetName(docModel, 'Sheet1')).to.equal(true);
                expect(odfOpConfig.isSimpleSheetName(docModel, 'Sheet1')).to.equal(true);
                expect(odfUiConfig.isSimpleSheetName(docModel, 'Sheet1')).to.equal(true);
                // whitespace
                expect(ooxOpConfig.isSimpleSheetName(docModel, 'Sheet 1')).to.equal(false);
                expect(ooxUiConfig.isSimpleSheetName(docModel, 'Sheet 1')).to.equal(false);
                expect(odfOpConfig.isSimpleSheetName(docModel, 'Sheet 1')).to.equal(false);
                expect(odfUiConfig.isSimpleSheetName(docModel, 'Sheet 1')).to.equal(false);
                // sheet range
                expect(ooxOpConfig.isSimpleSheetName(docModel, 'Sheet1:Sheet2')).to.equal(false);
                expect(ooxUiConfig.isSimpleSheetName(docModel, 'Sheet1:Sheet2')).to.equal(false);
                expect(odfOpConfig.isSimpleSheetName(docModel, 'Sheet1:Sheet2')).to.equal(false);
                expect(odfUiConfig.isSimpleSheetName(docModel, 'Sheet1:Sheet2')).to.equal(false);
                // booleans
                expect(ooxOpConfig.isSimpleSheetName(docModel, 'TRUE')).to.equal(false);
                expect(ooxUiConfig.isSimpleSheetName(docModel, 'TRUE')).to.equal(true);
                expect(ooxOpConfig.isSimpleSheetName(docModel, 'WAHR')).to.equal(true);
                expect(ooxUiConfig.isSimpleSheetName(docModel, 'WAHR')).to.equal(false);
                // A1 notation
                expect(ooxOpConfig.isSimpleSheetName(docModel, 'A1')).to.equal(false);
                expect(ooxUiConfig.isSimpleSheetName(docModel, 'A1')).to.equal(false);
                expect(odfOpConfig.isSimpleSheetName(docModel, 'A1')).to.equal(false);
                expect(odfUiConfig.isSimpleSheetName(docModel, 'A1')).to.equal(false);
                // native R1C1 notation
                expect(ooxOpConfig.isSimpleSheetName(docModel, 'R1C1')).to.equal(false);
                expect(ooxUiConfig.isSimpleSheetName(docModel, 'R1C1')).to.equal(false);
                expect(odfOpConfig.isSimpleSheetName(docModel, 'R1C1')).to.equal(false);
                expect(odfUiConfig.isSimpleSheetName(docModel, 'R1C1')).to.equal(false);
                // localized R1C1 notation
                expect(ooxOpConfig.isSimpleSheetName(docModel, 'Z1S1')).to.equal(true);
                expect(ooxUiConfig.isSimpleSheetName(docModel, 'Z1S1')).to.equal(false);
                expect(odfOpConfig.isSimpleSheetName(docModel, 'Z1S1')).to.equal(true);
                expect(odfUiConfig.isSimpleSheetName(docModel, 'Z1S1')).to.equal(true);
                // external path
                expect(ooxOpConfig.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1')).to.equal(false);
                expect(ooxUiConfig.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1')).to.equal(false);
                expect(odfOpConfig.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1')).to.equal(false);
                expect(odfUiConfig.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1')).to.equal(false);
            });
            it('should recognize simple sheet ranges', function () {
                expect(ooxOpConfig.isSimpleSheetName(docModel, 'Sheet1', { range: true })).to.equal(true);
                expect(ooxUiConfig.isSimpleSheetName(docModel, 'Sheet1', { range: true })).to.equal(true);
                expect(ooxOpConfig.isSimpleSheetName(docModel, 'Sheet 1', { range: true })).to.equal(false);
                expect(ooxUiConfig.isSimpleSheetName(docModel, 'Sheet 1', { range: true })).to.equal(false);
                expect(ooxOpConfig.isSimpleSheetName(docModel, 'Sheet1:Sheet2', { range: true })).to.equal(true);
                expect(ooxUiConfig.isSimpleSheetName(docModel, 'Sheet1:Sheet2', { range: true })).to.equal(true);
                expect(ooxOpConfig.isSimpleSheetName(docModel, 'Sheet1:Sheet 2', { range: true })).to.equal(false);
                expect(ooxUiConfig.isSimpleSheetName(docModel, 'Sheet1:Sheet 2', { range: true })).to.equal(false);
                expect(ooxOpConfig.isSimpleSheetName(docModel, 'Sheet1:TRUE', { range: true })).to.equal(false);
                expect(ooxUiConfig.isSimpleSheetName(docModel, 'Sheet1:TRUE', { range: true })).to.equal(true);
                expect(ooxOpConfig.isSimpleSheetName(docModel, 'Sheet1:WAHR', { range: true })).to.equal(true);
                expect(ooxUiConfig.isSimpleSheetName(docModel, 'Sheet1:WAHR', { range: true })).to.equal(false);
                expect(ooxOpConfig.isSimpleSheetName(docModel, 'Sheet1:A1', { range: true })).to.equal(false);
                expect(ooxUiConfig.isSimpleSheetName(docModel, 'Sheet1:A1', { range: true })).to.equal(false);
                expect(ooxOpConfig.isSimpleSheetName(docModel, 'Sheet1:R1C1', { range: true })).to.equal(false);
                expect(ooxUiConfig.isSimpleSheetName(docModel, 'Sheet1:R1C1', { range: true })).to.equal(false);
                expect(ooxOpConfig.isSimpleSheetName(docModel, 'Sheet1:Z1S1', { range: true })).to.equal(true);
                expect(ooxUiConfig.isSimpleSheetName(docModel, 'Sheet1:Z1S1', { range: true })).to.equal(false);
            });
            it('should recognize simple sheet names with document name', function () {
                expect(ooxOpConfig.isSimpleSheetName(docModel, 'Sheet1', { external: true })).to.equal(true);
                expect(ooxUiConfig.isSimpleSheetName(docModel, 'Sheet1', { external: true })).to.equal(true);
                expect(ooxOpConfig.isSimpleSheetName(docModel, 'Sheet 1', { external: true })).to.equal(false);
                expect(ooxUiConfig.isSimpleSheetName(docModel, 'Sheet 1', { external: true })).to.equal(false);
                expect(ooxOpConfig.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1', { external: true })).to.equal(true);
                expect(ooxUiConfig.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1', { external: true })).to.equal(true);
                expect(ooxOpConfig.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet 1', { external: true })).to.equal(false);
                expect(ooxUiConfig.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet 1', { external: true })).to.equal(false);
                expect(ooxOpConfig.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1:Sheet2', { external: true })).to.equal(false);
                expect(ooxUiConfig.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1:Sheet2', { external: true })).to.equal(false);
                expect(ooxOpConfig.isSimpleSheetName(docModel, '[path\\doc 1.xlsx]Sheet1', { external: true })).to.equal(false);
                expect(ooxUiConfig.isSimpleSheetName(docModel, '[path\\doc 1.xlsx]Sheet1', { external: true })).to.equal(false);
                expect(ooxOpConfig.isSimpleSheetName(docModel, '[path/doc1.xlsx]Sheet1', { external: true })).to.equal(false);
                expect(ooxUiConfig.isSimpleSheetName(docModel, '[path/doc1.xlsx]Sheet1', { external: true })).to.equal(false);
            });
            it('should recognize simple sheet ranges with document name', function () {
                expect(ooxOpConfig.isSimpleSheetName(docModel, 'Sheet1', { range: true, external: true })).to.equal(true);
                expect(ooxUiConfig.isSimpleSheetName(docModel, 'Sheet1', { range: true, external: true })).to.equal(true);
                expect(ooxOpConfig.isSimpleSheetName(docModel, 'Sheet 1', { range: true, external: true })).to.equal(false);
                expect(ooxUiConfig.isSimpleSheetName(docModel, 'Sheet 1', { range: true, external: true })).to.equal(false);
                expect(ooxOpConfig.isSimpleSheetName(docModel, 'Sheet1:Sheet2', { range: true, external: true })).to.equal(true);
                expect(ooxUiConfig.isSimpleSheetName(docModel, 'Sheet1:Sheet2', { range: true, external: true })).to.equal(true);
                expect(ooxOpConfig.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1', { range: true, external: true })).to.equal(true);
                expect(ooxUiConfig.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1', { range: true, external: true })).to.equal(true);
                expect(ooxOpConfig.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet 1', { range: true, external: true })).to.equal(false);
                expect(ooxUiConfig.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet 1', { range: true, external: true })).to.equal(false);
                expect(ooxOpConfig.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1:Sheet2', { range: true, external: true })).to.equal(true);
                expect(ooxUiConfig.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1:Sheet2', { range: true, external: true })).to.equal(true);
                expect(ooxOpConfig.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1:Sheet 2', { range: true, external: true })).to.equal(false);
                expect(ooxUiConfig.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1:Sheet 2', { range: true, external: true })).to.equal(false);
                expect(ooxOpConfig.isSimpleSheetName(docModel, '[path\\doc 1.xlsx]Sheet1:Sheet2', { range: true, external: true })).to.equal(false);
                expect(ooxUiConfig.isSimpleSheetName(docModel, '[path\\doc 1.xlsx]Sheet1:Sheet2', { range: true, external: true })).to.equal(false);
            });
        });

        describe('method "validateNameLabel"', function () {
            it('should exist', function () {
                expect(ooxOpConfig).to.respondTo('validateNameLabel');
                expect(ooxUiConfig).to.respondTo('validateNameLabel');
            });
            it('should reject invalid strings', function () {
                expect(ooxOpConfig.validateNameLabel(docModel, '')).to.equal('name:empty');
                expect(ooxUiConfig.validateNameLabel(docModel, '')).to.equal('name:empty');
                expect(odfOpConfig.validateNameLabel(docModel, '')).to.equal('name:empty');
                expect(odfUiConfig.validateNameLabel(docModel, '')).to.equal('name:empty');
                expect(ooxOpConfig.validateNameLabel(docModel, ' name')).to.equal('name:invalid');
                expect(ooxUiConfig.validateNameLabel(docModel, ' name')).to.equal('name:invalid');
                expect(odfOpConfig.validateNameLabel(docModel, ' name')).to.equal('name:invalid');
                expect(odfUiConfig.validateNameLabel(docModel, ' name')).to.equal('name:invalid');
                expect(ooxOpConfig.validateNameLabel(docModel, '1name')).to.equal('name:invalid');
                expect(ooxUiConfig.validateNameLabel(docModel, '1name')).to.equal('name:invalid');
                expect(odfOpConfig.validateNameLabel(docModel, '1name')).to.equal('name:invalid');
                expect(odfUiConfig.validateNameLabel(docModel, '1name')).to.equal('name:invalid');
                expect(ooxOpConfig.validateNameLabel(docModel, '/name')).to.equal('name:invalid');
                expect(ooxUiConfig.validateNameLabel(docModel, '/name')).to.equal('name:invalid');
                expect(odfOpConfig.validateNameLabel(docModel, '/name')).to.equal('name:invalid');
                expect(odfUiConfig.validateNameLabel(docModel, '/name')).to.equal('name:invalid');
                expect(ooxOpConfig.validateNameLabel(docModel, '?name')).to.equal('name:invalid');
                expect(ooxUiConfig.validateNameLabel(docModel, '?name')).to.equal('name:invalid');
                expect(odfOpConfig.validateNameLabel(docModel, '?name')).to.equal('name:invalid');
                expect(odfUiConfig.validateNameLabel(docModel, '?name')).to.equal('name:invalid');
                expect(ooxOpConfig.validateNameLabel(docModel, '.name')).to.equal('name:invalid');
                expect(ooxUiConfig.validateNameLabel(docModel, '.name')).to.equal('name:invalid');
                expect(odfOpConfig.validateNameLabel(docModel, '.name')).to.equal('name:invalid');
                expect(odfUiConfig.validateNameLabel(docModel, '.name')).to.equal('name:invalid');
            });
            it('should handle boolean literals', function () {
                expect(ooxOpConfig.validateNameLabel(docModel, 'true')).to.equal('name:invalid');
                expect(ooxUiConfig.validateNameLabel(docModel, 'true')).to.equal('');
                expect(odfOpConfig.validateNameLabel(docModel, 'true')).to.equal('');
                expect(odfUiConfig.validateNameLabel(docModel, 'true')).to.equal('');
                expect(ooxOpConfig.validateNameLabel(docModel, 'wahr')).to.equal('');
                expect(ooxUiConfig.validateNameLabel(docModel, 'wahr')).to.equal('name:invalid');
                expect(odfOpConfig.validateNameLabel(docModel, 'wahr')).to.equal('');
                expect(odfUiConfig.validateNameLabel(docModel, 'wahr')).to.equal('');
            });
            it('should reject A1 cell references', function () {
                // first cell in sheet
                expect(ooxOpConfig.validateNameLabel(docModel, 'a1')).to.equal('name:address');
                expect(ooxUiConfig.validateNameLabel(docModel, 'a1')).to.equal('name:address');
                expect(odfOpConfig.validateNameLabel(docModel, 'a1')).to.equal('name:address');
                expect(odfUiConfig.validateNameLabel(docModel, 'a1')).to.equal('name:address');
                // last cell in sheet
                expect(ooxOpConfig.validateNameLabel(docModel, 'iv65536')).to.equal('name:address');
                expect(ooxUiConfig.validateNameLabel(docModel, 'iv65536')).to.equal('name:address');
                expect(odfOpConfig.validateNameLabel(docModel, 'iv65536')).to.equal('name:address');
                expect(odfUiConfig.validateNameLabel(docModel, 'iv65536')).to.equal('name:address');
                // outside of sheet range
                expect(ooxOpConfig.validateNameLabel(docModel, 'iw65536')).to.equal('');
                expect(ooxUiConfig.validateNameLabel(docModel, 'iw65536')).to.equal('');
                expect(odfOpConfig.validateNameLabel(docModel, 'iw65536')).to.equal('');
                expect(odfUiConfig.validateNameLabel(docModel, 'iw65536')).to.equal('');
            });
            it('should reject R1C1 cell references', function () {
                // native absolute R1C1
                expect(ooxOpConfig.validateNameLabel(docModel, 'r1c1')).to.equal('name:address');
                expect(ooxUiConfig.validateNameLabel(docModel, 'r1c1')).to.equal('name:address');
                expect(odfOpConfig.validateNameLabel(docModel, 'r1c1')).to.equal('name:address');
                expect(odfUiConfig.validateNameLabel(docModel, 'r1c1')).to.equal('name:address');
                // native relative R1C1
                expect(ooxOpConfig.validateNameLabel(docModel, 'rc')).to.equal('name:address');
                expect(ooxUiConfig.validateNameLabel(docModel, 'rc')).to.equal('name:address');
                expect(odfOpConfig.validateNameLabel(docModel, 'rc')).to.equal('name:address');
                expect(odfUiConfig.validateNameLabel(docModel, 'rc')).to.equal('name:address');
                // localized absolute R1C1
                expect(ooxOpConfig.validateNameLabel(docModel, 'z1s1')).to.equal('');
                expect(ooxUiConfig.validateNameLabel(docModel, 'z1s1')).to.equal('name:address');
                expect(odfOpConfig.validateNameLabel(docModel, 'z1s1')).to.equal('');
                expect(odfUiConfig.validateNameLabel(docModel, 'z1s1')).to.equal('');
                // localized relative R1C1
                expect(ooxOpConfig.validateNameLabel(docModel, 'zs')).to.equal('');
                expect(ooxUiConfig.validateNameLabel(docModel, 'zs')).to.equal('name:address');
                expect(odfOpConfig.validateNameLabel(docModel, 'zs')).to.equal('');
                expect(odfUiConfig.validateNameLabel(docModel, 'zs')).to.equal('');
            });
            it('should accept valid names', function () {
                expect(ooxOpConfig.validateNameLabel(docModel, 'my_name')).to.equal('');
                expect(ooxUiConfig.validateNameLabel(docModel, 'my_name')).to.equal('');
                expect(odfOpConfig.validateNameLabel(docModel, 'my_name')).to.equal('');
                expect(odfUiConfig.validateNameLabel(docModel, 'my_name')).to.equal('');
                expect(ooxOpConfig.validateNameLabel(docModel, 'true1')).to.equal('');
                expect(ooxUiConfig.validateNameLabel(docModel, 'wahr1')).to.equal('');
                expect(odfOpConfig.validateNameLabel(docModel, 'true1')).to.equal('');
                expect(odfUiConfig.validateNameLabel(docModel, 'wahr1')).to.equal('');
                expect(ooxOpConfig.validateNameLabel(docModel, 'c1r1')).to.equal('');
                expect(ooxUiConfig.validateNameLabel(docModel, 'c1r1')).to.equal('');
                expect(odfOpConfig.validateNameLabel(docModel, 'c1r1')).to.equal('');
                expect(odfUiConfig.validateNameLabel(docModel, 'c1r1')).to.equal('');
            });
            it('should handle backslash and question mark', function () {
                expect(ooxOpConfig.validateNameLabel(docModel, '\\name?name.name\\name')).to.equal('');
                expect(ooxUiConfig.validateNameLabel(docModel, '\\name?name.name\\name')).to.equal('');
                expect(odfOpConfig.validateNameLabel(docModel, '\\name')).to.equal('name:invalid');
                expect(odfUiConfig.validateNameLabel(docModel, '\\name')).to.equal('name:invalid');
                expect(odfOpConfig.validateNameLabel(docModel, 'name?a')).to.equal('');
                expect(odfUiConfig.validateNameLabel(docModel, 'name?a')).to.equal('');
                expect(odfOpConfig.validateNameLabel(docModel, 'name.a')).to.equal('name:invalid');
                expect(odfUiConfig.validateNameLabel(docModel, 'name.a')).to.equal('name:invalid');
            });
        });

        describe('method "formatScalar"', function () {
            it('should exist', function () {
                expect(ooxOpConfig).to.respondTo('formatScalar');
                expect(ooxUiConfig).to.respondTo('formatScalar');
            });
            it('should convert values to native text', function () {
                expect(ooxOpConfig.formatScalar(42)).to.equal('42');
                expect(ooxOpConfig.formatScalar(-12.5)).to.equal('-12.5');
                expect(ooxOpConfig.formatScalar(1e300)).to.equal('1E+300');
                expect(ooxOpConfig.formatScalar(1e-300)).to.equal('1E-300');
                expect(ooxOpConfig.formatScalar(1e-310)).to.equal('0');
                expect(ooxOpConfig.formatScalar(Number.POSITIVE_INFINITY)).to.equal('#NUM!');
                expect(ooxOpConfig.formatScalar(Number.NaN)).to.equal('#NUM!');
                expect(ooxOpConfig.formatScalar('abc')).to.equal('"abc"');
                expect(ooxOpConfig.formatScalar('a"b"c')).to.equal('"a""b""c"');
                expect(ooxOpConfig.formatScalar('')).to.equal('""');
                expect(ooxOpConfig.formatScalar(false)).to.equal('FALSE');
                expect(ooxOpConfig.formatScalar(true)).to.equal('TRUE');
                expect(ooxOpConfig.formatScalar(ErrorCode.NULL)).to.equal('#NULL!');
                expect(ooxOpConfig.formatScalar(ErrorCode.DIV0)).to.equal('#DIV/0!');
                expect(ooxOpConfig.formatScalar(ErrorCode.VALUE)).to.equal('#VALUE!');
                expect(ooxOpConfig.formatScalar(ErrorCode.REF)).to.equal('#REF!');
                expect(ooxOpConfig.formatScalar(ErrorCode.NAME)).to.equal('#NAME?');
                expect(ooxOpConfig.formatScalar(ErrorCode.NUM)).to.equal('#NUM!');
                expect(ooxOpConfig.formatScalar(ErrorCode.NA)).to.equal('#N/A');
                expect(ooxOpConfig.formatScalar(ErrorCode.DATA)).to.equal('#GETTING_DATA');
                expect(ooxOpConfig.formatScalar(null)).to.equal('');
            });
            it('should convert values to localized text', function () {
                expect(ooxUiConfig.formatScalar(42)).to.equal('42');
                expect(ooxUiConfig.formatScalar(-12.5)).to.equal('-12,5');
                expect(ooxUiConfig.formatScalar(1e300)).to.equal('1E+300');
                expect(ooxUiConfig.formatScalar(1e-300)).to.equal('1E-300');
                expect(ooxUiConfig.formatScalar(1e-310)).to.equal('0');
                expect(ooxUiConfig.formatScalar(Number.POSITIVE_INFINITY)).to.equal('#ZAHL!');
                expect(ooxUiConfig.formatScalar(Number.NaN)).to.equal('#ZAHL!');
                expect(ooxUiConfig.formatScalar('abc')).to.equal('"abc"');
                expect(ooxUiConfig.formatScalar('a"b"c')).to.equal('"a""b""c"');
                expect(ooxUiConfig.formatScalar('')).to.equal('""');
                expect(ooxUiConfig.formatScalar(ErrorCode.NULL)).to.equal('#NULL!');
                expect(ooxUiConfig.formatScalar(ErrorCode.DIV0)).to.equal('#DIV/0!');
                expect(ooxUiConfig.formatScalar(ErrorCode.VALUE)).to.equal('#WERT!');
                expect(ooxUiConfig.formatScalar(ErrorCode.REF)).to.equal('#BEZUG!');
                expect(ooxUiConfig.formatScalar(ErrorCode.NAME)).to.equal('#NAME?');
                expect(ooxUiConfig.formatScalar(ErrorCode.NUM)).to.equal('#ZAHL!');
                expect(ooxUiConfig.formatScalar(ErrorCode.NA)).to.equal('#NV');
                expect(ooxUiConfig.formatScalar(ErrorCode.DATA)).to.equal('#GETTING_DATA');
                expect(ooxUiConfig.formatScalar(null)).to.equal('');
            });
        });

        describe('method "formatReference"', function () {
            it('should exist', function () {
                expect(ooxOpConfig).to.respondTo('formatReference');
                expect(ooxUiConfig).to.respondTo('formatReference');
            });
            var c1 = cell('1.$2'), c2 = cell('$3.4'), s1 = sheet(1), s2 = sheet(3), s3 = sheet(-1);
            it('should return the string representation of a cell address', function () {
                expect(ooxOpConfig.formatReference(docModel, null, null, c1, null)).to.equal('B$3');
                expect(ooxUiConfig.formatReference(docModel, null, null, c1, null)).to.equal('B$3');
                expect(odfOpConfig.formatReference(docModel, null, null, c1, null)).to.equal('[.B$3]');
                expect(odfUiConfig.formatReference(docModel, null, null, c1, null)).to.equal('B$3');
                expect(ooxOpConfig.formatReference(docModel, null, null, c1, null, true)).to.equal('#N/A');
                expect(ooxUiConfig.formatReference(docModel, null, null, c1, null, true)).to.equal('Z3S[1]');
                expect(odfOpConfig.formatReference(docModel, null, null, c1, null, true)).to.equal('#N/A');
                expect(odfUiConfig.formatReference(docModel, null, null, c1, null, true)).to.equal('R3C[1]');
            });
            it('should return the string representation of a cell range address', function () {
                expect(ooxOpConfig.formatReference(docModel, null, null, c1, c2)).to.equal('B$3:$D5');
                expect(ooxUiConfig.formatReference(docModel, null, null, c1, c2)).to.equal('B$3:$D5');
                expect(odfOpConfig.formatReference(docModel, null, null, c1, c2)).to.equal('[.B$3:.$D5]');
                expect(odfUiConfig.formatReference(docModel, null, null, c1, c2)).to.equal('B$3:$D5');
                expect(ooxOpConfig.formatReference(docModel, null, null, c1, c2, true)).to.equal('#N/A');
                expect(ooxUiConfig.formatReference(docModel, null, null, c1, c2, true)).to.equal('Z3S[1]:Z[4]S4');
                expect(odfOpConfig.formatReference(docModel, null, null, c1, c2, true)).to.equal('#N/A');
                expect(odfUiConfig.formatReference(docModel, null, null, c1, c2, true)).to.equal('R3C[1]:R[4]C4');
            });
            it('should return the string representation of a reference error', function () {
                expect(ooxOpConfig.formatReference(docModel, null, null, null, null)).to.equal('#REF!');
                expect(ooxUiConfig.formatReference(docModel, null, null, null, null)).to.equal('#BEZUG!');
                expect(odfOpConfig.formatReference(docModel, null, null, null, null)).to.equal('[.#REF!#REF!]');
                expect(odfUiConfig.formatReference(docModel, null, null, null, null)).to.equal('#BEZUG!');
                expect(ooxOpConfig.formatReference(docModel, null, null, null, null, true)).to.equal('#N/A');
                expect(ooxUiConfig.formatReference(docModel, null, null, null, null, true)).to.equal('#BEZUG!');
                expect(odfOpConfig.formatReference(docModel, null, null, null, null, true)).to.equal('#N/A');
                expect(odfUiConfig.formatReference(docModel, null, null, null, null, true)).to.equal('#BEZUG!');
            });
            it('should return the string representation of a cell address with sheet', function () {
                expect(ooxOpConfig.formatReference(docModel, s1, null, c1, null)).to.equal('Sheet2!B$3');
                expect(ooxUiConfig.formatReference(docModel, s1, null, c1, null)).to.equal('Sheet2!B$3');
                expect(odfOpConfig.formatReference(docModel, s1, null, c1, null)).to.equal('[$Sheet2.B$3]');
                expect(odfUiConfig.formatReference(docModel, s1, null, c1, null)).to.equal('$Sheet2!B$3');
                expect(ooxOpConfig.formatReference(docModel, s1, null, c1, null, true)).to.equal('#N/A');
                expect(ooxUiConfig.formatReference(docModel, s1, null, c1, null, true)).to.equal('Sheet2!Z3S[1]');
                expect(odfOpConfig.formatReference(docModel, s1, null, c1, null, true)).to.equal('#N/A');
                expect(odfUiConfig.formatReference(docModel, s1, null, c1, null, true)).to.equal('$Sheet2!R3C[1]');
            });
            it('should return the string representation of a cell range address with sheet', function () {
                expect(ooxOpConfig.formatReference(docModel, s1, null, c1, c2)).to.equal('Sheet2!B$3:$D5');
                expect(ooxUiConfig.formatReference(docModel, s1, null, c1, c2)).to.equal('Sheet2!B$3:$D5');
                expect(odfOpConfig.formatReference(docModel, s1, null, c1, c2)).to.equal('[$Sheet2.B$3:.$D5]');
                expect(odfUiConfig.formatReference(docModel, s1, null, c1, c2)).to.equal('$Sheet2!B$3:$D5');
                expect(ooxOpConfig.formatReference(docModel, s1, null, c1, c2, true)).to.equal('#N/A');
                expect(ooxUiConfig.formatReference(docModel, s1, null, c1, c2, true)).to.equal('Sheet2!Z3S[1]:Z[4]S4');
                expect(odfOpConfig.formatReference(docModel, s1, null, c1, c2, true)).to.equal('#N/A');
                expect(odfUiConfig.formatReference(docModel, s1, null, c1, c2, true)).to.equal('$Sheet2!R3C[1]:R[4]C4');
            });
            it('should return the string representation of a reference error with sheet', function () {
                expect(ooxOpConfig.formatReference(docModel, s1, null, null, null)).to.equal('Sheet2!#REF!');
                expect(ooxUiConfig.formatReference(docModel, s1, null, null, null)).to.equal('Sheet2!#BEZUG!');
                expect(odfOpConfig.formatReference(docModel, s1, null, null, null)).to.equal('[$Sheet2.#REF!#REF!]');
                expect(odfUiConfig.formatReference(docModel, s1, null, null, null)).to.equal('$Sheet2!#BEZUG!');
                expect(ooxOpConfig.formatReference(docModel, s1, null, null, null, true)).to.equal('#N/A');
                expect(ooxUiConfig.formatReference(docModel, s1, null, null, null, true)).to.equal('Sheet2!#BEZUG!');
                expect(odfOpConfig.formatReference(docModel, s1, null, null, null, true)).to.equal('#N/A');
                expect(odfUiConfig.formatReference(docModel, s1, null, null, null, true)).to.equal('$Sheet2!#BEZUG!');
            });
            it('should return the string representation of a reference with sheet range', function () {
                // cell address
                expect(ooxOpConfig.formatReference(docModel, s1, s2, c1, null)).to.equal('\'Sheet2:Sheet 4\'!B$3');
                expect(ooxUiConfig.formatReference(docModel, s1, s2, c1, null)).to.equal('\'Sheet2:Sheet 4\'!B$3');
                expect(odfOpConfig.formatReference(docModel, s1, s2, c1, null)).to.equal('[$Sheet2.B$3:$\'Sheet 4\'.B$3]');
                expect(odfUiConfig.formatReference(docModel, s1, s2, c1, null)).to.equal('$Sheet2:$\'Sheet 4\'!B$3');
                // cell range address
                expect(ooxOpConfig.formatReference(docModel, s1, s2, c1, c2)).to.equal('\'Sheet2:Sheet 4\'!B$3:$D5');
                expect(ooxUiConfig.formatReference(docModel, s1, s2, c1, c2)).to.equal('\'Sheet2:Sheet 4\'!B$3:$D5');
                expect(odfOpConfig.formatReference(docModel, s1, s2, c1, c2)).to.equal('[$Sheet2.B$3:$\'Sheet 4\'.$D5]');
                expect(odfUiConfig.formatReference(docModel, s1, s2, c1, c2)).to.equal('$Sheet2:$\'Sheet 4\'!B$3:$D5');
                // reference error
                expect(ooxOpConfig.formatReference(docModel, s1, s2, null, null)).to.equal('\'Sheet2:Sheet 4\'!#REF!');
                expect(ooxUiConfig.formatReference(docModel, s1, s2, null, null)).to.equal('\'Sheet2:Sheet 4\'!#BEZUG!');
                expect(odfOpConfig.formatReference(docModel, s1, s2, null, null)).to.equal('[$Sheet2.#REF!#REF!:$\'Sheet 4\'.#REF!#REF!]');
                expect(odfUiConfig.formatReference(docModel, s1, s2, null, null)).to.equal('$Sheet2:$\'Sheet 4\'!#BEZUG!');
            });
            it('should return the string representation of a sheet reference error', function () {
                // cell address
                expect(ooxOpConfig.formatReference(docModel, s3, null, c1, null)).to.equal('#REF!!B$3');
                expect(ooxUiConfig.formatReference(docModel, s3, null, c1, null)).to.equal('#BEZUG!!B$3');
                expect(odfOpConfig.formatReference(docModel, s3, null, c1, null)).to.equal('[$#REF!.B$3]');
                expect(odfUiConfig.formatReference(docModel, s3, null, c1, null)).to.equal('#BEZUG!!B$3');
                // cell range address
                expect(ooxOpConfig.formatReference(docModel, s3, null, c1, c2)).to.equal('#REF!!B$3:$D5');
                expect(ooxUiConfig.formatReference(docModel, s3, null, c1, c2)).to.equal('#BEZUG!!B$3:$D5');
                expect(odfOpConfig.formatReference(docModel, s3, null, c1, c2)).to.equal('[$#REF!.B$3:.$D5]');
                expect(odfUiConfig.formatReference(docModel, s3, null, c1, c2)).to.equal('#BEZUG!!B$3:$D5');
                // reference error
                expect(ooxOpConfig.formatReference(docModel, s3, null, null, null)).to.equal('#REF!!#REF!');
                expect(ooxUiConfig.formatReference(docModel, s3, null, null, null)).to.equal('#BEZUG!!#BEZUG!');
                expect(odfOpConfig.formatReference(docModel, s3, null, null, null)).to.equal('[$#REF!.#REF!#REF!]');
                expect(odfUiConfig.formatReference(docModel, s3, null, null, null)).to.equal('#BEZUG!!#BEZUG!');
            });
            it('should return string representation of a column range', function () {
                // relative->absolute
                expect(ooxOpConfig.formatReference(docModel, null, null, cell('1.$0'), cell('$3.$65535'))).to.equal('B:$D');
                expect(ooxUiConfig.formatReference(docModel, null, null, cell('1.$0'), cell('$3.$65535'))).to.equal('B:$D');
                expect(odfOpConfig.formatReference(docModel, null, null, cell('1.$0'), cell('$3.$65535'))).to.equal('[.B$1:.$D$65536]');
                expect(odfUiConfig.formatReference(docModel, null, null, cell('1.$0'), cell('$3.$65535'))).to.equal('B:$D');
                expect(ooxOpConfig.formatReference(docModel, null, null, cell('1.$0'), cell('$3.$65535'), true)).to.equal('#N/A');
                expect(ooxUiConfig.formatReference(docModel, null, null, cell('1.$0'), cell('$3.$65535'), true)).to.equal('S[1]:S4');
                expect(odfOpConfig.formatReference(docModel, null, null, cell('1.$0'), cell('$3.$65535'), true)).to.equal('#N/A');
                expect(odfUiConfig.formatReference(docModel, null, null, cell('1.$0'), cell('$3.$65535'), true)).to.equal('C[1]:C4');
                // absolute->relative
                expect(ooxOpConfig.formatReference(docModel, null, null, cell('$1.$0'), cell('3.$65535'))).to.equal('$B:D');
                expect(ooxUiConfig.formatReference(docModel, null, null, cell('$1.$0'), cell('3.$65535'))).to.equal('$B:D');
                expect(odfOpConfig.formatReference(docModel, null, null, cell('$1.$0'), cell('3.$65535'))).to.equal('[.$B$1:.D$65536]');
                expect(odfUiConfig.formatReference(docModel, null, null, cell('$1.$0'), cell('3.$65535'))).to.equal('$B:D');
                expect(ooxOpConfig.formatReference(docModel, null, null, cell('$1.$0'), cell('3.$65535'), true)).to.equal('#N/A');
                expect(ooxUiConfig.formatReference(docModel, null, null, cell('$1.$0'), cell('3.$65535'), true)).to.equal('S2:S[3]');
                expect(odfOpConfig.formatReference(docModel, null, null, cell('$1.$0'), cell('3.$65535'), true)).to.equal('#N/A');
                expect(odfUiConfig.formatReference(docModel, null, null, cell('$1.$0'), cell('3.$65535'), true)).to.equal('C2:C[3]');
            });
            it('should return string representation of a single column', function () {
                // absolute
                expect(ooxOpConfig.formatReference(docModel, null, null, cell('$1.$0'), cell('$1.$65535'))).to.equal('$B:$B');
                expect(ooxUiConfig.formatReference(docModel, null, null, cell('$1.$0'), cell('$1.$65535'))).to.equal('$B:$B');
                expect(odfOpConfig.formatReference(docModel, null, null, cell('$1.$0'), cell('$1.$65535'))).to.equal('[.$B$1:.$B$65536]');
                expect(odfUiConfig.formatReference(docModel, null, null, cell('$1.$0'), cell('$1.$65535'))).to.equal('$B:$B');
                expect(ooxOpConfig.formatReference(docModel, null, null, cell('$1.$0'), cell('$1.$65535'), true)).to.equal('#N/A');
                expect(ooxUiConfig.formatReference(docModel, null, null, cell('$1.$0'), cell('$1.$65535'), true)).to.equal('S2');
                expect(odfOpConfig.formatReference(docModel, null, null, cell('$1.$0'), cell('$1.$65535'), true)).to.equal('#N/A');
                expect(odfUiConfig.formatReference(docModel, null, null, cell('$1.$0'), cell('$1.$65535'), true)).to.equal('C2');
                // relative
                expect(ooxOpConfig.formatReference(docModel, null, null, cell('1.$0'), cell('1.$65535'))).to.equal('B:B');
                expect(ooxUiConfig.formatReference(docModel, null, null, cell('1.$0'), cell('1.$65535'))).to.equal('B:B');
                expect(odfOpConfig.formatReference(docModel, null, null, cell('1.$0'), cell('1.$65535'))).to.equal('[.B$1:.B$65536]');
                expect(odfUiConfig.formatReference(docModel, null, null, cell('1.$0'), cell('1.$65535'))).to.equal('B:B');
                expect(ooxOpConfig.formatReference(docModel, null, null, cell('1.$0'), cell('1.$65535'), true)).to.equal('#N/A');
                expect(ooxUiConfig.formatReference(docModel, null, null, cell('1.$0'), cell('1.$65535'), true)).to.equal('S[1]');
                expect(odfOpConfig.formatReference(docModel, null, null, cell('1.$0'), cell('1.$65535'), true)).to.equal('#N/A');
                expect(odfUiConfig.formatReference(docModel, null, null, cell('1.$0'), cell('1.$65535'), true)).to.equal('C[1]');
            });
            it('should return string representation of a column range with sheet', function () {
                // single sheet
                expect(ooxOpConfig.formatReference(docModel, sheet(1), null, cell('$1.$0'), cell('1.$65535'))).to.equal('Sheet2!$B:B');
                expect(ooxUiConfig.formatReference(docModel, sheet(1), null, cell('$1.$0'), cell('1.$65535'))).to.equal('Sheet2!$B:B');
                expect(odfOpConfig.formatReference(docModel, sheet(1), null, cell('$1.$0'), cell('1.$65535'))).to.equal('[$Sheet2.$B$1:.B$65536]');
                expect(odfUiConfig.formatReference(docModel, sheet(1), null, cell('$1.$0'), cell('1.$65535'))).to.equal('$Sheet2!$B:B');
                expect(ooxOpConfig.formatReference(docModel, sheet(1), null, cell('$1.$0'), cell('1.$65535'), true)).to.equal('#N/A');
                expect(ooxUiConfig.formatReference(docModel, sheet(1), null, cell('$1.$0'), cell('1.$65535'), true)).to.equal('Sheet2!S2:S[1]');
                expect(odfOpConfig.formatReference(docModel, sheet(1), null, cell('$1.$0'), cell('1.$65535'), true)).to.equal('#N/A');
                expect(odfUiConfig.formatReference(docModel, sheet(1), null, cell('$1.$0'), cell('1.$65535'), true)).to.equal('$Sheet2!C2:C[1]');
                // sheet range
                expect(ooxOpConfig.formatReference(docModel, sheet(1), sheet(3), cell('$1.$0'), cell('1.$65535'))).to.equal('\'Sheet2:Sheet 4\'!$B:B');
                expect(ooxUiConfig.formatReference(docModel, sheet(1), sheet(3), cell('$1.$0'), cell('1.$65535'))).to.equal('\'Sheet2:Sheet 4\'!$B:B');
                expect(odfOpConfig.formatReference(docModel, sheet(1), sheet(3), cell('$1.$0'), cell('1.$65535'))).to.equal('[$Sheet2.$B$1:$\'Sheet 4\'.B$65536]');
                expect(odfUiConfig.formatReference(docModel, sheet(1), sheet(3), cell('$1.$0'), cell('1.$65535'))).to.equal('$Sheet2:$\'Sheet 4\'!$B:B');
            });
            it('should not return string representation of a column range', function () {
                // missing absolute flags
                expect(ooxOpConfig.formatReference(docModel, null, null, cell('$1.$0'), cell('$3.65535'))).to.equal('$B$1:$D65536');
                expect(ooxUiConfig.formatReference(docModel, null, null, cell('$1.$0'), cell('$3.65535'))).to.equal('$B$1:$D65536');
                expect(odfOpConfig.formatReference(docModel, null, null, cell('$1.$0'), cell('$3.65535'))).to.equal('[.$B$1:.$D65536]');
                expect(odfUiConfig.formatReference(docModel, null, null, cell('$1.$0'), cell('$3.65535'))).to.equal('$B$1:$D65536');
                expect(ooxOpConfig.formatReference(docModel, null, null, cell('$1.0'), cell('$3.$65535'))).to.equal('$B1:$D$65536');
                expect(ooxUiConfig.formatReference(docModel, null, null, cell('$1.0'), cell('$3.$65535'))).to.equal('$B1:$D$65536');
                expect(odfOpConfig.formatReference(docModel, null, null, cell('$1.0'), cell('$3.$65535'))).to.equal('[.$B1:.$D$65536]');
                expect(odfUiConfig.formatReference(docModel, null, null, cell('$1.0'), cell('$3.$65535'))).to.equal('$B1:$D$65536');
                // wrong indexes
                expect(ooxOpConfig.formatReference(docModel, null, null, cell('$1.$1'), cell('$3.$65535'))).to.equal('$B$2:$D$65536');
                expect(ooxUiConfig.formatReference(docModel, null, null, cell('$1.$1'), cell('$3.$65535'))).to.equal('$B$2:$D$65536');
                expect(odfOpConfig.formatReference(docModel, null, null, cell('$1.$1'), cell('$3.$65535'))).to.equal('[.$B$2:.$D$65536]');
                expect(odfUiConfig.formatReference(docModel, null, null, cell('$1.$1'), cell('$3.$65535'))).to.equal('$B$2:$D$65536');
                expect(ooxOpConfig.formatReference(docModel, null, null, cell('$1.$0'), cell('$3.$65534'))).to.equal('$B$1:$D$65535');
                expect(ooxUiConfig.formatReference(docModel, null, null, cell('$1.$0'), cell('$3.$65534'))).to.equal('$B$1:$D$65535');
                expect(odfOpConfig.formatReference(docModel, null, null, cell('$1.$0'), cell('$3.$65534'))).to.equal('[.$B$1:.$D$65535]');
                expect(odfUiConfig.formatReference(docModel, null, null, cell('$1.$0'), cell('$3.$65534'))).to.equal('$B$1:$D$65535');
            });
            it('should return string representation of a row range', function () {
                // relative->absolute
                expect(ooxOpConfig.formatReference(docModel, null, null, cell('$0.1'), cell('$255.$3'))).to.equal('2:$4');
                expect(ooxUiConfig.formatReference(docModel, null, null, cell('$0.1'), cell('$255.$3'))).to.equal('2:$4');
                expect(odfOpConfig.formatReference(docModel, null, null, cell('$0.1'), cell('$255.$3'))).to.equal('[.$A2:.$IV$4]');
                expect(odfUiConfig.formatReference(docModel, null, null, cell('$0.1'), cell('$255.$3'))).to.equal('2:$4');
                expect(ooxOpConfig.formatReference(docModel, null, null, cell('$0.1'), cell('$255.$3'), true)).to.equal('#N/A');
                expect(ooxUiConfig.formatReference(docModel, null, null, cell('$0.1'), cell('$255.$3'), true)).to.equal('Z[1]:Z4');
                expect(odfOpConfig.formatReference(docModel, null, null, cell('$0.1'), cell('$255.$3'), true)).to.equal('#N/A');
                expect(odfUiConfig.formatReference(docModel, null, null, cell('$0.1'), cell('$255.$3'), true)).to.equal('R[1]:R4');
                // absolute->relative
                expect(ooxOpConfig.formatReference(docModel, null, null, cell('$0.$1'), cell('$255.3'))).to.equal('$2:4');
                expect(ooxUiConfig.formatReference(docModel, null, null, cell('$0.$1'), cell('$255.3'))).to.equal('$2:4');
                expect(odfOpConfig.formatReference(docModel, null, null, cell('$0.$1'), cell('$255.3'))).to.equal('[.$A$2:.$IV4]');
                expect(odfUiConfig.formatReference(docModel, null, null, cell('$0.$1'), cell('$255.3'))).to.equal('$2:4');
                expect(ooxOpConfig.formatReference(docModel, null, null, cell('$0.$1'), cell('$255.3'), true)).to.equal('#N/A');
                expect(ooxUiConfig.formatReference(docModel, null, null, cell('$0.$1'), cell('$255.3'), true)).to.equal('Z2:Z[3]');
                expect(odfOpConfig.formatReference(docModel, null, null, cell('$0.$1'), cell('$255.3'), true)).to.equal('#N/A');
                expect(odfUiConfig.formatReference(docModel, null, null, cell('$0.$1'), cell('$255.3'), true)).to.equal('R2:R[3]');
            });
            it('should return string representation of a single row', function () {
                // absolute
                expect(ooxOpConfig.formatReference(docModel, null, null, cell('$0.$1'), cell('$255.$1'))).to.equal('$2:$2');
                expect(ooxUiConfig.formatReference(docModel, null, null, cell('$0.$1'), cell('$255.$1'))).to.equal('$2:$2');
                expect(odfOpConfig.formatReference(docModel, null, null, cell('$0.$1'), cell('$255.$1'))).to.equal('[.$A$2:.$IV$2]');
                expect(odfUiConfig.formatReference(docModel, null, null, cell('$0.$1'), cell('$255.$1'))).to.equal('$2:$2');
                expect(ooxOpConfig.formatReference(docModel, null, null, cell('$0.$1'), cell('$255.$1'), true)).to.equal('#N/A');
                expect(ooxUiConfig.formatReference(docModel, null, null, cell('$0.$1'), cell('$255.$1'), true)).to.equal('Z2');
                expect(odfOpConfig.formatReference(docModel, null, null, cell('$0.$1'), cell('$255.$1'), true)).to.equal('#N/A');
                expect(odfUiConfig.formatReference(docModel, null, null, cell('$0.$1'), cell('$255.$1'), true)).to.equal('R2');
                // relative
                expect(ooxOpConfig.formatReference(docModel, null, null, cell('$0.1'), cell('$255.1'))).to.equal('2:2');
                expect(ooxUiConfig.formatReference(docModel, null, null, cell('$0.1'), cell('$255.1'))).to.equal('2:2');
                expect(odfOpConfig.formatReference(docModel, null, null, cell('$0.1'), cell('$255.1'))).to.equal('[.$A2:.$IV2]');
                expect(odfUiConfig.formatReference(docModel, null, null, cell('$0.1'), cell('$255.1'))).to.equal('2:2');
                expect(ooxOpConfig.formatReference(docModel, null, null, cell('$0.1'), cell('$255.1'), true)).to.equal('#N/A');
                expect(ooxUiConfig.formatReference(docModel, null, null, cell('$0.1'), cell('$255.1'), true)).to.equal('Z[1]');
                expect(odfOpConfig.formatReference(docModel, null, null, cell('$0.1'), cell('$255.1'), true)).to.equal('#N/A');
                expect(odfUiConfig.formatReference(docModel, null, null, cell('$0.1'), cell('$255.1'), true)).to.equal('R[1]');
            });
            it('should return string representation of a row range with sheet', function () {
                // single sheet
                expect(ooxOpConfig.formatReference(docModel, sheet(1), null, cell('$0.$1'), cell('$255.1'))).to.equal('Sheet2!$2:2');
                expect(ooxUiConfig.formatReference(docModel, sheet(1), null, cell('$0.$1'), cell('$255.1'))).to.equal('Sheet2!$2:2');
                expect(odfOpConfig.formatReference(docModel, sheet(1), null, cell('$0.$1'), cell('$255.1'))).to.equal('[$Sheet2.$A$2:.$IV2]');
                expect(odfUiConfig.formatReference(docModel, sheet(1), null, cell('$0.$1'), cell('$255.1'))).to.equal('$Sheet2!$2:2');
                expect(ooxOpConfig.formatReference(docModel, sheet(1), null, cell('$0.$1'), cell('$255.1'), true)).to.equal('#N/A');
                expect(ooxUiConfig.formatReference(docModel, sheet(1), null, cell('$0.$1'), cell('$255.1'), true)).to.equal('Sheet2!Z2:Z[1]');
                expect(odfOpConfig.formatReference(docModel, sheet(1), null, cell('$0.$1'), cell('$255.1'), true)).to.equal('#N/A');
                expect(odfUiConfig.formatReference(docModel, sheet(1), null, cell('$0.$1'), cell('$255.1'), true)).to.equal('$Sheet2!R2:R[1]');
                // sheet range
                expect(ooxOpConfig.formatReference(docModel, sheet(1), sheet(3), cell('$0.$1'), cell('$255.1'))).to.equal('\'Sheet2:Sheet 4\'!$2:2');
                expect(ooxUiConfig.formatReference(docModel, sheet(1), sheet(3), cell('$0.$1'), cell('$255.1'))).to.equal('\'Sheet2:Sheet 4\'!$2:2');
                expect(odfOpConfig.formatReference(docModel, sheet(1), sheet(3), cell('$0.$1'), cell('$255.1'))).to.equal('[$Sheet2.$A$2:$\'Sheet 4\'.$IV2]');
                expect(odfUiConfig.formatReference(docModel, sheet(1), sheet(3), cell('$0.$1'), cell('$255.1'))).to.equal('$Sheet2:$\'Sheet 4\'!$2:2');
            });
            it('should not return string representation of a row range', function () {
                // missing absolute flags
                expect(ooxOpConfig.formatReference(docModel, null, null, cell('$0.$1'), cell('255.$3'))).to.equal('$A$2:IV$4');
                expect(ooxUiConfig.formatReference(docModel, null, null, cell('$0.$1'), cell('255.$3'))).to.equal('$A$2:IV$4');
                expect(odfOpConfig.formatReference(docModel, null, null, cell('$0.$1'), cell('255.$3'))).to.equal('[.$A$2:.IV$4]');
                expect(odfUiConfig.formatReference(docModel, null, null, cell('$0.$1'), cell('255.$3'))).to.equal('$A$2:IV$4');
                expect(ooxOpConfig.formatReference(docModel, null, null, cell('0.$1'), cell('$255.$3'))).to.equal('A$2:$IV$4');
                expect(ooxUiConfig.formatReference(docModel, null, null, cell('0.$1'), cell('$255.$3'))).to.equal('A$2:$IV$4');
                expect(odfOpConfig.formatReference(docModel, null, null, cell('0.$1'), cell('$255.$3'))).to.equal('[.A$2:.$IV$4]');
                expect(odfUiConfig.formatReference(docModel, null, null, cell('0.$1'), cell('$255.$3'))).to.equal('A$2:$IV$4');
                // wrong indexes
                expect(ooxOpConfig.formatReference(docModel, null, null, cell('$1.$1'), cell('$255.$3'))).to.equal('$B$2:$IV$4');
                expect(ooxUiConfig.formatReference(docModel, null, null, cell('$1.$1'), cell('$255.$3'))).to.equal('$B$2:$IV$4');
                expect(odfOpConfig.formatReference(docModel, null, null, cell('$1.$1'), cell('$255.$3'))).to.equal('[.$B$2:.$IV$4]');
                expect(odfUiConfig.formatReference(docModel, null, null, cell('$1.$1'), cell('$255.$3'))).to.equal('$B$2:$IV$4');
                expect(ooxOpConfig.formatReference(docModel, null, null, cell('$0.$1'), cell('$254.$3'))).to.equal('$A$2:$IU$4');
                expect(ooxUiConfig.formatReference(docModel, null, null, cell('$0.$1'), cell('$254.$3'))).to.equal('$A$2:$IU$4');
                expect(odfOpConfig.formatReference(docModel, null, null, cell('$0.$1'), cell('$254.$3'))).to.equal('[.$A$2:.$IU$4]');
                expect(odfUiConfig.formatReference(docModel, null, null, cell('$0.$1'), cell('$254.$3'))).to.equal('$A$2:$IU$4');
            });
            it('should return string representation of a sheet range', function () {
                expect(ooxOpConfig.formatReference(docModel, null, null, cell('$0.$0'), cell('$255.$65535'))).to.equal('$1:$65536');
                expect(ooxUiConfig.formatReference(docModel, null, null, cell('$0.$0'), cell('$255.$65535'))).to.equal('$1:$65536');
                expect(odfOpConfig.formatReference(docModel, null, null, cell('$0.$0'), cell('$255.$65535'))).to.equal('[.$A$1:.$IV$65536]');
                expect(odfUiConfig.formatReference(docModel, null, null, cell('$0.$0'), cell('$255.$65535'))).to.equal('$1:$65536');
                expect(ooxOpConfig.formatReference(docModel, null, null, cell('$0.$0'), cell('$255.$65535'), true)).to.equal('#N/A');
                expect(ooxUiConfig.formatReference(docModel, null, null, cell('$0.$0'), cell('$255.$65535'), true)).to.equal('Z1:Z65536');
                expect(odfOpConfig.formatReference(docModel, null, null, cell('$0.$0'), cell('$255.$65535'), true)).to.equal('#N/A');
                expect(odfUiConfig.formatReference(docModel, null, null, cell('$0.$0'), cell('$255.$65535'), true)).to.equal('R1:R65536');
            });
            it('should handle simple/complex sheet names correctly', function () {
                var c1 = cell('0.0');
                // simple name
                expect(ooxOpConfig.formatReference(docModel, sheet('Sheet1'), null, c1, null)).to.equal('Sheet1!A1');
                expect(ooxUiConfig.formatReference(docModel, sheet('Sheet1'), null, c1, null)).to.equal('Sheet1!A1');
                expect(odfOpConfig.formatReference(docModel, sheet('Sheet1'), null, c1, null)).to.equal('[$Sheet1.A1]');
                expect(odfUiConfig.formatReference(docModel, sheet('Sheet1'), null, c1, null)).to.equal('$Sheet1!A1');
                // whitespace
                expect(ooxOpConfig.formatReference(docModel, sheet('Sheet 2'), null, c1, null)).to.equal('\'Sheet 2\'!A1');
                expect(ooxUiConfig.formatReference(docModel, sheet('Sheet 2'), null, c1, null)).to.equal('\'Sheet 2\'!A1');
                expect(odfOpConfig.formatReference(docModel, sheet('Sheet 2'), null, c1, null)).to.equal('[$\'Sheet 2\'.A1]');
                expect(odfUiConfig.formatReference(docModel, sheet('Sheet 2'), null, c1, null)).to.equal('$\'Sheet 2\'!A1');
                // leading digit
                expect(ooxOpConfig.formatReference(docModel, sheet('1Sheet'), null, c1, null)).to.equal('\'1Sheet\'!A1');
                expect(ooxUiConfig.formatReference(docModel, sheet('1Sheet'), null, c1, null)).to.equal('\'1Sheet\'!A1');
                expect(odfOpConfig.formatReference(docModel, sheet('1Sheet'), null, c1, null)).to.equal('[$\'1Sheet\'.A1]');
                expect(odfUiConfig.formatReference(docModel, sheet('1Sheet'), null, c1, null)).to.equal('$\'1Sheet\'!A1');
                // operator
                expect(ooxOpConfig.formatReference(docModel, sheet('Sheet+1'), null, c1, null)).to.equal('\'Sheet+1\'!A1');
                expect(ooxUiConfig.formatReference(docModel, sheet('Sheet+1'), null, c1, null)).to.equal('\'Sheet+1\'!A1');
                expect(odfOpConfig.formatReference(docModel, sheet('Sheet+1'), null, c1, null)).to.equal('[$\'Sheet+1\'.A1]');
                expect(odfUiConfig.formatReference(docModel, sheet('Sheet+1'), null, c1, null)).to.equal('$\'Sheet+1\'!A1');
                // simple sheet range
                expect(ooxOpConfig.formatReference(docModel, sheet('Sheet1'), sheet('Sheet2'), c1, null)).to.equal('Sheet1:Sheet2!A1');
                expect(ooxUiConfig.formatReference(docModel, sheet('Sheet1'), sheet('Sheet2'), c1, null)).to.equal('Sheet1:Sheet2!A1');
                expect(odfOpConfig.formatReference(docModel, sheet('Sheet1'), sheet('Sheet2'), c1, null)).to.equal('[$Sheet1.A1:$Sheet2.A1]');
                expect(odfUiConfig.formatReference(docModel, sheet('Sheet1'), sheet('Sheet2'), c1, null)).to.equal('$Sheet1:$Sheet2!A1');
                // complex first sheet
                expect(ooxOpConfig.formatReference(docModel, sheet('Sheet 1'), sheet('Sheet2'), c1, null)).to.equal('\'Sheet 1:Sheet2\'!A1');
                expect(ooxUiConfig.formatReference(docModel, sheet('Sheet 1'), sheet('Sheet2'), c1, null)).to.equal('\'Sheet 1:Sheet2\'!A1');
                expect(odfOpConfig.formatReference(docModel, sheet('Sheet 1'), sheet('Sheet2'), c1, null)).to.equal('[$\'Sheet 1\'.A1:$Sheet2.A1]');
                expect(odfUiConfig.formatReference(docModel, sheet('Sheet 1'), sheet('Sheet2'), c1, null)).to.equal('$\'Sheet 1\':$Sheet2!A1');
                // complex second sheet
                expect(ooxOpConfig.formatReference(docModel, sheet('Sheet1'), sheet('Sheet 2'), c1, null)).to.equal('\'Sheet1:Sheet 2\'!A1');
                expect(ooxUiConfig.formatReference(docModel, sheet('Sheet1'), sheet('Sheet 2'), c1, null)).to.equal('\'Sheet1:Sheet 2\'!A1');
                expect(odfOpConfig.formatReference(docModel, sheet('Sheet1'), sheet('Sheet 2'), c1, null)).to.equal('[$Sheet1.A1:$\'Sheet 2\'.A1]');
                expect(odfUiConfig.formatReference(docModel, sheet('Sheet1'), sheet('Sheet 2'), c1, null)).to.equal('$Sheet1:$\'Sheet 2\'!A1');
                // ODF: relative sheets
                expect(odfOpConfig.formatReference(docModel, sheet('Sheet1', false), null, c1, null)).to.equal('[Sheet1.A1]');
                expect(odfUiConfig.formatReference(docModel, sheet('Sheet1', false), null, c1, null)).to.equal('Sheet1!A1');
                expect(odfOpConfig.formatReference(docModel, sheet('Sheet 2', false), null, c1, null)).to.equal('[\'Sheet 2\'.A1]');
                expect(odfUiConfig.formatReference(docModel, sheet('Sheet 2', false), null, c1, null)).to.equal('\'Sheet 2\'!A1');
                expect(odfOpConfig.formatReference(docModel, sheet('Sheet1', false), sheet('Sheet2'), c1, null)).to.equal('[Sheet1.A1:$Sheet2.A1]');
                expect(odfUiConfig.formatReference(docModel, sheet('Sheet1', false), sheet('Sheet2'), c1, null)).to.equal('Sheet1:$Sheet2!A1');
                expect(odfOpConfig.formatReference(docModel, sheet('Sheet1'), sheet('Sheet2', false), c1, null)).to.equal('[$Sheet1.A1:Sheet2.A1]');
                expect(odfUiConfig.formatReference(docModel, sheet('Sheet1'), sheet('Sheet2', false), c1, null)).to.equal('$Sheet1:Sheet2!A1');
                expect(odfOpConfig.formatReference(docModel, sheet('Sheet1', false), sheet('Sheet2', false), c1, null)).to.equal('[Sheet1.A1:Sheet2.A1]');
                expect(odfUiConfig.formatReference(docModel, sheet('Sheet1', false), sheet('Sheet2', false), c1, null)).to.equal('Sheet1:Sheet2!A1');
                expect(odfOpConfig.formatReference(docModel, sheet('Sheet 1', false), sheet('Sheet2', false), c1, null)).to.equal('[\'Sheet 1\'.A1:Sheet2.A1]');
                expect(odfUiConfig.formatReference(docModel, sheet('Sheet 1', false), sheet('Sheet2', false), c1, null)).to.equal('\'Sheet 1\':Sheet2!A1');
                expect(odfOpConfig.formatReference(docModel, sheet('Sheet1', false), sheet('Sheet 2', false), c1, null)).to.equal('[Sheet1.A1:\'Sheet 2\'.A1]');
                expect(odfUiConfig.formatReference(docModel, sheet('Sheet1', false), sheet('Sheet 2', false), c1, null)).to.equal('Sheet1:\'Sheet 2\'!A1');
            });
            it('should treat booleans as complex sheet names', function () {
                var c1 = cell('0.0');
                expect(ooxOpConfig.formatReference(docModel, sheet('TRUE'), null, c1, null)).to.equal('\'TRUE\'!A1');
                expect(ooxUiConfig.formatReference(docModel, sheet('TRUE'), null, c1, null)).to.equal('TRUE!A1');
                expect(ooxOpConfig.formatReference(docModel, sheet('WAHR'), null, c1, null)).to.equal('WAHR!A1');
                expect(ooxUiConfig.formatReference(docModel, sheet('WAHR'), null, c1, null)).to.equal('\'WAHR\'!A1');
                expect(ooxOpConfig.formatReference(docModel, sheet('TRUE'), sheet('WAHR'), c1, null)).to.equal('\'TRUE:WAHR\'!A1');
                expect(ooxUiConfig.formatReference(docModel, sheet('TRUE'), sheet('WAHR'), c1, null)).to.equal('\'TRUE:WAHR\'!A1');
            });
            it('should treat cell addresses as complex sheet names', function () {
                var c1 = cell('0.0');
                expect(ooxOpConfig.formatReference(docModel, sheet('A1'), null, c1, null)).to.equal('\'A1\'!A1');
                expect(ooxUiConfig.formatReference(docModel, sheet('A1'), null, c1, null)).to.equal('\'A1\'!A1');
                expect(odfOpConfig.formatReference(docModel, sheet('A1'), null, c1, null)).to.equal('[$\'A1\'.A1]');
                expect(odfUiConfig.formatReference(docModel, sheet('A1'), null, c1, null)).to.equal('$\'A1\'!A1');
                expect(ooxOpConfig.formatReference(docModel, sheet('IW1'), null, c1, null)).to.equal('IW1!A1');
                expect(ooxUiConfig.formatReference(docModel, sheet('IW1'), null, c1, null)).to.equal('IW1!A1');
                expect(odfOpConfig.formatReference(docModel, sheet('IW1'), null, c1, null)).to.equal('[$IW1.A1]');
                expect(odfUiConfig.formatReference(docModel, sheet('IW1'), null, c1, null)).to.equal('$IW1!A1');
                expect(ooxOpConfig.formatReference(docModel, sheet('R1C1'), null, c1, null)).to.equal('\'R1C1\'!A1');
                expect(ooxUiConfig.formatReference(docModel, sheet('R1C1'), null, c1, null)).to.equal('\'R1C1\'!A1');
                expect(odfOpConfig.formatReference(docModel, sheet('R1C1'), null, c1, null)).to.equal('[$\'R1C1\'.A1]');
                expect(odfUiConfig.formatReference(docModel, sheet('R1C1'), null, c1, null)).to.equal('$\'R1C1\'!A1');
                expect(ooxOpConfig.formatReference(docModel, sheet('Z1S1'), null, c1, null)).to.equal('Z1S1!A1');
                expect(ooxUiConfig.formatReference(docModel, sheet('Z1S1'), null, c1, null)).to.equal('\'Z1S1\'!A1');
                expect(odfOpConfig.formatReference(docModel, sheet('Z1S1'), null, c1, null)).to.equal('[$Z1S1.A1]');
                expect(odfUiConfig.formatReference(docModel, sheet('Z1S1'), null, c1, null)).to.equal('$Z1S1!A1');
                expect(ooxOpConfig.formatReference(docModel, sheet(1), sheet('A1'), c1, null)).to.equal('\'Sheet2:A1\'!A1');
                expect(ooxUiConfig.formatReference(docModel, sheet(1), sheet('A1'), c1, null)).to.equal('\'Sheet2:A1\'!A1');
                expect(odfOpConfig.formatReference(docModel, sheet(1), sheet('A1'), c1, null)).to.equal('[$Sheet2.A1:$\'A1\'.A1]');
                expect(odfUiConfig.formatReference(docModel, sheet(1), sheet('A1'), c1, null)).to.equal('$Sheet2:$\'A1\'!A1');
            });
        });

        describe('method "formatName"', function () {
            it('should exist', function () {
                expect(ooxOpConfig).to.respondTo('formatName');
                expect(ooxUiConfig).to.respondTo('formatName');
            });
            it('should return the string representation', function () {
                var s1 = sheet(1), s2 = sheet(3), s3 = sheet(-1);
                expect(ooxOpConfig.formatName(docModel, null, 'name')).to.equal('name');
                expect(ooxUiConfig.formatName(docModel, null, 'name')).to.equal('name');
                expect(ooxOpConfig.formatName(docModel, s1, 'Name')).to.equal('Sheet2!Name');
                expect(ooxUiConfig.formatName(docModel, s1, 'Name')).to.equal('Sheet2!Name');
                expect(ooxOpConfig.formatName(docModel, s2, 'NAME')).to.equal('\'Sheet 4\'!NAME');
                expect(ooxUiConfig.formatName(docModel, s2, 'NAME')).to.equal('\'Sheet 4\'!NAME');
                expect(ooxOpConfig.formatName(docModel, s3, 'name')).to.equal('#REF!!name');
                expect(ooxUiConfig.formatName(docModel, s3, 'name')).to.equal('#BEZUG!!name');
            });
            it('should handle simple/complex sheet names correctly', function () {
                expect(ooxOpConfig.formatName(docModel, sheet('Sheet1'), 'name')).to.equal('Sheet1!name');
                expect(ooxUiConfig.formatName(docModel, sheet('Sheet1'), 'name')).to.equal('Sheet1!name');
                expect(ooxOpConfig.formatName(docModel, sheet('Sheet 2'), 'name')).to.equal('\'Sheet 2\'!name');
                expect(ooxUiConfig.formatName(docModel, sheet('Sheet 2'), 'name')).to.equal('\'Sheet 2\'!name');
                expect(ooxOpConfig.formatName(docModel, sheet('1Sheet'), 'name')).to.equal('\'1Sheet\'!name');
                expect(ooxUiConfig.formatName(docModel, sheet('1Sheet'), 'name')).to.equal('\'1Sheet\'!name');
                expect(ooxOpConfig.formatName(docModel, sheet('Sheet+1'), 'name')).to.equal('\'Sheet+1\'!name');
                expect(ooxUiConfig.formatName(docModel, sheet('Sheet+1'), 'name')).to.equal('\'Sheet+1\'!name');
            });
            it('should treat booleans as complex sheet names', function () {
                expect(ooxOpConfig.formatName(docModel, sheet('TRUE'), 'name')).to.equal('\'TRUE\'!name');
                expect(ooxUiConfig.formatName(docModel, sheet('TRUE'), 'name')).to.equal('TRUE!name');
                expect(ooxOpConfig.formatName(docModel, sheet('WAHR'), 'name')).to.equal('WAHR!name');
                expect(ooxUiConfig.formatName(docModel, sheet('WAHR'), 'name')).to.equal('\'WAHR\'!name');
            });
            it('should treat cell addresses as complex sheet names', function () {
                expect(ooxOpConfig.formatName(docModel, sheet('A1'), 'name')).to.equal('\'A1\'!name');
                expect(ooxUiConfig.formatName(docModel, sheet('A1'), 'name')).to.equal('\'A1\'!name');
                expect(ooxOpConfig.formatName(docModel, sheet('IW1'), 'name')).to.equal('IW1!name');
                expect(ooxUiConfig.formatName(docModel, sheet('IW1'), 'name')).to.equal('IW1!name');
                expect(ooxOpConfig.formatName(docModel, sheet('R1C1'), 'name')).to.equal('\'R1C1\'!name');
                expect(ooxUiConfig.formatName(docModel, sheet('R1C1'), 'name')).to.equal('\'R1C1\'!name');
                expect(ooxOpConfig.formatName(docModel, sheet('Z1S1'), 'name')).to.equal('Z1S1!name');
                expect(ooxUiConfig.formatName(docModel, sheet('Z1S1'), 'name')).to.equal('\'Z1S1\'!name');
            });
        });

        describe('method "getBooleanName"', function () {
            it('should exist', function () {
                expect(ooxOpConfig).to.respondTo('getBooleanName');
                expect(ooxUiConfig).to.respondTo('getBooleanName');
            });
            it('should return native boolean names', function () {
                expect(ooxOpConfig.getBooleanName(false)).to.equal('FALSE');
                expect(ooxOpConfig.getBooleanName(true)).to.equal('TRUE');
            });
            it('should return localized error names', function () {
                expect(ooxUiConfig.getBooleanName(false)).to.equal('FALSCH');
                expect(ooxUiConfig.getBooleanName(true)).to.equal('WAHR');
            });
        });

        describe('method "getBooleanValue"', function () {
            it('should exist', function () {
                expect(ooxOpConfig).to.respondTo('getBooleanValue');
                expect(ooxUiConfig).to.respondTo('getBooleanValue');
            });
            it('should parse native boolean names', function () {
                expect(ooxOpConfig.getBooleanValue('false')).to.equal(false);
                expect(ooxOpConfig.getBooleanValue('False')).to.equal(false);
                expect(ooxOpConfig.getBooleanValue('FALSE')).to.equal(false);
                expect(ooxOpConfig.getBooleanValue('true')).to.equal(true);
                expect(ooxOpConfig.getBooleanValue('True')).to.equal(true);
                expect(ooxOpConfig.getBooleanValue('TRUE')).to.equal(true);
            });
            it('should parse localized boolean names', function () {
                expect(ooxUiConfig.getBooleanValue('falsch')).to.equal(false);
                expect(ooxUiConfig.getBooleanValue('Falsch')).to.equal(false);
                expect(ooxUiConfig.getBooleanValue('FALSCH')).to.equal(false);
                expect(ooxUiConfig.getBooleanValue('wahr')).to.equal(true);
                expect(ooxUiConfig.getBooleanValue('Wahr')).to.equal(true);
                expect(ooxUiConfig.getBooleanValue('WAHR')).to.equal(true);
            });
            it('should return null for invalid names', function () {
                expect(ooxOpConfig.getBooleanValue('1')).to.equal(null);
                expect(ooxOpConfig.getBooleanValue('abc')).to.equal(null);
                expect(ooxOpConfig.getBooleanValue('FALSCH')).to.equal(null);
                expect(ooxUiConfig.getBooleanValue('1')).to.equal(null);
                expect(ooxUiConfig.getBooleanValue('abc')).to.equal(null);
                expect(ooxUiConfig.getBooleanValue('FALSE')).to.equal(null);
            });
        });

        describe('method "getErrorName"', function () {
            it('should exist', function () {
                expect(ooxOpConfig).to.respondTo('getErrorName');
                expect(ooxUiConfig).to.respondTo('getErrorName');
            });
            it('should return native error names', function () {
                expect(ooxOpConfig.getErrorName(ErrorCode.NULL)).to.equal('#NULL!');
                expect(ooxOpConfig.getErrorName(ErrorCode.DIV0)).to.equal('#DIV/0!');
                expect(ooxOpConfig.getErrorName(ErrorCode.VALUE)).to.equal('#VALUE!');
                expect(ooxOpConfig.getErrorName(ErrorCode.REF)).to.equal('#REF!');
                expect(ooxOpConfig.getErrorName(ErrorCode.NAME)).to.equal('#NAME?');
                expect(ooxOpConfig.getErrorName(ErrorCode.NUM)).to.equal('#NUM!');
                expect(ooxOpConfig.getErrorName(ErrorCode.NA)).to.equal('#N/A');
                expect(ooxOpConfig.getErrorName(ErrorCode.DATA)).to.equal('#GETTING_DATA');
            });
            it('should return localized error names', function () {
                expect(ooxUiConfig.getErrorName(ErrorCode.NULL)).to.equal('#NULL!');
                expect(ooxUiConfig.getErrorName(ErrorCode.DIV0)).to.equal('#DIV/0!');
                expect(ooxUiConfig.getErrorName(ErrorCode.VALUE)).to.equal('#WERT!');
                expect(ooxUiConfig.getErrorName(ErrorCode.REF)).to.equal('#BEZUG!');
                expect(ooxUiConfig.getErrorName(ErrorCode.NAME)).to.equal('#NAME?');
                expect(ooxUiConfig.getErrorName(ErrorCode.NUM)).to.equal('#ZAHL!');
                expect(ooxUiConfig.getErrorName(ErrorCode.NA)).to.equal('#NV');
                expect(ooxUiConfig.getErrorName(ErrorCode.DATA)).to.equal('#GETTING_DATA');
            });
        });

        describe('method "getErrorCode"', function () {
            it('should exist', function () {
                expect(ooxOpConfig).to.respondTo('getErrorCode');
                expect(ooxUiConfig).to.respondTo('getErrorCode');
            });
            it('should parse native error names', function () {
                expect(ooxOpConfig.getErrorCode('#NULL!')).to.equal(ErrorCode.NULL);
                expect(ooxOpConfig.getErrorCode('#null!')).to.equal(ErrorCode.NULL);
                expect(ooxOpConfig.getErrorCode('#DIV/0!')).to.equal(ErrorCode.DIV0);
                expect(ooxOpConfig.getErrorCode('#VALUE!')).to.equal(ErrorCode.VALUE);
                expect(ooxOpConfig.getErrorCode('#REF!')).to.equal(ErrorCode.REF);
                expect(ooxOpConfig.getErrorCode('#NAME?')).to.equal(ErrorCode.NAME);
                expect(ooxOpConfig.getErrorCode('#NUM!')).to.equal(ErrorCode.NUM);
                expect(ooxOpConfig.getErrorCode('#N/A')).to.equal(ErrorCode.NA);
                expect(ooxOpConfig.getErrorCode('#GETTING_DATA')).to.equal(ErrorCode.DATA);
            });
            it('should parse localized error names', function () {
                expect(ooxUiConfig.getErrorCode('#NULL!')).to.equal(ErrorCode.NULL);
                expect(ooxUiConfig.getErrorCode('#DIV/0!')).to.equal(ErrorCode.DIV0);
                expect(ooxUiConfig.getErrorCode('#WERT!')).to.equal(ErrorCode.VALUE);
                expect(ooxUiConfig.getErrorCode('#wert!')).to.equal(ErrorCode.VALUE);
                expect(ooxUiConfig.getErrorCode('#BEZUG!')).to.equal(ErrorCode.REF);
                expect(ooxUiConfig.getErrorCode('#NAME?')).to.equal(ErrorCode.NAME);
                expect(ooxUiConfig.getErrorCode('#ZAHL!')).to.equal(ErrorCode.NUM);
                expect(ooxUiConfig.getErrorCode('#NV')).to.equal(ErrorCode.NA);
                expect(ooxUiConfig.getErrorCode('#GETTING_DATA')).to.equal(ErrorCode.DATA);
            });
            it('should return null for invalid names', function () {
                expect(ooxOpConfig.getErrorCode('abc')).to.equal(null);
                expect(ooxOpConfig.getErrorCode('#NULL')).to.equal(null);
                expect(ooxOpConfig.getErrorCode('NULL')).to.equal(null);
                expect(ooxOpConfig.getErrorCode('#WERT!')).to.equal(null);
                expect(ooxUiConfig.getErrorCode('abc')).to.equal(null);
                expect(ooxUiConfig.getErrorCode('#NULL')).to.equal(null);
                expect(ooxUiConfig.getErrorCode('NULL')).to.equal(null);
                expect(ooxUiConfig.getErrorCode('#VALUE!')).to.equal(null);
            });
        });

        describe('method "getOperatorName"', function () {
            it('should exist', function () {
                expect(ooxOpConfig).to.respondTo('getOperatorName');
                expect(ooxUiConfig).to.respondTo('getOperatorName');
            });
            it('should return native operator names', function () {
                expect(ooxOpConfig.getOperatorName('add')).to.equal('+');
                expect(ooxOpConfig.getOperatorName('le')).to.equal('<=');
                expect(ooxOpConfig.getOperatorName('list')).to.equal(',');
                expect(ooxOpConfig.getOperatorName('isect')).to.equal(' ');
            });
            it('should return localized operator names', function () {
                expect(ooxUiConfig.getOperatorName('add')).to.equal('+');
                expect(ooxUiConfig.getOperatorName('le')).to.equal('<=');
                expect(ooxUiConfig.getOperatorName('list')).to.equal(';');
                expect(ooxUiConfig.getOperatorName('isect')).to.equal(' ');
            });
            it('should return null for invalid keys', function () {
                expect(ooxOpConfig.getOperatorName('abc')).to.equal(null);
                expect(ooxOpConfig.getOperatorName('ADD')).to.equal(null);
                expect(ooxOpConfig.getOperatorName('+')).to.equal(null);
            });
        });

        describe('method "getOperatorKey"', function () {
            it('should exist', function () {
                expect(ooxOpConfig).to.respondTo('getOperatorKey');
                expect(ooxUiConfig).to.respondTo('getOperatorKey');
            });
            it('should parse native operator names', function () {
                expect(ooxOpConfig.getOperatorKey('+')).to.equal('add');
                expect(ooxOpConfig.getOperatorKey('<=')).to.equal('le');
                expect(ooxOpConfig.getOperatorKey(',')).to.equal('list');
                expect(ooxOpConfig.getOperatorKey(' ')).to.equal('isect');
            });
            it('should parse localized operator names', function () {
                expect(ooxUiConfig.getOperatorKey('+')).to.equal('add');
                expect(ooxUiConfig.getOperatorKey('<=')).to.equal('le');
                expect(ooxUiConfig.getOperatorKey(';')).to.equal('list');
                expect(ooxUiConfig.getOperatorKey(' ')).to.equal('isect');
            });
            it('should return null for invalid names', function () {
                expect(ooxOpConfig.getOperatorKey('abc')).to.equal(null);
                expect(ooxOpConfig.getOperatorKey('add')).to.equal(null);
                expect(ooxOpConfig.getOperatorKey(';')).to.equal(null);
                expect(ooxUiConfig.getOperatorKey('abc')).to.equal(null);
                expect(ooxUiConfig.getOperatorKey('add')).to.equal(null);
                expect(ooxUiConfig.getOperatorKey(',')).to.equal(null);
            });
        });

        describe('method "getFunctionName"', function () {
            it('should exist', function () {
                expect(ooxOpConfig).to.respondTo('getFunctionName');
                expect(ooxUiConfig).to.respondTo('getFunctionName');
            });
            it('should return native function names', function () {
                expect(ooxOpConfig.getFunctionName('ABS')).to.equal('ABS');
                expect(ooxOpConfig.getFunctionName('SUM')).to.equal('SUM');
                expect(ooxOpConfig.getFunctionName('FORMULATEXT')).to.equal('_xlfn.FORMULATEXT');
            });
            it('should return localized function names', function () {
                expect(ooxUiConfig.getFunctionName('ABS')).to.equal('ABS');
                expect(ooxUiConfig.getFunctionName('SUM')).to.equal('SUMME');
                expect(ooxUiConfig.getFunctionName('FORMULATEXT')).to.equal('FORMELTEXT');
            });
            it('should return null for invalid keys', function () {
                expect(ooxOpConfig.getFunctionName('123')).to.equal(null);
                expect(ooxOpConfig.getFunctionName('abc')).to.equal(null);
                expect(ooxOpConfig.getFunctionName('abs')).to.equal(null);
                expect(ooxOpConfig.getFunctionName('ISLEAPYEAR')).to.equal(null);
                expect(ooxUiConfig.getFunctionName('123')).to.equal(null);
                expect(ooxUiConfig.getFunctionName('abc')).to.equal(null);
                expect(ooxUiConfig.getFunctionName('abs')).to.equal(null);
                expect(ooxUiConfig.getFunctionName('ISLEAPYEAR')).to.equal(null);
            });
        });

        describe('method "getFunctionKey"', function () {
            it('should exist', function () {
                expect(ooxOpConfig).to.respondTo('getFunctionKey');
                expect(ooxUiConfig).to.respondTo('getFunctionKey');
            });
            it('should parse native function names', function () {
                expect(ooxOpConfig.getFunctionKey('ABS')).to.equal('ABS');
                expect(ooxOpConfig.getFunctionKey('abs')).to.equal('ABS');
                expect(ooxOpConfig.getFunctionKey('SUM')).to.equal('SUM');
                expect(ooxOpConfig.getFunctionKey('sum')).to.equal('SUM');
                expect(ooxOpConfig.getFunctionKey('_xlfn.FORMULATEXT')).to.equal('FORMULATEXT');
            });
            it('should parse localized function names', function () {
                expect(ooxUiConfig.getFunctionKey('ABS')).to.equal('ABS');
                expect(ooxUiConfig.getFunctionKey('abs')).to.equal('ABS');
                expect(ooxUiConfig.getFunctionKey('SUMME')).to.equal('SUM');
                expect(ooxUiConfig.getFunctionKey('summe')).to.equal('SUM');
                expect(ooxUiConfig.getFunctionKey('FORMELTEXT')).to.equal('FORMULATEXT');
            });
            it('should return null for invalid names', function () {
                expect(ooxOpConfig.getFunctionKey('123')).to.equal(null);
                expect(ooxOpConfig.getFunctionKey('abc')).to.equal(null);
                expect(ooxOpConfig.getFunctionKey('SUMME')).to.equal(null);
                expect(ooxOpConfig.getFunctionKey('ISLEAPYEAR')).to.equal(null);
                expect(ooxOpConfig.getFunctionKey('FORMULATEXT')).to.equal(null);
                expect(ooxOpConfig.getFunctionKey('FORMELTEXT')).to.equal(null);
                expect(ooxUiConfig.getFunctionKey('123')).to.equal(null);
                expect(ooxUiConfig.getFunctionKey('abc')).to.equal(null);
                expect(ooxUiConfig.getFunctionKey('SUM')).to.equal(null);
                expect(ooxUiConfig.getFunctionKey('ISTSCHALTJAHR')).to.equal(null);
            });
        });
    });

    // ========================================================================
});
