/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define([
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/errorcode',
    'io.ox/office/spreadsheet/model/formula/formulautils'
], function (Utils, ErrorCode, FormulaUtils) {

    'use strict';

    // static class FormulaUtils ==============================================

    describe('Spreadsheet module FormulaUtils', function () {

        it('should exist', function () {
            expect(FormulaUtils).to.be.an('object');
        });

        // private helpers ----------------------------------------------------

        var sqrt = Math.sqrt,
            exp = Math.exp,
            log = Math.log;

        // simplified mock for the method FormulaContext.aggregateNumbers()
        function aggregateNumbers(values, init, agg, fin) {
            return fin(_.reduce(values, agg, init));
        }

        // simplified mock for the method FormulaContext.aggregateFiltered()
        function aggregateFiltered(source, crit, data, init, agg, fin) {
            return fin(_.reduce(data, agg, init));
        }

        // constants ----------------------------------------------------------

        describe('constant "MAX_PARAM_COUNT"', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('MAX_PARAM_COUNT').that.is.a('number').and.above(0);
            });
        });

        describe('constant "MAX_MATRIX_ROW_COUNT"', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('MAX_MATRIX_ROW_COUNT').that.is.a('number').and.above(0);
            });
        });

        describe('constant "MAX_MATRIX_COL_COUNT"', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('MAX_MATRIX_COL_COUNT').that.is.a('number').and.above(0);
            });
        });

        describe('constant "MAX_REF_LIST_SIZE"', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('MAX_REF_LIST_SIZE').that.is.a('number').and.above(0);
            });
        });

        describe('constant "MAX_CELL_ITERATION_COUNT"', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('MAX_CELL_ITERATION_COUNT').that.is.a('number').and.above(0);
            });
        });

        describe('constant "UNSUPPORTED_ERROR"', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('UNSUPPORTED_ERROR').that.is.an('object');
                expect(FormulaUtils.UNSUPPORTED_ERROR).to.have.a.property('key', 'unsupported');
            });
        });

        describe('constant "CIRCULAR_ERROR"', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('CIRCULAR_ERROR').that.is.an('object');
                expect(FormulaUtils.CIRCULAR_ERROR).to.have.a.property('key', 'circular');
            });
        });

        describe('constant "INTERNAL_ERROR"', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('INTERNAL_ERROR').that.is.an('object');
                expect(FormulaUtils.INTERNAL_ERROR).to.have.a.property('key', 'internal');
            });
        });

        // logger -------------------------------------------------------------

        describe('should support the logger interface"', function () {
            expect(FormulaUtils).itself.to.respondTo('log');
            expect(FormulaUtils).itself.to.respondTo('info');
            expect(FormulaUtils).itself.to.respondTo('warn');
            expect(FormulaUtils).itself.to.respondTo('error');
        });

        // static methods -----------------------------------------------------

        describe('method "throwInternal"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('throwInternal');
            });
            it('should always throw', function () {
                expect(FormulaUtils.throwInternal).to['throw'](ErrorCode).that.equals(FormulaUtils.INTERNAL_ERROR);
            });
        });

        describe('method "throwErrorCode"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('throwErrorCode');
            });
            it('should return scalar values', function () {
                expect(FormulaUtils.throwErrorCode(42)).to.equal(42);
                expect(FormulaUtils.throwErrorCode('abc')).to.equal('abc');
                expect(FormulaUtils.throwErrorCode(true)).to.equal(true);
                expect(FormulaUtils.throwErrorCode(new Date())).to.be.a('date');
                expect(FormulaUtils.throwErrorCode(null)).to.equal(null);
            });
            it('should throw error codes', function () {
                expect(FormulaUtils.throwErrorCode.bind(null, ErrorCode.NULL)).to['throw'](ErrorCode).that.equals(ErrorCode.NULL);
                expect(FormulaUtils.throwErrorCode.bind(null, ErrorCode.VALUE)).to['throw'](ErrorCode).that.equals(ErrorCode.VALUE);
            });
        });

        describe('method "trunc"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('trunc');
            });
            it('should round down positive numbers', function () {
                expect(FormulaUtils.trunc(1.1)).to.equal(1);
                expect(FormulaUtils.trunc(5.5)).to.equal(5);
                expect(FormulaUtils.trunc(9.9)).to.equal(9);
            });
            it('should round up negative numbers', function () {
                expect(FormulaUtils.trunc(-1.1)).to.equal(-1);
                expect(FormulaUtils.trunc(-5.5)).to.equal(-5);
                expect(FormulaUtils.trunc(-9.9)).to.equal(-9);
            });
            it('should return integers unmodified', function () {
                expect(FormulaUtils.trunc(-1)).to.equal(-1);
                expect(FormulaUtils.trunc(0)).to.equal(0);
                expect(FormulaUtils.trunc(1)).to.equal(1);
            });
        });

        describe('method "add"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('add');
            });
            it('should return the sum of two numbers', function () {
                expect(FormulaUtils.add(3, 2)).to.equal(5);
                expect(FormulaUtils.add(0, 2)).to.equal(2);
            });
            it('should return the concatenation of two strings', function () {
                expect(FormulaUtils.add('a', 'b')).to.equal('ab');
            });
        });

        describe('method "multiply"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('multiply');
            });
            it('should return the product of two numbers', function () {
                expect(FormulaUtils.multiply(3, 2)).to.equal(6);
            });
        });

        describe('method "divide"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('divide');
            });
            it('should return the quotient of two numbers', function () {
                expect(FormulaUtils.divide(3, 2)).to.equal(1.5);
                expect(FormulaUtils.divide(0, 2)).to.equal(0);
            });
            it('should throw the #DIV/0! error, if divisor is 0', function () {
                expect(FormulaUtils.divide.bind(null, 3, 0)).to['throw'](ErrorCode).that.equals(ErrorCode.DIV0);
                expect(FormulaUtils.divide.bind(null, 0, 0)).to['throw'](ErrorCode).that.equals(ErrorCode.DIV0);
            });
        });

        describe('method "modulo"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('modulo');
            });
            it('should return the remainder, if divisor is positive', function () {
                expect(FormulaUtils.modulo(-5, 3)).to.equal(1);
                expect(FormulaUtils.modulo(-4, 3)).to.equal(2);
                expect(FormulaUtils.modulo(-3, 3)).to.equal(0);
                expect(FormulaUtils.modulo(-2, 3)).to.equal(1);
                expect(FormulaUtils.modulo(-1, 3)).to.equal(2);
                expect(FormulaUtils.modulo(0, 3)).to.equal(0);
                expect(FormulaUtils.modulo(1, 3)).to.equal(1);
                expect(FormulaUtils.modulo(2, 3)).to.equal(2);
                expect(FormulaUtils.modulo(3, 3)).to.equal(0);
                expect(FormulaUtils.modulo(4, 3)).to.equal(1);
                expect(FormulaUtils.modulo(5, 3)).to.equal(2);
            });
            it('should return the remainder, if divisor is negative', function () {
                expect(FormulaUtils.modulo(-5, -3)).to.equal(-2);
                expect(FormulaUtils.modulo(-4, -3)).to.equal(-1);
                expect(FormulaUtils.modulo(-3, -3)).to.equal(0);
                expect(FormulaUtils.modulo(-2, -3)).to.equal(-2);
                expect(FormulaUtils.modulo(-1, -3)).to.equal(-1);
                expect(FormulaUtils.modulo(0, -3)).to.equal(0);
                expect(FormulaUtils.modulo(1, -3)).to.equal(-2);
                expect(FormulaUtils.modulo(2, -3)).to.equal(-1);
                expect(FormulaUtils.modulo(3, -3)).to.equal(0);
                expect(FormulaUtils.modulo(4, -3)).to.equal(-2);
                expect(FormulaUtils.modulo(5, -3)).to.equal(-1);
            });
            it('should throw the #DIV/0! error, if divisor is 0', function () {
                expect(FormulaUtils.modulo.bind(null, 3, 0)).to['throw'](ErrorCode).that.equals(ErrorCode.DIV0);
                expect(FormulaUtils.modulo.bind(null, 0, 0)).to['throw'](ErrorCode).that.equals(ErrorCode.DIV0);
            });
        });

        describe('method "powerOOXML"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('powerOOXML');
            });
            it('should return the power of two numbers', function () {
                expect(FormulaUtils.powerOOXML(3, 0)).to.equal(1);
                expect(FormulaUtils.powerOOXML(3, 1)).to.equal(3);
                expect(FormulaUtils.powerOOXML(3, 2)).to.equal(9);
                expect(FormulaUtils.powerOOXML(3, 3)).to.equal(27);
                expect(FormulaUtils.powerOOXML(9, 0.5)).to.almostEqual(3);
                expect(FormulaUtils.powerOOXML(81, 0.25)).to.almostEqual(3);
                expect(FormulaUtils.powerOOXML(4, -1)).to.almostEqual(0.25);
                expect(FormulaUtils.powerOOXML(2, -2)).to.almostEqual(0.25);
                expect(FormulaUtils.powerOOXML(0, 1)).to.equal(0);
                expect(FormulaUtils.powerOOXML(0, 2)).to.equal(0);
                expect(FormulaUtils.powerOOXML(0, 0.5)).to.equal(0);
            });
            it('should throw the #NUM! error, if both numbers are 0', function () {
                expect(FormulaUtils.powerOOXML.bind(null, 0, 0)).to['throw'](ErrorCode).that.equals(ErrorCode.NUM);
            });
            it('should throw the #DIV/0! error, if base is 0, and exponent is negative', function () {
                expect(FormulaUtils.powerOOXML.bind(null, 0, -1)).to['throw'](ErrorCode).that.equals(ErrorCode.DIV0);
            });
        });

        describe('method "powerODF"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('powerODF');
            });
            it('should return the power of two numbers', function () {
                expect(FormulaUtils.powerODF(3, 0)).to.equal(1);
                expect(FormulaUtils.powerODF(3, 1)).to.equal(3);
                expect(FormulaUtils.powerODF(3, 2)).to.equal(9);
                expect(FormulaUtils.powerODF(3, 3)).to.equal(27);
                expect(FormulaUtils.powerODF(9, 0.5)).to.almostEqual(3);
                expect(FormulaUtils.powerODF(81, 0.25)).to.almostEqual(3);
                expect(FormulaUtils.powerODF(4, -1)).to.almostEqual(0.25);
                expect(FormulaUtils.powerODF(2, -2)).to.almostEqual(0.25);
                expect(FormulaUtils.powerODF(0, 1)).to.equal(0);
                expect(FormulaUtils.powerODF(0, 2)).to.equal(0);
                expect(FormulaUtils.powerODF(0, 0.5)).to.equal(0);
            });
            it('should return 1, if both numbers are 0', function () {
                expect(FormulaUtils.powerODF(0, 0)).to.equal(1);
            });
            it('should throw the #NUM! error, if base is 0, and exponent is negative', function () {
                expect(FormulaUtils.powerODF.bind(null, 0, -1)).to['throw'](ErrorCode).that.equals(ErrorCode.NUM);
            });
        });

        describe('property "POWER"', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('POWER').that.is.an('object');
            });
            it('should contain resolvers for file formats', function () {
                expect(FormulaUtils.POWER).to.have.a.property('ooxml', FormulaUtils.powerOOXML);
                expect(FormulaUtils.POWER).to.have.a.property('odf', FormulaUtils.powerODF);
            });
        });

        describe('method "arctan2"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('arctan2');
            });
            it('should return the arc tangent of two numbers', function () {
                expect(FormulaUtils.arctan2(1, 0)).to.be.almostZero;
                expect(FormulaUtils.arctan2(1, 1)).to.almostEqual(0.25 * Math.PI);
                expect(FormulaUtils.arctan2(0, 1)).to.almostEqual(0.5 * Math.PI);
                expect(FormulaUtils.arctan2(-1, 1)).to.almostEqual(0.75 * Math.PI);
                expect(FormulaUtils.arctan2(-1, 0)).to.almostEqual(Math.PI);
                expect(FormulaUtils.arctan2(-1, -1)).to.almostEqual(-0.75 * Math.PI);
                expect(FormulaUtils.arctan2(0, -1)).to.almostEqual(-0.5 * Math.PI);
                expect(FormulaUtils.arctan2(1, -1)).to.almostEqual(-0.25 * Math.PI);
            });
            it('should throw the #DIV/0! error, if both numbers are 0', function () {
                expect(FormulaUtils.arctan2.bind(null, 0, 0)).to['throw'](ErrorCode).that.equals(ErrorCode.DIV0);
            });
        });

        describe('method "sinh"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('sinh');
            });
            it('should return the hyperbolic sine', function () {
                expect(FormulaUtils.sinh(-2)).to.almostEqual((exp(-2) - exp(2)) / 2);
                expect(FormulaUtils.sinh(-1)).to.almostEqual((exp(-1) - exp(1)) / 2);
                expect(FormulaUtils.sinh(0)).to.equal(0);
                expect(FormulaUtils.sinh(1)).to.almostEqual((exp(1) - exp(-1)) / 2);
                expect(FormulaUtils.sinh(2)).to.almostEqual((exp(2) - exp(-2)) / 2);
            });
            it('should use the native implementation', function () {
                if (Math.sinh) { expect(FormulaUtils.sinh).to.equal(Math.sinh); }
            });
        });

        describe('method "cosh"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('cosh');
            });
            it('should return the hyperbolic cosine', function () {
                expect(FormulaUtils.cosh(-2)).to.almostEqual((exp(-2) + exp(2)) / 2);
                expect(FormulaUtils.cosh(-1)).to.almostEqual((exp(-1) + exp(1)) / 2);
                expect(FormulaUtils.cosh(0)).to.equal(1);
                expect(FormulaUtils.cosh(1)).to.almostEqual((exp(1) + exp(-1)) / 2);
                expect(FormulaUtils.cosh(2)).to.almostEqual((exp(2) + exp(-2)) / 2);
            });
            it('should use the native implementation', function () {
                if (Math.cosh) { expect(FormulaUtils.cosh).to.equal(Math.cosh); }
            });
        });

        describe('method "tanh"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('tanh');
            });
            it('should return the hyperbolic tangent', function () {
                expect(FormulaUtils.tanh(-2)).to.almostEqual((exp(-4) - 1) / (exp(-4) + 1));
                expect(FormulaUtils.tanh(-1)).to.almostEqual((exp(-2) - 1) / (exp(-2) + 1));
                expect(FormulaUtils.tanh(0)).to.equal(0);
                expect(FormulaUtils.tanh(1)).to.almostEqual((exp(2) - 1) / (exp(2) + 1));
                expect(FormulaUtils.tanh(2)).to.almostEqual((exp(4) - 1) / (exp(4) + 1));
            });
            it('should use the native implementation', function () {
                if (Math.tanh) { expect(FormulaUtils.tanh).to.equal(Math.tanh); }
            });
        });

        describe('method "asinh"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('asinh');
            });
            it('should return the argument resulting in the hyperbolic sine', function () {
                expect(FormulaUtils.asinh(-2)).to.almostEqual(log(sqrt(5) - 2));
                expect(FormulaUtils.asinh(-1)).to.almostEqual(log(sqrt(2) - 1));
                expect(FormulaUtils.asinh(0)).to.equal(0);
                expect(FormulaUtils.asinh(1)).to.almostEqual(log(sqrt(2) + 1));
                expect(FormulaUtils.asinh(2)).to.almostEqual(log(sqrt(5) + 2));
            });
            it('should use the native implementation', function () {
                if (Math.asinh) { expect(FormulaUtils.asinh).to.equal(Math.asinh); }
            });
        });

        describe('method "acosh"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('acosh');
            });
            it('should return the argument resulting in the hyperbolic cosine', function () {
                expect(FormulaUtils.acosh(-1)).not.to.be.finite;
                expect(FormulaUtils.acosh(0)).not.to.be.finite;
                expect(FormulaUtils.acosh(1)).to.equal(0);
                expect(FormulaUtils.acosh(2)).to.almostEqual(log(2 + sqrt(3)));
                expect(FormulaUtils.acosh(3)).to.almostEqual(log(3 + sqrt(8)));
            });
            it('should use the native implementation', function () {
                if (Math.acosh) { expect(FormulaUtils.acosh).to.equal(Math.acosh); }
            });
        });

        describe('method "atanh"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('atanh');
            });
            it('should return the argument resulting in the hyperbolic tangent', function () {
                expect(FormulaUtils.atanh(-2)).not.to.be.finite;
                expect(FormulaUtils.atanh(-1)).not.to.be.finite;
                expect(FormulaUtils.atanh(-0.5)).to.almostEqual(log(1 / 3) / 2);
                expect(FormulaUtils.atanh(0)).to.equal(0);
                expect(FormulaUtils.atanh(0.5)).to.almostEqual(log(3) / 2);
                expect(FormulaUtils.atanh(1)).not.to.be.finite;
                expect(FormulaUtils.atanh(2)).not.to.be.finite;
            });
            it('should use the native implementation', function () {
                if (Math.atanh) { expect(FormulaUtils.atanh).to.equal(Math.atanh); }
            });
        });

        describe('method "acoth"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('acoth');
            });
            it('should return the argument resulting in the hyperbolic tangent', function () {
                expect(FormulaUtils.acoth(-3)).to.almostEqual(log(1 / 2) / 2);
                expect(FormulaUtils.acoth(-2)).to.almostEqual(log(1 / 3) / 2);
                expect(FormulaUtils.acoth(-1)).not.to.be.finite;
                expect(FormulaUtils.acoth(0)).not.to.be.finite;
                expect(FormulaUtils.acoth(1)).not.to.be.finite;
                expect(FormulaUtils.acoth(2)).to.almostEqual(log(3) / 2);
                expect(FormulaUtils.acoth(3)).to.almostEqual(log(2) / 2);
            });
            it('should use the native implementation', function () {
                if (Math.acoth) { expect(FormulaUtils.acoth).to.equal(Math.acoth); }
            });
        });

        describe('method "compareNumbers"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('compareNumbers');
            });
            it('should return -1, if first number is less than second number', function () {
                expect(FormulaUtils.compareNumbers(-3, 0)).to.equal(-1);
                expect(FormulaUtils.compareNumbers(0, 3)).to.equal(-1);
            });
            it('should return 0, if the numbers are equal', function () {
                expect(FormulaUtils.compareNumbers(-3, -3)).to.equal(0);
                expect(FormulaUtils.compareNumbers(0, 0)).to.equal(0);
                expect(FormulaUtils.compareNumbers(3, 3)).to.equal(0);
                expect(FormulaUtils.compareNumbers(0, Utils.MIN_NUMBER / 2)).to.equal(0);
            });
            it('should return 1, if first number is greater than second number', function () {
                expect(FormulaUtils.compareNumbers(3, 0)).to.equal(1);
                expect(FormulaUtils.compareNumbers(0, -3)).to.equal(1);
            });
            it('should return NaN, if either number is not finite', function () {
                expect(FormulaUtils.compareNumbers(3, 1 / 0)).to.be.NaN;
                expect(FormulaUtils.compareNumbers(1 / 0, 3)).to.be.NaN;
                expect(FormulaUtils.compareNumbers(1 / 0, 1 / 0)).to.be.NaN;
                expect(FormulaUtils.compareNumbers(3, Number.NaN)).to.be.NaN;
                expect(FormulaUtils.compareNumbers(Number.NaN, 3)).to.be.NaN;
                expect(FormulaUtils.compareNumbers(Number.NaN, Number.NaN)).to.be.NaN;
            });
        });

        describe('method "compareStrings"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('compareStrings');
            });
            it('should return -1, if first string is less than second string (ignoring case)', function () {
                expect(FormulaUtils.compareStrings('a', 'b')).to.equal(-1);
                expect(FormulaUtils.compareStrings('A', 'b')).to.equal(-1);
                expect(FormulaUtils.compareStrings('a', 'B')).to.equal(-1);
                expect(FormulaUtils.compareStrings('A', 'B')).to.equal(-1);
                expect(FormulaUtils.compareStrings('a', 'aa')).to.equal(-1);
                expect(FormulaUtils.compareStrings('ab', 'ba')).to.equal(-1);
                expect(FormulaUtils.compareStrings('', 'a')).to.equal(-1);
            });
            it('should return 0, if the strings are equal (ignoring case)', function () {
                expect(FormulaUtils.compareStrings('a', 'a')).to.equal(0);
                expect(FormulaUtils.compareStrings('a', 'A')).to.equal(0);
                expect(FormulaUtils.compareStrings('A', 'a')).to.equal(0);
                expect(FormulaUtils.compareStrings('A', 'A')).to.equal(0);
                expect(FormulaUtils.compareStrings('abc', 'abc')).to.equal(0);
                expect(FormulaUtils.compareStrings('', '')).to.equal(0);
            });
            it('should return 1, if first string is greater than second string (ignoring case)', function () {
                expect(FormulaUtils.compareStrings('b', 'a')).to.equal(1);
                expect(FormulaUtils.compareStrings('B', 'a')).to.equal(1);
                expect(FormulaUtils.compareStrings('b', 'A')).to.equal(1);
                expect(FormulaUtils.compareStrings('B', 'A')).to.equal(1);
                expect(FormulaUtils.compareStrings('aa', 'a')).to.equal(1);
                expect(FormulaUtils.compareStrings('ba', 'ab')).to.equal(1);
                expect(FormulaUtils.compareStrings('a', '')).to.equal(1);
            });
            it('should return -1, if first string is less than second string (regarding case)', function () {
                expect(FormulaUtils.compareStrings('a', 'A', { withCase: true })).to.equal(-1);
                expect(FormulaUtils.compareStrings('a', 'b', { withCase: true })).to.equal(-1);
                expect(FormulaUtils.compareStrings('a', 'B', { withCase: true })).to.equal(-1);
                expect(FormulaUtils.compareStrings('a', 'c', { withCase: true })).to.equal(-1);
                expect(FormulaUtils.compareStrings('a', 'C', { withCase: true })).to.equal(-1);
                expect(FormulaUtils.compareStrings('a', 'aa', { withCase: true })).to.equal(-1);
                expect(FormulaUtils.compareStrings('ba', 'bA', { withCase: true })).to.equal(-1);
                expect(FormulaUtils.compareStrings('', 'a', { withCase: true })).to.equal(-1);
            });
            it('should return 0, if the strings are equal (regarding case)', function () {
                expect(FormulaUtils.compareStrings('a', 'a', { withCase: true })).to.equal(0);
                expect(FormulaUtils.compareStrings('A', 'A', { withCase: true })).to.equal(0);
                expect(FormulaUtils.compareStrings('abc', 'abc', { withCase: true })).to.equal(0);
                expect(FormulaUtils.compareStrings('', '', { withCase: true })).to.equal(0);
            });
            it('should return 1, if first string is greater than second string (regarding case)', function () {
                expect(FormulaUtils.compareStrings('A', 'a', { withCase: true })).to.equal(1);
                expect(FormulaUtils.compareStrings('b', 'a', { withCase: true })).to.equal(1);
                expect(FormulaUtils.compareStrings('B', 'a', { withCase: true })).to.equal(1);
                expect(FormulaUtils.compareStrings('c', 'a', { withCase: true })).to.equal(1);
                expect(FormulaUtils.compareStrings('C', 'a', { withCase: true })).to.equal(1);
                expect(FormulaUtils.compareStrings('aa', 'a', { withCase: true })).to.equal(1);
                expect(FormulaUtils.compareStrings('bA', 'ba', { withCase: true })).to.equal(1);
                expect(FormulaUtils.compareStrings('a', '', { withCase: true })).to.equal(1);
            });
        });

        describe('method "compareBooleans"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('compareBooleans');
            });
            it('should return the result of Boolean comparison', function () {
                expect(FormulaUtils.compareBooleans(false, false)).to.equal(0);
                expect(FormulaUtils.compareBooleans(false, true)).to.equal(-1);
                expect(FormulaUtils.compareBooleans(true, false)).to.equal(1);
                expect(FormulaUtils.compareBooleans(true, true)).to.equal(0);
            });
        });

        describe('method "compareErrorCodes"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('compareErrorCodes');
            });
            it('should return the result of error code comparison', function () {
                expect(FormulaUtils.compareErrorCodes(ErrorCode.NULL, ErrorCode.NULL)).to.equal(0);
                expect(FormulaUtils.compareErrorCodes(ErrorCode.NULL, ErrorCode.DIV0)).to.equal(-1);
                expect(FormulaUtils.compareErrorCodes(ErrorCode.DIV0, ErrorCode.NULL)).to.equal(1);
            });
        });

        describe('method "compareValues"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('compareValues');
            });
            it('should compare null to other types', function () {
                expect(FormulaUtils.compareValues(null, 0)).to.equal(-1);
                expect(FormulaUtils.compareValues(null, '')).to.equal(-1);
                expect(FormulaUtils.compareValues(null, false)).to.equal(-1);
                expect(FormulaUtils.compareValues(null, ErrorCode.NULL)).to.equal(-1);
            });
            it('should compare numbers to other types', function () {
                expect(FormulaUtils.compareValues(0, null)).to.equal(1);
                expect(FormulaUtils.compareValues(0, '0')).to.equal(-1);
                expect(FormulaUtils.compareValues(0, false)).to.equal(-1);
                expect(FormulaUtils.compareValues(0, ErrorCode.NULL)).to.equal(-1);
            });
            it('should compare strings to other types', function () {
                expect(FormulaUtils.compareValues('', null)).to.equal(1);
                expect(FormulaUtils.compareValues('0', 0)).to.equal(1);
                expect(FormulaUtils.compareValues('false', false)).to.equal(-1);
                expect(FormulaUtils.compareValues('#NULL!', ErrorCode.NULL)).to.equal(-1);
            });
            it('should compare booleans to other types', function () {
                expect(FormulaUtils.compareValues(false, null)).to.equal(1);
                expect(FormulaUtils.compareValues(false, 0)).to.equal(1);
                expect(FormulaUtils.compareValues(false, 'false')).to.equal(1);
                expect(FormulaUtils.compareValues(false, ErrorCode.NULL)).to.equal(-1);
            });
            it('should compare error codes to other types', function () {
                expect(FormulaUtils.compareValues(ErrorCode.NULL, null)).to.equal(1);
                expect(FormulaUtils.compareValues(ErrorCode.NULL, 0)).to.equal(1);
                expect(FormulaUtils.compareValues(ErrorCode.NULL, '#NULL!')).to.equal(1);
                expect(FormulaUtils.compareValues(ErrorCode.NULL, false)).to.equal(1);
            });
            it('should compare numbers', function () {
                expect(FormulaUtils.compareValues(0, 1)).to.equal(-1);
                expect(FormulaUtils.compareValues(1, 1)).to.equal(0);
                expect(FormulaUtils.compareValues(1, 0)).to.equal(1);
                expect(FormulaUtils.compareValues(3, 1 / 0)).to.be.NaN;
            });
            it('should compare strings (ignoring case)', function () {
                expect(FormulaUtils.compareValues('a', 'a')).to.equal(0);
                expect(FormulaUtils.compareValues('a', 'A')).to.equal(0);
                expect(FormulaUtils.compareValues('a', 'b')).to.equal(-1);
                expect(FormulaUtils.compareValues('a', 'B')).to.equal(-1);
                expect(FormulaUtils.compareValues('b', 'a')).to.equal(1);
                expect(FormulaUtils.compareValues('B', 'a')).to.equal(1);
            });
            it('should compare strings (regarding case)', function () {
                expect(FormulaUtils.compareValues('a', 'a', { withCase: true })).to.equal(0);
                expect(FormulaUtils.compareValues('a', 'A', { withCase: true })).to.equal(-1);
                expect(FormulaUtils.compareValues('A', 'a', { withCase: true })).to.equal(1);
                expect(FormulaUtils.compareValues('a', 'B', { withCase: true })).to.equal(-1);
                expect(FormulaUtils.compareValues('A', 'b', { withCase: true })).to.equal(-1);
            });
            it('should compare booleans', function () {
                expect(FormulaUtils.compareValues(false, false)).to.equal(0);
                expect(FormulaUtils.compareValues(false, true)).to.equal(-1);
                expect(FormulaUtils.compareValues(true, false)).to.equal(1);
                expect(FormulaUtils.compareValues(true, true)).to.equal(0);
            });
            it('should compare error codes', function () {
                expect(FormulaUtils.compareValues(ErrorCode.NULL, ErrorCode.NULL)).to.equal(0);
                expect(FormulaUtils.compareValues(ErrorCode.NULL, ErrorCode.DIV0)).to.equal(-1);
                expect(FormulaUtils.compareValues(ErrorCode.DIV0, ErrorCode.NULL)).to.equal(1);
            });
            it('should convert null value to zero', function () {
                expect(FormulaUtils.compareValues(null, -1, { convertNull: true })).to.equal(1);
                expect(FormulaUtils.compareValues(null, 0, { convertNull: true })).to.equal(0);
                expect(FormulaUtils.compareValues(null, 1, { convertNull: true })).to.equal(-1);
                expect(FormulaUtils.compareValues(-1, null, { convertNull: true })).to.equal(-1);
                expect(FormulaUtils.compareValues(0, null, { convertNull: true })).to.equal(0);
                expect(FormulaUtils.compareValues(1, null, { convertNull: true })).to.equal(1);
            });
            it('should convert null value to empty string', function () {
                expect(FormulaUtils.compareValues(null, '', { convertNull: true })).to.equal(0);
                expect(FormulaUtils.compareValues(null, 'a', { convertNull: true })).to.equal(-1);
                expect(FormulaUtils.compareValues('', null, { convertNull: true })).to.equal(0);
                expect(FormulaUtils.compareValues('a', null, { convertNull: true })).to.equal(1);
            });
            it('should convert null value to FALSE', function () {
                expect(FormulaUtils.compareValues(null, false, { convertNull: true })).to.equal(0);
                expect(FormulaUtils.compareValues(null, true, { convertNull: true })).to.equal(-1);
                expect(FormulaUtils.compareValues(false, null, { convertNull: true })).to.equal(0);
                expect(FormulaUtils.compareValues(true, null, { convertNull: true })).to.equal(1);
            });
            it('should not convert null value to error code', function () {
                expect(FormulaUtils.compareValues(null, ErrorCode.NULL, { convertNull: true })).to.equal(-1);
                expect(FormulaUtils.compareValues(ErrorCode.NULL, null, { convertNull: true })).to.equal(1);
            });
            it('should compare two null values', function () {
                expect(FormulaUtils.compareValues(null, null)).to.equal(0);
                expect(FormulaUtils.compareValues(null, null, { convertNull: true })).to.equal(0);
            });
        });

        describe('method "isValidMatrixSize"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('isValidMatrixSize');
            });
            it('should return true for valid matrix sizes', function () {
                expect(FormulaUtils.isValidMatrixSize(1, 1)).to.equal(true);
                expect(FormulaUtils.isValidMatrixSize(FormulaUtils.MAX_MATRIX_ROW_COUNT, 1)).to.equal(true);
                expect(FormulaUtils.isValidMatrixSize(1, FormulaUtils.MAX_MATRIX_COL_COUNT)).to.equal(true);
                expect(FormulaUtils.isValidMatrixSize(FormulaUtils.MAX_MATRIX_ROW_COUNT, FormulaUtils.MAX_MATRIX_COL_COUNT)).to.equal(true);
            });
            it('should return false for invalid matrix sizes', function () {
                expect(FormulaUtils.isValidMatrixSize(0, 1)).to.equal(false);
                expect(FormulaUtils.isValidMatrixSize(1, 0)).to.equal(false);
                expect(FormulaUtils.isValidMatrixSize(FormulaUtils.MAX_MATRIX_ROW_COUNT + 1, 1)).to.equal(false);
                expect(FormulaUtils.isValidMatrixSize(1, FormulaUtils.MAX_MATRIX_COL_COUNT + 1)).to.equal(false);
            });
        });

        describe('method "ensureMatrixSize"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('ensureMatrixSize');
            });
            it('should do nothing for valid matrix sizes', function () {
                expect(FormulaUtils.ensureMatrixSize(1, 1)).to.equal(undefined);
                expect(FormulaUtils.ensureMatrixSize(FormulaUtils.MAX_MATRIX_ROW_COUNT, 1)).to.equal(undefined);
                expect(FormulaUtils.ensureMatrixSize(1, FormulaUtils.MAX_MATRIX_COL_COUNT)).to.equal(undefined);
                expect(FormulaUtils.ensureMatrixSize(FormulaUtils.MAX_MATRIX_ROW_COUNT, FormulaUtils.MAX_MATRIX_COL_COUNT)).to.equal(undefined);
            });
            it('should throw for invalid matrix sizes', function () {
                expect(FormulaUtils.ensureMatrixSize.bind(null, 0, 1)).to['throw'](ErrorCode).that.equals(FormulaUtils.UNSUPPORTED_ERROR);
                expect(FormulaUtils.ensureMatrixSize.bind(null, 1, 0)).to['throw'](ErrorCode).that.equals(FormulaUtils.UNSUPPORTED_ERROR);
                expect(FormulaUtils.ensureMatrixSize.bind(null, FormulaUtils.MAX_MATRIX_ROW_COUNT + 1, 1)).to['throw'](ErrorCode).that.equals(FormulaUtils.UNSUPPORTED_ERROR);
                expect(FormulaUtils.ensureMatrixSize.bind(null, 1, FormulaUtils.MAX_MATRIX_COL_COUNT + 1)).to['throw'](ErrorCode).that.equals(FormulaUtils.UNSUPPORTED_ERROR);
            });
        });

        describe('method "implementNumericAggregation"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('implementNumericAggregation');
            });
            it('should return an aggregator function', function () {
                var agg = sinon.spy(FormulaUtils.add), fin = sinon.spy(_.identity), opt = {}, spy = sinon.spy(aggregateNumbers);
                var resolver = FormulaUtils.implementNumericAggregation(1, agg, fin, opt);
                expect(resolver).to.be.a('function');
                sinon.assert.callCount(agg, 0);
                sinon.assert.callCount(fin, 0);
                var result = resolver.call({ aggregateNumbers: spy }, 42, 12.5);
                sinon.assert.callCount(agg, 2);
                sinon.assert.callCount(fin, 1);
                sinon.assert.callCount(spy, 1);
                var args = spy.getCall(0).args;
                expect(args).to.have.length(5);
                expect(args[0]).to.be.an('arguments').and.have.length(2);
                expect(args[0][0]).to.equal(42);
                expect(args[0][1]).to.equal(12.5);
                expect(args[1]).to.equal(1);
                expect(args[2]).to.equal(agg);
                expect(args[3]).to.equal(fin);
                expect(args[4]).to.equal(opt);
                expect(result).to.equal(55.5);
            });
        });

        describe('method "implementNumericAggregationWithArray"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('implementNumericAggregationWithArray');
            });
            it('should return an aggregator function', function () {
                var fin = sinon.spy(_.identity), opt = {}, spy = sinon.spy(aggregateNumbers);
                var resolver = FormulaUtils.implementNumericAggregationWithArray(fin, opt);
                expect(resolver).to.be.a('function');
                sinon.assert.callCount(fin, 0);
                var result = resolver.call({ aggregateNumbers: spy }, 42, 12.5);
                sinon.assert.callCount(fin, 1);
                sinon.assert.calledWithExactly(fin, [42, 12.5], 54.5);
                sinon.assert.callCount(spy, 1);
                var args = spy.getCall(0).args;
                expect(args).to.have.length(5);
                expect(args[0]).to.be.an('arguments').and.have.length(2);
                expect(args[0][0]).to.equal(42);
                expect(args[0][1]).to.equal(12.5);
                expect(args[1]).to.equal(0);
                expect(args[2]).to.be.a('function');
                expect(args[3]).to.be.a('function').and.not.equal(fin);
                expect(args[4]).to.equal(opt);
                expect(result).to.deep.equal([42, 12.5]);
            });
        });

        describe('method "implementFilterAggregation"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('implementFilterAggregation');
            });
            it('should return an aggregator function', function () {
                var agg = sinon.spy(FormulaUtils.add), fin = sinon.spy(_.identity), spy = sinon.spy(aggregateFiltered);
                var resolver = FormulaUtils.implementFilterAggregation(1, agg, fin);
                expect(resolver).to.be.a('function');
                sinon.assert.callCount(agg, 0);
                sinon.assert.callCount(fin, 0);
                var data = [42, 12.5];
                var result = resolver.call({ aggregateFiltered: spy }, 'src', '>1', data);
                sinon.assert.callCount(agg, 2);
                sinon.assert.callCount(fin, 1);
                sinon.assert.callCount(spy, 1);
                sinon.assert.calledWithExactly(spy, 'src', '>1', data, 1, agg, fin);
                expect(result).to.equal(55.5);
            });
        });

        describe('method "valueToString"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('valueToString');
            });
            it('should convert strings', function () {
                expect(FormulaUtils.valueToString('abc')).to.equal('"abc"');
                expect(FormulaUtils.valueToString('a"b"c')).to.equal('"a""b""c"');
                expect(FormulaUtils.valueToString('')).to.equal('""');
            });
            it('should convert dates', function () {
                var now = new Date();
                expect(FormulaUtils.valueToString(now)).to.equal(now.toISOString());
            });
            it('should convert null and booleans', function () {
                expect(FormulaUtils.valueToString(null)).to.equal('null');
                expect(FormulaUtils.valueToString(true)).to.equal('TRUE');
                expect(FormulaUtils.valueToString(false)).to.equal('FALSE');
            });
            it('should convert numbers', function () {
                expect(FormulaUtils.valueToString(42)).to.equal('42');
                expect(FormulaUtils.valueToString(-12.5)).to.equal('-12.5');
            });
            it('should convert error codes', function () {
                expect(FormulaUtils.valueToString(ErrorCode.REF)).to.equal('#REF');
            });
        });

        describe('method "logTokens"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('logTokens');
            });
        });
    });

    // ========================================================================
});
